﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testG3d_FileIoUtility.h"
#include "testG3d_ViewerTestAllocator.h"

#include <nn/nn_Assert.h>

#include <cstdio>
#include <io.h>
#include <algorithm>
#include <nn/fs.h>
#include <nn/g3d/g3d_ResFile.h>
#include <nn/nn_Allocator.h>


size_t G3dTest::GetResFileAlignment(const char* path, G3dTest::ViewerTestAllocator& allocator)
{
    // ResFileのアライメントはBinaryFileHeaderから取得
    nn::fs::FileHandle fileHandle;

    nn::Result result = nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());

    void* buffer;
    buffer = allocator.Allocate(sizeof(nn::util::BinaryFileHeader), 8);

    size_t readSize;
    result = nn::fs::ReadFile(&readSize, fileHandle, 0, buffer, sizeof(nn::util::BinaryFileHeader));
    NN_ASSERT(result.IsSuccess());

    nn::fs::CloseFile(fileHandle);

    size_t alignment = reinterpret_cast<nn::util::BinaryFileHeader*>(buffer)->GetAlignment();
    allocator.Free(buffer);

    return alignment;
}


nn::g3d::ResFile* G3dTest::LoadResFile(const char* path, nn::gfx::Device& device, G3dTest::ViewerTestAllocator& allocator)
{
    size_t size;
    size_t alignment = GetResFileAlignment(path, allocator);
    void* pFile = LoadFile(path, allocator, &size, alignment);
    NN_ASSERT(nn::g3d::ResFile::IsValid(pFile));

    // リソースファイル内のポインタ情報をセットアップする
    static_cast<nn::util::BinaryFileHeader*>(pFile)->GetRelocationTable()->Relocate();
    nn::g3d::ResFile* pResFile = nn::g3d::ResFile::ResCast(pFile);
    NN_ASSERT_NOT_NULL(pResFile);
    pResFile->Setup(&device);
    return pResFile;
}

void* G3dTest::LoadFile(const char* path, ViewerTestAllocator& allocator, size_t* pSize, size_t alignment)
{
    alignment = std::max<size_t>(alignment, nn::DefaultAlignment);

    nn::fs::FileHandle fileHandle;
    int64_t fileSize = 0;
    nn::Result result = nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());

    result = nn::fs::GetFileSize(&fileSize, fileHandle);
    NN_ASSERT(result.IsSuccess());

    // テキストファイル NUL 終端させるために多めにメモリ確保して 0 で埋める。
    void* pBuffer = allocator.Allocate(static_cast<size_t>(fileSize) + nn::DefaultAlignment, alignment);
    result = nn::fs::ReadFile(pSize, fileHandle, 0, pBuffer, static_cast<size_t>(fileSize));
    NN_ASSERT(result.IsSuccess());

    memset((int8_t*)pBuffer + fileSize, 0, nn::DefaultAlignment);
    nn::fs::CloseFile(fileHandle);
    return pBuffer;
}

void G3dTest::FreeFile(void* pFileBuffer, ViewerTestAllocator& allocator)
{
    allocator.Free(pFileBuffer);
}

bool G3dTest::Exists(const char* path)
{
    nn::fs::FileHandle fileHandle;
    nn::Result result = nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read);
    return result.IsSuccess();
}
