﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testG3d_Main.h"
#include "nnt/g3d/testG3d_TestUtility.h"

#if defined(NN_BUILD_CONFIG_SPEC_NX)

TEST_F(G3dTest, ResShaderFile)
{
    nn::g3d::ResShaderArchive* pShaderArchive = GetResShaderFile()->GetResShaderArchive();
    EXPECT_TRUE(pShaderArchive != NULL);
    const nn::util::BinaryFileHeader* pHeader = GetResShaderFile()->GetFileHeader();
    EXPECT_TRUE(pHeader != NULL);
    EXPECT_EQ(0, pHeader->GetFileName().compare("demo"));
}

TEST_F(G3dTest, ResShaderArchive)
{
    nn::g3d::ResShaderArchive* pShaderArchive = GetResShaderFile()->GetResShaderArchive();
    EXPECT_STREQ("demo", pShaderArchive->GetName());
    EXPECT_STREQ("", pShaderArchive->GetPath());
    void* ptr = GetResShaderFile();
    pShaderArchive->SetUserPtr(ptr);
    EXPECT_TRUE(ptr == pShaderArchive->GetUserPtr());
#if defined(NN_BUILD_CONFIG_SPEC_NX)
    EXPECT_FALSE(pShaderArchive->HasSource());
#else
    EXPECT_TRUE(pShaderArchive->HasSource());
#endif
    EXPECT_TRUE(pShaderArchive->HasBinary());
    EXPECT_TRUE(pShaderArchive->IsBinaryAvailable());
#if defined(NN_BUILD_CONFIG_SPEC_NX) && defined(NN_BUILD_CONFIG_OS_WIN)
    EXPECT_TRUE(pShaderArchive->HasIr());
#else
    EXPECT_FALSE(pShaderArchive->HasIr());
#endif

    EXPECT_EQ(4, pShaderArchive->GetShadingModelCount());
    nn::g3d::ResShadingModel* pShadingModel = pShaderArchive->FindShadingModel("basic");
    EXPECT_TRUE(pShadingModel != NULL);
    EXPECT_TRUE(pShadingModel == pShaderArchive->GetShadingModel(0));
    EXPECT_STREQ("basic", pShaderArchive->GetShadingModelName(pShaderArchive->FindShadingModelIndex("basic")));
}

TEST_F(G3dTest, ResShadingModel)
{
    nn::g3d::ResShaderArchive* pShaderArchive = GetResShaderFile()->GetResShaderArchive();
    nn::g3d::ResShadingModel* pShadingModel = pShaderArchive->FindShadingModel("basic");
    EXPECT_STREQ("basic", pShadingModel->GetName());
    void* ptr = GetResShaderFile();
    pShadingModel->SetUserPtr(ptr);
    EXPECT_TRUE(ptr == pShadingModel->GetUserPtr());
    EXPECT_TRUE(pShadingModel->GetGfxResShaderFile() != NULL);
    // オプション
    EXPECT_EQ(6, pShadingModel->GetStaticOptionCount());
    nn::g3d::ResShaderOption* pShaderOption = pShadingModel->FindStaticOption("normal_map");
    EXPECT_TRUE(pShaderOption != NULL);
    EXPECT_TRUE(pShaderOption == pShadingModel->GetStaticOption(pShadingModel->FindStaticOptionIndex("normal_map")));
    EXPECT_STREQ("normal_map", pShadingModel->GetStaticOptionName(pShadingModel->FindStaticOptionIndex("normal_map")));
    EXPECT_EQ(3, pShadingModel->GetDynamicOptionCount());
    pShaderOption = pShadingModel->FindDynamicOption("skinning");
    EXPECT_TRUE(pShaderOption != NULL);
    EXPECT_TRUE(pShaderOption == pShadingModel->GetDynamicOption(pShadingModel->FindDynamicOptionIndex("skinning")));
    EXPECT_STREQ("skinning", pShadingModel->GetDynamicOptionName(pShadingModel->FindDynamicOptionIndex("skinning")));

    // 頂点属性
    EXPECT_EQ(12, pShadingModel->GetAttrCount());
    nn::g3d::ResAttrVar* pAttrVar = pShadingModel->FindAttr("_t0");
    EXPECT_TRUE(pAttrVar != NULL);
    EXPECT_TRUE(pAttrVar == pShadingModel->GetAttr(pShadingModel->FindAttrIndex("_t0")));
    EXPECT_STREQ("_t0", pShadingModel->GetAttrName(pShadingModel->FindAttrIndex("_t0")));

    // サンプラー
    EXPECT_EQ(4, pShadingModel->GetSamplerCount());
    nn::g3d::ResSamplerVar* pSamplerVar = pShadingModel->FindSampler("_a1");
    EXPECT_TRUE(pSamplerVar != NULL);
    EXPECT_TRUE(pSamplerVar == pShadingModel->GetSampler(pShadingModel->FindSamplerIndex("_a1")));
    EXPECT_STREQ("_a1", pShadingModel->GetSamplerName(pShadingModel->FindSamplerIndex("_a1")));

    // ユニフォームブロック
    EXPECT_EQ(6, pShadingModel->GetUniformBlockCount());
    nn::g3d::ResUniformBlockVar* pUniformBlockVar = pShadingModel->FindUniformBlock("mat");
    EXPECT_TRUE(pUniformBlockVar != NULL);
    EXPECT_TRUE(pUniformBlockVar == pShadingModel->GetUniformBlock(pShadingModel->FindUniformBlockIndex("mat")));
    EXPECT_STREQ("mat", pShadingModel->GetUniformBlockName(pShadingModel->FindUniformBlockIndex("mat")));
    EXPECT_EQ(21, pShadingModel->GetUniformCount());
    EXPECT_EQ(pShadingModel->FindUniformBlockIndex("mat"), pShadingModel->GetSystemBlockIndex(nn::g3d::ResUniformBlockVar::Type_Material));
    EXPECT_TRUE(pShadingModel->GetUniform(10) != NULL);

    // シェーダプログラム
    EXPECT_EQ(1536, pShadingModel->GetShaderProgramCount());
    nn::g3d::ResShaderProgram* pProgram = pShadingModel->GetShaderProgram(1535);
    EXPECT_TRUE(pProgram != NULL);
    EXPECT_EQ(2, pShadingModel->GetKeyLength());
    EXPECT_EQ(1, pShadingModel->GetStaticKeyLength());
    EXPECT_EQ(1, pShadingModel->GetDynamicKeyLength());
    EXPECT_EQ(384, pShadingModel->GetDefaultProgramIndex());
    int defaultProgramIndex = pShadingModel->GetDefaultProgramIndex();
    const nn::Bit32* pKey = pShadingModel->GetKey(defaultProgramIndex);
    EXPECT_EQ(defaultProgramIndex, pShadingModel->FindProgramIndex(pKey));
    pProgram = pShadingModel->GetShaderProgram(defaultProgramIndex);
    EXPECT_EQ(defaultProgramIndex, pShadingModel->GetProgramIndex(pProgram));
    nn::Bit32 staticKey;
    nn::Bit32 dynamicKey;
    pShadingModel->WriteDefaultStaticKey(&staticKey);
    pShadingModel->WriteDefaultDynamicKey(&dynamicKey);
    EXPECT_EQ(pKey[0], staticKey);
    EXPECT_EQ(pKey[1], dynamicKey);
    nn::Bit32 invalidKey[2] = {0};
    pShadingModel->WriteInvalidDynamicKey(invalidKey);
    EXPECT_EQ(0xFFFFFFFF, invalidKey[0]);
    EXPECT_EQ(0, invalidKey[1]);
    const nn::Bit32* pStaticKey = pShadingModel->GetStaticKey(defaultProgramIndex);
    const nn::Bit32* pDynamicKey = pShadingModel->GetDynamicKey(defaultProgramIndex);
    EXPECT_EQ(staticKey, *pStaticKey);
    EXPECT_EQ(dynamicKey, *pDynamicKey);
    nn::g3d::ShaderRange range;
    EXPECT_TRUE(pShadingModel->FindProgramRange(&range, pStaticKey));
    EXPECT_EQ(defaultProgramIndex, pShadingModel->FindProgramIndex(range, pDynamicKey));

    // デバッグ
    const char* pExpectStr = "vertex_color:0\talbedo_tex:1\tmulti_tex:0\tnormal_map:0\tspec_mask:0\trim_light:0";
    char str[256];
    pShadingModel->PrintStaticOptionTo(str, sizeof(str), pStaticKey);
    EXPECT_STREQ(pExpectStr, str);
    pShadingModel->PrintDynamicOptionTo(str, sizeof(str), pDynamicKey);
    pExpectStr = "skinning:0\tcoord:c_Local\tsystem_id:0";
    EXPECT_STREQ(pExpectStr, str);
    nn::g3d::ResShadingModel::PrintKeyTo(str, sizeof(str), pStaticKey, pShadingModel->GetStaticKeyLength());
    pExpectStr = "40000000";
    EXPECT_STREQ(pExpectStr, str);
}

TEST_F(G3dTest, ResShadingProgram)
{
    nn::g3d::ResShaderArchive* pShaderArchive = GetResShaderFile()->GetResShaderArchive();
    nn::g3d::ResShadingModel* pShadingModel = pShaderArchive->FindShadingModel("basic");
    int index = pShadingModel->GetDefaultProgramIndex();
    nn::g3d::ResShaderProgram* pProgram = pShadingModel->GetShaderProgram(index);
    EXPECT_EQ(-1, pProgram->GetSamplerLocation(0, nn::g3d::Stage::Stage_Vertex));
    EXPECT_EQ(-1, pProgram->GetSamplerLocation(0, nn::g3d::Stage::Stage_Geometry));
    EXPECT_EQ(0, pProgram->GetSamplerLocation(0, nn::g3d::Stage::Stage_Pixel));
    EXPECT_EQ(-1, pProgram->GetSamplerLocation(0, nn::g3d::Stage::Stage_Compute));

#if defined(NN_BUILD_CONFIG_SPEC_NX)
    int blockIndex = pShadingModel->FindUniformBlockIndex("mat");
    EXPECT_EQ(3, pProgram->GetUniformBlockLocation(blockIndex, nn::g3d::Stage::Stage_Vertex));
    EXPECT_EQ(-1, pProgram->GetUniformBlockLocation(blockIndex, nn::g3d::Stage::Stage_Geometry));
    EXPECT_EQ(3, pProgram->GetUniformBlockLocation(blockIndex, nn::g3d::Stage::Stage_Pixel));
    EXPECT_EQ(-1, pProgram->GetUniformBlockLocation(blockIndex, nn::g3d::Stage::Stage_Compute));
    blockIndex = pShadingModel->FindUniformBlockIndex("opt");
    EXPECT_EQ(-1, pProgram->GetUniformBlockLocation(blockIndex, nn::g3d::Stage::Stage_Vertex));
    EXPECT_EQ(-1, pProgram->GetUniformBlockLocation(blockIndex, nn::g3d::Stage::Stage_Geometry));
    EXPECT_EQ(2, pProgram->GetUniformBlockLocation(blockIndex, nn::g3d::Stage::Stage_Pixel));
    EXPECT_EQ(-1, pProgram->GetUniformBlockLocation(blockIndex, nn::g3d::Stage::Stage_Compute));
#else
    int blockIndex = pShadingModel->FindUniformBlockIndex("mat");
    EXPECT_EQ(1, pProgram->GetUniformBlockLocation(blockIndex, nn::g3d::Stage::Stage_Pixel));
#endif
    pShadingModel->UpdateProgram(GetDevice(), index);
    EXPECT_TRUE(pProgram->GetShader() != NULL);
    EXPECT_TRUE(pProgram->IsAttrActive(pShadingModel->FindAttrIndex("_u0")));
#if defined(NN_BUILD_CONFIG_SPEC_NX)
    EXPECT_TRUE(pProgram->IsAttrActive(pShadingModel->FindAttrIndex("_t0")));
#else
    EXPECT_FALSE(pProgram->IsAttrActive(pShadingModel->FindAttrIndex("_t0")));
#endif
    EXPECT_TRUE(pProgram->GetShadingModel() == pShadingModel);
}

TEST_F(G3dTest, ResShaderInfo)
{
    nn::g3d::ResShaderArchive* pShaderArchive = GetResShaderFile()->GetResShaderArchive();
    nn::g3d::ResShadingModel* pShadingModel = pShaderArchive->FindShadingModel( "basic_ssbo_test" );
    const nn::g3d::ResShaderInfo* pShaderInfo = pShadingModel->GetShaderInfo();

    //
    // シンボルテーブルのアクセステストです。fsdbの要素順に配列に詰められています。
    //

    // sampler
    auto pSymbolNameArray = pShaderInfo->pSamplerTable.Get();
    EXPECT_STREQ( "albedoTex0", pSymbolNameArray[nn::g3d::Stage_Vertex].Get()->GetData() );
    EXPECT_STREQ( "", pSymbolNameArray[nn::g3d::Stage_Geometry].Get()->GetData() );
    EXPECT_STREQ( "albedoTex0", pSymbolNameArray[nn::g3d::Stage_Pixel].Get()->GetData() );
    EXPECT_STREQ( "", pSymbolNameArray[nn::g3d::Stage_Compute].Get()->GetData() );
    pSymbolNameArray += nn::g3d::Stage_End;
    EXPECT_STREQ( "albedoTex1", pSymbolNameArray[nn::g3d::Stage_Vertex].Get()->GetData() );
    EXPECT_STREQ( "", pSymbolNameArray[nn::g3d::Stage_Geometry].Get()->GetData() );
    EXPECT_STREQ( "albedoTex1", pSymbolNameArray[nn::g3d::Stage_Pixel].Get()->GetData() );
    EXPECT_STREQ( "", pSymbolNameArray[nn::g3d::Stage_Compute].Get()->GetData() );
    pSymbolNameArray += nn::g3d::Stage_End;
    EXPECT_STREQ( "normalTex", pSymbolNameArray[nn::g3d::Stage_Vertex].Get()->GetData() );
    EXPECT_STREQ( "", pSymbolNameArray[nn::g3d::Stage_Geometry].Get()->GetData() );
    EXPECT_STREQ( "normalTex", pSymbolNameArray[nn::g3d::Stage_Pixel].Get()->GetData() );
    EXPECT_STREQ( "", pSymbolNameArray[nn::g3d::Stage_Compute].Get()->GetData() );
    pSymbolNameArray += nn::g3d::Stage_End;
    EXPECT_STREQ( "specularTex", pSymbolNameArray[nn::g3d::Stage_Vertex].Get()->GetData() );
    EXPECT_STREQ( "", pSymbolNameArray[nn::g3d::Stage_Geometry].Get()->GetData() );
    EXPECT_STREQ( "specularTex", pSymbolNameArray[nn::g3d::Stage_Pixel].Get()->GetData() );
    EXPECT_STREQ( "", pSymbolNameArray[nn::g3d::Stage_Compute].Get()->GetData() );

    // ubo
    pSymbolNameArray = pShaderInfo->pUniformBlockTable.Get();
    EXPECT_STREQ( "Shape", pSymbolNameArray[nn::g3d::Stage_Vertex].Get()->GetData() );
    EXPECT_STREQ( "", pSymbolNameArray[nn::g3d::Stage_Geometry].Get()->GetData() );
    EXPECT_STREQ( "Shape", pSymbolNameArray[nn::g3d::Stage_Pixel].Get()->GetData() );
    EXPECT_STREQ( "", pSymbolNameArray[nn::g3d::Stage_Compute].Get()->GetData() );
    pSymbolNameArray += nn::g3d::Stage_End;
    EXPECT_STREQ( "View", pSymbolNameArray[nn::g3d::Stage_Vertex].Get()->GetData() );
    EXPECT_STREQ( "", pSymbolNameArray[nn::g3d::Stage_Geometry].Get()->GetData() );
    EXPECT_STREQ( "View", pSymbolNameArray[nn::g3d::Stage_Pixel].Get()->GetData() );
    EXPECT_STREQ( "", pSymbolNameArray[nn::g3d::Stage_Compute].Get()->GetData() );
    pSymbolNameArray += nn::g3d::Stage_End;
    EXPECT_STREQ( "Env", pSymbolNameArray[nn::g3d::Stage_Vertex].Get()->GetData() );
    EXPECT_STREQ( "", pSymbolNameArray[nn::g3d::Stage_Geometry].Get()->GetData() );
    EXPECT_STREQ( "Env", pSymbolNameArray[nn::g3d::Stage_Pixel].Get()->GetData() );
    EXPECT_STREQ( "", pSymbolNameArray[nn::g3d::Stage_Compute].Get()->GetData() );
    pSymbolNameArray += nn::g3d::Stage_End;
    EXPECT_STREQ( "Option", pSymbolNameArray[nn::g3d::Stage_Vertex].Get()->GetData() );
    EXPECT_STREQ( "", pSymbolNameArray[nn::g3d::Stage_Geometry].Get()->GetData() );
    EXPECT_STREQ( "Option", pSymbolNameArray[nn::g3d::Stage_Pixel].Get()->GetData() );
    EXPECT_STREQ( "", pSymbolNameArray[nn::g3d::Stage_Compute].Get()->GetData() );
    pSymbolNameArray += nn::g3d::Stage_End;
    EXPECT_STREQ( "Material", pSymbolNameArray[nn::g3d::Stage_Vertex].Get()->GetData() );
    EXPECT_STREQ( "", pSymbolNameArray[nn::g3d::Stage_Geometry].Get()->GetData() );
    EXPECT_STREQ( "Material", pSymbolNameArray[nn::g3d::Stage_Pixel].Get()->GetData() );
    EXPECT_STREQ( "", pSymbolNameArray[nn::g3d::Stage_Compute].Get()->GetData() );

    // ssbo
    pSymbolNameArray = pShaderInfo->pShaderStorageBlockTable.Get();
    EXPECT_STREQ( "Skeleton", pSymbolNameArray[nn::g3d::Stage_Vertex].Get()->GetData() );
    EXPECT_STREQ( "", pSymbolNameArray[nn::g3d::Stage_Geometry].Get()->GetData() );
    EXPECT_STREQ( "Skeleton", pSymbolNameArray[nn::g3d::Stage_Pixel].Get()->GetData() );
    EXPECT_STREQ( "", pSymbolNameArray[nn::g3d::Stage_Compute].Get()->GetData() );
    pSymbolNameArray += nn::g3d::Stage_End;
    EXPECT_STREQ( "DummyBuffer0", pSymbolNameArray[nn::g3d::Stage_Vertex].Get()->GetData() );
    EXPECT_STREQ( "", pSymbolNameArray[nn::g3d::Stage_Geometry].Get()->GetData() );
    EXPECT_STREQ( "DummyBuffer0", pSymbolNameArray[nn::g3d::Stage_Pixel].Get()->GetData() );
    EXPECT_STREQ( "", pSymbolNameArray[nn::g3d::Stage_Compute].Get()->GetData() );
    pSymbolNameArray += nn::g3d::Stage_End;
    EXPECT_STREQ( "DummyBuffer1", pSymbolNameArray[nn::g3d::Stage_Vertex].Get()->GetData() );
    EXPECT_STREQ( "", pSymbolNameArray[nn::g3d::Stage_Geometry].Get()->GetData() );
    EXPECT_STREQ( "DummyBuffer1", pSymbolNameArray[nn::g3d::Stage_Pixel].Get()->GetData() );
    EXPECT_STREQ( "", pSymbolNameArray[nn::g3d::Stage_Compute].Get()->GetData() );

    // streamout
    pSymbolNameArray = pShaderInfo->pStreamout.Get();
    EXPECT_EQ( nullptr, pSymbolNameArray );    // 未使用の時はnullptr
}

TEST_F(G3dTest, ResUniformBlockVar)
{
    nn::g3d::ResShaderArchive* pShaderArchive = GetResShaderFile()->GetResShaderArchive();
    nn::g3d::ResShadingModel* pShadingModel = pShaderArchive->FindShadingModel("basic");
    // ユニフォームブロック
    nn::g3d::ResUniformBlockVar* pUniformBlockVar = pShadingModel->FindUniformBlock("mat");
    EXPECT_EQ(nn::g3d::ResUniformBlockVar::Type_Material, pUniformBlockVar->GetType());
    EXPECT_EQ(pShadingModel->FindUniformBlockIndex("mat"), pUniformBlockVar->GetIndex());
    EXPECT_EQ(11, pUniformBlockVar->GetUniformCount());
    EXPECT_TRUE(pUniformBlockVar == pShadingModel->GetUniformBlock(pShadingModel->FindUniformBlockIndex("mat")));
    EXPECT_STREQ("texsrt2", pUniformBlockVar->GetUniformName(pUniformBlockVar->FindUniformIndex("texsrt2")));
    nn::util::BytePtr pDefaultValue(const_cast<void*>(pUniformBlockVar->GetDefault()));
    EXPECT_TRUE(pDefaultValue.Get() != NULL);
    // ユニフォーム
    nn::g3d::ResUniformVar* pUniformVar = pShadingModel->GetUniform(10);
    EXPECT_EQ(pUniformVar->GetIndex(), 10);
    pUniformVar = pUniformBlockVar->FindUniform("spec_int");
    EXPECT_EQ(pUniformVar->GetBlockIndex(), pShadingModel->FindUniformBlockIndex("mat"));
    pDefaultValue.Advance(pUniformVar->GetOffset());
    EXPECT_FLOAT_EQ(0.2f, *(pDefaultValue.Get<float>()));
}

TEST_F(G3dTest, ResShaderStorageBlockVar)
{
    nn::g3d::ResShaderArchive* pShaderArchive = GetResShaderFile()->GetResShaderArchive();
    nn::g3d::ResShadingModel* pShadingModel = pShaderArchive->FindShadingModel("basic_ssbo_test");

    EXPECT_EQ(3, pShadingModel->GetShaderStorageBlockCount());
    EXPECT_EQ(pShadingModel->FindShaderStorageBlockIndex("skel"), pShadingModel->GetSystemShaderStorageBlockIndex(nn::g3d::ResShaderStorageBlockVar::Type_Skeleton));
    EXPECT_TRUE(pShadingModel->GetShaderStorage(1) != NULL);

    // SSBO type=none
    nn::g3d::ResShaderStorageBlockVar* pShaderStorageBlockVar = pShadingModel->FindShaderStorageBlock("dummy_buffer0");
    int bufferIdx = pShaderStorageBlockVar->GetIndex();
    EXPECT_EQ(pShadingModel->FindShaderStorageBlockIndex("dummy_buffer0"), bufferIdx);
    EXPECT_EQ(nn::g3d::ResShaderStorageBlockVar::Type_None, pShaderStorageBlockVar->GetType());
    EXPECT_EQ(2, pShaderStorageBlockVar->GetShaderStorageCount());
    EXPECT_EQ(32, pShaderStorageBlockVar->GetSize());
    EXPECT_TRUE(pShaderStorageBlockVar == pShadingModel->GetShaderStorageBlock(pShadingModel->FindShaderStorageBlockIndex("dummy_buffer0")));
    EXPECT_STREQ("dummy_buffer0_val0", pShaderStorageBlockVar->GetShaderStorageName(pShaderStorageBlockVar->FindShaderStorageIndex("dummy_buffer0_val0")));
    nn::util::BytePtr pDefaultValue(const_cast<void*>(pShaderStorageBlockVar->GetDefault()));

    // SsboVar
    nn::g3d::ResShaderStorageVar* pShaderStorageVar = pShadingModel->GetShaderStorage(2);
    EXPECT_EQ(pShaderStorageVar->GetIndex(), 2);
    pShaderStorageVar = pShaderStorageBlockVar->FindShaderStorage("dummy_buffer0_val1");
    EXPECT_EQ(pShaderStorageVar->GetBlockIndex(), pShadingModel->FindShaderStorageBlockIndex("dummy_buffer0"));
    EXPECT_EQ(16, pShaderStorageVar->GetOffset());    // DummyBuffer0{ float dummyBuffer0Val0; vec4 dummyBuffer0Val1; }なのでオフセットはパディングが入り、16が正解値

    // SSBO type=skeleton
    pShaderStorageBlockVar = pShadingModel->FindShaderStorageBlock("skel");
    EXPECT_EQ(nn::g3d::ResShaderStorageBlockVar::Type_Skeleton, pShaderStorageBlockVar->GetType());
    EXPECT_EQ(pShadingModel->FindShaderStorageBlockIndex("skel"), pShaderStorageBlockVar->GetIndex());
    EXPECT_EQ(0, pShaderStorageBlockVar->GetShaderStorageCount());    // アノテーションをメンバに振っていないので0が正解値
    EXPECT_TRUE(pShaderStorageBlockVar == pShadingModel->GetShaderStorageBlock(pShadingModel->FindShaderStorageBlockIndex("skel")));
    EXPECT_EQ(2048, pShaderStorageBlockVar->GetSize());        // 任意長配列の場合はシェーダで利用可能な最大サイズが入る。リフレクションでは2048が返っているのでとりあえずこれを正解値とする。

    // ResShaderProgram ShaderStorageBlock インターフェーステスト
    nn::g3d::ResShaderProgram* pProgram = pShadingModel->GetShaderProgram(0);
    EXPECT_EQ( 2, pProgram->GetShaderStorageBlockLocation( pShadingModel->FindShaderStorageBlockIndex("skel"), nn::g3d::Stage_Vertex ) );    // "skel"のみlayoutで2を指定している。
    EXPECT_EQ( 0, pProgram->GetShaderStorageBlockLocation( pShadingModel->FindShaderStorageBlockIndex("dummy_buffer0"), nn::g3d::Stage_Vertex ) );
    EXPECT_EQ( 1, pProgram->GetShaderStorageBlockLocation( pShadingModel->FindShaderStorageBlockIndex("dummy_buffer1"), nn::g3d::Stage_Vertex ) );
}

TEST_F(G3dTest, ResSamplerVar)
{
    nn::g3d::ResShaderArchive* pShaderArchive = GetResShaderFile()->GetResShaderArchive();
    nn::g3d::ResShadingModel* pShadingModel = pShaderArchive->FindShadingModel("basic");
    nn::g3d::ResSamplerVar* pSamplerVar = pShadingModel->FindSampler("_a1");
    EXPECT_STREQ("_a0", pSamplerVar->GetAlt());
    EXPECT_EQ(pSamplerVar->GetIndex(), pShadingModel->FindSamplerIndex("_a1"));
}

TEST_F(G3dTest, ResShaderOption)
{
    // ブランチなしシェーダアーカイブ
    nn::g3d::ResShaderArchive* pShaderArchive = GetResShaderFile()->GetResShaderArchive();
    nn::g3d::ResShadingModel* pShadingModel = pShaderArchive->FindShadingModel("basic");
    {
        nn::g3d::ResShaderOption* pOption = pShadingModel->FindStaticOption("multi_tex");
        EXPECT_EQ(4, pOption->GetChoiceCount());
        EXPECT_EQ(0, pOption->GetDefaultIndex());
        EXPECT_EQ(-1, pOption->GetBranchOffset());
        EXPECT_STREQ("multi_tex", pOption->GetName());
        EXPECT_EQ(2, pOption->FindChoiceIndex("2"));
        EXPECT_STREQ("2", pOption->GetChoiceName(2));
        EXPECT_TRUE(pOption->IsStatic());
        EXPECT_FALSE(pOption->IsDynamic());
        EXPECT_FALSE(pOption->IsBranch());
    }
    {
        nn::g3d::ResShaderOption* pOption = pShadingModel->FindDynamicOption("coord");
        EXPECT_EQ(2, pOption->GetChoiceCount());
        EXPECT_EQ(0, pOption->GetDefaultIndex());
        EXPECT_EQ(-1, pOption->GetBranchOffset());
        EXPECT_STREQ("coord", pOption->GetName());
        EXPECT_EQ(1, pOption->FindChoiceIndex("c_World"));
        EXPECT_STREQ("c_World", pOption->GetChoiceName(1));
        EXPECT_FALSE(pOption->IsStatic());
        EXPECT_TRUE(pOption->IsDynamic());
        EXPECT_FALSE(pOption->IsBranch());
    }

    // ブランチありシェーダアーカイブ
    pShaderArchive = GetResShaderFile(1)->GetResShaderArchive();
    pShadingModel = pShaderArchive->FindShadingModel("basic");
    {
        nn::g3d::ResShaderOption* pOption = pShadingModel->FindStaticOption("multi_tex");
        EXPECT_EQ(4, pOption->GetChoiceCount());
        EXPECT_EQ(0, pOption->GetDefaultIndex());
        EXPECT_EQ(8, pOption->GetBranchOffset());
        EXPECT_STREQ("multi_tex", pOption->GetName());
        EXPECT_EQ(2, pOption->FindChoiceIndex("2"));
        EXPECT_STREQ("2", pOption->GetChoiceName(2));
        EXPECT_TRUE(pOption->IsStatic());
        EXPECT_FALSE(pOption->IsDynamic());
        EXPECT_TRUE(pOption->IsBranch());
    }
    {
        nn::g3d::ResShaderOption* pOption = pShadingModel->FindDynamicOption("coord");
        EXPECT_EQ(2, pOption->GetChoiceCount());
        EXPECT_EQ(0, pOption->GetDefaultIndex());
        EXPECT_EQ(-1, pOption->GetBranchOffset());
        EXPECT_STREQ("coord", pOption->GetName());
        EXPECT_EQ(1, pOption->FindChoiceIndex("c_World"));
        EXPECT_STREQ("c_World", pOption->GetChoiceName(1));
        EXPECT_FALSE(pOption->IsStatic());
        EXPECT_TRUE(pOption->IsDynamic());
        EXPECT_TRUE(pOption->IsBranch());
    }

    // シェーダキーの書き込みテスト
    {
        nn::Bit32 key = 0;
        nn::g3d::ResShaderOption* pOption = pShadingModel->FindStaticOption("vertex_color");
        pOption->WriteStaticKey(&key, 1);
        EXPECT_EQ(1, pOption->ReadStaticKey(&key));
        pOption = pShadingModel->FindStaticOption("albedo_tex");
        pOption->WriteStaticKey(&key, 0);
        EXPECT_EQ(0, pOption->ReadStaticKey(&key));
        pOption = pShadingModel->FindStaticOption("multi_tex");
        pOption->WriteStaticKey(&key, 3);
        EXPECT_EQ(3, pOption->ReadStaticKey(&key));
        EXPECT_EQ(0xb0000000, key);
        pOption = pShadingModel->FindDynamicOption("skinning");
        pOption->WriteDynamicKey(&key, 4);
        EXPECT_EQ(4, pOption->ReadDynamicKey(&key));
        pOption = pShadingModel->FindDynamicOption("coord");
        pOption->WriteDynamicKey(&key, 1);
        EXPECT_EQ(1, pOption->ReadDynamicKey(&key));
        EXPECT_EQ(0x90000000, key);
    }
}

namespace
{
// ShadingModelObjの必要バッファサイズ計算
size_t CalculateBuildShadingModelObjBufferSize(nn::g3d::ResShadingModel* pResShadingModel, int shadingModelObjBufferingCount)
{
    nn::util::MemorySplitter::MemoryBlock memoryBlock;
    size_t defaultAlignment = memoryBlock.GetAlignment();

    size_t size = pResShadingModel->GetStaticKeyLength() * sizeof(nn::Bit32);
    size = nn::util::align_up(size, defaultAlignment);
    size += pResShadingModel->GetStaticKeyLength() * sizeof(nn::Bit32);
    size = nn::util::align_up(size, defaultAlignment);
    size += nn::g3d::FlagSet::CalcBufferSize(pResShadingModel->GetStaticOptionCount(), shadingModelObjBufferingCount);
    size += sizeof(nn::gfx::Buffer) * shadingModelObjBufferingCount;
    size = nn::util::align_up(size, defaultAlignment);
    return size;
}
}

TEST_F(G3dTest, ShadingModelObj)
{
    const int shadingModelObjBufferingCount = 2;

    nn::gfx::Device* pDevice = GetDevice();
    // ブランチなしシェーダアーカイブ
    {
        nn::g3d::ResShaderArchive* pShaderArchive = GetResShaderFile()->GetResShaderArchive();
        nn::g3d::ResShadingModel* pResShadingModel = pShaderArchive->FindShadingModel("basic");
        nn::g3d::ShadingModelObj::Builder builder(pResShadingModel);
        builder.BufferingCount(shadingModelObjBufferingCount);
        builder.CalculateMemorySize();

        size_t size = builder.GetWorkMemorySize();
        size_t expectedSize = CalculateBuildShadingModelObjBufferSize(pResShadingModel, shadingModelObjBufferingCount);
        EXPECT_EQ(expectedSize, size);
        SimplePtr buffer(nnt::g3d::AlignedAllocate<void>(size, nn::g3d::ShadingModelObj::Alignment_Buffer));
        nn::g3d::ShadingModelObj shadingModelObj;
        builder.Build(&shadingModelObj, buffer.Get(), size);
        EXPECT_TRUE(buffer.Get() == shadingModelObj.GetBufferPtr());
        EXPECT_EQ(shadingModelObjBufferingCount, shadingModelObj.GetBufferingCount());
        size_t blockBufferSize = shadingModelObj.CalculateBlockBufferSize(pDevice);
        const nn::g3d::ResUniformBlockVar* pBlock = pResShadingModel->FindUniformBlock("opt");
        EXPECT_EQ(blockBufferSize, 0);
        ptrdiff_t offset = AllocateMemoryPool(blockBufferSize, shadingModelObj.GetBlockBufferAlignment(pDevice));
        EXPECT_TRUE(shadingModelObj.SetupBlockBuffer(pDevice, GetWriteCombineMemoryPool(), offset, blockBufferSize));
        EXPECT_FALSE(shadingModelObj.IsBlockBufferValid());
        EXPECT_FALSE(GetWriteCombineMemoryPool() == shadingModelObj.GetMemoryPoolPtr());
        EXPECT_EQ(offset, shadingModelObj.GetMemoryPoolOffset());
        EXPECT_EQ(pBlock->GetSize(), shadingModelObj.GetOptionBlockSize());
        shadingModelObj.SetUserPtr(pShaderArchive);
        EXPECT_TRUE(pShaderArchive == shadingModelObj.GetUserPtr<nn::g3d::ResShaderArchive>());
        // オプション関連チェック
        EXPECT_EQ(6, shadingModelObj.GetStaticOptionCount());
        EXPECT_EQ(1, shadingModelObj.GetStaticKeyLength());
        const nn::g3d::ResShaderOption* pResShaderOption = shadingModelObj.FindStaticOption("multi_tex");
        EXPECT_TRUE(pResShaderOption == shadingModelObj.GetStaticOption(shadingModelObj.FindStaticOptionIndex("multi_tex")));
        EXPECT_STREQ("multi_tex", shadingModelObj.GetStaticOptionName(shadingModelObj.FindStaticOptionIndex("multi_tex")));
        // キー関連チェック
        const nn::Bit32* pDefaultKey = pResShadingModel->GetKey(pResShadingModel->GetDefaultProgramIndex());
        const nn::Bit32* pStaticKey = shadingModelObj.GetStaticKey();
        EXPECT_EQ(*pDefaultKey, *pStaticKey);
        shadingModelObj.WriteStaticKey(shadingModelObj.FindStaticOptionIndex("multi_tex"), 3);
        EXPECT_EQ(3, shadingModelObj.ReadStaticKey(shadingModelObj.FindStaticOptionIndex("multi_tex")));
        pStaticKey = shadingModelObj.GetStaticKey();
        EXPECT_NE(*pDefaultKey, *pStaticKey);
        shadingModelObj.ClearStaticKey();
        EXPECT_EQ(*pDefaultKey, *pStaticKey);
        // プログラム範囲関連チェック
        EXPECT_TRUE(shadingModelObj.UpdateShaderRange());
        const nn::g3d::ShaderRange& shaderRange = shadingModelObj.GetShaderRange();
        nn::g3d::ShaderRange defaultRange;
        pResShadingModel->FindProgramRange(&defaultRange, pDefaultKey);
        EXPECT_TRUE(memcmp(defaultRange.pBegin, shaderRange.pBegin, 2) == 0);
        EXPECT_TRUE(memcmp(defaultRange.pEnd, shaderRange.pEnd, 2) == 0);
        {
            shadingModelObj.WriteStaticKey(shadingModelObj.FindStaticOptionIndex("multi_tex"), 3);
            for (int idxBuffer = 0; idxBuffer < shadingModelObjBufferingCount; ++idxBuffer)
            {
                shadingModelObj.CalculateOptionBlock(idxBuffer);
            }
        }

        // デバッグ
        char str[256];
        shadingModelObj.PrintKeyTo(str, sizeof(str));
        const char* pExpectStr = "70000000";
        EXPECT_STREQ(pExpectStr, str);
        shadingModelObj.PrintOptionTo(str, sizeof(str));
        pExpectStr = "vertex_color:0\talbedo_tex:1\tmulti_tex:3\tnormal_map:0\tspec_mask:0\trim_light:0";
        EXPECT_STREQ(pExpectStr, str);
        shadingModelObj.PrintRawKeyTo(str, sizeof(str));
        pExpectStr = "70000000";
        EXPECT_STREQ(pExpectStr, str);
        shadingModelObj.PrintRawOptionTo(str, sizeof(str));
        pExpectStr = "vertex_color:0\talbedo_tex:1\tmulti_tex:3\tnormal_map:0\tspec_mask:0\trim_light:0";
        EXPECT_STREQ(pExpectStr, str);
        FreeMemoryPool();
    }
    // ブランチありシェーダアーカイブ
    {
        nn::g3d::ResShaderArchive* pShaderArchive = GetResShaderFile(1)->GetResShaderArchive();
        nn::g3d::ResShadingModel* pResShadingModel = pShaderArchive->FindShadingModel("basic");
        nn::g3d::ShadingModelObj::Builder builder(pResShadingModel);
        builder.BufferingCount(shadingModelObjBufferingCount);
        builder.CalculateMemorySize();

        size_t size = builder.GetWorkMemorySize();
        SimplePtr buffer(nnt::g3d::AlignedAllocate<void>(size, nn::g3d::ShadingModelObj::Alignment_Buffer));
        nn::g3d::ShadingModelObj shadingModelObj;
        builder.Build(&shadingModelObj, buffer.Get(), size);
        EXPECT_TRUE(buffer.Get() == shadingModelObj.GetBufferPtr());
        EXPECT_EQ(shadingModelObjBufferingCount, shadingModelObj.GetBufferingCount());
        size_t blockBufferSize = shadingModelObj.CalculateBlockBufferSize(pDevice);
        const nn::g3d::ResUniformBlockVar* pBlock = pResShadingModel->FindUniformBlock("opt");
        size_t expectBlockBufferSize = nn::util::align_up(pBlock->GetSize(), shadingModelObj.GetBlockBufferAlignment(pDevice)) * shadingModelObj.GetBufferingCount();
        EXPECT_EQ(blockBufferSize, expectBlockBufferSize);
        ptrdiff_t offset = AllocateMemoryPool(blockBufferSize, shadingModelObj.GetBlockBufferAlignment(pDevice));
        EXPECT_TRUE(shadingModelObj.SetupBlockBuffer(pDevice, GetWriteCombineMemoryPool(), offset, blockBufferSize));
        EXPECT_TRUE(shadingModelObj.IsBlockBufferValid());
        EXPECT_TRUE(GetWriteCombineMemoryPool() == shadingModelObj.GetMemoryPoolPtr());
        EXPECT_EQ(offset, shadingModelObj.GetMemoryPoolOffset());
        EXPECT_EQ(pBlock->GetSize(), shadingModelObj.GetOptionBlockSize());
        shadingModelObj.SetUserPtr(pShaderArchive);
        EXPECT_TRUE(pShaderArchive == shadingModelObj.GetUserPtr<nn::g3d::ResShaderArchive>());
        // オプション関連チェック
        EXPECT_EQ(6, shadingModelObj.GetStaticOptionCount());
        EXPECT_EQ(1, shadingModelObj.GetStaticKeyLength());
        const nn::g3d::ResShaderOption* pResShaderOption = shadingModelObj.FindStaticOption("multi_tex");
        EXPECT_TRUE(pResShaderOption == shadingModelObj.GetStaticOption(shadingModelObj.FindStaticOptionIndex("multi_tex")));
        EXPECT_STREQ("multi_tex", shadingModelObj.GetStaticOptionName(shadingModelObj.FindStaticOptionIndex("multi_tex")));
        // キー関連チェック
        const nn::Bit32* pDefaultKey = pResShadingModel->GetKey(pResShadingModel->GetDefaultProgramIndex());
        const nn::Bit32* pStaticKey = shadingModelObj.GetStaticKey();
        EXPECT_EQ(*pDefaultKey, *pStaticKey);
        shadingModelObj.WriteStaticKey(shadingModelObj.FindStaticOptionIndex("multi_tex"), 3);
        EXPECT_EQ(3, shadingModelObj.ReadStaticKey(shadingModelObj.FindStaticOptionIndex("multi_tex")));
        pStaticKey = shadingModelObj.GetStaticKey();
        EXPECT_EQ(*pDefaultKey, *pStaticKey);
        shadingModelObj.ClearStaticKey();
        EXPECT_EQ(*pDefaultKey, *pStaticKey);
        // プログラム範囲関連チェック
        EXPECT_TRUE(shadingModelObj.UpdateShaderRange());
        const nn::g3d::ShaderRange& shaderRange = shadingModelObj.GetShaderRange();
        nn::g3d::ShaderRange defaultRange;
        pResShadingModel->FindProgramRange(&defaultRange, pDefaultKey);
        EXPECT_TRUE(memcmp(defaultRange.pBegin, shaderRange.pBegin, 2) == 0);
        EXPECT_TRUE(memcmp(defaultRange.pEnd, shaderRange.pEnd, 2) == 0);

        // オプションブロックが更新されることを確認
        size_t optionSize = shadingModelObj.GetOptionBlockSize();
        SimplePtr saveBuffer(nnt::g3d::Allocate(optionSize));
        {
            for (int idxBuffer = 0; idxBuffer < shadingModelObjBufferingCount; ++idxBuffer)
            {
                nn::gfx::Buffer* pOptionBlock = shadingModelObj.GetOptionBlock(idxBuffer);
                nn::util::BytePtr bytePtr(pOptionBlock->Map<int>());
                bytePtr.Advance(pResShadingModel->FindUniformBlock("opt")->FindUniform("multi_tex")->GetOffset());
                EXPECT_EQ(0, *bytePtr.Get<int>());
                pOptionBlock->Unmap();
            }
        }
        {
            shadingModelObj.WriteStaticKey(shadingModelObj.FindStaticOptionIndex("multi_tex"), 3);
            for (int idxBuffer = 0; idxBuffer < shadingModelObjBufferingCount; ++idxBuffer)
            {
                shadingModelObj.CalculateOptionBlock(idxBuffer);

                nn::gfx::Buffer* pOptionBlock = shadingModelObj.GetOptionBlock(idxBuffer);
                nn::util::BytePtr bytePtr(pOptionBlock->Map<int>());
                bytePtr.Advance(pResShadingModel->FindUniformBlock("opt")->FindUniform("multi_tex")->GetOffset());
                EXPECT_EQ(3, *bytePtr.Get<int>());
                pOptionBlock->Unmap();
            }
        }

        // デバッグ
        char str[256];
        shadingModelObj.PrintKeyTo(str, sizeof(str));
        const char* pExpectStr = "70000000";
        EXPECT_STREQ(pExpectStr, str);
        shadingModelObj.PrintOptionTo(str, sizeof(str));
        pExpectStr = "vertex_color:0\talbedo_tex:1\tmulti_tex:3\tnormal_map:0\tspec_mask:0\trim_light:0";
        EXPECT_STREQ(pExpectStr, str);
        shadingModelObj.PrintRawKeyTo(str, sizeof(str));
        pExpectStr = "40000000";
        EXPECT_STREQ(pExpectStr, str);
        shadingModelObj.PrintRawOptionTo(str, sizeof(str));
        pExpectStr = "vertex_color:0\talbedo_tex:1\tmulti_tex:0\tnormal_map:0\tspec_mask:0\trim_light:0";
        EXPECT_STREQ(pExpectStr, str);

        shadingModelObj.CleanupBlockBuffer(pDevice);
        FreeMemoryPool();
    }
} // NOLINT

namespace
{
// ShaderSelectorの必要バッファサイズ計算
size_t CalculateBuildShaderSelectorBufferSize(nn::g3d::ResShadingModel* pResShadingModel)
{
    nn::util::MemorySplitter::MemoryBlock memoryBlock;
    size_t defaultAlignment = memoryBlock.GetAlignment();

    size_t size = pResShadingModel->GetDynamicKeyLength() * sizeof(nn::Bit32);
    size = nn::util::align_up(size, defaultAlignment);
    size += pResShadingModel->GetDynamicKeyLength() * sizeof(nn::Bit32);
    size = nn::util::align_up(size, defaultAlignment);
    size += pResShadingModel->GetDynamicKeyLength() * sizeof(nn::Bit32);
    return size;
}
}
TEST_F(G3dTest, ShaderSelector)
{
    nn::gfx::Device* pDevice = GetDevice();
    // ブランチなしシェーダアーカイブ
    {
        nn::g3d::ResShaderArchive* pShaderArchive = GetResShaderFile()->GetResShaderArchive();
        nn::g3d::ResShadingModel* pResShadingModel = pShaderArchive->FindShadingModel("basic");
        nn::g3d::ShadingModelObj::Builder shadingModelBuilder(pResShadingModel);
        shadingModelBuilder.CalculateMemorySize();
        // ShadingModelObj初期化
        size_t shadingModelBufferSize = shadingModelBuilder.GetWorkMemorySize();
        SimplePtr shadingModelBuffer(nnt::g3d::AlignedAllocate<void>(shadingModelBufferSize, nn::g3d::ShadingModelObj::Alignment_Buffer));
        nn::g3d::ShadingModelObj shadingModelObj;
        shadingModelBuilder.Build(&shadingModelObj, shadingModelBuffer.Get(), shadingModelBufferSize);
        size_t blockBufferSize = shadingModelObj.CalculateBlockBufferSize(pDevice);
        EXPECT_EQ(blockBufferSize, 0);
        ptrdiff_t offset = AllocateMemoryPool(blockBufferSize, shadingModelObj.GetBlockBufferAlignment(pDevice));
        EXPECT_TRUE(shadingModelObj.SetupBlockBuffer(pDevice, GetWriteCombineMemoryPool(), offset, blockBufferSize));
        EXPECT_FALSE(shadingModelObj.IsBlockBufferValid());
        EXPECT_FALSE(GetWriteCombineMemoryPool() == shadingModelObj.GetMemoryPoolPtr());
        EXPECT_EQ(offset, shadingModelObj.GetMemoryPoolOffset());
        shadingModelObj.UpdateShaderRange();

        // ShaderSelector初期化
        nn::g3d::ShaderSelector::Builder shaderSelectorBuilder(&shadingModelObj);
        nn::g3d::ShaderSelector shaderSelector;
        shaderSelectorBuilder.CalculateMemorySize();
        size_t shaderSelectorBufferSize = shaderSelectorBuilder.GetWorkMemorySize();
        size_t expectSize = CalculateBuildShaderSelectorBufferSize(pResShadingModel);
        EXPECT_EQ(expectSize, shaderSelectorBufferSize);
        SimplePtr shaderSelectorBuffer(nnt::g3d::AlignedAllocate<void>(shaderSelectorBufferSize, nn::g3d::ShaderSelector::Alignment_Buffer));
        shaderSelectorBuilder.Build(&shaderSelector, shaderSelectorBuffer.Get(), shaderSelectorBufferSize);
        EXPECT_EQ(shaderSelectorBuffer.Get(), shaderSelector.GetBufferPtr());
        EXPECT_EQ(2 + 1/*システム用*/, shaderSelector.GetDynamicOptionCount());
        EXPECT_EQ(1, shaderSelector.GetDynamicKeyLength());
        EXPECT_TRUE(shaderSelector.FindDynamicOption("coord") == shaderSelector.GetDynamicOption(shaderSelector.FindDynamicOptionIndex("coord")));
        EXPECT_STREQ("coord", shaderSelector.GetDynamicOptionName(shaderSelector.FindDynamicOptionIndex("coord")));

        const nn::g3d::ResShaderProgram* pDefaultProgram = shaderSelector.GetDefaultProgram();
        {
            EXPECT_TRUE(shaderSelector.UpdateVariation(pDevice));
            const nn::g3d::ResShaderProgram* pProgram = shaderSelector.GetProgram();
            EXPECT_EQ(pDefaultProgram, pProgram);
        }
        EXPECT_EQ(0, shaderSelector.ReadDynamicKey(shaderSelector.FindDynamicOptionIndex("coord")));
        shaderSelector.WriteDynamicKey(shaderSelector.FindDynamicOptionIndex("coord"), 1);
        EXPECT_EQ(1, shaderSelector.ReadDynamicKey(shaderSelector.FindDynamicOptionIndex("coord")));
        const nn::Bit32* pDynamicKey = shaderSelector.GetDynamicKey();
        EXPECT_EQ(0x10000000, *pDynamicKey);
        {
            EXPECT_TRUE(shaderSelector.UpdateVariation(pDevice));
            const nn::g3d::ResShaderProgram* pProgram = shaderSelector.GetProgram();
            EXPECT_NE(pDefaultProgram, pProgram);
        }
        shaderSelector.ClearDynamicKey();
        EXPECT_EQ(0, shaderSelector.ReadDynamicKey(shaderSelector.FindDynamicOptionIndex("coord")));
        {
            EXPECT_TRUE(shaderSelector.UpdateVariation(pDevice));
            const nn::g3d::ResShaderProgram* pProgram = shaderSelector.GetProgram();
            EXPECT_EQ(pDefaultProgram, pProgram);
        }
        EXPECT_EQ(&shadingModelObj, shaderSelector.GetShadingModel());

        shaderSelector.WriteDynamicKey(shaderSelector.FindDynamicOptionIndex("coord"), 1);
        char str[256];
        shaderSelector.PrintKeyTo(str, sizeof(str));
        const char* pExpectStr = "10000000";
        EXPECT_STREQ(pExpectStr, str);
        shaderSelector.PrintRawKeyTo(str, sizeof(str));
        pExpectStr = "10000000";
        EXPECT_STREQ(pExpectStr, str);
        shaderSelector.PrintOptionTo(str, sizeof(str));
        pExpectStr = "skinning:0\tcoord:c_World\tsystem_id:0";
        EXPECT_STREQ(pExpectStr, str);
        shaderSelector.PrintRawOptionTo(str, sizeof(str));
        pExpectStr = "skinning:0\tcoord:c_World\tsystem_id:0";
        EXPECT_STREQ(pExpectStr, str);

        FreeMemoryPool();
    }
    // ブランチありシェーダアーカイブ
    {
        nn::g3d::ResShaderArchive* pShaderArchive = GetResShaderFile(1)->GetResShaderArchive();
        nn::g3d::ResShadingModel* pResShadingModel = pShaderArchive->FindShadingModel("basic");
        nn::g3d::ShadingModelObj::Builder shadingModelBuilder(pResShadingModel);
        shadingModelBuilder.CalculateMemorySize();
        // ShadingModelObj初期化
        size_t shadingModelBufferSize = shadingModelBuilder.GetWorkMemorySize();
        SimplePtr shadingModelBuffer(nnt::g3d::AlignedAllocate<void>(shadingModelBufferSize, nn::g3d::ShadingModelObj::Alignment_Buffer));
        nn::g3d::ShadingModelObj shadingModelObj;
        shadingModelBuilder.Build(&shadingModelObj, shadingModelBuffer.Get(), shadingModelBufferSize);
        size_t blockBufferSize = shadingModelObj.CalculateBlockBufferSize(pDevice);
        ptrdiff_t offset = AllocateMemoryPool(blockBufferSize, shadingModelObj.GetBlockBufferAlignment(pDevice));
        EXPECT_TRUE(shadingModelObj.SetupBlockBuffer(pDevice, GetWriteCombineMemoryPool(), offset, blockBufferSize));
        EXPECT_TRUE(shadingModelObj.IsBlockBufferValid());
        shadingModelObj.UpdateShaderRange();

        // ShaderSelector初期化
        nn::g3d::ShaderSelector::Builder shaderSelectorBuilder(&shadingModelObj);
        nn::g3d::ShaderSelector shaderSelector;
        shaderSelectorBuilder.CalculateMemorySize();
        size_t shaderSelectorBufferSize = shaderSelectorBuilder.GetWorkMemorySize();
        size_t expectSize = CalculateBuildShaderSelectorBufferSize(pResShadingModel);
        EXPECT_EQ(expectSize, shaderSelectorBufferSize);
        SimplePtr shaderSelectorBuffer(nnt::g3d::AlignedAllocate<void>(shaderSelectorBufferSize, nn::g3d::ShaderSelector::Alignment_Buffer));
        shaderSelectorBuilder.Build(&shaderSelector, shaderSelectorBuffer.Get(), shaderSelectorBufferSize);
        EXPECT_EQ(shaderSelectorBuffer.Get(), shaderSelector.GetBufferPtr());
        EXPECT_EQ(2 + 1/*システム用*/, shaderSelector.GetDynamicOptionCount());
        EXPECT_EQ(1, shaderSelector.GetDynamicKeyLength());
        EXPECT_TRUE(shaderSelector.FindDynamicOption("coord") == shaderSelector.GetDynamicOption(shaderSelector.FindDynamicOptionIndex("coord")));
        EXPECT_STREQ("coord", shaderSelector.GetDynamicOptionName(shaderSelector.FindDynamicOptionIndex("coord")));

        const nn::g3d::ResShaderProgram* pDefaultProgram = shaderSelector.GetDefaultProgram();
        {
            EXPECT_TRUE(shaderSelector.UpdateVariation(pDevice));
            const nn::g3d::ResShaderProgram* pProgram = shaderSelector.GetProgram();
            EXPECT_EQ(pDefaultProgram, pProgram);
        }
        EXPECT_EQ(0, shaderSelector.ReadDynamicKey(shaderSelector.FindDynamicOptionIndex("coord")));
        shaderSelector.WriteDynamicKey(shaderSelector.FindDynamicOptionIndex("coord"), 1);
        EXPECT_EQ(1, shaderSelector.ReadDynamicKey(shaderSelector.FindDynamicOptionIndex("coord")));
        const nn::Bit32* pDynamicKey = shaderSelector.GetDynamicKey();
        // ブランチを有効にしているオプションは反映されない
        EXPECT_EQ(0x00000000, *pDynamicKey);
        {
            EXPECT_TRUE(shaderSelector.UpdateVariation(pDevice));
            const nn::g3d::ResShaderProgram* pProgram = shaderSelector.GetProgram();
            EXPECT_EQ(pDefaultProgram, pProgram);
        }
        shaderSelector.ClearDynamicKey();
        EXPECT_EQ(0, shaderSelector.ReadDynamicKey(shaderSelector.FindDynamicOptionIndex("coord")));
        {
            EXPECT_TRUE(shaderSelector.UpdateVariation(pDevice));
            const nn::g3d::ResShaderProgram* pProgram = shaderSelector.GetProgram();
            EXPECT_EQ(pDefaultProgram, pProgram);
        }
        EXPECT_EQ(&shadingModelObj, shaderSelector.GetShadingModel());

        shaderSelector.WriteDynamicKey(shaderSelector.FindDynamicOptionIndex("coord"), 1);
        char str[256];
        shaderSelector.PrintKeyTo(str, sizeof(str));
        const char* pExpectStr = "10000000";
        EXPECT_STREQ(pExpectStr, str);
        // ブランチを有効にしているオプションは反映されない
        shaderSelector.PrintRawKeyTo(str, sizeof(str));
        pExpectStr = "00000000";
        EXPECT_STREQ(pExpectStr, str);
        shaderSelector.PrintOptionTo(str, sizeof(str));
        pExpectStr = "skinning:0\tcoord:c_World\tsystem_id:0";
        EXPECT_STREQ(pExpectStr, str);
        // ブランチを有効にしているオプションは反映されない
        shaderSelector.PrintRawOptionTo(str, sizeof(str));
        pExpectStr = "skinning:0\tcoord:c_Local\tsystem_id:0";
        EXPECT_STREQ(pExpectStr, str);

        shadingModelObj.CleanupBlockBuffer(pDevice);
        FreeMemoryPool();
    }
} // NOLINT

#endif // NN_BUILD_CONFIG_SPEC_NX
