﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SampleSources/g3ddemo_ViewerUtility.h"
#include "SampleSources/Town.h"
#include <nn/g3d/g3d_Viewer.h>

namespace g3ddemo = nn::g3d::demo;

float TownPointLight::pointLightMaxRadius = 400.0f;

namespace
{
    const int g_BufferingCount = 2;
}

// ポイントライトの描画設定を初期化
void TownPointLight::Initialize(nn::gfx::Device* pDevice, nn::g3d::ResModel* pResModel, nn::g3d::ResShadingModel* pResShadingModel) NN_NOEXCEPT
{
    NN_ASSERT(!IsInitialized());

    // ModelAnimObjリストを作成
    g3ddemo::CreateVector(&m_ModelAnimObjs, maxLightCount);

    // RenderModelObjリストを作成
    g3ddemo::CreateVector(&m_RenderModelObjs, maxLightCount);

    // モデルの描画設定を構築
    m_pRenderModel = nns::g3d::CreateRenderModel(pDevice, pResModel);
    m_pRenderModel->CreateDrawPass(pDevice, pResShadingModel);
    SetupRenderState(m_pRenderModel);

    m_IsInitialized = true;
}

// ポイントライトを破棄
void TownPointLight::Finalize(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_NOT_NULL(pDevice);
    NN_ASSERT(nn::gfx::IsInitialized(*pDevice));

    // RenderModelObj の破棄
    for (nns::g3d::RenderModelObj* pRenderModelObj : m_RenderModelObjs)
    {
        nns::g3d::DestroyRenderModelObj(pRenderModelObj);
    }

    // ModelAnimObj の破棄
    for (nns::g3d::ModelAnimObj* pModelAnimObj : m_ModelAnimObjs)
    {
        nns::g3d::DestroyModelAnimObj(pDevice,pModelAnimObj);
    }

    // RenderModel を破棄
    nns::g3d::DestroyRenderModel(pDevice, m_pRenderModel);

    // RenderModelObjリストを破棄
    g3ddemo::DestroyVector(&m_RenderModelObjs);

    // ModelAnimObjリストを破棄
    g3ddemo::DestroyVector(&m_ModelAnimObjs);

    m_IsInitialized = false;
}

// ポイントライトの追加
void TownPointLight::AddLight(nn::gfx::Device* pDevice, const nn::util::Float3& color, const nn::util::Vector3fType& position, const nn::util::Vector3fType& scale) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_NOT_NULL(pDevice);
    NN_ASSERT(nn::gfx::IsInitialized(*pDevice));

     // モデルおよびアニメーションの計算処理を簡略化するModelAnimObjを作成
    nns::g3d::ModelAnimObj* pModelAnimObj = nullptr;
    {
        nn::g3d::ModelObj::Builder builder(m_pRenderModel->GetResModel());
        builder.ShapeBufferingCount(g_BufferingCount);
        builder.SkeletonBufferingCount(g_BufferingCount);
        builder.MaterialBufferingCount(g_BufferingCount);
        pModelAnimObj = nns::g3d::CreateModelAnimObj(pDevice, builder);
        m_ModelAnimObjs.PushBack(pModelAnimObj);
    }

    // モデルインスタンスに初期値を設定
    {
        pModelAnimObj->SetScale(scale);
        pModelAnimObj->SetTranslate(position);

        nn::g3d::ModelObj* pModelObj = pModelAnimObj->GetModelObj();
        nn::g3d::MaterialObj* pMaterialObj = pModelObj->GetMaterial(0);
        int locCenter = pMaterialObj->FindShaderParamIndex("center");
        int locRadius = pMaterialObj->FindShaderParamIndex("radius");
        int locColor = pMaterialObj->FindShaderParamIndex("color");
        VectorStore(pMaterialObj->EditShaderParam<nn::util::Float3>(locCenter), position);
        *pMaterialObj->EditShaderParam<float>(locRadius) = TownPointLight::pointLightMaxRadius;
        *pMaterialObj->EditShaderParam<nn::util::Float3>(locColor) = color;
    }

    // 描画設定とモデルインスタンスを結びつける
    nns::g3d::RenderModelObj* pRenderModelObj = nns::g3d::CreateRenderModelObj(pModelAnimObj->GetModelObj(), m_pRenderModel);
    g3ddemo::WriteSkinningOption(pRenderModelObj);
    BindUniformBlockAndSampler(pRenderModelObj);
    m_RenderModelObjs.PushBack(pRenderModelObj);
}

// 更新処理
void TownPointLight::Calculate(nn::gfx::Device* pDevice, int uniformBufferIndex, bool enableAnimation) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_NOT_NULL(pDevice);
    NN_ASSERT(nn::gfx::IsInitialized(*pDevice));

    if (enableAnimation)
    {
        // ライトのアニメーション
        m_PointLightRadius += m_PointLightRadiusDelta;
        if (m_PointLightRadius < 0.0f)
        {
            m_PointLightRadiusDelta = fabs(m_PointLightRadiusDelta);
            m_PointLightRadius += m_PointLightRadiusDelta;
        }
        else if (m_PointLightRadius > pointLightMaxRadius)
        {
            m_PointLightRadiusDelta = -fabs(m_PointLightRadiusDelta);
            m_PointLightRadius += m_PointLightRadiusDelta;
        }

        for (nns::g3d::ModelAnimObj* pModelAnimObj : m_ModelAnimObjs)
        {
            nn::g3d::MaterialObj* pMaterialObj = pModelAnimObj->GetModelObj()->GetMaterial(0);
            int index = pMaterialObj->FindShaderParamIndex("radius");
            *pMaterialObj->EditShaderParam<float>(index) = m_PointLightRadius;
        }
    }

    for (nns::g3d::ModelAnimObj* pModelAnimObj : m_ModelAnimObjs)
    {
        pModelAnimObj->Calculate();
        pModelAnimObj->CalculateUniformBlock(uniformBufferIndex);
    }

    for (nns::g3d::RenderModelObj* pRenderModelObj : m_RenderModelObjs)
    {
        pRenderModelObj->UpdateShader(pDevice);
    }
}

// 描画処理
void TownPointLight::Draw(nn::gfx::CommandBuffer* pCommandBuffer, int uniformBufferIndex) const NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_ASSERT(nn::gfx::IsInitialized(*pCommandBuffer));

    for (nns::g3d::RenderModelObj* pRenderModelObj : m_RenderModelObjs)
    {
        pRenderModelObj->Draw(pCommandBuffer, ViewType_Default, uniformBufferIndex);
    }
}

void ViewVolumeRenderer::Initialize()
{
    g3ddemo::InitializePrimitiveRenderer(&m_pRenderer, &m_MemoryOffset);
    NN_ASSERT_NOT_NULL(m_pRenderer);
}

void ViewVolumeRenderer::Finalize() NN_NOEXCEPT
{
    g3ddemo::FinalizePrimitiveRenderer(&m_pRenderer, &m_MemoryOffset);
}

void ViewVolumeRenderer::Draw(nn::gfx::CommandBuffer* pCommandBuffer, const nn::util::Matrix4x4fType& invViewProjectionMatrix, const nn::util::Color4u8Type& color) NN_NOEXCEPT
{
    pCommandBuffer->SetRasterizerState(m_pRenderer->GetRasterizerState(nn::gfx::PrimitiveTopologyType_Triangle, nn::gfx::CullMode_None, nn::gfx::FillMode_Solid));
    m_pRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType_DepthTest);
    m_pRenderer->SetColor(color);
    m_pRenderer->SetModelMatrix(&invViewProjectionMatrix);
    m_pRenderer->DrawCube(pCommandBuffer, nns::gfx::PrimitiveRenderer::Surface_Normal, NN_UTIL_VECTOR_3F_INITIALIZER(.0f, .0f, .0f), NN_UTIL_VECTOR_3F_INITIALIZER(2.0f, 2.0f, 2.0f));

    m_pRenderer->SetLineWidth(2.0f);
    nn::util::Color4u8Type lineColor;
    memset(lineColor.v, 255, sizeof(lineColor.v));
    m_pRenderer->SetColor(lineColor);
    m_pRenderer->DrawCube(pCommandBuffer, nns::gfx::PrimitiveRenderer::Surface_Wired, NN_UTIL_VECTOR_3F_INITIALIZER(.0f, .0f, .0f), NN_UTIL_VECTOR_3F_INITIALIZER(2.0f, 2.0f, 2.0f));
}

