﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nnt/g3d/testG3d_TestUtility.h>
#include "testG3d_MiiG3dMiiCharModelBase.h"
#include "testG3d_MiiG3dTestModelSimple.h"

// マテリアル名と mii のパーツを紐づけるテーブルです。
const MaterialNameDrawTypeTable MiiG3dTestModelSimple::s_MaterialNameDrawTypeTableArray[] =
{
    { "beard",                nn::mii::CharModel::DrawType_Beard, 0 },
    { "faceline",            nn::mii::CharModel::DrawType_Faceline, 0 },
    { "forehead",    nn::mii::CharModel::DrawType_Forehead, 0 },
    { "hair",        nn::mii::CharModel::DrawType_Hair, 0 },
    { "hat",            nn::mii::CharModel::DrawType_Hat, 0 },
    { "nose",        nn::mii::CharModel::DrawType_Nose, 0 },
    { "noseline",    nn::mii::CharModel::DrawType_Noseline, 0 },
    { "glass",    nn::mii::CharModel::DrawType_Glass, 0 },
    { "mask",    nn::mii::CharModel::DrawType_Mask, 0 },
};
const size_t MiiG3dTestModelSimple::s_MaterialNameDrawTypeTableArraySize =
                        sizeof( MiiG3dTestModelSimple::s_MaterialNameDrawTypeTableArray ) / sizeof( MaterialNameDrawTypeTable );


// ボーン名と CreateModel/NoseType を紐づけるテーブルです。
const BoneNameCreateModelTypeNoseType MiiG3dTestModelSimple::s_BoneNameCreateModelTypeNoseTypeTableArray[] =
{
    { "mii_common",        nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Normal },
    { "beard",            nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Normal },
    { "glass",            nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Normal },
    { "forehead_normal",    nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Normal },
    { "forehead_hat",        nn::mii::CreateModelType_Hat, nn::mii::CreateNoseType_Normal },
    { "hair_normal",        nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Normal },
    { "hair_hat",            nn::mii::CreateModelType_Hat, nn::mii::CreateNoseType_Normal },
    { "hat_normal",            nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Normal },
    { "hat_hat",            nn::mii::CreateModelType_Hat, nn::mii::CreateNoseType_Normal },
    { "nose_normal",        nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Normal },
    { "nose_flatten",        nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Flatten },
    { "noseline_normal",    nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Normal },
    { "noseline_flatten",    nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Flatten },
};
const size_t MiiG3dTestModelSimple::s_BoneNameCreateModelTypeNoseTypeTableArraySize =
                        sizeof( MiiG3dTestModelSimple::s_BoneNameCreateModelTypeNoseTypeTableArray ) / sizeof( BoneNameCreateModelTypeNoseType );


const char* MiiG3dTestModelSimple::s_VisibleBoneNameArray[] =
{
    "mii_common",
    //"beard",        //!< NXAddon0.11.0 でモデルデータが改変され含まれなくなった。
    "forehead_hat",
    "hair_hat",
    //"nose_normal",        //!< NXAddon0.11.0 でモデルデータが改変され含まれなくなった。
    "noseline_normal",
};
const size_t MiiG3dTestModelSimple::s_VisibleBoneNameArraySize = sizeof( MiiG3dTestModelSimple::s_VisibleBoneNameArray ) / sizeof( char* );


void MiiG3dTestModelSimple::MaterialCallback( nn::g3d::ResMaterial* pMaterial, const nn::mii::DrawParam* pDrawParam, void* pUserParam )
{
    auto pCbParam = static_cast<MaterialCallbackParam*>( pUserParam );

    const int ShaderParamCount = pMaterial->GetShaderParamCount();
    for( int idxShaderParam = 0; idxShaderParam < ShaderParamCount; ++idxShaderParam )
    {
        auto pShaderParam = pMaterial->GetShaderParam( idxShaderParam );
        const char* pShaderParamId = pShaderParam->GetId();

        if ( pCbParam->shaderParamModulateTypeId == std::string( pShaderParamId ) )
        {
            NN_SDK_ASSERT( pShaderParam->GetSrcSize() == sizeof( int32_t ), "Modulate type param size is not consistent." );
            void* pSrcParamBuffer = pMaterial->GetSrcParam( idxShaderParam );
            *static_cast<int32_t*>(pSrcParamBuffer) = static_cast<int32_t>(pDrawParam->GetModulateType());
        }
        else if ( pCbParam->shaderParamConstColorId == std::string( pShaderParamId ) )
        {
            const size_t ConstColorSize = sizeof(float) * 9;    // vec3 * 3
            NN_SDK_ASSERT( pShaderParam->GetSrcSize() == ConstColorSize, "Modulate type param size is not consistent." );
            NN_UNUSED( ConstColorSize );
            void* pSrcParamBuffer = pMaterial->GetSrcParam(idxShaderParam);
            for ( int idxConstColor = 0; idxConstColor < nn::mii::DrawParam::ConstantColorNum; ++idxConstColor )
            {
                const nn::mii::Color3* pConstColor = pDrawParam->GetConstantColor( idxConstColor );
                if ( pConstColor )
                {
                    memcpy( pSrcParamBuffer, pConstColor, sizeof( nn::mii::Color3 ) );
                }
            }
        }
    }
    const int RenderInfoCount = pMaterial->GetRenderInfoCount();
    for( int idxRenderInfo = 0; idxRenderInfo < RenderInfoCount; ++idxRenderInfo )
    {
        auto pRenderInfo = pMaterial->GetRenderInfo( idxRenderInfo );
        const char* pRenderInfoName = pRenderInfo->GetName();
        if( pRenderInfoName == std::string( pCbParam->renderInfoCullingName ) )
        {
            NN_SDK_ASSERT( nn::g3d::ResRenderInfo::Type_Int == pRenderInfo->GetType(), "RenderInfo culling type mismatch." );
            nn::gfx::CullMode cullMode = pDrawParam->GetCullMode();    // none, front, back
            *pRenderInfo->ToData().intValueArray.Get() = static_cast<int>(cullMode);
        }
    }
}


void MiiG3dTestModelSimple::DrawParamCallback( nn::g3d::mii::DrawParamCallbackOutput* pDrawParamCallbackOutput,
                                        const nn::g3d::ResModel* pModel,
                                        const int shapeIndex, const nn::mii::CharModel* pCharModel, void* pCbParam )
{
    auto& cbParam = *reinterpret_cast<DrawParamCallbackParam*>( pCbParam );

    //!< テストの正解値として shapeIndex に対応する nn::g3d::mii::DrawParamCallbackOutput を保存します。
    //!< 最初に本コールバック関数が呼ばれた時に全 shape 分のサイズを確保します。
    if( cbParam.m_pG3dMiiCharModel->GetDrawParamCallbackOutputArraySize() == 0 )
    {
        NN_ASSERT( pModel->GetShapeCount() != 0 );
        cbParam.m_pG3dMiiCharModel->ResizeDrawParamCallbackOutputArray( pModel->GetShapeCount() );
    }

    auto pShape = pModel->GetShape( shapeIndex );
    const char* pMatName = pModel->GetMaterial( pShape->GetMaterialIndex() )->GetName();
    int idxCharModelMaterialBindTable = FindMiiBindTableIndex( pMatName, cbParam.m_pMaterialNameDrawTypeTableArray,
                                                                cbParam.m_MaterialNameDrawTypeTableArraySize );

    // Mii のパーツではない場合
    if( idxCharModelMaterialBindTable == -1 )
    {
        pDrawParamCallbackOutput->isMiiFace = false;
        pDrawParamCallbackOutput->pDrawParam = nullptr;
        cbParam.m_pG3dMiiCharModel->SetDrawParamCallbackOutput( shapeIndex, *pDrawParamCallbackOutput );
        return;
    }

    // Mii のパーツの場合、DrawParam を取得します。
    const char* pBoneName = pModel->GetSkeleton()->GetBoneName( pShape->GetBoneIndex() );
    int idxCharModelBoneBindTable = FindMiiBindTableIndex( pBoneName, cbParam.pBoneNameCreateModelTypeNoseTypeArray,
                                                            cbParam.m_BoneNameCreateModelTypeNoseTypeArraySize );
    NN_ASSERT( idxCharModelBoneBindTable != -1 );
    auto& boneBindTbl = cbParam.pBoneNameCreateModelTypeNoseTypeArray[idxCharModelBoneBindTable];
    auto& matBindTbl = cbParam.m_pMaterialNameDrawTypeTableArray[idxCharModelMaterialBindTable];
    pDrawParamCallbackOutput->isMiiFace = true;
    pDrawParamCallbackOutput->pDrawParam = pCharModel->GetDrawParam( matBindTbl.type1, boneBindTbl.type1, boneBindTbl.type2, matBindTbl.type2 );
    pDrawParamCallbackOutput->pBoundingBox = &pCharModel->GetBoundingBox( boneBindTbl.type1, boneBindTbl.type2 );
    cbParam.m_pG3dMiiCharModel->SetDrawParamCallbackOutput( shapeIndex, *pDrawParamCallbackOutput );
}
