﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "testG3d_Main.h"

// 頂点バッファーのアラインメント
const size_t g_VertexBufferAlignment = 8;
const size_t g_IndexBufferAlignment = 8;

void CheckResDicDataEqual(const nn::util::ResDicData& data, const nn::util::ResDicData& referenceData)
{
    // 各値
    EXPECT_EQ(data.count, referenceData.count);
    for (int i = 0; i < data.count; ++i)
    {
        EXPECT_EQ(data.entries[1 + i].refBit, referenceData.entries[1 + i].refBit);
        EXPECT_EQ(data.entries[1 + i].children[0], referenceData.entries[1 + i].children[0]);
        EXPECT_EQ(data.entries[1 + i].children[1], referenceData.entries[1 + i].children[1]);
        EXPECT_EQ(std::strcmp(data.entries[1 + i].pKey.Get()->GetData(), referenceData.entries[1 + i].pKey.Get()->GetData()), 0);
    }
}

void CheckResUserDataEqual(const nn::gfx::ResUserData& pData, const nn::gfx::ResUserData& pReferenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(pData.ToData()), 64);

    // ポインター
    EXPECT_EQ(std::strcmp(pData.ToData().pName.Get()->GetData(), pReferenceData.ToData().pName.Get()->GetData()), 0);
    switch (pData.ToData().type)
    {
    case nn::gfx::ResUserData::Type_Int:
        {
            for (int i = 0; i < static_cast<int>(pData.ToData().count); ++i)
            {
                EXPECT_EQ(pData.GetInt()[i], pReferenceData.GetInt()[i]);
            }
        }
        break;
    case nn::gfx::ResUserData::Type_Float:
        {
            for (int i = 0; i < static_cast<int>(pData.ToData().count); ++i)
            {
                EXPECT_EQ(pData.GetFloat()[i], pReferenceData.GetFloat()[i]);
            }
        }
        break;
    case nn::gfx::ResUserData::Type_String:
        {
            for (int i = 0; i < static_cast<int>(pData.ToData().count); ++i)
            {
                EXPECT_EQ(std::strcmp(pData.GetString(i), pReferenceData.GetString(i)), 0);
            }
        }
        break;
    case nn::gfx::ResUserData::Type_Stream:
        {
            EXPECT_EQ(pData.GetStreamSize(), pReferenceData.GetStreamSize());
            EXPECT_EQ(memcmp(pData.GetStream(), pReferenceData.GetStream(), pData.GetStreamSize()), 0);
        }
        break;
    default:
        break;
    }

    // 各値
    EXPECT_EQ(pData.ToData().count, pReferenceData.ToData().count);
    EXPECT_EQ(pData.ToData().type, pReferenceData.ToData().type);

    // 予約領域
    EXPECT_EQ(sizeof(pData.ToData().reserved), 43);
}

void CheckResBoneDataEqual(const nn::g3d::ResBoneData& data, const nn::g3d::ResBoneData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 96);

    // ポインター
    EXPECT_EQ(std::strcmp(data.pName.Get()->GetData(), referenceData.pName.Get()->GetData()), 0);
    for (int i = 0; i < data.userDataCount; ++i)
    {
        CheckResUserDataEqual(data.pUserDataArray.Get()[i], referenceData.pUserDataArray.Get()[i]);
    }
    if (data.userDataCount > 0)
    {
        CheckResDicDataEqual(data.pUserDataDic.Get()->ToData(), referenceData.pUserDataDic.Get()->ToData());
    }

    // 各値
    EXPECT_EQ(data.index, referenceData.index);
    EXPECT_EQ(data.parentIndex, referenceData.parentIndex);
    EXPECT_EQ(data.smoothMtxIndex, referenceData.smoothMtxIndex);
    EXPECT_EQ(data.rigidMtxIndex, referenceData.rigidMtxIndex);
    EXPECT_EQ(data.billboardIndex, referenceData.billboardIndex);
    EXPECT_EQ(data.userDataCount, referenceData.userDataCount);
    EXPECT_EQ(data.flag, referenceData.flag);
    nn::util::Vector3fType scale, rScale;
    nn::util::VectorLoad(&scale, data.scale);
    nn::util::VectorLoad(&rScale, referenceData.scale);
    EXPECT_TRUE(IsVector3fEqual(scale, rScale));
    nn::util::Vector4fType rotate, rRotate;
    nn::util::VectorLoad(&rotate, data.rotate.quat);
    nn::util::VectorLoad(&rRotate, referenceData.rotate.quat);
    EXPECT_TRUE(IsVector4fEqual(rotate, rRotate));
    nn::util::Vector3fType translate, rTranslate;
    nn::util::VectorLoad(&translate, data.translate);
    nn::util::VectorLoad(&rTranslate, referenceData.translate);
    EXPECT_TRUE(IsVector3fEqual(translate, rTranslate));

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved), 16);
}

void CheckResSkeletonDataEqual(const nn::g3d::ResSkeletonData& data, const nn::g3d::ResSkeletonData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 88);

    // ポインター
    for (int i = 0; i < data.boneCount; ++i)
    {
        CheckResBoneDataEqual(data.pBoneArray.Get()[i].ToData(), referenceData.pBoneArray.Get()[i].ToData());
    }
    if (data.boneCount > 0)
    {
        CheckResDicDataEqual(data.pBoneDic.Get()->ToData(), referenceData.pBoneDic.Get()->ToData());
    }
    for (int i = 0; i < data.smoothMtxCount; ++i)
    {
        EXPECT_EQ(data.pMtxToBoneTable.Get()[i], referenceData.pMtxToBoneTable.Get()[i]);

        nn::util::Matrix4x3fType mtx, rMtx;
        nn::util::MatrixLoad(&mtx, data.pInvModelMatrixArray.Get()[i]);
        nn::util::MatrixLoad(&rMtx, referenceData.pInvModelMatrixArray.Get()[i]);
        EXPECT_TRUE(IsMtx43Equal(mtx, rMtx));
    }
    if (referenceData.pMirroringBoneTable.Get() != NULL)
    {
        for (int i = 0; i < data.boneCount; ++i)
        {
            EXPECT_EQ(data.pMirroringBoneTable.Get()[i], referenceData.pMirroringBoneTable.Get()[i]);
        }
    }

    // 各値
    EXPECT_EQ(data.flag, referenceData.flag);
    EXPECT_EQ(data.boneCount, referenceData.boneCount);
    EXPECT_EQ(data.smoothMtxCount, referenceData.smoothMtxCount);
    EXPECT_EQ(data.rigidMtxCount, referenceData.rigidMtxCount);

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved1), 8);
    EXPECT_EQ(sizeof(data.reserved2), 6);
}

void CheckResVertexAttrDataEqual(const nn::g3d::ResVertexAttrData& data, const nn::g3d::ResVertexAttrData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 16);

    // ポインタ
    EXPECT_EQ(std::strcmp(data.pName.Get()->GetData(), referenceData.pName.Get()->GetData()), 0);

    // 各値
    EXPECT_EQ(data.format, referenceData.format);
    EXPECT_EQ(data.offset, referenceData.offset);
    EXPECT_EQ(data.bufferIndex, referenceData.bufferIndex);

    // フラグ
    EXPECT_EQ(sizeof(data.flag), 1);
    EXPECT_EQ(data.flag, referenceData.flag);
    EXPECT_EQ(data.flag, 0);
}

void CheckResVertexDataEqual(const nn::g3d::ResVertexData& data, const nn::g3d::ResVertexData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 96);

    // ポインタ
    for (int i = 0; i < data.vertexAttrCount; ++i)
    {
        CheckResVertexAttrDataEqual(data.pVertexAttrArray.Get()[i].ToData(), referenceData.pVertexAttrArray.Get()[i].ToData());
    }
    if (data.vertexAttrCount > 0)
    {
        CheckResDicDataEqual(data.pVertexAttrDic.Get()->ToData(), referenceData.pVertexAttrDic.Get()->ToData());
    }
    // メモリープールのチェックは ResFile で行う
    for (int i = 0; i < data.vertexBufferCount; ++i)
    {
        const nn::gfx::Buffer* pBuffer = &data.pVertexBufferArray.Get()[i];
        const nn::gfx::Buffer* pRBuffer = &referenceData.pVertexBufferArray.Get()[i];
        const void* pMapBuffer = pBuffer->Map();
        const void* pRMapBuffer = pRBuffer->Map();
        size_t bufferSize = nn::util::align_up(data.pVertexBufferInfoArray.Get()[i].GetSize(), g_VertexBufferAlignment);
        EXPECT_EQ(memcmp(pMapBuffer, pRMapBuffer, bufferSize), 0);
        pBuffer->Unmap();
        pRBuffer->Unmap();

        // ポインタ先の値が VertexBuffer のアドレスを指している事をチェック
        EXPECT_EQ(data.pVertexBufferPtrArray.Get()[i], &data.pVertexBufferArray.Get()[i]);
        EXPECT_EQ(memcmp(&data.pVertexBufferInfoArray.Get()[i], &referenceData.pVertexBufferInfoArray.Get()[i], sizeof(nn::gfx::BufferInfo)), 0);
        EXPECT_EQ(memcmp(&data.pVertexBufferStateInfoArray.Get()[i], &referenceData.pVertexBufferStateInfoArray.Get()[i], sizeof(nn::gfx::VertexBufferStateInfo)), 0);
    }

    // 各値
    EXPECT_EQ(data.memoryPoolOffset, referenceData.memoryPoolOffset);
    EXPECT_EQ(data.vertexAttrCount, referenceData.vertexAttrCount);
    EXPECT_EQ(data.vertexBufferCount, referenceData.vertexBufferCount);
    EXPECT_EQ(data.index, referenceData.index);
    EXPECT_EQ(data.count, referenceData.count);
    EXPECT_EQ(data.influenceCount, referenceData.influenceCount);

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved), 3);
}

void CheckResMeshDataEqual(const nn::g3d::ResMeshData& data, const nn::g3d::ResMeshData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 56);

    // ポインタ
    for (int i = 0; i < data.subMeshCount; ++i)
    {
        EXPECT_EQ(data.pSubMeshArray.Get()[i].ToData().offset, referenceData.pSubMeshArray.Get()[i].ToData().offset);
        EXPECT_EQ(data.pSubMeshArray.Get()[i].ToData().count, referenceData.pSubMeshArray.Get()[i].ToData().count);
    }
    // メモリープールのチェックは ResFile で行う
    const nn::gfx::Buffer* pBuffer = data.pIndexBuffer.Get();
    const nn::gfx::Buffer* pRBuffer = referenceData.pIndexBuffer.Get();
    const void* pMapBuffer = pBuffer->Map();
    const void* pRMapBuffer = pRBuffer->Map();
    size_t bufferSize = nn::util::align_up(data.pIndexBufferInfo.Get()->GetSize(), g_IndexBufferAlignment);
    EXPECT_EQ(memcmp(pMapBuffer, pRMapBuffer, bufferSize), 0);
    pBuffer->Unmap();
    pRBuffer->Unmap();
    EXPECT_EQ(memcmp(data.pIndexBufferInfo.Get(), referenceData.pIndexBufferInfo.Get(), sizeof(nn::gfx::BufferInfo)), 0);

    // 各値
    EXPECT_EQ(data.memoryPoolOffset, referenceData.memoryPoolOffset);
    EXPECT_EQ(data.primType, referenceData.primType);
    EXPECT_EQ(data.format, referenceData.format);
    EXPECT_EQ(data.count, referenceData.count);
    EXPECT_EQ(data.offset, referenceData.offset);
    EXPECT_EQ(data.subMeshCount, referenceData.subMeshCount);

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved), 2);
}

void CheckResShapeDataEqual(const nn::g3d::ResShapeData& data, const nn::g3d::ResShapeData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 112);

    // ポインタ
    EXPECT_EQ(std::strcmp(data.pName.Get()->GetData(), referenceData.pName.Get()->GetData()), 0);
    CheckResVertexDataEqual(data.pVertex.Get()->ToData(), referenceData.pVertex.Get()->ToData());
    for (int i = 0; i < data.meshCount; ++i)
    {
        CheckResMeshDataEqual(data.pMeshArray.Get()[i].ToData(), referenceData.pMeshArray.Get()[i].ToData());
    }
    for (int i = 0; i < data.skinBoneIndexCount; ++i)
    {
        EXPECT_EQ(data.pSkinBoneIndexArray.Get()[i], referenceData.pSkinBoneIndexArray.Get()[i]);
    }
    if (data.keyShapeCount > 0)
    {
        for (int keyShapeIndex = 0; keyShapeIndex < data.keyShapeCount; ++keyShapeIndex)
        {
            for (int i = 0; i < 20; ++i)
            {
                EXPECT_EQ(data.pKeyShapeArray.Get()[keyShapeIndex].ToData().targetAttribIndices[i], referenceData.pKeyShapeArray.Get()[keyShapeIndex].ToData().targetAttribIndices[i]);
            }
        }
        CheckResDicDataEqual(data.pKeyShapeDic.Get()->ToData(), referenceData.pKeyShapeDic.Get()->ToData());
    }
    for (int i = 0; i < data.meshCount; ++i)
    {
        nn::util::Vector3fType center, rCenter;
        nn::util::VectorLoad(&center, data.pSubMeshBoundingArray.Get()[i].center);
        nn::util::VectorLoad(&rCenter, referenceData.pSubMeshBoundingArray.Get()[i].center);
        EXPECT_TRUE(IsVector3fEqual(center, rCenter));
        nn::util::Vector3fType extent, rExtent;
        nn::util::VectorLoad(&extent, data.pSubMeshBoundingArray.Get()[i].extent);
        nn::util::VectorLoad(&rExtent, referenceData.pSubMeshBoundingArray.Get()[i].extent);
        EXPECT_TRUE(IsVector3fEqual(extent, rExtent));

        EXPECT_EQ(data.pRadiusArray.Get()[i], referenceData.pRadiusArray.Get()[i]);
    }

    // 各値
    EXPECT_EQ(data.flag, referenceData.flag);
    EXPECT_EQ(data.index, referenceData.index);
    EXPECT_EQ(data.materialIndex, referenceData.materialIndex);
    EXPECT_EQ(data.boneIndex, referenceData.boneIndex);
    EXPECT_EQ(data.vertexIndex, referenceData.vertexIndex);
    EXPECT_EQ(data.skinBoneIndexCount, referenceData.skinBoneIndexCount);
    EXPECT_EQ(data.vtxSkinCount, referenceData.vtxSkinCount);
    EXPECT_EQ(data.meshCount, referenceData.meshCount);
    EXPECT_EQ(data.keyShapeCount, referenceData.keyShapeCount);
    EXPECT_EQ(data.targetAttrCount, referenceData.targetAttrCount);

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved), 6);
}

void CheckResRenderInfoDataEqual(const nn::g3d::ResRenderInfoData& data, const nn::g3d::ResRenderInfoData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 24);

    // ポインタ
    EXPECT_EQ(std::strcmp(data.pName.Get()->GetData(), referenceData.pName.Get()->GetData()), 0);
    for (int i = 0; i < data.arrayLength; ++i)
    {
        EXPECT_EQ(data.intValueArray.Get()[i], referenceData.intValueArray.Get()[i]);
    }
    EXPECT_EQ(data.arrayLength, referenceData.arrayLength);
    EXPECT_EQ(data.type, referenceData.type);

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved), 5);
}

void CheckResShaderParamDataEqual(const nn::g3d::ResShaderParamData& data, const nn::g3d::ResShaderParamData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 32);

    EXPECT_EQ(std::strcmp(data.pId.Get()->GetData(), referenceData.pId.Get()->GetData()), 0);

    // ポインタ
    EXPECT_EQ(data.type, referenceData.type);
    EXPECT_EQ(data.srcSize, referenceData.srcSize);
    EXPECT_EQ(data.srcOffset, referenceData.srcOffset);
    EXPECT_EQ(data.offset, referenceData.offset);
    EXPECT_EQ(data.dependedIndex, referenceData.dependedIndex);
    EXPECT_EQ(data.dependIndex, referenceData.dependIndex);

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved), 4);
}

void CheckResMaterialDataEqual(const nn::g3d::ResMaterialData& data, const nn::g3d::ResMaterialData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 184);

    // ポインタ
    EXPECT_EQ(std::strcmp(data.pName.Get()->GetData(), referenceData.pName.Get()->GetData()), 0);
    for (int i = 0; i < data.renderInfoCount; ++i)
    {
        CheckResRenderInfoDataEqual(data.pRenderInfoArray.Get()[i].ToData(), referenceData.pRenderInfoArray.Get()[i].ToData());
    }
    if (data.renderInfoCount > 0)
    {
        CheckResDicDataEqual(data.pRenderInfoDic.Get()->ToData(), referenceData.pRenderInfoDic.Get()->ToData());
    }
    if(data.pShaderAssign.Get())
    {
        const nn::g3d::ResShaderAssignData& shderAssign = data.pShaderAssign.Get()->ToData();
        const nn::g3d::ResShaderAssignData& referenceSahderAssign = referenceData.pShaderAssign.Get()->ToData();
        EXPECT_EQ(std::strcmp(shderAssign.pShaderArchiveName.Get()->GetData(), referenceSahderAssign.pShaderArchiveName.Get()->GetData()), 0);
        EXPECT_EQ(std::strcmp(shderAssign.pShadingModelName.Get()->GetData(), referenceSahderAssign.pShadingModelName.Get()->GetData()), 0);
        for (int i = 0; i < shderAssign.attrAssignCount; ++i)
        {
            EXPECT_EQ(memcmp(shderAssign.pAttrAssignArray.Get()[i].Get()->GetData(), referenceSahderAssign.pAttrAssignArray.Get()[i].Get()->GetData(), shderAssign.pAttrAssignArray.Get()[i].Get()->GetLength()), 0);
        }
        if (shderAssign.attrAssignCount > 0)
        {
            CheckResDicDataEqual(shderAssign.pAttrAssignDic.Get()->ToData(), referenceSahderAssign.pAttrAssignDic.Get()->ToData());
        }
        for (int i = 0; i < shderAssign.samplerAssignCount; ++i)
        {
            EXPECT_EQ(memcmp(shderAssign.pSamplerAssignArray.Get()[i].Get()->GetData(), referenceSahderAssign.pSamplerAssignArray.Get()[i].Get()->GetData(), shderAssign.pSamplerAssignArray.Get()[i].Get()->GetLength()), 0);
        }
        if (shderAssign.samplerAssignCount > 0)
        {
            CheckResDicDataEqual(shderAssign.pSamplerAssignDic.Get()->ToData(), referenceSahderAssign.pSamplerAssignDic.Get()->ToData());
        }
        for (int i = 0; i < shderAssign.shaderOptionCount; ++i)
        {
            EXPECT_EQ(memcmp(shderAssign.pShaderOptionArray.Get()[i].Get()->GetData(), referenceSahderAssign.pShaderOptionArray.Get()[i].Get()->GetData(), shderAssign.pShaderOptionArray.Get()[i].Get()->GetLength()), 0);
        }
        if (shderAssign.shaderOptionCount > 0)
        {
            CheckResDicDataEqual(shderAssign.pShaderOptionDic.Get()->ToData(), referenceSahderAssign.pShaderOptionDic.Get()->ToData());
        }
        EXPECT_EQ(shderAssign.revision, referenceSahderAssign.revision);
        EXPECT_EQ(shderAssign.attrAssignCount, referenceSahderAssign.attrAssignCount);
        EXPECT_EQ(shderAssign.samplerAssignCount, referenceSahderAssign.samplerAssignCount);
        EXPECT_EQ(shderAssign.shaderOptionCount, referenceSahderAssign.shaderOptionCount);
    }
    for (int i = 0; i < data.textureCount; ++i)
    {
        // ポインタの示す先は異なるので比較しない
        // EXPECT_EQ(memcmp(&data.pTextureArray.Get()[i], &referenceData.pTextureArray.Get()[i], sizeof(nn::gfx::TextureView)), 0);
        EXPECT_EQ(memcmp(data.pTextureNameArray.Get()[i].Get()->GetData(), referenceData.pTextureNameArray.Get()[i].Get()->GetData(), data.pTextureNameArray.Get()[i].Get()->GetLength()), 0);
    }
    for (int i = 0; i < data.samplerCount; ++i)
    {
        // ポインタの示す先は異なるので比較しない
        //EXPECT_EQ(memcmp(&data.pSamplerArray.Get()[i], &referenceData.pSamplerArray.Get()[i], sizeof(nn::gfx::Sampler)), 0);
        EXPECT_EQ(memcmp(&data.pSamplerInfoArray.Get()[i], &referenceData.pSamplerInfoArray.Get()[i], sizeof(nn::gfx::SamplerInfo)), 0);
    }
    if (data.samplerCount > 0)
    {
        CheckResDicDataEqual(data.pSamplerInfoDic.Get()->ToData(), referenceData.pSamplerInfoDic.Get()->ToData());
    }
    for (int i = 0; i < data.shaderParamCount; ++i)
    {
        CheckResShaderParamDataEqual(data.pShaderParamArray.Get()[i].ToData(), referenceData.pShaderParamArray.Get()[i].ToData());
    }
    if (data.shaderParamCount > 0)
    {
        CheckResDicDataEqual(data.pShaderParamDic.Get()->ToData(), referenceData.pShaderParamDic.Get()->ToData());
    }
    EXPECT_EQ(memcmp(data.pSrcParam.Get(), referenceData.pSrcParam.Get(), data.srcParamSize), 0);
    for (int i = 0; i < data.userDataCount; ++i)
    {
        CheckResUserDataEqual(data.pUserDataArray.Get()[i], referenceData.pUserDataArray.Get()[i]);
    }
    if (data.userDataCount > 0)
    {
        CheckResDicDataEqual(data.pUserDataDic.Get()->ToData(), referenceData.pUserDataDic.Get()->ToData());
    }
    EXPECT_EQ(memcmp(data.pVolatileFlag.Get(), referenceData.pVolatileFlag.Get(), nn::util::align_up(data.shaderParamCount, 32) >> 3), 0);
    for (int i = 0; i < data.samplerCount; ++i)
    {
        EXPECT_EQ(memcmp(&data.pSamplerSlotArray.Get()[i], &referenceData.pSamplerSlotArray.Get()[i], sizeof(nn::gfx::DescriptorSlot)), 0);
    }
    for (int i = 0; i < data.textureCount; ++i)
    {
        EXPECT_EQ(memcmp(&data.pTextureSlotArray.Get()[i], &referenceData.pTextureSlotArray.Get()[i], sizeof(nn::gfx::DescriptorSlot)), 0);
    }

    // 各値
    EXPECT_EQ(data.flag, referenceData.flag);
    EXPECT_EQ(data.index, referenceData.index);
    EXPECT_EQ(data.renderInfoCount, referenceData.renderInfoCount);
    EXPECT_EQ(data.samplerCount, referenceData.samplerCount);
    EXPECT_EQ(data.textureCount, referenceData.textureCount);
    EXPECT_EQ(data.shaderParamCount, referenceData.shaderParamCount);
    EXPECT_EQ(data.shaderParamVolatileCount, referenceData.shaderParamVolatileCount);
    EXPECT_EQ(data.srcParamSize, referenceData.srcParamSize);
    EXPECT_EQ(data.rawParamSize, referenceData.rawParamSize);
    EXPECT_EQ(data.userDataCount, referenceData.userDataCount);

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved), 4);
}

void CheckResModelDataEqual(const nn::g3d::ResModelData& data, const nn::g3d::ResModelData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 120);

    // ポインタ
    EXPECT_EQ(std::strcmp(data.pName.Get()->GetData(), referenceData.pName.Get()->GetData()), 0);
    EXPECT_EQ(std::strcmp(data.pPath.Get()->GetData(), referenceData.pPath.Get()->GetData()), 0);

    CheckResSkeletonDataEqual(data.pSkeleton.Get()->ToData(), referenceData.pSkeleton.Get()->ToData());
    for (int i = 0; i < data.vertexCount; ++i)
    {
        CheckResVertexDataEqual(data.pVertexArray.Get()[i].ToData(), referenceData.pVertexArray.Get()[i].ToData());
    }
    for (int i = 0; i < data.shapeCount; ++i)
    {
        CheckResShapeDataEqual(data.pShapeArray.Get()[i].ToData(), referenceData.pShapeArray.Get()[i].ToData());
    }
    if (data.shapeCount > 0)
    {
        CheckResDicDataEqual(data.pShapeDic.Get()->ToData(), referenceData.pShapeDic.Get()->ToData());
    }
    for (int i = 0; i < data.materialCount; ++i)
    {
        CheckResMaterialDataEqual(data.pMaterialArray.Get()[i].ToData(), referenceData.pMaterialArray.Get()[i].ToData());
    }
    if (data.materialCount > 0)
    {
        CheckResDicDataEqual(data.pMaterialDic.Get()->ToData(), referenceData.pMaterialDic.Get()->ToData());
    }
    for (int i = 0; i < data.userDataCount; ++i)
    {
        CheckResUserDataEqual(data.pUserDataArray.Get()[i], referenceData.pUserDataArray.Get()[i]);
    }
    if (data.userDataCount > 0)
    {
        CheckResDicDataEqual(data.pUserDataDic.Get()->ToData(), referenceData.pUserDataDic.Get()->ToData());
    }

    // 各値
    EXPECT_EQ(data.vertexCount, referenceData.vertexCount);
    EXPECT_EQ(data.shapeCount, referenceData.shapeCount);
    EXPECT_EQ(data.materialCount, referenceData.materialCount);
    EXPECT_EQ(data.userDataCount, referenceData.userDataCount);

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved), 8);
}

void CheckResAnimCurveDataEqual(const nn::g3d::ResAnimCurveData& data, const nn::g3d::ResAnimCurveData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 48);

    if (data.flag & nn::g3d::ResAnimCurve::FrameType_Quant8)
    {
        EXPECT_EQ(memcmp(data.pFrameArray.Get(), referenceData.pFrameArray.Get(), sizeof(uint8_t) * data.keyCount), 0);
    }
    else if (data.flag & nn::g3d::ResAnimCurve::FrameType_Quant16)
    {
        EXPECT_EQ(memcmp(data.pFrameArray.Get(), referenceData.pFrameArray.Get(), sizeof(uint16_t) * data.keyCount), 0);
    }
    else if (data.flag & nn::g3d::ResAnimCurve::FrameType_Quant32)
    {
        EXPECT_EQ(memcmp(data.pFrameArray.Get(), referenceData.pFrameArray.Get(), sizeof(float) * data.keyCount), 0);
    }

    int coefSize = 1;
    if (data.flag & nn::g3d::ResAnimCurve::CurveType_Cubic)
    {
        coefSize = 4;
    }
    else if (data.flag & nn::g3d::ResAnimCurve::CurveType_Linear)
    {
        coefSize = 2;
    }
    if (data.flag & nn::g3d::ResAnimCurve::KeyType_Quant8)
    {
        EXPECT_EQ(memcmp(data.pKeyArray.Get(), referenceData.pKeyArray.Get(), sizeof(uint8_t) * data.keyCount * coefSize), 0);
    }
    else if (data.flag & nn::g3d::ResAnimCurve::KeyType_Quant16)
    {
        EXPECT_EQ(memcmp(data.pKeyArray.Get(), referenceData.pKeyArray.Get(), sizeof(uint16_t) * data.keyCount * coefSize), 0);
    }
    else if (data.flag & nn::g3d::ResAnimCurve::KeyType_Quant32)
    {
        EXPECT_EQ(memcmp(data.pKeyArray.Get(), referenceData.pKeyArray.Get(), sizeof(float) * data.keyCount * coefSize), 0);
    }

    EXPECT_EQ(data.flag, referenceData.flag);
    EXPECT_EQ(data.keyCount , referenceData.keyCount);
    EXPECT_EQ(data.targetOffset, referenceData.targetOffset);
    EXPECT_EQ(data.startFrame, referenceData.startFrame);
    EXPECT_EQ(data.endFrame, referenceData.endFrame);
    EXPECT_EQ(data.fScale, referenceData.fScale);
    EXPECT_EQ(data.fOffset, referenceData.fOffset);
    EXPECT_EQ(data.fDelta, referenceData.fDelta);

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved), 4);
}

void CheckResBoneAnimDataEqual(const nn::g3d::ResBoneAnimData& data, const nn::g3d::ResBoneAnimData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 40);

    // ポインタ
    EXPECT_EQ(std::strcmp(data.pName.Get()->GetData(), referenceData.pName.Get()->GetData()), 0);
    for (int i = 0; i < data.curveCount; ++i)
    {
        CheckResAnimCurveDataEqual(data.pCurveArray.Get()[i].ToData(), referenceData.pCurveArray.Get()[i].ToData());
    }
    int baseValueSize = 0;
    if (data.flag & nn::g3d::ResBoneAnim::Flag_BaseScale)
    {
        baseValueSize += 3;
    }
    if (data.flag & nn::g3d::ResBoneAnim::Flag_BaseTranslate)
    {
        baseValueSize += 3;
    }
    if (data.flag & nn::g3d::ResBoneAnim::Flag_BaseRotate)
    {
        baseValueSize += 4;
    }
    EXPECT_EQ(memcmp(data.pBaseValueArray.Get(), referenceData.pBaseValueArray.Get(), sizeof(float) * baseValueSize), 0);

    // 各値
    EXPECT_EQ(data.flag, referenceData.flag);
    EXPECT_EQ(data.beginRotate, referenceData.beginRotate);
    EXPECT_EQ(data.beginTranslate, referenceData.beginTranslate);
    EXPECT_EQ(data.curveCount, referenceData.curveCount);
    EXPECT_EQ(data.beginBaseTranslate, referenceData.beginBaseTranslate);
    EXPECT_EQ(data.beginCurve, referenceData.beginCurve);

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved), 4);
}

void CheckResSkeletalAnimDataEqual(const nn::g3d::ResSkeletalAnimData& data, const nn::g3d::ResSkeletalAnimData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 96);

    // ポインタ
    EXPECT_EQ(std::strcmp(data.pName.Get()->GetData(), referenceData.pName.Get()->GetData()), 0);
    EXPECT_EQ(std::strcmp(data.pPath.Get()->GetData(), referenceData.pPath.Get()->GetData()), 0);
    EXPECT_EQ(data.pBindSkeleton.Get(), referenceData.pBindSkeleton.Get());

    for (int i = 0; i < data.boneAnimCount; ++i)
    {
        EXPECT_EQ(data.pBindIndexArray.Get()[i], referenceData.pBindIndexArray.Get()[i]);
        CheckResBoneAnimDataEqual(data.pBoneAnimArray.Get()[i].ToData(), referenceData.pBoneAnimArray.Get()[i].ToData());
    }
    for (int i = 0; i < data.userDataCount; ++i)
    {
        CheckResUserDataEqual(data.pUserDataArray.Get()[i], referenceData.pUserDataArray.Get()[i]);
    }
    if (data.userDataCount > 0)
    {
        CheckResDicDataEqual(data.pUserDataDic.Get()->ToData(), referenceData.pUserDataDic.Get()->ToData());
    }

    // 各値
    EXPECT_EQ(data.flag, referenceData.flag);
    EXPECT_EQ(data.frameCount, referenceData.frameCount);
    EXPECT_EQ(data.curveCount, referenceData.curveCount);
    EXPECT_EQ(data.bakedSize, referenceData.bakedSize);
    EXPECT_EQ(data.boneAnimCount, referenceData.boneAnimCount);
    EXPECT_EQ(data.userDataCount, referenceData.userDataCount);

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved), 4);
}

void CheckResAnimConstantDataEqual(const nn::g3d::ResAnimConstantData& data, const nn::g3d::ResAnimConstantData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 8);

    EXPECT_EQ(data.targetOffset, referenceData.targetOffset);
    EXPECT_EQ(data.fValue, referenceData.fValue);
}

void CheckShaderParamAnimInfoEqual(const nn::g3d::ResShaderParamAnimInfo& data, const nn::g3d::ResShaderParamAnimInfo& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 24);

    // ポインタ
    EXPECT_EQ(std::strcmp(data.pName.Get()->GetData(), referenceData.pName.Get()->GetData()), 0);

    // 各値
    EXPECT_EQ(data.beginCurve, referenceData.beginCurve);
    EXPECT_EQ(data.floatCurveCount, referenceData.floatCurveCount);
    EXPECT_EQ(data.intCurveCount, referenceData.intCurveCount);
    EXPECT_EQ(data.beginConstant, referenceData.beginConstant);
    EXPECT_EQ(data.constantCount, referenceData.constantCount);
    EXPECT_EQ(data.subbindIndex, referenceData.subbindIndex);

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved), 4);
}

void CheckTexturePatternAnimInfoEqual(const nn::g3d::ResTexturePatternAnimInfo& data, const nn::g3d::ResTexturePatternAnimInfo& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 16);

    // ポインタ
    EXPECT_EQ(std::strcmp(data.pName.Get()->GetData(), referenceData.pName.Get()->GetData()), 0);

    // 各値
    EXPECT_EQ(data.curveIndex, referenceData.curveIndex);
    EXPECT_EQ(data.constantIndex, referenceData.constantIndex);
    EXPECT_EQ(data.subbindIndex, referenceData.subbindIndex);

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved), 3);
}

void CheckResPerMaterialAnimDataEqual(const nn::g3d::ResPerMaterialAnimData& data, const nn::g3d::ResPerMaterialAnimData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 64);

    // ポインタ
    EXPECT_EQ(std::strcmp(data.pName.Get()->GetData(), referenceData.pName.Get()->GetData()), 0);
    for (int i = 0; i < data.shaderParamAnimCount; ++i)
    {
        CheckShaderParamAnimInfoEqual(data.pShaderParamAnimInfoArray.Get()[i], referenceData.pShaderParamAnimInfoArray.Get()[i]);
    }
    for (int i = 0; i < data.texturePatternAnimCount; ++i)
    {
        CheckTexturePatternAnimInfoEqual(data.pTexturePatternAnimInfoArray.Get()[i], referenceData.pTexturePatternAnimInfoArray.Get()[i]);
    }
    for (int i = 0; i < data.curveCount; ++i)
    {
        CheckResAnimCurveDataEqual(data.pCurveArray.Get()[i].ToData(), referenceData.pCurveArray.Get()[i].ToData());
    }
    for (int i = 0; i < data.constantCount; ++i)
    {
        CheckResAnimConstantDataEqual(data.pConstantArray.Get()[i].ToData(), referenceData.pConstantArray.Get()[i].ToData());
    }

    // 各値
    EXPECT_EQ(data.beginShaderParamCurveIndex, referenceData.beginShaderParamCurveIndex);
    EXPECT_EQ(data.beginTexturePatternCurveIndex, referenceData.beginTexturePatternCurveIndex);
    EXPECT_EQ(data.beginVisibilityCurveIndex, referenceData.beginVisibilityCurveIndex);
    EXPECT_EQ(data.visibilityCurveIndex, referenceData.visibilityCurveIndex);
    EXPECT_EQ(data.visibilityConstantIndex, referenceData.visibilityConstantIndex);
    EXPECT_EQ(data.shaderParamAnimCount, referenceData.shaderParamAnimCount);
    EXPECT_EQ(data.texturePatternAnimCount, referenceData.texturePatternAnimCount);
    EXPECT_EQ(data.constantCount, referenceData.constantCount);
    EXPECT_EQ(data.curveCount, referenceData.curveCount);

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved), 6);
}

void CheckResMaterialAnimDataEqual(const nn::g3d::ResMaterialAnimData& data, const nn::g3d::ResMaterialAnimData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 120);

    // ポインタ
    EXPECT_EQ(std::strcmp(data.pName.Get()->GetData(), referenceData.pName.Get()->GetData()), 0);
    EXPECT_EQ(std::strcmp(data.pPath.Get()->GetData(), referenceData.pPath.Get()->GetData()), 0);
    EXPECT_EQ(data.pBindModel.Get(), referenceData.pBindModel.Get());

    for (int i = 0; i < data.perMaterialAnimCount; ++i)
    {
        EXPECT_EQ(data.pBindIndexArray.Get()[i], referenceData.pBindIndexArray.Get()[i]);
        CheckResPerMaterialAnimDataEqual(data.pPerMaterialAnimArray.Get()[i].ToData(), referenceData.pPerMaterialAnimArray.Get()[i].ToData());
    }
    for (int i = 0; i < data.textureCount; ++i)
    {
        EXPECT_EQ(data.pTextureArray.Get()[i], referenceData.pTextureArray.Get()[i]);
        EXPECT_EQ(std::strcmp(data.pTextureNameArray.Get()[i].Get()->GetData(), referenceData.pTextureNameArray.Get()[i].Get()->GetData()), 0);
        EXPECT_EQ(data.pTextureSlotArray.Get()[i].ToData()->value, referenceData.pTextureSlotArray.Get()[i].ToData()->value);
    }
    for (int i = 0; i < data.userDataCount; ++i)
    {
        CheckResUserDataEqual(data.pUserDataArray.Get()[i], referenceData.pUserDataArray.Get()[i]);
    }
    if (data.userDataCount > 0)
    {
        CheckResDicDataEqual(data.pUserDataDic.Get()->ToData(), referenceData.pUserDataDic.Get()->ToData());
    }

    // 各値
    EXPECT_EQ(data.flag, referenceData.flag);
    EXPECT_EQ(data.userDataCount, referenceData.userDataCount);
    EXPECT_EQ(data.perMaterialAnimCount, referenceData.perMaterialAnimCount);
    EXPECT_EQ(data.curveCount, referenceData.curveCount);
    EXPECT_EQ(data.frameCount, referenceData.frameCount);
    EXPECT_EQ(data.bakedSize, referenceData.bakedSize);
    EXPECT_EQ(data.shaderParamAnimCount, referenceData.shaderParamAnimCount);
    EXPECT_EQ(data.texturePatternAnimCount, referenceData.texturePatternAnimCount);
    EXPECT_EQ(data.visibilityAnimCount, referenceData.visibilityAnimCount);
    EXPECT_EQ(data.textureCount, referenceData.textureCount);
}

void CheckResBoneVisAnimDataEqual(const nn::g3d::ResBoneVisibilityAnimData& data, const nn::g3d::ResBoneVisibilityAnimData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 104);

    // ポインタ
    EXPECT_EQ(std::strcmp(data.pName.Get()->GetData(), referenceData.pName.Get()->GetData()), 0);
    EXPECT_EQ(std::strcmp(data.pPath.Get()->GetData(), referenceData.pPath.Get()->GetData()), 0);
    EXPECT_EQ(data.pBindModel.Get(), referenceData.pBindModel.Get());

    for (int i = 0; i < data.animCount; ++i)
    {
        EXPECT_EQ(data.pBindIndexArray.Get()[i], referenceData.pBindIndexArray.Get()[i]);
        EXPECT_EQ(std::strcmp(data.pNameArray.Get()[i].Get()->GetData(), referenceData.pNameArray.Get()[i].Get()->GetData()), 0);
        EXPECT_EQ(data.pBaseValueArray.Get()[i], referenceData.pBaseValueArray.Get()[i]);
    }
    for (int i = 0; i < data.curveCount; ++i)
    {
        CheckResAnimCurveDataEqual(data.pCurveArray.Get()[i].ToData(), referenceData.pCurveArray.Get()[i].ToData());
    }
    for (int i = 0; i < data.userDataCount; ++i)
    {
        CheckResUserDataEqual(data.pUserDataArray.Get()[i], referenceData.pUserDataArray.Get()[i]);
    }
    if (data.userDataCount > 0)
    {
        CheckResDicDataEqual(data.pUserDataDic.Get()->ToData(), referenceData.pUserDataDic.Get()->ToData());
    }

    EXPECT_EQ(data.flag, referenceData.flag);
    EXPECT_EQ(data.userDataCount, referenceData.userDataCount);
    EXPECT_EQ(data.frameCount, referenceData.frameCount);
    EXPECT_EQ(data.animCount, referenceData.animCount);
    EXPECT_EQ(data.curveCount, referenceData.curveCount);
    EXPECT_EQ(data.bakedSize, referenceData.bakedSize);
}


void CheckResKeyShapeAnimInfoEqual(const nn::g3d::ResKeyShapeAnimInfo& data, const nn::g3d::ResKeyShapeAnimInfo& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 16);

    // ポインタ
    EXPECT_EQ(std::strcmp(data.pName.Get()->GetData(), referenceData.pName.Get()->GetData()), 0);

    EXPECT_EQ(data.curveIndex, referenceData.curveIndex);
    EXPECT_EQ(data.subbindIndex, referenceData.subbindIndex);

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved), 6);
}

void CheckResVertexShapeAnimDataEqual(const nn::g3d::ResVertexShapeAnimData& data, const nn::g3d::ResVertexShapeAnimData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 48);

    // ポインタ
    EXPECT_EQ(std::strcmp(data.pName.Get()->GetData(), referenceData.pName.Get()->GetData()), 0);
    for (int i = 0; i < data.curveCount; ++i)
    {
        CheckResAnimCurveDataEqual(data.pCurveArray.Get()[i].ToData(), referenceData.pCurveArray.Get()[i].ToData());
        CheckResKeyShapeAnimInfoEqual(data.pKeyShapeAnimInfoArray.Get()[i], referenceData.pKeyShapeAnimInfoArray.Get()[i]);
    }
    for (int i = 0; i < data.keyShapeAnimCount; ++i)
    {
        EXPECT_EQ(data.pBaseValueArray.Get()[i], referenceData.pBaseValueArray.Get()[i]);
    }

    EXPECT_EQ(data.curveCount, referenceData.curveCount);
    EXPECT_EQ(data.keyShapeAnimCount, referenceData.keyShapeAnimCount);
    EXPECT_EQ(data.beginCurve, referenceData.beginCurve);
    EXPECT_EQ(data.beginKeyShapeAnim, referenceData.beginKeyShapeAnim);

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved), 4);
}

void CheckResShapeAnimDataEqual(const nn::g3d::ResShapeAnimData& data, const nn::g3d::ResShapeAnimData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 96);

    // ポインタ
    EXPECT_EQ(std::strcmp(data.pName.Get()->GetData(), referenceData.pName.Get()->GetData()), 0);
    EXPECT_EQ(std::strcmp(data.pPath.Get()->GetData(), referenceData.pPath.Get()->GetData()), 0);
    EXPECT_EQ(data.pBindModel.Get(), referenceData.pBindModel.Get());

    for (int i = 0; i < data.vertexShapeAnimCount; ++i)
    {
        EXPECT_EQ(data.pBindIndexArray.Get()[i], referenceData.pBindIndexArray.Get()[i]);
        CheckResVertexShapeAnimDataEqual(data.pVertexShapeAnimArray.Get()[i].ToData(), referenceData.pVertexShapeAnimArray.Get()[i].ToData());
    }
    for (int i = 0; i < data.userDataCount; ++i)
    {
        CheckResUserDataEqual(data.pUserDataArray.Get()[i], referenceData.pUserDataArray.Get()[i]);
    }
    if (data.userDataCount > 0)
    {
        CheckResDicDataEqual(data.pUserDataDic.Get()->ToData(), referenceData.pUserDataDic.Get()->ToData());
    }

    EXPECT_EQ(data.flag, referenceData.flag);
    EXPECT_EQ(data.userDataCount, referenceData.userDataCount);
    EXPECT_EQ(data.vertexShapeAnimCount, referenceData.vertexShapeAnimCount);
    EXPECT_EQ(data.keyShapeAnimCount, referenceData.keyShapeAnimCount);
    EXPECT_EQ(data.frameCount, referenceData.frameCount);
    EXPECT_EQ(data.bakedSize, referenceData.bakedSize);
    EXPECT_EQ(data.curveCount, referenceData.curveCount);

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved), 6);
}

void CheckResCameraAnimDataEqual(const nn::g3d::ResCameraAnimData& data, const nn::g3d::ResCameraAnimData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 72);

    EXPECT_EQ(std::strcmp(data.pName.Get()->GetData(), referenceData.pName.Get()->GetData()), 0);

    for (int i = 0; i < data.curveCount; ++i)
    {
        CheckResAnimCurveDataEqual(data.pCurveArray.Get()[i].ToData(), referenceData.pCurveArray.Get()[i].ToData());
    }
    // BaseValue"Array" とあるがデータサイズは CameraAnimResult 構造体一つ分
    EXPECT_EQ(memcmp(data.pBaseValueArray.Get(), referenceData.pBaseValueArray.Get(), sizeof(nn::g3d::CameraAnimResult)), 0);
    for (int i = 0; i < data.userDataCount; ++i)
    {
        CheckResUserDataEqual(data.pUserDataArray.Get()[i], referenceData.pUserDataArray.Get()[i]);
    }
    if (data.userDataCount > 0)
    {
        CheckResDicDataEqual(data.pUserDataDic.Get()->ToData(), referenceData.pUserDataDic.Get()->ToData());
    }

    EXPECT_EQ(data.flag, referenceData.flag);
    EXPECT_EQ(data.frameCount, referenceData.frameCount);
    EXPECT_EQ(data.curveCount, referenceData.curveCount);
    EXPECT_EQ(data.userDataCount, referenceData.userDataCount);
    EXPECT_EQ(data.bakedSize, referenceData.bakedSize);
}

void CheckResLightAnimDataEqual(const nn::g3d::ResLightAnimData& data, const nn::g3d::ResLightAnimData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 96);

    EXPECT_EQ(std::strcmp(data.pName.Get()->GetData(), referenceData.pName.Get()->GetData()), 0);

    for (int i = 0; i < data.curveCount; ++i)
    {
        CheckResAnimCurveDataEqual(data.pCurveArray.Get()[i].ToData(), referenceData.pCurveArray.Get()[i].ToData());
    }
    int baseValueCount = 0;
    static const int tblDimension[] = { 1, 3, 3, 3, 2, 2, 3, 3 };
    static const int paramCount = sizeof(tblDimension) / sizeof(int);
    for (int idxParam = 0; idxParam < paramCount; ++idxParam)
    {
        if (data.flag & (nn::g3d::ResLightAnim::Flag_BaseEnable << idxParam))
        {
            baseValueCount += tblDimension[idxParam];
        }
    }
    EXPECT_EQ(memcmp(data.pBaseValueArray.Get(), referenceData.pBaseValueArray.Get(), sizeof(float) * baseValueCount), 0);
    for (int i = 0; i < data.userDataCount; ++i)
    {
        CheckResUserDataEqual(data.pUserDataArray.Get()[i], referenceData.pUserDataArray.Get()[i]);
    }
    if (data.userDataCount > 0)
    {
        CheckResDicDataEqual(data.pUserDataDic.Get()->ToData(), referenceData.pUserDataDic.Get()->ToData());
    }
    EXPECT_EQ(std::strcmp(data.pLightType.Get()->GetData(), referenceData.pLightType.Get()->GetData()), 0);
    EXPECT_EQ(std::strcmp(data.pDistAttnFunc.Get()->GetData(), referenceData.pDistAttnFunc.Get()->GetData()), 0);
    EXPECT_EQ(std::strcmp(data.pAngleAttnFunc.Get()->GetData(), referenceData.pAngleAttnFunc.Get()->GetData()), 0);

    EXPECT_EQ(data.flag, referenceData.flag);
    EXPECT_EQ(data.userDataCount, referenceData.userDataCount);
    EXPECT_EQ(data.frameCount, referenceData.frameCount);
    EXPECT_EQ(data.curveCount, referenceData.curveCount);
    EXPECT_EQ(data.lightTypeIndex, referenceData.lightTypeIndex);
    EXPECT_EQ(data.distAttnFuncIndex, referenceData.distAttnFuncIndex);
    EXPECT_EQ(data.angleAttnFuncIndex, referenceData.angleAttnFuncIndex);
    EXPECT_EQ(data.bakedSize, referenceData.bakedSize);
}

void CheckResFogAnimDataEqual(const nn::g3d::ResFogAnimData& data, const nn::g3d::ResFogAnimData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 80);

    EXPECT_EQ(std::strcmp(data.pName.Get()->GetData(), referenceData.pName.Get()->GetData()), 0);

    for (int i = 0; i < data.curveCount; ++i)
    {
        CheckResAnimCurveDataEqual(data.pCurveArray.Get()[i].ToData(), referenceData.pCurveArray.Get()[i].ToData());
    }
    // BaseValue"Array" とあるがデータサイズは CameraAnimResult 構造体一つ分
    EXPECT_EQ(memcmp(data.pBaseValueArray.Get(), referenceData.pBaseValueArray.Get(), sizeof(nn::g3d::FogAnimResult)), 0);
    for (int i = 0; i < data.userDataCount; ++i)
    {
        CheckResUserDataEqual(data.pUserDataArray.Get()[i], referenceData.pUserDataArray.Get()[i]);
    }
    if (data.userDataCount > 0)
    {
        CheckResDicDataEqual(data.pUserDataDic.Get()->ToData(), referenceData.pUserDataDic.Get()->ToData());
    }
    EXPECT_EQ(std::strcmp(data.pDistAttnFunc.Get()->GetData(), referenceData.pDistAttnFunc.Get()->GetData()), 0);

    EXPECT_EQ(data.flag, referenceData.flag);
    EXPECT_EQ(data.frameCount, referenceData.frameCount);
    EXPECT_EQ(data.curveCount, referenceData.curveCount);
    EXPECT_EQ(data.distAttnFuncIndex, referenceData.distAttnFuncIndex);
    EXPECT_EQ(data.userDataCount, referenceData.userDataCount);
    EXPECT_EQ(data.bakedSize, referenceData.bakedSize);
}

void CheckResSceneAnimDataEqual(const nn::g3d::ResSceneAnimData& data, const nn::g3d::ResSceneAnimData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 104);

    // ポインタ
    EXPECT_EQ(std::strcmp(data.pName.Get()->GetData(), referenceData.pName.Get()->GetData()), 0);
    EXPECT_EQ(std::strcmp(data.pPath.Get()->GetData(), referenceData.pPath.Get()->GetData()), 0);

    for (int i = 0; i < data.cameraAnimCount; ++i)
    {
        CheckResCameraAnimDataEqual(data.pCameraAnimArray.Get()[i].ToData(), referenceData.pCameraAnimArray.Get()[i].ToData());
    }
    if (data.cameraAnimCount > 0)
    {
        CheckResDicDataEqual(data.pCameraAnimDic.Get()->ToData(), referenceData.pCameraAnimDic.Get()->ToData());
    }
    for (int i = 0; i < data.lightAnimCount; ++i)
    {
        CheckResLightAnimDataEqual(data.pLightAnimArray.Get()[i].ToData(), referenceData.pLightAnimArray.Get()[i].ToData());
    }
    if (data.lightAnimCount > 0)
    {
        CheckResDicDataEqual(data.pLightAnimDic.Get()->ToData(), referenceData.pLightAnimDic.Get()->ToData());
    }
    for (int i = 0; i < data.fogAnimCount; ++i)
    {
        CheckResFogAnimDataEqual(data.pFogAnimArray.Get()[i].ToData(), referenceData.pFogAnimArray.Get()[i].ToData());
    }
    if (data.fogAnimCount > 0)
    {
        CheckResDicDataEqual(data.pFogAnimDic.Get()->ToData(), referenceData.pFogAnimDic.Get()->ToData());
    }
    for (int i = 0; i < data.userDataCount; ++i)
    {
        CheckResUserDataEqual(data.pUserDataArray.Get()[i], referenceData.pUserDataArray.Get()[i]);
    }
    if (data.userDataCount > 0)
    {
        CheckResDicDataEqual(data.pUserDataDic.Get()->ToData(), referenceData.pUserDataDic.Get()->ToData());
    }

    EXPECT_EQ(data.userDataCount, referenceData.userDataCount);
    EXPECT_EQ(data.cameraAnimCount, referenceData.cameraAnimCount);
    EXPECT_EQ(data.lightAnimCount, referenceData.lightAnimCount);
    EXPECT_EQ(data.fogAnimCount, referenceData.fogAnimCount);
}

void CheckResFileDataEqual(const nn::g3d::ResFileData& data, const nn::g3d::ResFileData& referenceData) NN_NOEXCEPT
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 208);

    // ポインタ
    EXPECT_EQ(std::strcmp(data.pName.Get()->GetData(), referenceData.pName.Get()->GetData()), 0);
    for (int i = 0; i < data.modelCount; ++i)
    {
        CheckResModelDataEqual(data.pModelArray.Get()[i].ToData(), referenceData.pModelArray.Get()[i].ToData());
    }
    if (data.modelCount > 0)
    {
        CheckResDicDataEqual(data.pModelDic.Get()->ToData(), referenceData.pModelDic.Get()->ToData());
    }
    for (int i = 0; i < data.skeletalAnimCount; ++i)
    {
        CheckResSkeletalAnimDataEqual(data.pSkeletalAnimArray.Get()[i].ToData(), referenceData.pSkeletalAnimArray.Get()[i].ToData());
    }
    if (data.skeletalAnimCount > 0)
    {
        CheckResDicDataEqual(data.pSkeletalAnimDic.Get()->ToData(), referenceData.pSkeletalAnimDic.Get()->ToData());
    }
    for (int i = 0; i < data.materialAnimCount; ++i)
    {
        CheckResMaterialAnimDataEqual(data.pMaterialAnimArray.Get()->ToData(), referenceData.pMaterialAnimArray.Get()->ToData());
    }
    if (data.materialAnimCount > 0)
    {
        CheckResDicDataEqual(data.pMaterialAnimDic.Get()->ToData(), referenceData.pMaterialAnimDic.Get()->ToData());
    }
    for (int i = 0; i < data.boneVisAnimCount; ++i)
    {
        CheckResBoneVisAnimDataEqual(data.pBoneVisAnimArray.Get()->ToData(), referenceData.pBoneVisAnimArray.Get()->ToData());
    }
    if (data.boneVisAnimCount > 0)
    {
        CheckResDicDataEqual(data.pBoneVisAnimDic.Get()->ToData(), referenceData.pBoneVisAnimDic.Get()->ToData());
    }
    for (int i = 0; i < data.shapeAnimCount; ++i)
    {
        CheckResShapeAnimDataEqual(data.pShapeAnimArray.Get()->ToData(), referenceData.pShapeAnimArray.Get()->ToData());
    }
    if (data.shapeAnimCount > 0)
    {
        CheckResDicDataEqual(data.pShapeAnimDic.Get()->ToData(), referenceData.pShapeAnimDic.Get()->ToData());
    }
    for (int i = 0; i < data.sceneAnimCount; ++i)
    {
        CheckResSceneAnimDataEqual(data.pSceneAnimArray.Get()->ToData(), referenceData.pSceneAnimArray.Get()->ToData());
    }
    if (data.sceneAnimCount > 0)
    {
        CheckResDicDataEqual(data.pSceneAnimDic.Get()->ToData(), referenceData.pSceneAnimDic.Get()->ToData());
    }
    // ポインタの示す先は異なるので比較しない
    //EXPECT_EQ(data.pBufferMemoryPool.Get(), referenceData.pBufferMemoryPool.Get());
    for (int modelIndex = 0; modelIndex < data.modelCount; ++modelIndex)
    {
        const nn::g3d::ResModel& model = data.pModelArray.Get()[modelIndex];
        for (int vertexIndex = 0; vertexIndex < model.GetVertexCount(); ++vertexIndex)
        {
            EXPECT_EQ(data.pBufferMemoryPool.Get(), model.GetVertex(vertexIndex)->ToData().pMemoryPool.Get());
        }
        for (int shapeIndex = 0; shapeIndex < model.GetShapeCount(); ++shapeIndex)
        {
            for (int meshIndex = 0; meshIndex < model.GetShape(shapeIndex)->GetMeshCount(); ++meshIndex)
            {
                EXPECT_EQ(data.pBufferMemoryPool.Get(), model.GetShape(shapeIndex)->GetMesh(meshIndex)->ToData().pMemoryPool.Get());
            }
        }
    }
    // MemoryPoolInfo 内の pMemory の指す先が異なるので比較しない
    //EXPECT_EQ(memcmp(data.pBufferMemoryPoolInfo.Get(), referenceData.pBufferMemoryPoolInfo.Get(), sizeof(nn::gfx::MemoryPool::InfoType)), 0);

    for (int i = 0; i < data.externalFileCount; ++i)
    {
        EXPECT_EQ(data.pExternalFileArray.Get()[i].GetSize(), referenceData.pExternalFileArray.Get()[i].GetSize());
        // ポインタの指す先が異なるので比較しない
        //EXPECT_EQ(memcmp(&data.pExternalFileArray.Get()[i], &referenceData.pExternalFileArray.Get()[i], data.pExternalFileArray.Get()[i].GetSize()), 0);
    }
    if (data.externalFileCount > 0)
    {
        CheckResDicDataEqual(data.pExternalFileDic.Get()->ToData(), referenceData.pExternalFileDic.Get()->ToData());
    }

    // ポインタの指す先が異なるので比較しない
    //EXPECT_EQ(data.pStringPool.Get(), referenceData.pStringPool.Get());
    EXPECT_EQ(memcmp(data.pStringPool.Get(), referenceData.pStringPool.Get(), data.sizeStringPool), 0);

    // 各値
    EXPECT_EQ(data.sizeStringPool, referenceData.sizeStringPool);
    EXPECT_EQ(data.modelCount, data.modelCount);
    EXPECT_EQ(data.skeletalAnimCount, referenceData.skeletalAnimCount);
    EXPECT_EQ(data.materialAnimCount, referenceData.materialAnimCount);
    EXPECT_EQ(data.boneVisAnimCount, referenceData.boneVisAnimCount);
    EXPECT_EQ(data.shapeAnimCount, referenceData.shapeAnimCount);
    EXPECT_EQ(data.sceneAnimCount, referenceData.sceneAnimCount);
    EXPECT_EQ(data.externalFileCount, referenceData.externalFileCount);

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved), 6);
}
