﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testG3d_FrameCapture.h"

#include <cstdlib>
#include <cstring>
#include <malloc.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nnt/graphics/testGraphics_Png.h>

namespace nnt{ namespace g3d{

    FrameCapture::FrameCapture()
        : m_IsInitialized(false),
          m_AllocateFunction(DefaultAllocateFunction),
          m_AllocateUserParam(NULL),
          m_DeallocateFunction(DefaultDeallocateFunction),
          m_DeallocateUserParam(NULL),
          m_ScreenWidth(0),
          m_ScreenHeight(0),
          m_pDevice(NULL),
          m_pMemoryPool(NULL),
          m_CapturedImageBuffer(NULL),
          m_CapturedImageBufferSize(0),
          m_CompressionWorkBuffer(NULL),
          m_CompressionWorkBufferSize(0),
          m_CompressedImageBuffer(NULL),
          m_CompressedImageBufferSize(0)
    {
    }

    FrameCapture::~FrameCapture()
    {
        this->Finalize();
    }

    //-------------------------------------------

    void* FrameCapture::DefaultAllocateFunction(size_t size, int alignment, void* userParam)
    {
        NN_UNUSED(alignment);
        NN_UNUSED(userParam);
        return static_cast<void*>(new char[size]);
    }
    void FrameCapture::DefaultDeallocateFunction(void* ptr, void* userParam)
    {
        NN_UNUSED(userParam);
        char* ptr2 = static_cast<char*>(ptr);
        delete[] ptr2;
    }

    void* FrameCapture::AllocateMemory(size_t size, int alignment)
    {
        return m_AllocateFunction(size, alignment, m_AllocateUserParam);
    }
    void FrameCapture::DeallocateMemory(void* ptr)
    {
        m_DeallocateFunction(ptr, m_DeallocateUserParam);
    }

    //-------------------------------------------

    void FrameCapture::Initialize(
        nn::gfx::Device* pDevice,
        nn::gfx::MemoryPool* pMemoryPool,
        nn::util::BytePtr* pMemoryPoolBase,
        nn::util::BytePtr* pMemoryPoolCurrent,
        int screenWidth,
        int screenHeight)
    {
        m_ScreenWidth  = screenWidth;
        m_ScreenHeight = screenHeight;
        m_pDevice      = pDevice;
        m_pMemoryPool  = pMemoryPool;

        size_t imageBufferSize = screenWidth * screenHeight * 4;
        {
            nn::gfx::Buffer::InfoType info;
            info.SetDefault();
            info.SetSize(imageBufferSize);
            info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);

            if (NN_STATIC_CONDITION(nn::gfx::Texture::IsMemoryPoolRequired))
            {
                pMemoryPoolCurrent->AlignUp(nn::gfx::Buffer::GetBufferAlignment(pDevice, info));
                size_t size = info.GetSize();
                m_CaptureBuffer.Initialize(pDevice, info, pMemoryPool, pMemoryPoolBase->Distance(pMemoryPoolCurrent->Get()), size);

                pMemoryPoolCurrent->Advance(static_cast<ptrdiff_t>(size));
            }
            else
            {
                m_CaptureBuffer.Initialize(pDevice, info, NULL, 0, 0);
            }
        }

        //size_t imageBufferSize = screenWidth * screenHeight * 4;
        m_CapturedImageBuffer = static_cast<uint8_t*>(AllocateMemory(imageBufferSize, NN_ALIGNOF(std::max_align_t)));
        m_CapturedImageBufferSize = imageBufferSize;

        m_CompressedImageBuffer = AllocateMemory(imageBufferSize, NN_ALIGNOF(std::max_align_t));
        m_CompressedImageBufferSize = imageBufferSize;

        m_IsInitialized = true;
    }

    void FrameCapture::Finalize()
    {
        if(!m_IsInitialized)
        {
            return;
        }
        m_CaptureBuffer.Finalize(m_pDevice);

        DeallocateMemory(m_CapturedImageBuffer);
        m_CapturedImageBuffer = NULL;
        m_CapturedImageBufferSize = 0;

        DeallocateMemory(m_CompressionWorkBuffer);
        m_CompressionWorkBuffer = NULL;
        m_CompressionWorkBufferSize = 0;

        DeallocateMemory(m_CompressedImageBuffer);
        m_CompressedImageBuffer = NULL;
        m_CompressedImageBufferSize = 0;

        m_IsInitialized = false;
    }

    //-------------------------------------------

    void FrameCapture::PushCaptureCommand(nn::gfx::CommandBuffer* pCommandBuffer, nn::gfx::Texture* pSrcTexture)
    {
        nn::gfx::TextureCopyRegion region;
        region.SetDefault();
        region.SetWidth(m_ScreenWidth);
        region.SetHeight(m_ScreenHeight);
        pCommandBuffer->CopyImageToBuffer(&m_CaptureBuffer, 0, pSrcTexture, region);
    }

    //-------------------------------------------

    void FrameCapture::FetchCaptureResult()
    {
        m_CaptureBuffer.InvalidateMappedRange( 0, 4 * m_ScreenWidth * m_ScreenHeight);

        const uint8_t* pBuffer = static_cast<uint8_t*>(m_CaptureBuffer.Map());
        size_t stride = 4 * m_ScreenWidth;

        int yDst = m_ScreenHeight - 1;
        for (int ySrc = 0; ySrc < m_ScreenHeight; ySrc++, yDst--)
        {
            const uint8_t* pSrc = pBuffer + stride * ySrc;
            uint8_t* pDst = m_CapturedImageBuffer + stride * yDst;

            int xDst = m_ScreenWidth;
            for (int xSrc = 0; xSrc < m_ScreenWidth; xSrc++, xDst--)
            {
                // R
                *pDst = *pSrc;
                pDst++;
                pSrc++;
                // G
                *pDst = *pSrc;
                pDst++;
                pSrc++;
                // B
                *pDst = *pSrc;
                pDst++;
                pSrc++;
                // A
                *pDst = 0xFF;
                pDst++;
                pSrc++;
            }
        }

        m_CaptureBuffer.Unmap();
    }

    //-------------------------------------------
    void* PngIoAllocateMemory(size_t size, size_t alignment, void* pArgument)
    {
        NN_UNUSED(alignment);
        NN_UNUSED(pArgument);
        return malloc(size);
    }

    void PngIoFreeMemory(void* ptr, void* pArgument)
    {
        NN_UNUSED(pArgument);
        free(ptr);
    }

    void FrameCapture::SaveToPng(const char* destPath)
    {
        // 作業用アロケータ
        nnt::graphics::AllocatorFunction allocator;
        allocator.allocateFunction = PngIoAllocateMemory;
        allocator.freeFunction = PngIoFreeMemory;

        // png設定
        size_t imgDataSize = m_ScreenWidth * m_ScreenHeight * 4;
        nnt::graphics::PngIhdr ihdr;
        ihdr.SetDefault();
        ihdr.bitDepth = 8;
        ihdr.channels = 4;
        ihdr.colorType = nnt::graphics::PngColorType_RgbAlpha;
        ihdr.width = m_ScreenWidth;
        ihdr.height = m_ScreenHeight;

        // 保存
        nnt::graphics::PngIO::Write(
            destPath,
            m_CapturedImageBuffer,
            imgDataSize,
            &ihdr,
            &allocator);
    }

}}

