﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <stddef.h>
#include <string>

namespace nnt { namespace g3d {
    class CaptureSetting
    {
    public:
        struct Camera
        {
            float posX;
            float posY;
            float posZ;
            float targetX;
            float targetY;
            float targetZ;
            float frame; // 0 未満は未指定扱いとする
            int shapeIndex; // 0 未満は未指定扱いとする
            int sumeshIndex; // 0 未満は未指定扱いとする

            Camera()
                : frame(-1.0f)
                , shapeIndex(-1)
                , sumeshIndex(-1)
            {
            }
        };

        struct Resource
        {
            char* path;
        };

        struct Model
        {
            char* name;
            float posX;
            float posY;
            float posZ;
            bool isAutoAnimBindEnabled;
            char* autoBindAnimName;
            bool isMirroringEnabled;
            char* retargetingHostName;

            Model()
                : isAutoAnimBindEnabled(false)
                , isMirroringEnabled(false)
            {
            }
        };

    public:
        CaptureSetting() NN_NOEXCEPT;

        void LoadSetting(const char* path) NN_NOEXCEPT;

        const char* GetOutputImageName() const NN_NOEXCEPT
        {
            return m_OutputImageName;
        }

        int GetLodLevel() const NN_NOEXCEPT
        {
            return m_LodLevel;
        }

        int GetResourceCount() const NN_NOEXCEPT
        {
            return m_ResourceCount;
        }

        int GetModelCount() const NN_NOEXCEPT
        {
            return m_ModelCount;
        }

        int GetCameraCount() const NN_NOEXCEPT
        {
            return m_CameraCount;
        }

        const Resource* GetResource(int index) const NN_NOEXCEPT
        {
            return &m_Resources[index];
        }

        const Model* GetModel(int index) const NN_NOEXCEPT
        {
            return &m_Models[index];
        }

        const Model* GetModel(const char* name) const NN_NOEXCEPT
        {
            for (int modelIndex = 0; modelIndex < m_ModelCount; ++modelIndex)
            {
                const Model* pModelSetting = &m_Models[modelIndex];
                if (strcmp(name, pModelSetting->name) == 0)
                {
                    return pModelSetting;
                }
            }

            return NULL;
        }

        const Camera* GetCamera(int index) const NN_NOEXCEPT
        {
            return &m_Cameras[index];
        }

    private:
        void ParseStringToModelSetting(char* pSourceString) NN_NOEXCEPT;
        void ParseStringToCameraSetting(char* pSourceString) NN_NOEXCEPT;
        void ParseStringToSettingNameAndValue(char** pOutName, char** pOutValue, char* pSourceString) NN_NOEXCEPT;

    private:
        static const int ResourceCountMax = 32;
        static const int ModelCountMax = 32;
        static const int CameraCountMax = 16;
        static const size_t ReadBufferSizeMax = 4096;

        char* m_OutputImageName;
        int m_LodLevel;
        int m_ResourceCount;
        int m_ModelCount;
        int m_CameraCount;
        Resource m_Resources[ResourceCountMax];
        Model m_Models[ModelCountMax];
        Camera m_Cameras[CameraCountMax];
        char m_ReadStringBuffer[ReadBufferSizeMax];
        char* m_pCurrentReadStringBuffer;
        size_t m_CurrentReadStringBufferSize;
    };

}}
