﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nnt.h>

#include "testEft_ProgramOptions.h"
#include "testEft_Capture.h"
#include "framework/testEft_TestHeap.h"

//---------------------------------------------------------------------------
//
//---------------------------------------------------------------------------
static const uint32_t   MemSizeForEffect    = 256 * 1024 * 1024;                          // システム用メモリサイズ
static const uint32_t   MemSizeForGfx       = 256 * 1024 * 1024;                          // GFX用メモリサイズ

extern nnt::eft::ProgramOptions                   g_ProgramOptions;
static nnt::eft::TestHeap                         g_TestHeap;                             // テスト用ヒープ
static nnt::eft::TextureDescriptorIndexAllocator  g_TextureDescPoolAllocator;             //!< テクスチャデスクリプタインデックスアロケータ
static nnt::eft::SamplerDescriptorIndexAllocator  g_SamplerDescPoolAllocator;             //!< サンプラーデスクリプタインデックスアロケータ


//------------------------------------------------------------------------------
//  テクスチャディスクリプタスロットをアロケート
//------------------------------------------------------------------------------
bool AllocateSlotForFontTexture_( nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::TextureView& textureView, void* pUserData )
{
    return g_TextureDescPoolAllocator.AllocateSlotForFontTexture( dstSlot, textureView, pUserData );
}

//------------------------------------------------------------------------------
//  Font サンプラディスクリプタスロットをアロケート
//------------------------------------------------------------------------------
bool AllocateSlotForFontSampler_(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    return g_SamplerDescPoolAllocator.AllocateSlotForFontSampler( dstSlot, sampler, pUserData );
}

//---------------------------------------------------------------------------
// メイン
//---------------------------------------------------------------------------
TEST(ScreenCaptureGfx, All)
{
    // メモリ関連の初期化
    NN_LOG("Initialize memory heap.\n");
    nn::mem::StandardAllocator allocatorForEft;
    nn::mem::StandardAllocator allocatorForGfx;
    nn::util::BytePtr pEftMemoryHeap(NULL);
    nn::util::BytePtr pGfxMemoryHeap(NULL);
    {
        // エフェクトシステム用メモリブロックの取得
        pEftMemoryHeap.Reset(new uint8_t[MemSizeForEffect]);
        NN_ASSERT(pEftMemoryHeap.Get());
        allocatorForEft.Initialize(pEftMemoryHeap.Get(), MemSizeForEffect);

        pGfxMemoryHeap.Reset(new uint8_t[MemSizeForGfx]);
        NN_ASSERT(pGfxMemoryHeap.Get());
        allocatorForGfx.Initialize(pGfxMemoryHeap.Get(), MemSizeForGfx);
    }
    // ヒープを設定
    g_TestHeap.SetStandardAllocator( &allocatorForEft );

    // ファイルシステム初期化
    NN_LOG("Initialize filesystem\n");
    nn::fs::MountHostRoot();

    // 描画システムの初期化
    NN_LOG("Initialize render system\n");
    nnt::eft::EftRenderSystem renderSystem;                       // 描画システム
    renderSystem.Initialize(&allocatorForGfx, g_ProgramOptions.screenWidth, g_ProgramOptions.screenHeight);

    g_TextureDescPoolAllocator.SetDescriptorPool( renderSystem.GetTextureDescriptorPool() );
    g_SamplerDescPoolAllocator.SetDescriptorPool( renderSystem.GetSamplerDescriptorPool() );

    renderSystem.InitializeDescriptors( &g_TextureDescPoolAllocator );

    // エフェクトシステムを初期化
    NN_LOG("Initialize effect system\n");
    nnt::eft::EffectSystem effectSystem;                          // エフェクトシステム
    effectSystem.Initialize( &g_TestHeap, &renderSystem );
    effectSystem.GetSystem()->RegisterSamplerToDescriptorPool( AllocateSlotForFontSampler_, NULL );
    effectSystem.GetSystem()->RegisterTextureViewToDescriptorPool( AllocateSlotForFontTexture_, NULL );

    // シードの設定
    nn::vfx::detail::Random::SetGlobalSeed( true, 32 );

    // エフェクトリソースの読み込み
    NN_LOG("Load effect resource\n");
    uint32_t resId = effectSystem.EntryResource( g_ProgramOptions.ptclPath.GetString() );
    effectSystem.GetSystem()->GetResource(resId)->RegisterTextureViewToDescriptorPool( AllocateSlotForFontTexture_, NULL );

    // 0番目のエミッタセットを生成
    effectSystem.CreateEmitterSet( 0, 0/*resId*/, 0 );
    NN_UNUSED(resId);

    // フレームキャプチャの生成
    nnt::eft::FrameCapture frameCapture;
    frameCapture.Initialize(
        renderSystem.GetDevice(),
        renderSystem.GetMemoryPoolAllocator(),
        renderSystem.GetInvisibleMemoryPoolAllocator(),
        g_ProgramOptions.screenWidth,
        g_ProgramOptions.screenHeight);

    // キャプチャ
    NN_LOG("Start capture\n");
    nnt::eft::TestEftCapture testCapture;                         // キャプチャー
    testCapture.Initialize( &renderSystem, &effectSystem, &frameCapture );

    testCapture.Run(g_ProgramOptions.outDir.GetString(), g_ProgramOptions.numberOfCapturingFrames, g_ProgramOptions.capturingFrames);

    // 解放処理
    NN_LOG("End capture\n");
    testCapture.Finalize();
    frameCapture.Finalize();

    // リソース破棄
    NN_LOG("Finalize resource\n");
    effectSystem.ClearResource();

    // エフェクトシステムの終了処理
    NN_LOG("Finalize effect system\n");
    effectSystem.Finalize();

    // 描画システムの終了処理
    NN_LOG("Finalize render system\n");
    renderSystem.Finalize();

    NN_LOG("Finalize filesystem\n");
    nn::fs::UnmountHostRoot();

    // アロケータ終了処理
    NN_LOG("Finalize memory heap\n");
    allocatorForEft.Finalize();
    allocatorForGfx.Finalize();

    // ヒープメモリーの終了処理
    delete[] reinterpret_cast<uint8_t*>(pEftMemoryHeap.Get());
    delete[] reinterpret_cast<uint8_t*>(pGfxMemoryHeap.Get());

    NN_LOG("succeed\n");
    SUCCEED();
}
