﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testEft_ProgramOptions.h"

#include <cstdlib>
#include <nn/nn_Assert.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>

namespace nnt{ namespace eft{
    namespace{
        static const int MaxOptionNameLength = 10;

        static bool IsEqualString(const char* str0, const char* str1, int maxLength = MaxOptionNameLength)
        {
            int len0 = nn::util::Strnlen(str0, maxLength + 1);
            int len1 = nn::util::Strnlen(str1, maxLength + 1);
            if(len0 > maxLength || len1 > maxLength || len0 != len1)
            {
                return false;
            }
            return nn::util::Strncmp(str0, str1, maxLength + 1) == 0;
        }
    }

    void ProgramOptions::Analyze(ProgramOptions* pOutValue, int argc, char** argv)
    {
        // 引数解析
        for (int i = 1; i < argc; i++)
        {
            if (IsEqualString(argv[i], "-outDir"))
            {
                NN_ASSERT(i + 1 < argc);
                pOutValue->outDir.SetString(argv[i + 1]);
                pOutValue->outDir.NormalizeAssign();
                i++;
                continue;
            }

            // エフェクトリソース
            if (IsEqualString(argv[i], "-ptcl"))
            {
                NN_ASSERT(i + 1 < argc);
                pOutValue->ptclPath.SetString(argv[i + 1]);
                pOutValue->ptclPath.NormalizeAssign();
                i++;
                continue;
            }

            // 描画するフレーム指定
            if (IsEqualString(argv[i], "-frame"))
            {
                static const int MaxFrameStringLength = 1023;
                NN_ASSERT(i + 1 < argc);
                char* pFrameStr = const_cast<char*>(argv[i + 1]);
                i++;
                int length = nn::util::Strnlen(pFrameStr, MaxFrameStringLength + 1);
                NN_ASSERT(length <= MaxFrameStringLength, "Too long capturing frame specification");

                int frame = -1;
                pOutValue->numberOfCapturingFrames = 0;
                for(int pos = 0; pos < length; pos++)
                {
                    char c = pFrameStr[pos];
                    if(c >= '0' && c <= '9')
                    {
                        NN_ASSERT(pOutValue->numberOfCapturingFrames < MaxNumberOfCapturingFrames, "Too may capturing frames");
                        if(frame < 0)
                        {
                            frame = c - '0';
                        }
                        else
                        {
                            frame *= 10;
                            frame += c - '0';
                        }
                    }
                    else
                    {
                        if(frame >= 0)
                        {
                            pOutValue->capturingFrames[pOutValue->numberOfCapturingFrames] = static_cast<float>(frame);
                            pOutValue->numberOfCapturingFrames++;
                        }
                        frame = -1;
                    }
                }
                if(frame >= 0)
                {
                    pOutValue->capturingFrames[pOutValue->numberOfCapturingFrames] = static_cast<float>(frame);
                    pOutValue->numberOfCapturingFrames++;
                }
                continue;
            }

            if (IsEqualString(argv[i], "-width"))
            {
                NN_ASSERT(i + 1 < argc);
                pOutValue->screenWidth = std::atoi(argv[i + 1]);
                i++;
                continue;
            }

            if (IsEqualString(argv[i], "-height"))
            {
                NN_ASSERT(i + 1 < argc);
                pOutValue->screenHeight = std::atoi(argv[i + 1]);
                i++;
                continue;
            }

        }
    }

}}
