﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/gfx.h>
#include <nn/vi.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include "testEft_MemoryPoolAllocator.h"
#include "testEft_DescriptorIndexAllocator.h"

namespace nnt{
namespace eft{

/**
 * @brief 描画システムクラス
 *
 * @details Windows版のみの実装です
 */
class EftRenderSystem
{
public:
        //static const nn::gfx::ImageFormat ColorBufferImageFormat = nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm;
        //static const nn::gfx::ImageFormat ColorBufferImageFormat = nn::gfx::ImageFormat_R16_G16_B16_A16_Float;
        static const nn::gfx::ImageFormat ColorBufferImageFormat = nn::gfx::ImageFormat_R32_G32_B32_A32_Float;

public:
    /**
     * @brief コンストラクタ
     *
     */
    EftRenderSystem();

    /**
     * @brief デストラクタ
     *
     */
    ~EftRenderSystem();

    /**
     * @brief 初期化
     *
     * @param[in] allocator     アロケータ
     * @return     成功したときtrueを返します。
     */
    bool Initialize(
        nn::mem::StandardAllocator* pAllocator,
        int width,
        int height);

    /**
     * @brief 解放処理
     *
     */
    void Finalize();

   /**
     * @brief ディスプレイの取得
     *
     * @return     ディスプレイのインスタンス
     */
    nn::vi::Display* GetDisplay() { return m_Display; }

   /**
     * @brief レイヤの取得
     *
     * @return     レイヤのインスタンス
     */
    nn::vi::Layer* GetLayer() { return m_Layer; }

    /**
     * @brief デバイスの取得
     *
     * @return     デバイスのインスタンス
     */
    nn::gfx::Device* GetDevice() { return &m_Device; }

    /**
     * @brief スクリーン幅の取得
     *
     * @return     スクリーン幅
     */
    int GetScreenWidth() const { return m_ScreenWidth; }

    /**
     * @brief スクリーン高さの取得
     *
     * @return      スクリーン高さ
     */
    int GetScreenHeight() const{ return m_ScreenHeight; }

    /**
     * @brief スワップチェーンの取得
     *
     * @return     スワップチェーンのインスタンス
     */
    nn::gfx::SwapChain* GetCurrentSwapChain() { return m_CurrentSwapChain; }

    /**
     * @brief キューの取得
     *
     * @return     キューのインスタンス
     */
    nn::gfx::Queue* GetQueue() { return &m_Queue; }

    /**
     * @brief コマンドバッファの取得
     *
     * @return     コマンドバッファのインスタンス
     */
    nn::gfx::CommandBuffer* GetCommandBuffer() { return &m_CommandBuffer; }

    /**
     * @brief メモリプールの取得
     *
     * @return メモリプールのインスタンス
     */
    MemortPoolAllocator* GetMemoryPoolAllocator() { return &m_MemoryPoolAllocator; }

    /**
     * @brief メモリプールの取得
     *
     * @return メモリプールのインスタンス
     */
    MemortPoolAllocator* GetInvisibleMemoryPoolAllocator() { return &m_InvisibleMemPoolAllocator; }

    /**
     * @brief ビューポートシザーステートの取得
     *
     * @return     ビューポートシザーステートのインスタンス
     */
    nn::gfx::ViewportScissorState* GetViewportScissorState() { return &m_ViewportScissor; }

    /**
     * @brief ラスタライザステートの取得
     *
     * @return     ラスタライザステートのインスタンス
     */
    nn::gfx::RasterizerState* GetRasterizerState() { return &m_RasterizerState; }

    /**
     * @brief 深度ステンシルステートの取得
     *
     * @return     深度ステンシルステートのインスタンス
     */
    nn::gfx::DepthStencilState* GetDepthStencilState() { return &m_DepthStencilState; }

    /**
     * @brief カラーバッファの取得
     *
     * @return      カラーバッファのインスタンス
     */
    nn::gfx::Texture* GetColorBuffer() { return &m_ColorBuffer; }

    /**
     * @brief カラーバッファビューの取得
     *
     * @return     カラーバッファビューのインスタンス
     */
    nn::gfx::ColorTargetView* GetColorTargetView() { return &m_ColorBufferView; }

    /**
     * @brief 深度ステンシルビューの取得
     *
     * @return     深度ステンシルビューのインスタンス
     */
    nn::gfx::DepthStencilView* GetDepthStencilView() { return &m_DepthStencilView; }

    /**
     * @brief 同期フェンスの取得
     *
     * @return     同期フェンスのインスタンス
     */
    nn::gfx::Fence* GetFence() { return &m_Fence; }

    /**
     * @brief テクスチャデスクリプタプールの取得
     *
     * @return     デスクリプタインデックスアロケータ
     */
    nn::gfx::DescriptorPool* GetTextureDescriptorPool() { return &m_TextureDescriptorPool; }

    /**
     * @brief サンプラデスクリプタプールの取得
     *
     * @return     デスクリプタインデックスアロケータ
     */
    nn::gfx::DescriptorPool* GetSamplerDescriptorPool() { return &m_SamplerDescriptorPool; }

    /**
     * @brief コマンドバッファのリセット
     *
     */
    void ResetCommandBuffer();

   /**
     * @brief ディスクリプタの初期化
     *
     */
    void InitializeDescriptors( TextureDescriptorIndexAllocator* textureDescPoolAllocator );

    /**
     * @brief カラーバッファ用テクスチャの取得
     */
    nn::gfx::Texture* GetColorTextureView() { return &m_ColorBuffer; }

    /**
     * @brief 深度ステンシルバッファ用テクスチャの取得
     */
    nn::gfx::Texture* GetDepthStencilBuffer() { return &m_DepthStencilBuffer; }

    /**
     * @brief カラーコピーバッファの取得
     */
    nn::gfx::Texture* GetColorCopyBuffer(){ return &m_ColorCopyBuffer; }

    /**
     * @brief デプス・ステンシルコピーバッファの取得
     */
    nn::gfx::Texture* GetDepthStencilCopyBuffer(){ return &m_DepthStencilCopyBuffer; }

    /**
     * @brief テクスチャコピー
     */
    void CopyTextureBuffer( nn::gfx::Texture* dstTexture, nn::gfx::Texture* srcTexture );

    /**
     * @brief カラーデスクリプタスロットの取得
     */
    nn::gfx::DescriptorSlot* GetColorDescSlot(){ return &m_ColorDescSlot; }

    /**
     * @brief 深度ステンシルデスクリプタスロットの取得
     */
    nn::gfx::DescriptorSlot* GetDepthStencilDescSlot(){ return &m_DepthStencilDescSlot; }

    /**
     * @brief カラーコピーバッファデスクリプタスロットの取得
     */
    nn::gfx::DescriptorSlot* GetColorCopyBufferDescSlot(){ return &m_ColorCopyBufferDescSlot; }

    /**
     * @brief デプス・ステンシルコピーデスクリプタスロットの取得
     */
    nn::gfx::DescriptorSlot* GetDepthStencilCopyBufferDescSlot(){ return &m_DepthStencilCopyBufferDescSlot; }

private:
    /**
     * @brief メモリプールの初期化
     *
     */
    void InitializeMemoryPool();

    /**
     * @brief レイヤの初期化
     *
     */
    void InitializeLayer();

    /**
     * @brief デバイスの初期化
     *
     */
    void InitializeDevice();

    /**
     * @brief スワップチェーンの初期化
     *
     */
    void InitializeSwapChain( int width, int height );

    /**
     * @brief キューの初期化
     *
     */
    void InitializeQueue();

    /**
     * @brief コマンドバッファの初期化
     *
     */
    void InitializeCommandBuffer();
    void InitializeNestedCommandBuffer();

    /**
     * @brief ビューポートシザーの初期化
     *
     */
    void InitializeViewportScissor();

    /**
     * @brief ラスタライザの初期化
     *
     */
    void InitializeRasterizerState();

    /**
     * @brief 深度ステンシルステートの初期化
     *
     */
    void InitializeDepthStencilState();

    /**
     * @brief カラーバッファ用のテクスチャの初期化
     *
     */
    void InitializeColorBuffer();

    /**
     * @brief カラーコピーバッファ用のテクスチャの初期化
     *
     */
    void InitializeColorCopyBuffer();

    /**
     * @brief カラーバッファビューの初期化
     *
     */
    void InitializeColorBufferView();

    /**
     * @brief 深度ステンシルバッファ用のテクスチャの初期化
     *
     */
    void InitializeDepthStencilBuffer();

    /**
     * @brief 深度ステンシルコピーバッファ用のテクスチャの初期化
     *
     */
    void InitializeDepthStencilCopyBuffer();

    /**
     * @brief 深度ステンシルビューの初期化
     *
     */
    void InitializeDepthStencilView();

    /**
     * @brief 同期フェンスの初期化
     *
     */
    void InitializeFence();

private:
    nn::mem::StandardAllocator*      m_pAllocator;           //!< メモリアロケータ

    nn::vi::Display*                 m_Display;             //!< ディスプレイハンドル
    nn::vi::Layer*                   m_Layer;               //!< レイヤハンドル

    nn::gfx::Device                  m_Device;              //!< デバイス
    nn::gfx::SwapChain               m_RgbSwapChain;        //!< ノンリニアスワップチェーン
    nn::gfx::SwapChain               m_SrgbSwapChain;       //!< リニアスワップチェーン
    nn::gfx::SwapChain*              m_CurrentSwapChain;    //!< カレントスワップチェーン
    nn::gfx::Queue                   m_Queue;               //!< キュー
    nn::gfx::CommandBuffer           m_CommandBuffer;       //!< コマンドバッファ
    nn::gfx::ViewportScissorState    m_ViewportScissor;     //!< ビューポートシザー
    nn::gfx::RasterizerState         m_RasterizerState;     //!< ライスタライザ
    nn::gfx::DepthStencilState       m_DepthStencilState;   //!< 深度ステンシルステート

    nn::gfx::Texture                 m_ColorBuffer;         //!< カラーバッファ用テクスチャ
    nn::gfx::ColorTargetView         m_ColorBufferView;     //!< カラーバッファビュー
    nn::gfx::TextureView             m_ColorTextureView;
    nn::gfx::DescriptorSlot          m_ColorDescSlot;       //!< カラーデスクリプタスロット

    nn::gfx::Texture                m_ColorCopyBuffer;              //!< カラーコピーバッファ
    nn::gfx::TextureView            m_ColorCopyBufferTextureView;   //!< カラーコピーバッファテクスチャビュー
    nn::gfx::DescriptorSlot         m_ColorCopyBufferDescSlot;      //!< カラーコピーバッファデスクリプタスロット

    nn::gfx::Texture                 m_DepthStencilBuffer;          //!< 深度ステンシルバッファ用テクスチャ
    nn::gfx::DepthStencilView        m_DepthStencilView;            //!< 深度ステンシルビュー
    nn::gfx::TextureView             m_DepthStencilTextureView;
    nn::gfx::DescriptorSlot          m_DepthStencilDescSlot;        //!< 深度ステンシルデスクリプタスロット

    nn::gfx::Texture                m_DepthStencilCopyBuffer;               //!< デプス・ステンシルコピーバッファ
    nn::gfx::TextureView            m_DepthStencilCopyBufferTextureView;    //!< デプス・ステンシルコピーテクスチャビュー
    nn::gfx::DescriptorSlot         m_DepthStencilCopyBufferDescSlot;       //!< デプス・ステンシルコピーデスクリプタスロット

    nn::gfx::Fence                   m_Fence;                       //!< 同期フェンス

    MemortPoolAllocator             m_MemoryPoolAllocator;          //!< メモリプールアロケータ
    MemortPoolAllocator             m_CommandBuffersMemoryPool;     //!< コマンドバッファメモリプール
    MemortPoolAllocator             m_InvisibleMemPoolAllocator;    //!< メモリプールアロケータ

    nn::gfx::CommandBuffer          m_NestedCommandBuffer;          //!< コマンドバッファ
    nn::gfx::DescriptorPool         m_TextureDescriptorPool;        //!< テクスチャデスクリプタプール
    nn::gfx::DescriptorPool         m_SamplerDescriptorPool;        //!< サンプラーデスクリプタプール
    int                             m_ScreenWidth;                  //!< スクリーン幅
    int                             m_ScreenHeight;                 //!< スクリーン高さ
    int                             m_Width;                        //!< デバイスの幅
    int                             m_Height;                       //!< デバイスの高さ
    void*                           m_Hwnd;                         //!< ウィンドウハンドル

};

}
}
