﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/vfx/vfx_System.h>
#include "testEft_RenderSystem.h"

namespace nnt{
namespace eft{

/**
 * @brief エフェクトシステムクラス
 *
 * @details Windows版のみの実装です
 */
class EffectSystem
{
public:
    /**
     * @brief コンストラクタ
     *
     */
    EffectSystem();

    /**
     * @brief デストラクタ
     *
     */
    ~EffectSystem();

    /**
     * @brief 初期化
     *
     * @param[in] allocator         アロケータ
     * @param[in] pRenderSystem     描画システム
     * @return     成功したときtrueを返します。
     */
    bool Initialize( nn::vfx::Heap* pHeap, EftRenderSystem* pRenderSystem );

    /**
     * @brief 解放処理
     *
     */
    void Finalize();

    /**
     * @brief エフェクトリソースの登録
     *
     * @param[in] filePath         ptclのファイルパス
     * @return    成功したときresourceIdを返します。
     */
    uint32_t EntryResource( const char* filePath );

    /**
     * @brief エフェクトリソースの破棄
     *
     */
    void ClearResource();

    /**
     * @brief 指定リソースIDのエミッタセット数の取得
     *
     * @param[in] resId         リソースID
     * @return    エミッタセット数
     */
    uint32_t GetEmitterSetNum( uint32_t resId );

    /**
     * @brief 指定IDのエミッタセットを生成
     *
     * @param[in] emitterSetId    エミッタセットID
     * @param[in] resourceId      リソースID
     * @param[in] groupId         グループID
     */
    void CreateEmitterSet( uint32_t emitterSetId, uint32_t resourceId, uint32_t groupId );

    /**
     * @brief エミッタセットの破棄
     *
     */
    void DestroyEmitterSet();

    /**
     * @brief 定期処理
     *
     */
    void Calc();

    /**
     * @brief 描画処理
     *
     * @param[in] viewMatrix    ビュー行列
     * @param[in] projMatrix    プロジェクション行列
     * @param[in] eyePos        視点
     * @param[in] screenNear    ニアー
     * @param[in] screenFar     ファー
     */
    void DrawEffectSystem( const nn::util::Matrix4x3fType& viewMatrix, const nn::util::Matrix4x4fType& projMatrix, const nn::util::Vector3fType& eyePos, float screenNear, float screenFar );

    /**
     * @brief リソースIDの取得
     *
     * @return    リソースID
     */
    uint32_t GetResourceId();

    /**
     * @brief エフェクト管理の取得
     *
     * @return    エフェクト管理
     */
    nn::vfx::System* GetSystem() { return m_pSystem; }

    /**
     * @brief エフェクト操作ハンドルの取得
     *
     * @return    エフェクト操作ハンドル
     *
     */
    nn::vfx::Handle* GetHandle() { return &m_EsetHandle; }

private:

    nn::vfx::Heap*      m_pHeap;          //!< ヒープ
    nn::vfx::System*    m_pSystem;        //!< エフェクト管理
    nn::vfx::Handle     m_EsetHandle;     //!< エフェクト操作ハンドル
    void*               m_pPtclResource;  //!< PTCLリソース
    EftRenderSystem*    m_pRenderSystem;  //!< 描画システム
    uint32_t            m_ResourceId;     //!< リソースID
};

/**
 * @brief カスタムアクションクラス
 *
 */
class CustomActionTest
{
public:
    /**
     * @brief 初期化
     *
     * @param[in] system    エフェクトシステム
     */
    static void Initialize( nn::vfx::System* system )
    {
        // エミッタビルボード アクション
        nn::vfx::CallbackSet action1CbSet;
        action1CbSet.emitterMatrixSet = _EmitterMatrixSetCallback;
        system->SetCallback( nn::vfx::CallbackId_CustomAction1, action1CbSet );

        // パーティクル位置操作 アクション
        nn::vfx::CallbackSet action2CbSet;
        action2CbSet.particleEmit = _ParticleEmitCallback;
        system->SetCallback( nn::vfx::CallbackId_CustomAction2, action2CbSet );
    }

    /**
     * @brief 描画設定後コールバック
     *
     * @param[in] arg    引数
     */
    static void _EmitterMatrixSetCallback( nn::vfx::EmitterMatrixSetArg& arg )
    {
        NN_UNUSED(arg);
    }

    /**
     * @brief パーティクル生成後コールバック
     *
     * @param[in] arg    引数
     */
    static bool _ParticleEmitCallback( nn::vfx::ParticleEmitArg& arg )
    {
        const nn::vfx::ResCustomActionData* actionParam = reinterpret_cast< const nn::vfx::ResCustomActionData* >( arg.pEmitter->GetCustomActionParameter() );
        arg.SetWorldPos(
            static_cast<float>(actionParam->integerValue[0]),
            static_cast<float>(actionParam->integerValue[1]),
            static_cast<float>(actionParam->integerValue[2]) );
        return true;
    }

};

}
}
