﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cassert>
#include <sstream>

#include <nw/eft/typeDef2.h>
#include <nw/eft/eft2_EndianUtil.h>

#include <zlib.h>

#include "testEft_FileManager.h"
#include "testEft_PngFile.h"

using namespace std;


//!------------------------------------------------------------------------------
//! @brief  8bitデータをストリームに出力します。
//!
//! @param[in] pStream 出力ストリーム
//! @param[in] pData   データ
//! @param[in] pCrc    CRC
//!------------------------------------------------------------------------------
void Output8( ostream* pStream, const void* pData, u32* pCrc = NULL );


//!------------------------------------------------------------------------------
//! @brief  32bitデータをストリームに出力します。
//!
//! @param[in] pStream 出力ストリーム
//! @param[in] pData   データ
//! @param[in] pCrc    CRC
//!------------------------------------------------------------------------------
void Output32( ostream* pStream, const void* pData, u32* pCrc = NULL );


//!------------------------------------------------------------------------------
//! @brief  バイト列データをストリームに出力します。
//!
//! @param[in] pStream 出力ストリーム
//! @param[in] pData   データ
//! @param[in] size    データサイズ
//! @param[in] pCrc    CRC
//!------------------------------------------------------------------------------
void OutputBytes( ostream* pStream, const void* pData, u32 size, u32* pCrc = NULL );


//!------------------------------------------------------------------------------
//! @brief  圧縮率を上げるためピクセルデータにフィルターをかけます。
//!
//! @param[in] pInput  入力データ
//! @param[in] width   画像の横幅
//! @param[in] height  画像の縦幅
//! @param[in] pOutput 出力データ
//!------------------------------------------------------------------------------
void CompressFilter( const u8* pInput, u32 width, u32 height, u8* pOutput );


//!------------------------------------------------------------------------------
//! @brief  8bitデータをストリームから入力します。
//!
//! @param[in] pStream 入力ストリーム
//! @return 入力データ
//!------------------------------------------------------------------------------
u8 Input8( u8** pStream );


//!------------------------------------------------------------------------------
//! @brief  32bitデータをストリームから入力します。
//!
//! @param[in] pStream 入力ストリーム
//! @return 入力データ
//!------------------------------------------------------------------------------
u32 Input32( u8** pStream );


//!------------------------------------------------------------------------------
//! @brief  バイト列データをストリームから入力します。
//!
//! @param[in] pStream 入力ストリーム
//! @param[in] pData   データ
//! @param[in] size    データサイズ
//!------------------------------------------------------------------------------
void InputBytes( u8** pStream, void* pData, u32 size );


//!------------------------------------------------------------------------------
//! @brief  フィルタを適用したピクセルデータを復元します。
//!
//! @param[in] pInput  入力データ
//! @param[in] width   画像の横幅
//! @param[in] height  画像の縦幅
//! @param[in] pOutput 出力データ
//!------------------------------------------------------------------------------
void DecompressFilter( const u8* pInput, u32 width, u32 height, u8* pOutput );


//------------------------------------------------------------------------------
//  イメージをPNGに保存します。
//
//  参考:
//  PNG画像を自力で読む [http://kanow.jp/programming/png-loader.xhtml]
//  zlib 入門 [http://oku.edu.mie-u.ac.jp/~okumura/compression/zlib.html]
//------------------------------------------------------------------------------
bool WritePng( u32 width, u32 height, const u8* pImage, const char* pPath )
{
    // メモリーストリーム
    ostringstream stream;

    // PNGヘッダを出力
    {
        u8 header[] = { 0x89, 0x50, 0x4e, 0x47, 0x0d, 0x0a, 0x1a, 0x0a };
        OutputBytes( &stream, header, sizeof( header ) );
    }

    // IHDRチャンクを出力
    {
        u32 ihdrLength        = 13;
        u8  ihdrHeader[]      = { 'I', 'H', 'D', 'R' };
        u32 ihdrWidth         = width;
        u32 ihdrHeight        = height;
        u8  ihdrDepth         = 8;
        u8  ihdrColorType     = 2;
        u8  ihdrCompressType  = 0;
        u8  ihdrFilterType    = 0;
        u8  ihdrInterlaceType = 0;

        // CRCを初期化
        u32 crc = crc32( 0L, Z_NULL, 0 );

        // チャンク長を出力
        Output32( &stream, &ihdrLength );

        // チャンクデータを出力
        OutputBytes( &stream, ihdrHeader, sizeof( ihdrHeader ), &crc );
        Output32( &stream, &ihdrWidth, &crc );
        Output32( &stream, &ihdrHeight, &crc );
        Output8( &stream, &ihdrDepth, &crc );
        Output8( &stream, &ihdrColorType, &crc );
        Output8( &stream, &ihdrCompressType, &crc );
        Output8( &stream, &ihdrFilterType, &crc );
        Output8( &stream, &ihdrInterlaceType, &crc );

        // CRCを出力
        Output32( &stream, &crc );
    }

    // 圧縮データのバッファを作成
    // TODO: 下記サイズが間違っている: 8ではなく3が正しい
    // TODO: exrepoに移す際に要修正、下記サイズだとReadPngが正しく実行できない
    u32 compressBufferSize = (1 + width * 3) * height + 128 * 1024;  // 念のため少し大きめのバッファサイズを確保
    u8* pCompressBuffer = new u8[compressBufferSize];

    u32 compressedSize;

    // ピクセルデータを圧縮
    {
        // zstreamを作成
        z_stream zstream;

        zstream.zalloc = Z_NULL;
        zstream.zfree = Z_NULL;
        zstream.opaque = Z_NULL;

        int res = deflateInit(&zstream, Z_DEFAULT_COMPRESSION);
        assert(res == Z_OK);

        // 圧縮率を上げるためピクセルデータにフィルターをかける
        // TODO: 下記サイズが間違っている: 8ではなく3が正しい
        // TODO: exrepoに移す際に要修正、下記サイズだとReadPngが正しく実行できない
        u32 filteredSize = (1 + width * 3) * height;
        u8* pFilteredImage = new u8[filteredSize];

        CompressFilter( pImage, width, height, pFilteredImage );

        zstream.next_in   = pFilteredImage;
        zstream.avail_in  = filteredSize;
        zstream.next_out  = pCompressBuffer;
        zstream.avail_out = compressBufferSize;

        // データを圧縮
        res = deflate(&zstream, Z_NO_FLUSH);
        assert(res == Z_OK);

        res = deflate(&zstream, Z_FINISH);
        assert(res == Z_STREAM_END);

        compressedSize = zstream.total_out;

        // zstreamを破棄
        res = deflateEnd(&zstream);
        assert(res == Z_OK);

        delete [] pFilteredImage;
    }

    // IDATチャンクを出力
    {
        u32 idatLength = compressedSize;
        u8  idatHeader[] = { 'I', 'D', 'A', 'T' };

        // CRCを初期化
        u32 crc = crc32( 0L, Z_NULL, 0 );

        // チャンク長を出力
        Output32( &stream, &idatLength );

        // チャンクデータを出力
        OutputBytes( &stream, idatHeader, sizeof( idatHeader ), &crc );
        OutputBytes( &stream, pCompressBuffer, compressedSize, &crc );

        // CRCを出力
        Output32( &stream, &crc );
    }

    // 圧縮データを破棄
    delete[] pCompressBuffer;

    // IENDチャンクを出力
    {
        u32 iendLength = 0;
        u8  iendHeader[] = { 'I', 'E', 'N', 'D' };

        // CRCを初期化
        u32 crc = crc32( 0L, Z_NULL, 0 );

        // チャンク長を出力
        Output32( &stream, &iendLength );

        // チャンクデータを出力
        OutputBytes( &stream, iendHeader, sizeof( iendHeader ), &crc );

        // CRCを出力
        Output32( &stream, &crc );
    }

    // ファイルにデータを保存
    bool resWrite = FileManager::GetInstance().WriteFile(
        reinterpret_cast<const u8*>( stream.str().c_str() ), static_cast<u32>(stream.str().size()), pPath );

    return resWrite;
}


//------------------------------------------------------------------------------
//  イメージをPNGから入力します。
//------------------------------------------------------------------------------
bool ReadPng( u32& width, u32& height, u8** pImage, const char* pPath, nw::eft2::Heap* heap )
{
    // ファイルからデータを入力
    u8* pData;
    u32 size;
    bool resRead = FileManager::GetInstance().ReadFile( pPath, &pData, &size );

    u8* dataHead = pData;

    // エラーチェック
    if (!resRead)
    { return false; }

    // PNGヘッダを入力
    {
        u8 header[8];
        InputBytes( &pData, &header, sizeof( header ) );
    }

    // IHDRチャンクを出力
    {
        // チャンク長を入力
        Input32( &pData );

        // チャンクデータを入力
        u8 ihdrHeader[4];
        InputBytes( &pData, ihdrHeader, sizeof( ihdrHeader ) );
        width = Input32( &pData );
        height = Input32( &pData );
        Input8( &pData );
        Input8( &pData );
        Input8( &pData );
        Input8( &pData );
        Input8( &pData );

        // CRCを入力
        Input32( &pData );
    }

    // 圧縮データのバッファを作成
    u32 decompressBufferSize = (width * 3) * height;
    *pImage = (u8*)heap->Alloc(decompressBufferSize);

    // IDATチャンクを入力
    {
        // チャンク長を入力
        u32 length = Input32( &pData );

        // チャンクデータを入力
        u8  idatHeader[4];
        InputBytes( &pData, idatHeader, sizeof( idatHeader ) );
        u8* pCompressBuffer = new u8[length];
        InputBytes( &pData, pCompressBuffer, length );

        // zstreamを作成
        z_stream zstream;

        zstream.zalloc = Z_NULL;
        zstream.zfree = Z_NULL;
        zstream.opaque = Z_NULL;

        u32 filtedImageSize = (1 + width * 3) * height;
        u8* pFilteredImage = new u8[filtedImageSize];

        zstream.next_in   = pCompressBuffer;
        zstream.avail_in  = length;
        zstream.next_out  = pFilteredImage;
        zstream.avail_out = filtedImageSize;

        int res = inflateInit2(&zstream, MAX_WBITS);
        assert(res == Z_OK);

        // データを解凍
        res = inflate(&zstream, Z_FINISH);
        assert(res == Z_STREAM_END);

        // zstreamを破棄
        res = inflateEnd(&zstream);
        assert(res == Z_OK);

        // フィルターを展開
        DecompressFilter( pFilteredImage, width, height, *pImage );

        delete [] pCompressBuffer;
        delete [] pFilteredImage;

        // CRCを入力
        Input32( &pData );
    }

    // IENDチャンクを入力
    {
        // チャンク長を入力
        Input32( &pData );

        // チャンクデータを入力
        u8  iendHeader[4];
        InputBytes( &pData, iendHeader, sizeof( iendHeader ) );

        // CRCを入力
        Input32( &pData );
    }

    FileManager::GetInstance().ReleaseData(dataHead);

    return resRead;
}


//------------------------------------------------------------------------------
//  8bitデータをストリームに出力します。
//------------------------------------------------------------------------------
void Output8( ostream* pStream, const void* pData, u32* pCrc )
{
    pStream->write( static_cast<const char*>( pData ), 1 );

    if (pCrc != NULL)
    {
        *pCrc = crc32( *pCrc, static_cast<const Bytef*>( pData ), 1 );
    }
}


//------------------------------------------------------------------------------
//  32bitデータをストリームに出力します。
//------------------------------------------------------------------------------
void Output32( ostream* pStream, const void* pData, u32* pCrc )
{
    // ビッグエンディアンになるようにWin版のみフリップ
#if EFT_IS_WIN
    const u8* pBytes = reinterpret_cast<const u8*>( pData );
    u32 data;
    for ( int i = 0; i < 4; ++i )
    {
        data = data << 8 | pBytes[i];
    }
#endif
#if EFT_IS_CAFE
    u32 data = *static_cast<const u32*>( pData );
#endif

    pStream->write( reinterpret_cast<const char*>( &data ), 4 );

    if (pCrc != NULL)
    {
        *pCrc = crc32( *pCrc, reinterpret_cast<const Bytef*>( &data ), 4 );
    }
}


//------------------------------------------------------------------------------
//  バイト列データをストリームに出力します。
//------------------------------------------------------------------------------
void OutputBytes( ostream* pStream, const void* pData, u32 size, u32* pCrc )
{
    pStream->write( static_cast<const char*>( pData ), size );

    if (pCrc != NULL)
    {
        *pCrc = crc32( *pCrc, static_cast<const Bytef*>( pData ), size );
    }
}


//------------------------------------------------------------------------------
//  圧縮率を上げるため画像にフィルターをかけます。
//  TODO: フィルター処理
//------------------------------------------------------------------------------
void CompressFilter( const u8* pInput, u32 width, u32 height, u8* pOutput )
{
    for ( u32 y = 0; y < height; ++y )
    {
        // Win版はOpenGLの仕様に合わせて上下反転させる
#if EFT_IS_WIN
        const u8* pInputLine = &pInput[(height - 1 - y) * ( width * 3 )];
        u8* pOutputLine = &pOutput[y * ( 1 + width * 3 )];
#endif
#if EFT_IS_CAFE
        const u8* pInputLine = &pInput[y * ( width * 3 )];
        u8* pOutputLine = &pOutput[y * ( 1 + width * 3 )];
#endif

        pOutputLine[0] = 0;

        for ( u32 x = 0; x < width; ++x )
        {
            pOutputLine[1 + ( x * 3 ) + 0] = pInputLine[( x * 3 ) + 0];
            pOutputLine[1 + ( x * 3 ) + 1] = pInputLine[( x * 3 ) + 1];
            pOutputLine[1 + ( x * 3 ) + 2] = pInputLine[( x * 3 ) + 2];
        }
    }
}


//!------------------------------------------------------------------------------
//! @brief  8bitデータをストリームから入力します。
//!
//! @param[in] pStream 入力ストリーム
//! @return 入力データ
//!------------------------------------------------------------------------------
u8 Input8( u8** pStream )
{
    u8 data = *(*pStream);
    *pStream = *pStream + 1;
    return data;
}


//!------------------------------------------------------------------------------
//! @brief  32bitデータをストリームから入力します。
//!
//! @param[in] pStream 入力ストリーム
//! @return 入力データ
//!------------------------------------------------------------------------------
u32 Input32( u8** pStream )
{
    u32 data;
    memcpy( &data, *pStream, 4 );

    // リトルエンディアンになるようにWin版のみフリップ
#if EFT_IS_WIN
    u32 temp = data;
    data = 0;
    const u8* pBytes = reinterpret_cast<const u8*>( &temp );
    for ( int i = 0; i < 4; ++i )
    {
        data = data << 8 | pBytes[i];
    }
#endif

    *pStream = *pStream + 4;

    return data;
}


//!------------------------------------------------------------------------------
//! @brief  バイト列データをストリームから入力します。
//!
//! @param[in] pStream 入力ストリーム
//! @param[in] pData   データ
//! @param[in] size    データサイズ
//!------------------------------------------------------------------------------
void InputBytes( u8** pStream, void* pData, u32 size )
{
    memcpy( pData, *pStream, size );
    *pStream = *pStream + size;
}


//!------------------------------------------------------------------------------
//! @brief  フィルタを適用したピクセルデータを復元します。
//!
//! @param[in] pInput  入力データ
//! @param[in] width   画像の横幅
//! @param[in] height  画像の縦幅
//! @param[in] pOutput 出力データ
//!------------------------------------------------------------------------------
void DecompressFilter( const u8* pInput, u32 width, u32 height, u8* pOutput )
{
    for ( u32 y = 0; y < height; ++y )
    {
        // Win版はOpenGLの仕様に合わせて上下反転させる
#if EFT_IS_WIN
        const u8* pInputLine = &pInput[y * ( 1 + width * 3 )];
        u8* pOutputLine = &pOutput[(height - 1 - y) * ( width * 3 )];
#endif
#if EFT_IS_CAFE
        const u8* pInputLine = &pInput[y * ( 1 + width * 3 )];
        u8* pOutputLine = &pOutput[y * ( width * 3 )];
#endif

        for ( u32 x = 0; x < width; ++x )
        {
            pOutputLine[( x * 3 ) + 0] = pInputLine[1 + ( x * 3 ) + 0];
            pOutputLine[( x * 3 ) + 1] = pInputLine[1 + ( x * 3 ) + 1];
            pOutputLine[( x * 3 ) + 2] = pInputLine[1 + ( x * 3 ) + 2];
        }
    }
}
