﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/eft2_Heap.h>
#include <nw/ut/ut_MemoryAllocator.h>

//---------------------------------------------------------------------------
//! @brief        エフェクトヒープ Nwラッパ版
//---------------------------------------------------------------------------
class TestHeap : public nw::eft2::Heap
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    TestHeap()
    {
        mAllocator      = NULL;
        mCounter        = 0;
        mAllocedSize    = 0;
    }

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    explicit TestHeap( nw::ut::MemoryAllocator* allocator )
    {
        mAllocator  = allocator;
    }

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    ~TestHeap()
    {
        Finalize();
    }

    //---------------------------------------------------------------------------
    //! @brief        nw::ut::IAllocatorインスタンスを設定します。
    //---------------------------------------------------------------------------
    void SetNwAllocator( nw::ut::IAllocator* allocator )
    {
        mAllocator  = allocator;
    }

    //---------------------------------------------------------------------------
    //! @brief        nw::ut::IAllocatorインスタンスを取得します。
    //---------------------------------------------------------------------------
    nw::ut::IAllocator* GetNwAllocator() const
    {
        return mAllocator;
    }

    //---------------------------------------------------------------------------
    //! @brief        メモリを確保します。
    //---------------------------------------------------------------------------
    virtual void* Alloc( u32 size, s32 alignment = nw::eft2::Heap::EFT_HEAP_DEAFULT_ALIGNMENT )
    {
        void* ptr = mAllocator->Alloc( size, alignment );
//      void* ptr =  _aligned_malloc( size, alignment );
        EFT_ASSERT( ptr );
        mCounter++;
        mAllocedSize += size;

        return ptr;
    }

    //---------------------------------------------------------------------------
    //! @brief        メモリを解放します。
    //---------------------------------------------------------------------------
    virtual void Free( void* ptr )
    {
        EFT_ASSERT( ptr );

        if ( mCounter > 0 )
        {
            mCounter--;
        }
        else
        {
            EFT_ERR( "[EFTUT] Heap Free Error! " );
        }

        return mAllocator->Free( ptr );
//      return _aligned_free( ptr );
    }

    //---------------------------------------------------------------------------
    //! @brief        メモリ確保カウンターを取得します。
    //---------------------------------------------------------------------------
    s32 GetCounter() const
    {
        return mCounter;
    }

    //---------------------------------------------------------------------------
    //! @brief        確保したメモリサイズを取得します。
    //---------------------------------------------------------------------------
    s32 GetAllocedSize() const
    {
        return mAllocedSize;
    }

    //---------------------------------------------------------------------------
    //! @brief        終了処理
    //---------------------------------------------------------------------------
    void Finalize()
    {
        mCounter = 0;
        mAllocedSize = 0;
    }

private:
    nw::ut::IAllocator*      mAllocator;
    u32                      mCounter;
    u32                      mAllocedSize;
};
