﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testEft_RenderSystem.h"
#if defined( NN_BUILD_CONFIG_OS_WIN )
#include <nn/nn_Windows.h>
#endif
#include <nn/nn_Log.h>

namespace nnt{
namespace eft{

//------------------------------------------------------------------------------
//  コールバック
//------------------------------------------------------------------------------
void OutOfMemoryEventCallback( nn::gfx::CommandBuffer* pCommandBuffer, const nn::gfx::OutOfMemoryEventArg& arg )
{
    NN_UNUSED( pCommandBuffer );
    NN_UNUSED( arg );
}

//------------------------------------------------------------------------------
//  デストラクタ
//------------------------------------------------------------------------------
EftRenderSystem::EftRenderSystem()
    : m_pAllocator(nullptr)
    , m_CurrentSwapChain(nullptr)
{
}

//------------------------------------------------------------------------------
//  コンストラクタ
//------------------------------------------------------------------------------
EftRenderSystem::~EftRenderSystem()
{
}

//------------------------------------------------------------------------------
//  初期化
//------------------------------------------------------------------------------
bool EftRenderSystem::Initialize(
    nn::mem::StandardAllocator* pAllocator,
    int width,
    int height)
{
    m_pAllocator = pAllocator;
    m_ScreenWidth = width;
    m_ScreenHeight = height;

    // レイヤを初期化
    InitializeLayer();

    // ライブラリを初期化
    nn::gfx::Initialize();

    InitializeDevice();

    // メモリプール初期化
    InitializeMemoryPool();

#if defined( NN_BUILD_CONFIG_OS_WIN )
    RECT rw, rc;
    ::GetWindowRect(static_cast<HWND>(m_Hwnd), &rw);
    ::GetClientRect(static_cast<HWND>(m_Hwnd), &rc);

    int new_width = (rw.right - rw.left) - (rc.right - rc.left) + width;
    int new_height = (rw.bottom - rw.top) - (rc.bottom - rc.top) + height;

    BOOL result = SetWindowPos(static_cast<HWND>(m_Hwnd), NULL, 0, 0, new_width, new_height, SWP_NOMOVE | SWP_NOZORDER);
    if (result != TRUE)
    {
        DWORD lastError = GetLastError();
        NN_LOG("SetWindowPos() is error : 0x%08X", lastError);
    }
#endif
    NN_LOG("Screen Width=%d Height=%d\n", m_Width, m_Height);

    // テクスチャサンプラプールを初期化
    {
        nn::gfx::DescriptorPool::InfoType info;
        info.SetDefault();
        info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_Sampler );
        info.SetSlotCount( 1024 );
        size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( &m_Device, info );
        size_t alignment = nn::gfx::DescriptorPool::GetDescriptorPoolAlignment( &m_Device, info );
        ptrdiff_t offset = m_MemoryPoolAllocator.AllocOffset( size, alignment );
        m_SamplerDescriptorPool.Initialize( &m_Device, info, m_MemoryPoolAllocator.GetMemoryPool(), offset, size );
    }

    // テクスチャデスクリプタプールを初期化
    {
        nn::gfx::DescriptorPool::InfoType info;
        info.SetDefault();
        info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_TextureView );
        info.SetSlotCount( 1024 * 32 );
        size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( &m_Device, info );
        size_t alignment = nn::gfx::DescriptorPool::GetDescriptorPoolAlignment( &m_Device, info );
        ptrdiff_t offset = m_MemoryPoolAllocator.AllocOffset(size, alignment);
        m_TextureDescriptorPool.Initialize(&m_Device, info, m_MemoryPoolAllocator.GetMemoryPool(), offset, size);
    }

    InitializeSwapChain(width, height);

    // キューを初期化
    InitializeQueue();

    // コマンドバッファを初期化
    InitializeCommandBuffer();

    // ビューポートシザーを初期化
    InitializeViewportScissor();

    // ラスタライザステートを初期化
    InitializeRasterizerState();

    // 深度ステンシルステートを初期化
    InitializeDepthStencilState();

    // カラーバッファ用のテクスチャを初期化
    InitializeColorBuffer();

    // カラーバッファビューを初期化
    InitializeColorBufferView();

    // 深度ステンシルバッファ用のテクスチャを初期化
    InitializeDepthStencilBuffer();

    // 深度ステンシルビューの初期化
    InitializeDepthStencilView();

    // 同期フェンスを初期化
    InitializeFence();

    return true;
}

//------------------------------------------------------------------------------
//  解放処理
//------------------------------------------------------------------------------
void EftRenderSystem::Finalize()
{
    m_CommandBuffersMemoryPool.Finalize( &m_Device, m_pAllocator );
    m_MemoryPoolAllocator.Finalize( &m_Device, m_pAllocator );
    m_InvisibleMemPoolAllocator.Finalize( &m_Device, m_pAllocator );

    m_TextureDescriptorPool.Finalize( &m_Device );
    m_SamplerDescriptorPool.Finalize( &m_Device );

    m_Fence.Finalize(&m_Device);
    m_DepthStencilView.Finalize(&m_Device);
    m_DepthStencilBuffer.Finalize(&m_Device);
    m_ColorBufferView.Finalize(&m_Device);
    m_ColorBuffer.Finalize(&m_Device);
    m_DepthStencilState.Finalize(&m_Device);
    m_RasterizerState.Finalize(&m_Device);
    m_ViewportScissor.Finalize(&m_Device);
    m_CommandBuffer.Finalize(&m_Device);
    m_Queue.Finalize(&m_Device);
#ifndef NN_BUILD_CONFIG_OS_HORIZON
    m_RgbSwapChain.Finalize(&m_Device);
#endif
    m_SrgbSwapChain.Finalize(&m_Device);
    m_CurrentSwapChain = NULL;
    m_ColorTextureView.Finalize(&m_Device);
    m_DepthStencilTextureView.Finalize(&m_Device);
    m_Device.Finalize();

    nn::gfx::Finalize();

    nn::vi::DestroyLayer( m_Layer );
    nn::vi::CloseDisplay( m_Display );
    nn::vi::Finalize();
}

//------------------------------------------------------------------------------
//  メモリプールの初期化
//------------------------------------------------------------------------------
void EftRenderSystem::InitializeMemoryPool()
{
    const size_t size = 1024 * 1024 * 64;

    // テクスチャ等のInvisibleなメモリプールサイズ
    const size_t invisibleMemPoolSize = 1024 * 1024 * 64;

    m_MemoryPoolAllocator.Initialize(&m_Device, m_pAllocator, size, nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached);

    m_InvisibleMemPoolAllocator.Initialize(&m_Device, m_pAllocator, invisibleMemPoolSize,
        nn::gfx::MemoryPoolProperty_CpuInvisible |
        nn::gfx::MemoryPoolProperty_GpuCached |
        nn::gfx::MemoryPoolProperty_Compressible );
}

//------------------------------------------------------------------------------
//  レイヤの初期処理
//------------------------------------------------------------------------------
void EftRenderSystem::InitializeLayer()
{
    nn::vi::Initialize();

    nn::Result result = nn::vi::OpenDefaultDisplay( &m_Display );
    NN_ABORT_UNLESS_RESULT_SUCCESS( result );
    NN_UNUSED( result );

    m_Width = 1280;
    m_Height = 720;

    result = nn::vi::CreateLayer( &m_Layer, m_Display );

    NN_ABORT_UNLESS_RESULT_SUCCESS( result );

    result = nn::vi::SetLayerScalingMode( m_Layer, nn::vi::ScalingMode_FitToLayer );
    NN_ABORT_UNLESS_RESULT_SUCCESS( result );

    nn::vi::GetNativeWindow((nn::vi::NativeWindowHandle*)&m_Hwnd, m_Layer);
}

//------------------------------------------------------------------------------
//  デバイスの初期化
//------------------------------------------------------------------------------
void EftRenderSystem::InitializeDevice()
{
    nn::gfx::Device::InfoType info;
    info.SetDefault();
    info.SetDebugMode( nn::gfx::DebugMode_Disable );
    info.SetApiVersion( nn::gfx::ApiMajorVersion, nn::gfx::ApiMinorVersion );
    m_Device.Initialize( info );
}

//------------------------------------------------------------------------------
//  スワップチェーンの初期化
//------------------------------------------------------------------------------
void EftRenderSystem::InitializeSwapChain( int width, int height)
{
    nn::gfx::SwapChain::InfoType swapInfo;

    swapInfo.SetDefault();
    swapInfo.SetLayer( m_Layer );
    swapInfo.SetWidth( width );
    swapInfo.SetHeight( height );
    swapInfo.SetBufferCount( 2 );

    size_t size = 0;

    // リニアなスワップチェーンを作成
    swapInfo.SetFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb );

    size = m_RgbSwapChain.CalculateScanBufferSize( &m_Device, swapInfo );
    m_InvisibleMemPoolAllocator.AlignUp( nn::gfx::SwapChain::GetScanBufferAlignment( &m_Device, swapInfo ) );
    m_SrgbSwapChain.Initialize( &m_Device, swapInfo,
        m_InvisibleMemPoolAllocator.GetMemoryPool(),
        m_InvisibleMemPoolAllocator.GetCurrentPos(), size );
    m_InvisibleMemPoolAllocator.Advance( size );

    // ノンリニアなスワップチェーンを作成
#ifndef NN_BUILD_CONFIG_OS_HORIZON
    swapInfo.SetFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );
    size = m_RgbSwapChain.CalculateScanBufferSize( &m_Device, swapInfo );
    m_InvisibleMemPoolAllocator.AlignUp( nn::gfx::SwapChain::GetScanBufferAlignment( &m_Device, swapInfo ) );
    m_RgbSwapChain.Initialize( &m_Device, swapInfo,
                                m_InvisibleMemPoolAllocator.GetMemoryPool(),
                                m_InvisibleMemPoolAllocator.GetCurrentPos(), size );
    m_InvisibleMemPoolAllocator.Advance( size );
    m_CurrentSwapChain = &m_RgbSwapChain;
#else
    m_CurrentSwapChain = &m_SrgbSwapChain;
#endif
}

//------------------------------------------------------------------------------
//  キューの初期化
//------------------------------------------------------------------------------
void EftRenderSystem::InitializeQueue()
{
    nn::gfx::Queue::InfoType info;
    info.SetDefault();
    info.SetCapability( nn::gfx::QueueCapability_Graphics | nn::gfx::QueueCapability_Compute);
    m_Queue.Initialize( &m_Device, info );
}

//------------------------------------------------------------------------------
//  コマンドバッファの初期化
//------------------------------------------------------------------------------
void EftRenderSystem::InitializeCommandBuffer()
{
    const size_t size = 1024 * 1024 * 4/*+ 1024 * 256 */;
    m_CommandBuffersMemoryPool.Initialize(&m_Device, m_pAllocator, size, nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached);

    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    info.SetQueueCapability( nn::gfx::QueueCapability_Graphics | nn::gfx::QueueCapability_Compute);
    info.SetCommandBufferType( nn::gfx::CommandBufferType_Direct );
    m_CommandBuffer.Initialize( &m_Device, info );

    ResetCommandBuffer();
//    m_CommandBuffer.AddCommandMemory( m_CommandBuffersMemoryPool.GetMemoryPool(), m_CommandBuffersMemoryPool.GetCurrentPos(), 1024 * 1024 * 2 );
//    m_CommandBuffer.AddControlMemory( m_MemoryPoolAllocator.Alloc( 1024 * 32, 32 ), 1024 * 32 );
//    m_CommandBuffer.SetEventCallback( CommandBufferEventCallback, NULL );
}

//------------------------------------------------------------------------------
//  コマンドバッファの初期化
//------------------------------------------------------------------------------
void EftRenderSystem::InitializeNestedCommandBuffer()
{
    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    info.SetQueueCapability( nn::gfx::QueueCapability_Graphics );
    info.SetCommandBufferType( nn::gfx::CommandBufferType_Nested );
    m_NestedCommandBuffer.Initialize( &m_Device, info );
}

//------------------------------------------------------------------------------
//  コマンドバッファのリセット
//------------------------------------------------------------------------------
void EftRenderSystem::ResetCommandBuffer()
{
    m_CommandBuffersMemoryPool.Reset();
    m_CommandBuffer.Reset();

    size_t size = 1024 * 1024 * 2;
    size_t alignment = nn::gfx::CommandBuffer::GetCommandMemoryAlignment( &m_Device );
    ptrdiff_t offset = m_CommandBuffersMemoryPool.AllocOffset(size, alignment);
    m_CommandBuffer.AddCommandMemory( m_CommandBuffersMemoryPool.GetMemoryPool(), offset, size );

    size = 1024 * 32;
    alignment = nn::gfx::CommandBuffer::GetControlMemoryAlignment(&m_Device);
    m_CommandBuffer.AddControlMemory(m_CommandBuffersMemoryPool.Alloc(size, alignment), size);

    m_CommandBuffer.SetOutOfCommandMemoryEventCallback( OutOfMemoryEventCallback );
    m_CommandBuffer.SetOutOfControlMemoryEventCallback( OutOfMemoryEventCallback );
}

//------------------------------------------------------------------------------
//  ビューポートシザーの初期化
//------------------------------------------------------------------------------
void EftRenderSystem::InitializeViewportScissor()
{
    nn::gfx::ViewportScissorState::InfoType info;
    info.SetDefault();
    info.SetScissorEnabled( true );

    nn::gfx::ViewportStateInfo viewportInfo;
    viewportInfo.SetDefault();
    viewportInfo.SetWidth( static_cast<float>( m_ScreenWidth ) );
    viewportInfo.SetHeight( static_cast<float>( m_ScreenHeight ) );

    nn::gfx::ScissorStateInfo scissorInfo;
    scissorInfo.SetDefault();
    scissorInfo.SetWidth( m_ScreenWidth );
    scissorInfo.SetHeight( m_ScreenHeight );

    info.SetViewportStateInfoArray( &viewportInfo, 1 );
    info.SetScissorStateInfoArray( &scissorInfo, 1 );
    m_ViewportScissor.Initialize( &m_Device, info );
}

//------------------------------------------------------------------------------
//  ラスタライザステートの初期化
//------------------------------------------------------------------------------
void EftRenderSystem::InitializeRasterizerState()
{
    nn::gfx::RasterizerState::InfoType info;
    info.SetDefault();
    info.SetCullMode( nn::gfx::CullMode_None );
    info.SetPrimitiveTopologyType( nn::gfx::PrimitiveTopologyType_Triangle );
    info.SetScissorEnabled( true );
    info.SetDepthClipEnabled( false );
    m_RasterizerState.Initialize( &m_Device, info );
}

//------------------------------------------------------------------------------
//  深度ステンシルステートの初期化
//------------------------------------------------------------------------------
void EftRenderSystem::InitializeDepthStencilState()
{
    nn::gfx::DepthStencilState::InfoType info;
    info.SetDefault();
    info.SetDepthTestEnabled( false );
    info.SetDepthWriteEnabled( false );
    m_DepthStencilState.Initialize( &m_Device, info );
}

//------------------------------------------------------------------------------
//  カラーバッファ用のテクスチャの初期化
//------------------------------------------------------------------------------
void EftRenderSystem::InitializeColorBuffer()
{
    nn::gfx::Texture::InfoType info;
    info.SetDefault();
    info.SetWidth( m_ScreenWidth );
    info.SetHeight( m_ScreenHeight );
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_ColorBuffer );
    info.SetImageStorageDimension( nn::gfx::ImageStorageDimension_2d );
    info.SetImageFormat( ColorBufferImageFormat );
    info.SetMipCount( 1 );
    info.SetDepth( 1 );

    size_t size = nn::gfx::Texture::CalculateMipDataSize( &m_Device, info );
    size_t alignment = nn::gfx::Texture::CalculateMipDataAlignment(&m_Device, info);
    ptrdiff_t offset = m_InvisibleMemPoolAllocator.AllocOffset( size, alignment );
    m_ColorBuffer.Initialize( &m_Device, info, m_InvisibleMemPoolAllocator.GetMemoryPool(), offset, size );
}

//------------------------------------------------------------------------------
//  カラーバッファビューの初期化
//------------------------------------------------------------------------------
void EftRenderSystem::InitializeColorBufferView()
{
    nn::gfx::ColorTargetView::InfoType info;
    info.SetDefault();
    info.SetImageDimension( nn::gfx::ImageDimension_2d );
    info.SetImageFormat( ColorBufferImageFormat );
    info.SetTexturePtr( &m_ColorBuffer );
    m_ColorBufferView.Initialize( &m_Device, info );

}

//------------------------------------------------------------------------------
//  深度ステンシルバッファ用のテクスチャの初期化
//------------------------------------------------------------------------------
void EftRenderSystem::InitializeDepthStencilBuffer()
{
    nn::gfx::Texture::InfoType info;
    info.SetDefault();
    info.SetWidth( m_ScreenWidth );
    info.SetHeight( m_ScreenHeight );
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_DepthStencil );
    info.SetImageStorageDimension( nn::gfx::ImageStorageDimension_2d );
    info.SetImageFormat( nn::gfx::ImageFormat_D16_Unorm );
    info.SetMipCount( 1 );

    size_t size = nn::gfx::Texture::CalculateMipDataSize( &m_Device, info );
    size_t alignment = nn::gfx::Texture::CalculateMipDataAlignment(&m_Device, info);
    ptrdiff_t offset = m_InvisibleMemPoolAllocator.AllocOffset(size, alignment);
    m_DepthStencilBuffer.Initialize( &m_Device, info, m_InvisibleMemPoolAllocator.GetMemoryPool(), offset, size );
}

//------------------------------------------------------------------------------
//  深度ステンシルビューの初期化
//------------------------------------------------------------------------------
void EftRenderSystem::InitializeDepthStencilView()
{
    nn::gfx::DepthStencilView::InfoType info;
    info.SetDefault();
    info.SetImageDimension( nn::gfx::ImageDimension_2d );
    info.SetTexturePtr( &m_DepthStencilBuffer );
    m_DepthStencilView.Initialize( &m_Device, info );
}

//------------------------------------------------------------------------------
//  同期フェンスの初期化
//------------------------------------------------------------------------------
void EftRenderSystem::InitializeFence()
{
    nn::gfx::Fence::InfoType info;
    info.SetDefault();
    m_Fence.Initialize( &m_Device, info );
}

void EftRenderSystem::InitializeDescriptors( TextureDescriptorIndexAllocator* textureDescPoolAllocator )
{

    // カラーコピーテクスチャビューを初期化
    nn::gfx::TextureView::InfoType viewInfo;
    viewInfo.SetDefault();
    viewInfo.SetImageDimension( nn::gfx::ImageDimension_2d );
    viewInfo.SetImageFormat( ColorBufferImageFormat );
    viewInfo.SetTexturePtr( &m_ColorBuffer );
    m_ColorTextureView.Initialize( &m_Device, viewInfo );

    // デスクリプタスロット設定
    textureDescPoolAllocator->AllocateSlotForFontTexture( &m_ColorDescSlot, m_ColorTextureView, NULL );

    // 深度ステンシルテクスチャビューを初期化
    nn::gfx::TextureView::InfoType texViewInfo;
    texViewInfo.SetDefault();
    texViewInfo.SetImageDimension( nn::gfx::ImageDimension_2d );
    texViewInfo.SetImageFormat( nn::gfx::ImageFormat_D16_Unorm );
    texViewInfo.SetTexturePtr( &m_DepthStencilBuffer );
    m_DepthStencilTextureView.Initialize( &m_Device, texViewInfo );

    // デスクリプタスロット設定
    textureDescPoolAllocator->AllocateSlotForFontTexture( &m_DepthStencilDescSlot, m_DepthStencilTextureView, NULL );
}

}
}
