﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/vfx/vfx_System.h>

namespace nnt{
namespace eft{

//------------------------------------------------------------------------------
//  デスクリプタインデックス アロケータ
//------------------------------------------------------------------------------
class IDescriptorIndexAllocator
{
public:
    //---------------------------------------------------------------------------
    //! @brief  コンストラクタ
    //---------------------------------------------------------------------------
    IDescriptorIndexAllocator()
        : m_pDescPool( nullptr )
        , m_Index( 256 )
    {
    }

    //---------------------------------------------------------------------------
    //! @brief      コンストラクタ
    //! @param[in]  pPool   TBD
    //---------------------------------------------------------------------------
    void SetDescriptorPool( nn::gfx::DescriptorPool* pPool )
    {
        m_pDescPool = pPool;
    }

    //---------------------------------------------------------------------------
    //! @brief  コンストラクタ
    //---------------------------------------------------------------------------
    nn::gfx::DescriptorPool* GetDescriptorPool()
    {
        return m_pDescPool;
    }

    //---------------------------------------------------------------------------
    //! @brief      デスクリプタプールのインデックスを取得
    //! @param[in]  requireCount    TBD
    //---------------------------------------------------------------------------
    int GetDescriptorPoolIndex( int requireCount )
    {
        int ret = m_Index;
        m_Index += requireCount;
        return ret;
    }

protected:
    nn::gfx::DescriptorPool*    m_pDescPool;    //!< デスクリプタプールへのポインタ
    int                         m_Index;        //!< インデックス
};

//------------------------------------------------------------------------------
//  テクスチャデスクリプタインデックス アロケータ
//------------------------------------------------------------------------------
class TextureDescriptorIndexAllocator : public IDescriptorIndexAllocator
{
public:
    //---------------------------------------------------------------------------
    //! @brief      デスクリプタスロットを確保
    //! @param[out]  dstSlot        確保したデスクリプタスロットを出力
    //! @param[int]  textureView    テクスチャビュー
    //---------------------------------------------------------------------------
    bool AllocateSlotForFontTexture( nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::TextureView& textureView, void* pUserData )
    {
        NN_UNUSED(pUserData);

        int slot = GetDescriptorPoolIndex( 1 );
        m_pDescPool->BeginUpdate();
        {
            m_pDescPool->SetTextureView( slot, &textureView);
        }
        m_pDescPool->EndUpdate();
        m_pDescPool->GetDescriptorSlot( dstSlot, slot );
        return true;
    }
};

//------------------------------------------------------------------------------
//  サンプラデスクリプタインデックス アロケータ
//------------------------------------------------------------------------------
class SamplerDescriptorIndexAllocator : public IDescriptorIndexAllocator
{
public:
    //---------------------------------------------------------------------------
    //! @brief      デスクリプタスロットを確保
    //! @param[out]  dstSlot        確保したデスクリプタスロットを出力
    //! @param[int]  textureView    テクスチャビュー
    //---------------------------------------------------------------------------
    bool AllocateSlotForFontSampler( nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::Sampler& sampler, void* pUserData )
    {
        NN_UNUSED(pUserData);

        int slot = GetDescriptorPoolIndex( 1 );
        m_pDescPool->BeginUpdate();
        {
            m_pDescPool->SetSampler( slot, &sampler );
        }
        m_pDescPool->EndUpdate();
        m_pDescPool->GetDescriptorSlot( dstSlot, slot );
        return true;
    }
};

}
}

