﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/typeDef2.h>
#include <nw/eft/eft2_System.h>


class HandlingTestSet;

//------------------------------------------------------------------------------
//  テスト実装状況
//  o: 実装済
//  +: コマンド追加済み、テスト未実装
//  /: 似たコマンドがあるため実装を省略
//  x: 未実装 (保留)
//------------------------------------------------------------------------------
// o SetAddVel(const nw::math::VEC3& velAdd)
// o SetAllDirectionalVel(f32 velScale)
// o SetAlpha(f32 alpha)
// o SetCalcEnable(bool flag)
// - SetColor(f32 r, f32 g, f32 b)
// - SetColor(const nw::math::VEC4& color)
// o SetColor(const nw::ut::Color4f& color)
// o SetDirectional(const nw::math::VEC3& dir)
// o SetDirectionalVel(f32 velScale)
// o SetDrawPriority(u8 priority)
// o SetEmissionIntervalScale(f32 ratio)
// o SetEmissionParticleScale(nw::math::VEC2& ratio)
// - SetEmissionParticleScale(f32 scale)
// - SetEmissionParticleScale(const nw::math::VEC2& scale)
// o SetEmissionRatioScale(f32 ratio)
// - SetEmissionScale(f32 scale) = SetEmissionParticleScale
// o SetEmitterColor(const nw::math::VEC4& color0,
// - SetEmitterColor0(const nw::math::VEC4& color1)
// - SetEmitterColor1(const nw::math::VEC4& color0)
// o SetEmitterScale(const nw::math::VEC3& scale)
// o SetEmitterVisible(const char* emitterName, bool flag)
// - SetEmitterVolumeScale(const nw::math::VEC3& scale) = SetEmitterScale
// x SetFinalizeCallback(EmitterSetFinalizeCallback callback)
// o SetInitRotate(const nw::math::VEC3& rot)
// - SetInitRotate(const Vector3i& rot)
// - SetManualParticleEmission()
// o SetManualParticleEmission(bool emissionEnable)
// o SetManualParticleEmissionWithParticleCount(u32 particle
// - SetMatrix(const nw::math::MTX34& matrixSRT) = SetMtx
// o SetMatrixFast(const nw::math::MTX34& matrixSRT)
// - SetMatrixFast(const nw::math::MTX34& matrixRT, f32
// - SetMatrixFast(const nw::math::MTX34& matrixRT,
// o SetMtx(const nw::math::MTX34& matrixSRT)
// - SetMtxFast(const nw::math::MTX34& matrixSRT)
// - SetMtxFast(const nw::math::MTX34& matrixRT, f32 scale)
// - SetMtxFast(const nw::math::MTX34& matrixRT, const nw::
// o SetParticleLifeScale(f32 ratio)
// o SetParticleScale(f32 scale)
// - SetParticleScale(f32 scaleX, f32 scaleY)
// - SetParticleScale(const nw::math::VEC2& scale)
// - SetParticleScale(nw::math::VEC2& ratio)
// - SetPos(const nw::math::VEC3& pos) = SetMatrix
// - SetPos(const nw::math::VEC3& pos, f32 scale) = SetMatrix
// - SetPos(const nw::math::VEC3& pos, const nw::math::VEC3 = SetMatrix
// o SetRandomVel(f32 velScale)
// o SetRuntimeUserData(u32 data)
// o SetStartFrame(s32 frame)
// - SetStopCalc(bool isStopCalc) = SetCalcEnable
// - SetStopCalcAndDraw(bool isStopCalcAndDraw) = SetCalcEnable(), SetVisible()
// - SetStopDraw(bool isStopDraw) = SetVisible
// x SetSystem(System* sys)
// x SetViewFlag(DrawViewFlag flag)
// o SetVisible(bool flag)


//------------------------------------------------------------------------------
//! @brief  操作コマンド
//------------------------------------------------------------------------------
class HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] type     コマンドタイプ
    //! @param[in] exeFrame 実行フレーム番号
    //------------------------------------------------------------------------------
    NN_IMPLICIT HandlingCommand( int exeFrame );

    //------------------------------------------------------------------------------
    //! @brief  デストラクタです。
    //------------------------------------------------------------------------------
    virtual ~HandlingCommand() {};

    //------------------------------------------------------------------------------
    //! @brief  コマンド開始フレーム番号を取得します。
    //!
    //! @return コマンド開始フレーム番号を返します。
    //------------------------------------------------------------------------------
    virtual int GetBeginFrame() const;

    //------------------------------------------------------------------------------
    //! @brief  コマンド終了フレーム番号を取得します。
    //!
    //! @return コマンド終了フレーム番号を返します。
    //------------------------------------------------------------------------------
    virtual int GetEndFrame() const;

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行するかどうか取得します。
    //!
    //! @param[in] numFrame フレーム番号
    //! @return コマンドを実行するときtrueを返します。
    //------------------------------------------------------------------------------
    virtual bool DoExecute( int numFrame ) const;

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest ) = 0;

private:
    int                 m_exeFrame;  //!< 実行フレーム番号
};


//------------------------------------------------------------------------------
//! @brief  エミッタセット作成コマンド
//------------------------------------------------------------------------------
class CreateEsetCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] esetId   エミッタセットID
    //------------------------------------------------------------------------------
    CreateEsetCommand( int exeFrame, u32 esetId );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    u32 m_esetId;  //!< エミッタセットID
};


//------------------------------------------------------------------------------
//! @brief  エミッタセット削除コマンド
//------------------------------------------------------------------------------
class KillEsetCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame    実行フレーム番号
    //! @param[in] pEmitterSet エミッタセット
    //------------------------------------------------------------------------------
    KillEsetCommand( int exeFrame, nw::eft2::EmitterSet* pEmitterSet );

    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame        実行フレーム番号
    //! @param[in] pEmitterSetName エミッタセット名
    //! @param[in] resID           リソースID
    //------------------------------------------------------------------------------
    KillEsetCommand( int exeFrame, const char* pEmitterSetName, u32 resId );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    nw::eft2::EmitterSet* m_pEmitterSet;      //!< エミッタセット
    const char*           m_pEmitterSetName;  //!< エミッタセット名
    u32                   m_resId;            //!< リソースID
};


//------------------------------------------------------------------------------
//! @brief  エミッタセット全削除コマンド
//------------------------------------------------------------------------------
class KillAllEsetCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //------------------------------------------------------------------------------
    NN_IMPLICIT KillAllEsetCommand( int exeFrame );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );
};


//------------------------------------------------------------------------------
//! @brief  フェードコマンド
//------------------------------------------------------------------------------
class FadeCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //------------------------------------------------------------------------------
    NN_IMPLICIT FadeCommand( int exeFrame );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );
};


//------------------------------------------------------------------------------
//! @brief  速度への加算値設定コマンド
//------------------------------------------------------------------------------
class SetAddVelCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] velocity 速度への加算値
    //------------------------------------------------------------------------------
    SetAddVelCommand( int exeFrame, const nw::math::VEC3& velocity );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    nw::math::VEC3 m_velocity;  //!< 速度への加算値
};


//------------------------------------------------------------------------------
//! @brief  全方向速度スケール設定コマンド
//------------------------------------------------------------------------------
class SetAllDirectionalVelCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] velocity 全方向速度スケール
    //------------------------------------------------------------------------------
    SetAllDirectionalVelCommand( int exeFrame, f32 scale );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    f32 m_scale;  //!< 全方向速度スケール
};


//------------------------------------------------------------------------------
//! @brief  描画アルファ設定コマンド
//------------------------------------------------------------------------------
class SetAlphaCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] alpha    描画アルファ
    //------------------------------------------------------------------------------
    SetAlphaCommand( int exeFrame, f32 alpha );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    f32 m_alpha;  //!< 描画アルファ
};


//------------------------------------------------------------------------------
//! @brief  計算処理の有効/無効設定コマンド
//------------------------------------------------------------------------------
class SetCalcEnableCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] enabled  計算処理の有効/無効
    //------------------------------------------------------------------------------
    SetCalcEnableCommand( int exeFrame, bool enabled );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    bool m_enabled;  //!< 計算処理の有効/無効
};


//------------------------------------------------------------------------------
//! @brief  カラー設定コマンド
//------------------------------------------------------------------------------
class SetColorCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] color    カラー
    //------------------------------------------------------------------------------
    SetColorCommand( int exeFrame, const nw::math::VEC4& color );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    nw::math::VEC4 m_color;  //!< カラー
};


//------------------------------------------------------------------------------
//! @brief  指定方向設定コマンド
//------------------------------------------------------------------------------
class SetDirectionalCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame  実行フレーム番号
    //! @param[in] direction 指定方向
    //------------------------------------------------------------------------------
    SetDirectionalCommand( int exeFrame, const nw::math::VEC3& direction );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    nw::math::VEC3 m_direction;  //!< 指定方向
};


//------------------------------------------------------------------------------
//! @brief  指定方向速度スケール設定コマンド
//------------------------------------------------------------------------------
class SetDirectionalVelCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] scale    指定方向速度スケール
    //------------------------------------------------------------------------------
    SetDirectionalVelCommand( int exeFrame, float scale );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    float m_scale;  //!< 指定方向速度スケール
};


//------------------------------------------------------------------------------
//! @brief  描画優先度設定コマンド
//------------------------------------------------------------------------------
class SetDrawPriorityCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] priority 描画優先度
    //------------------------------------------------------------------------------
    SetDrawPriorityCommand( int exeFrame, u8 priority );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    u8 m_priority;  //!< 描画優先度
};


//------------------------------------------------------------------------------
//! @brief  放出間隔スケール設定コマンド
//------------------------------------------------------------------------------
class SetEmissionIntervalScaleCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] scale    寿命スケール
    //------------------------------------------------------------------------------
    SetEmissionIntervalScaleCommand( int exeFrame, f32 scale );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    f32 m_scale;  //!< スケール
};


//------------------------------------------------------------------------------
//! @brief  放出スケール設定コマンド
//------------------------------------------------------------------------------
class SetEmissionParticleScaleCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] scale    放出スケール
    //------------------------------------------------------------------------------
    SetEmissionParticleScaleCommand( int exeFrame, const nw::math::VEC2& scale );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    nw::math::VEC2 m_scale;  //!< 放出スケール
};


//------------------------------------------------------------------------------
//! @brief  放出レートスケール設定コマンド
//------------------------------------------------------------------------------
class SetEmissionRatioScaleCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] scale    寿命スケール
    //------------------------------------------------------------------------------
    SetEmissionRatioScaleCommand( int exeFrame, f32 scale );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    f32 m_scale;  //!< スケール
};


//------------------------------------------------------------------------------
//! @brief  エミッタ乗算カラー設定コマンド
//------------------------------------------------------------------------------
class SetEmitterColorCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] color0   エミッタ乗算カラー0
    //! @param[in] color1   エミッタ乗算カラー1
    //------------------------------------------------------------------------------
    SetEmitterColorCommand( int exeFrame, const nw::math::VEC4& color0, const nw::math::VEC4& color1 );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    nw::math::VEC4 m_color0;  //!< エミッタ乗算カラー0
    nw::math::VEC4 m_color1;  //!< エミッタ乗算カラー1
};


//------------------------------------------------------------------------------
//! @brief  エミッタスケール設定コマンド
//------------------------------------------------------------------------------
class SetEmitterScaleCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] scale    エミッタスケール
    //------------------------------------------------------------------------------
    SetEmitterScaleCommand( int exeFrame, const nw::math::VEC3& scale );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    nw::math::VEC3 m_scale;  //!< エミッタスケール
};


//------------------------------------------------------------------------------
//! @brief  エミッタ描画処理の有効/無効設定コマンド
//------------------------------------------------------------------------------
class SetEmitterVisibleCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] visible  エミッタ描画処理の有効/無効
    //------------------------------------------------------------------------------
    SetEmitterVisibleCommand( int exeFrame, u32 emitterIndex, bool visible );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    u32  m_emitterIndex;  //!< エミッタインデックス
    bool m_visible;       //!< エミッタ描画処理の有効/無効
};


//------------------------------------------------------------------------------
//! @brief  初期角度設定コマンド
//------------------------------------------------------------------------------
class SetInitRotateCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] rotate   初期角度
    //------------------------------------------------------------------------------
    SetInitRotateCommand( int exeFrame, const nw::math::VEC3& rotate );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    nw::math::VEC3 m_rotate;  //!< 初期角度
};


//------------------------------------------------------------------------------
//! @brief  手動放出設定コマンド
//------------------------------------------------------------------------------
class SetManualParticleEmissionCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] manual   手動放出設定
    //------------------------------------------------------------------------------
    SetManualParticleEmissionCommand( int exeFrame, bool manual );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    bool m_manual;  //!< 手動放出設定
};


//------------------------------------------------------------------------------
//! @brief  手動放出設定コマンド
//------------------------------------------------------------------------------
class SetParticleCountCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] count    パーティクル数
    //------------------------------------------------------------------------------
    SetParticleCountCommand( int exeFrame, u32 count );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    u32 m_count;  //!< パーティクル数
};


//------------------------------------------------------------------------------
//! @brief  行列設定コマンド
//------------------------------------------------------------------------------
class SetMatrixCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] matrix   行列
    //------------------------------------------------------------------------------
    SetMatrixCommand( int exeFrame, const nw::math::MTX34& matrix );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    nw::math::MTX34 m_matrix;  //!< 行列
};


//------------------------------------------------------------------------------
//! @brief  スケール抜き行列設定コマンド
//------------------------------------------------------------------------------
class SetMatrixFastCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] matrix   行列
    //! @param[in] scale    スケール
    //------------------------------------------------------------------------------
    SetMatrixFastCommand( int exeFrame, const nw::math::MTX34& matrix, const nw::math::VEC3& scale );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    nw::math::MTX34 m_matrix;  //!< 行列
    nw::math::VEC3  m_scale;   //!< スケール
};


//------------------------------------------------------------------------------
//! @brief  寿命スケール設定コマンド
//------------------------------------------------------------------------------
class SetParticleLifeScaleCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] scale    寿命スケール
    //------------------------------------------------------------------------------
    SetParticleLifeScaleCommand( int exeFrame, f32 scale );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    f32 m_scale;  //!< 寿命スケール
};


//------------------------------------------------------------------------------
//! @brief  パーティクルスケール設定コマンド
//------------------------------------------------------------------------------
class SetParticleScaleCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] scale    パーティクルスケール
    //------------------------------------------------------------------------------
    SetParticleScaleCommand( int exeFrame, const nw::math::VEC2& scale );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    const nw::math::VEC2& m_scale;  //!< パーティクルスケール
};


//------------------------------------------------------------------------------
//! @brief  初期速度ランダムスケール設定コマンド
//------------------------------------------------------------------------------
class SetRandomVelCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] scale    初期速度ランダムスケール
    //------------------------------------------------------------------------------
    SetRandomVelCommand( int exeFrame, float scale );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    float m_scale;  //!< 初期速度ランダムスケール
};


//------------------------------------------------------------------------------
//! @brief  ユーザーデータ設定コマンド
//------------------------------------------------------------------------------
class SetRuntimeUserDataCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] data     ユーザーデータ
    //------------------------------------------------------------------------------
    SetRuntimeUserDataCommand( int exeFrame, u32 data );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    u32 m_data;  //!< ユーザーデータ
};


//------------------------------------------------------------------------------
//! @brief  開始フレーム設定コマンド
//------------------------------------------------------------------------------
class SetStartFrameCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] frame    開始フレーム
    //------------------------------------------------------------------------------
    SetStartFrameCommand( int exeFrame, s32 frame );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    s32 m_frame;  //!< 開始フレーム
};


//------------------------------------------------------------------------------
//! @brief  表示/非表示設定コマンド
//------------------------------------------------------------------------------
class SetVisibleCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] visible  表示/非表示
    //------------------------------------------------------------------------------
    SetVisibleCommand( int exeFrame, bool visible );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    bool m_visible;  //!< 表示/非表示
};
