﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testEft_HandlingCommand.h"
#include "testEft_HandlingTestSet.h"


//------------------------------------------------------------------------------
// HandlingCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
HandlingCommand::HandlingCommand( int exeFrame )
    : m_exeFrame( exeFrame )
{
}


//------------------------------------------------------------------------------
//  コマンド開始フレーム番号を取得します。
//------------------------------------------------------------------------------
int HandlingCommand::GetEndFrame() const
{
    return m_exeFrame;
}


//------------------------------------------------------------------------------
//  コマンド終了フレーム番号を取得します。
//------------------------------------------------------------------------------
int HandlingCommand::GetBeginFrame() const
{
    return m_exeFrame;
}


//------------------------------------------------------------------------------
//  コマンドを実行するかどうか取得します。
//------------------------------------------------------------------------------
bool HandlingCommand::DoExecute( int numFrame ) const
{
    return ( numFrame == m_exeFrame );
}


//------------------------------------------------------------------------------
// CreateEsetCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
CreateEsetCommand::CreateEsetCommand( int exeFrame, u32 esetId )
    : HandlingCommand( exeFrame ), m_esetId( esetId )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void CreateEsetCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::System* pEffectSystem = pTest->GetEffectSystem();
    nw::eft2::Handle* pEsetHandle = pTest->GetEsetHandle();

    bool resCreate = pEffectSystem->CreateEmitterSetID( pEsetHandle, m_esetId, 0, 0 );
    assert(resCreate == true);

#ifndef _DEBUG
    NW_UNUSED_VARIABLE(resCreate);
#endif
}


//------------------------------------------------------------------------------
// KillEsetCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
KillEsetCommand::KillEsetCommand( int exeFrame, nw::eft2::EmitterSet* pEmitterSet )
    : HandlingCommand( exeFrame ), m_pEmitterSet( pEmitterSet ), m_pEmitterSetName(), m_resId()
{
}


//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
KillEsetCommand::KillEsetCommand( int exeFrame, const char* pEmitterSetName, u32 resId )
    : HandlingCommand( exeFrame ), m_pEmitterSet(), m_pEmitterSetName( pEmitterSetName ), m_resId( resId )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void KillEsetCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::System* pEffectSystem = pTest->GetEffectSystem();

    if (m_pEmitterSet != NULL)
    {
        pEffectSystem->KillEmitterSet( m_pEmitterSet );
    }
    else
    {
        pEffectSystem->KillEmitterSet( m_pEmitterSetName, m_resId );
    }
}


//------------------------------------------------------------------------------
// KillAllEsetCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
KillAllEsetCommand::KillAllEsetCommand( int exeFrame )
    : HandlingCommand( exeFrame )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void KillAllEsetCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::System* pEffectSystem = pTest->GetEffectSystem();

    pEffectSystem->KillAllEmitterSet();
}


//------------------------------------------------------------------------------
// FadeCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
FadeCommand::FadeCommand( int exeFrame )
    : HandlingCommand( exeFrame )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void FadeCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->Fade();
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetAllDirectionalVelCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetAddVelCommand::SetAddVelCommand( int exeFrame, const nw::math::VEC3& velocity )
    : HandlingCommand( exeFrame ), m_velocity( velocity )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetAddVelCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetAddVel( m_velocity );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetAllDirectionalVelCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetAllDirectionalVelCommand::SetAllDirectionalVelCommand( int exeFrame, f32 scale )
    : HandlingCommand( exeFrame ), m_scale( scale )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetAllDirectionalVelCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetAllDirectionalVel( m_scale );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetAlphaCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetAlphaCommand::SetAlphaCommand( int exeFrame, f32 alpha )
    : HandlingCommand( exeFrame ), m_alpha( alpha )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetAlphaCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetAlpha( m_alpha );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetCalcEnableCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetCalcEnableCommand::SetCalcEnableCommand( int exeFrame, bool enabled )
    : HandlingCommand( exeFrame ), m_enabled( enabled )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetCalcEnableCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetCalcEnable( m_enabled );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetColorCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetColorCommand::SetColorCommand( int exeFrame, const nw::math::VEC4& color )
    : HandlingCommand( exeFrame ), m_color( color )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetColorCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetColor( m_color );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetDirectionalCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetDirectionalCommand::SetDirectionalCommand( int exeFrame, const nw::math::VEC3& direction )
    : HandlingCommand( exeFrame ), m_direction( direction )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetDirectionalCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetDirectional( m_direction );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetDirectionalVelCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetDirectionalVelCommand::SetDirectionalVelCommand( int exeFrame, float scale )
    : HandlingCommand( exeFrame ), m_scale( scale )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetDirectionalVelCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetDirectionalVel( m_scale );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetDrawPriorityCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetDrawPriorityCommand::SetDrawPriorityCommand( int exeFrame, u8 priority )
    : HandlingCommand( exeFrame ), m_priority( priority )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetDrawPriorityCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetDrawPriority( m_priority );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetEmissionInternalScaleCommand

//------------------------------------------------------------------------------
//  コストラクタです。
//------------------------------------------------------------------------------
SetEmissionIntervalScaleCommand::SetEmissionIntervalScaleCommand( int exeFrame, f32 scale )
    : HandlingCommand( exeFrame ), m_scale( scale )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetEmissionIntervalScaleCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetEmissionIntervalScale( m_scale );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetEmissionParticleScaleCommand

//------------------------------------------------------------------------------
//  コストラクタです。
//------------------------------------------------------------------------------
SetEmissionParticleScaleCommand::SetEmissionParticleScaleCommand( int exeFrame, const nw::math::VEC2& scale )
    : HandlingCommand( exeFrame ), m_scale( scale )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetEmissionParticleScaleCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetEmissionParticleScale( m_scale );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetEmissionRatioScaleCommand

//------------------------------------------------------------------------------
//  コストラクタです。
//------------------------------------------------------------------------------
SetEmissionRatioScaleCommand::SetEmissionRatioScaleCommand( int exeFrame, f32 scale )
    : HandlingCommand( exeFrame ), m_scale( scale )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetEmissionRatioScaleCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetEmissionRatioScale( m_scale );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetEmitterColorCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetEmitterColorCommand::SetEmitterColorCommand( int exeFrame, const nw::math::VEC4& color0, const nw::math::VEC4& color1 )
    : HandlingCommand( exeFrame ), m_color0( color0 ), m_color1( color1 )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetEmitterColorCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetEmitterColor( nw::math::VEC4( m_color0 ), nw::math::VEC4( m_color1 ) );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetEmitterScaleCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetEmitterScaleCommand::SetEmitterScaleCommand( int exeFrame, const nw::math::VEC3& scale )
    : HandlingCommand( exeFrame ), m_scale( scale )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetEmitterScaleCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetEmitterScale( m_scale );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetEmitterVisibleCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetEmitterVisibleCommand::SetEmitterVisibleCommand( int exeFrame, u32 emitterIndex, bool visible )
    : HandlingCommand( exeFrame ), m_emitterIndex( emitterIndex ), m_visible( visible )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetEmitterVisibleCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        const nw::eft2::Emitter* pEmitter = pEset->GetAliveEmitter( m_emitterIndex );
        if ( pEmitter != NULL )
        {
            const char* pEmitterName = pEmitter->GetName();
            if ( pEmitterName != NULL )
            {
                pEset->SetEmitterVisible( pEmitterName, m_visible );
            }
        }
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetInitRotateCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetInitRotateCommand::SetInitRotateCommand( int exeFrame, const nw::math::VEC3& rotate )
    : HandlingCommand( exeFrame ), m_rotate( rotate )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetInitRotateCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetInitRotate( m_rotate );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetManualParticleEmissionCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetManualParticleEmissionCommand::SetManualParticleEmissionCommand( int exeFrame, bool manual )
    : HandlingCommand( exeFrame ), m_manual( manual )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetManualParticleEmissionCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetManualParticleEmission( m_manual );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetManualParticleEmissionWithCountCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetParticleCountCommand::SetParticleCountCommand( int exeFrame, u32 count )
    : HandlingCommand( exeFrame ), m_count( count )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetParticleCountCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetManualParticleEmissionWithParticleCount( m_count );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetMatrixCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetMatrixCommand::SetMatrixCommand( int exeFrame, const nw::math::MTX34& matrix )
    : HandlingCommand( exeFrame ), m_matrix( matrix )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetMatrixCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetMatrix( m_matrix );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetMatrixFastCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetMatrixFastCommand::SetMatrixFastCommand( int exeFrame, const nw::math::MTX34& matrix, const nw::math::VEC3& scale )
    : HandlingCommand( exeFrame ), m_matrix( matrix ), m_scale( scale )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetMatrixFastCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetMatrixFast( m_matrix, m_scale );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetParticleLifeScaleCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetParticleLifeScaleCommand::SetParticleLifeScaleCommand( int exeFrame, f32 scale )
    : HandlingCommand( exeFrame ), m_scale( scale )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetParticleLifeScaleCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetParticleLifeScale( m_scale );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetParticleScaleCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetParticleScaleCommand::SetParticleScaleCommand( int exeFrame, const nw::math::VEC2& scale )
    : HandlingCommand( exeFrame ), m_scale( scale )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetParticleScaleCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetParticleScale( m_scale );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetRandomVelCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetRandomVelCommand::SetRandomVelCommand( int exeFrame, float scale )
    : HandlingCommand( exeFrame ), m_scale( scale )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetRandomVelCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetRandomVel( m_scale );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetRuntimeUserDataCommand

//------------------------------------------------------------------------------
//  コストラクタです。
//------------------------------------------------------------------------------
SetRuntimeUserDataCommand::SetRuntimeUserDataCommand( int exeFrame, u32 data )
    : HandlingCommand( exeFrame ), m_data( data )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetRuntimeUserDataCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetRuntimeUserData( m_data );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetStartFrameCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetStartFrameCommand::SetStartFrameCommand( int exeFrame, s32 frame )
    : HandlingCommand( exeFrame ), m_frame( frame )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetStartFrameCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetStartFrame( m_frame );
        pEset = pEset->GetNext();
    }
}


//------------------------------------------------------------------------------
// SetVisibleCommand

//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
SetVisibleCommand::SetVisibleCommand( int exeFrame, bool visible )
    : HandlingCommand( exeFrame ), m_visible( visible )
{
}


//------------------------------------------------------------------------------
//  コマンドを実行します。
//------------------------------------------------------------------------------
void SetVisibleCommand::Execute( HandlingTestSet* pTest )
{
    nw::eft2::EmitterSet* pEset = pTest->GetEsetHandle()->GetEmitterSet();

    while ( pEset != NULL )
    {
        pEset->SetVisible( m_visible );
        pEset = pEset->GetNext();
    }
}
