﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// testEft_EmitterSetHandling.cppからincludeするための書き方になっています
#include "EmitterSetHandlingPtcl.h"
#include "testEft_HandlingCommand.h"

//---------------------------------------------------------------------------
//! @brief  基本テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SimpleTest )
{
    {
        const char* pTestName = "SimpleTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        testSet.AddCommand( new CreateEsetCommand( 0, 0 ) );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  エミッタセット全削除テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, KillAllEsetTest )
{
    {
        const char* pTestName = "KillAllEsetTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        testSet.AddCommand( new CreateEsetCommand( 0, 0 ) );
        testSet.AddCommand( new KillAllEsetCommand( 40 ) );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  フェードテストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, FadeTest )
{
    {
        const char* pTestName = "FadeTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        testSet.AddCommand( new CreateEsetCommand( 0, FadeTest ) );
        testSet.AddCommand( new FadeCommand( 25 ) );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  速度への加算値設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetAddVelTest )
{
    {
        const char* pTestName = "SetAddVelTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        testSet.AddCommand( new CreateEsetCommand( 0, 0 ) );
        testSet.AddCommand( new SetAddVelCommand( 30, nw::math::VEC3( 1.0f, 2.0f, 3.0f ) ) );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  全方向速度スケール設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetAllDirectionalVelTest )
{
    {
        const char* pTestName = "SetAllDirectionalVelTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        testSet.AddCommand( new CreateEsetCommand( 0, 0 ) );
        testSet.AddCommand( new SetAllDirectionalVelCommand( 30,  4.0f ) );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  描画アルファ設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetAlphaTest )
{
    {
        const char* pTestName = "SetAlphaTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        testSet.AddCommand( new CreateEsetCommand( 0, 0 ) );
        testSet.AddCommand( new SetAlphaCommand( 30, 0.3f ) );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  計算処理の有効/無効設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetCalcEnableTest )
{
    {
        const char* pTestName = "SetCalcEnableTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        testSet.AddCommand( new CreateEsetCommand( 0, 0 ) );
        testSet.AddCommand( new SetCalcEnableCommand( 30, false ) );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  描画カラー設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetColorTest )
{
    {
        const char* pTestName = "SetColorTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        testSet.AddCommand( new CreateEsetCommand( 0, 0 ) );
        testSet.AddCommand( new SetColorCommand( 30, nw::math::VEC4( 0.8f, 0.6f, 0.4f, 1.0f ) ) );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  指定方向設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetDirectionalTest )
{
    {
        const char* pTestName = "SetDirectionalTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        testSet.AddCommand( new CreateEsetCommand( 0, 0 ) );
        testSet.AddCommand( new SetDirectionalCommand( 30, nw::math::VEC3( 2.0f, 1.5f, 0.7f ) ) );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  指定方向速度スケール設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetDirectionalVelTest )
{
    {
        const char* pTestName = "SetDirectionalVelTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        testSet.AddCommand( new CreateEsetCommand( 0, 0 ) );
        testSet.AddCommand( new SetDirectionalCommand( 30, nw::math::VEC3( 2.0f, 1.5f, 0.7f ) ) );
        testSet.AddCommand( new SetDirectionalVelCommand( 30, 6.5f ) );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  描画プライオリティ設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetDrawPriorityTest )
{
    {
        const char* pTestName = "SetDrawPriorityTest";
        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        testSet.AddCommand( new CreateEsetCommand( 0, 0 ) );
        testSet.AddCommand( new SetDrawPriorityCommand( 30, 0 ) );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  放出スケール設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetEmissionScaleTest )
{
    {
        const char* pTestName = "SetEmissionScaleTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        testSet.AddCommand( new CreateEsetCommand( 0, 0 ) );
        testSet.AddCommand( new SetEmissionParticleScaleCommand( 30, nw::math::VEC2( 2.0f, 1.5f ) ) );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  放出間隔スケール設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetEmissionIntervalScaleTest )
{
    {
        const char* pTestName = "SetEmissionIntervalScaleTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        testSet.AddCommand( new CreateEsetCommand( 0, 0 ) );
        testSet.AddCommand( new SetEmissionIntervalScaleCommand( 30, 0.6f ) );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  放出レートスケール設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetEmissionRatioScaleTest )
{
    {
        const char* pTestName = "SetEmissionRatioScaleTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        testSet.AddCommand( new CreateEsetCommand( 0, 0 ) );
        testSet.AddCommand( new SetEmissionRatioScaleCommand( 30, 0.6f ) );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  エミッタカラー設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetEmitterColorTest )
{
    {
        const char* pTestName = "SetEmitterColorTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        testSet.AddCommand( new CreateEsetCommand( 0, 0 ) );
        testSet.AddCommand( new SetEmitterColorCommand( 30, nw::math::VEC4( 0.2f, 0.4f, 0.8f, 0.5f ), nw::math::VEC4( 0.9f, 0.5f, 0.1f, 0.3f ) ) );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  エミッタスケール設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetEmitterScaleTest )
{
    {
        const char* pTestName = "SetEmitterScaleTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        testSet.AddCommand( new CreateEsetCommand( 0, 0 ) );
        testSet.AddCommand( new SetEmitterScaleCommand( 30, nw::math::VEC3( 2.0f, 1.5f, 0.8f ) ) );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  エミッタ描画処理の有効/無効設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetEmitterVisibleTest )
{
    {
        const char* pTestName = "SetEmitterVisibleTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        testSet.AddCommand( new CreateEsetCommand( 0, 0 ) );
        testSet.AddCommand( new SetEmitterVisibleCommand( 30, 0, false ) );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  初期角度設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetInitRotateTest )
{
    {
        const char* pTestName = "SetInitRotateTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        testSet.AddCommand( new CreateEsetCommand( 0, 0 ) );
        testSet.AddCommand( new SetInitRotateCommand( 30, nw::math::VEC3( 30.0f, 60.0f, 270.0f ) ) );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  手動放出設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetManualParticleEmissionTest )
{
    {
        const char* pTestName = "SetManualParticleEmissionTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        testSet.AddCommand( new CreateEsetCommand( 0, 0 ) );
        testSet.AddCommand( new SetManualParticleEmissionCommand( 30, true ) );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  手動放出設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetParticleCountTest )
{
    {
        const char* pTestName = "SetParticleCountTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        testSet.AddCommand( new CreateEsetCommand( 0, 0 ) );
        testSet.AddCommand( new SetParticleCountCommand( 30, 8 ) );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  行列設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetMatrixTest )
{
    {
        const char* pTestName = "SetMatrixTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        // NOTE:SetMatrixTest と SetMatrixFastTest は計算結果が同じになる事
        nw::math::MTX34 matrix;
        nw::math::VEC3 scale;
        nw::math::VEC3 rotate;
        nw::math::VEC3 translate;

        nw::math::MTX34Identity( &matrix );
        scale.Set( 0.5f, 1.0f, 1.0f );
        rotate.Set( nw::math::DegToRad( 0.0f ), nw::math::DegToRad( 0.0f ), nw::math::DegToRad( 0.0f ) );
        translate.Set( 0.0f, 0.0f, 0.0f );
        nw::math::MTX34RotXYZRad( &matrix, rotate.x, rotate.y, rotate.z );
        nw::math::MTX34MultScale( &matrix, &matrix, &scale );
        matrix.SetColumn( 3, translate );
        testSet.AddCommand( new CreateEsetCommand( 0, TrackingTest ) );
        testSet.AddCommand( new SetMatrixCommand( 0, matrix ) );

        nw::math::MTX34Identity( &matrix );
        scale.Set( 1.0f, 0.5f, 1.0f );
        rotate.Set( nw::math::DegToRad( 45.0f ), nw::math::DegToRad( 0.0f ), nw::math::DegToRad( 0.0f ) );
        translate.Set( 0.0f, 10.0f, 0.0f );
        nw::math::MTX34RotXYZRad( &matrix, rotate.x, rotate.y, rotate.z );
        nw::math::MTX34MultScale( &matrix, &matrix, &scale );
        matrix.SetColumn( 3, translate );
        testSet.AddCommand( new SetMatrixCommand( 5, matrix ) );

        nw::math::MTX34Identity( &matrix );
        scale.Set( 0.75f, 0.75f, 0.0f );
        rotate.Set( nw::math::DegToRad( 0.0f ), nw::math::DegToRad( 0.0f ), nw::math::DegToRad( 45.0f ) );
        translate.Set( 0.0f, -10.0f, 5.0f );
        nw::math::MTX34RotXYZRad( &matrix, rotate.x, rotate.y, rotate.z );
        nw::math::MTX34MultScale(&matrix, &matrix, &scale );
        matrix.SetColumn( 3, translate );
        testSet.AddCommand( new SetMatrixCommand( 20, matrix ) );

        nw::math::MTX34Identity( &matrix );
        scale.Set( 1.25f, 1.25f, 1.25f );
        rotate.Set( nw::math::DegToRad( 0.0f ), nw::math::DegToRad( 0.0f ), nw::math::DegToRad( 135.0f ) );
        translate.Set( 0.0f, 5.0f, 0.0f );
        nw::math::MTX34RotXYZRad( &matrix, rotate.x, rotate.y, rotate.z );
        nw::math::MTX34MultScale( &matrix, &matrix, &scale );
        matrix.SetColumn( 3, translate );
        testSet.AddCommand( new SetMatrixCommand( 45, matrix ) );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  スケール抜き行列設定テストを行います。
//!
//! 注意）正解画像は SetMatrixTest で生成したものを利用する事
//---------------------------------------------------------------------------
TEST( HandlingTest, SetMatrixFastTest )
{
    {
        const char* pTestName = "SetMatrixFastTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        nw::math::MTX34 matrix;
        nw::math::VEC3 scale;
        nw::math::VEC3 rotate;
        nw::math::VEC3 translate;

        nw::math::MTX34Identity( &matrix );
        scale.Set( 0.5f, 1.0f, 1.0f );
        rotate.Set( nw::math::DegToRad( 0.0f ), nw::math::DegToRad( 0.0f ), nw::math::DegToRad( 0.0f ) );
        translate.Set( 0.0f, 0.0f, 0.0f );
        nw::math::MTX34RotXYZRad( &matrix, rotate.x, rotate.y, rotate.z );
        matrix.SetColumn( 3, translate );
        testSet.AddCommand( new CreateEsetCommand( 0, TrackingTest ) );
        testSet.AddCommand( new SetMatrixFastCommand( 0, matrix, scale ) );

        nw::math::MTX34Identity( &matrix );
        scale.Set( 1.0f, 0.5f, 1.0f );
        rotate.Set( nw::math::DegToRad( 45.0f ), nw::math::DegToRad( 0.0f ), nw::math::DegToRad( 0.0f ) );
        translate.Set( 0.0f, 10.0f, 0.0f );
        nw::math::MTX34RotXYZRad( &matrix, rotate.x, rotate.y, rotate.z );
        matrix.SetColumn( 3, translate );
        testSet.AddCommand( new SetMatrixFastCommand( 5, matrix, scale ) );

        nw::math::MTX34Identity( &matrix );
        scale.Set( 0.75f, 0.75f, 0.0f );
        rotate.Set( nw::math::DegToRad( 0.0f ), nw::math::DegToRad( 0.0f ), nw::math::DegToRad( 45.0f ) );
        translate.Set( 0.0f, -10.0f, 5.0f );
        nw::math::MTX34RotXYZRad( &matrix, rotate.x, rotate.y, rotate.z );
        matrix.SetColumn( 3, translate );
        testSet.AddCommand( new SetMatrixFastCommand( 20, matrix, scale ) );

        nw::math::MTX34Identity( &matrix );
        scale.Set( 1.25f, 1.25f, 1.25f );
        rotate.Set( nw::math::DegToRad( 0.0f ), nw::math::DegToRad( 0.0f ), nw::math::DegToRad( 135.0f ) );
        translate.Set( 0.0f, 5.0f, 0.0f );
        nw::math::MTX34RotXYZRad( &matrix, rotate.x, rotate.y, rotate.z );
        matrix.SetColumn( 3, translate );
        testSet.AddCommand( new SetMatrixFastCommand( 45, matrix, scale ) );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  寿命スケール設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetParticleLifeScaleTest )
{
    {
        const char* pTestName = "SetParticleLifeScaleTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        testSet.AddCommand( new CreateEsetCommand( 0, 0 ) );
        testSet.AddCommand( new SetParticleLifeScaleCommand( 30, 0.6f ) );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  パーティクルスケール設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetParticleScaleTest )
{
    {
        const char* pTestName = "SetParticleScaleTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        testSet.AddCommand( new CreateEsetCommand( 0, 0 ) );
        testSet.AddCommand( new SetParticleScaleCommand( 30, nw::math::VEC2( 2.0f, 1.5f ) ) );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  初期速度ランダムスケール設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetRandomVelTest )
{
    {
        const char* pTestName = "SetRandomVelTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        testSet.AddCommand( new CreateEsetCommand( 0, 0 ) );
        testSet.AddCommand( new SetRandomVelCommand( 30, 3.2f ) );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  ユーザーデータ設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetRuntimeUserDataTest )
{
    {
        const char* pTestName = "SetRuntimeUserDataTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        testSet.AddCommand( new CreateEsetCommand( 0, 0 ) );
        testSet.AddCommand( new SetRuntimeUserDataCommand( 30, 3939 ) );

        RunLoop(&testSet, pTestName);
    }
}


//---------------------------------------------------------------------------
//! @brief  開始フレーム設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetStartFrameTest )
{
    {
        const char* pTestName = "SetStartFrameTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        testSet.AddCommand( new CreateEsetCommand( 0, StartFrameTest ) );
        testSet.AddCommand( new SetStartFrameCommand( 0, 15 ) );

        RunLoop(&testSet, pTestName);
    }
}


//---------------------------------------------------------------------------
//! @brief  表示/非表示設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetVisibleTest )
{
    {
        const char* pTestName = "SetVisibleTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem );
        testSet.SetEsetHandle( &g_EsetHandle );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        testSet.AddCommand( new CreateEsetCommand( 0, 0 ) );
        testSet.AddCommand( new SetVisibleCommand( 30, false ) );

        RunLoop(&testSet, pTestName);
    }
}
