﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cctype>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/teamcity/testTeamcity_Logger.h>
#include <nnt/nnt_Argument.h>
#include <nnt/graphics/testGraphics_GetHostExecutableFilepath.h>

#include "testGraphics_Path.h"

#include <regex>
#include <string>
#include <iostream>
#include <sstream>
#include <unordered_map>

struct ProgramArgs
{
    void SetDefault()
    {
        LogFile.SetString( "" );
        CompareFile.SetString( "" );
        dictionary.clear();
    }
    nnt::graphics::Path LogFile;
    nnt::graphics::Path CompareFile;
    std::unordered_map<std::string, double> dictionary;
};

ProgramArgs g_Args;

//----------------------------------------------------------------------------------------------------
static void* FsAllocate(size_t s)
{
    return malloc(s);
}

//----------------------------------------------------------------------------------------------------
static void FsDeallocate(void* p, size_t)
{
    free(p);
}

TEST( ComparePerfWorksData, InternalConsistency )
{
    // Specific Tests
    const double Epsilon = 0.0001;

    EXPECT_GE( g_Args.dictionary[ "system__time_duration" ], g_Args.dictionary[ "gpu__time_duration" ] );
    EXPECT_EQ( g_Args.dictionary[ "gpu__vs_invocations" ], g_Args.dictionary[ "ia__vertex_count" ] );
    EXPECT_LE( fabs( g_Args.dictionary[ "gpu__time_duration" ] - ( g_Args.dictionary[ "gpu__time_end" ] - g_Args.dictionary[ "gpu__time_start" ] ) ), Epsilon );
    EXPECT_LE( fabs( 100.0 * g_Args.dictionary[ "zcull__fragments_trivially_accepted" ] / g_Args.dictionary[ "zcull__fragments_tested" ]
        - g_Args.dictionary[ "zcull__fragments_trivially_accepted_pct" ] ), Epsilon );
    EXPECT_LE( fabs( 100.0 * g_Args.dictionary[ "zcull__tiles_accepted" ] / g_Args.dictionary[ "zcull__tiles_tested" ]
        - g_Args.dictionary[ "zcull__tiles_accepted_pct" ] ), Epsilon );
    EXPECT_LE( fabs( 100.0 * g_Args.dictionary[ "zcull__tiles_rejected" ] / g_Args.dictionary[ "zcull__tiles_tested" ]
        - g_Args.dictionary[ "zcull__tiles_rejected_pct" ] ), Epsilon );
    EXPECT_LE( fabs( 100.0 * g_Args.dictionary[ "zcull__tiles_trivially_accepted" ] / g_Args.dictionary[ "zcull__tiles_tested" ]
        - g_Args.dictionary[ "zcull__tiles_trivially_accepted_pct" ] ), Epsilon );
}

TEST( ComparePerfWorksData, TestPercentages )
{
    // Global Tests
    std::regex percentRgx( "_pct" );
    for (std::unordered_map<std::string, double>::iterator it = g_Args.dictionary.begin(); it != g_Args.dictionary.end(); it++ )
    {
        EXPECT_GE( it->second, 0.0f ) << it->first << " invalid value " << it->second << std::endl;

        if ( std::regex_search( it->first, percentRgx ) )
        {
            double percentage = it->second;
            EXPECT_GE( percentage, 0.0f ) << it->first;
            EXPECT_LE( percentage, 100.0f ) << it->first;
        }
    }

}

TEST( ComparePerfWorksData, TestComparisonFile )
{
    if ( !g_Args.CompareFile.IsEmpty() )
    {
        nn::fs::FileHandle compareFile;
        nn::Result result = nn::fs::OpenFile( &compareFile, g_Args.CompareFile.GetString(), nn::fs::OpenMode_Read );
        NN_ASSERT( result.IsSuccess() );

        int64_t size = 0;
        result = nn::fs::GetFileSize( &size, compareFile );
        NN_ASSERT( result.IsSuccess() );

        void* pData = malloc( size );
        result = nn::fs::ReadFile( compareFile, 0, pData, size );
        NN_ASSERT( result.IsSuccess() );

        nn::fs::CloseFile( compareFile );

        std::istringstream input;
        input.str( reinterpret_cast< char* >( pData ) );

        // Collect test data into Dictionary
        std::string line;
        while ( std::getline(input, line) )
        {
            std::regex lineCaptureRgx( "(\\w+),(\\d+.\\d+)" );

            std::smatch sm;
            if ( std::regex_search( line, sm, lineCaptureRgx ) )
            {
                std::unordered_map< std::string, double>::iterator it = g_Args.dictionary.find( sm[ 1 ] );
                EXPECT_NE( it, g_Args.dictionary.end() );
                if ( it != g_Args.dictionary.end() )
                {
                    EXPECT_EQ( g_Args.dictionary[ it->first ], atof( sm[ 2 ].str().c_str() ) ) << it->first << " mismatch with comparison file!";
                }
            }
        }
    }
}

static void ParseLogFile()
{
    bool bIsRecording = false;
    std::regex rangeRgx( "RANGES");
    std::regex lineCaptureRgx( "\\|--* (\\w+) *= *(\\d+.\\d+)");
    std::regex rangeIdsRgx( "rangeIds" );

    nn::fs::FileHandle inputLog;
    nn::Result result = nn::fs::OpenFile( &inputLog, g_Args.LogFile.GetString(), nn::fs::OpenMode_Read );
    NN_ASSERT( result.IsSuccess() );

    int64_t size = 0;
    result = nn::fs::GetFileSize( &size, inputLog );
    NN_ASSERT( result.IsSuccess() );

    void* pData = malloc( size );
    result = nn::fs::ReadFile( inputLog, 0, pData, size );
    NN_ASSERT( result.IsSuccess() );

    nn::fs::CloseFile( inputLog );

    std::istringstream input;
    input.str( reinterpret_cast< char* >( pData ) );

    // Collect test data into Dictionary
    std::string line;
    while ( std::getline(input, line) )
    {
        // STACK/RANGE data are the same so stop after the RANGE
        if ( std::regex_search( line, rangeRgx ) )
        {
            break;
        }

        if ( bIsRecording )
        {
            std::smatch sm;
            if ( std::regex_search( line, sm, lineCaptureRgx ) )
            {
                g_Args.dictionary.insert( std::pair< std::string, double >( sm[ 1 ], atof( sm[ 2 ].str().c_str() ) ) );
            }
        }
        else
        {
            bIsRecording = std::regex_search( line, rangeIdsRgx );
        }
    }

    EXPECT_GT( g_Args.dictionary.size(), 0 );
}

static void ParseArguments( int argc, char** argv )
{
    g_Args.SetDefault();

    for ( int i = 1; i < argc; i++ )
    {
        NN_LOG( "argv[%d] = %s\n", i, argv[ i ] );
        const char* arg = argv[ i ];
        if ( std::strcmp( arg, "--compare-file" ) == 0 )
        {
            NN_ASSERT(i + 1 < argc);
            NN_LOG("argv[%d] = %s\n", i + 1, argv[i + 1]);
            g_Args.CompareFile = nnt::graphics::Path(argv[ i + 1 ]).Normalize();
            i += 1;
        }
        else if ( std::strcmp( arg, "--log-file" ) == 0 )
        {
            NN_ASSERT(i + 1 < argc);
            NN_LOG("argv[%d] = %s\n", i + 1, argv[i + 1]);
            g_Args.LogFile = nnt::graphics::Path(argv[ i + 1 ]).Normalize();
            i += 1;
        }
        else if ( std::strcmp( arg, "--help" ) == 0 )
        {
            FAIL() << "--log-file filePath [--compare-file compareFile]";
        }
    }

    if ( g_Args.LogFile.IsEmpty() )
    {
        FAIL() << "Must specify --log-file";
    }
}

extern "C" void nnMain()
{
    int argc = ::nnt::GetHostArgc();
    char** argv = ::nnt::GetHostArgv();

    nn::fs::SetAllocator(FsAllocate, FsDeallocate);
    nn::fs::MountHostRoot();

    NN_LOG("Running nnMain() from testGraphics_TestMain.cpp\n");
    ::testing::InitGoogleTest(&argc, argv);

    ::testing::TestEventListeners& listeners =
        ::testing::UnitTest::GetInstance()->listeners();
    ::testing::TestEventListener* defaultResultPrinter =
        listeners.Release(listeners.default_result_printer());
#if defined(NN_BUILD_CONFIG_HARDWARE_BDSLIMX6) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    listeners.Append(new ::nnt::teamcity::ServiceMessageLogger());
#endif
    listeners.Append(defaultResultPrinter);

    ParseArguments( argc, argv );

    // Setup Dictionary
    ParseLogFile();

    int result = RUN_ALL_TESTS();

    ::nnt::Exit(result);
}
