﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testGraphics_ImageCompareResult.h"

#include <malloc.h>
#include <algorithm>
#include <cstring>
#include <sstream>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/util/util_FormatString.h>
#include "testGraphics_PngIO.h"
#include "testGraphics_PixelwiseImageComparison.h"
#include "testGraphics_Path.h"

namespace{
    static const int MaxPathLength = 512;
    static const int MaxHtmlTableCellValueLength = 1024;
    static const int HtmlTableImageColumnWidth = 384;
    static const int HtmlTableHeaderColumnWidth = 60;
    static const int HtmlTableResultColumnWidth = 40;
    static const int HtmlTableSummaryColumnWidth = 200;
    static const int HtmlTableTitleRowHeight = 40;
    static const int HtmlTableImageRowHeight = 216;
    static const char* ImageSpecImage      = "captured";
    static const char* ImageSpecReference  = "reference";
    static const char* ImageSpecDifference = "difference";

    void CreateDirectories(const nnt::graphics::Path* path)
    {
        if(path->IsRoot())
        {
            return;
        }
        nn::Result result = nn::fs::CreateDirectory(path->GetString());
        if(result.IsSuccess() || nn::fs::ResultPathAlreadyExists::Includes(result))
        {
            return;
        }
        else if(nn::fs::ResultPathNotFound::Includes(result))
        {
            nnt::graphics::Path parent;
            path->GetParent(&parent);
            CreateDirectories(&parent);
            result = nn::fs::CreateDirectory(path->GetString());
            NN_ASSERT(result.IsSuccess());
        }
        else
        {
            NN_ASSERT(result.IsSuccess());
        }
    }
}

namespace nnt{ namespace graphics{


    ImageCompareResult::ImageCompareResult()
        : m_ResultDirPath(NULL),
          m_NumberOfTotalFrames(0),
          m_NumberOfOkFrames(0),
          m_NumberOfNgFrames(0)
    {
    }
    ImageCompareResult::~ImageCompareResult()
    {
        this->Finalize();
    }

    void ImageCompareResult::Initialize(const char* resultDirPath, const Allocator* allocator)
    {
        NN_ASSERT(resultDirPath);
        m_Allocator = allocator;
        m_ResultDirPath.SetString(resultDirPath);

        m_ResultHtmlTable.headerName = "Results";
        m_NgResultHtmlTable.headerName = "Results";

        m_NumberOfTotalFrames = 0;
        m_NumberOfOkFrames = 0;
        m_NumberOfNgFrames = 0;


        // すでに存在していれば削除します。
        nn::Result result = nn::fs::CleanDirectoryRecursively(resultDirPath);
        if (nn::fs::ResultTargetLocked::Includes(result))
        {
            NN_ASSERT("Can not clean a result directory.[%s]\n", resultDirPath);
        }

        m_CapturedImageDirectoryPath.SetString(m_ResultDirPath.Combine(Path(ImageSpecImage)).GetString());
        m_ReferenceImageDirectoryPath.SetString(m_ResultDirPath.Combine(Path(ImageSpecReference)).GetString());
        m_DifferenceImageDirectoryPath.SetString(m_ResultDirPath.Combine(Path(ImageSpecDifference)).GetString());

        CreateDirectories(&m_CapturedImageDirectoryPath);
        CreateDirectories(&m_ReferenceImageDirectoryPath);
        CreateDirectories(&m_DifferenceImageDirectoryPath);
    }
    void ImageCompareResult::Finalize()
    {
    }

    //-----------------------------------------------------------

    const char* ImageCompareResult::GetCapturedImageDirectoryPath() const NN_NOEXCEPT
    {
        return m_CapturedImageDirectoryPath.GetString();
    }
    const char* ImageCompareResult::GetReferenceImageDirectoryPath() const NN_NOEXCEPT
    {
        return m_ReferenceImageDirectoryPath.GetString();
    }
    const char* ImageCompareResult::GetDifferenceImageDirectoryPath() const NN_NOEXCEPT
    {
        return m_DifferenceImageDirectoryPath.GetString();
    }

    //-----------------------------------------------------------

    bool ImageCompareResult::HasNgFrame() const
    {
        return m_NumberOfNgFrames != 0;
    }

    //-----------------------------------------------------------

    static void CreateResultHtmlTitleCellValue(char* pOutBuffer, int outBufferSize, const char* imageSpec, const char* title)
    {
        int result = nn::util::SNPrintf(pOutBuffer, outBufferSize, "%s/%s", imageSpec, title);
        NN_ASSERT(result < outBufferSize);
    }
    static void CreateResultHtmlFrameCellValue(char* pOutBuffer, int outBufferSize, const char* name)
    {
        int result = nn::util::SNPrintf(pOutBuffer, outBufferSize, "%s", name);
        NN_ASSERT(result < outBufferSize);
    }
    static void CreateResultHtmlImageCellValue(char* pOutBuffer, int outBufferSize, const char* imagePath, int width, int height)
    {
        int result = nn::util::SNPrintf(
            pOutBuffer, outBufferSize,
            "<a href=\"%s\">"
                "<img src=\"%s\" width=\"%d\" height=\"%d\" />"
            "</a>",
            imagePath,
                imagePath, width, height
            /* nothing */
            );
        NN_ASSERT(result < outBufferSize);
    }
    static void CreateResultHtmlEmptyImageCellValue(char* pOutBuffer, int outBufferSize)
    {
        int result = nn::util::SNPrintf(
            pOutBuffer, outBufferSize,
            "NoImage"
            );
        NN_ASSERT(result < outBufferSize);
    }
    static void CreateResultHtmlResultCellValue(char* pOutBuffer, int outBufferSize, bool isOk)
    {
        if(isOk)
        {
            int result = nn::util::SNPrintf(pOutBuffer, outBufferSize, "<font color=\"#00ff00\"><strong>ok</strong></font>");
            NN_ASSERT(result < outBufferSize);
        }
        else
        {
            int result = nn::util::SNPrintf(pOutBuffer, outBufferSize, "<font color=\"#ff0000\"><strong>NG</strong></font>");
            NN_ASSERT(result < outBufferSize);
        }
    }

    void ImageCompareResult::AddDataHeader(const char* title, nnt::gfx::NntGfxResultHtml::Table& table)
    {
        char cellValue[MaxHtmlTableCellValueLength] = {};
        nnt::gfx::NntGfxResultHtml::TableLine row;
        {
            nnt::gfx::NntGfxResultHtml::TableData cell;
            cell.width  = HtmlTableHeaderColumnWidth;
            cell.height = HtmlTableTitleRowHeight;

            cell.outValue = "Frame";
            row.dataArray.push_back(cell);
        }
        {
            nnt::gfx::NntGfxResultHtml::TableData cell;
            cell.width  = HtmlTableImageColumnWidth;
            cell.height = HtmlTableTitleRowHeight;

            CreateResultHtmlTitleCellValue(cellValue, MaxHtmlTableCellValueLength, ImageSpecImage, title);
            cell.outValue = cellValue;
            row.dataArray.push_back(cell);

            CreateResultHtmlTitleCellValue(cellValue, MaxHtmlTableCellValueLength, ImageSpecReference, title);
            cell.outValue = cellValue;
            row.dataArray.push_back(cell);

            CreateResultHtmlTitleCellValue(cellValue, MaxHtmlTableCellValueLength, ImageSpecDifference, title);
            cell.outValue = cellValue;
            row.dataArray.push_back(cell);
        }
        {
            nnt::gfx::NntGfxResultHtml::TableData cell;
            cell.width  = HtmlTableResultColumnWidth;
            cell.height = HtmlTableTitleRowHeight;

            cell.outValue = "Result";
            row.dataArray.push_back(cell);
        }
        table.lineArray.push_back(row);
    }

    static void CopyImageFile(const Path& srcPath, const Path& dstPath)
    {
        if(srcPath.IsEmpty() || dstPath.IsEmpty())
        {
            return;
        }
        const int64_t BufferSize = 512;
        char buffer[BufferSize];
        if(dstPath.IsEqual(srcPath))
        {
            return;
        }

        nn::Result result;
        nn::fs::FileHandle srcFile;
        nn::fs::FileHandle dstFile;

        result = nn::fs::OpenFile(&srcFile, srcPath.GetString(), nn::fs::OpenMode_Read);
        NN_ASSERT(result.IsSuccess());

        int64_t size;
        result = nn::fs::GetFileSize(&size, srcFile);
        NN_ASSERT(result.IsSuccess());

        result = nn::fs::CreateFile(dstPath.GetString(), size);
        NN_ASSERT(result.IsSuccess() || nn::fs::ResultPathAlreadyExists::Includes(result));
        result = nn::fs::OpenFile(&dstFile, dstPath.GetString(), nn::fs::OpenMode_Write);
        NN_ASSERT(result.IsSuccess());
        result = nn::fs::SetFileSize(dstFile, size);
        NN_ASSERT(result.IsSuccess());

        int64_t pos = 0;
        while(size > 0)
        {
            int64_t sizeToCopy = std::min(size, BufferSize);
            result = nn::fs::ReadFile(srcFile, pos, buffer, static_cast<size_t>(sizeToCopy));
            NN_ASSERT(result.IsSuccess());
            result = nn::fs::WriteFile(dstFile, pos, buffer, static_cast<size_t>(sizeToCopy), nn::fs::WriteOption::MakeValue(0));
            NN_ASSERT(result.IsSuccess());
            size -= sizeToCopy;
            pos += sizeToCopy;
        }
        NN_ASSERT(size == 0);

        result = nn::fs::FlushFile(dstFile);
        NN_ASSERT(result.IsSuccess());

        nn::fs::CloseFile(dstFile);
        nn::fs::CloseFile(srcFile);

    }

    void ImageCompareResult::AddCompareResult(
            bool success,
            int difference,
            const char* name,
            const char* capImgPath,
            const char* refImgPath,
            const char* difImgPath)
    {
        NN_UNUSED(difference);
        // ファイルを出力ディレクトリにコピー
        Path capPath(capImgPath);
        Path refPath(refImgPath);
        Path difPath(difImgPath);
        CopyImageFile(capPath, m_CapturedImageDirectoryPath.Combine(capPath.GetFilename()));
        CopyImageFile(refPath, m_ReferenceImageDirectoryPath.Combine(refPath.GetFilename()));
        CopyImageFile(difPath, m_DifferenceImageDirectoryPath.Combine(difPath.GetFilename()));
        Path relCapRoot(ImageSpecImage);
        Path relRefRoot(ImageSpecReference);
        Path relDifRoot(ImageSpecDifference);
        // 結果 HTML に登録
        {
            char cellValue[MaxHtmlTableCellValueLength] = {};
            nnt::gfx::NntGfxResultHtml::TableLine row;
            {
                nnt::gfx::NntGfxResultHtml::TableData cell;
                cell.width  = HtmlTableHeaderColumnWidth;
                cell.height = HtmlTableImageRowHeight;

                CreateResultHtmlFrameCellValue(cellValue, MaxHtmlTableCellValueLength, name);
                cell.outValue = cellValue;
                row.dataArray.push_back(cell);
            }
            {
                nnt::gfx::NntGfxResultHtml::TableData cell;
                cell.width  = HtmlTableImageColumnWidth;
                cell.height = HtmlTableImageRowHeight;

                if(!capPath.IsEmpty())
                {
                    CreateResultHtmlImageCellValue(cellValue, MaxHtmlTableCellValueLength, relCapRoot.Combine(capPath.GetFilename()).GetString(), cell.width, cell.height);
                }
                else
                {
                    CreateResultHtmlEmptyImageCellValue(cellValue, MaxHtmlTableCellValueLength);
                }
                cell.outValue = cellValue;
                row.dataArray.push_back(cell);

                if(!refPath.IsEmpty())
                {
                    CreateResultHtmlImageCellValue(cellValue, MaxHtmlTableCellValueLength, relRefRoot.Combine(refPath.GetFilename()).GetString(), cell.width, cell.height);
                }
                else
                {
                    CreateResultHtmlEmptyImageCellValue(cellValue, MaxHtmlTableCellValueLength);
                }
                cell.outValue = cellValue;
                row.dataArray.push_back(cell);

                if(!difPath.IsEmpty())
                {
                    CreateResultHtmlImageCellValue(cellValue, MaxHtmlTableCellValueLength, relDifRoot.Combine(difPath.GetFilename()).GetString(), cell.width, cell.height);
                }
                else
                {
                    CreateResultHtmlEmptyImageCellValue(cellValue, MaxHtmlTableCellValueLength);
                }
                cell.outValue = cellValue;
                row.dataArray.push_back(cell);
            }
            {
                nnt::gfx::NntGfxResultHtml::TableData cell;
                cell.width  = HtmlTableResultColumnWidth;
                cell.height = HtmlTableImageRowHeight;

                CreateResultHtmlResultCellValue(cellValue, MaxHtmlTableCellValueLength, success);
                cell.outValue = cellValue;
                row.dataArray.push_back(cell);
            }
            m_ResultHtmlTable.lineArray.push_back(row);

            if (!success)
            {
                m_NgResultHtmlTable.lineArray.push_back(row);
            }

        }
        // カウンタを更新
        m_NumberOfTotalFrames++;
        if(success)
        {
            m_NumberOfOkFrames++;
        }
        else
        {
            m_NumberOfNgFrames++;
        }
    }

    static std::string CreateResultHtmlString(nnt::gfx::NntGfxResultHtml& resultHtml)
    {
        typedef nnt::gfx::NntGfxResultHtml::Table Table;
        typedef nnt::gfx::NntGfxResultHtml::TableLine TableLine;
        typedef nnt::gfx::NntGfxResultHtml::TableData TableData;
        int tableCount = static_cast<int>(resultHtml.GetArray()->size());
        std::stringstream sStream;

        sStream << "<html>" << std::endl;
        sStream << "<body>" << std::endl;

        for ( int i = 0; i < tableCount; i++ )
        {
            Table table = resultHtml.GetArray()->at(i);

            sStream << "<h4>" << table.headerName.c_str() << "</h4>" << std::endl;
            sStream << "<table border=\"1\">" << std::endl;

            int lineCount = static_cast<int>(table.lineArray.size());
            for ( int j = 0; j < lineCount; j++ )
            {
                TableLine tableLine = table.lineArray.at(j);
                int dataCount = static_cast<int>(tableLine.dataArray.size());

                sStream << "<tr>" << std::endl;

                for ( int n = 0; n < dataCount; n++ )
                {
                    TableData data = tableLine.dataArray.at(n);

                    sStream << "    <td width=\"" << data.width << "\" height=\"" << data.height << "\" align=\"center\">" << data.outValue.c_str() << "</td>" << std::endl;
                }

                sStream << "</tr>" << std::endl;
            }
            sStream << "</table>" << std::endl << std::endl;
        }

        sStream << "</html>" << std::endl;
        sStream << "</body>" << std::endl;

        return sStream.str();
    }

    void ImageCompareResult::SetupSummaryTable(nnt::gfx::NntGfxResultHtml::Table* pOutValue)
    {
        char cellValue[MaxHtmlTableCellValueLength] = {};
        nnt::gfx::NntGfxResultHtml::TableLine row;
        nnt::gfx::NntGfxResultHtml::TableData cell;
        pOutValue->headerName = "Summary";
        cell.height = HtmlTableTitleRowHeight;
        {
            cell.width = HtmlTableHeaderColumnWidth;
            cell.outValue = "Total";
            row.dataArray.push_back(cell);

            cell.width = HtmlTableSummaryColumnWidth;
            nn::util::SNPrintf(cellValue, MaxHtmlTableCellValueLength, "%d frames", m_NumberOfTotalFrames);
            cell.outValue = cellValue;
            row.dataArray.push_back(cell);
        }
        pOutValue->lineArray.push_back(row);
        row.dataArray.clear();
        {
            cell.width = HtmlTableHeaderColumnWidth;
            cell.outValue = "OK";
            row.dataArray.push_back(cell);

            cell.width = HtmlTableSummaryColumnWidth;
            if(m_NumberOfOkFrames == m_NumberOfTotalFrames)
            {
                nn::util::SNPrintf(cellValue, MaxHtmlTableCellValueLength, "<font color=\"#00ff00\"><strong>%d frames</strong></font>", m_NumberOfOkFrames);
            }
            else
            {
                nn::util::SNPrintf(cellValue, MaxHtmlTableCellValueLength, "%d frames", m_NumberOfOkFrames);
            }
            cell.outValue = cellValue;
            row.dataArray.push_back(cell);
        }
        pOutValue->lineArray.push_back(row);
        row.dataArray.clear();
        {
            cell.width = HtmlTableHeaderColumnWidth;
            cell.outValue = "NG";
            row.dataArray.push_back(cell);

            cell.width = HtmlTableSummaryColumnWidth;
            if(m_NumberOfNgFrames > 0)
            {
                nn::util::SNPrintf(cellValue, MaxHtmlTableCellValueLength, "<font color=\"#ff0000\"><strong>%d frames</strong></font>", m_NumberOfNgFrames);
                cell.outValue = cellValue;
            }
            else
            {
                cell.outValue = "0";
            }
            row.dataArray.push_back(cell);
        }
        pOutValue->lineArray.push_back(row);
    }

    void ImageCompareResult::SaveResultHtml()
    {
        nnt::gfx::NntGfxResultHtml resultHtml;
        {
            // サマリのテーブルを作成
            nnt::gfx::NntGfxResultHtml::Table table;
            SetupSummaryTable(&table);
            resultHtml.GetArray()->push_back(table);
            // 結果のテーブルを追加
            resultHtml.GetArray()->push_back(m_ResultHtmlTable);
        }

        Path outPath = m_ResultDirPath.Combine(Path("index.html"));
        std::string data = CreateResultHtmlString(resultHtml);

        // 書き出し
        {
            const char* path = outPath.GetString();
            size_t fileSize = data.size();
            nn::Result result;
            result = nn::fs::DeleteFile(path);
            NN_ASSERT(result.IsSuccess() || nn::fs::ResultPathNotFound::Includes(result));
            nn::fs::FileHandle file = {};
            result = nn::fs::CreateFile(path, static_cast<int64_t>(fileSize));
            NN_ASSERT(result.IsSuccess());
            result = nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Write);
            NN_ASSERT(result.IsSuccess());
            result = nn::fs::WriteFile(file, 0, data.c_str(), fileSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
            NN_ASSERT(result.IsSuccess());
            nn::fs::CloseFile(file);
        }
    }

    void ImageCompareResult::SaveNgResultHtml(bool success)
    {
        Path outPath = m_ResultDirPath.Combine(Path("error.html"));

        if (success)
        {
            // 書き出し
            {
                const char* path = outPath.GetString();
                nn::Result result;
                result = nn::fs::DeleteFile(path);
                NN_ASSERT(result.IsSuccess() || nn::fs::ResultPathNotFound::Includes(result));
            }
        }
        else {
            nnt::gfx::NntGfxResultHtml resultHtml;
            {
                // サマリのテーブルを作成
                nnt::gfx::NntGfxResultHtml::Table table;
                SetupSummaryTable(&table);
                resultHtml.GetArray()->push_back(table);
                // 結果のテーブルを追加
                resultHtml.GetArray()->push_back(m_NgResultHtmlTable);
            }

            std::string data = CreateResultHtmlString(resultHtml);

            // 書き出し
            {
                const char* path = outPath.GetString();
                size_t fileSize = data.size();
                nn::Result result;
                result = nn::fs::DeleteFile(path);
                NN_ASSERT(result.IsSuccess() || nn::fs::ResultPathNotFound::Includes(result));
                nn::fs::FileHandle file = {};
                result = nn::fs::CreateFile(path, static_cast<int64_t>(fileSize));
                NN_ASSERT(result.IsSuccess());
                result = nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Write);
                NN_ASSERT(result.IsSuccess());
                result = nn::fs::WriteFile(file, 0, data.c_str(), fileSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
                NN_ASSERT(result.IsSuccess());
                nn::fs::CloseFile(file);
            }
        }
    }

}}
