﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Span.h>
#include <nn/devicecode/devicecode_Builder.h>

#include <nn/gpio/driver/gpio_Lib.h>
#include <nn/gpio/driver/gpio_Pad.h>
#include <nn/gpio/driver/gpio_DriverService.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#define NNT_GPIO_STUBDRIVER_DEVICECODE_TEST01 NN_DEVICECODE_MAKE_CODE(TEST, 0x0001)
#define NNT_GPIO_STUBDRIVER_DEVICECODE_TEST02 NN_DEVICECODE_MAKE_CODE(TEST, 0x0002)
#define NNT_GPIO_STUBDRIVER_DEVICECODE_TEST03 NN_DEVICECODE_MAKE_CODE(TEST, 0x0003)
#define NNT_GPIO_STUBDRIVER_DEVICECODE_TEST04 NN_DEVICECODE_MAKE_CODE(TEST, 0x0004)
#define NNT_GPIO_STUBDRIVER_DEVICECODE_TEST05 NN_DEVICECODE_MAKE_CODE(TEST, 0x0005)
#define NNT_GPIO_STUBDRIVER_DEVICECODE_TEST06 NN_DEVICECODE_MAKE_CODE(TEST, 0x0006)
#define NNT_GPIO_STUBDRIVER_DEVICECODE_TEST07 NN_DEVICECODE_MAKE_CODE(TEST, 0x0007)
#define NNT_GPIO_STUBDRIVER_DEVICECODE_TEST08 NN_DEVICECODE_MAKE_CODE(TEST, 0x0008)
#define NNT_GPIO_STUBDRIVER_DEVICECODE_TEST09 NN_DEVICECODE_MAKE_CODE(TEST, 0x0009)
#define NNT_GPIO_STUBDRIVER_DEVICECODE_TEST0A NN_DEVICECODE_MAKE_CODE(TEST, 0x000A)

namespace
{
    int g_LoopCount = 0; // 汎用カウンタ変数

    //
    // テスト用の仮想デバイス・ドライバ群
    // StubPad : 仮想ピン。各種状態を直接取得・変更可能。 StubInterruptHandler の待機対象イベントを注入することで割り込みの挙動をエミュレート可能
    // StubDriver : 仮想ドライバ。コンストラクト時に指定した 1 つ以上の仮想ピンを管理対象にする
    // StubInterruptHandler : 仮想割り込みハンドラ。コンストラクト時に指定した StubDriver に対応する
    //
    class StubPad: public nn::gpio::driver::Pad
    {
    public:
        StubPad(nn::DeviceCode deviceCode, int padNumber) NN_NOEXCEPT :
            Pad(padNumber),
            m_DeviceCode(deviceCode),
            m_Value(nn::gpio::GpioValue_Low),
            m_Direction(nn::gpio::Direction_Input),
            m_InterruptMode(nn::gpio::InterruptMode_LowLevel),
            m_InterruptEnabled(false),
            m_InterruptStatus(nn::gpio::InterruptStatus_Inactive),
            m_pInterruptEvent(nullptr)
        {}

        void SetInterruptEvent(nn::os::Event* pInterruptEvent) NN_NOEXCEPT { m_pInterruptEvent = pInterruptEvent; }

        nn::DeviceCode GetDeviceCode() const NN_NOEXCEPT { return m_DeviceCode; }

        nn::gpio::Direction GetDirection() const NN_NOEXCEPT { return m_Direction; }
        void SetDirection(nn::gpio::Direction direction) NN_NOEXCEPT { m_Direction = direction; }

        nn::gpio::GpioValue GetValue() const NN_NOEXCEPT { return m_Value; }
        void SetValue(nn::gpio::GpioValue newValue) NN_NOEXCEPT
        {
            // value と interruptMode に合わせて interruptStatus を操作する
            auto prevValue = m_Value;
            m_Value = newValue;

            if ( m_InterruptEnabled )
            {
                switch ( m_InterruptMode )
                {
                    case nn::gpio::InterruptMode_AnyEdge:
                        if ( prevValue != m_Value )
                        {
                            ActivateInterrupt();
                        }
                        break;
                    case nn::gpio::InterruptMode_FallingEdge:
                        if ( prevValue != m_Value && m_Value == nn::gpio::GpioValue_Low )
                        {
                            ActivateInterrupt();
                        }
                        break;
                    case nn::gpio::InterruptMode_RisingEdge:
                        if ( prevValue != m_Value && m_Value == nn::gpio::GpioValue_High )
                        {
                            ActivateInterrupt();
                        }
                        break;
                    case nn::gpio::InterruptMode_HighLevel:
                        if ( m_Value == nn::gpio::GpioValue_High )
                        {
                            ActivateInterrupt();
                        }
                        else
                        {
                            DeactivateInterrupt();
                        }
                        break;
                    case nn::gpio::InterruptMode_LowLevel:
                        if ( m_Value == nn::gpio::GpioValue_Low )
                        {
                            ActivateInterrupt();
                        }
                        else
                        {
                            DeactivateInterrupt();
                        }
                        break;
                } // NOLINT(style/switch_default)
            }
        }

        nn::gpio::InterruptMode GetInterruptMode() const NN_NOEXCEPT { return m_InterruptMode; }
        void SetInterruptMode(nn::gpio::InterruptMode interruptMode) NN_NOEXCEPT { m_InterruptMode = interruptMode; }

        bool GetInterruptEnabled() const NN_NOEXCEPT { return m_InterruptEnabled; }
        void SetInterruptEnabled(bool enable) NN_NOEXCEPT
        {
            m_InterruptEnabled = enable;
            if ( enable )
            {
                switch ( m_InterruptMode )
                {
                    case nn::gpio::InterruptMode_AnyEdge:
                    case nn::gpio::InterruptMode_FallingEdge:
                    case nn::gpio::InterruptMode_RisingEdge:
                        break;
                    case nn::gpio::InterruptMode_HighLevel:
                        if ( m_Value == nn::gpio::GpioValue_High )
                        {
                            ActivateInterrupt();
                        }
                        else
                        {
                            DeactivateInterrupt();
                        }
                        break;
                    case nn::gpio::InterruptMode_LowLevel:
                        if ( m_Value == nn::gpio::GpioValue_Low )
                        {
                            ActivateInterrupt();
                        }
                        else
                        {
                            DeactivateInterrupt();
                        }
                        break;
                } // NOLINT(style/switch_default)
            }
            else
            {
                DeactivateInterrupt();
            }
        }

        nn::gpio::InterruptStatus GetInterruptStatus() const NN_NOEXCEPT { return m_InterruptStatus; }
        void ClearInterruptStatus() NN_NOEXCEPT
        {
            DeactivateInterrupt();
        }

        nn::os::SdkMutex& GetInterruptControlMutex() const NN_NOEXCEPT { return m_InterruptControlMutex; }

    private:
        void ActivateInterrupt() NN_NOEXCEPT
        {
            m_InterruptStatus = nn::gpio::InterruptStatus_Active;
            if ( m_pInterruptEvent )
            {
                m_pInterruptEvent->Signal();
            }
        }
        void DeactivateInterrupt() NN_NOEXCEPT
        {
            m_InterruptStatus = nn::gpio::InterruptStatus_Inactive;
            if ( m_pInterruptEvent )
            {
                m_pInterruptEvent->Clear();
            }
        }

    private:
        nn::DeviceCode m_DeviceCode;
        nn::gpio::GpioValue m_Value;
        nn::gpio::Direction m_Direction;
        nn::gpio::InterruptMode m_InterruptMode;
        bool m_InterruptEnabled;
        nn::gpio::InterruptStatus m_InterruptStatus;
        mutable nn::os::SdkMutex m_InterruptControlMutex;
        nn::os::Event* m_pInterruptEvent;
    };
    StubPad PadInfoList0[] = {
        { NNT_GPIO_STUBDRIVER_DEVICECODE_TEST01, 0x1 },
        { NNT_GPIO_STUBDRIVER_DEVICECODE_TEST02, 0x2 },
    };
    StubPad PadInfoList1[] = {
        { NNT_GPIO_STUBDRIVER_DEVICECODE_TEST03, 0x3 },
        { NNT_GPIO_STUBDRIVER_DEVICECODE_TEST04, 0x4 },
    };
    StubPad PadInfoList2[] = {
        { NNT_GPIO_STUBDRIVER_DEVICECODE_TEST05, 0x5 },
        { NNT_GPIO_STUBDRIVER_DEVICECODE_TEST06, 0x6 },
    };
    StubPad PadInfoList3[] = {
        { NNT_GPIO_STUBDRIVER_DEVICECODE_TEST07, 0x7 },
        { NNT_GPIO_STUBDRIVER_DEVICECODE_TEST08, 0x8 },
    };
    nn::util::Span<StubPad> PadInfoSpan[] =
    {
        nn::util::MakeSpan(PadInfoList0),
        nn::util::MakeSpan(PadInfoList1),
        nn::util::MakeSpan(PadInfoList2),
        nn::util::MakeSpan(PadInfoList3),
    };

    class StubDriver :
        public nn::gpio::driver::IGpioDriver
    {
        NN_DDSF_CAST_SAFE_DECL;

    public:
        explicit StubDriver(const nn::util::Span<StubPad>& padList) NN_NOEXCEPT :
            m_PadList(padList)
        {}

        // IGpioDriver: ドライバ初期化
        virtual void InitializeDriver() NN_NOEXCEPT NN_OVERRIDE
        {
            for ( auto&& pad : m_PadList )
            {
                RegisterDevice(&pad);
                nn::gpio::driver::RegisterDeviceCode(pad.GetDeviceCode(), &pad);
            }
        }

        // IGpioDriver: パッド制御
        virtual nn::Result GetDirection(nn::gpio::Direction* pOut, nn::gpio::driver::Pad* pPad) const NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ABORT_UNLESS_NOT_NULL(pOut);
            NN_ABORT_UNLESS_NOT_NULL(pPad);
            *pOut = pPad->SafeCastTo<StubPad>().GetDirection();
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result SetDirection(nn::gpio::driver::Pad* pPad, nn::gpio::Direction direction) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ABORT_UNLESS_NOT_NULL(pPad);
            pPad->SafeCastTo<StubPad>().SetDirection(direction);
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result GetValue(nn::gpio::GpioValue* pOut, nn::gpio::driver::Pad* pPad) const NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ABORT_UNLESS_NOT_NULL(pOut);
            NN_ABORT_UNLESS_NOT_NULL(pPad);
            *pOut = pPad->SafeCastTo<StubPad>().GetValue();
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result SetValue(nn::gpio::driver::Pad* pPad, nn::gpio::GpioValue value) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ABORT_UNLESS_NOT_NULL(pPad);
            pPad->SafeCastTo<StubPad>().SetValue(value);
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result GetInterruptMode(nn::gpio::InterruptMode* pOut, nn::gpio::driver::Pad* pPad) const NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ABORT_UNLESS_NOT_NULL(pOut);
            NN_ABORT_UNLESS_NOT_NULL(pPad);
            *pOut = pPad->SafeCastTo<StubPad>().GetInterruptMode();
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result SetInterruptMode(nn::gpio::driver::Pad* pPad, nn::gpio::InterruptMode mode) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ABORT_UNLESS_NOT_NULL(pPad);
            pPad->SafeCastTo<StubPad>().SetInterruptMode(mode);
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result SetInterruptEnabled(nn::gpio::driver::Pad* pPad, bool enable) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ABORT_UNLESS_NOT_NULL(pPad);
            pPad->SafeCastTo<StubPad>().SetInterruptEnabled(enable);
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result GetInterruptStatus(nn::gpio::InterruptStatus* pOut, nn::gpio::driver::Pad* pPad) const NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ABORT_UNLESS_NOT_NULL(pOut);
            NN_ABORT_UNLESS_NOT_NULL(pPad);
            *pOut = pPad->SafeCastTo<StubPad>().GetInterruptStatus();
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result ClearInterruptStatus(nn::gpio::driver::Pad* pPad) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ABORT_UNLESS_NOT_NULL(pPad);
            pPad->SafeCastTo<StubPad>().ClearInterruptStatus();
            NN_RESULT_SUCCESS;
        }
        virtual nn::os::SdkMutex& GetInterruptControlMutex(const nn::gpio::driver::Pad& pad) const NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(pad);
            return pad.SafeCastTo<const StubPad>().GetInterruptControlMutex();
        }

        void HandleEvent() NN_NOEXCEPT
        {
            for ( auto&& pad : m_PadList )
            {
                std::lock_guard<decltype(GetInterruptControlMutex(pad))> lock(GetInterruptControlMutex(pad));

                if ( pad.GetInterruptStatus() == nn::gpio::InterruptStatus_Active )
                {
                    pad.SetInterruptEnabled(false);
                    pad.SignalInterruptBoundEvent();
                }
            }
        }

    private:
        nn::util::Span<StubPad> m_PadList;
    };

    class StubInterruptHandler :
        public nn::ddsf::IEventHandler
    {
    public:
        explicit StubInterruptHandler(StubDriver* pDriver) NN_NOEXCEPT :
            m_pDriver(pDriver)
        {
            IEventHandler::Initialize(m_InterruptEvent.GetBase());
        }
        nn::os::Event* GetInterruptEvent() NN_NOEXCEPT
        {
            return &m_InterruptEvent;
        }
        virtual void HandleEvent() NN_NOEXCEPT NN_OVERRIDE
        {
            m_pDriver->HandleEvent();
        }
    private:
        StubDriver* m_pDriver{ nullptr };
        nn::os::Event m_InterruptEvent{ nn::os::EventClearMode_ManualClear };
    };
}

NN_DDSF_CAST_SAFE_DEFINE(StubDriver, nn::gpio::driver::IGpioDriver);

//-------------------------------------------------------------------------

//! 初期化終了のみの単純なテスト
TEST(testGpio_SubsystemUnitTest, ImmediateFinalize)
{
    auto& padInfoSpan = PadInfoSpan[0];
    StubDriver driver(padInfoSpan);
    StubInterruptHandler interruptHandler(&driver);

    nn::gpio::driver::RegisterDriver(&driver);
    nn::gpio::driver::Initialize();
    nn::gpio::driver::RegisterInterruptHandler(&interruptHandler);

    nn::gpio::driver::UnregisterInterruptHandler(&interruptHandler);
    nn::gpio::driver::Finalize();
    nn::gpio::driver::UnregisterDriver(&driver);
}

//! 複数のドライバオブジェクト登録のテスト
TEST(testGpio_SubsystemUnitTest, RegisterDriver)
{
    StubDriver driver0(PadInfoSpan[0]);
    StubDriver driver1(PadInfoSpan[1]);
    StubDriver driver2(PadInfoSpan[2]);

    EXPECT_DEATH_IF_SUPPORTED(nn::gpio::driver::RegisterDriver(nullptr), "");

    nn::gpio::driver::RegisterDriver(&driver0);
    nn::gpio::driver::RegisterDriver(&driver1);
    nn::gpio::driver::RegisterDriver(&driver2);
    NN_UTIL_SCOPE_EXIT
    {
        // Finalize 後なら成功する
        nn::gpio::driver::UnregisterDriver(&driver0);
        nn::gpio::driver::UnregisterDriver(&driver1);
        nn::gpio::driver::UnregisterDriver(&driver2);
    };

    nn::gpio::driver::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        nn::gpio::driver::Finalize();
    };

    // サブシステム初期化後のドライバ登録・解除は不可
    StubDriver anotherDriver(PadInfoSpan[3]);
    EXPECT_DEATH_IF_SUPPORTED(nn::gpio::driver::RegisterDriver(&anotherDriver), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::gpio::driver::UnregisterDriver(&driver0), "");

    // 簡単なセッション導通のテスト
    // 仮想ドライバとデバイスに通しているので、操作結果を直接比較してのテストが可能
    for ( auto&& padList : nn::util::MakeSpan(PadInfoSpan, 3) )
    {
        for ( auto&& pad : padList )
        {
            nn::gpio::driver::GpioPadSession session;
            NNT_ASSERT_RESULT_SUCCESS(nn::gpio::driver::OpenSession(&session, pad.GetDeviceCode()));
            NN_UTIL_SCOPE_EXIT
            {
                nn::gpio::driver::CloseSession(&session);
            };

            nn::gpio::driver::SetValue(&session, nn::gpio::GpioValue_High);
            EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::driver::GetValue(&session));
            EXPECT_EQ(nn::gpio::GpioValue_High, pad.GetValue());
            nn::gpio::driver::SetValue(&session, nn::gpio::GpioValue_Low);

            nn::gpio::driver::SetDirection(&session, nn::gpio::Direction_Output);
            EXPECT_EQ(nn::gpio::Direction_Output, nn::gpio::driver::GetDirection(&session));
            EXPECT_EQ(nn::gpio::Direction_Output, pad.GetDirection());
            nn::gpio::driver::SetDirection(&session, nn::gpio::Direction_Input);
        }
    }
}

//! StubPad のエッジトリガの挙動シミュレータを使用した簡単なインターフェーステスト
TEST(testGpio_SubsystemUnitTest, InterruptEdge)
{
    auto& padInfoSpan = PadInfoSpan[0];
    StubDriver driver(padInfoSpan);
    nn::gpio::driver::RegisterDriver(&driver);
    NN_UTIL_SCOPE_EXIT
    {
        nn::gpio::driver::UnregisterDriver(&driver);
    };

    nn::gpio::driver::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        nn::gpio::driver::Finalize();
    };

    StubInterruptHandler interruptHandler(&driver);
    nn::gpio::driver::RegisterInterruptHandler(&interruptHandler);
    NN_UTIL_SCOPE_EXIT
    {
        nn::gpio::driver::UnregisterInterruptHandler(&interruptHandler);
    };

    auto& pad = padInfoSpan[0]; // テスト対象ピン
    pad.SetInterruptEvent(interruptHandler.GetInterruptEvent());

    nn::os::SystemEvent event; // session より長寿命にしないとセッション破棄時の自動 Unbind で二重 Destroy するリスクあり
    nn::gpio::driver::GpioPadSession session;
    NNT_ASSERT_RESULT_SUCCESS(nn::gpio::driver::OpenSession(&session, pad.GetDeviceCode()));
    NN_UTIL_SCOPE_EXIT
    {
        nn::gpio::driver::CloseSession(&session);
    };

    nn::gpio::driver::SetDirection(&session, nn::gpio::Direction_Input);
    nn::gpio::driver::SetInterruptEnable(&session, false);
    nn::gpio::driver::ClearInterruptStatus(&session);

    // 以下テスト本体 ------------------------

    nn::gpio::driver::SetInterruptMode(&session, nn::gpio::InterruptMode_RisingEdge);

    nn::gpio::driver::BindInterrupt(event.GetBase(), &session); // 中で CreateSystemEvent される
    NN_UTIL_SCOPE_EXIT
    {
        nn::gpio::driver::UnbindInterrupt(&session);
    };

    nn::gpio::driver::SetInterruptEnable(&session, true);

    // ピン状態を変更（中で割り込みハンドラがトリガーされる）
    pad.SetValue(nn::gpio::GpioValue_High);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50)); // HandleEvent が走るの待ち

    // シグナル状態になったことをテスト
    EXPECT_TRUE(event.TryWait());
    EXPECT_FALSE(interruptHandler.GetInterruptEvent()->TryWait()); // これは HandleEvent 側ですぐに落とされる
    EXPECT_EQ(nn::gpio::InterruptStatus_Inactive, nn::gpio::driver::GetInterruptStatus(&session)); // これも HandleEvent 側ですぐに落とされる
    nn::gpio::driver::ClearInterruptStatus(&session);
    event.Clear();

    // この状態では再度ピン状態が変更されても割り込みは発生しないことをテスト
    pad.SetValue(nn::gpio::GpioValue_Low);
    pad.SetValue(nn::gpio::GpioValue_High);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
    EXPECT_FALSE(event.TryWait());
    EXPECT_FALSE(interruptHandler.GetInterruptEvent()->TryWait());
    EXPECT_EQ(nn::gpio::InterruptStatus_Inactive, nn::gpio::driver::GetInterruptStatus(&session));

    // エッジトリガなので、SetInterruptEnable で再有効にするだけではまだシグナルは来ないことをテスト
    nn::gpio::driver::SetInterruptEnable(&session, true);
    EXPECT_FALSE(event.TryWait());
    EXPECT_FALSE(interruptHandler.GetInterruptEvent()->TryWait());
    EXPECT_EQ(nn::gpio::InterruptStatus_Inactive, nn::gpio::driver::GetInterruptStatus(&session));

    // この状態でピン状態を変更して初めて次の割り込みが発生することをテスト
    pad.SetValue(nn::gpio::GpioValue_Low);
    pad.SetValue(nn::gpio::GpioValue_High);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50)); // HandleEvent が走るの待ち
    EXPECT_TRUE(event.TryWait());
    EXPECT_FALSE(interruptHandler.GetInterruptEvent()->TryWait()); // これは HandleEvent 側ですぐに落とされる
    EXPECT_EQ(nn::gpio::InterruptStatus_Inactive, nn::gpio::driver::GetInterruptStatus(&session)); // これも HandleEvent 側ですぐに落とされる
    nn::gpio::driver::ClearInterruptStatus(&session);
    event.Clear();
}

//! StubPad のレベルトリガの挙動シミュレータを使用した簡単なインターフェーステスト
TEST(testGpio_SubsystemUnitTest, InterruptLevel)
{
    auto& padInfoSpan = PadInfoSpan[0];
    StubDriver driver(padInfoSpan);
    nn::gpio::driver::RegisterDriver(&driver);
    NN_UTIL_SCOPE_EXIT
    {
        nn::gpio::driver::UnregisterDriver(&driver);
    };

    nn::gpio::driver::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        nn::gpio::driver::Finalize();
    };

    StubInterruptHandler interruptHandler(&driver);
    nn::gpio::driver::RegisterInterruptHandler(&interruptHandler);
    NN_UTIL_SCOPE_EXIT
    {
        nn::gpio::driver::UnregisterInterruptHandler(&interruptHandler);
    };

    auto& pad = padInfoSpan[0]; // テスト対象ピン
    pad.SetInterruptEvent(interruptHandler.GetInterruptEvent());

    nn::os::SystemEvent event; // session より長寿命にしないとセッション破棄時の自動 Unbind で二重 Destroy するリスクあり
    nn::gpio::driver::GpioPadSession session;
    NNT_ASSERT_RESULT_SUCCESS(nn::gpio::driver::OpenSession(&session, pad.GetDeviceCode()));
    NN_UTIL_SCOPE_EXIT
    {
        nn::gpio::driver::CloseSession(&session);
    };

    nn::gpio::driver::SetDirection(&session, nn::gpio::Direction_Input);
    nn::gpio::driver::SetInterruptEnable(&session, false);
    nn::gpio::driver::ClearInterruptStatus(&session);

    // 以下テスト本体 ------------------------

    nn::gpio::driver::SetInterruptMode(&session, nn::gpio::InterruptMode_HighLevel);

    nn::gpio::driver::BindInterrupt(event.GetBase(), &session); // 中で CreateSystemEvent される
    NN_UTIL_SCOPE_EXIT
    {
        nn::gpio::driver::UnbindInterrupt(&session);
    };

    nn::gpio::driver::SetInterruptEnable(&session, true);

    // ピン状態を変更（中で割り込みハンドラがトリガーされる）
    pad.SetValue(nn::gpio::GpioValue_High);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50)); // HandleEvent が走るの待ち

    // シグナル状態になったことをテスト
    EXPECT_TRUE(event.TryWait());
    EXPECT_FALSE(interruptHandler.GetInterruptEvent()->TryWait()); // これはレベルトリガでも HandleEvent 側ですぐに落とされる
    EXPECT_EQ(nn::gpio::InterruptStatus_Inactive, nn::gpio::driver::GetInterruptStatus(&session)); // これも HandleEvent 側ですぐに落とされる
    nn::gpio::driver::ClearInterruptStatus(&session);
    event.Clear();

    nn::gpio::driver::UnbindInterrupt(&session);

    // アンバインド状態では割り込み有効にしても実際にはアクティブにならない
    nn::gpio::driver::SetInterruptEnable(&session, true);
    // EXPECT_FALSE(event.TryWait()); // Unbind 内で DestroySystemEvent されている
    EXPECT_FALSE(interruptHandler.GetInterruptEvent()->TryWait());
    EXPECT_EQ(nn::gpio::InterruptStatus_Inactive, nn::gpio::driver::GetInterruptStatus(&session));

    // 割り込み有効状態からバインドした瞬間に割り込みがアクティブになる
    nn::gpio::driver::BindInterrupt(event.GetBase(), &session);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50)); // HandleEvent が走るの待ち

    EXPECT_TRUE(event.TryWait());
    EXPECT_FALSE(interruptHandler.GetInterruptEvent()->TryWait()); // これはレベルトリガでも HandleEvent 側ですぐに落とされる
    EXPECT_EQ(nn::gpio::InterruptStatus_Inactive, nn::gpio::driver::GetInterruptStatus(&session)); // これも HandleEvent 側ですぐに落とされる
    nn::gpio::driver::ClearInterruptStatus(&session);
    event.Clear();

    // ピン状態を変更して割り込み要因を取り除く
    pad.SetValue(nn::gpio::GpioValue_Low);

    // 要因を取り除いたので、割り込みを有効にしても今度はシグナルは来ない
    nn::gpio::driver::SetInterruptEnable(&session, true);
    EXPECT_FALSE(event.TryWait());
    EXPECT_FALSE(interruptHandler.GetInterruptEvent()->TryWait());
    EXPECT_EQ(nn::gpio::InterruptStatus_Inactive, nn::gpio::driver::GetInterruptStatus(&session));
}

//! 割り込み制御関連 API の内部の排他制御のテスト。割り込み要因を発生させるスレッド / 割り込みハンドラスレッド / クライアントスレッドから関連 API を一定時間叩く
TEST(testGpio_SubsystemUnitTest, ConcurrentInterruptHandling)
{
    auto& padInfoSpan = PadInfoSpan[0];
    StubDriver driver(padInfoSpan);
    nn::gpio::driver::RegisterDriver(&driver);
    NN_UTIL_SCOPE_EXIT
    {
        nn::gpio::driver::UnregisterDriver(&driver);
    };

    nn::gpio::driver::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        nn::gpio::driver::Finalize();
    };

    StubInterruptHandler interruptHandler(&driver);
    nn::gpio::driver::RegisterInterruptHandler(&interruptHandler);
    NN_UTIL_SCOPE_EXIT
    {
        nn::gpio::driver::UnregisterInterruptHandler(&interruptHandler);
    };

    auto& pad = padInfoSpan[0]; // テスト対象ピン
    pad.SetInterruptEvent(interruptHandler.GetInterruptEvent());

    nn::os::SystemEvent event; // session より長寿命にしないとセッション破棄時の自動 Unbind で二重 Destroy するリスクあり
    nn::gpio::driver::GpioPadSession session;
    NNT_ASSERT_RESULT_SUCCESS(nn::gpio::driver::OpenSession(&session, pad.GetDeviceCode()));
    NN_UTIL_SCOPE_EXIT
    {
        nn::gpio::driver::CloseSession(&session);
    };

    nn::gpio::driver::SetDirection(&session, nn::gpio::Direction_Input);
    nn::gpio::driver::SetInterruptEnable(&session, false);
    nn::gpio::driver::ClearInterruptStatus(&session);

    // 以下テスト本体 ------------------------

    nn::gpio::driver::SetInterruptMode(&session, nn::gpio::InterruptMode_AnyEdge);

    nn::gpio::driver::BindInterrupt(event.GetBase(), &session); // 中で CreateSystemEvent される
    NN_UTIL_SCOPE_EXIT
    {
        nn::gpio::driver::UnbindInterrupt(&session);
    };

    nn::gpio::driver::SetInterruptEnable(&session, true);

    // ピン状態を変更して割り込みを一定時間発生させ続けるスレッドを作成
    g_LoopCount = 500;
    static NN_ALIGNAS(nn::os::StackRegionAlignment) char s_PinControlThreadStack[nn::os::StackRegionAlignment];
    nn::os::ThreadType pinControlThread;
    nn::os::CreateThread(&pinControlThread,
        [](void* arg)
        {
            auto& pad = *reinterpret_cast<StubPad*>(arg);
            while ( g_LoopCount > 0 ) // TORIAEZU: ラムダ式にキャプチャを使うと同等の関数ポインタが得られなくなるため、loop count はグローバル変数として参照
            {
                --g_LoopCount;
                pad.SetValue(nn::gpio::GpioValue_Low);
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
                pad.SetValue(nn::gpio::GpioValue_High);
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
                pad.SetValue(nn::gpio::GpioValue_Low);
                pad.SetValue(nn::gpio::GpioValue_High);
            }
        }, &pad, s_PinControlThreadStack, sizeof(s_PinControlThreadStack), nn::os::DefaultThreadPriority);
    nn::os::StartThread(&pinControlThread);
    NN_UTIL_SCOPE_EXIT
    {
        nn::os::WaitThread(&pinControlThread);
        nn::os::DestroyThread(&pinControlThread);
    };

    // スレッドが終わるまで
    while ( event.TimedWait(nn::TimeSpan::FromMilliSeconds(200)) )
    {
        nn::gpio::driver::ClearInterruptStatus(&session);
        event.Clear();
        switch ( std::rand() % 4 )
        {
            case 0:
                nn::gpio::driver::SetInterruptEnable(&session, true);
                break;
            case 1:
                nn::gpio::driver::UnbindInterrupt(&session);
                nn::gpio::driver::BindInterrupt(event.GetBase(), &session);
                nn::gpio::driver::SetInterruptEnable(&session, true);
                break;
            case 2:
                nn::gpio::driver::UnbindInterrupt(&session);
                nn::gpio::driver::SetInterruptEnable(&session, true);
                nn::gpio::driver::BindInterrupt(event.GetBase(), &session);
                break;
            case 3:
                nn::gpio::driver::UnbindInterrupt(&session);
                nn::gpio::driver::CloseSession(&session);
                nn::gpio::driver::OpenSession(&session, pad.GetDeviceCode());
                nn::gpio::driver::BindInterrupt(event.GetBase(), &session);
                nn::gpio::driver::SetInterruptEnable(&session, true);
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
        }
    }
    EXPECT_EQ(0, g_LoopCount);
}

