﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <memory>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/fs.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_IEventNotifier.h>
#include <nn/fs/fs_IStorage.h>

#include <nn/os.h>

#include <nn/sf/sf_ObjectFactory.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_Content.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_ResultHandler.h>

#include <nn/fs/fs_ContentStorage.h>
#include <nn/fs/fs_MountPrivate.h>
#include <nn/fs/fs_Directory.h>
#include <nn/ncm/ncm_ContentIdUtil.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ncm/ncm_ContentStorageImpl.h>
#include <nn/util/util_StringView.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_InitializationApi.h>

#define NN_DETAIL_GC_RESULT_DO_LOG(r, ...)      \
    { \
        const auto& _nn_result_do_temporary((r)); \
        if(_nn_result_do_temporary.IsFailure()) \
        { \
            NN_LOG(__VA_ARGS__); \
            NN_LOG("Failure (Module:%d, Description:%d)...\n", _nn_result_do_temporary.GetModule(), _nn_result_do_temporary.GetDescription()); \
            NN_RESULT_THROW(r); \
        } \
    }

#define NN_DETAIL_GC_RESULT_LOG(r) \
    { \
        const auto& _nn_result_do_temporary((r)); \
        if(_nn_result_do_temporary.IsFailure()) \
        { \
            NN_LOG("Failure (Module:%d, Description:%d)...\n", _nn_result_do_temporary.GetModule(), _nn_result_do_temporary.GetDescription()); \
            NN_RESULT_THROW(_nn_result_do_temporary); \
        } \
    }

using namespace nn;
using namespace nn::fs;

namespace {

int g_InsertRemoveMaxWaitTime = 30;

#if 0
// バッファのダンプ
void DumpBuffer(const void* buffer, size_t size)
{
    for(size_t i = 0; i < size; ++i)
    {
        if(i % 4   == 0)
        {
            NN_LOG(" ");
        }
        if(i % 32  == 0)
        {
            NN_LOG("\n");
        }

        uint8_t value8 = static_cast<const uint8_t*>(buffer)[i];
        NN_LOG("%02x", value8);
    }

    NN_LOG("\n\n");
}
#endif

struct Path
{
    char string[768];
};

class Time
{
public:
    Time() : m_StartTime(0), m_EndTime(0) {}
    ~Time() {}

    void Start()
    {
        m_StartTime = nn::os::GetSystemTick();
    }

    int64_t GetElapsedTimeMsec()
    {
        m_EndTime = nn::os::GetSystemTick();
        return (m_EndTime - m_StartTime).ToTimeSpan().GetMilliSeconds();
    }

    int64_t GetElapsedTimeSec()
    {
        return GetElapsedTimeMsec() / 1000;
    }

private:
    nn::os::Tick m_StartTime;
    nn::os::Tick m_EndTime;
};

Result SearchContentMetaFromGameCard(const char* mountName, std::string& path) NN_NOEXCEPT
{
    auto rootDirectoryPath = std::string(mountName) + ":/";
    const char* contentMetaExtension = ".cnmt.nca";
    fs::DirectoryHandle directory;
    NN_DETAIL_GC_RESULT_DO_LOG(fs::OpenDirectory(&directory, rootDirectoryPath.c_str(), fs::OpenDirectoryMode_File), "fs::OpenDirectory in ContentMeta Failure\n");
    NN_UTIL_SCOPE_EXIT
    {
        fs::CloseDirectory(directory);
    };

    while (NN_STATIC_CONDITION(true))
    {
        fs::DirectoryEntry entry;
        int64_t outRead;
        NN_DETAIL_GC_RESULT_DO_LOG(fs::ReadDirectory(&outRead, &entry, directory, 1), "fs::ReadDirectory in ContentMeta Failure\n");

        if (nn::util::string_view(entry.name).substr(strlen(entry.name) - strlen(contentMetaExtension))
            == nn::util::string_view(contentMetaExtension))
        {
            path = ::std::string(rootDirectoryPath) + entry.name;
        }

        if (outRead == 0)
        {
            break;
        }

        // NN_LOG("Directory Name : %s\n", entry.name);
        // NN_LOG("Directory Size : %lld\n", entry.fileSize);
    }

    NN_RESULT_SUCCESS;
}

nn::Result ListupFile(nn::fs::GameCardPartition partition)
{
    NN_LOG("Listup File Start\n");

    GameCardHandle handle;

    // ゲームカードのハンドルを取得
    NN_DETAIL_GC_RESULT_DO_LOG(GetGameCardHandle(&handle), "fs::GetGameCardHandle Failure\n");

    // normal area をマウント
    NN_DETAIL_GC_RESULT_DO_LOG(nn::fs::MountGameCardPartition("@GcApp", handle, partition), "fs::MountGameCardPartition Failure\n");
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount("@GcApp");
    };

    ::std::string contentMetaPath;
    //コンテンツメタファイルを探す 拡張子 .cnmt.nca
    if(SearchContentMetaFromGameCard("@GcApp", contentMetaPath).IsFailure())
    {
        NN_LOG("Find content meta Failure!\n");
    }
    NN_LOG("Content meta path: %s\n", contentMetaPath.c_str());

    Path commonPath;
    nn::fs::ConvertToFsCommonPath(commonPath.string, sizeof(commonPath.string), contentMetaPath.c_str());

    const char* mountMetaName = "@GcMeta";
    NN_DETAIL_GC_RESULT_DO_LOG(nn::fs::MountContent(mountMetaName, commonPath.string, fs::ContentType::ContentType_Meta), "fs::MountContent Failure\n");
    NN_UTIL_SCOPE_EXIT
    {
        fs::Unmount(mountMetaName);
    };

    auto rootDirectoryPath = std::string(mountMetaName) + ":/";

    fs::DirectoryHandle directory;
    NN_DETAIL_GC_RESULT_DO_LOG(fs::OpenDirectory(&directory, rootDirectoryPath.c_str(), fs::OpenDirectoryMode_File), "fs:OpenDirectory Failure\n");
    NN_UTIL_SCOPE_EXIT
    {
        fs::CloseDirectory(directory);
    };

    while (NN_STATIC_CONDITION(true))
    {
        fs::DirectoryEntry entry;
        int64_t outRead;
        NN_DETAIL_GC_RESULT_DO_LOG(fs::ReadDirectory(&outRead, &entry, directory, 1), "fs::ReadDirectory Failure\n");
        if (outRead == 0)
        {
            break;
        }

        // NN_LOG("Directory Name : %s\n", entry.name);
        // NN_LOG("Directory Size : %lld\n", entry.fileSize);
    }
    NN_LOG("Listup File End\n");

    NN_RESULT_SUCCESS;
}

nn::Result ReadNormalArea()
{
    return ListupFile(nn::fs::GameCardPartition::Normal);
}

#ifdef ENABLE_READ_CUP_AREA
nn::Result ReadCupArea()
{
    return ListupFile(nn::fs::GameCardPartition::Update);
}
#endif

nn::Result ReadSecureArea()
{
    return ListupFile(nn::fs::GameCardPartition::Secure);
}


// ゲームカードが挿入されていなければ挿入を待つ
void WaitAttach()
{
    if( !IsGameCardInserted() )
    {
        Result result;

        NN_LOG("Waiting attach of game card.\n");

        // 挿抜イベント通知をオープン
        std::unique_ptr<nn::fs::IEventNotifier> deviceDetectionEventNotifier;
        result = OpenGameCardDetectionEventNotifier(&deviceDetectionEventNotifier);
        if( result.IsFailure() )
        {
            NN_ABORT("failed to open notifier. (%d:%d)\n", result.GetModule(), result.GetDescription());
        }

        // システムイベントと紐づけ
        nn::os::SystemEventType deviceDetectionEvent;
        result = deviceDetectionEventNotifier->BindEvent(&deviceDetectionEvent, nn::os::EventClearMode_ManualClear);
        if( result.IsFailure() )
        {
            NN_ABORT("failed to bind system event to notifier. (%d:%d)\n", result.GetModule(), result.GetDescription());
        }
        // 挿入を待つ
        Time waitTimer;
        waitTimer.Start();
        while( !IsGameCardInserted() )
        {
            // イベント通知を待つ
            bool isEventOccuured = nn::os::TimedWaitSystemEvent(&deviceDetectionEvent, nn::TimeSpan::FromMilliSeconds(10));
            if (isEventOccuured)
            {
                nn::os::ClearSystemEvent(&deviceDetectionEvent);
                waitTimer.Start();
            }
            if (waitTimer.GetElapsedTimeSec() > g_InsertRemoveMaxWaitTime)
            {
                NN_ABORT("wait attach timeout\n");
            }
        }

        // イベントを破棄
        nn::os::DestroySystemEvent(&deviceDetectionEvent);
    }
}

// 抜去のイベント通知を 1 回待つ
void WaitDetach()
{
    if( IsGameCardInserted() )
    {
        NN_LOG("Waiting detach of game card.\n");

        Result result;

        // 挿抜イベント通知をオープン
        std::unique_ptr<nn::fs::IEventNotifier> deviceDetectionEventNotifier;
        result = OpenGameCardDetectionEventNotifier(&deviceDetectionEventNotifier);
        if( result.IsFailure() )
        {
            NN_ABORT("failed to open notifier. (%d:%d)\n", result.GetModule(), result.GetDescription());
        }

        // システムイベントと紐づけ
        nn::os::SystemEventType deviceDetectionEvent;
        result = deviceDetectionEventNotifier->BindEvent(&deviceDetectionEvent, nn::os::EventClearMode_ManualClear);
        if( result.IsFailure() )
        {
            NN_ABORT("failed to bind system event to notifier. (%d:%d)\n", result.GetModule(), result.GetDescription());
        }

        // イベント通知を待つ
        Time waitTimer;
        waitTimer.Start();
        while( IsGameCardInserted() )
        {
            // イベント通知を待つ
            bool isEventOccuured = nn::os::TimedWaitSystemEvent(&deviceDetectionEvent, nn::TimeSpan::FromMilliSeconds(10));
            if (isEventOccuured)
            {
                nn::os::ClearSystemEvent(&deviceDetectionEvent);
                waitTimer.Start();
            }
            if (waitTimer.GetElapsedTimeSec() > g_InsertRemoveMaxWaitTime)
            {
                NN_ABORT("wait attach timeout\n");
            }
        }

        // イベントを破棄
        nn::os::DestroySystemEvent(&deviceDetectionEvent);
    }
}

}

void CheckResult(nn::Result result)
{
    if (! nn::fs::ResultGameCardAccessFailed::Includes(result))
    {
        if(nn::fs::ResultHandledByAllProcess::Includes(result))
        {
            NN_LOG(" - nn::fs::ResultHandledByAllProcess\n");
        }
        else if(nn::fs::ResultHandledBySystemProcess::Includes(result))
        {
            NN_LOG(" - nn::fs::ResultHandledBySystemProcess\n");
        }
        else if(nn::fs::ResultInternal::Includes(result))
        {
            // 出てほしくはないが、SIGLO-34435, SPRD-3195 の検討結果として出る想定
            NN_LOG("Warning: nn::fs::ResultInternal\n");
        }
        else
        {
            NN_LOG("Error: not fs result\n");
        }

        // GameCard 以外のエラー Result は期待しない
        NN_ABORT("Error: detected not gc failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
    }
}


#define NN_DETAIL_GC_RESULT_CONTINUE(r) \
    { \
        const auto& _nn_result_do_temporary((r)); \
        if(_nn_result_do_temporary.IsFailure()) \
        { \
            NN_LOG("Detected Result Failure (Module:%d, Description:%d)\n", _nn_result_do_temporary.GetModule(), _nn_result_do_temporary.GetDescription()); \
            CheckResult(_nn_result_do_temporary); \
            continue; \
        } \
    }

extern "C" void nnMain()
{
    nn::fs::SetEnabledAutoAbort(false);
    nn::ns::Initialize();

    nn::ns::GameCardStopper stopper;
    nn::ns::GetGameCardStopper(&stopper);

    // 挿抜の試行回数
    int maxTryNum = 3;

    // 引数取得
    {
        int argc = nn::os::GetHostArgc();
        char** argv = nn::os::GetHostArgv();
        if(argc == 1)
        {
            NN_LOG("usage: CardInsertionEndurance <InsertRemoveCount> <TimeOutSec>\n");
        }
        if(argc > 1)
        {
            maxTryNum = atoi(argv[1]);
        }
        if(argc > 2)
        {
            g_InsertRemoveMaxWaitTime = atoi(argv[2]);
        }
    }
    NN_LOG("## Game card insert/remove test for [%d] times\n", maxTryNum);
    NN_LOG("timeout: %d sec\n", g_InsertRemoveMaxWaitTime);

    for (int i = 0; i < maxTryNum; i++)
    {
        WaitDetach();

        NN_LOG("Test loop: %d\n", i);

        WaitAttach();

        NN_DETAIL_GC_RESULT_CONTINUE(ReadNormalArea());
#ifdef ENABLE_READ_CUP_AREA
        NN_DETAIL_GC_RESULT_CONTINUE(ReadCupArea());
#endif
        NN_DETAIL_GC_RESULT_CONTINUE(ReadSecureArea());
    }

    NN_LOG("Test Finished\n");
}
