﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstring>
#include <cstdlib>
#include <cstdio>

#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>

#include <nn/gc/gc.h>
#include <nn/dd.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_GameCardForInspection.h>

#include <nn/spl/spl_Api.h>

#include "GcCommon.h"

const size_t WorkBufferSize          = nn::util::align_up(nn::gc::GcWorkBufferSize, nn::os::MemoryBlockUnitSize);
const size_t MemoryHeapSize          = 32 * 1024 * 1024;
const size_t DataBufferSize          = MemoryHeapSize / 2;
const size_t MemoryHeapSizeForMalloc = MemoryHeapSize - WorkBufferSize - DataBufferSize;

char*     g_pWorkBuffer;
uintptr_t g_WorkBufferAddress;
char*     g_pDataBuffer;
uintptr_t g_DataBufferAddress;
uintptr_t g_MallocBufferAddress;

nn::dd::DeviceAddressSpaceType g_Das;
nn::dd::DeviceVirtualAddress g_WorkBufferDeviceVirtualAddress;
nn::dd::DeviceVirtualAddress g_DataBufferDeviceVirtualAddress;

const size_t ThreadNum = 2;
nn::os::ThreadType g_Thread[ThreadNum];
static const int ThreadPriority = 9;
static const size_t ThreadStackSize = 64 * 1024;
char g_ThreadName[] = "GcTest";
NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[ThreadNum][ ThreadStackSize ];

// イベント
nn::os::EventType g_ThreadEndEvent;
nn::os::EventType g_MeasureStartEvent[ThreadNum];
nn::os::EventType g_MeasureEndEvent[ThreadNum];

const size_t ReadStartPageAddress = 0x80000;
size_t g_ChunkPageSize = 0;//3MB
int64_t g_Time = 0;
int64_t g_MaxTime = 0;

class Time
{
public:
    Time() : m_StartTime(0), m_EndTime(0) {}
    ~Time() {}

    void Start()
    {
        m_StartTime = nn::os::GetSystemTick();
    }

    int64_t GetElapsedTime()
    {
        m_EndTime = nn::os::GetSystemTick();
        return (m_EndTime - m_StartTime).ToTimeSpan().GetMicroSeconds();
    }

private:
    nn::os::Tick m_StartTime;
    nn::os::Tick m_EndTime;
};

void PrintArray(const char* buffer, const size_t bufferLength) NN_NOEXCEPT
{
    for(size_t i = 0; i < bufferLength; i++)
    {
        if(i != 0 && i % 16 == 0)
        {
            NN_LOG("\n");
        }

        NN_LOG("%02X ", buffer[i]);
    }
    NN_LOG("\n");
}

extern "C" void nninitStartup()
{
    nn::Result result = nn::os::SetMemoryHeapSize(MemoryHeapSize);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot set MemoryHeapSize.");

    // メモリヒープから WorkBuffer で使用するメモリ領域を確保
    result = nn::os::AllocateMemoryBlock(&g_WorkBufferAddress, WorkBufferSize);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot AllocateMemoryBlock0.");

    // メモリヒープから DataBuffer で使用するメモリ領域を確保
    result = nn::os::AllocateMemoryBlock(&g_DataBufferAddress, DataBufferSize);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot AllocateMemoryBlock1.");

    // メモリヒープから malloc で使用するメモリ領域を確保
    result = nn::os::AllocateMemoryBlock(&g_MallocBufferAddress, MemoryHeapSizeForMalloc);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot AllocateMemoryBlock2.");

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator(reinterpret_cast<void*>(g_MallocBufferAddress), MemoryHeapSizeForMalloc);
}

void InitializeDriver()
{
    nn::spl::InitializeForFs();

    NN_LOG("Read From Partition\n");

    ReadGcCalibrationPartitionToSetInternalKeys();

    NN_LOG("Finalize gc driver from fs\n");

    nn::fs::FinalizeGameCardDriver();

    // クロック制御は pcv 経由で行う
    nn::sdmmc::SwitchToPcvClockResetControl();

    nn::os::ChangeThreadPriority(nn::os::GetCurrentThread(), 10);

    NN_LOG("Initialize gc driver.\n");

    g_pWorkBuffer = reinterpret_cast<char*>(g_WorkBufferAddress);
    g_pDataBuffer = reinterpret_cast<char*>(g_DataBufferAddress);

    SetupDeviceAddressSpace(&g_Das, nn::dd::DeviceName::DeviceName_Sdmmc2a);

    g_WorkBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&g_Das, g_WorkBufferAddress, WorkBufferSize, 0);
    nn::gc::Initialize(g_pWorkBuffer, WorkBufferSize, g_WorkBufferDeviceVirtualAddress);

    g_DataBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&g_Das, g_DataBufferAddress, DataBufferSize, g_WorkBufferDeviceVirtualAddress + WorkBufferSize);
    nn::gc::RegisterDeviceVirtualAddress(g_DataBufferAddress, DataBufferSize, g_DataBufferDeviceVirtualAddress);
}

void FinalizeDriver()
{
    nn::gc::UnregisterDeviceVirtualAddress(g_DataBufferAddress, DataBufferSize, g_DataBufferDeviceVirtualAddress);
    UnmapDeviceAddressSpaceAligned(&g_Das, g_DataBufferAddress, DataBufferSize, g_DataBufferDeviceVirtualAddress);

    nn::gc::Finalize();
    UnmapDeviceAddressSpaceAligned(&g_Das, g_WorkBufferAddress, WorkBufferSize, g_WorkBufferDeviceVirtualAddress);

    CleanDeviceAddressSpace(&g_Das, nn::dd::DeviceName::DeviceName_Sdmmc2a);

    nn::spl::Finalize();
}

const size_t ReadMaxNum = 64;
void RunThreadFunction(void* pThreadNum) NN_NOEXCEPT
{
    size_t threadNumber = *reinterpret_cast<size_t*>(pThreadNum);
    uint32_t startAddr = ReadStartPageAddress;
    nn::os::WaitEvent(&g_MeasureStartEvent[threadNumber]);
    Time timer;
    for(size_t i = 0; i < ReadMaxNum; i++ )
    {
        timer.Start();
        nn::gc::Read(g_pDataBuffer, DataBufferSize, startAddr * (threadNumber + 1), g_ChunkPageSize);
        auto time = timer.GetElapsedTime();
        g_Time += time;
        if(time > g_MaxTime)
        {
            g_MaxTime = time;
        }
        startAddr += g_ChunkPageSize;
    }
    nn::os::SignalEvent(&g_MeasureEndEvent[threadNumber]);
}

void InitializeThread()
{
    // 待ちイベントの初期設定
    static char threadName[ThreadNum][20];

    for(size_t i = 0; i < ThreadNum; i++)
    {
        nn::os::InitializeEvent(&g_MeasureStartEvent[i], false, nn::os::EventClearMode_AutoClear);
        nn::os::InitializeEvent(&g_MeasureEndEvent[i], false, nn::os::EventClearMode_AutoClear);
//        nn::Result result = nn::os::CreateThread( &g_Thread[i], (i == 0) ? RunThreadFunction: RunThreadFunction2, &g_Thread[i], g_ThreadStack[i], ThreadStackSize, ThreadPriority );
        nn::Result result = nn::os::CreateThread(&g_Thread[i], RunThreadFunction, &i, g_ThreadStack[i], ThreadStackSize, ThreadPriority);
        if(!result.IsSuccess())
        {
            NN_ABORT("failed to create a thread\n");
        }
        char threadNumber[4];
        strcpy(threadName[i], g_ThreadName);
        sprintf(threadNumber, "%zu", i);
        strcat(threadName[i], threadNumber);
        nn::os::SetThreadNamePointer(&g_Thread[i], threadName[i]);
        nn::os::StartThread(&g_Thread[i]);
    }
}

void FinalizeThread()
{
    for(size_t i = 0; i < ThreadNum; i++)
    {
        // スレッドの終了
        nn::os::WaitThread(&g_Thread[i]);
        nn::os::DestroyThread(&g_Thread[i]);
        nn::os::FinalizeEvent(&g_MeasureStartEvent[i]);
        nn::os::FinalizeEvent(&g_MeasureEndEvent[i]);
    }
}

void ExecuteMeasurement(size_t pageSize)
{
    InitializeThread();
    g_ChunkPageSize = pageSize;
    g_Time = 0;
    g_MaxTime = 0;
    Time timer;
    timer.Start();
    // スレッド開始シグナル
    for(size_t i = 0; i < ThreadNum; i++)
    {
        nn::os::SignalEvent(&g_MeasureStartEvent[i]);
    }
    // 全スレッド終了待ち
    for(size_t i = 0; i < ThreadNum; i++)
    {
        nn::os::WaitEvent(&g_MeasureEndEvent[i]);
    }
    auto time = timer.GetElapsedTime();
//    NN_LOG("TIME  = %lld us\n",time);
    // スレッド数 * チャンク(ページ)サイズ * 読んだ回数 * 1ページのバイト数 * 1000(ms) * 1000(us) / 計測にかかった時間(us) * 1024(KB) * 1024(MB)
    float speed  = ThreadNum * g_ChunkPageSize * ReadMaxNum * 512 * 1000.0 * 1000.0 / (time * 1024 * 1024);
    NN_LOG("%05d Page : %06.3f MB/s (ave %07lld us, max %07lld us)\n", g_ChunkPageSize, speed, time / (ThreadNum * ReadMaxNum), g_MaxTime);
    FinalizeThread();
}

extern "C" void nnMain()
{
    NN_LOG("Hello, I'm gc writer program.\n");

    InitializeDriver();

    NN_LOG("\n\nFinished. Enable card bus.\n");

    nn::Result result = nn::gc::Activate();
    if(result.IsFailure())
    {
        NN_LOG("Test Failure\n");
        return;
    }

    NN_LOG("\n\nFinished. SetCardToSecureMode\n");

    result = nn::gc::SetCardToSecureMode();
    if(result.IsFailure())
    {
        NN_LOG("Test Failure");
        return;
    }

    size_t i = 1;
    const size_t MaxChunkPageSize = 0x4000;//8MB
    while(i <= MaxChunkPageSize)
    {
        ExecuteMeasurement(i);
        i *= 2;
    }

    nn::os::FreeMemoryBlock(g_MallocBufferAddress, MemoryHeapSizeForMalloc);
    nn::os::FreeMemoryBlock(g_DataBufferAddress, DataBufferSize);
    nn::os::FreeMemoryBlock(g_WorkBufferAddress, WorkBufferSize);
    FinalizeDriver();

    NN_LOG("Test Success\n");

}

