﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>

#include <nn/nn_Log.h>
// #include <nn/nn_Assert.h>
// #include <nn/nn_Common.h>
// #include <nn/os.h>
// #include <nn/dd.h>

// #include <cstdio>
// #include <cstring>
// #include <string>
// #include <iostream>

#include "SdFileAccessor.h"

#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Code.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_Result.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

using namespace ::nn::fs;

#define NN_L3_ENABLE_HOST_IO

SdFileAccessor::SdFileAccessor()
{
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    #if defined(NN_L3_ENABLE_HOST_IO)
        nn::Result result = nn::fs::MountHost("sdcard", "L:");
    #else
        nn::Result result = nn::fs::MountSdCardForDebug("sdcard");
    #endif
    if(result.IsFailure())
    {
        m_IsSdInserted = false;
        NN_LOG("Mount SD Failure\n");
    }
    else
    {
        m_IsSdInserted = true;
        NN_LOG("SD card mounted\n");
    }
}

bool SdFileAccessor::IsEntryExisted(nn::fs::DirectoryEntry* workDirectoryEntryBuffer, const size_t directoryEntryBufferLength, const char* directoryPath, const char* targetName, int openDirectoryMode)
{
    int64_t directoryEntryListLength = 0;
    GetDirectoryList(workDirectoryEntryBuffer, &directoryEntryListLength, directoryEntryBufferLength, directoryPath);
    if ( directoryEntryListLength > 0 )
    {
        for(size_t i=0; i<directoryEntryListLength; i++)
        {
            size_t entryNameLength = strnlen(workDirectoryEntryBuffer[i].name, sizeof(workDirectoryEntryBuffer[i].name));
            size_t targetNameLength = strnlen(targetName, sizeof(workDirectoryEntryBuffer[i].name));
            if ( entryNameLength == targetNameLength && std::memcmp(workDirectoryEntryBuffer[i].name, targetName, targetNameLength) == 0 )
            {
                return true;
            }
        }
    }
    return false;
}

void SdFileAccessor::GetDirectoryOrFileList(nn::fs::DirectoryEntry* outDirectoryEntryBuffer, int64_t* outDirectoryEntryCount, const size_t directoryEntryBufferLength, const char* directoryPath, int openDirectoryMode)
{
    NN_LOG("Open directory %s\n", directoryPath);
    nn::fs::DirectoryHandle directory;
    nn::Result result = nn::fs::OpenDirectory(&directory, directoryPath, openDirectoryMode);
    if(result.IsFailure())
    {
        NN_LOG("Open Directory Failure\n");
    }

    NN_LOG("Read Directory\n");
    result = nn::fs::ReadDirectory(outDirectoryEntryCount, outDirectoryEntryBuffer, directory, directoryEntryBufferLength);
    if(result.IsFailure())
    {
        NN_LOG("Read Directory Failure\n");
    }
    NN_LOG("%d Directories Found\n", *outDirectoryEntryCount);

    nn::fs::CloseDirectory(directory);
}

nn::Result SdFileAccessor::Read(char* outBuffer, size_t* outReadLength, const size_t bufferLength, const char* filePath)
{
    nn::Result result;
    NN_LOG("Open File %s\n", filePath);
    nn::fs::FileHandle file;
    result = OpenFile(&file, filePath, nn::fs::OpenMode::OpenMode_Read);
    if(result.IsFailure())
    {
        NN_LOG("Open File Failure\n");
        return result;
    }
    NN_LOG("Read File\n");
    result = ReadFile(outReadLength, file, 0, outBuffer, bufferLength);
    if(result.IsFailure())
    {
        NN_LOG("Read File Failure\n");
    }
    else
    {
        NN_LOG("Read File Success\n");
    }
    CloseFile(file);
    return result;
}

nn::Result SdFileAccessor::Write(const char* buffer, const size_t bufferLength, const char* filePath)
{
    nn::Result result;
    NN_LOG("Open File %d\n", filePath);
    nn::fs::FileHandle file;
    result = OpenFile(&file, filePath, nn::fs::OpenMode::OpenMode_Write);
    if(result.IsFailure())
    {
        NN_LOG("Open File Failure\n");
        result = nn::fs::CreateFile(filePath, bufferLength);
        if(result.IsFailure())
        {
            NN_LOG("File Create Error\n");
            NN_LOG("failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            return result;
        }
    }
    auto option = nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag::WriteOptionFlag_Flush);
    result = WriteFile(file, 0, buffer, bufferLength, option);
    if(result.IsFailure())
    {
        NN_LOG("Write File Failure\n");
    }
    else
    {
        NN_LOG("Write File Success\n");
    }
    CloseFile(file);
    return result;
}

