﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstring>
#include <cstdlib>

#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>

#include <nn/sdmmc/sdmmc_Common.h>
#include <nn/gc/gc.h>
#include <nn/dd.h>
#include <nn/gc/detail/gc_GcCrypto.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_GameCardForInspection.h>
#include <nn/fs/fs_MmcPrivate.h>

#include <nn/spl/spl_Api.h>

#include "GcCommon.h"

const size_t WorkBufferSize          = nn::util::align_up(nn::gc::GcWorkBufferSize, nn::os::MemoryBlockUnitSize);
const size_t MemoryHeapSize          = 16 * 1024 * 1024;
const size_t DataBufferSize          = MemoryHeapSize / 2;
const size_t MemoryHeapSizeForMalloc = MemoryHeapSize / 4;

char*     g_pWorkBuffer;
uintptr_t g_WorkBufferAddress;
char*     g_pDataBuffer;
uintptr_t g_DataBufferAddress;
uintptr_t g_MallocBufferAddress;
nn::os::EventType g_CardDetectionEvent;

nn::dd::DeviceAddressSpaceType g_Das;
nn::dd::DeviceVirtualAddress g_WorkBufferDeviceVirtualAddress;
nn::dd::DeviceVirtualAddress g_DataBufferDeviceVirtualAddress;

char g_CalKey[] = {
0xd4,0xd0,0x7f,0xb6,0xe0,0xb8,0x63,0x82,0x26,0xfd,0x76,0x88,0x7a,0x86,0x8f,0x03
};

void PrintArray(const char* buffer, const size_t bufferLength) NN_NOEXCEPT
{
    for(size_t i=0; i<bufferLength; i++)
    {
        if(i!=0 && i % 16 == 0)
        {
            NN_LOG("\n");
        }

        NN_LOG("%02X ", buffer[i]);
    }
    NN_LOG("\n");
}

extern "C" void nninitStartup()
{
    nn::Result result = nn::os::SetMemoryHeapSize(MemoryHeapSize);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot set MemoryHeapSize.");

    // メモリヒープから WorkBuffer で使用するメモリ領域を確保
    result = nn::os::AllocateMemoryBlock( &g_WorkBufferAddress, WorkBufferSize );
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot AllocateMemoryBlock0.");

    // メモリヒープから DataBuffer で使用するメモリ領域を確保
    result = nn::os::AllocateMemoryBlock( &g_DataBufferAddress, DataBufferSize );
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot AllocateMemoryBlock1.");

    // メモリヒープから malloc で使用するメモリ領域を確保
    result = nn::os::AllocateMemoryBlock( &g_MallocBufferAddress, MemoryHeapSizeForMalloc );
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot AllocateMemoryBlock2.");

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(g_MallocBufferAddress), MemoryHeapSizeForMalloc );
}

// 挿抜があった際に呼ばれるコールバック関数：GC のスレッドをブロックしてしまうため、イベントのシグナルのみにとどめる必要がある
void SignalDetectionEvents(void *pParameter)
{
    nn::os::SignalEvent( &g_CardDetectionEvent );
}

void InitializeDriver()
{
    nn::spl::InitializeForFs();

    NN_LOG("Read From Partition\n");

    ReadGcCalibrationPartitionToSetInternalKeys();

    NN_LOG("Finalize gc driver from fs\n");

    nn::fs::FinalizeGameCardDriver();

    NN_LOG("Initialize gc driver.\n");

    // クロック制御は pcv 経由で行う
    nn::sdmmc::SwitchToPcvClockResetControl();

    // 自信で監視するイベントを用意する
    nn::os::InitializeEvent( &g_CardDetectionEvent, false, nn::os::EventClearMode_ManualClear );

    g_pWorkBuffer = reinterpret_cast<char*>(g_WorkBufferAddress);
    g_pDataBuffer = reinterpret_cast<char*>(g_DataBufferAddress);

    SetupDeviceAddressSpace(&g_Das, nn::dd::DeviceName::DeviceName_Sdmmc2a);

    g_WorkBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&g_Das, g_WorkBufferAddress, WorkBufferSize, 0);
    nn::gc::Initialize(g_pWorkBuffer, WorkBufferSize, g_WorkBufferDeviceVirtualAddress);

    g_DataBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&g_Das, g_DataBufferAddress, DataBufferSize, g_WorkBufferDeviceVirtualAddress + WorkBufferSize);
    nn::gc::RegisterDeviceVirtualAddress(g_DataBufferAddress, DataBufferSize, g_DataBufferDeviceVirtualAddress);

    nn::gc::RegisterDetectionEventCallback( SignalDetectionEvents, nullptr );
}

void FinalizeDriver()
{
    nn::gc::UnregisterDeviceVirtualAddress(g_DataBufferAddress, DataBufferSize, g_DataBufferDeviceVirtualAddress);
    UnmapDeviceAddressSpaceAligned(&g_Das, g_DataBufferAddress, DataBufferSize, g_DataBufferDeviceVirtualAddress);

    nn::gc::Finalize();
    UnmapDeviceAddressSpaceAligned(&g_Das, g_WorkBufferAddress, WorkBufferSize, g_WorkBufferDeviceVirtualAddress);

    CleanDeviceAddressSpace(&g_Das, nn::dd::DeviceName::DeviceName_Sdmmc2a);

    nn::spl::Finalize();
}

void CheckHs400()
{
    nn::fs::MmcSpeedMode speedMode;
    nn::Result result = nn::fs::GetMmcSpeedMode(&speedMode);
    if (result.IsFailure())
    {
        NN_LOG("nn::fs::GetMmcSpeedMode() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        NN_LOG("Test Failure\n");
        return;
    }
    if (speedMode != nn::fs::MmcSpeedMode_Hs400)
    {
        NN_LOG("MMC Speed Mode: %d\n", speedMode);
        NN_LOG("Test Failure\n");
        return;
    }
}

extern "C" void nnMain()
{
    NN_LOG("Check Hs400 First\n");

    CheckHs400();

    NN_LOG("Setting End.Gc Test Start.\n");

    InitializeDriver();

    while(! nn::gc::IsCardInserted())
    {
        NN_LOG("@ Waiting for Game Card Insert ... \n");
        nn::os::WaitEvent( &g_CardDetectionEvent );
        // イベントをクリア
        nn::os::ClearEvent( &g_CardDetectionEvent );
    }

    NN_LOG("\n\nFinished. Enable card bus.\n");

    nn::Result result = nn::gc::Activate();
    if(result.IsFailure())
    {
        NN_LOG("Test Failure\n");
        return;
    }

    NN_LOG("\n\nFinished. 2 page Read Test\n");

    result = nn::gc::Read(g_pDataBuffer, DataBufferSize, 0, 2);
    if(result.IsFailure())
    {
        NN_LOG("Test Failure");
        return;
    }

    NN_LOG("\n\nFinished. SetCardToSecureMode\n");

    result = nn::gc::SetCardToSecureMode();
    if(result.IsFailure())
    {
        NN_LOG("Test Failure");
        return;
    }

    result = nn::gc::Read(g_pDataBuffer, DataBufferSize, 0x80000, 2);
    if(result.IsFailure())
    {
        NN_LOG("Test Failure");
        return;
    }

    NN_LOG("\n\nFinished. Deactivate.\n");

    nn::gc::Deactivate();
    NN_LOG("\n\nFinalize.\n");
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(1));
    FinalizeDriver();

    nn::os::FreeMemoryBlock(g_MallocBufferAddress, MemoryHeapSizeForMalloc);
    nn::os::FreeMemoryBlock(g_DataBufferAddress, DataBufferSize);
    nn::os::FreeMemoryBlock(g_WorkBufferAddress, WorkBufferSize);
    NN_LOG("Test Success\n");
}

