﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <cstdlib>
#include <vector>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_BitTypes.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/gfx.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/gfx/util/gfx_ObjectHolder.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/os/os_MemoryHeap.h>
#include <nn/util/util_BitUtil.h>
#include <nn/util/util_MathTypes.h>
#include <nn/vi.h>

#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <nv/nv_MemoryManagement.h>

namespace
{
const size_t MemoryPoolSize = 64 * 1024 * 1024;
const size_t MemoryPoolSizeInvisible = 32 * 1024 * 1024;

const size_t CommandBufferCommandMemoryChunkSize = 512 * 1024;
const size_t CommandBufferControlMemoryChunkSize = 64 * 1024;

const uint32_t ScreenWidth = 1280;
const uint32_t ScreenHeight = 720;

const int SwapInterval = 1;

const uint32_t CharCountMax = 2048;

const int FrameRate = 60;
}

struct Color
{
    static const nn::util::Unorm8x4 White;
    static const nn::util::Unorm8x4 Black;
    static const nn::util::Unorm8x4 Gray;
    static const nn::util::Unorm8x4 Green;
    static const nn::util::Unorm8x4 Orange;
    static const nn::util::Unorm8x4 Red;
    static const nn::util::Unorm8x4 Blue;
};

//const nn::util::Unorm8x4 Color::White  = { { 255, 255, 255, 255 } };
//const nn::util::Unorm8x4 Color::Black  = { {  10,  10,  10, 255 } };
//const nn::util::Unorm8x4 Color::Gray   = { { 153, 153, 153, 255 } };
//const nn::util::Unorm8x4 Color::Green  = { { 153, 255,   0, 255 } };
//const nn::util::Unorm8x4 Color::Orange = { { 255, 153,   0, 255 } };
//const nn::util::Unorm8x4 Color::Red    = { { 255, 0,   0, 255 } };


class ApplicationHeap
{
    NN_DISALLOW_COPY(ApplicationHeap);
    NN_DISALLOW_MOVE(ApplicationHeap);

private:
    void* m_pMemoryHeap;
    nn::mem::StandardAllocator m_ApplicationHeapHandle;

public:
    ApplicationHeap() NN_NOEXCEPT
        : m_pMemoryHeap(0)
    {
    }

    void Initialize(size_t size) NN_NOEXCEPT
    {
        m_pMemoryHeap = new nn::Bit8[size];

        m_ApplicationHeapHandle.Initialize(m_pMemoryHeap, size);
    }

    void Finalize() NN_NOEXCEPT
    {
        m_ApplicationHeapHandle.Finalize();

        delete[] reinterpret_cast<nn::Bit8*>(m_pMemoryHeap);
    }

    void* Allocate(size_t size, size_t alignment) NN_NOEXCEPT
    {
        return m_ApplicationHeapHandle.Allocate(size, alignment);
    }

    void* Allocate(size_t size) NN_NOEXCEPT
    {
        return this->Allocate(size, sizeof(void*));
    }

    void Deallocate(void* ptr)
    {
        if (ptr != 0)
        {
            m_ApplicationHeapHandle.Free(ptr);
        }
    }
};

class GraphicsSystem
{
    NN_DISALLOW_COPY(GraphicsSystem);
    NN_DISALLOW_MOVE(GraphicsSystem);

private:
    ApplicationHeap* m_pApplicationHeap;

    nn::gfx::MemoryPool m_MemoryPool;
    void* m_pMemoryPool;
    ptrdiff_t m_pMemoryPoolOffset;
    ptrdiff_t m_pMemoryPoolCommandOffset;

    nn::gfx::MemoryPool m_MemoryPoolInvisible;
    void* m_pMemoryPoolInvisible;
    ptrdiff_t m_pMemoryPoolInvisibleOffset;

    nn::gfx::DescriptorPool m_TextureDescriptorPool;
    int m_TextureDescriptorBaseIndex;

    nn::gfx::DescriptorPool m_SamplerDescriptorPool;
    int m_SamplerDescriptorBaseIndex;

    nn::vi::Display* m_pDisplay;
    int m_DisplayWidth;
    int m_DisplayHeight;
    nn::vi::Layer* m_pLayer;
    nn::vi::NativeWindowHandle m_NativeWindowHandle;
    nn::gfx::Device m_Device;
    nn::gfx::SwapChain m_SwapChain;
    nn::gfx::Queue m_Queue;
    nn::gfx::Fence m_Fence;
    nn::gfx::Texture m_ColorTargetTexture;
    nn::gfx::ColorTargetView m_ColorTargetView;
    nn::gfx::Texture m_DepthStencilTexture;
    nn::gfx::DepthStencilView m_DepthStencilView;
    nn::gfx::CommandBuffer m_CommandBuffer;
    nn::gfx::ViewportScissorState m_ViewportScissorState;
    nn::gfx::BlendState m_BlendState;
    nn::gfx::DepthStencilState m_DepthStencilState;
    nn::gfx::RasterizerState m_RasterizerState;

    std::vector<void*> m_CommandBufferControlMemoryChunks;

public:
    GraphicsSystem() NN_NOEXCEPT
        : m_pApplicationHeap(0)
        , m_pMemoryPool(0)
        , m_pMemoryPoolOffset(0)
        , m_pMemoryPoolCommandOffset(0)
        , m_pMemoryPoolInvisible(0)
        , m_pMemoryPoolInvisibleOffset(0)
        , m_TextureDescriptorBaseIndex(0)
        , m_SamplerDescriptorBaseIndex(0)
        , m_pDisplay(0)
        , m_DisplayWidth(1280)
        , m_DisplayHeight(720)
        , m_pLayer(0)
    {
    }

    void SetApplicationHeap(ApplicationHeap* pValue) NN_NOEXCEPT
    {
        m_pApplicationHeap = pValue;
    }

    nn::gfx::DescriptorPool& GetTextureDescriptorPool() NN_NOEXCEPT
    {
        return m_TextureDescriptorPool;
    }

    int GetTextureDescriptorBaseIndex() const NN_NOEXCEPT
    {
        return m_TextureDescriptorBaseIndex;
    }

    nn::gfx::DescriptorPool& GetSamplerDescriptorPool() NN_NOEXCEPT
    {
        return m_SamplerDescriptorPool;
    }

    int GetSamplerDescriptorBaseIndex() const NN_NOEXCEPT
    {
        return m_SamplerDescriptorBaseIndex;
    }

    nn::vi::NativeWindowHandle GetNativeWindowHandle() NN_NOEXCEPT
    {
        return m_NativeWindowHandle;
    }

    nn::gfx::Device& GetDevice() NN_NOEXCEPT
    {
        return m_Device;
    }

    nn::gfx::CommandBuffer& GetCommandBuffer() NN_NOEXCEPT
    {
        return m_CommandBuffer;
    }

    void Initialize() NN_NOEXCEPT
    {
        nn::vi::Initialize();
        this->InitializeDisplay();
        nn::gfx::Initialize();
        this->InitializeDevice();
        this->InitializeMemoryPool();
        this->InitializeMemoryPoolInvisible();
        this->InitializeSwapChain();
        this->InitializeQueue();
        this->InitializeFence();
        this->InitializeColorTargetTexture();
        this->InitializeColorTargetView();
        this->InitializeDepthStencilTexture();
        this->InitializeDepthStencilView();
        this->InitializeTextureDescriptorPool();
        this->InitializeSamplerDescriptorPool();
        this->InitializeCommandBuffer();
        this->InitializeViewportScissorState();
        this->InitializeBlendState();
        this->InitializeDepthStencilState();
        this->InitializeRasterizerState();
    }

    void Finalize() NN_NOEXCEPT
    {
        m_RasterizerState.Finalize(&m_Device);
        m_DepthStencilState.Finalize(&m_Device);
        this->FinalizeBlendState();
        this->FinalizeViewportScissorState();
        this->ResetCommandBuffer();
        m_CommandBuffer.Finalize(&m_Device);
        m_SamplerDescriptorPool.Finalize(&m_Device);
        m_TextureDescriptorPool.Finalize(&m_Device);
        m_DepthStencilView.Finalize(&m_Device);
        m_DepthStencilTexture.Finalize(&m_Device);
        m_ColorTargetView.Finalize(&m_Device);
        m_ColorTargetTexture.Finalize(&m_Device);
        m_Fence.Finalize(&m_Device);
        m_Queue.Finalize(&m_Device);
        m_SwapChain.Finalize(&m_Device);
        this->FinalizeMemoryPoolInvisible();
        this->FinalizeMemoryPool();
        m_Device.Finalize();
        nn::gfx::Finalize();
        nn::vi::DestroyLayer(m_pLayer);
        nn::vi::CloseDisplay(m_pDisplay);
        nn::vi::Finalize();
    }

    void BeginDraw() NN_NOEXCEPT
    {
        this->ResetCommandBuffer();
        this->AddCommandMemoryToCommandBuffer(&m_CommandBuffer);
        this->AddControlMemoryToCommandBuffer(&m_CommandBuffer);

        m_CommandBuffer.Begin();

        m_CommandBuffer.SetDescriptorPool(&m_TextureDescriptorPool);
        m_CommandBuffer.SetDescriptorPool(&m_SamplerDescriptorPool);

        m_CommandBuffer.ClearColor(
            &m_ColorTargetView,
            static_cast<float>(Color::Black.v[0]) / Color::White.v[0],
            static_cast<float>(Color::Black.v[1]) / Color::White.v[1],
            static_cast<float>(Color::Black.v[2]) / Color::White.v[2],
            static_cast<float>(Color::Black.v[3]) / Color::White.v[3],
            0);

        m_CommandBuffer.ClearDepthStencil(
            &m_DepthStencilView,
            1.0f,
            0,
            nn::gfx::DepthStencilClearMode_DepthStencil,
            0);

        nn::gfx::ColorTargetView* pColorTargetView = &m_ColorTargetView;
        m_CommandBuffer.SetRenderTargets(
            1,
            &pColorTargetView,
            &m_DepthStencilView);

        m_CommandBuffer.SetViewportScissorState(&m_ViewportScissorState);
        m_CommandBuffer.SetBlendState(&m_BlendState);
        m_CommandBuffer.SetDepthStencilState(&m_DepthStencilState);
        m_CommandBuffer.SetRasterizerState(&m_RasterizerState);
    }

    void EndDraw() NN_NOEXCEPT
    {
        m_CommandBuffer.End();

        m_Queue.ExecuteCommand(&m_CommandBuffer, &m_Fence);
        m_Queue.Flush();

        m_Queue.CopyToScanBuffer(&m_SwapChain, &m_ColorTargetView);
        m_Queue.Present(&m_SwapChain, SwapInterval);
        m_Queue.Flush();
    }

    void Synchronize(const nn::TimeSpan& timeout) NN_NOEXCEPT
    {
        m_Fence.Sync(timeout);
    }

private:
    static void OutOfCommandMemoryEventCallback(
        nn::gfx::CommandBuffer* pCommandBuffer,
        const nn::gfx::OutOfMemoryEventArg& arg) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pCommandBuffer);
        NN_ASSERT_NOT_NULL(pCommandBuffer->GetUserPtr());
        NN_UNUSED(arg);

        reinterpret_cast<GraphicsSystem*>(
            pCommandBuffer->GetUserPtr()
            )->AddCommandMemoryToCommandBuffer(pCommandBuffer);
    }

    static void OutOfControlMemoryEventCallback(
        nn::gfx::CommandBuffer* pCommandBuffer,
        const nn::gfx::OutOfMemoryEventArg& arg) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pCommandBuffer);
        NN_ASSERT_NOT_NULL(pCommandBuffer->GetUserPtr());
        NN_UNUSED(arg);

        reinterpret_cast<GraphicsSystem*>(
            pCommandBuffer->GetUserPtr()
            )->AddControlMemoryToCommandBuffer(pCommandBuffer);
    }

    void InitializeDisplay() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&m_pDisplay));

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::vi::CreateLayer(&m_pLayer,
                                m_pDisplay));

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::vi::SetLayerScalingMode(m_pLayer,
                                        nn::vi::ScalingMode_FitToLayer));

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::vi::GetNativeWindow(&m_NativeWindowHandle, m_pLayer));
    }

    void InitializeDevice() NN_NOEXCEPT
    {
        nn::gfx::Device::InfoType info;
        info.SetApiVersion(nn::gfx::ApiMajorVersion, nn::gfx::ApiMinorVersion);
        info.SetDefault();

        m_Device.Initialize(info);
    }

    void InitializeMemoryPool() NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(m_pApplicationHeap);

        nn::gfx::MemoryPoolInfo info;
        info.SetDefault();
        info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuUncached |
                                   nn::gfx::MemoryPoolProperty_GpuCached|
                                   nn::gfx::MemoryPoolProperty_Compressible);

        const size_t aligment =
            nn::gfx::MemoryPool::GetPoolMemoryAlignment(&m_Device, info);

        m_pMemoryPool = m_pApplicationHeap->Allocate(MemoryPoolSize, aligment);

        info.SetPoolMemory(m_pMemoryPool, MemoryPoolSize);

        m_MemoryPool.Initialize(&m_Device, info);

        m_pMemoryPoolOffset = 0;

        m_pMemoryPoolCommandOffset = 0;
    }

    void InitializeMemoryPoolInvisible() NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(m_pApplicationHeap);

        nn::gfx::MemoryPoolInfo info;
        info.SetDefault();
        info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuInvisible |
                                   nn::gfx::MemoryPoolProperty_GpuCached |
                                   nn::gfx::MemoryPoolProperty_Compressible);

        const size_t aligment =
            nn::gfx::MemoryPool::GetPoolMemoryAlignment(&m_Device, info);

        m_pMemoryPoolInvisible =
            m_pApplicationHeap->Allocate(MemoryPoolSizeInvisible, aligment);

        info.SetPoolMemory(m_pMemoryPoolInvisible, MemoryPoolSizeInvisible);

        m_MemoryPoolInvisible.Initialize(&m_Device, info);

        m_pMemoryPoolInvisibleOffset = 0;
    }

    void InitializeSwapChain() NN_NOEXCEPT
    {
        nn::gfx::SwapChain::InfoType info;
        info.SetDefault();
        info.SetLayer(m_pLayer);
        info.SetWidth(m_DisplayWidth);
        info.SetHeight(m_DisplayHeight);
        info.SetBufferCount(2);

        m_pMemoryPoolInvisibleOffset = nn::util::align_up(
            m_pMemoryPoolInvisibleOffset,
            nn::gfx::SwapChain::GetScanBufferAlignment(&m_Device, info));

        size_t size = m_SwapChain.CalculateScanBufferSize(&m_Device, info);

        m_SwapChain.Initialize(&m_Device,
                               info,
                               &m_MemoryPoolInvisible,
                               m_pMemoryPoolInvisibleOffset,
                               size);

        m_pMemoryPoolInvisibleOffset += size;
    }

    void InitializeQueue() NN_NOEXCEPT
    {
        nn::gfx::Queue::InfoType info;
        info.SetDefault();
        info.SetCapability(nn::gfx::QueueCapability_Graphics);

        m_Queue.Initialize(&m_Device, info);
    }

    void InitializeFence() NN_NOEXCEPT
    {
        nn::gfx::Fence::InfoType info;
        info.SetDefault();

        m_Fence.Initialize(&m_Device, info);
    }

    void InitializeColorTargetTexture() NN_NOEXCEPT
    {
        nn::gfx::Texture::InfoType info;
        info.SetDefault();
        info.SetWidth(m_DisplayWidth);
        info.SetHeight(m_DisplayHeight);
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);
        info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        info.SetImageFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
        info.SetMipCount(1);
        info.SetDepth(1);

        m_pMemoryPoolInvisibleOffset = nn::util::align_up(
            m_pMemoryPoolInvisibleOffset,
            nn::gfx::Texture::CalculateMipDataAlignment(&m_Device, info));

        size_t size = nn::gfx::Texture::CalculateMipDataSize(&m_Device, info);

        m_ColorTargetTexture.Initialize(&m_Device,
                                        info,
                                        &m_MemoryPoolInvisible,
                                        m_pMemoryPoolInvisibleOffset,
                                        size);

        m_pMemoryPoolInvisibleOffset += size;
    }

    void InitializeColorTargetView() NN_NOEXCEPT
    {
        nn::gfx::ColorTargetView::InfoType info;
        info.SetDefault();
        info.SetImageDimension(nn::gfx::ImageDimension_2d);
        info.SetImageFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
        info.SetTexturePtr(&m_ColorTargetTexture);

        m_ColorTargetView.Initialize(&m_Device, info);
    }

    void InitializeDepthStencilTexture() NN_NOEXCEPT
    {
        nn::gfx::Texture::InfoType info;
        info.SetDefault();
        info.SetWidth(m_DisplayWidth);
        info.SetHeight(m_DisplayHeight);
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_DepthStencil);
        info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        info.SetImageFormat(nn::gfx::ImageFormat_D16_Unorm);
        info.SetMipCount(1);
        info.SetDepth(1);

        m_pMemoryPoolInvisibleOffset = nn::util::align_up(
            m_pMemoryPoolInvisibleOffset,
            nn::gfx::Texture::CalculateMipDataAlignment(&m_Device, info));

        size_t size = nn::gfx::Texture::CalculateMipDataSize(&m_Device, info);

        m_DepthStencilTexture.Initialize(&m_Device,
                                         info,
                                         &m_MemoryPoolInvisible,
                                         m_pMemoryPoolInvisibleOffset,
                                         size);

        m_pMemoryPoolInvisibleOffset += size;
    }

    void InitializeDepthStencilView() NN_NOEXCEPT
    {
        nn::gfx::DepthStencilView::InfoType info;
        info.SetDefault();
        info.SetImageDimension(nn::gfx::ImageDimension_2d);
        info.SetTexturePtr(&m_DepthStencilTexture);

        m_DepthStencilView.Initialize(&m_Device, info);
    }

    void InitializeTextureDescriptorPool() NN_NOEXCEPT
    {
#if NN_GFX_IS_TARGET_NVN
        nn::gfx::Device::DataType& type = nn::gfx::AccessorToData(m_Device);
        nvnDeviceGetInteger(type.pNvnDevice,
                            NVN_DEVICE_INFO_RESERVED_TEXTURE_DESCRIPTORS,
                            &m_TextureDescriptorBaseIndex);
#endif
        nn::gfx::DescriptorPool::InfoType info;
        info.SetDefault();
        info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_TextureView);
        info.SetSlotCount(m_TextureDescriptorBaseIndex + 1);

        m_pMemoryPoolOffset = nn::util::align_up(
            m_pMemoryPoolOffset,
            nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(
                &m_Device, info));

        size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(
            &m_Device, info);

        m_TextureDescriptorPool.Initialize(&m_Device,
                                           info,
                                           &m_MemoryPool,
                                           m_pMemoryPoolOffset,
                                           size);

        m_pMemoryPoolOffset += size;
    }

    void InitializeSamplerDescriptorPool() NN_NOEXCEPT
    {
#if NN_GFX_IS_TARGET_NVN
        nn::gfx::Device::DataType& type = nn::gfx::AccessorToData(m_Device);
        nvnDeviceGetInteger(type.pNvnDevice,
                            NVN_DEVICE_INFO_RESERVED_SAMPLER_DESCRIPTORS,
                            &m_SamplerDescriptorBaseIndex);
#endif
        nn::gfx::DescriptorPool::InfoType info;
        info.SetDefault();
        info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_Sampler);
        info.SetSlotCount(m_SamplerDescriptorBaseIndex + 1);

        m_pMemoryPoolOffset = nn::util::align_up(
            m_pMemoryPoolOffset,
            nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(
                &m_Device, info));

        size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(
            &m_Device, info);

        m_SamplerDescriptorPool.Initialize(&m_Device,
                                           info,
                                           &m_MemoryPool,
                                           m_pMemoryPoolOffset,
                                           size);

        m_pMemoryPoolOffset += size;
    }

    void InitializeCommandBuffer() NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(m_pApplicationHeap);

        nn::gfx::CommandBuffer::InfoType info;
        info.SetDefault();
        m_CommandBuffer.Initialize(&m_Device, info);
        m_CommandBuffer.SetUserPtr(this);
        m_CommandBuffer.SetOutOfCommandMemoryEventCallback(
            OutOfCommandMemoryEventCallback);
        m_CommandBuffer.SetOutOfControlMemoryEventCallback(
            OutOfControlMemoryEventCallback);

        m_pMemoryPoolOffset = nn::util::align_up(
            m_pMemoryPoolOffset,
            nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&m_Device));

        m_pMemoryPoolCommandOffset = m_pMemoryPoolOffset;
    }

    void InitializeViewportScissorState() NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(m_pApplicationHeap);

        nn::gfx::ViewportScissorState::InfoType info;
        info.SetDefault();
        info.SetScissorEnabled(true);

        nn::gfx::ViewportStateInfo viewportStateInfo;
        viewportStateInfo.SetDefault();
        viewportStateInfo.SetWidth(static_cast<float>(m_DisplayWidth));
        viewportStateInfo.SetHeight(static_cast<float>(m_DisplayHeight));
        info.SetViewportStateInfoArray(&viewportStateInfo, 1);

        nn::gfx::ScissorStateInfo scissorStateInfo;
        scissorStateInfo.SetDefault();
        scissorStateInfo.SetWidth(m_DisplayWidth);
        scissorStateInfo.SetHeight(m_DisplayHeight);
        info.SetScissorStateInfoArray(&scissorStateInfo, 1);

        size_t memorySize =
            nn::gfx::ViewportScissorState::GetRequiredMemorySize(info);
        m_ViewportScissorState.SetMemory(
            m_pApplicationHeap->Allocate(memorySize), memorySize);

        m_ViewportScissorState.Initialize(&m_Device, info);
    }

    void InitializeBlendState() NN_NOEXCEPT
    {
        nn::gfx::BlendState::InfoType info;
        info.SetDefault();

        nn::gfx::BlendTargetStateInfo targetInfo;
        targetInfo.SetDefault();
        targetInfo.SetChannelMask(nn::gfx::ChannelMask_Red |
                                  nn::gfx::ChannelMask_Green |
                                  nn::gfx::ChannelMask_Blue);
        targetInfo.SetBlendEnabled(true);
        targetInfo.SetColorBlendFunction(nn::gfx::BlendFunction_Add);
        targetInfo.SetDestinationColorBlendFactor(
            nn::gfx::BlendFactor_OneMinusSourceAlpha);
        targetInfo.SetSourceColorBlendFactor(nn::gfx::BlendFactor_SourceAlpha);
        info.SetBlendTargetStateInfoArray(&targetInfo, 1);

        size_t memorySize = nn::gfx::BlendState::GetRequiredMemorySize(info);
        m_BlendState.SetMemory(m_pApplicationHeap->Allocate(memorySize),
                               memorySize);

        m_BlendState.Initialize(&m_Device, info);
    }

    void InitializeDepthStencilState() NN_NOEXCEPT
    {
        nn::gfx::DepthStencilState::InfoType info;
        info.SetDefault();
        info.SetDepthTestEnabled(false);
        info.SetDepthWriteEnabled(true);

        m_DepthStencilState.Initialize(&m_Device, info);
    }

    void InitializeRasterizerState() NN_NOEXCEPT
    {
        nn::gfx::RasterizerState::InfoType info;
        info.SetDefault();
        info.SetCullMode(nn::gfx::CullMode_None);
        info.SetScissorEnabled(true);
        info.SetDepthClipEnabled(false);

        m_RasterizerState.Initialize(&m_Device, info);
    }

    void FinalizeMemoryPool() NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(m_pApplicationHeap);

        m_MemoryPool.Finalize(&m_Device);

        m_pApplicationHeap->Deallocate(m_pMemoryPool);
    }

    void FinalizeMemoryPoolInvisible() NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(m_pApplicationHeap);

        m_MemoryPoolInvisible.Finalize(&m_Device);

        m_pApplicationHeap->Deallocate(m_pMemoryPoolInvisible);
    }

    void FinalizeViewportScissorState() NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(m_pApplicationHeap);

        void* ptr = m_ViewportScissorState.GetMemory();

        m_ViewportScissorState.Finalize(&m_Device);

        m_pApplicationHeap->Deallocate(ptr);
    }

    void FinalizeBlendState() NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(m_pApplicationHeap);

        void* ptr = m_BlendState.GetMemory();

        m_BlendState.Finalize(&m_Device);

        m_pApplicationHeap->Deallocate(ptr);
    }

    void ResetCommandBuffer() NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(m_pApplicationHeap);

        m_CommandBuffer.Reset();

        m_pMemoryPoolOffset = m_pMemoryPoolCommandOffset;

        const size_t size = m_CommandBufferControlMemoryChunks.size();

        for (size_t i = 0; i < size; ++i)
        {
            m_pApplicationHeap->Deallocate(
                m_CommandBufferControlMemoryChunks[i]);
        }

        m_CommandBufferControlMemoryChunks.clear();
    }

    void AddCommandMemoryToCommandBuffer(nn::gfx::CommandBuffer* pCommandBuffer
                                         ) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pCommandBuffer);

        m_pMemoryPoolOffset = nn::util::align_up(
            m_pMemoryPoolOffset,
            nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&m_Device));

        pCommandBuffer->AddCommandMemory(&m_MemoryPool,
                                         m_pMemoryPoolOffset,
                                         CommandBufferCommandMemoryChunkSize);

        m_pMemoryPoolOffset += CommandBufferCommandMemoryChunkSize;
    }

    void AddControlMemoryToCommandBuffer(nn::gfx::CommandBuffer* pCommandBuffer
                                         ) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pCommandBuffer);
        NN_ASSERT_NOT_NULL(m_pApplicationHeap);

        void* pControlMemoryChunk = m_pApplicationHeap->Allocate(
            CommandBufferControlMemoryChunkSize,
            nn::gfx::CommandBuffer::GetControlMemoryAlignment(&m_Device));

        pCommandBuffer->AddControlMemory(pControlMemoryChunk,
                                         CommandBufferControlMemoryChunkSize);

        m_CommandBufferControlMemoryChunks.push_back(pControlMemoryChunk);
    }
};

class FontSystem
{
    NN_DISALLOW_COPY(FontSystem);
    NN_DISALLOW_MOVE(FontSystem);

private:
    ApplicationHeap* m_pApplicationHeap;

    GraphicsSystem* m_pGraphicsSystem;

    void* m_pDebugFontMemory;

    nn::gfx::MemoryPool m_DebugFontMemoryPool;
    void* m_pDebugFontMemoryPool;

    nn::gfx::util::DebugFontTextWriter m_DebugFontTextWriter;

public:
    FontSystem() NN_NOEXCEPT
        : m_pApplicationHeap(0)
        , m_pGraphicsSystem(0)
        , m_pDebugFontMemory(0)
        , m_pDebugFontMemoryPool(0)
    {
    }

    void SetApplicationHeap(ApplicationHeap* pValue) NN_NOEXCEPT
    {
        m_pApplicationHeap = pValue;
    }

    void SetGraphicsSystem(GraphicsSystem* pValue)
    {
        m_pGraphicsSystem = pValue;
    }

    void Initialize() NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(m_pApplicationHeap);
        NN_ASSERT_NOT_NULL(m_pGraphicsSystem);

        nn::gfx::util::DebugFontTextWriterInfo info;
        info.SetDefault();
        info.SetCharCountMax(CharCountMax);
        info.SetUserMemoryPoolEnabled(true);

        nn::gfx::Device* pDevice = &m_pGraphicsSystem->GetDevice();

        const size_t memorySize =
            nn::gfx::util::DebugFontTextWriter::GetRequiredMemorySize(
                pDevice, info);

        m_pDebugFontMemory = m_pApplicationHeap->Allocate(memorySize);

        const size_t memoryPoolSize =
            nn::gfx::util::DebugFontTextWriter::GetRequiredMemoryPoolSize(
                pDevice, info);

        nn::gfx::MemoryPoolInfo memoryPoolInfo;
        memoryPoolInfo.SetDefault();
        memoryPoolInfo.SetMemoryPoolProperty(
            nn::gfx::MemoryPoolProperty_CpuUncached |
            nn::gfx::MemoryPoolProperty_GpuCached);

        m_pDebugFontMemoryPool = m_pApplicationHeap->Allocate(
            memoryPoolSize,
            nn::gfx::MemoryPool::GetPoolMemoryAlignment(pDevice,
                                                        memoryPoolInfo));

        memoryPoolInfo.SetPoolMemory(m_pDebugFontMemoryPool, memoryPoolSize);

        m_DebugFontMemoryPool.Initialize(pDevice, memoryPoolInfo);

        m_DebugFontTextWriter.Initialize(pDevice,
                                         info,
                                         m_pDebugFontMemory,
                                         memorySize,
                                         &m_DebugFontMemoryPool,
                                         0,
                                         memoryPoolSize);

        m_DebugFontTextWriter.SetDisplayWidth(static_cast<int>(ScreenWidth));

        m_DebugFontTextWriter.SetDisplayHeight(static_cast<int>(ScreenHeight));

        m_DebugFontTextWriter.SetTextureDescriptor(
            &m_pGraphicsSystem->GetTextureDescriptorPool(),
            m_pGraphicsSystem->GetTextureDescriptorBaseIndex());

        m_DebugFontTextWriter.SetSamplerDescriptor(
            &m_pGraphicsSystem->GetSamplerDescriptorPool(),
            m_pGraphicsSystem->GetSamplerDescriptorBaseIndex());
    }

    void Finalize() NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(m_pApplicationHeap);
        NN_ASSERT_NOT_NULL(m_pGraphicsSystem);

        m_DebugFontTextWriter.Finalize();

        m_DebugFontMemoryPool.Finalize(&m_pGraphicsSystem->GetDevice());

        m_pApplicationHeap->Deallocate(m_pDebugFontMemoryPool);

        m_pApplicationHeap->Deallocate(m_pDebugFontMemory);
    }

    nn::gfx::util::DebugFontTextWriter& GetDebugFontTextWriter() NN_NOEXCEPT
    {
        return m_DebugFontTextWriter;
    }

    void Draw() NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(m_pGraphicsSystem);

        m_DebugFontTextWriter.Draw(&m_pGraphicsSystem->GetCommandBuffer());
    }
};

