﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>

#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_Vibration.h>

#include <nn/os.h>
#include <nn/util/util_Vector.h>

#include <nnt/nntest.h>

#include "../Common/testGamePad_Common.h"

namespace
{
    const int VibrationDeviceCountMax = 2;

#if false
    void CheckVibrationValue(const nn::hid::VibrationValue* pSrc, const nn::hid::VibrationValue* pDst)
    {
        ASSERT_NEAR(pSrc->amplitudeHigh, pDst->amplitudeHigh, 0.01);
        ASSERT_NEAR(pSrc->amplitudeLow, pDst->amplitudeLow, 0.01);
        ASSERT_NEAR(pSrc->frequencyHigh, pDst->frequencyHigh, 0.1);
        ASSERT_NEAR(pSrc->frequencyLow, pDst->frequencyLow, 0.1);
    }
#endif

    bool CheckVibrationValue(const nn::hid::VibrationValue* pSrc, const nn::hid::VibrationValue* pDst)
    {
        float diff = pSrc->amplitudeHigh - pDst->amplitudeHigh;
        if (diff > 0.01 || diff < -0.01)
        {
            return false;
        }

        diff = pSrc->amplitudeLow - pDst->amplitudeLow;
        if (diff > 0.01 || diff < -0.01)
        {
            return false;
        }

        diff = pSrc->frequencyHigh - pDst->frequencyHigh;
        if (diff > 0.1 || diff < -0.1)
        {
            return false;
        }

        diff = pSrc->frequencyLow - pDst->frequencyLow;
        if (diff > 0.1 || diff < -0.1)
        {
            return false;
        }

        return true;
    }

    template<typename T>
    bool IsConnected(const nn::hid::NpadStyleSet& style,
        const ::nn::hid::NpadIdType& npadId,
        int handleIndex) NN_NOEXCEPT
    {
        if (T::Index != nn::hid::NpadStyleJoyDual::Index)
        {
            return style.Test<T>();
        }

        // JoyDual の時は左右ジョイコンの接続状態を見る
        ::nn::hid::NpadJoyDualState state;
        ::nn::hid::GetNpadState(&state, npadId);
        bool isConnected = false;

        switch (handleIndex)
        {
        case 0: // Left
            isConnected = state.attributes.Test<::nn::hid::NpadJoyAttribute::IsLeftConnected>();
            break;
        case 1: // Right
            isConnected = state.attributes.Test<::nn::hid::NpadJoyAttribute::IsRightConnected>();
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        return isConnected;
    }

    template<typename T>
    void ReadingStateTest(const ::nn::hid::NpadIdType& npadId) NN_NOEXCEPT
    {
        // 振動子の有無を確認
        if ((::nnt::gamepad::GetControllerTypeFromNpad(npadId) == ::nnt::gamepad::ControllerType::Undefined) ||
            (::nnt::gamepad::GetControllerTypeFromNpad(npadId) == ::nnt::gamepad::ControllerType::UsbCon))
        {
            return;
        }


        // ハンドルの取得
        ::nn::hid::VibrationDeviceHandle handles[VibrationDeviceCountMax];
        int handleCount = nn::hid::GetVibrationDeviceHandles(
            handles, VibrationDeviceCountMax, npadId, T::Mask);

        const ::nn::hid::VibrationValue vib = nn::hid::VibrationValue::Make(1.0f, 160.0f, 0.0f, 320.0f);
        const ::nn::hid::VibrationValue zero = nn::hid::VibrationValue::Make();

        // 現時点では接続しているときだけ確認　いまだとJoy-ConはJoyDualの時だけ有効になる
        nn::hid::NpadStyleSet style = ::nn::hid::GetNpadStyleSet(npadId);

        // 接続確認
        bool isConnected = true;

        for (int i = 0; i < handleCount; i++)
        {
            isConnected &= IsConnected<T>(style, npadId, i);
        }

        // 接続していなければテスト終了
        if (isConnected == false)
        {
            return;
        }

        for (int i = 0; i < handleCount; i++)
        {
            // 振動子を初期化します
            ::nn::hid::InitializeVibrationDevice(handles[i]);
        }

        // アンプの初期化待ち
        ::nn::os::SleepThread(
            ::nn::TimeSpan::FromMilliSeconds(500));

        for (int i = 0; i < handleCount; i++)
        {
            ::nn::hid::VibrationDeviceInfo deviceInfo;

            // 振動子の情報を取得する
            ::nn::hid::GetVibrationDeviceInfo(&deviceInfo, handles[i]);

            // 振動値を一旦クリア
            ::nn::hid::SendVibrationValue(handles[i], zero);
        }

        for (int i = 0; i < handleCount; i++)
        {
            bool result = false;

            // 実際に振動させてみて値を確認する　最大5回まで試す
            for (int count = 0; count < 5; count++)
            {
                // 振動値を送信します
                ::nn::hid::SendVibrationValue(handles[i], vib);

                // 送信待ち
                ::nn::os::SleepThread(
                    ::nn::TimeSpan::FromMilliSeconds(50));

                // 現在の振動を取得します
                ::nn::hid::VibrationValue ret;
                ::nn::hid::GetActualVibrationValue(&ret, handles[i]);

                // 振動値の確認
                result = CheckVibrationValue(&ret, &vib);

                NN_LOG("Try:%d/5 NpadId:0x%x StyleIndex:%x, HandleIndex:%x/%x isConnected:%d result:%d %1.2f %1.2f %1.2f %1.2f\n",
                    count, npadId, T::Index, i, handleCount, isConnected, result,
                    ret.amplitudeLow, ret.frequencyLow, ret.amplitudeHigh, ret.frequencyHigh);

                // 一度でも成功すればループを抜ける
                if (result)
                {
                    break;
                }
            }

            // 振動停止
            ::nn::hid::SendVibrationValue(handles[i], zero);

            EXPECT_TRUE(result);
        }
    }
} // namespace

//!< 振動子に設定した値が正しく取得できるか(GetActualVibrationValueを使った確認)
TEST(Vibration, NormalTest1)
{
    ::nnt::gamepad::Initialize();

    // ProCon USB 無効
    ::nnt::gamepad::DisableUsbConnect();

    // コントローラの再接続
    ::nnt::gamepad::DisconnectAll();
    ::nnt::gamepad::ConnectAll();

    // 各 NpadId、操作形態で確認
    for (const auto& id : nnt::gamepad::NpadIds)
    {
        ReadingStateTest<::nn::hid::NpadStyleFullKey>(id);
        ReadingStateTest<::nn::hid::NpadStyleHandheld>(id);
        ReadingStateTest<::nn::hid::NpadStyleJoyDual>(id);
        ReadingStateTest<::nn::hid::NpadStyleJoyLeft>(id);
        ReadingStateTest<::nn::hid::NpadStyleJoyRight>(id);
    }

    ::nnt::gamepad::DisconnectAll();
}

#if false
//!< 振動子に設定した値が正しく取得できるか(加速度を使って確認)
TEST(Vibration, NormalTest2)
{
}

#endif
