﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_NpadCommon.h>
#include <nn/hid/hid_NpadFullKey.h>
#include <nn/hid/hid_NpadJoyCommon.h>
#include <nn/hid/hid_NpadJoyDual.h>
#include <nn/hid/hid_NpadSixAxisSensor.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/os.h>
#include <nn/util/util_Vector.h>

#include <nnt/nntest.h>

#include "../Common/testGamePad_Common.h"
#include "../Common/testGamePad_SixAxisSensor.h"

namespace
{

template<typename T>
bool IsConnected(const nn::hid::NpadStyleSet& style,
                 const ::nn::hid::NpadIdType& npadId,
                 int handleIndex) NN_NOEXCEPT
{
    if (T::Index != nn::hid::NpadStyleJoyDual::Index)
    {
        return style.Test<T>();
    }

    // JoyDual の時は左右ジョイコンの接続状態を見る
    ::nn::hid::NpadJoyDualState state;
    ::nn::hid::GetNpadState(&state, npadId);
    bool isConnected = false;

    switch (handleIndex)
    {
    case 0: // Left
        isConnected = state.attributes.Test<::nn::hid::NpadJoyAttribute::IsLeftConnected>();
        break;
    case 1: // Right
        isConnected = state.attributes.Test<::nn::hid::NpadJoyAttribute::IsRightConnected>();
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    return isConnected;
}

template<typename T>
void ReadingStateTest(const ::nn::hid::NpadIdType& npadId) NN_NOEXCEPT
{
    // 6 軸非搭載デバイスはスキップ
    if ((::nnt::gamepad::GetControllerTypeFromNpad(npadId) == ::nnt::gamepad::ControllerType::Undefined) ||
        (::nnt::gamepad::GetControllerTypeFromNpad(npadId) == ::nnt::gamepad::ControllerType::UsbCon))
    {
        return;
    }

    ::nn::hid::SixAxisSensorState states[::nn::hid::SixAxisSensorStateCountMax];
    ::nn::hid::SixAxisSensorHandle handles[::nn::hid::SixAxisSensorStateCountMax];

    // ハンドルの取得
    int handleCount = nn::hid::GetSixAxisSensorHandles(handles,
                                                       ::nn::hid::NpadSixAxisSensorHandleCountMax,
                                                       npadId,
                                                       T::Mask);
    for (int i = 0; i < handleCount; i++)
    {
        nn::hid::StartSixAxisSensor(handles[i]);

        ::nn::os::SleepThread(
            ::nn::TimeSpan::FromMilliSeconds(
                ::nn::hid::SixAxisSensorStateCountMax *
                ::nnt::gamepad::GetSixAxisSensorSamplingInterval().GetMilliSeconds()));

        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(npadId);

        if (IsConnected<T>(style, npadId, i))
        {
            int count = nn::hid::GetSixAxisSensorStates(states,
                                                        ::nn::hid::SixAxisSensorStateCountMax,
                                                        handles[i]);

            ASSERT_GE(::nn::hid::SixAxisSensorStateCountMax, count);

            NN_LOG("NpadId:0x%x, StyleIndex:%x, HandleIndex:%x\n", npadId, T::Index, i);
            ::nnt::gamepad::PrintSixAxisSensorState(states[0]);

            // 加速度値が 1G 付近か
            ::nn::util::Vector3f accelerometer;
            ::nn::util::VectorLoad(&accelerometer, states[0].acceleration);
            ASSERT_NEAR(accelerometer.Length(), 1.0f, 0.1);

            // 角速度の各成分が 0.0f 付近か
            for (const auto& angularVelocity : states[0].angularVelocity.v)
            {
                ASSERT_NEAR(angularVelocity, 0.0f, 0.1f);
            }

            const auto SamplingNumber = states[0].samplingNumber;

            // 1 [sec] 待つ
            ::nn::os::SleepThread(
                ::nn::TimeSpan::FromSeconds(1)
            );

            nn::hid::GetSixAxisSensorState(&states[0], handles[i]);

            const auto ExpectedSamplingFrequency = 1000.f / ::nnt::gamepad::GetSixAxisSensorSamplingInterval().GetMilliSeconds();
            const auto ActualSamplingFrequency = states[0].samplingNumber - SamplingNumber;
            const auto PermissiblePacketLostRate = 0.3f;
            NN_LOG("ActualSamplingFrequency: %lld\n", ActualSamplingFrequency);
            ASSERT_NEAR(ExpectedSamplingFrequency,
                        ActualSamplingFrequency,
                        ExpectedSamplingFrequency * PermissiblePacketLostRate);

        }
    }
}

} // namespace


//!< 6軸センサーの入力状態が正しく取得できるか
TEST(SixAxisSensor, StateReading)
{
    ::nnt::gamepad::Initialize();

    // ProCon USB 無効
    ::nnt::gamepad::DisableUsbConnect();

    // コントローラの再接続
    ::nnt::gamepad::DisconnectAll();
    ::nnt::gamepad::ConnectAll();

    // 各 NpadId、操作形態で六軸の状態取得
    for (const auto& id : nnt::gamepad::NpadIds)
    {
        ReadingStateTest<::nn::hid::NpadStyleFullKey>(id);
        ReadingStateTest<::nn::hid::NpadStyleHandheld>(id);
        ReadingStateTest<::nn::hid::NpadStyleJoyDual>(id);
        ReadingStateTest<::nn::hid::NpadStyleJoyLeft>(id);
        ReadingStateTest<::nn::hid::NpadStyleJoyRight>(id);
    }

    // ProCon USB 有効
    ::nnt::gamepad::EnableUsbConnect();

    // コントローラの再接続
    ::nnt::gamepad::DisconnectAll();
    ::nnt::gamepad::ConnectAll();

    // 各 NpadId、操作形態で六軸の状態取得
    for (const auto& id : nnt::gamepad::NpadIds)
    {
        ReadingStateTest<::nn::hid::NpadStyleFullKey>(id);
        ReadingStateTest<::nn::hid::NpadStyleHandheld>(id);
        ReadingStateTest<::nn::hid::NpadStyleJoyDual>(id);
        ReadingStateTest<::nn::hid::NpadStyleJoyLeft>(id);
        ReadingStateTest<::nn::hid::NpadStyleJoyRight>(id);
    }

    ::nnt::gamepad::DisconnectAll();
}
