﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/system/hid_Npad.h>
#include <nnt/nntest.h>

#include "../Common/testGamePad_Common.h"
#include "../Common/testGamePad_Npad.h"


namespace
{
//!< System 向けに有効なスタイルのマスク
const nn::hid::NpadStyleSet NpadSystemStyles = nn::hid::system::NpadStyleSystem::Mask | nn::hid::system::NpadStyleSystemExt::Mask;

//!< デバイス毎に対応するスタイルを取得する関数
nn::hid::NpadStyleSet GetSupportedNpadStyleSet(const nn::hid::system::NpadDeviceTypeSet deviceType)
{
    auto supportedStyle = nn::hid::GetSupportedNpadStyleSet();
    nn::hid::NpadStyleSet returnStyle;
    returnStyle.Reset();

    // デバイス毎に対応するスタイルをセットする
    if (deviceType.Test<nn::hid::system::NpadDeviceType::JoyConLeft>() &&
        deviceType.Test<nn::hid::system::NpadDeviceType::JoyConRight>())
    {
        if (supportedStyle.Test<nn::hid::NpadStyleJoyDual>())
        {
            returnStyle = nn::hid::NpadStyleJoyDual::Mask;
        }
        else
        {
            returnStyle = nn::hid::NpadStyleFullKey::Mask;
        }
    }
    else if (deviceType.Test<nn::hid::system::NpadDeviceType::JoyConLeft>())
    {
        returnStyle = nn::hid::NpadStyleJoyLeft::Mask;
    }
    else if (deviceType.Test<nn::hid::system::NpadDeviceType::JoyConRight>())
    {
        returnStyle = nn::hid::NpadStyleJoyLeft::Mask;
    }
    else if (deviceType.Test<nn::hid::system::NpadDeviceType::FullKeyController>())
    {
        returnStyle = nn::hid::NpadStyleFullKey::Mask;
    }
    else if (deviceType.Test<nn::hid::system::NpadDeviceType::HandheldJoyLeft>() &&
        deviceType.Test<nn::hid::system::NpadDeviceType::HandheldJoyRight>())
    {
        returnStyle = nn::hid::NpadStyleHandheld::Mask;
    }

    // デバイスがひとつでも有効の場合はシステムのスタイルを有効にする
    if (deviceType.IsAnyOn())
    {
        returnStyle |= NpadSystemStyles;
    }

    // サポートされているスタイルでマスクする
    return returnStyle &= supportedStyle;
}

//!< NpadStyleSet を無効化した場合に、非サポートのコントローラーが切断されるか
void TestLimitStyleAndDetach(const nn::hid::NpadStyleSet styleToTest,
                             const nn::hid::system::NpadDeviceTypeSet deviceToBeDetached)
{
    nn::hid::system::NpadDeviceTypeSet originalDeviceType[nnt::gamepad::NpadIdCountMax];

    // テストマシンの構成に応じた Npad の初期化
    nnt::gamepad::Initialize();

    // USB有線通信を無効化
    nnt::gamepad::DisableUsbConnect();

    // コントローラを切断
    nnt::gamepad::DisconnectAll();

    // コントローラを接続
    nnt::gamepad::ConnectAll();

    // フルコンの接続されている Npad を探索
    for (int i = 0; i < nnt::gamepad::NpadIdCountMax; i++)
    {
        originalDeviceType[i] = nn::hid::system::GetNpadDeviceType(nnt::gamepad::NpadIds[i]);
    }

    // NpadStyleFullKey を無効化
    auto supportedStyleSet = nn::hid::GetSupportedNpadStyleSet();
    supportedStyleSet ^= styleToTest;
    nn::hid::SetSupportedNpadStyleSet(supportedStyleSet);

    // フルコンの接続されている Npad からデバイスが切断されたことを確認
    for (int i = 0; i < nnt::gamepad::NpadIdCountMax; i++)
    {
        auto deviceType = nn::hid::system::GetNpadDeviceType(nnt::gamepad::NpadIds[i]);
        NN_LOG("NpadId: %d\n", nnt::gamepad::NpadIds[i]);
        // 対応するデバイスが Detach されたことの検証
        if ((originalDeviceType[i] & deviceToBeDetached).IsAnyOn())
        {
            NN_LOG("Check expected device detach\n");
            EXPECT_EQ((deviceType & deviceToBeDetached), nn::hid::system::NpadDeviceTypeSet());
        }
        NN_LOG("Check expected device undetached\n");
        // 関係ないデバイスの接続は維持されていることの検証
        EXPECT_EQ((originalDeviceType[i] & ~deviceToBeDetached), deviceType);
    }

    // コントローラを切断
    nnt::gamepad::DisconnectAll();
}

}

//!< デバイスに対して正しいステートが取得されるか
TEST(NpadSupportedStyleTest, StyleOnDeviceTypeSet)
{
    // テストマシンの構成に応じた Npad の初期化
    nnt::gamepad::Initialize();

    // USB有線通信を無効化
    nnt::gamepad::DisableUsbConnect();

    // コントローラを切断
    nnt::gamepad::DisconnectAll();

    // コントローラを接続
    nnt::gamepad::ConnectAll();

    // 各 NpadId, 操作携帯で入力状態が取得できているかチェック
    for (const auto& id : nnt::gamepad::NpadIds)
    {
        auto deviceType = nn::hid::system::GetNpadDeviceType(id);
        auto style = nn::hid::GetNpadStyleSet(id);

        EXPECT_EQ(style, GetSupportedNpadStyleSet(deviceType));
    }

    // コントローラを切断
    nnt::gamepad::DisconnectAll();
}

//!< NpadStyleFullKey を無効にしたら、フルコンが切断されるか
TEST(NpadSupportedStyleTest, DetachOnFullKeyDisable)
{
    TestLimitStyleAndDetach(
        nn::hid::NpadStyleFullKey::Mask,
        nn::hid::system::NpadDeviceType::FullKeyController::Mask);
}
