﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <nn/nn_Log.h>
#include "../../Common/testGamePad_Common.h"
#include "../../Common/testGamePad_IrSensor.h"

namespace nnt { namespace irsensor {
    nn::irsensor::IrCameraHandle IrSensorTest::s_Handles[nnt::gamepad::NpadIdCountMax];
    int IrSensorTest::s_HandleCount;
}} // namaspace nnt::irsensor

namespace
{
enum IrCameraMode
{
    IrCameraMode_Moment = 0,
    IrCameraMode_Clustering = 1,
    IrCameraMode_ImageTransfer,
    IrCameraMode_HandAnalysis,
    IrCameraMode_Pointing,
    IrCameraMode_IrLed,
    IrCameraMode_AdaptiveClustering,
    IrCameraMode_Num
};

const size_t IrCameraModeCount = IrCameraMode_Num;

NN_ALIGNAS(0x1000)
uint8_t g_WorkBuffer[nn::irsensor::ImageTransferProcessorWorkBufferSize320x240];
uint8_t g_OutBuffer[nn::irsensor::ImageTransferProcessorImageSize320x240];

const int NotReadyWaitLoopCountMax = 1000;
const nn::TimeSpan PollingInterval = nn::TimeSpan::FromMilliSeconds(15);

template <typename T, size_t N>
class TestCase
{
    NN_STATIC_ASSERT(0 < N);
private:
    void SetValue(T val) NN_NOEXCEPT;
    void Test(T test) NN_NOEXCEPT;
public:
    TestCase() NN_NOEXCEPT;
    void Clear() NN_NOEXCEPT;
    bool IsCompleted() NN_NOEXCEPT;
    bool GetNextTest(T* pNextTest) NN_NOEXCEPT;
    void PrintTestPass() NN_NOEXCEPT;
private:
    bool m_IsC1TestCompleted[N * N];        // 実行済みの 0スイッチカバレッジテスト
    bool m_Is1STestCompleted[N * N];        // 実行済みの 1スイッチカバレッジテスト
    int m_CompletedC1TestCount;
    int m_Completed1STestCount;
    std::vector<T> m_PrevValue;
};

class ModeTransitionTest : public ::nnt::irsensor::IrSensorTest
{
protected:

    void RunImageProcessor(const nn::irsensor::IrCameraHandle& handle, const nn::irsensor::MomentProcessorConfig& config) NN_NOEXCEPT;
    void RunImageProcessor(const nn::irsensor::IrCameraHandle& handle, const nn::irsensor::ClusteringProcessorConfig& config) NN_NOEXCEPT;
    void RunImageProcessor(const nn::irsensor::IrCameraHandle& handle, const nn::irsensor::ImageTransferProcessorConfig& config) NN_NOEXCEPT;
    void RunImageProcessor(const nn::irsensor::IrCameraHandle& handle, const nn::irsensor::HandAnalysisConfig& config) NN_NOEXCEPT;
    void RunImageProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;
    void RunImageProcessor(const nn::irsensor::IrCameraHandle& handle, const nn::irsensor::IrLedProcessorConfig& config) NN_NOEXCEPT;
    void RunImageProcessor(const nn::irsensor::IrCameraHandle& handle, const nn::irsensor::AdaptiveClusteringProcessorConfig& config) NN_NOEXCEPT;

    void DoTestStaticModeTransition(const nn::irsensor::IrCameraHandle& handle, const IrCameraMode mode) NN_NOEXCEPT;
    void DoTestDynamicModeTransition(const nn::irsensor::IrCameraHandle& handle, const IrCameraMode mode) NN_NOEXCEPT;

    void DoTestRunMoment(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;
    void DoTestRunClustering(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;
    void DoTestRunImageTransfer(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;
    void DoTestRunHandAnalysis(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;
    void DoTestRunPointing(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;
    void DoTestRunIrLed(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;
    void DoTestRunAdaptiveClustering(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;

    void DoTestStopImageProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;
};

//==================================================
// テストケースを生成します
//==================================================

template <typename T, size_t N>
void TestCase<T, N>::SetValue(T val) NN_NOEXCEPT
{
    m_PrevValue.push_back(val);
    if (m_PrevValue.size() > 2)
    {
        m_PrevValue.erase(m_PrevValue.begin());
    }
}

template <typename T, size_t N>
void TestCase<T, N>::Test(T test) NN_NOEXCEPT
{
    if (!m_PrevValue.empty())
    {
        // 過去に最低1回以上テストを行っている場合
        if (m_IsC1TestCompleted[m_PrevValue.back() * N + test] == false)
        {
            ++m_CompletedC1TestCount;
            m_IsC1TestCompleted[m_PrevValue.back() * N + test] = true;
        }
        if (m_PrevValue.size() == 2)
        {
            // 過去に2回テストを行っている場合
            if (m_Is1STestCompleted[m_PrevValue.front() * N + test] == false)
            {
                ++m_Completed1STestCount;
                m_Is1STestCompleted[m_PrevValue.front() * N + test] = true;
            }
        }
    }
    SetValue(test);
}

template <typename T, size_t N>
TestCase<T, N>::TestCase() NN_NOEXCEPT
{
    Clear();
}

template <typename T, size_t N>
void TestCase<T, N>::Clear() NN_NOEXCEPT
{
    for (auto& val : m_IsC1TestCompleted)
    {
        val = false;
    }
    for (auto& val : m_Is1STestCompleted)
    {
        val = false;
    }
    m_PrevValue.resize(0);
    m_CompletedC1TestCount = 0;
    m_Completed1STestCount = 0;
}

template <typename T, size_t N>
bool TestCase<T, N>::IsCompleted() NN_NOEXCEPT
{
    return ((m_CompletedC1TestCount == N * N) && (m_Completed1STestCount == N * N));
}

template <typename T, size_t N>
bool TestCase<T, N>::GetNextTest(T* pNextTest) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pNextTest);

    T& nextTest = *pNextTest;
    nextTest = static_cast<T>(0);

    if (IsCompleted())
    {
        return false;
    }

    if (!m_PrevValue.empty())
    {
        std::vector<T> testCur; // testCur.front() を次にテストすべき値として返します

        //--------------------------------------------------------------------
        // [前々回のテスト] から遷移する場合の 1スイッチカバレッジテスト で
        // 未実施のものを列挙しテスト候補に追加します
        //--------------------------------------------------------------------
        if (m_PrevValue.size() > 1)
        {
            for (auto val = m_PrevValue.front() * N; val < (m_PrevValue.front() + 1) * N; ++val)
            {
                if (m_Is1STestCompleted[val] == false)
                {
                    testCur.push_back(static_cast<T>(val % N));
                }
            }
        }

        //--------------------------------------------------------------------
        // [前回のテスト] から遷移する場合の 0スイッチカバレッジテスト で
        // 未実施のものを列挙しテスト候補に追加します
        //--------------------------------------------------------------------
        for (auto val = m_PrevValue.back() * N; val < (m_PrevValue.back() + 1) * N; ++val)
        {
            if (m_IsC1TestCompleted[val] == false)
            {
                auto itr = std::find(testCur.begin(), testCur.end(), static_cast<T>(val % N));
                size_t index = std::distance(testCur.begin(), itr);

                if (index != testCur.size())
                {
                    // 0スイッチテストと 1スイッチテストが共に未実施の組み合わせは優先的に実行します
                    testCur.erase(itr);
                    testCur.insert(testCur.begin(), static_cast<T>(val % N));
                }
                else
                {
                    testCur.push_back(static_cast<T>(val % N));
                }
            }
        }
        //--------------------------------------------------------------------
        // [前回のテスト]、[前々回のテスト] から遷移する場合のテストが
        // 全て実施済みの時、未実施の0スイッチカバレッジテストの組み合わせを検索し
        // 遷移前の状態をテスト候補に追加します
        //--------------------------------------------------------------------
        for (auto i = 0; i < NN_ARRAY_SIZE(m_IsC1TestCompleted) && testCur.empty(); ++i)
        {
            if (m_IsC1TestCompleted[i] == false)
            {
                // 未実施の 0 スイッチカバレッジテストがあった場合
                testCur.push_back(static_cast<T>(i / N));
                break;
            }
        }
        //--------------------------------------------------------------------
        // [前回のテスト]、[前々回のテスト] から遷移する場合のテストが
        // 全て実施済みの時、未実施の1スイッチカバレッジテストの組み合わせを検索し
        // 遷移前の状態をテスト候補に追加します
        //--------------------------------------------------------------------
        for (auto i = 0; i < NN_ARRAY_SIZE(m_Is1STestCompleted) && testCur.empty(); ++i)
        {
            if (m_Is1STestCompleted[i] == false)
            {
                // 未実施の 1 スイッチカバレッジテストがあった場合
                testCur.push_back(static_cast<T>(i / N));
                break;
            }
        }

        EXPECT_FALSE(testCur.empty());      // 実行すべきテストが存在せず、かつ全テストケースを網羅していない場合はエラー

        if (testCur.empty())
        {
            return false;
        }
        else
        {
            nextTest = testCur.front();
        }
    }

    Test(nextTest);
    return true;
}

template <typename T, size_t N>
void TestCase<T, N>::PrintTestPass() NN_NOEXCEPT
{
    Clear();
    IrCameraMode mode;
    while (GetNextTest(&mode))
    {
        NN_LOG("[%d]→", mode);
    }
    NN_LOG("Exit\n");
    Clear();
}

//==================================================
// IRセンサの各プロセッサの処理の開始
//==================================================

void ModeTransitionTest::RunImageProcessor(const nn::irsensor::IrCameraHandle& handle, const nn::irsensor::MomentProcessorConfig& config) NN_NOEXCEPT
{
    RunMomentProcessor(handle, config);
    nn::TimeSpanType startTime = nn::os::GetSystemTick().ToTimeSpan();
    auto counter = 0;
    while (NN_STATIC_CONDITION(true))
    {
        nn::irsensor::MomentProcessorState state;
        nn::Result result = nn::irsensor::GetMomentProcessorState(&state, handle);

        if (nn::irsensor::ResultIrsensorNotReady::Includes(result))
        {
            counter++;
            nn::os::SleepThread(PollingInterval);
            ASSERT_LT(counter, NotReadyWaitLoopCountMax);
        }
        else
        {
            NN_LOG("# RunMomentProcessor (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
            break;
        }
    }
    NNT_IRSENSOR_EXIT_SUCCESS;
}

void ModeTransitionTest::RunImageProcessor(const nn::irsensor::IrCameraHandle& handle, const nn::irsensor::ClusteringProcessorConfig& config) NN_NOEXCEPT
{
    RunClusteringProcessor(handle, config);
    nn::TimeSpanType startTime = nn::os::GetSystemTick().ToTimeSpan();
    auto counter = 0;
    while (NN_STATIC_CONDITION(true))
    {
        nn::irsensor::ClusteringProcessorState state;
        nn::Result result = nn::irsensor::GetClusteringProcessorState(&state, handle);
        if (nn::irsensor::ResultIrsensorNotReady::Includes(result))
        {
            counter++;
            nn::os::SleepThread(PollingInterval);
            ASSERT_LT(counter, NotReadyWaitLoopCountMax);
        }
        else
        {
            NN_LOG("# RunClusteringProcessor (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
            break;
        }
    }
    NNT_IRSENSOR_EXIT_SUCCESS;
}

void ModeTransitionTest::RunImageProcessor(const nn::irsensor::IrCameraHandle& handle, const nn::irsensor::ImageTransferProcessorConfig& config) NN_NOEXCEPT
{
    auto buffer = g_WorkBuffer;
    auto image = g_OutBuffer;
    RunImageTransferProcessor(handle, config, buffer, nn::irsensor::ImageTransferProcessorWorkBufferSize320x240);
    nn::TimeSpanType startTime = nn::os::GetSystemTick().ToTimeSpan();
    auto counter = 0;
    while (NN_STATIC_CONDITION(true))
    {
        nn::irsensor::ImageTransferProcessorState state;
        nn::Result result = nn::irsensor::GetImageTransferProcessorState(&state, image, nn::irsensor::ImageTransferProcessorImageSize320x240, handle);
        if (nn::irsensor::ResultIrsensorNotReady::Includes(result))
        {
            counter++;
            nn::os::SleepThread(PollingInterval);
            ASSERT_LT(counter, NotReadyWaitLoopCountMax);
        }
        else
        {
            NN_LOG("# RunImageTransferProcessor (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
            break;
        }
    }
    NNT_IRSENSOR_EXIT_SUCCESS;
}

void ModeTransitionTest::RunImageProcessor(const nn::irsensor::IrCameraHandle& handle, const nn::irsensor::HandAnalysisConfig& config) NN_NOEXCEPT
{
    RunHandAnalysis(handle, config);
    nn::TimeSpanType startTime = nn::os::GetSystemTick().ToTimeSpan();
    auto counter = 0;
    auto stateCount = 0;
    while (NN_STATIC_CONDITION(true))
    {
        nn::irsensor::HandAnalysisSilhouetteState state;
        nn::Result result = nn::irsensor::GetHandAnalysisSilhouetteState(&state, &stateCount, 1, 0, handle);
        if (nn::irsensor::ResultIrsensorNotReady::Includes(result))
        {
            counter++;
            nn::os::SleepThread(PollingInterval);
            ASSERT_LT(counter, NotReadyWaitLoopCountMax);
        }
        else
        {
            NN_LOG("# RunHandAnalysis (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
            break;
        }
    }
    NNT_IRSENSOR_EXIT_SUCCESS;
}

void ModeTransitionTest::RunImageProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    RunPointingProcessor(handle);
    nn::TimeSpanType startTime = nn::os::GetSystemTick().ToTimeSpan();
    auto counter = 0;
    auto stateCount = 0;
    while (NN_STATIC_CONDITION(true))
    {
        nn::irsensor::PointingProcessorState state;
        nn::Result result = nn::irsensor::GetPointingProcessorStates(&state, &stateCount, 1, handle);
        if (nn::irsensor::ResultIrsensorNotReady::Includes(result))
        {
            counter++;
            nn::os::SleepThread(PollingInterval);
            ASSERT_LT(counter, NotReadyWaitLoopCountMax);
        }
        else
        {
            NN_LOG("# RunPointingProcessor (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
            break;
        }
    }
    NNT_IRSENSOR_EXIT_SUCCESS;
}

void ModeTransitionTest::RunImageProcessor(const nn::irsensor::IrCameraHandle& handle, const nn::irsensor::IrLedProcessorConfig& config) NN_NOEXCEPT
{
    RunIrLedProcessor(handle, config);
    nn::TimeSpanType startTime = nn::os::GetSystemTick().ToTimeSpan();
    auto counter = 0;
    while (NN_STATIC_CONDITION(true))
    {
        nn::Result result = nn::irsensor::GetIrLedProcessorState(handle);
        if (nn::irsensor::ResultIrsensorNotReady::Includes(result))
        {
            counter++;
            nn::os::SleepThread(PollingInterval);
            ASSERT_LT(counter, NotReadyWaitLoopCountMax);
        }
        else
        {
            NN_LOG("# RunIrLedProcessor (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
            break;
        }
    }
    NNT_IRSENSOR_EXIT_SUCCESS;
}

void ModeTransitionTest::RunImageProcessor(const nn::irsensor::IrCameraHandle& handle, const nn::irsensor::AdaptiveClusteringProcessorConfig& config) NN_NOEXCEPT
{
    nn::irsensor::RunAdaptiveClusteringProcessor(handle, config);
    nn::TimeSpanType startTime = nn::os::GetSystemTick().ToTimeSpan();
    auto counter = 0;
    auto stateCount = 0;
    while (NN_STATIC_CONDITION(true))
    {
        nn::irsensor::AdaptiveClusteringProcessorState state;
        nn::Result result = nn::irsensor::GetAdaptiveClusteringProcessorStates(&state, &stateCount, 1, 0, handle);
        if (nn::irsensor::ResultIrsensorNotReady::Includes(result))
        {
            counter++;
            nn::os::SleepThread(PollingInterval);
            ASSERT_LT(counter, NotReadyWaitLoopCountMax);
        }
        else
        {
            NN_LOG("# RunAdaptiveClustering (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
            break;
        }
    }
    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// イメージプロセッサの処理の 開始/停止
//==================================================
// 実行後に停止する
void ModeTransitionTest::DoTestStaticModeTransition(const nn::irsensor::IrCameraHandle& handle, const IrCameraMode mode) NN_NOEXCEPT
{
    switch (mode)
    {
    case ::IrCameraMode_Moment:
        NNT_IRSENSOR_EXPECT_EXIT(DoTestRunMoment(handle), NNT_IRSENSOR_EXIT_0, "");
        break;
    case ::IrCameraMode_Clustering:
        NNT_IRSENSOR_EXPECT_EXIT(DoTestRunClustering(handle), NNT_IRSENSOR_EXIT_0, "");
        break;
    case ::IrCameraMode_ImageTransfer:
        NNT_IRSENSOR_EXPECT_EXIT(DoTestRunImageTransfer(handle), NNT_IRSENSOR_EXIT_0, "");
        break;
    case ::IrCameraMode_HandAnalysis:
        NNT_IRSENSOR_EXPECT_EXIT(DoTestRunHandAnalysis(handle), NNT_IRSENSOR_EXIT_0, "");
        break;
    case ::IrCameraMode_Pointing:
        NNT_IRSENSOR_EXPECT_EXIT(DoTestRunPointing(handle), NNT_IRSENSOR_EXIT_0, "");
        break;
    case ::IrCameraMode_IrLed:
        NNT_IRSENSOR_EXPECT_EXIT(DoTestRunIrLed(handle), NNT_IRSENSOR_EXIT_0, "");
        break;
    case ::IrCameraMode_AdaptiveClustering:
        NNT_IRSENSOR_EXPECT_EXIT(DoTestRunAdaptiveClustering(handle), NNT_IRSENSOR_EXIT_0, "");
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    DoTestStopImageProcessor(handle);

    NNT_IRSENSOR_EXIT_SUCCESS;
}

// 実行後に停止しない
void ModeTransitionTest::DoTestDynamicModeTransition(const nn::irsensor::IrCameraHandle& handle, const IrCameraMode mode) NN_NOEXCEPT
{
    switch (mode)
    {
    case ::IrCameraMode_Moment:
        NNT_IRSENSOR_EXPECT_EXIT(DoTestRunMoment(handle), NNT_IRSENSOR_EXIT_0, "");
        break;
    case ::IrCameraMode_Clustering:
        NNT_IRSENSOR_EXPECT_EXIT(DoTestRunClustering(handle), NNT_IRSENSOR_EXIT_0, "");
        break;
    case ::IrCameraMode_ImageTransfer:
        NNT_IRSENSOR_EXPECT_EXIT(DoTestRunImageTransfer(handle), NNT_IRSENSOR_EXIT_0, "");
        break;
    case ::IrCameraMode_HandAnalysis:
        NNT_IRSENSOR_EXPECT_EXIT(DoTestRunHandAnalysis(handle), NNT_IRSENSOR_EXIT_0, "");
        break;
    case ::IrCameraMode_Pointing:
        NNT_IRSENSOR_EXPECT_EXIT(DoTestRunPointing(handle), NNT_IRSENSOR_EXIT_0, "");
        break;
    case ::IrCameraMode_IrLed:
        NNT_IRSENSOR_EXPECT_EXIT(DoTestRunIrLed(handle), NNT_IRSENSOR_EXIT_0, "");
        break;
    case ::IrCameraMode_AdaptiveClustering:
        NNT_IRSENSOR_EXPECT_EXIT(DoTestRunAdaptiveClustering(handle), NNT_IRSENSOR_EXIT_0, "");
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    NNT_IRSENSOR_EXIT_SUCCESS;
}

// モーメント
void ModeTransitionTest::DoTestRunMoment(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    nn::irsensor::MomentProcessorConfig momentConfig;
    nn::irsensor::GetMomentProcessorDefaultConfig(&momentConfig);

    RunImageProcessor(handle, momentConfig);
    NNT_IRSENSOR_EXIT_SUCCESS;
}

// クラスタリング
void ModeTransitionTest::DoTestRunClustering(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    nn::irsensor::ClusteringProcessorConfig clusteringConfig;
    nn::irsensor::GetClusteringProcessorDefaultConfig(&clusteringConfig);

    RunImageProcessor(handle, clusteringConfig);
    NNT_IRSENSOR_EXIT_SUCCESS;
}

// イメージ転送
void ModeTransitionTest::DoTestRunImageTransfer(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    nn::irsensor::ImageTransferProcessorConfig imageTransferConfig;
    nn::irsensor::GetImageTransferProcessorDefaultConfig(&imageTransferConfig);

    RunImageProcessor(handle, imageTransferConfig);
    NNT_IRSENSOR_EXIT_SUCCESS;
}

// ハンドアナリシス
void ModeTransitionTest::DoTestRunHandAnalysis(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    nn::irsensor::HandAnalysisConfig handAnalysisConfig;
    handAnalysisConfig.mode = nn::irsensor::HandAnalysisMode_Silhouette;

    RunImageProcessor(handle, handAnalysisConfig);
    NNT_IRSENSOR_EXIT_SUCCESS;
}

// ポインティング
void ModeTransitionTest::DoTestRunPointing(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    RunImageProcessor(handle);
    NNT_IRSENSOR_EXIT_SUCCESS;
}

// IRLed
void ModeTransitionTest::DoTestRunIrLed(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    nn::irsensor::IrLedProcessorConfig irLedConfig;
    nn::irsensor::GetIrLedProcessorDefaultConfig(&irLedConfig);

    RunImageProcessor(handle, irLedConfig);
    NNT_IRSENSOR_EXIT_SUCCESS;
}

// AdaptiveClustering
void ModeTransitionTest::DoTestRunAdaptiveClustering(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    nn::irsensor::AdaptiveClusteringProcessorConfig adaptiveClusteringConfig;
    adaptiveClusteringConfig.mode = nn::irsensor::AdaptiveClusteringMode_DynamicFov;

    RunImageProcessor(handle, adaptiveClusteringConfig);
    NNT_IRSENSOR_EXIT_SUCCESS;
}

// 停止処理
void ModeTransitionTest::DoTestStopImageProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    nn::irsensor::StopImageProcessorAsync(handle);

    nn::TimeSpanType startTime = nn::os::GetSystemTick().ToTimeSpan();
    auto counter = 0;
    while (NN_STATIC_CONDITION(true))
    {
        auto status = nn::irsensor::GetImageProcessorStatus(handle);

        if (status == nn::irsensor::ImageProcessorStatus_Running)
        {
            counter++;
            nn::os::SleepThread(PollingInterval);
            ASSERT_LT(counter, NotReadyWaitLoopCountMax);
        }
        else
        {
            NN_LOG("# StopImageProcessor (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
            break;
        }
    }
    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// 状態遷移のテスト
//==================================================

// プロセッサの停止を伴う静的な状態遷移のテスト
TEST_F(ModeTransitionTest, ModeTransitionTestStatically)
{
    nnt::gamepad::Initialize();

    // コントローラの再接続
    nnt::gamepad::DisconnectAll();
    nnt::gamepad::ConnectAll();

    NN_LOG("#\n# Static Transition.\n#\n");
    for (const auto& handle : nnt::irsensor::IrSensorTest::s_Handles)
    {
        // IRカメラが利用できない場合はスキップ
        if (GetIrCameraStatus(handle) != nn::irsensor::IrCameraStatus_Available)
        {
            continue;
        }

        ::TestCase<IrCameraMode, IrCameraModeCount> test;

        NN_LOG("#-----------------------------------------------\n");
        NN_LOG("# IrCameraHandle : 0x%x\n", handle);
        NN_LOG("#-----------------------------------------------\n");
        IrCameraMode mode;
        test.PrintTestPass();
        while (test.GetNextTest(&mode))
        {
            DoTestStaticModeTransition(handle, mode);
        }
    }
    nnt::gamepad::DisconnectAll();
}

// プロセッサの停止を伴わない動的な状態遷移のテスト
TEST_F(ModeTransitionTest, ModeTransitionTestDynamically)
{
    nnt::gamepad::Initialize();

    // コントローラの再接続
    nnt::gamepad::DisconnectAll();
    nnt::gamepad::ConnectAll();

    NN_LOG("#\n# Dynamic Transition.\n#\n");
    for (const auto& handle : nnt::irsensor::IrSensorTest::s_Handles)
    {
        // IRカメラが利用できない場合はスキップ
        if (GetIrCameraStatus(handle) != nn::irsensor::IrCameraStatus_Available)
        {
            continue;
        }

        ::TestCase<IrCameraMode, IrCameraModeCount> test;

        NN_LOG("#-----------------------------------------------\n");
        NN_LOG("# IrCameraHandle : 0x%x\n", handle);
        NN_LOG("#-----------------------------------------------\n");
        IrCameraMode mode;
        test.PrintTestPass();
        while (test.GetNextTest(&mode))
        {
            DoTestDynamicModeTransition(handle, mode);
        }

        DoTestStopImageProcessor(handle);
    }
    nnt::gamepad::DisconnectAll();
}

} // namespace
