﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_NpadJoyCommon.h>
#include <nn/hid/hid_NpadSixAxisSensor.h>
#include <nn/hid/hid_ConsoleSixAxisSensor.h>
#include <nn/hid/tmp/hid_ConsoleSixAxisSensor.h>

#include <nn/os.h>
#include <nn/util/util_Vector.h>

#include <nnt/nntest.h>

#include "../Common/testGamePad_Common.h"
#include "../Common/testGamePad_ConsoleSixAxisSensor.h"

namespace
{

//!< テスト因子 (ref. http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=230884778)
struct TestFactor
{
    ::nn::hid::NpadHandheldActivationMode mode;
    bool handheldSampling;
    bool consoleSampling;
    bool expectedHandheldSamplingState;

    void show() const
    {
        NN_LOG("--------\n");
        NN_LOG("handheldSampling = %x, consoleSampling = %x, expectedHandheldSamplingState = %x\n"
            , handheldSampling
            , consoleSampling
            , expectedHandheldSamplingState
        );
        NN_LOG("--------\n");
    }
};


bool IsSensorSampled(const ::nn::hid::SixAxisSensorHandle& handle)
{
    // 入力検証
    ::nn::hid::SixAxisSensorState state = {};

    ::nn::hid::GetSixAxisSensorState(&state, handle);
    const auto SamplingNumber = state.samplingNumber;

    // 1 周期待つ
    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nnt::gamepad::GetConsoleSixAxisSensorSamplingInterval().GetMilliSeconds() * 5
        )
    );

    ::nn::hid::GetSixAxisSensorState(&state, handle);
    ::nnt::gamepad::PrintConsoleSixAxisSensorState(state);

    return (SamplingNumber != state.samplingNumber);
}

void VerifySamplingState(::nn::hid::ConsoleSixAxisSensorHandle& consoleHandle,
                         ::nn::hid::SixAxisSensorHandle& handheldHandle,
                         const TestFactor& factor) NN_NOEXCEPT
{
    // テスト環境の設定
    if (factor.consoleSampling)
    {
        ::nn::hid::StartSixAxisSensor(consoleHandle);
    }
    else
    {
        ::nn::hid::StopSixAxisSensor(consoleHandle);
    }

    if (factor.handheldSampling)
    {
        ::nn::hid::StartSixAxisSensor(handheldHandle);
    }
    else
    {
        ::nn::hid::StopSixAxisSensor(handheldHandle);
    }

    ::nn::hid::SetNpadHandheldActivationMode(factor.mode);

    EXPECT_EQ(factor.expectedHandheldSamplingState, IsSensorSampled(handheldHandle));
}

} // namespace


//!< 6軸センサーの入力状態が正しく取得できるか
TEST(ConsoleSixAxisSensor, VerifySamplingState)
{
    ::nnt::gamepad::Initialize();

    ::nnt::gamepad::ConnectAll();

    ::nn::hid::InitializeConsoleSixAxisSensor();

    ::nn::hid::ConsoleSixAxisSensorHandle consoleHandle;
    ::nn::hid::GetSixAxisSensorHandle(&consoleHandle);

    ::nn::hid::SixAxisSensorHandle handheldHandle;
    ::nn::hid::GetSixAxisSensorHandles(
        &handheldHandle,
        1,
        ::nn::hid::NpadId::Handheld,
        ::nn::hid::NpadStyleHandheld::Mask
    );

    // テスト因子の設定
    const TestFactor Factors[] = {
        // NpadHandheldActivationMode, handheldSampling, consoleSampling, expectedHandheldSamplingState
        { ::nn::hid::NpadHandheldActivationMode_None,   true,  true,  true  },
    };

    for (const auto& factor : Factors)
    {
        factor.show();
        VerifySamplingState(consoleHandle, handheldHandle, factor);
    }
}
