﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nnd/ftm/ftm.h>

namespace
{

nn::Result ReadFirmwareFile(char* buffer, void* pParameter, int64_t offset, size_t size) NN_NOEXCEPT
{
    ::nn::fs::FileHandle* pFileHandle = reinterpret_cast<::nn::fs::FileHandle*>(pParameter);
    NN_RESULT_DO(nn::fs::ReadFile(*pFileHandle, offset, buffer, size));
    NN_RESULT_SUCCESS;
}

} // namespace

extern "C" void nnMain()
{
    NN_LOG("/// FTM Fw Updater ///\n");

    std::string mountName = "sd";
    std::string fileName  = "NTD_4CD.fts256";
    std::string path = mountName + ":/" + fileName;

    nn::Result result = nn::fs::MountSdCardForDebug(mountName.c_str());
    NN_ASSERT(result.IsSuccess(), "Failed to mount sd");

    nn::fs::FileHandle fileHandle;
    result = nn::fs::OpenFile(&fileHandle, path.c_str(), nn::fs::OpenMode_Read);
    NN_ASSERT(result.IsSuccess(), "Failed to open file");

    int64_t fileSize;
    result = nn::fs::GetFileSize(&fileSize, fileHandle);
    NN_ASSERT(result.IsSuccess(), "Failed to get file size");

    nnd::ftm::Initialize();

    // バージョン取得前にリセットが必要
    // ファームウェアが壊れている場合は失敗する
    result = nnd::ftm::ResetDevice();
    if (result.IsFailure())
    {
        NN_LOG("Firmware may be corrupted\n");
    }

    // アップデート前のバージョンを取得
    nnd::ftm::FirmwareVersion version;
    result = nnd::ftm::ReadFirmwareVersion(&version);
    NN_ASSERT(result.IsSuccess(), "Failed to read version");
    NN_LOG("Current Version = 0x%08X\n", version.release);

    // アップデートを実施
    NN_LOG("Update Firmware ...\n");
    nn::os::Tick tick = nn::os::GetSystemTick();
    result = nnd::ftm::UpdateFirmware(ReadFirmwareFile, &fileHandle, static_cast<size_t>(fileSize));
    NN_ASSERT(result.IsSuccess(), "Failed to update firmware");
    NN_LOG("Update Time = %d ms\n", ConvertToTimeSpan(nn::os::GetSystemTick() - tick).GetMilliSeconds());

    // アップデート後のバージョンを取得
    result = nnd::ftm::ReadFirmwareVersion(&version);
    NN_ASSERT(result.IsSuccess(), "Failed to read version");
    NN_LOG("Current Version = 0x%08X\n", version.release);

    nnd::ftm::Finalize();

    nn::fs::CloseFile(fileHandle);

    nn::fs::Unmount(mountName.c_str());

    NN_LOG("[[ PASSED !! ]]\n");
}
