﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <mutex>
#include <cstring>
#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nnd/ftm/ftm.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>

namespace
{
    nn::os::ThreadType    g_ThreadRead;
    nn::os::ThreadType    g_ThreadParse;
    const size_t          threadStackSize = 8192;
    NN_ALIGNAS(4096) char g_ThreadStackRead[threadStackSize];
    NN_ALIGNAS(4096) char g_ThreadStackParse[threadStackSize];

    class TouchBuffer
    {
    public:
        NN_IMPLICIT TouchBuffer(size_t size) : BufferSize(size), wp(0), rp(0), dataSize(0), mutex(false)
        {
            buffer = new char[BufferSize];
        }

        ~TouchBuffer()
        {
            delete[] buffer;
        }

        size_t Write(const char* pBuffer, size_t size)
        {
            std::lock_guard<nn::os::Mutex> lock(mutex);
            size_t writeSize = (size < BufferSize) ? size : BufferSize;
            size_t remain = BufferSize - wp;
            if (writeSize > remain)
            {
                std::memcpy(&buffer[wp], pBuffer, remain);
                std::memcpy(&buffer[0], pBuffer + remain, writeSize - remain);
            }
            else
            {
                std::memcpy(&buffer[wp], pBuffer, writeSize);
            }

            dataSize = (dataSize + writeSize) > BufferSize ? BufferSize : (dataSize + writeSize);
            wp = (wp + writeSize) % BufferSize;

            if((rp < wp) && (wp - rp != dataSize))
            {
                rp = wp;
                dataSize = BufferSize;
            }

            return writeSize;
        }

        size_t Read(char* pOutBuffer, size_t size)
        {
            std::lock_guard<nn::os::Mutex> lock(mutex);
            size_t readSize = (dataSize < size) ? dataSize: size;
            size_t remain = BufferSize - rp;

            if (readSize > remain)
            {
                std::memcpy(pOutBuffer, &buffer[rp], remain);
                std::memcpy(pOutBuffer + remain, &buffer[0], readSize - remain);
            }
            else
            {
                std::memcpy(pOutBuffer, &buffer[rp], readSize);
            }

            dataSize -= readSize;
            rp = (rp + readSize) % BufferSize;

            return readSize;
        }

        size_t GetDataSize()
        {
            std::lock_guard<nn::os::Mutex> lock(mutex);
            return dataSize;
        }

    private:
        const size_t    BufferSize;
        char*           buffer;
        uint32_t        wp;
        uint32_t        rp;
        size_t          dataSize;
        nn::os::Mutex   mutex;
    };

    nn::Result Initialize()
    {
        nnd::ftm::Initialize();
        NN_RESULT_DO(nnd::ftm::ResetDevice());
        NN_RESULT_DO(nnd::ftm::SleepOutDevice());
        NN_RESULT_DO(nnd::ftm::ActivateSensing());
        NN_LOG("FTM is Ready !\n");

        NN_RESULT_SUCCESS;
    }

    void ThreadEntryReadEvent(void *arg)
    {
        TouchBuffer* touchBuffer = reinterpret_cast<TouchBuffer*>(arg);

        nn::os::SystemEventType event;
        nnd::ftm::BindInterrupt(&event);
        nnd::ftm::SetInterruptEnable(true);

        const size_t FifoSizeMax = nnd::ftm::FtmMaxEventReportCount * nnd::ftm::FtmMaxEventReportByteSize;
        char readData[FifoSizeMax];
        int leftCount;
        int readCount;
        int totalCount;
        bool isOverflow;

        for ( ; ; )
        {
            for ( ; ; )
            {
                nnd::ftm::ReadEventReports(readData, &leftCount, &readCount, &isOverflow, 1);
                totalCount = readCount;

                if (leftCount != 0)
                {
                    nnd::ftm::ReadEventReports(&readData[nnd::ftm::GetEventReportByteSize()], &leftCount, &readCount, &isOverflow, leftCount);
                    totalCount += readCount;
                }

                touchBuffer->Write(readData, totalCount * nnd::ftm::GetEventReportByteSize());

                if (leftCount == 0)
                {
                    break;
                }
            }

            // 次のイベント発生まで待つ
            nn::os::WaitSystemEvent(&event);
            nn::os::ClearSystemEvent(&event);

            // イベント発生後に割り込み禁止となるため再び有効にする
            nnd::ftm::SetInterruptEnable(true);
        }
    }

    void ThreadEntryParseEvent(void *arg)
    {
        TouchBuffer* touchBuffer = reinterpret_cast<TouchBuffer*>(arg);

        const size_t FifoSizeMax = nnd::ftm::FtmMaxEventReportCount * nnd::ftm::FtmMaxEventReportByteSize;
        char readData[FifoSizeMax];
        size_t remain;

        for ( ; ; )
        {
            nn::os::YieldThread();
            remain = touchBuffer->GetDataSize();

            // バッファにデータがたまっていれば解析を始める
            if (remain != 0)
            {
                size_t readSize = (remain < FifoSizeMax) ? remain : FifoSizeMax;
                readSize = touchBuffer->Read(readData, readSize);

                uint32_t totalEventCount = readSize / nnd::ftm::GetEventReportByteSize();
                nnd::ftm::EventReport report[nnd::ftm::FtmMaxEventReportCount];
                nnd::ftm::ParseEventReports(report, readData, totalEventCount);

                for (uint32_t event = 0; event < totalEventCount; event++)
                {
                    nn::os::YieldThread();
                    if (report[event].eventId == nnd::ftm::EventId::NoEvents)
                    {
                        continue;
                    }

                    switch (report[event].eventId)
                    {
                    case nnd::ftm::EventId::TouchEnter:
                        NN_FALL_THROUGH;

                    case nnd::ftm::EventId::TouchLeave:
                        NN_FALL_THROUGH;

                    case nnd::ftm::EventId::TouchMotion:
                    {
                        nnd::ftm::TouchEventReport* pTouchReport = &(report[event].content.touchReport);
                        NN_LOG("EID: %d TID: %d, (x, y, major, minor) = (%4d, %3d, %4d, %4d), milliDeg.: %+6d",
                               report[event].eventId, pTouchReport->touchId, pTouchReport->x, pTouchReport->y,
                               pTouchReport->major, pTouchReport->minor, pTouchReport->milliDegree);
                        NN_LOG(" [");
                        for(size_t i = 0; i < nnd::ftm::GetEventReportByteSize(); i++)
                        {
                            NN_LOG("%02X",readData[event * nnd::ftm::GetEventReportByteSize() + i]);
                        }
                        NN_LOG("]\n");
                    }
                    break;

                    case nnd::ftm::EventId::Status:
                    {
                        switch (report[event].content.statusReport)
                        {
                            case nnd::ftm::StatusType::AutoTuneMutual:
                                NN_LOG("StatusType: AutoTuneMutual\n");
                                break;

                            case nnd::ftm::StatusType::AutoTuneSelf:
                                NN_LOG("StatusType: AutoTuneSelf\n");
                                break;

                            case nnd::ftm::StatusType::FlashWriteConfig:
                                NN_LOG("StatusType: FlashWriteConfig\n");
                                break;

                            case nnd::ftm::StatusType::FlashWriteNodeCompensationMemory:
                                NN_LOG("StatusType: FlashWriteNodeCompensationMemory\n");
                                break;

                            case nnd::ftm::StatusType::ForceCalSelfAndMutual:
                                NN_LOG("StatusType: ForceCalSelfAndMutual\n");
                                break;

                            case nnd::ftm::StatusType::ForceCalSelfOnly:
                                NN_LOG("StatusType: ForceCalSelfOnly\n");
                                break;

                            case nnd::ftm::StatusType::Others:
                                NN_LOG("StatusType: Others\n");
                                break;

                            default:
                                NN_LOG("StatusType = %d\n", report[event].content.statusReport);
                                NN_UNEXPECTED_DEFAULT;
                        }
                        break;
                    }
                    case nnd::ftm::EventId::Inform:
                    {
                        nnd::ftm::GpioInformEventReport* pGpioInform = &report[event].content.gpioReport;
                        NN_LOG("(GPIO0, GPIO1, GPIO2) = (%d, %d, %d)\n",
                               pGpioInform->gpio0, pGpioInform->gpio1, pGpioInform->gpio2);
                    }
                    break;

                    case nnd::ftm::EventId::NoEvents:
                        NN_FALL_THROUGH;

                    case nnd::ftm::EventId::Error:
                        NN_FALL_THROUGH;

                    case nnd::ftm::EventId::ControllerReady:
                        NN_FALL_THROUGH;

                    case nnd::ftm::EventId::SleepOutContollerReady:
                        NN_FALL_THROUGH;

                    case nnd::ftm::EventId::Others:
                        NN_LOG("EventID = %d\n", report[event].eventId);
                    break;

                    default:
                        NN_LOG("EventID = %d\n", report[event].eventId);
                        NN_UNEXPECTED_DEFAULT;
                    }
                }
            }
        }
    }
}

extern "C" void nnMain()
{
    NN_LOG("/// FTM Event Monitor ///\n");

    nn::Result result = Initialize();
    if (result.IsFailure())
    {
        NN_LOG("Module:%d Description:%d Inner:%d\n", result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
        return;
    }

    size_t bufferSize = 32 * nnd::ftm::FtmMaxEventReportCount * nnd::ftm::FtmMaxEventReportByteSize;
    TouchBuffer *touchBuffer = new TouchBuffer(bufferSize);

    result = nn::os::CreateThread(&g_ThreadRead, ThreadEntryReadEvent, touchBuffer,
                                  g_ThreadStackRead, threadStackSize, nn::os::DefaultThreadPriority);
    NN_ASSERT(result.IsSuccess(), "Failed to Create Thread for Read");

    result = nn::os::CreateThread(&g_ThreadParse, ThreadEntryParseEvent, touchBuffer,
                                  g_ThreadStackParse, threadStackSize, nn::os::DefaultThreadPriority);
    NN_ASSERT(result.IsSuccess(), "Failed to Create Thread for Parse");

    nn::os::StartThread(&g_ThreadRead);
    nn::os::StartThread(&g_ThreadParse);

    nn::os::DestroyThread(&g_ThreadRead);
    nn::os::DestroyThread(&g_ThreadParse);

    delete touchBuffer;

    nnd::ftm::ResetDevice();
    nnd::ftm::Finalize();
}
