﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fssystem/fs_ZeroBitmapFile.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>


namespace {

    class TestZeroBitmapFile : public ::testing::Test
    {
    public:
        TestZeroBitmapFile() NN_NOEXCEPT
          : m_pZeroBitmapFile(nullptr),
            m_Mt(nnt::fs::util::GetRandomSeed())
        {
        }

        virtual ~TestZeroBitmapFile() NN_NOEXCEPT NN_OVERRIDE {}
        void CreateTestData(bool bCleanBitmap, bool bIncompleteSize) NN_NOEXCEPT;
        void CheckZeroFillRead() NN_NOEXCEPT;
        void CheckFractionWrite() NN_NOEXCEPT;
        void CheckAppendWrite() NN_NOEXCEPT;
        void CheckIncompleteTailWrite() NN_NOEXCEPT;

    protected:
        static const int32_t SaveDataBlockSize = nn::fs::detail::DefaultSaveDataBlockSize;
#if defined(NN_BUILD_CONFIG_OS_WIN32) || defined(NN_BUILD_CONFIG_CPU_CORTEX_A57_AARCH32)
        static const int64_t testBufferSize = 128 * 1024 * 1024;
#else
        static const int64_t testBufferSize = 512 * 1024 * 1024;
#endif
        static const int64_t bitmapSize = testBufferSize / SaveDataBlockSize / 8;

    protected:
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            if(m_pZeroBitmapFile) delete m_pZeroBitmapFile;
        }

    private:
        static char g_Buffer[testBufferSize];
        static nn::Bit8 g_Bitmap[bitmapSize];
        nnt::fs::util::MemoryFile m_File;
        nn::fssystem::ZeroBitmapFile* m_pZeroBitmapFile;
        int64_t m_TestFileSize;
        std::mt19937 m_Mt;
    };

    char TestZeroBitmapFile::g_Buffer[TestZeroBitmapFile::testBufferSize];
    nn::Bit8 TestZeroBitmapFile::g_Bitmap[TestZeroBitmapFile::bitmapSize];

    void TestZeroBitmapFile::CreateTestData(bool bCleanBitmap, bool bIncompleteSize) NN_NOEXCEPT
    {
        if(m_pZeroBitmapFile) delete m_pZeroBitmapFile;

        m_TestFileSize = testBufferSize - (bIncompleteSize ? std::uniform_int_distribution<int64_t>(1, SaveDataBlockSize - 1)(m_Mt) : 0);

        nnt::fs::util::FillBufferWithRandomValue(g_Buffer, static_cast<size_t>(m_TestFileSize));
        if(bCleanBitmap)
        {
            memset(g_Bitmap, 0xff, bitmapSize);
        }
        else
        {
            nnt::fs::util::FillBufferWithRandomValue(g_Bitmap, static_cast<size_t>(bitmapSize));
        }

        m_File.SetSize(m_TestFileSize);
        m_File.Write(0, g_Buffer, static_cast<size_t>(m_TestFileSize), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
        std::unique_ptr<nnt::fs::util::MemoryFile> pMemFile(new nnt::fs::util::MemoryFile());
        pMemFile->SetSize(m_TestFileSize);
        pMemFile->Write(0, g_Buffer, static_cast<size_t>(m_TestFileSize), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
        std::unique_ptr<nn::Bit8[], nn::fs::detail::Deleter> pBitmap = nn::fs::detail::MakeUnique<nn::Bit8[]>(bitmapSize);
        memcpy(pBitmap.get(), g_Bitmap, bitmapSize);

        m_pZeroBitmapFile = new nn::fssystem::ZeroBitmapFile();
        NNT_ASSERT_RESULT_SUCCESS(m_pZeroBitmapFile->Initialize(std::move(pMemFile), std::move(pBitmap), bitmapSize, SaveDataBlockSize, static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend)));
    }

    void TestZeroBitmapFile::CheckZeroFillRead() NN_NOEXCEPT
    {
        CreateTestData(false, false);

        char* pZeroFileBuffer = g_Buffer;
        char* pMemFileBuffer = &g_Buffer[SaveDataBlockSize * 8];
        for(int i = 0; i < bitmapSize; ++i)
        {
            size_t size;
            int64_t offset = i * SaveDataBlockSize * 8;
            NNT_ASSERT_RESULT_SUCCESS(m_pZeroBitmapFile->Read(&size, offset, pZeroFileBuffer, SaveDataBlockSize * 8, nn::fs::ReadOption::MakeValue(0)));
            NNT_ASSERT_RESULT_SUCCESS(m_File.Read(&size, offset, pMemFileBuffer, SaveDataBlockSize * 8, nn::fs::ReadOption::MakeValue(0)));

            for(int bit = 0; bit < 8; ++bit)
            {
                // ビットが立ってたら未使用領域
                if(g_Bitmap[i] & (1 << bit))
                {
                    bool bZeroCheck = true;
                    for(int j = 0; j < SaveDataBlockSize; ++j)
                    {
                        if(pZeroFileBuffer[bit * SaveDataBlockSize + j] != 0)
                        {
                            bZeroCheck = false;
                            break;
                        }
                    }
                    ASSERT_TRUE(bZeroCheck);
                }
                else
                {
                    bool bCompareCheck = true;
                    for(int j = 0; j < SaveDataBlockSize; ++j)
                    {
                        if(pZeroFileBuffer[bit * SaveDataBlockSize + j] != pMemFileBuffer[bit * SaveDataBlockSize + j])
                        {
                            bCompareCheck = false;
                            break;
                        }
                    }
                    ASSERT_TRUE(bCompareCheck);
                }
            }
        }
    }

    void TestZeroBitmapFile::CheckFractionWrite() NN_NOEXCEPT
    {
        CreateTestData(true, false);

        char* pWriteBuffer = g_Buffer;
        char* pReadBuffer = &g_Buffer[SaveDataBlockSize * 2];

        int64_t start = 0;
        while(start < m_TestFileSize)
        {
            auto offset = std::uniform_int_distribution<int64_t>(0, SaveDataBlockSize)(m_Mt);
            auto writeSize = SaveDataBlockSize + std::uniform_int_distribution<int64_t>(0, SaveDataBlockSize)(m_Mt);
            if(start + offset + writeSize > m_TestFileSize)
            {
                writeSize = m_TestFileSize - (start + offset);
            }
            nnt::fs::util::FillBufferWithRandomValue(pWriteBuffer, static_cast<size_t>(writeSize));

            NNT_ASSERT_RESULT_SUCCESS(m_pZeroBitmapFile->Write(start + offset, pWriteBuffer, static_cast<size_t>(writeSize), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
            int64_t readSize = nn::util::align_up(start + offset + writeSize, SaveDataBlockSize) - start;
            size_t size;
            NNT_ASSERT_RESULT_SUCCESS(m_pZeroBitmapFile->Read(&size, start, pReadBuffer, static_cast<size_t>(readSize), nn::fs::ReadOption::MakeValue(0)));

            // 前方余白チェック
            bool bZeroCheck = true;
            for(int i = 0; i < offset; ++i)
            {
                if(pReadBuffer[i] != 0)
                {
                    bZeroCheck = false;
                    break;
                }
            }
            ASSERT_TRUE(bZeroCheck);

            // 書き込みデータチェック
            bool bCompareCheck = true;
            for(int i = 0; i < writeSize; ++i)
            {
                if(pReadBuffer[offset + i] != pWriteBuffer[i])
                {
                    bCompareCheck = false;
                    break;
                }
            }
            ASSERT_TRUE(bCompareCheck);

            // 後方余白チェック
            bZeroCheck = true;
            for(int i = 0; i < (readSize - offset - writeSize); ++i)
            {
                if(pReadBuffer[offset + writeSize + i] != 0)
                {
                    bZeroCheck = false;
                    break;
                }
            }
            ASSERT_TRUE(bZeroCheck);

            start += readSize;
        }

    }

    void TestZeroBitmapFile::CheckAppendWrite() NN_NOEXCEPT
    {
        CreateTestData(true, false);

        char* pWriteBuffer = g_Buffer;

        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultUsableSpaceNotEnough, m_pZeroBitmapFile->Write(m_TestFileSize - SaveDataBlockSize, pWriteBuffer, SaveDataBlockSize * 2, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));

        // ファイル伸長チェック
        int64_t fileSize;
        m_pZeroBitmapFile->GetSize(&fileSize);
        ASSERT_TRUE(fileSize == m_TestFileSize);
    }

    void TestZeroBitmapFile::CheckIncompleteTailWrite() NN_NOEXCEPT
    {
        CreateTestData(true, true);

        char* pWriteBuffer = g_Buffer;
        char* pReadBuffer = &g_Buffer[SaveDataBlockSize * 2];

        int64_t offset = nn::util::align_down(m_TestFileSize, SaveDataBlockSize);
        int64_t writeSize = std::uniform_int_distribution<int64_t>(1, m_TestFileSize - offset - 1)(m_Mt);
        nnt::fs::util::FillBufferWithRandomValue(pWriteBuffer, static_cast<size_t>(writeSize));
        NNT_ASSERT_RESULT_SUCCESS(m_pZeroBitmapFile->Write(offset, pWriteBuffer, static_cast<size_t>(writeSize), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));

        // ファイル伸長チェック
        int64_t fileSize;
        m_pZeroBitmapFile->GetSize(&fileSize);
        ASSERT_TRUE(fileSize == m_TestFileSize);

        size_t readSize;
        NNT_ASSERT_RESULT_SUCCESS(m_pZeroBitmapFile->Read(&readSize, offset, pReadBuffer, static_cast<size_t>(m_TestFileSize - offset), nn::fs::ReadOption::MakeValue(0)));

        // 書き込みデータチェック
        bool bCompareCheck = true;
        for(int i = 0; i < writeSize; ++i)
        {
            if(pReadBuffer[i] != pWriteBuffer[i])
            {
                bCompareCheck = false;
                break;
            }
        }
        ASSERT_TRUE(bCompareCheck);

        // 後方余白チェック
        bool bZeroCheck = true;
        for(int i = 0; i < (m_TestFileSize - writeSize - offset); ++i)
        {
            if(pReadBuffer[writeSize + i] != 0)
            {
                bZeroCheck = false;
                break;
            }
        }
        ASSERT_TRUE(bZeroCheck);
    }

}

// ビットマップに従ってゼロフィルされたデータが読み出せるかのテスト
TEST_F(TestZeroBitmapFile, ZeroFillRead)
{
    CheckZeroFillRead();
}

// 半端なサイズの書き込みでブロック内の余白がゼロフィルされるかのテスト
TEST_F(TestZeroBitmapFile, FractionWrite)
{
    CheckFractionWrite();
}

// 追記でファイルの伸長が起こらない事のテスト
TEST_F(TestZeroBitmapFile, AppendWrite)
{
    CheckAppendWrite();
}

// 半端なサイズのファイルへの終端ゼロフィルのテスト
TEST_F(TestZeroBitmapFile, IncompleteTailWrite)
{
    CheckIncompleteTailWrite();
}

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetEnabledAutoAbort(false);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
