﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <cstring>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fssystem/fs_AlignmentMatchingStorage.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Compression.h>
#include <nn/util/util_Decompression.h>
#include <nn/util/util_BitUtil.h>

#include "fs_HierarchicalSha256Storage.h"

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

#include "testFs_Unit_StorageLayerTestCase.h"

using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nn::fssystem;
using namespace nnt::fs::util;

using namespace nn::crypto;


class HierarchicalSha256StorageTestBase : public ::testing::Test
{
protected:

    static const int LayerCount = 3;

public:
    void SetUpStorage(IStorage* pTargetStorage, int hashTargetBlockSize) NN_NOEXCEPT
    {
        int64_t storageSize;
        NNT_ASSERT_RESULT_SUCCESS(pTargetStorage->GetSize(&storageSize));

        int hashCount = static_cast<int>(nn::util::align_up(storageSize, hashTargetBlockSize) / hashTargetBlockSize);
        size_t hashStorageSize = hashCount * Sha256Generator::HashSize;

        std::unique_ptr<char[]> tmpBuffer(new char[hashTargetBlockSize]);

        // 1段目ハッシュ計算
        m_HashStorageBuffer.reset(new char[hashStorageSize]);

        const int dataHashSizeRatio = hashTargetBlockSize / Sha256Generator::HashSize;
        int64_t restSize = storageSize;
        int64_t offset = 0;
        while( restSize > 0 )
        {
            size_t processSize = static_cast<size_t>(std::min<int64_t>(restSize, hashTargetBlockSize));
            char hash[Sha256Generator::HashSize];

            NNT_ASSERT_RESULT_SUCCESS(pTargetStorage->Read(offset, tmpBuffer.get(), processSize));
            GenerateSha256Hash(hash, sizeof(hash), tmpBuffer.get(), processSize);
            memcpy(&m_HashStorageBuffer.get()[offset / dataHashSizeRatio], hash, sizeof(hash));

            offset += processSize;
            restSize -= processSize;
        }

        // マスターハッシュ計算
        GenerateSha256Hash(m_MasterHash, Sha256Generator::HashSize, m_HashStorageBuffer.get(), hashStorageSize);

        // 各 Storage 化
        m_pHashMemoryStorage.reset(new MemoryStorage(m_HashStorageBuffer.get(), hashStorageSize));
        m_pMasterHashMemoryStorage.reset(new MemoryStorage(m_MasterHash, sizeof(m_MasterHash)));

        m_HashBuffer.reset(new char[hashStorageSize]);

        // HierarchicalSha256Storage 化
        m_pShaStorage.reset(new HierarchicalSha256Storage());
        IStorage* pStorages[3] =
        {
            m_pMasterHashMemoryStorage.get(),
            m_pHashMemoryStorage.get(),
            pTargetStorage
        };
        NNT_ASSERT_RESULT_SUCCESS(m_pShaStorage->Initialize(pStorages, LayerCount, hashTargetBlockSize, m_HashBuffer.get(), hashTargetBlockSize));

        m_pStorage.reset(new AlignmentMatchingStoragePooledBuffer<1>(m_pShaStorage.get(), hashTargetBlockSize));
    }

    nn::fs::IStorage* GetStorage() NN_NOEXCEPT
    {
        return m_pStorage.get();
    }

private:
    char m_MasterHash[HierarchicalSha256Storage::HashSize];
    std::unique_ptr<IStorage> m_pMasterHashMemoryStorage;

    std::unique_ptr<char[]> m_HashStorageBuffer;
    std::unique_ptr<IStorage> m_pHashMemoryStorage;

    std::unique_ptr<HierarchicalSha256Storage> m_pShaStorage;

    std::unique_ptr<IStorage> m_pStorage;

    std::unique_ptr<char[]> m_HashBuffer;
};

class HierarchicalSha256StorageTest : public HierarchicalSha256StorageTestBase
{
protected:
    static const int HashTargetBlockSize = 512 * 1024;
    static const int StorageSize = 16 * 1024 * 1024;

protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        SetUpMemoryStorage(StorageSize);
    }

    void SetUpMemoryStorage(int64_t storageSize) NN_NOEXCEPT
    {
        size_t memoryStorageSize = static_cast<size_t>(storageSize);

        // 2段目(データ領域)
        m_pDataMemoryStorage.reset(new AccessCountedMemoryStorage);
        GetMemoryStorage()->Initialize(memoryStorageSize);
        FillBufferWith32BitCount(GetMemoryStorage()->GetBuffer(), memoryStorageSize, 0);

        SetUpStorage(m_pDataMemoryStorage.get(), HashTargetBlockSize);
    }

    nnt::fs::util::AccessCountedMemoryStorage* GetMemoryStorage() NN_NOEXCEPT
    {
        return m_pDataMemoryStorage.get();
    }

private:
    std::unique_ptr<AccessCountedMemoryStorage> m_pDataMemoryStorage;
};

class HierarchicalSha256StorageTestWithHostFile : public HierarchicalSha256StorageTestBase, public nnt::fs::util::PrepareWorkDirFixture
{
protected:
    static const int HashTargetBlockSize = 512 * 1024;

protected:
    void SetUpFileStorage() NN_NOEXCEPT
    {
        SetUpFileStorage(static_cast<int64_t>(8) * 1024 * 1024 * 1024, true);
    }

    void SetUpFileStorage(int64_t fileSize, bool isFillEnabled) NN_NOEXCEPT
    {
        CreateWorkRootPath();
        m_DirectoryPath = GetWorkRootPath();

        nn::fs::CreateFile((m_DirectoryPath + "/file").c_str(), fileSize);

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&m_File, (m_DirectoryPath + "/file").c_str(), nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));

        m_pDataFileHandleStorage.reset(new FileHandleStorage(m_File)); // todo: close

        if( isFillEnabled )
        {
            const int tmpBufferSize = 1024 * 1024;
            std::unique_ptr<char[]> tmpBuffer(new char[tmpBufferSize]);
            for( int64_t i = 0; i < 8 * 1024; i++ )
            {
                FillBufferWith32BitCount(tmpBuffer.get(), tmpBufferSize, i * tmpBufferSize);
                NNT_ASSERT_RESULT_SUCCESS(m_pDataFileHandleStorage->Write(i * tmpBufferSize, tmpBuffer.get(), tmpBufferSize));
            }
        }

        SetUpStorage(m_pDataFileHandleStorage.get(), HashTargetBlockSize);
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(m_File));
        nn::fs::CloseFile(m_File);
        DeleteWorkRootPath();
    }

private:
    nnt::fs::util::String m_DirectoryPath;
    nn::fs::FileHandle m_File;
    std::unique_ptr<IStorage> m_pDataFileHandleStorage;
};

class HierarchicalSha256StorageTestWithVirtualMemoryStorage : public HierarchicalSha256StorageTestBase
{
protected:
    static const int HashTargetBlockSize = 2 * 1024 * 1024;
    static const int BufferSize = 256;

    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        const int64_t FileSize = static_cast<int64_t>(64) * 1024 * 1024 * 1024 + BufferSize;

        m_BaseStorage.Initialize(FileSize);

        SetUpStorage(&m_BaseStorage, HashTargetBlockSize);
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        m_BaseStorage.Finalize();
    }

private:
    nnt::fs::util::VirtualMemoryStorage m_BaseStorage;
};

TEST_F(HierarchicalSha256StorageTest, GetSize)
{
    TestGetSize(GetStorage(), StorageSize);
}

TEST_F(HierarchicalSha256StorageTest, StorageContent)
{
    TestStorageContent(GetStorage(), 512 * 1024);
    TestStorageContent(GetStorage(), 1024 * 1024);
}

TEST_F(HierarchicalSha256StorageTest, BoundaryOffsetSize)
{
    TestBoundaryOffsetSize(GetStorage(), HashTargetBlockSize);
}
TEST_F(HierarchicalSha256StorageTest, WriteRead)
{
    TestWriteReadStorage(GetStorage(), 512 * 1024);
    TestWriteReadStorage(GetStorage(), 1024 * 1024);
}

TEST_F(HierarchicalSha256StorageTest, ConcurrentWriteRead)
{
    TestConcurrentWriteRead(GetStorage());
}

TEST_F(HierarchicalSha256StorageTest, Invalidate)
{
    GetMemoryStorage()->ResetAccessCounter();
    NNT_EXPECT_RESULT_SUCCESS(GetStorage()->OperateRange(
        nn::fs::OperationId::Invalidate,
        0,
        HashTargetBlockSize));
    EXPECT_GT(GetMemoryStorage()->GetInvalidateTimes(), 0);
}



typedef HierarchicalSha256StorageTestWithHostFile HierarchicalSha256StorageTestHeavy;

#if 0 // 時間かかりすぎるので無効化
TEST_F(HierarchicalSha256StorageTestHeavy, GetSize)
{
    SetUpFileStorage();
    TestGetSize(GetStorage(), 8ULL * 1024 * 1024 * 1024);
}

TEST_F(HierarchicalSha256StorageTestHeavy, StorageContent)
{
    SetUpFileStorage();
    TestStorageContent(GetStorage());
}

TEST_F(HierarchicalSha256StorageTestHeavy, BoundaryOffsetSize)
{
    SetUpFileStorage();
    TestBoundaryOffsetSize(GetStorage(), HashTargetBlockSize);
}

TEST_F(HierarchicalSha256StorageTestHeavy, WriteRead)
{
    SetUpFileStorage();
    TestWriteReadStorage(GetStorage());
}

TEST_F(HierarchicalSha256StorageTestHeavy, ConcurrentWriteRead)
{
    SetUpFileStorage();
    TestConcurrentWriteRead(GetStorage());
}
#endif

TEST_F(HierarchicalSha256StorageTestWithVirtualMemoryStorage, WriteReadLargeOffsetHeavy)
{
    TestWriteReadStorageWithLargeOffset(GetStorage(), BufferSize);
}

TEST_F(HierarchicalSha256StorageTest, DetectCorruption)
{
    char buffer[32];
    const auto memory = reinterpret_cast<char*>(GetMemoryStorage()->GetBuffer());

    // 先頭ブロック
    {
        memory[0] ^= 0x01; // 壊す
        NNT_EXPECT_RESULT_FAILURE(ResultHierarchicalSha256HashVerificationFailed, GetStorage()->Read(                      0, buffer, 1)); // 検知
        NNT_EXPECT_RESULT_FAILURE(ResultHierarchicalSha256HashVerificationFailed, GetStorage()->Read(HashTargetBlockSize - 1, buffer, 1)); // 検知

        // (次のブロックには影響無し)
        NNT_EXPECT_RESULT_SUCCESS(GetStorage()->Read(HashTargetBlockSize, buffer, sizeof(buffer)));

        memory[0] ^= 0x01;
    }

    // 末尾ブロック
    {
        memory[StorageSize - 1] ^= 0x01; // 壊す
        NNT_EXPECT_RESULT_FAILURE(ResultHierarchicalSha256HashVerificationFailed, GetStorage()->Read(StorageSize - 1,                   buffer, 1)); // 検知
        NNT_EXPECT_RESULT_FAILURE(ResultHierarchicalSha256HashVerificationFailed, GetStorage()->Read(StorageSize - HashTargetBlockSize, buffer, 1)); // 検知

        // (前のブロックには影響無し)
        NNT_EXPECT_RESULT_SUCCESS(GetStorage()->Read(StorageSize - HashTargetBlockSize - 1, buffer, 1));

        memory[StorageSize - 1] ^= 0x01;
    }
}
