﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/fssystem/fs_StorageLayoutTypeSetter.h>
#include <nn/fssystem/fs_ServiceContext.h>
#include <nn/fs/fsa/fs_IDirectory.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>

class TestStorage : public nn::fs::IStorage
{
public:
    explicit TestStorage(int flag) NN_NOEXCEPT
        : m_StorageFlag(flag)
    {
    }

public:
    virtual nn::Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(offset);
        NN_UNUSED(buffer);
        NN_UNUSED(size);
        NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(offset);
        NN_UNUSED(buffer);
        NN_UNUSED(size);
        NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(size);
        NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(outValue);
        NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result OperateRange(
        void* outBuffer,
        size_t outBufferSize,
        nn::fs::OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(outBuffer);
        NN_UNUSED(outBufferSize);
        NN_UNUSED(operationId);
        NN_UNUSED(offset);
        NN_UNUSED(size);
        NN_UNUSED(inBuffer);
        NN_UNUSED(inBufferSize);
        NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
        NN_RESULT_SUCCESS;
    }

private:
    int m_StorageFlag;
};

class TestFileSystem : public nn::fs::fsa::IFileSystem
{
public:
    explicit TestFileSystem(int flag) NN_NOEXCEPT
        : m_StorageFlag(flag)
    {
    }

private:
    class TestFile : public nn::fs::fsa::IFile
    {
    public:
        explicit TestFile(int flag) NN_NOEXCEPT
            : m_StorageFlag(flag)
        {
        }

    private:
        virtual nn::Result DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(outValue);
            NN_UNUSED(offset);
            NN_UNUSED(buffer);
            NN_UNUSED(size);
            NN_UNUSED(option);
            NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result DoWrite(int64_t offset, const void* buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(offset);
            NN_UNUSED(buffer);
            NN_UNUSED(size);
            NN_UNUSED(option);
            NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result DoFlush() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(size);
            NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result DoGetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(outValue);
            NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result DoOperateRange(
            void* outBuffer,
            size_t outBufferSize,
            nn::fs::OperationId operationId,
            int64_t offset,
            int64_t size,
            const void* inBuffer,
            size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(outBuffer);
            NN_UNUSED(outBufferSize);
            NN_UNUSED(operationId);
            NN_UNUSED(offset);
            NN_UNUSED(size);
            NN_UNUSED(inBuffer);
            NN_UNUSED(inBufferSize);
            NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
            NN_RESULT_SUCCESS;
        }

    private:
        int m_StorageFlag;
    };

    class TestDirectory : public nn::fs::fsa::IDirectory
    {
    public:
        explicit TestDirectory(int flag) NN_NOEXCEPT
            : m_StorageFlag(flag)
        {
        }

        virtual nn::Result DoRead(int64_t* outValue, nn::fs::DirectoryEntry* outEntries, int64_t count) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(outValue);
            NN_UNUSED(outEntries);
            NN_UNUSED(count);
            NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result DoGetEntryCount(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(outValue);
            NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
            NN_RESULT_SUCCESS;
        }

    private:
        int m_StorageFlag;
    };

private:
    virtual nn::Result DoCreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(path);
        NN_UNUSED(size);
        NN_UNUSED(option);
        NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result DoDeleteFile(const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(path);
        NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result DoCreateDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(path);
        NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result DoDeleteDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(path);
        NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result DoDeleteDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(path);
        NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(path);
        NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result DoRenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(currentPath);
        NN_UNUSED(newPath);
        NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result DoRenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(currentPath);
        NN_UNUSED(newPath);
        NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result DoGetEntryType(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(outValue);
        NN_UNUSED(path);
        NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result DoGetFreeSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(outValue);
        NN_UNUSED(path);
        NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result DoGetTotalSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(outValue);
        NN_UNUSED(path);
        NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result DoOpenFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(path);
        NN_UNUSED(mode);
        NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());

        NN_ASSERT_NOT_NULL(outValue);
        *outValue = std::unique_ptr<nn::fs::fsa::IFile>(new TestFile(m_StorageFlag));

        NN_RESULT_SUCCESS;
    }

    virtual nn::Result DoOpenDirectory(std::unique_ptr<nn::fs::fsa::IDirectory>* outValue, const char* path, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(path);
        NN_UNUSED(mode);
        NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());

        NN_ASSERT_NOT_NULL(outValue);
        *outValue = std::unique_ptr<nn::fs::fsa::IDirectory>(new TestDirectory(m_StorageFlag));

        NN_RESULT_SUCCESS;
    }

    virtual nn::Result DoCommit() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result DoFlush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result DoGetFileTimeStampRaw(nn::fs::FileTimeStampRaw* outTimeStamp, const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(outTimeStamp);
        NN_UNUSED(path);
        NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result DoQueryEntry(char* outBuffer, size_t outBufferSize, const char* inBuffer, size_t inBufferSize, nn::fs::fsa::QueryId queryId, const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(outBuffer);
        NN_UNUSED(outBufferSize);
        NN_UNUSED(inBuffer);
        NN_UNUSED(inBufferSize);
        NN_UNUSED(queryId);
        NN_UNUSED(path);
        NN_RESULT_THROW_UNLESS(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag() == m_StorageFlag, nn::fs::ResultUnexpected());
        NN_RESULT_SUCCESS;
    }

private:
    int m_StorageFlag;
};

class StorageLayoutTypeSetterTest : public ::testing::Test
{
public:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::fssystem::RegisterServiceContext(&m_Context);
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::fssystem::UnregisterServiceContext();
    }

private:
    nn::fssystem::ServiceContext m_Context;
};

TEST_F(StorageLayoutTypeSetterTest, TestStorage)
{
    const int StorageFlag = nn::fssystem::StorageFlag_Mmc;
    auto pBaseStorage = std::make_shared<TestStorage>(StorageFlag);
    NN_ASSERT_NOT_NULL(pBaseStorage.get());
    nn::fssystem::StorageLayoutTypeSetStorage storage(pBaseStorage, StorageFlag);

    const size_t BufferSize = 256;
    char buffer[BufferSize];
    const int64_t Size = 256;
    int64_t outSize;

    NNT_EXPECT_RESULT_SUCCESS(storage.Read(0, buffer, BufferSize));
    NNT_EXPECT_RESULT_SUCCESS(storage.Write(0, buffer, BufferSize));
    NNT_EXPECT_RESULT_SUCCESS(storage.Flush());
    NNT_EXPECT_RESULT_SUCCESS(storage.SetSize(Size));
    NNT_EXPECT_RESULT_SUCCESS(storage.GetSize(&outSize));
    NNT_EXPECT_RESULT_SUCCESS(storage.OperateRange(nullptr, 0, nn::fs::OperationId::Invalidate, 0, 0, nullptr, 0));
}

TEST_F(StorageLayoutTypeSetterTest, TestFileSystem)
{
    const int StorageFlag = nn::fssystem::StorageFlag_Mmc;
    auto pBaseFileSystem = std::make_shared<TestFileSystem>(StorageFlag);
    NN_ASSERT_NOT_NULL(pBaseFileSystem.get());
    nn::fssystem::StorageLayoutTypeSetFileSystem fileSystem(pBaseFileSystem, StorageFlag);

    const char* Path = "test";
    const size_t BufferSize = 256;
    char buffer[BufferSize];
    const int64_t Size = 256;
    int64_t outSize;
    nn::fs::DirectoryEntryType outDirectoryEntryType;
    nn::fs::FileTimeStampRaw outTimeStamp;

    NNT_EXPECT_RESULT_SUCCESS(fileSystem.CreateFile(Path, Size));
    NNT_EXPECT_RESULT_SUCCESS(fileSystem.DeleteFile(Path));
    NNT_EXPECT_RESULT_SUCCESS(fileSystem.CreateDirectory(Path));
    NNT_EXPECT_RESULT_SUCCESS(fileSystem.DeleteDirectory(Path));
    NNT_EXPECT_RESULT_SUCCESS(fileSystem.DeleteDirectoryRecursively(Path));
    NNT_EXPECT_RESULT_SUCCESS(fileSystem.CleanDirectoryRecursively(Path));
    NNT_EXPECT_RESULT_SUCCESS(fileSystem.RenameFile(Path, Path));
    NNT_EXPECT_RESULT_SUCCESS(fileSystem.RenameDirectory(Path, Path));
    NNT_EXPECT_RESULT_SUCCESS(fileSystem.GetEntryType(&outDirectoryEntryType, Path));
    NNT_EXPECT_RESULT_SUCCESS(fileSystem.Commit());
    NNT_EXPECT_RESULT_SUCCESS(fileSystem.GetFreeSpaceSize(&outSize, Path));
    NNT_EXPECT_RESULT_SUCCESS(fileSystem.GetTotalSpaceSize(&outSize, Path));
    NNT_EXPECT_RESULT_SUCCESS(fileSystem.Flush());
    NNT_EXPECT_RESULT_SUCCESS(fileSystem.GetFileTimeStampRaw(&outTimeStamp, Path));
    NNT_EXPECT_RESULT_SUCCESS(fileSystem.QueryEntry(buffer, BufferSize, buffer, Size, nn::fs::fsa::QueryId::SetConcatenationFileAttribute, Path));
}

TEST_F(StorageLayoutTypeSetterTest, TestFile)
{
    const int StorageFlag = nn::fssystem::StorageFlag_Mmc;
    auto pBaseFileSystem = std::make_shared<TestFileSystem>(StorageFlag);
    NN_ASSERT_NOT_NULL(pBaseFileSystem.get());
    nn::fssystem::StorageLayoutTypeSetFileSystem fileSystem(pBaseFileSystem, StorageFlag);

    std::unique_ptr<nn::fs::fsa::IFile> pFile;
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&pFile, "test", static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)));

    const size_t BufferSize = 256;
    char buffer[BufferSize];
    const int64_t Size = 256;
    int64_t outSize;
    size_t outReadSize;

    NNT_EXPECT_RESULT_SUCCESS(pFile->Read(&outReadSize, 0, buffer, BufferSize, nn::fs::ReadOption::MakeValue(0)));
    NNT_EXPECT_RESULT_SUCCESS(pFile->Write(0, buffer, BufferSize, nn::fs::WriteOption::MakeValue(0)));
    NNT_EXPECT_RESULT_SUCCESS(pFile->Flush());
    NNT_EXPECT_RESULT_SUCCESS(pFile->SetSize(Size));
    NNT_EXPECT_RESULT_SUCCESS(pFile->GetSize(&outSize));
    NNT_EXPECT_RESULT_SUCCESS(pFile->OperateRange(nullptr, 0, nn::fs::OperationId::Invalidate, 0, 0, nullptr, 0));
}

TEST_F(StorageLayoutTypeSetterTest, TestDirectory)
{
    const int StorageFlag = nn::fssystem::StorageFlag_Mmc;
    auto pBaseFileSystem = std::make_shared<TestFileSystem>(StorageFlag);
    NN_ASSERT_NOT_NULL(pBaseFileSystem.get());
    nn::fssystem::StorageLayoutTypeSetFileSystem fileSystem(pBaseFileSystem, StorageFlag);

    std::unique_ptr<nn::fs::fsa::IDirectory> pDirectory;
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenDirectory(&pDirectory, "test", nn::fs::OpenDirectoryMode_All));

    int64_t outCount;
    nn::fs::DirectoryEntry outDirectoryEntry;

    NNT_EXPECT_RESULT_SUCCESS(pDirectory->Read(&outCount, &outDirectoryEntry, 1));
    NNT_EXPECT_RESULT_SUCCESS(pDirectory->GetEntryCount(&outCount));
}
