﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include "fs_SlotCache.h"

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/util/util_Optional.h>

using namespace nn;
using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nnt::fs::util;

namespace {

class SlotCacheSimple : public ::testing::Test
{
protected:
    SlotCacheSimple()
    {
    }

    virtual void SetUp() NN_NOEXCEPT
    {
        for (int i = 0; i < 4; i++)
        {
            memset(key[i], 0x0A + i, KeySize);
        }

        entryArray[0].emplace(1);
        entryArray[1].emplace(2);
        entryArray[2].emplace(3);
        for (auto& entry : entryArray)
        {
            cache.AddEntry(&entry.value());
        }

    }

    virtual void TearDown() NN_NOEXCEPT
    {
    }

protected:
    static const int KeySize = 16;
    static const int EntryCount = 3;

protected:
    SlotCache cache;
    char key[4][KeySize];

    nn::util::optional<SlotCacheEntry> entryArray[EntryCount];

};


TEST_F(SlotCacheSimple, LruLow)
{
    std::unique_ptr<SlotCacheAccessor> accessor;

    // LRU を追い出すこと
    {
        // MRU 3, 2, 1 LRU

        NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, cache.Find(&accessor, key[0], KeySize, 0));

        NNT_EXPECT_RESULT_SUCCESS(cache.AllocateLowPriority(&accessor, key[0], KeySize, 0));
        accessor.reset();
        NNT_EXPECT_RESULT_SUCCESS(cache.AllocateLowPriority(&accessor, key[1], KeySize, 0));
        accessor.reset();
        NNT_EXPECT_RESULT_SUCCESS(cache.AllocateLowPriority(&accessor, key[2], KeySize, 0));
        accessor.reset();

        // 3C, 2B, 1A

        NNT_EXPECT_RESULT_SUCCESS(cache.Find(&accessor, key[2], KeySize, 0));
        EXPECT_EQ(3, accessor->GetSlotIndex());
        accessor.reset();

        // 3C, 2B, 1A

        NNT_EXPECT_RESULT_SUCCESS(cache.AllocateLowPriority(&accessor, key[3], KeySize, 0));
        EXPECT_EQ(1, accessor->GetSlotIndex());
        accessor.reset();

        // 1D, 3C, 2B

        NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, cache.Find(&accessor, key[0], KeySize, 0));
        accessor.reset();

        // MRU 更新
        NNT_EXPECT_RESULT_SUCCESS(cache.Find(&accessor, key[1], KeySize, 0));
        EXPECT_EQ(2, accessor->GetSlotIndex());
        accessor.reset();

        // 2B, 1D, 3C

        NNT_EXPECT_RESULT_SUCCESS(cache.AllocateLowPriority(&accessor, key[0], KeySize, 0));
        EXPECT_EQ(3, accessor->GetSlotIndex());
        accessor.reset();

        // 3A, 2B, 1D

        NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, cache.Find(&accessor, key[2], KeySize, 0));
        accessor.reset();
    }

}


TEST_F(SlotCacheSimple, LruHigh)
{
    std::unique_ptr<SlotCacheAccessor> accessor;

    // LRU を追い出すこと
    {
        // MRU 3, 2, 1 LRU

        NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, cache.Find(&accessor, key[0], KeySize, 0));

        NNT_EXPECT_RESULT_SUCCESS(cache.AllocateHighPriority(&accessor, key[0], KeySize, 0));
        accessor.reset();
        NNT_EXPECT_RESULT_SUCCESS(cache.AllocateHighPriority(&accessor, key[1], KeySize, 0));
        accessor.reset();
        NNT_EXPECT_RESULT_SUCCESS(cache.AllocateHighPriority(&accessor, key[2], KeySize, 0));
        accessor.reset();

        // 3C, 2B, 1A

        NNT_EXPECT_RESULT_SUCCESS(cache.Find(&accessor, key[2], KeySize, 0));
        EXPECT_EQ(3, accessor->GetSlotIndex());
        accessor.reset();

        // 3C, 2B, 1A

        NNT_EXPECT_RESULT_SUCCESS(cache.AllocateHighPriority(&accessor, key[3], KeySize, 0));
        EXPECT_EQ(1, accessor->GetSlotIndex());
        accessor.reset();

        // 1D, 3C, 2B

        NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, cache.Find(&accessor, key[0], KeySize, 0));
        accessor.reset();

        // MRU 更新
        NNT_EXPECT_RESULT_SUCCESS(cache.Find(&accessor, key[1], KeySize, 0));
        EXPECT_EQ(2, accessor->GetSlotIndex());
        accessor.reset();

        // 2B, 1D, 3C

        NNT_EXPECT_RESULT_SUCCESS(cache.AllocateHighPriority(&accessor, key[0], KeySize, 0));
        EXPECT_EQ(3, accessor->GetSlotIndex());
        accessor.reset();

        // 3A, 2B, 1D

        NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, cache.Find(&accessor, key[2], KeySize, 0));
        accessor.reset();
    }

}

TEST_F(SlotCacheSimple, HighLow)
{
    std::unique_ptr<SlotCacheAccessor> accessor;

    // High LRU より Low 優先で確保されること
    {

        NNT_EXPECT_RESULT_SUCCESS(cache.AllocateLowPriority(&accessor, key[3], KeySize, 0));
        accessor.reset();
        NNT_EXPECT_RESULT_SUCCESS(cache.AllocateLowPriority(&accessor, key[1], KeySize, 0));
        accessor.reset();
        NNT_EXPECT_RESULT_SUCCESS(cache.AllocateLowPriority(&accessor, key[0], KeySize, 0));
        accessor.reset();

        // L 3A, 2B, 1D

        NNT_EXPECT_RESULT_SUCCESS(cache.AllocateHighPriority(&accessor, key[1], KeySize, 0));
        accessor.reset();
        NNT_EXPECT_RESULT_SUCCESS(cache.AllocateHighPriority(&accessor, key[2], KeySize, 0));
        accessor.reset();

        // H 2C, 1B
        // L 3A,

        NNT_EXPECT_RESULT_SUCCESS(cache.Find(&accessor, key[0], KeySize, 0));
        EXPECT_EQ(3, accessor->GetSlotIndex());
        accessor.reset();

        // H 2C, 1B
        // L 3A,

        NNT_EXPECT_RESULT_SUCCESS(cache.AllocateLowPriority(&accessor, key[3], KeySize, 0));
        EXPECT_EQ(3, accessor->GetSlotIndex());
        accessor.reset();

        // H 2C, 1B
        // L 3D,

        NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, cache.Find(&accessor, key[0], KeySize, 0));
        accessor.reset();
    }

}


TEST_F(SlotCacheSimple, Key)
{
    std::unique_ptr<SlotCacheAccessor> accessor;

    // key1 + key2 でマッチ判定されること
    {
        NNT_EXPECT_RESULT_SUCCESS(cache.AllocateLowPriority(&accessor, key[0], KeySize, 0));
        accessor.reset();
        NNT_EXPECT_RESULT_SUCCESS(cache.AllocateLowPriority(&accessor, key[1], KeySize, 1));
        accessor.reset();

        NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, cache.Find(&accessor, key[0], KeySize, 1));
        NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, cache.Find(&accessor, key[1], KeySize, 0));

        NNT_EXPECT_RESULT_SUCCESS(cache.Find(&accessor, key[0], KeySize, 0));
        accessor.reset();
        NNT_EXPECT_RESULT_SUCCESS(cache.Find(&accessor, key[1], KeySize, 1));
        accessor.reset();
    }

}


}

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
