﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/fssystem/fs_PartitionFileSystem.h>
#include <nn/fs/fsa/fs_Registrar.h>

#include <nn/fs/fs_ResultHandler.h>

#include <nn/fssystem/fs_PartitionFileSystemMeta.h>
#include <nn/crypto/crypto_Sha256Generator.h>

#include <nn/fs/fs_SubStorage.h>
#include <nn/gc/gc.h>

#include <nn/fs/fs_GameCard.h>
#include <nn/util/util_BitUtil.h>

#include <algorithm>
#include <array>

using namespace nn;
using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nn::fssystem;
using namespace nn::fs::detail;
using namespace nnt::fs::util;

using nn::util::align_up;
typedef Sha256PartitionFileSystemMeta::FileEntryForConstruct FileEntryForConstruct;



namespace {

class Sha256PartitionFs : public ::testing::TestWithParam<int>
{
};

void CheckEntries(const Sha256PartitionFileSystemMeta* pMeta, const Vector<Sha256PartitionFileSystemMeta::FileEntryForConstruct>& entryArray)
{
    for (auto entry : entryArray)
    {
        auto index = pMeta->GetEntryIndex(entry.name);
        EXPECT_TRUE(index >= 0);

        auto pMetaEntry = pMeta->GetEntry(index);

        NNT_FS_UTIL_EXPECT_MEMCMPEQ(entry.hash, pMetaEntry->hash, sizeof(entry.hash));
        EXPECT_EQ(entry.hashTargetOffset, pMetaEntry->hashTargetOffset);
        EXPECT_EQ(entry.hashTargetSize  , pMetaEntry->hashTargetSize);
        EXPECT_EQ(entry.offset          , pMetaEntry->offset);
        EXPECT_EQ(entry.size            , pMetaEntry->size);
        EXPECT_STREQ(entry.name         , pMeta->GetEntryName(index));
    }
}

//! メタデータを生成し、各 Initialize 方法で正しく初期化できること
TEST(Sha256PartitionFsMeta, Construct)
{
    Vector<Sha256PartitionFileSystemMeta::FileEntryForConstruct> entryArray;
    Sha256PartitionFileSystemMeta::FileEntryForConstruct entry0 = { 0,  10,  5,  5, {0}, "entry0"};
    entryArray.push_back(entry0);
    Sha256PartitionFileSystemMeta::FileEntryForConstruct entry1 = { 10, 20, 10, 10, {1}, "entry1"};
    entryArray.push_back(entry1);
    Sha256PartitionFileSystemMeta::FileEntryForConstruct entry2 = { 30, 30,  0, 30, {2}, "entry2"};
    entryArray.push_back(entry2);

    size_t headerSize;
    size_t bodySize = 4 * 1024;
    NNT_EXPECT_RESULT_SUCCESS(Sha256PartitionFileSystemMeta::QueryMetaDataSize(&headerSize, &entryArray[0], static_cast<int>(entryArray.size())));
    std::unique_ptr<char[]> storageBuffer(new char[headerSize + bodySize]);
    NNT_EXPECT_RESULT_SUCCESS(Sha256PartitionFileSystemMeta::ConstructMetaData(storageBuffer.get(), headerSize, &entryArray[0], static_cast<int>(entryArray.size())));

    MemoryStorage fsStorage(storageBuffer.get(), headerSize + bodySize);

    //! バッファ渡し
    {
        std::unique_ptr<char[]> metaDataBuffer(new char[headerSize]);
        Sha256PartitionFileSystemMeta meta;
        NNT_EXPECT_RESULT_SUCCESS(meta.Initialize(&fsStorage, metaDataBuffer.get(), headerSize));
        CheckEntries(&meta, entryArray);
    }

    //! アロケータ指定
    {
        std::unique_ptr<char[]> metaDataBuffer(new char[headerSize]);
        Sha256PartitionFileSystemMeta meta;
        NNT_EXPECT_RESULT_SUCCESS(meta.Initialize(&fsStorage, GetTestLibraryAllocator()));
        CheckEntries(&meta, entryArray);
    }

    //! アロケータ + ヘッダハッシュ指定
    {
        char hash[crypto::Sha256Generator::HashSize] = {0};
        std::unique_ptr<char[]> metaDataBuffer(new char[headerSize]);
        Sha256PartitionFileSystemMeta meta;

        // ヘッダハッシュ不整合
        NNT_EXPECT_RESULT_FAILURE(ResultSha256PartitionHashVerificationFailed, meta.Initialize(&fsStorage, GetTestLibraryAllocator(), hash, sizeof(hash)));

        crypto::GenerateSha256Hash(hash, sizeof(hash), storageBuffer.get(), headerSize);
        NNT_EXPECT_RESULT_SUCCESS(meta.Initialize(&fsStorage, GetTestLibraryAllocator(), hash, sizeof(hash)));

        CheckEntries(&meta, entryArray);
    }

    //! ヘッダ壊れ
    {
        auto corruptedStorageBuffer = AllocateBuffer(headerSize + bodySize);
        std::memcpy(corruptedStorageBuffer.get(), storageBuffer.get(), headerSize + bodySize);
        corruptedStorageBuffer.get()[0] = 0x00;
        MemoryStorage corruptedFsStorage(corruptedStorageBuffer.get(), headerSize + bodySize);

        std::unique_ptr<char[]> metaDataBuffer(new char[headerSize]);
        Sha256PartitionFileSystemMeta meta;

        NNT_EXPECT_RESULT_FAILURE(ResultSha256PartitionSignatureVerificationFailed, meta.Initialize(&corruptedFsStorage, metaDataBuffer.get(), headerSize));

        NNT_EXPECT_RESULT_FAILURE(ResultSha256PartitionSignatureVerificationFailed, meta.Initialize(&corruptedFsStorage, GetTestLibraryAllocator()));

        char hash[crypto::Sha256Generator::HashSize] = { 0 };
        NNT_EXPECT_RESULT_FAILURE(ResultSha256PartitionSignatureVerificationFailed, meta.Initialize(&corruptedFsStorage, GetTestLibraryAllocator(), hash, sizeof(hash)));
    }

}


void DumpMovedFs(std::unique_ptr<IFileSystem>&& fs)
{
    Register("tmp", std::move(fs));
    NNT_EXPECT_RESULT_SUCCESS(DumpDirectoryRecursive("tmp:/"));
    Unregister("tmp");
}


//! ハッシュ検証対象領域について改竄検知ができること
TEST_P(Sha256PartitionFs, PartifionFs_IntegrityCheck)
{
    // 2KB ファイル、 0 ～ 768 B がハッシュ検証対象
    // TODO: 他のパラメータでテスト
    const int HashTargetOffset = 0;
    const int FileSize = 2048;
    int HashTargetSize = GetParam();

    Vector<Sha256PartitionFileSystemMeta::FileEntryForConstruct> entryArray;
    {
        Sha256PartitionFileSystemMeta::FileEntryForConstruct entry0 = { 0, FileSize, HashTargetOffset, static_cast<const uint32_t>(HashTargetSize), {0}, "entry0"};
        entryArray.push_back(entry0);
    }

    size_t headerSize;
    size_t bodySize = 4 * 1024;
    NNT_EXPECT_RESULT_SUCCESS(Sha256PartitionFileSystemMeta::QueryMetaDataSize(&headerSize, &entryArray[0], static_cast<int>(entryArray.size())));
    std::unique_ptr<char[]> storageBuffer(new char[headerSize + bodySize]);

    // ファイルデータ
    char* pFileData = storageBuffer.get() + headerSize;
    FillBufferWith32BitCount(pFileData, FileSize, 0);

    // ハッシュ埋め込み
    crypto::GenerateSha256Hash(entryArray[0].hash, sizeof(entryArray[0].hash), storageBuffer.get() + headerSize + HashTargetOffset, HashTargetSize);

    // メタデータ生成
    NNT_EXPECT_RESULT_SUCCESS(Sha256PartitionFileSystemMeta::ConstructMetaData(storageBuffer.get(), headerSize, &entryArray[0], static_cast<int>(entryArray.size())));
    MemoryStorage fsStorage(storageBuffer.get(), headerSize + bodySize);

    {
        std::unique_ptr<Sha256PartitionFileSystem> fs(new Sha256PartitionFileSystem());
        NNT_EXPECT_RESULT_SUCCESS(fs->Initialize(&fsStorage));
        Register("tmp", std::move(fs));

        // 改竄無し
        NNT_EXPECT_RESULT_SUCCESS(DumpDirectoryRecursive("tmp:/"));

        // ハッシュ範囲は改竄を検知できる
        {
            const int TargetOffsets[] = {
                HashTargetOffset,
                HashTargetOffset + HashTargetSize / 2,
                HashTargetOffset + HashTargetSize - 1,
            };
            for(auto offset : TargetOffsets)
            {
                auto backUp = pFileData[offset];
                pFileData[offset] = 'A';
                NNT_EXPECT_RESULT_FAILURE(ResultSha256PartitionHashVerificationFailed, DumpDirectoryRecursive("tmp:/", false));
                pFileData[offset] = backUp;
            }
        }

        // ハッシュ範囲外は改竄を検知できない
        {
            const int TargetOffsets[] = {
                //0,
                //HashTargetOffset - 1,
                HashTargetOffset + HashTargetSize + 1,
                FileSize - 1,
            };
            for(auto offset : TargetOffsets)
            {
                auto backUp = pFileData[offset];
                pFileData[offset] = 'A';
                NNT_EXPECT_RESULT_SUCCESS(DumpDirectoryRecursive("tmp:/"));
                pFileData[offset] = backUp;
            }
        }

        Unregister("tmp");
    }

    // TODO: テスト分離
    //! 各 Initialize 方法で正しく読めること
    {
        std::shared_ptr<IStorage> sharedStorage(new SubStorage(&fsStorage, 0, headerSize + bodySize));

        {
            std::unique_ptr<Sha256PartitionFileSystem> fs(new Sha256PartitionFileSystem());
            NNT_EXPECT_RESULT_SUCCESS(fs->Initialize(sharedStorage.get()));
            DumpMovedFs(std::move(fs));
        }

        {
            std::unique_ptr<Sha256PartitionFileSystem> fs(new Sha256PartitionFileSystem());
            NNT_EXPECT_RESULT_SUCCESS(fs->Initialize(sharedStorage));
            DumpMovedFs(std::move(fs));
        }

        {
            std::unique_ptr<Sha256PartitionFileSystem> fs(new Sha256PartitionFileSystem());
            NNT_EXPECT_RESULT_SUCCESS(fs->Initialize(sharedStorage, GetTestLibraryAllocator()));
            DumpMovedFs(std::move(fs));
        }

        {
            std::unique_ptr<Sha256PartitionFileSystemMeta> meta(new Sha256PartitionFileSystemMeta());
            NNT_EXPECT_RESULT_SUCCESS(meta->Initialize(sharedStorage.get(), GetTestLibraryAllocator()));

            std::unique_ptr<Sha256PartitionFileSystem> fs(new Sha256PartitionFileSystem());
            NNT_EXPECT_RESULT_SUCCESS(fs->Initialize(std::move(meta), sharedStorage));
            DumpMovedFs(std::move(fs));
        }
    }


    // TODO: テスト分離
    //! ハッシュ対象近辺の read が正しくできること
    {
        std::unique_ptr<Sha256PartitionFileSystem> fs(new Sha256PartitionFileSystem());
        NNT_EXPECT_RESULT_SUCCESS(fs->Initialize(&fsStorage));

        std::unique_ptr<IFile> file;
        NNT_EXPECT_RESULT_SUCCESS(fs->OpenFile(&file, "/entry0", static_cast<OpenMode>(OpenMode_Read)));

        const struct
        {
            int64_t offset;
            size_t size;
        } Ranges[] =
        {
            {0, 256},
            {0, 512},
            {0, 1024},
            {4, 256},
            {4, 1532},
            {256, 256},
            {256, 512},
            {512, 256},
            {512, 512},
            {512, 1024},
            {768, 256},
            {1024, 256},
        };

        char buffer[2048];
        for(auto range : Ranges)
        {
            NNT_FS_SCOPED_TRACE("%d, %d", range.offset, range.size);

            ASSERT_LE(range.size, sizeof(buffer));

            size_t readSize;

            InvalidateVariable(buffer, sizeof(buffer));
            if ((HashTargetOffset < range.offset) &&
                range.offset < (HashTargetOffset + HashTargetSize) &&
                ((HashTargetOffset + HashTargetSize) < (range.offset + range.size)))
            {
                // ハッシュ対象内外の領域両方の部分読出しは未サポート
                NNT_EXPECT_RESULT_FAILURE(ResultInvalidSha256PartitionHashTarget, file->Read(&readSize, range.offset, buffer, range.size, ReadOption()));
            }
            else
            {
                NNT_EXPECT_RESULT_SUCCESS(file->Read(&readSize, range.offset, buffer, range.size, ReadOption()));
                EXPECT_TRUE(IsFilledWith32BitCount(buffer, range.size, range.offset));
            }
        }

    }
} // NOLINT(impl/function_size)

INSTANTIATE_TEST_CASE_P(WithVariousHashTargetSize,
                        Sha256PartitionFs,
                        ::testing::Values(768, 1536));

#if 0
void DumpSha256PartitionFs(IStorage* pStorage)
{
    std::unique_ptr<Sha256PartitionFileSystem> fs(new Sha256PartitionFileSystem());
    NNT_EXPECT_RESULT_SUCCESS(fs->Initialize(pStorage));
    DumpMovedFs(std::move(fs));
}

//! xci ファイルの内容をダンプ
TEST(Xci, DumpAll)
{
    MountHostRoot();

    FileHandle xciHandle;
    const nnt::fs::util::String RootPath(nnt::fs::util::GetHostTemporaryPath());
    NNT_ASSERT_RESULT_SUCCESS(OpenFile(&xciHandle, RootPath.append("/test.xci").c_str(), OpenMode_Read));
    int64_t xciSize;
    NNT_ASSERT_RESULT_SUCCESS(GetFileSize(&xciSize, xciHandle));
    FileHandleStorage xciStorage(xciHandle);
    SubStorage xciBodyStorage(&xciStorage, 64 * 1024, xciSize - 64 * 1024);


    // root
    {
        DumpSha256PartitionFs(&xciBodyStorage);

        std::unique_ptr<Sha256PartitionFileSystem> rootPartitionFs(new Sha256PartitionFileSystem());
        NNT_EXPECT_RESULT_SUCCESS(rootPartitionFs->Initialize(&xciBodyStorage));
        {
            std::unique_ptr<IFile> updateFile;
            NNT_EXPECT_RESULT_SUCCESS(rootPartitionFs->OpenFile(&updateFile, "/update", static_cast<OpenMode>(OpenMode_Read | OpenMode_Write)));
            FileStorage updateFileStorage(updateFile.get());
            DumpSha256PartitionFs(&updateFileStorage);
        }

        {
            std::unique_ptr<IFile> normalFile;
            NNT_EXPECT_RESULT_SUCCESS(rootPartitionFs->OpenFile(&normalFile, "/normal", static_cast<OpenMode>(OpenMode_Read | OpenMode_Write)));
            FileStorage normalFileStorage(normalFile.get());
            DumpSha256PartitionFs(&normalFileStorage);
        }

        {
            std::unique_ptr<IFile> secureFile;
            NNT_EXPECT_RESULT_SUCCESS(rootPartitionFs->OpenFile(&secureFile, "/secure", static_cast<OpenMode>(OpenMode_Read | OpenMode_Write)));
            FileStorage secureFileStorage(secureFile.get());
            DumpSha256PartitionFs(&secureFileStorage);
        }

    }



    CloseFile(xciHandle);
    UnmountHostRoot();
}
#endif

//! 4 GB を超えるオフセットのテスト
TEST(Sha256PartitionFs, ReadWriteLargeOffset)
{
    const size_t AccessSize = 1024;
    const int HashTargetSize = 768;
    const int HashTargetOffset = AccessSize;
    const int64_t FileSize = LargeOffsetMax + AccessSize;

    Vector<Sha256PartitionFileSystemMeta::FileEntryForConstruct> entryArray;
    {
        Sha256PartitionFileSystemMeta::FileEntryForConstruct entry0 =
        {
            0,
            static_cast<uint64_t>(FileSize),
            HashTargetOffset,
            HashTargetSize,
            {0},
            "entry0"
        };
        entryArray.push_back(entry0);
    }

    size_t metaDataSize = 0;
    NNT_EXPECT_RESULT_SUCCESS(Sha256PartitionFileSystemMeta::QueryMetaDataSize(
        &metaDataSize,
        &entryArray[0],
        static_cast<int>(entryArray.size())));
    auto metaDataBuffer = nnt::fs::util::AllocateBuffer(metaDataSize);

    // ハッシュ埋め込み
    auto zeroBuffer = nnt::fs::util::AllocateBuffer(HashTargetSize);
    std::memset(zeroBuffer.get(), 0, HashTargetSize);
    crypto::GenerateSha256Hash(
        entryArray[0].hash,
        sizeof(entryArray[0].hash),
        zeroBuffer.get(),
        HashTargetSize);

    // メタデータ生成
    NNT_EXPECT_RESULT_SUCCESS(Sha256PartitionFileSystemMeta::ConstructMetaData(
        metaDataBuffer.get(),
        metaDataSize,
        &entryArray[0],
        static_cast<int>(entryArray.size())));

    nnt::fs::util::VirtualMemoryStorage baseStorage(nnt::fs::util::LargeOffsetMax * 2);
    baseStorage.Write(0, metaDataBuffer.get(), metaDataSize);

    nn::fssystem::Sha256PartitionFileSystem fileSystem;
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.Initialize(&baseStorage));

    auto readBuffer = nnt::fs::util::AllocateBuffer(AccessSize);
    auto writeBuffer = nnt::fs::util::AllocateBuffer(AccessSize);

    {
        // AllowAppend なし
        std::unique_ptr<nn::fs::fsa::IFile> pFile;
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(
            &pFile,
            "/entry0",
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)));
        nnt::fs::util::TestFileAccessWithLargeOffset(pFile.get(), AccessSize);

        // 範囲外への読み書きが失敗する
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            pFile->Write(FileSize, writeBuffer.get(), AccessSize, nn::fs::WriteOption()));

        size_t readSize = 0;
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultOutOfRange,
            pFile->Read(&readSize, FileSize + 1, readBuffer.get(), AccessSize, nn::fs::ReadOption()));
    }
    {
        // AllowAppend あり
        std::unique_ptr<nn::fs::fsa::IFile> pFile;
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(
            &pFile,
            "/entry0",
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend)));

        // 範囲外への読み書きが失敗する
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            pFile->Write(FileSize, writeBuffer.get(), AccessSize, nn::fs::WriteOption()));

        size_t readSize = 0;
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultOutOfRange,
            pFile->Read(&readSize, FileSize + 1, readBuffer.get(), AccessSize, nn::fs::ReadOption()));
    }
}

}


extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetEnabledAutoAbort(false);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
