﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <array>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/spl/spl_Api.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_PosixTime.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_Utility.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fs/fs_DebugPrivate.h>
#include <nn/crypto/crypto_Compare.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/crypto/crypto_RsaPssSha256Signer.h>
#include <nn/crypto/crypto_Aes128CmacGenerator.h>
#include <nn/crypto/crypto_Csrng.h>

#include <nn/fs/fs_SaveDataTransferVersion2.h>
#include <nn/fssrv/fssrv_SaveDataTransferCryptoConfiguration.h>
#include "detail/fssrv_SaveDataTransferVersion2.h"
#include "detail/fssrv_SaveDataTransferChunkPorter.h"
#include "detail/fssrv_SaveDataTransferChunkIterator.h"

#if !defined(NN_BUILD_CONFIG_OS_WIN)
#include "../../Processes/fs/fs_CryptoConfiguration.h"
#else
#include "shim/fs_FileSystemCreatorInterfaces.h"
#endif

typedef nn::fs::SaveDataTransferManagerVersion2 SaveDataTransferManagerForCloudBackUp;
typedef nn::fs::InitialDataVersion2 InitialDataForCloudBackUp;

#include <nn/fs/fs_SaveDataPrivate.h>

using namespace nn;
using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nnt::fs::util;



namespace {

const char KeySeedFileName[] = "keySeed";
const char InitialDataMacFileName[] = "initialDataMac";


String GenerateStorageFileName(SaveDataChunkId id)
{
    return ToString(static_cast<int64_t>(id));
}

void GenerateAadForTest(nn::fs::detail::InitialDataAad* aad)
{
    nnt::fs::util::FillBufferWith8BitCount(aad->data, aad->Size, 0);
}


void CalculateSaveDataHash(Hash *outValue, const SaveDataInfo& info)
{
    const size_t WorkBufferSize = 4 * 1024 * 1024;
    auto workBuffer = AllocateBuffer(WorkBufferSize);

    crypto::Sha256Generator sha;
    sha.Initialize();

    // save
    {
        nnt::fs::util::Hash hash;
        NN_ABORT_UNLESS_RESULT_SUCCESS(MountSaveData("_hash", info.applicationId, info.saveDataUserId));
        CalculateDirectoryTreeHash(&hash, "_hash:/", workBuffer.get(), WorkBufferSize);
        Unmount("_hash");

        sha.Update(&hash, sizeof(hash));
    }

    // internal storage
    {
        nnt::fs::util::Hash hash;
        NN_ABORT_UNLESS_RESULT_SUCCESS(MountSaveDataInternalStorage("_hash", info.saveDataSpaceId, info.saveDataId));

        CalculateFileHash(&hash, "_hash:/AllocationTableControlArea", workBuffer.get(), WorkBufferSize);
        sha.Update(&hash, sizeof(hash));
        CalculateFileHash(&hash, "_hash:/AllocationTableMeta", workBuffer.get(), WorkBufferSize);
        sha.Update(&hash, sizeof(hash));
        CalculateFileHash(&hash, "_hash:/AllocationTableDataWithZeroFree", workBuffer.get(), WorkBufferSize);
        sha.Update(&hash, sizeof(hash));

        Unmount("_hash");
    }

    // thumbnail
    {
        const size_t headerSize = 32;
        const size_t  bodySize = nn::fs::detail::ThumbnailFileSize - 32;
        char thumbnailHeader[headerSize];
        char thumbnailBody[bodySize];
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::ReadSaveDataThumbnailFile(info.applicationId.value, info.saveDataUserId, thumbnailHeader, headerSize, thumbnailBody, bodySize)
        );

        sha.Update(thumbnailHeader, headerSize);
        sha.Update(thumbnailBody, bodySize);
    }

    sha.GetHash(outValue->value, sizeof(outValue->value));
}


Result GetExportSizeByDryRun(int64_t* outValue, nnt::fs::util::Hash* outHash, ISaveDataDivisionExporter* pExporter, SaveDataChunkId id, char* buffer, size_t BufferSize)
{
    int64_t size = 0;
    std::unique_ptr<ISaveDataChunkExporter> chunkExporter;
    NN_RESULT_DO(pExporter->OpenSaveDataChunkExporter(&chunkExporter, id));

    int64_t checkSum = 0; // 簡易チェックサム

    while (true)
    {
        size_t pulledSize;
        NN_RESULT_DO(chunkExporter->Pull(&pulledSize, buffer, BufferSize));
        if (pulledSize > 0)
        {
            size += pulledSize;

            if (outHash)
            {
                for (size_t i = 0; i < pulledSize; i++)
                {
                    checkSum += buffer[i];
                }
            }
        }
        else
        {
            *outValue = size;
            if (outHash)
            {
                memset(outHash->value, 0, sizeof(outHash->value));
                memcpy(outHash->value, &checkSum, sizeof(checkSum));
            }
            NN_RESULT_SUCCESS;
        }
    }

}


void ReadInitialData(InitialDataForCloudBackUp* pOutValue, String backupStoragePath)
{
    auto initialDataRaw = AllocateBuffer(32);
    size_t initialDataRawSize;

    auto initialDataName = GenerateStorageFileName(SaveDataChunkIdForInitialData);
    NNT_ASSERT_RESULT_SUCCESS(ReadFileAtOnce(&initialDataRawSize, &initialDataRaw, (backupStoragePath + initialDataName).c_str()));
    InitialDataForCloudBackUp initialData = {};
    ASSERT_EQ(sizeof(InitialDataForCloudBackUp), initialDataRawSize);
    memcpy(&initialData, initialDataRaw.get(), initialDataRawSize);

    *pOutValue = initialData;
}

nn::fs::SaveDataTransferManagerVersion2::KeySeedPackage GetKeySeedPackageFromServer(const SaveDataTransferManagerForCloudBackUp::Challenge& challenge, const KeySeed& keySeed, const InitialDataMac& mac, bool falsify = false)
{
    nn::fs::SaveDataTransferManagerVersion2::KeySeedPackage ksp;
    nn::fs::detail::SdaId sdaId = { 0 }; // TORIAEZU
    nnt::fs::util::GenerateKeySeedPackage(&ksp, TestApplicationId0, sdaId, keySeed, mac, challenge, falsify);

    return ksp;
}

ISaveDataDivisionExporter::KeySeed GetKeySeed(String backupStoragePath)
{
    size_t size;
    auto buffer = AllocateBuffer(32);
    ISaveDataDivisionExporter::KeySeed keySeed;
    NN_ABORT_UNLESS_RESULT_SUCCESS(ReadFileAtOnce(&size, &buffer, (backupStoragePath + KeySeedFileName).c_str()));
    NN_ABORT_UNLESS_EQUAL(size, ISaveDataDivisionExporter::KeySeed::Size);
    memcpy(keySeed.data, buffer.get(), keySeed.Size);
    return keySeed;
}

ISaveDataDivisionExporter::InitialDataMac GetInitialDataMac(String backupStoragePath)
{
    size_t size;
    auto buffer = AllocateBuffer(32);
    ISaveDataDivisionExporter::InitialDataMac mac;
    NN_ABORT_UNLESS_RESULT_SUCCESS(ReadFileAtOnce(&size, &buffer, (backupStoragePath + InitialDataMacFileName).c_str()));
    NN_ABORT_UNLESS_EQUAL(size, ISaveDataDivisionExporter::InitialDataMac::Size);
    memcpy(mac.data, buffer.get(), mac.Size);
    return mac;
}

// バックアップストレージ上のチャンクの書き出し・読み込みヘルパクラス
class ChunkAccessor
{
public:
    struct Buffer {

        Buffer(decltype(AllocateBuffer(0)) buffer, size_t validDataSize)
            : buffer(std::move(buffer))
            , validDataSize(validDataSize)
        {}

    public:
        decltype(AllocateBuffer(0)) buffer;
        size_t validDataSize;
    };


public:
    ChunkAccessor(String backupStoragePath, SaveDataChunkId id)
    {
        auto name = GenerateStorageFileName(id);
        NN_ABORT_UNLESS_RESULT_SUCCESS(OpenFile(&m_Handle, (backupStoragePath + name).c_str(), OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend));
    }

    ~ChunkAccessor()
    {
        CloseFile(m_Handle);
    }

    Buffer Pull(size_t size)
    {
        auto buffer = AllocateBuffer(size);
        NN_ABORT_UNLESS_NOT_NULL(buffer.get());

        size_t readSize;
        NN_ABORT_UNLESS_RESULT_SUCCESS(ReadFile(&readSize, m_Handle, m_Offset, buffer.get(), size, ReadOption()));

        Buffer resultBuffer(std::move(buffer), readSize);
        m_Offset += readSize;

        return resultBuffer;
    }

    void Push(Buffer buffer, size_t size)
    {
        NN_UNUSED(buffer);
        NN_UNUSED(size);
        NN_ABORT(); // not implemented
    }

private:
    FileHandle m_Handle;
    int64_t m_Offset = 0;
};


// 先頭から順に書き込まれたデータの sha256 を計算するストレージ
class Sha256Storage : public IStorage
{
public:
    explicit Sha256Storage(int64_t size) NN_NOEXCEPT
        : m_Size(size)
    {
        m_Generator.Initialize();
    }

    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(offset);
        NN_UNUSED(buffer);
        NN_UNUSED(size);
        NN_ABORT();
    }

    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_ABORT_UNLESS(offset == m_Offset);
        m_Generator.Update(buffer, size);
        m_Offset += size;
        NN_RESULT_SUCCESS;
    }

    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_ABORT();
    }

    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        *outValue = m_Size;
        NN_RESULT_SUCCESS;
    }

public:
    void GetHash(nnt::fs::util::Hash* pOutValue)
    {
        m_Generator.GetHash(pOutValue, sizeof(nnt::fs::util::Hash));
    }

private:
    int64_t m_Offset = 0;
    const int64_t m_Size;
    crypto::Sha256Generator m_Generator;
};


// エクスポートされたチャンクデータファイルの暗号化・圧縮をほどいて生のデータの sha256 を取り出す
void ExtractAndCalculateHashOfChunkRawData(
    Hash* pOutValue,
    String backupStoragePath,
    SaveDataChunkId id,
    const fssrv::SaveDataTransferCryptoConfiguration& configuration,
    const fssrv::detail::SaveDataTransferManagerVersion2::KeySeedPackage::Content& keySeedPackage,
    int64_t size
) NN_NOEXCEPT
{
    fssrv::detail::SaveDataChunkImporter importer(configuration, keySeedPackage);

    auto sha256Storage = fssystem::AllocateShared<Sha256Storage>(size);
    NN_ASSERT_NOT_NULL(sha256Storage);

    NNT_ASSERT_RESULT_SUCCESS(importer.Initialize(sha256Storage, size));

    const size_t BufferSize = 4 * 1024 * 1024;

    ChunkAccessor chunk(backupStoragePath, id);
    while (true)
    {
        auto buffer = chunk.Pull(BufferSize);
        if (buffer.validDataSize == 0)
        {
            break;
        }
        NNT_ASSERT_RESULT_SUCCESS(importer.Push(nn::sf::InBuffer(static_cast<const char*>(buffer.buffer.get()), buffer.validDataSize), buffer.validDataSize));
    }

    sha256Storage->GetHash(pOutValue);
}


// InitialData の復号・検証
void DecryptAndVerifyInitialData(
    fssrv::detail::InitialDataVersion2Detail::Content* pOutValue,
    String backupStoragePath,
    const fssrv::SaveDataTransferCryptoConfiguration& configuration,
    const fssrv::detail::SaveDataTransferManagerVersion2::KeySeedPackage::Content& keySeedPackage
) NN_NOEXCEPT
{
    InitialDataForCloudBackUp encryptedInitialData;
    ReadInitialData(&encryptedInitialData, backupStoragePath);

    fssrv::detail::InitialDataVersion2Detail untrustedInitialData;
    memcpy(&untrustedInitialData, &encryptedInitialData, sizeof(untrustedInitialData));


    fssrv::detail::InitialDataVersion2Detail::Content initialData;

    // AES-GCM 復号・検証
    {
        // TODO: 正式な鍵
        // TODO: 鍵世代
        char key[16];
        configuration.pGenerateAesKey(key, sizeof(key), fssrv::SaveDataTransferCryptoConfiguration::KeyIndex::CloudBackUpData, keySeedPackage.keySeed, sizeof(keySeedPackage.keySeed));

        char mac[crypto::Aes128GcmDecryptor::MacSize];
        auto decryptedSize = crypto::DecryptAes128Gcm(
            &initialData, sizeof(fssrv::detail::InitialDataVersion2Detail::Content),
            &mac, sizeof(mac),
            key, sizeof(key),
            untrustedInitialData.aesGcmStreamHeader.iv, sizeof(untrustedInitialData.aesGcmStreamHeader.iv),
            &untrustedInitialData.content, sizeof(untrustedInitialData.content),
            nullptr, 0
        );
        NN_UNUSED(decryptedSize);

        EXPECT_TRUE(crypto::IsSameBytes(mac, untrustedInitialData.aesGcmStreamTail.mac, sizeof(mac)));
    }

    // ksp の cmac と検証
    {
        // TODO: 正式な鍵
        // TODO: 鍵世代
        char key[16];
        configuration.pGenerateAesKey(key, sizeof(key), fssrv::SaveDataTransferCryptoConfiguration::KeyIndex::CloudBackUpToken, nullptr, 0);

        char mac[crypto::Aes128CmacGenerator::MacSize];
        crypto::GenerateAes128Cmac(mac, sizeof(mac), &initialData, sizeof(fssrv::detail::InitialDataVersion2Detail::Content), key, sizeof(key));
#if 0
        NN_RESULT_THROW_UNLESS(crypto::IsSameBytes(mac, m_KeySeedPackage.initialDataMac, sizeof(mac)), fs::ResultSaveDataTransferInitialDataKeySeedPackageMacVerificationFailed());
#else
        if (!crypto::IsSameBytes(mac, keySeedPackage.initialDataMac, sizeof(mac)))
        {
            NN_SDK_LOG("fs::ResultSaveDataTransferInitialDataKeySeedPackageMacVerificationFailed() (ignored)\n");
        }
#endif
    }

    *pOutValue = initialData;
}


// バックアップの initialData, チャンクのハッシュ列を取得
void CalculateBackupHash(
    fssrv::detail::InitialDataVersion2Detail::Content* pOutInitialData,
    std::vector<Hash>* pOutHashArray,
    String backupStoragePath,
    const fssrv::SaveDataTransferCryptoConfiguration& configuration
)
{
    fssrv::detail::SaveDataTransferManagerVersion2::KeySeedPackage::Content keySeedPackage;
    {
        auto keySeed = GetKeySeed(backupStoragePath);
        auto initialDataMac = GetInitialDataMac(backupStoragePath);
        memcpy(keySeedPackage.keySeed, keySeed.data, keySeed.Size);
        memcpy(keySeedPackage.initialDataMac, initialDataMac.data, initialDataMac.Size);
    }

    fssrv::detail::InitialDataVersion2Detail::Content initialData;
    DecryptAndVerifyInitialData(&initialData, backupStoragePath, configuration, keySeedPackage);

    // TODO: initialData の要素の妥当性をテスト

    for (int i = 0; i < initialData.singleSaveData.divisionCount; i++)
    {
        Hash hash;
        auto id = static_cast<SaveDataChunkId>(i + fssrv::detail::IdBaseForDivisionIndex);
        ExtractAndCalculateHashOfChunkRawData(&hash, backupStoragePath, id, configuration, keySeedPackage, initialData.singleSaveData.chunkSize[i]);
        pOutHashArray->push_back(hash);
    }

    memcpy(pOutInitialData, &initialData, sizeof(initialData));
}

void ExportSaveData(int64_t *outExportTotalSize, int *outExportChunkCount, String backupStoragePath, const SaveDataInfo& infoSrc, bool isDiffExport, int divisionCount)
{
    int64_t exportTotalSize = 0;
    int     exportChunkCount = 0;

    SaveDataTransferManagerForCloudBackUp manager;

    std::unique_ptr<ISaveDataDivisionExporter> exporter;

    if (isDiffExport)
    {
        SaveDataTransferManagerForCloudBackUp::Challenge challenge;
        NNT_ASSERT_RESULT_SUCCESS(manager.GetChallenge(&challenge));

        auto keySeed        = GetKeySeed(backupStoragePath);
        auto initialDataMac = GetInitialDataMac(backupStoragePath);

        auto token = GetKeySeedPackageFromServer(challenge, keySeed, initialDataMac);
        NNT_ASSERT_RESULT_SUCCESS(manager.SetToken(token));

        InitialDataForCloudBackUp initialData;
        ReadInitialData(&initialData, backupStoragePath);

        // ここでサーバにセーブの duplicate をリクエスト

        NNT_ASSERT_RESULT_SUCCESS(manager.OpenSaveDataDiffExporter(&exporter, initialData, infoSrc.saveDataSpaceId, infoSrc.saveDataId));

        // Aad を検証
        {
            nn::fs::ISaveDataDivisionExporter::InitialDataAad aad1;
            exporter->GetImportInitialDataAad(&aad1);

            nn::fs::ISaveDataDivisionExporter::InitialDataAad aad2;
            GenerateAadForTest(&aad2);

            NNT_FS_UTIL_EXPECT_MEMCMPEQ(aad1.data, aad2.data, nn::fs::detail::InitialDataAad::Size);
        }
    }
    else
    {
        // ここでサーバにセーブ新規作成をリクエスト

        NNT_ASSERT_RESULT_SUCCESS(manager.OpenSaveDataFullExporter(&exporter, infoSrc.saveDataSpaceId, infoSrc.saveDataId));
    }

    // Aad を設定
    {
        nn::fs::ISaveDataDivisionExporter::InitialDataAad aad;
        GenerateAadForTest(&aad);
        exporter->SetExportInitialDataAad(aad);
    }

    // exporter を開いた状態でのコミット ID の取得
    {
        SaveDataCommitId commitId;
        NNT_EXPECT_RESULT_SUCCESS(GetSaveDataCommitId(&commitId, infoSrc.saveDataSpaceId, infoSrc.saveDataId));
        NN_LOG("CommitId: %lld\n", commitId);
        // TODO: 差し替わり検知のため fs に渡す？
    }

    // 現セーブサイズ、前回の分割数、サーバからのヒントなどから今回の分割数を決める
    exporter->SetDivisionCount(divisionCount);

    std::unique_ptr<ISaveDataChunkIterator> iter;
    NNT_ASSERT_RESULT_SUCCESS(exporter->OpenSaveDataDiffChunkIterator(&iter));

    // 差分のある sf について（新規なら全 sf について）ループ
    for (; !iter->IsEnd(); iter->Next())
    {
        NN_LOG("chunk %5d :", iter->GetId());
        exportChunkCount++;

        std::unique_ptr<ISaveDataChunkExporter> chunkExporter;
        NNT_ASSERT_RESULT_SUCCESS(exporter->OpenSaveDataChunkExporter(&chunkExporter, iter->GetId()));

        const size_t BufferSize = 1024 * 1024;
        auto buffer = AllocateBuffer(BufferSize);

        // もし UL 開始前にサイズ確定が必要なら空回ししてでも取得
        int64_t size;
        NNT_ASSERT_RESULT_SUCCESS(GetExportSizeByDryRun(&size, nullptr, exporter.get(), iter->GetId(), buffer.get(), BufferSize));

        auto name = GenerateStorageFileName(iter->GetId());
        DeleteFile((backupStoragePath + name).c_str());
        NNT_ASSERT_RESULT_SUCCESS(CreateFile((backupStoragePath + name).c_str(), size));

        FileHandle handle;
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, (backupStoragePath + name).c_str(), OpenMode_Write));
        NN_UTIL_SCOPE_EXIT
        {
            CloseFile(handle);
        };

        int64_t offset = 0;
        while (true)
        {
            size_t pulledSize;
            NNT_ASSERT_RESULT_SUCCESS(chunkExporter->Pull(&pulledSize, buffer.get(), BufferSize));
            if (pulledSize > 0)
            {
                NNT_ASSERT_RESULT_SUCCESS(WriteFile(handle, offset, buffer.get(), pulledSize, WriteOption::MakeValue(WriteOptionFlag_Flush)));
                offset += pulledSize;
                NN_LOG("  pulled %d bytes\n", pulledSize);
                exportTotalSize += pulledSize;
            }
            else
            {
                break;
            }
        }
    }

    // TODO: sf の数が減った場合の削除

    // Finalize して keyseed, initialDataMac を取得
    if (isDiffExport)
    {
        nn::fs::ISaveDataDivisionExporter::InitialDataMac mac;
        NNT_ASSERT_RESULT_SUCCESS(exporter->FinalizeDiffExport(&mac));

        DeleteFile((backupStoragePath + InitialDataMacFileName).c_str());
        NNT_ASSERT_RESULT_SUCCESS(CreateAndWriteFileAtOnce((backupStoragePath + InitialDataMacFileName).c_str(), mac.data, mac.Size));
    }
    else
    {
        nn::fs::ISaveDataDivisionExporter::KeySeed keySeed;
        nn::fs::ISaveDataDivisionExporter::InitialDataMac mac;
        NNT_ASSERT_RESULT_SUCCESS(exporter->FinalizeFullExport(&keySeed, &mac));

        DeleteFile((backupStoragePath + KeySeedFileName).c_str());
        NNT_ASSERT_RESULT_SUCCESS(CreateAndWriteFileAtOnce((backupStoragePath + "keySeed").c_str(), keySeed.data, keySeed.Size));

        DeleteFile((backupStoragePath + InitialDataMacFileName).c_str());
        NNT_ASSERT_RESULT_SUCCESS(CreateAndWriteFileAtOnce((backupStoragePath + InitialDataMacFileName).c_str(), mac.data, mac.Size));
    }


    // UL 完了をコミット
    // server.commitSaveData(newSaveDataFilesId)

    *outExportChunkCount = exportChunkCount;
    *outExportTotalSize = exportTotalSize;
}

void ImportSaveDataCore(int64_t *outImportTotalSize, int *outImportChunkCount, ISaveDataDivisionImporter* importer, String backupStoragePath)
{
    int64_t importTotalSize = 0;
    int     importChunkCount = 0;

    // 必要なら、適用開始前に差分ファイルを全てローカルに保存
    {
        std::unique_ptr<ISaveDataChunkIterator> iter;
        NNT_ASSERT_RESULT_SUCCESS(importer->OpenSaveDataDiffChunkIterator(&iter));

        int64_t requiredTotalStorageSize = 0;

        // 差分のある sf について（新規なら全 sf について）ループ
        for (; !iter->IsEnd(); iter->Next())
        {
            auto name = GenerateStorageFileName(iter->GetId());

            FileHandle handle;
            int64_t size;
            NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, (backupStoragePath + name).c_str(), OpenMode_Read));
            NN_UTIL_SCOPE_EXIT
            {
                CloseFile(handle);
            };
            NNT_ASSERT_RESULT_SUCCESS(GetFileSize(&size, handle));
            requiredTotalStorageSize += size;
        }

        // 空き容量判定・差分 sf をローカルに保存する等
        NN_LOG("Required size: %lld\n", requiredTotalStorageSize);
    }

    // TBD: いったん importer を解放してアンロックし、差分 sf を DL し終えてから importer を再オープンしてインポートするユースケースに対応する場合、
    // 更新が間に挟まっていないことをチェックするためのコンテキスト出力・入力が必要

    // ここからセーブデータを破壊的に拡張・更新、または作成
    int64_t restSize;
    do
    {
        NNT_ASSERT_RESULT_SUCCESS(importer->InitializeImport(&restSize, 512 * 1024));
        NN_LOG("InitializeImport: restSize: %d\n", restSize);
    } while (restSize > 0);

    // Aad を検証
    {
        nn::fs::ISaveDataDivisionExporter::InitialDataAad aad1;
        importer->GetImportInitialDataAad(&aad1);

        nn::fs::ISaveDataDivisionExporter::InitialDataAad aad2;
        GenerateAadForTest(&aad2);

        NNT_FS_UTIL_EXPECT_MEMCMPEQ(aad1.data, aad2.data, nn::fs::detail::InitialDataAad::Size);
    }

    {
        std::unique_ptr<ISaveDataChunkIterator> iter;
        NNT_ASSERT_RESULT_SUCCESS(importer->OpenSaveDataDiffChunkIterator(&iter));

        // 差分のある sf について（新規なら全 sf について）ループ
        for (; !iter->IsEnd(); iter->Next())
        {
            NN_LOG("chunk %5d :", iter->GetId());
            importChunkCount++;
            std::unique_ptr<ISaveDataChunkImporter> chunkImporter;
            NNT_ASSERT_RESULT_SUCCESS(importer->OpenSaveDataChunkImporter(&chunkImporter, iter->GetId()));

            auto name = GenerateStorageFileName(iter->GetId());

            const size_t BufferSize = 1024 * 1024;
            auto buffer = AllocateBuffer(BufferSize);
            FileHandle handle;
            NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, (backupStoragePath + name).c_str(), OpenMode_Read));
            int64_t offset = 0;
            while (true)
            {
                size_t readSize;
                NNT_ASSERT_RESULT_SUCCESS(ReadFile(&readSize, handle, offset, buffer.get(), BufferSize, ReadOption()));

                if (readSize > 0)
                {
                    NN_LOG("  push %d bytes\n", readSize);
                    importTotalSize += readSize;
                    NNT_ASSERT_RESULT_SUCCESS(chunkImporter->Push(buffer.get(), readSize));
                }
                else
                {
                    break;
                }

                offset += readSize;
            }
            CloseFile(handle);

        }
    }

    NNT_ASSERT_RESULT_SUCCESS(importer->FinalizeImport());

    *outImportChunkCount = importChunkCount;
    *outImportTotalSize = importTotalSize;

}

void ImportSaveDataDiff(int64_t *outImportTotalSize, int *outImportChunkCount, String backupStoragePath, const SaveDataInfo& infoDst, bool duplicate)
{
    SaveDataTransferManagerForCloudBackUp manager;
    SaveDataTransferManagerForCloudBackUp::Challenge challenge;
    NNT_ASSERT_RESULT_SUCCESS(manager.GetChallenge(&challenge));

    auto keySeed = GetKeySeed(backupStoragePath);
    auto initialDataMac = GetInitialDataMac(backupStoragePath);
    auto ksp = GetKeySeedPackageFromServer(challenge, keySeed, initialDataMac);
    NNT_ASSERT_RESULT_SUCCESS(manager.SetKeySeedPackage(ksp));

    InitialDataForCloudBackUp initialData;
    ReadInitialData(&initialData, backupStoragePath);

    std::unique_ptr<ISaveDataDivisionImporter> importer;
    if (duplicate)
    {
        NNT_ASSERT_RESULT_SUCCESS(manager.OpenSaveDataDuplicateDiffImporter(&importer, initialData, infoDst.saveDataSpaceId, infoDst.saveDataId));
    }
    else
    {
        NNT_ASSERT_RESULT_SUCCESS(manager.OpenSaveDataDiffImporter(&importer, initialData, infoDst.saveDataSpaceId, infoDst.saveDataId));
    }

    // importer を開いた状態でのコミット ID の取得
    {
        SaveDataCommitId commitId;
        NNT_EXPECT_RESULT_SUCCESS(GetSaveDataCommitId(&commitId, infoDst.saveDataSpaceId, infoDst.saveDataId));
        NN_LOG("CommitId: %lld\n", commitId);
        // TODO: 差し替わり検知のため fs に渡す？
    }

    ImportSaveDataCore(outImportTotalSize, outImportChunkCount, importer.get(), backupStoragePath);
}

void ImportSaveData(int64_t *outImportTotalSize, int *outImportChunkCount, String backupStoragePath, SaveDataSpaceId spaceId, const UserId& userId)
{
    SaveDataTransferManagerForCloudBackUp manager;
    SaveDataTransferManagerForCloudBackUp::Challenge challenge;
    NNT_ASSERT_RESULT_SUCCESS(manager.GetChallenge(&challenge));
    auto keySeed = GetKeySeed(backupStoragePath);
    auto initialDataMac = GetInitialDataMac(backupStoragePath);
    auto ksp = GetKeySeedPackageFromServer(challenge, keySeed, initialDataMac);
    NNT_ASSERT_RESULT_SUCCESS(manager.SetKeySeedPackage(ksp));

    InitialDataForCloudBackUp initialData;
    ReadInitialData(&initialData, backupStoragePath);

    std::unique_ptr<ISaveDataDivisionImporter> importer;
    NNT_ASSERT_RESULT_SUCCESS(manager.OpenSaveDataFullImporter(&importer, initialData, userId, spaceId));

    ImportSaveDataCore(outImportTotalSize, outImportChunkCount, importer.get(), backupStoragePath);
}


struct DiffImportParam
{
    bool isDuplicate;
};

const DiffImportParam DiffImportParamArray[] =
{
    { true },
    { false },
};

// DuplicateDiff / Diff のバリエーションのための fixture
class CloudBackupDiffImport : public ::testing::TestWithParam<DiffImportParam>
{
};

struct ImportParam
{
    bool isDuplicate;
    bool isDiff;
};

const ImportParam ImportParamArray[] =
{
    { true, true },
    { false, true },
    { false, false },
};

// Import のバリエーションのための fixture
class CloudBackupImport : public ::testing::TestWithParam<ImportParam>
{
};


#if !defined(NN_BUILD_CONFIG_OS_WIN)
// ksp の署名検証が通ること
// @pre
//      - 開発版実機版である
//      - 未 OverrideSaveDataTransferTokenSignVerificationKey() である
TEST(TestKsp, ksp)
{
    SaveDataTransferManagerForCloudBackUp manager;
    SaveDataTransferManagerForCloudBackUp::Challenge challenge;
    NNT_ASSERT_RESULT_SUCCESS(manager.GetChallenge(&challenge));

    char ksp[] =
    {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xe9, 0xd9, 0x8c, 0x77, 0x62, 0x47, 0xcf, 0x67, 0xf3, 0xde, 0x00, 0x19, 0xbc, 0x83, 0xe4, 0x9f,
        0xc1, 0x18, 0x00, 0xec, 0x69, 0x3f, 0x14, 0x02, 0xe6, 0xe0, 0x37, 0x06, 0x92, 0xe6, 0x3c, 0x6a,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0xbb, 0x0d, 0x5d, 0x28, 0x77, 0xae, 0x69, 0x6e, 0x9a, 0xfe, 0xe7, 0x3b, 0x23, 0xbc, 0xcf,
        0xc5, 0xdd, 0xc3, 0xfb, 0xdb, 0x8c, 0x5d, 0xde, 0x89, 0x89, 0xfa, 0x80, 0x30, 0x11, 0x75, 0x2a,
        0x08, 0x4c, 0x2f, 0x43, 0x69, 0x98, 0x3f, 0x54, 0x11, 0x83, 0xe6, 0xba, 0x53, 0x60, 0x3f, 0x20,
        0xb0, 0x20, 0x42, 0x68, 0x6e, 0x79, 0xab, 0x34, 0xc8, 0x05, 0xc5, 0x55, 0x86, 0xad, 0x99, 0x9d,
        0xaa, 0xa6, 0x2d, 0xc4, 0x20, 0x51, 0x35, 0x8b, 0xdd, 0x49, 0xd2, 0x77, 0x16, 0x5b, 0x5c, 0xd8,
        0xca, 0xc3, 0x63, 0xc7, 0x72, 0x81, 0xb9, 0xeb, 0x24, 0x10, 0xeb, 0x62, 0x5c, 0x9e, 0xcf, 0x20,
        0xe9, 0xa1, 0xb0, 0x97, 0x9a, 0x25, 0xc7, 0xf6, 0x89, 0xa8, 0x1a, 0x12, 0x40, 0xc8, 0xc3, 0x6d,
        0xa2, 0x73, 0x4d, 0x92, 0xc5, 0xa7, 0x92, 0xcf, 0xd9, 0xa2, 0xd0, 0xd0, 0x54, 0xde, 0x03, 0x21,
        0xf2, 0x1b, 0xed, 0x44, 0xd1, 0x2d, 0x11, 0xc2, 0xd7, 0x74, 0x9c, 0x08, 0x0a, 0x41, 0xd9, 0x3d,
        0x21, 0x0e, 0x89, 0xcb, 0xd0, 0x26, 0x97, 0x61, 0x6d, 0x43, 0x1a, 0xbd, 0x76, 0xfb, 0xd8, 0xc0,
        0x9e, 0x7a, 0xf4, 0xe5, 0x24, 0x0d, 0x06, 0xc7, 0x3a, 0x18, 0x31, 0x48, 0x2c, 0xcf, 0x64, 0x04,
        0x10, 0x48, 0xdb, 0x30, 0x49, 0xb0, 0xbb, 0x6f, 0xa6, 0x0b, 0x75, 0x7a, 0x98, 0x55, 0x91, 0xcc,
        0x4a, 0xa3, 0xf0, 0xd8, 0xef, 0xea, 0x29, 0xde, 0xd7, 0xc2, 0xf6, 0x4e, 0x81, 0x37, 0xa6, 0x22,
        0xee, 0x0f, 0x50, 0x5f, 0x26, 0xb2, 0x80, 0x4b, 0xa7, 0x8a, 0x6d, 0x80, 0x02, 0x47, 0xe2, 0xf1,
        0x0d, 0x28, 0x5a, 0xd9, 0x60, 0x93, 0x96, 0x16, 0xfc, 0xf8, 0x09, 0x6d, 0xa6, 0x2f, 0xa0, 0x73,
        0xad, 0x09, 0x4e, 0xa1, 0xd8, 0xf7, 0x12, 0x8c, 0x5b, 0x15, 0xfc, 0xfd, 0xc9, 0x8c, 0xe7, 0x41,
        0xb3, 0x0e, 0x75, 0x73, 0xe2, 0xd8, 0x23, 0x79, 0x66, 0x4f, 0x0e, 0x57, 0xa3, 0xe3, 0x7b, 0xa2,
        0x78, 0x05, 0x73, 0x9f, 0x16, 0xc5, 0xdc, 0x67, 0x03, 0x5c, 0xaa, 0xfb, 0xa4, 0x50, 0x6a, 0xab,
        0x90, 0x2d, 0x7a, 0x74, 0x63, 0xa6, 0xf9, 0x00, 0x00, 0x43, 0x1c, 0xbb, 0x25, 0xab, 0xd5, 0xbf,
        0x41, 0x0d, 0xc2, 0x43, 0xa4, 0x31, 0x70, 0x2b, 0x67, 0xd8, 0x8e, 0x34, 0xd6, 0xb8, 0x78, 0x2e,
        0xfa, 0xe9, 0xd0, 0x8b, 0x2e, 0x00, 0x10, 0x51, 0xd5, 0x44, 0x0c, 0xce, 0x04, 0x07, 0xbd, 0x34,
        0x06, 0xa7, 0x46, 0xbf, 0x5a, 0x3c, 0xbf, 0x35, 0x26, 0x2c, 0xd3, 0x92, 0x07, 0xec, 0xdc, 0x2d,
        0x08, 0x0d, 0x87, 0xc5, 0xac, 0xf1, 0x9c, 0xf3, 0x3d, 0xb1, 0x8f, 0xd0, 0x8d, 0xf7, 0x15, 0x78,
        0x20, 0xd3, 0xdc, 0x5a, 0xb8, 0xc0, 0x7d, 0xc5, 0xb7, 0xe0, 0xcc, 0x8e, 0x55, 0x1e, 0x01, 0xda,
    };

    SaveDataTransferManagerForCloudBackUp::KeySeedPackage kspBuffer;
    memcpy(&kspBuffer, ksp, 512);

#if 0 // 一時的に challenge 検証無効化
    NNT_EXPECT_RESULT_SUCCESS(manager.SetToken(kspBuffer));
#else
    // 署名検証は通過、challenge 検証はリプレイ ksp なので失敗
    //NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultSaveDataTransferTokenChallengeVerificationFailed, manager.SetToken(kspBuffer));
#endif

}
#endif



// 新規 export, import, 差分 export, import, 複製差分 import
// （兼サンプル）
TEST(CloudBackup, ExportImportSimple)
{
    DeleteAllTestSaveData();

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

    int divisionCount = 100;

    const size_t WorkBufferSize = 4 * 1024 * 1024;
    auto workBuffer = AllocateBuffer(WorkBufferSize);

    const nn::fs::UserId TestUserId = { { 0, 1 } };
    const nn::fs::UserId TestUserId2 = { { 0, 2 } };

    // エクスポート先ストレージ
    nnt::fs::util::TemporaryHostDirectory tempPath;
    tempPath.Create();

    NNT_ASSERT_RESULT_SUCCESS(MountHost("cloud", tempPath.GetPath().c_str()));
    String backupStoragePath("cloud:/");

    // エクスポート元セーブを生成 (src)
    NNT_ASSERT_RESULT_SUCCESS(CreateAndMountSaveData("save", TestUserId));
    NNT_ASSERT_RESULT_SUCCESS(CreateTestDirectoryTreeRandomly("save:/", 512 * 1024, 8));
    CommitSaveData("save");
    NN_LOG("\n\n");
    NN_LOG("Src: \n");
    ListDirectoryRecursive("save:/");

    nnt::fs::util::Hash hashSrc1;
    NNT_ASSERT_RESULT_SUCCESS(CalculateDirectoryTreeHash(&hashSrc1, "save:/", workBuffer.get(), WorkBufferSize));
    DumpBuffer(&hashSrc1, sizeof(hashSrc1));
    Unmount("save");

    SaveDataSpaceId spaceId = SaveDataSpaceId::User;

    nn::util::optional<SaveDataInfo> infoSrc;
    FindSaveData(&infoSrc, spaceId, [&](SaveDataInfo i) {
        return i.saveDataUserId == TestUserId;
    });


    // エクスポート
    NN_LOG("\n\n");
    NN_LOG("Export (new)\n");
    {
        int exportCount;
        int64_t exportSize;
        ExportSaveData(&exportSize, &exportCount, backupStoragePath, *infoSrc, false, divisionCount);
        EXPECT_EQ(divisionCount + 1, exportCount);
        EXPECT_LT(exportSize, 65 * 1024 * 1024);
    }

    // 別のセーブ (dst) にインポート
    NN_LOG("\n\n");
    NN_LOG("Import (new)\n");
    {
        int importCount;
        int64_t importSize;
        ImportSaveData(&importSize, &importCount, backupStoragePath, SaveDataSpaceId::User, TestUserId2);
        EXPECT_LT(importCount, divisionCount);
        EXPECT_LT(importSize, 5 * 1024 * 1024);
    }

    // dst 確認
    {
        nn::util::optional<SaveDataInfo> infoDst;
        FindSaveData(&infoDst, spaceId, [&](SaveDataInfo i) {
            return i.saveDataUserId == TestUserId2;
        });
        EXPECT_TRUE(infoDst != util::nullopt);

        NNT_ASSERT_RESULT_SUCCESS(CreateAndMountSaveData("save", TestUserId2));
        NN_LOG("\n\n");
        NN_LOG("Imported: \n");
        ListDirectoryRecursive("save:/");
        nnt::fs::util::Hash hashDst1;
        NNT_ASSERT_RESULT_SUCCESS(CalculateDirectoryTreeHash(&hashDst1, "save:/", workBuffer.get(), WorkBufferSize));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(&hashSrc1, &hashDst1, sizeof(hashDst1));
        Unmount("save");
    }

    // src 更新
    nnt::fs::util::Hash hashSrc2;
    {
        NNT_ASSERT_RESULT_SUCCESS(CreateAndMountSaveData("save", TestUserId));
        NNT_ASSERT_RESULT_SUCCESS(CreateFileWith32BitCount("save:/___additional_file2", 1024 * 1024, 0));
        CommitSaveData("save");
        NN_LOG("\n\n");
        NN_LOG("Updated: \n");
        ListDirectoryRecursive("save:/");
        NNT_ASSERT_RESULT_SUCCESS(CalculateDirectoryTreeHash(&hashSrc2, "save:/", workBuffer.get(), WorkBufferSize));
        EXPECT_FALSE(crypto::IsSameBytes(&hashSrc1, &hashSrc2, sizeof(hashSrc1)));
        DumpBuffer(&hashSrc2, sizeof(hashSrc2));
        Unmount("save");
    }

    // 差分エクスポート
    NN_LOG("\n\n");
    NN_LOG("Export (diff)\n");
    {
        int exportCount;
        int64_t exportSize;
        ExportSaveData(&exportSize, &exportCount, backupStoragePath, *infoSrc, true, divisionCount);
        EXPECT_LT(exportCount, divisionCount + 1);
        EXPECT_LT(exportSize, 65 * 1024 * 1024);
    }

    nn::util::optional<SaveDataInfo> infoDst;
    FindSaveData(&infoDst, spaceId, [&](SaveDataInfo i) {
        return i.saveDataUserId == TestUserId2;
    });

    // 差分インポート
    NN_LOG("Import (diff)\n");
    NN_LOG("\n\n");
    {
        int importCount;
        int64_t importSize;
        ImportSaveDataDiff(&importSize, &importCount, backupStoragePath, *infoDst, false);
        EXPECT_LT(importCount, divisionCount + 1);
        EXPECT_LT(importSize, 65 * 1024 * 1024);
    }

    // dst 確認
    {
        NNT_ASSERT_RESULT_SUCCESS(CreateAndMountSaveData("save", TestUserId2));
        NN_LOG("\n\n");
        NN_LOG("Imported: \n");
        ListDirectoryRecursive("save:/");
        nnt::fs::util::Hash hashDst2;
        NNT_ASSERT_RESULT_SUCCESS(CalculateDirectoryTreeHash(&hashDst2, "save:/", workBuffer.get(), WorkBufferSize));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(&hashSrc2, &hashDst2, sizeof(hashDst2));
        Unmount("save");
    }

    // src さらに更新
    nnt::fs::util::Hash hashSrc3;
    {
        NNT_ASSERT_RESULT_SUCCESS(CreateAndMountSaveData("save", TestUserId));
        NNT_ASSERT_RESULT_SUCCESS(CreateFileWith32BitCount("save:/___additional_file3", 1024 * 1024, 0));
        CommitSaveData("save");
        NN_LOG("\n\n");
        NN_LOG("Updated: \n");
        ListDirectoryRecursive("save:/");
        NNT_ASSERT_RESULT_SUCCESS(CalculateDirectoryTreeHash(&hashSrc3, "save:/", workBuffer.get(), WorkBufferSize));
        EXPECT_FALSE(crypto::IsSameBytes(&hashSrc1, &hashSrc3, sizeof(hashSrc1)));
        DumpBuffer(&hashSrc3, sizeof(hashSrc3));
        Unmount("save");
    }

    // (さらに差分エクスポート)
    NN_LOG("\n\n");
    NN_LOG("Export (diff)\n");
    {
        int exportCount;
        int64_t exportSize;
        ExportSaveData(&exportSize, &exportCount, backupStoragePath, *infoSrc, true, divisionCount);
        EXPECT_LT(exportCount, divisionCount + 1);
        EXPECT_LT(exportSize, 65 * 1024 * 1024);
    }

    // 複製差分インポート
    NN_LOG("Import (duplicate diff)\n");
    NN_LOG("\n\n");
    {
        int importCount;
        int64_t importSize;
        ImportSaveDataDiff(&importSize, &importCount, backupStoragePath, *infoDst, true);
        EXPECT_LT(importCount, divisionCount + 1);
        EXPECT_LT(importSize, 65 * 1024 * 1024);
    }

    // dst 確認
    {
        NNT_ASSERT_RESULT_SUCCESS(CreateAndMountSaveData("save", TestUserId2));
        NN_LOG("\n\n");
        NN_LOG("Imported: \n");
        ListDirectoryRecursive("save:/");
        nnt::fs::util::Hash hashDst3;
        NNT_ASSERT_RESULT_SUCCESS(CalculateDirectoryTreeHash(&hashDst3, "save:/", workBuffer.get(), WorkBufferSize));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(&hashSrc3, &hashDst3, sizeof(hashDst3));
        Unmount("save");
    }

    // TODO: 新規スワップインポート

    Unmount("cloud");

} // NOLINT(impl/function_size)


#if 0
// ksp の検証
TEST(CloudBackup, InvalidKsp)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

    // invalid challenge
    {
        SaveDataTransferManagerForCloudBackUp manager;
        SaveDataTransferManagerForCloudBackUp::Challenge challenge;
        NNT_ASSERT_RESULT_SUCCESS(manager.GetChallenge(&challenge));
        challenge.data[0] ^= 0xFF;

        auto ksp = GetTokenFromServer(challenge);
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultSaveDataTransferTokenChallengeVerificationFailed, manager.SetKeySeedPackage(ksp));
    }

    // invalid mac
    {
        SaveDataTransferManagerForCloudBackUp manager;
        SaveDataTransferManagerForCloudBackUp::Challenge challenge;
        NNT_ASSERT_RESULT_SUCCESS(manager.GetChallenge(&challenge));

        // mac 対象の改竄
        auto ksp = GetTokenFromServer(challenge);
        auto pKspImpl = reinterpret_cast<nn::fssrv::detail::SaveDataTransferManagerVersion2::KeySeedPackage*>(&ksp);
        pKspImpl->encryptedArea.signedArea.keySeed[0] ^= 0xFF;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultSaveDataTransferTokenMacVerificationFailed, manager.SetKeySeedPackage(ksp));
    }

    // invalid sign
    {
        SaveDataTransferManagerForCloudBackUp manager;
        SaveDataTransferManagerForCloudBackUp::Challenge challenge;
        NNT_ASSERT_RESULT_SUCCESS(manager.GetChallenge(&challenge));

        // 署名対象の改竄
        auto ksp = GetTokenFromServer(challenge, 0, true);
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultSaveDataTransferTokenSignatureVerificationFailed, manager.SetKeySeedPackage(ksp));
    }
}
#endif


const int UpdateFileArrayTestCaseCountMax = 5;
std::vector<int> GetUpdateFileArray(int divisionCount, int testCaseIndex, int fileCountMax)
{
    std::vector<std::vector<int>> fileArrayArray;

    // 1 チャンク相当のファイル数（概算）
    int fileCountPerChunk = fileCountMax / divisionCount;

    { // 先頭
        std::vector<int> fileArray = { 0 };
        fileArrayArray.push_back(fileArray);
    }
    { // 末尾
        std::vector<int> fileArray = { fileCountMax - 1 };
        fileArrayArray.push_back(fileArray);
    }
    { // 全チャンク
        std::vector<int> fileArray;
        for (int i = 0; i < fileCountMax; i++)
        {
            fileArray.push_back(i);
        }
        fileArrayArray.push_back(fileArray);
    }

    { // 連続複数チャンク (0, 1)
        std::vector<int> fileArray = { 0, fileCountPerChunk };
        fileArrayArray.push_back(fileArray);
    }

#if 0
    { // 連続複数チャンク (1, 2)
        std::vector<int> fileArray = { fileCountPerChunk, fileCountPerChunk * 2 };
        fileArrayArray.push_back(fileArray);
    }

    { // 離散複数チャンク (0, 2)
        std::vector<int> fileArray = { 0, fileCountPerChunk * 2 };
        fileArrayArray.push_back(fileArray);
    }
#endif

    { // 離散複数チャンク (1, 4)
        std::vector<int> fileArray = { fileCountPerChunk, fileCountPerChunk * 4 };
        fileArrayArray.push_back(fileArray);
    }

    NN_ASSERT(UpdateFileArrayTestCaseCountMax == fileArrayArray.size());
    NN_ASSERT(testCaseIndex < static_cast<int>(fileArrayArray.size()));
    return fileArrayArray[testCaseIndex];
}


#if 0
    // Full set
    const int TestParamDivisionCount[] = { 1, 2, 3, 4, 8, 16, 32, 127, 128 };
    const int64_t TestParamSaveDataSize[] =
    {
        48 * 1024,
        64 * 1024,
        1024 * 1024,
        16 * 1024 * 1024,
        64 * 1024 * 1024,
        2000ULL * 1024 * 1024,
    };
    auto TestParamUpdateFileArrayTestCaseIndex = testing::Range(0, UpdateFileArrayTestCaseCountMax);
#else
    const int TestParamDivisionCount[] = { 1, 3, 16, 128 };
    const int64_t TestParamSaveDataSize[] =
    {
        48 * 1024,
        1024 * 1024,
    };
    const int64_t TestParamSaveDataSizeHeavy[] =
    {
        64 * 1024 * 1024,
        // 2000ULL * 1024 * 1024,
    };
#endif


class CloudBackupDiffExportVsFullExport : public ::testing::TestWithParam<int64_t>
{
};

// 48KB ~ 1MB
INSTANTIATE_TEST_CASE_P(CloudBackupDiffExportVsFullExport,
    CloudBackupDiffExportVsFullExport,
    testing::ValuesIn(TestParamSaveDataSize)
);

// 64MB ~ 4GB
INSTANTIATE_TEST_CASE_P(CloudBackupDiffExportVsFullExportHeavy,
    CloudBackupDiffExportVsFullExport,
    testing::ValuesIn(TestParamSaveDataSizeHeavy)
);


// 様々な条件で差分エクスポートの結果が新規エクスポートと同じバックアップ状況になること
// TODO: 組み合わせの見直し、エクスポート回数削減などテスト時間の改善
TEST_P(CloudBackupDiffExportVsFullExport, VerifyDiffExportData)
{
    // TODO: testing::Combine を利用
    auto AvailableSize = GetParam();
    for (auto DivisionCount : TestParamDivisionCount)
    {
        for (auto UpdateFileTestCaseIndex = 0; UpdateFileTestCaseIndex < UpdateFileArrayTestCaseCountMax; UpdateFileTestCaseIndex++)
        {
            NN_SDK_LOG("---------------------------------\n");
            NN_SDK_LOG("DivisionCount: %d\n", DivisionCount);
            NN_SDK_LOG("AvailableSize, JournalSize: %lld\n", AvailableSize);
            NN_SDK_LOG("UpdateFileTestCaseIndex: %d\n", UpdateFileTestCaseIndex);

            const int64_t JournalSize = AvailableSize;
            const int64_t BlockSize = 16 * 1024;
            const int64_t FileSize = std::max(BlockSize, nn::util::align_up((AvailableSize - BlockSize * 2) / 256, BlockSize));
            const int FileCountMax = std::min(static_cast<int>((AvailableSize - BlockSize * 2) / FileSize), 256);

            auto& configuration = *GetSaveDataTransferCryptoConfiguration(false);


            DeleteAllTestSaveData();
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));


            const nn::fs::UserId TestUserId = { { 0, 1 } };

            nnt::fs::util::TemporaryHostDirectory tempPathDiffExport;
            tempPathDiffExport.Create();
            NNT_ASSERT_RESULT_SUCCESS(MountHost("cloudD", tempPathDiffExport.GetPath().c_str()));
            String backupStoragePathDiffExport("cloudD:/");

            // ～256 個のファイルで埋める
            SaveDataInfo infoSrc;
            {
                NNT_ASSERT_RESULT_SUCCESS(CreateAndFindSaveData(&infoSrc, TestApplicationId0, TestUserId, AvailableSize, JournalSize));
                NNT_ASSERT_RESULT_SUCCESS(MountSaveData("save", TestApplicationId0, TestUserId));
                for (int i = 0; i < FileCountMax; i++)
                {
                    String name;
                    name += "save:/" + ToString(i);
                    NNT_ASSERT_RESULT_SUCCESS(CreateFileWith32BitCount(name.c_str(), FileSize, i));
                }
                CommitSaveData("save");
                Unmount("save");
            }

            {
                SaveDataCommitId commitId;
                GetSaveDataCommitId(&commitId, SaveDataSpaceId::User, infoSrc.saveDataId);
                NN_SDK_LOG("%s %d: %llx\n", __FUNCTION__, __LINE__, commitId);
            }

            // 更新前で新規エクスポート
            {
                int exportCount;
                int64_t exportSize;
                ExportSaveData(&exportSize, &exportCount, backupStoragePathDiffExport, infoSrc, false, DivisionCount);
            }

            // ファイルをテストパラメータに応じて更新
            {
                auto UpdateFileArray = GetUpdateFileArray(DivisionCount, UpdateFileTestCaseIndex, FileCountMax);
                NNT_ASSERT_RESULT_SUCCESS(MountSaveData("save", TestApplicationId0, TestUserId));
                for (auto i : UpdateFileArray)
                {
                    if (i >= FileCountMax)
                    {
                        continue;
                    }

                    String name;
                    name += "save:/" + ToString(i);
                    NNT_ASSERT_RESULT_SUCCESS(WriteFileWith32BitCount(name.c_str(), FileSize, i + 0x1000));
                }
                CommitSaveData("save");
                Unmount("save");
            }

            {
                SaveDataCommitId commitId;
                GetSaveDataCommitId(&commitId, SaveDataSpaceId::User, infoSrc.saveDataId);
                NN_SDK_LOG("%s %d: %llx\n", __FUNCTION__, __LINE__, commitId);
            }

            // 更新後を差分エクスポート
            fssrv::detail::InitialDataVersion2Detail::Content initialDataDiffExport;
            std::vector<Hash> chunkHashArrayDiffExport;
            {
                int exportCount;
                int64_t exportSize;
                ExportSaveData(&exportSize, &exportCount, backupStoragePathDiffExport, infoSrc, true, DivisionCount);

                CalculateBackupHash(&initialDataDiffExport, &chunkHashArrayDiffExport, backupStoragePathDiffExport, configuration);
            }

            // 更新後を別ストレージに新規エクスポート
            nnt::fs::util::TemporaryHostDirectory tempPathFullExport;
            tempPathFullExport.Create();
            NNT_ASSERT_RESULT_SUCCESS(MountHost("cloudF", tempPathFullExport.GetPath().c_str()));
            String backupStoragePathFullExport("cloudF:/");

            fssrv::detail::InitialDataVersion2Detail::Content initialDataFullExport;
            std::vector<Hash> chunkHashArrayFullExport;
            {
                int exportCount;
                int64_t exportSize;
                ExportSaveData(&exportSize, &exportCount, backupStoragePathFullExport, infoSrc, false, DivisionCount);

                CalculateBackupHash(&initialDataFullExport, &chunkHashArrayFullExport, backupStoragePathFullExport, configuration);
            }

            // 一致確認
            {
                // TODO: 値の妥当性チェック

                // mac は iv 変化により変わるので確認しない（mac とデータの一貫性は CalculateBackupHash() 内で検証済み）
                memset(initialDataFullExport.singleSaveData.chunkMac, 0xBB, sizeof(initialDataFullExport.singleSaveData.chunkMac));
                memset(initialDataDiffExport.singleSaveData.chunkMac, 0xBB, sizeof(initialDataDiffExport.singleSaveData.chunkMac));

                //DumpBuffer(&initialDataFullExport.singleSaveData, sizeof(initialDataFullExport.singleSaveData));
                //DumpBuffer(&initialDataDiffExport.singleSaveData, sizeof(initialDataDiffExport.singleSaveData));

                // initialData
                NNT_FS_UTIL_EXPECT_MEMCMPEQ(&initialDataFullExport.singleSaveData, &initialDataDiffExport.singleSaveData, sizeof(initialDataDiffExport.singleSaveData));

                // chunks
                NNT_FS_UTIL_EXPECT_MEMCMPEQ(&chunkHashArrayFullExport[0], &chunkHashArrayDiffExport[0], sizeof(Hash) * chunkHashArrayDiffExport.size());
            }

            Unmount("cloudF");
            Unmount("cloudD");
        }
    }
}



// 空回し時に同じデータが取得できること
// TODO: DiffExport でもテスト
TEST(CloudBackup, GenerateDigestByExportDryRun)
{
    DeleteAllTestSaveData();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

    const int DivisionCount = 2;

    SaveDataInfo infoSrc;
    NNT_ASSERT_RESULT_SUCCESS(CreateAndFindSaveData(&infoSrc));

    SaveDataTransferManagerForCloudBackUp manager;
    std::unique_ptr<ISaveDataDivisionExporter> exporter;
    NNT_ASSERT_RESULT_SUCCESS(manager.OpenSaveDataFullExporter(&exporter, infoSrc.saveDataSpaceId, infoSrc.saveDataId));

    exporter->SetDivisionCount(DivisionCount);

    std::unique_ptr<ISaveDataChunkIterator> iter;
    NNT_ASSERT_RESULT_SUCCESS(exporter->OpenSaveDataDiffChunkIterator(&iter));

    for (; !iter->IsEnd(); iter->Next())
    {
        std::unique_ptr<ISaveDataChunkExporter> chunkExporter;
        NNT_ASSERT_RESULT_SUCCESS(exporter->OpenSaveDataChunkExporter(&chunkExporter, iter->GetId()));

        const size_t BufferSize = 1024 * 1024;
        auto buffer = AllocateBuffer(BufferSize);

        // 2回 export して同じデータが得られること
        int64_t size;
        nnt::fs::util::Hash checkSum[2];
        NNT_ASSERT_RESULT_SUCCESS(GetExportSizeByDryRun(&size, &checkSum[0], exporter.get(), iter->GetId(), buffer.get(), BufferSize));
        NNT_ASSERT_RESULT_SUCCESS(GetExportSizeByDryRun(&size, &checkSum[1], exporter.get(), iter->GetId(), buffer.get(), BufferSize));

        NNT_FS_UTIL_EXPECT_MEMCMPEQ(&checkSum[0], &checkSum[1], sizeof(nnt::fs::util::Hash));
    }
}

// GetRestRawDataSize() で妥当な値が取得できること
// TODO: DiffExport でもテスト
TEST(CloudBackup, GetRestRawDataSize)
{
    DeleteAllTestSaveData();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

    const int DivisionCount = 4;

    SaveDataInfo infoSrc;
    NNT_ASSERT_RESULT_SUCCESS(CreateAndFindSaveData(&infoSrc));
    NNT_ASSERT_RESULT_SUCCESS(MountSaveData("save", TestApplicationId0, TestUserId0));
    NNT_ASSERT_RESULT_SUCCESS(CreateFileWith32BitCount("save:/file", 300 * 1024, 0)); // 一部は有効なデータで埋める
    CommitSaveData("save");
    Unmount("save");

    SaveDataTransferManagerForCloudBackUp manager;
    std::unique_ptr<ISaveDataDivisionExporter> exporter;
    NNT_ASSERT_RESULT_SUCCESS(manager.OpenSaveDataFullExporter(&exporter, infoSrc.saveDataSpaceId, infoSrc.saveDataId));

    exporter->SetDivisionCount(DivisionCount);

    std::unique_ptr<ISaveDataChunkIterator> iter;
    NNT_ASSERT_RESULT_SUCCESS(exporter->OpenSaveDataDiffChunkIterator(&iter));

    for (; !iter->IsEnd(); iter->Next())
    {
        std::unique_ptr<ISaveDataChunkExporter> chunkExporter;
        NNT_ASSERT_RESULT_SUCCESS(exporter->OpenSaveDataChunkExporter(&chunkExporter, iter->GetId()));

        auto totalSize = chunkExporter->GetRestRawDataSize();
        if (iter->GetId() == SaveDataChunkIdForInitialData)
        {
            EXPECT_EQ(sizeof(fssrv::detail::InitialDataVersion2Detail::Content), totalSize);
        }
        else
        {
            // とりあえず決め打ちでチェック：avialable size = 1MB のセーブの internalStorageFs の 4 分割サイズ
            EXPECT_EQ(331818, totalSize);
        }

        const size_t BufferSize = 256;
        auto buffer = AllocateBuffer(BufferSize);

        int64_t lastRestSize = totalSize;
        while (NN_STATIC_CONDITION(true))
        {
            size_t pulledSize;
            NNT_ASSERT_RESULT_SUCCESS(chunkExporter->Pull(&pulledSize, buffer.get(), BufferSize));
            auto restSize = chunkExporter->GetRestRawDataSize();

            // 今回値 <= 前回値 であること、完了時 0 になっていること
            EXPECT_LE(restSize, lastRestSize);
            lastRestSize = restSize;
            if (pulledSize < BufferSize)
            {
                EXPECT_EQ(0, restSize);
                break;
            }
        }

    }
}


// InitialData の改竄を検知できること
TEST(CloudBackup, InvalidInitialData)
{
    DeleteAllTestSaveData();

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

    int divisionCount = 2;

    const nn::fs::UserId TestUserId = { { 0, 1 } };
    const nn::fs::UserId TestUserId2 = { { 0, 2 } };

    // エクスポート先ストレージ
    nnt::fs::util::TemporaryHostDirectory tempPath;
    tempPath.Create();

    NNT_ASSERT_RESULT_SUCCESS(MountHost("cloud", tempPath.GetPath().c_str()));
    String backupStoragePath("cloud:/");

    // エクスポート元セーブを生成 (src)
    NNT_ASSERT_RESULT_SUCCESS(CreateAndMountSaveData("save", TestUserId));
    NNT_ASSERT_RESULT_SUCCESS(CreateTestDirectoryTreeRandomly("save:/", 512 * 1024, 8));
    CommitSaveData("save");
    Unmount("save");

    SaveDataSpaceId spaceId = SaveDataSpaceId::User;

    nn::util::optional<SaveDataInfo> infoSrc;
    FindSaveData(&infoSrc, spaceId, [&](SaveDataInfo i) {
        return i.saveDataUserId == TestUserId;
    });


    // エクスポート
    {
        int exportCount;
        int64_t exportSize;
        ExportSaveData(&exportSize, &exportCount, backupStoragePath, *infoSrc, false, divisionCount);
    }


    {
        SaveDataTransferManagerForCloudBackUp manager;
        SaveDataTransferManagerForCloudBackUp::Challenge challenge;
        NNT_ASSERT_RESULT_SUCCESS(manager.GetChallenge(&challenge));
        auto keySeed = GetKeySeed(backupStoragePath);
        auto initialDataMac = GetInitialDataMac(backupStoragePath);
        auto ksp = GetKeySeedPackageFromServer(challenge, keySeed, initialDataMac);
        NNT_ASSERT_RESULT_SUCCESS(manager.SetKeySeedPackage(ksp));

        // 改竄
        InitialDataForCloudBackUp initialData;
        ReadInitialData(&initialData, backupStoragePath);
        auto pInitialDataDetail = reinterpret_cast<fssrv::detail::InitialDataVersion2Detail*>(&initialData);
        memset(&pInitialDataDetail->content, 0xAA, 1);

        std::unique_ptr<ISaveDataDivisionImporter> importer;
        std::unique_ptr<ISaveDataDivisionExporter> exporter;

        // 新規インポート
        NNT_ASSERT_RESULT_FAILURE(ResultSaveDataTransferInitialDataMacVerificationFailed, manager.OpenSaveDataFullImporter(&importer, initialData, TestUserId2, spaceId));

        // 差分インポート
        NNT_ASSERT_RESULT_FAILURE(ResultSaveDataTransferInitialDataMacVerificationFailed, manager.OpenSaveDataDiffImporter(&importer, initialData, spaceId, infoSrc->saveDataId));

        // 差分エクスポート
        NNT_ASSERT_RESULT_FAILURE(ResultSaveDataTransferInitialDataMacVerificationFailed, manager.OpenSaveDataDiffExporter(&exporter, initialData, spaceId, infoSrc->saveDataId));
    }

    Unmount("cloud");

}


// ksp と InitialData の差し替えを検知できること
TEST(CloudBackup, KspAndInitialDataConsistency)
{
    DeleteAllTestSaveData();

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

    int divisionCount = 2;

    const size_t WorkBufferSize = 4 * 1024 * 1024;
    auto workBuffer = AllocateBuffer(WorkBufferSize);

    const nn::fs::UserId TestUserId = { { 0, 1 } };
    const nn::fs::UserId TestUserId2 = { { 0, 2 } };

    // エクスポート先ストレージ1
    nnt::fs::util::TemporaryHostDirectory tempPath1;
    tempPath1.Create();
    NNT_ASSERT_RESULT_SUCCESS(MountHost("cloud1", tempPath1.GetPath().c_str()));
    String backupStoragePath1("cloud1:/");

    // エクスポート先ストレージ2
    nnt::fs::util::TemporaryHostDirectory tempPath2;
    tempPath2.Create();
    NNT_ASSERT_RESULT_SUCCESS(MountHost("cloud2", tempPath2.GetPath().c_str()));
    String backupStoragePath2("cloud2:/");

    // エクスポート元セーブを生成 (src1)
    NNT_ASSERT_RESULT_SUCCESS(CreateAndMountSaveData("save", TestUserId));
    NNT_ASSERT_RESULT_SUCCESS(CreateTestDirectoryTreeRandomly("save:/", 512 * 1024, 8));
    CommitSaveData("save");
    Unmount("save");

    // エクスポート元セーブを生成 (src2)
    NNT_ASSERT_RESULT_SUCCESS(CreateAndMountSaveData("save", TestUserId2));
    NNT_ASSERT_RESULT_SUCCESS(CreateTestDirectoryTreeRandomly("save:/", 512 * 1024, 8));
    CommitSaveData("save");
    Unmount("save");


    SaveDataSpaceId spaceId = SaveDataSpaceId::User;

    nn::util::optional<SaveDataInfo> infoSrc1;
    FindSaveData(&infoSrc1, spaceId, [&](SaveDataInfo i) {
        return i.saveDataUserId == TestUserId;
    });

    nn::util::optional<SaveDataInfo> infoSrc2;
    FindSaveData(&infoSrc2, spaceId, [&](SaveDataInfo i) {
        return i.saveDataUserId == TestUserId2;
    });


    // src1 をエクスポート
    {
        int exportCount;
        int64_t exportSize;
        ExportSaveData(&exportSize, &exportCount, backupStoragePath1, *infoSrc1, false, divisionCount);
    }

    // src2 をエクスポート
    {
        int exportCount;
        int64_t exportSize;
        ExportSaveData(&exportSize, &exportCount, backupStoragePath2, *infoSrc2, false, divisionCount);
    }


    // 1 の initialData を使って 2 を操作
    {
        // 2 の ksp
        SaveDataTransferManagerForCloudBackUp manager;
        SaveDataTransferManagerForCloudBackUp::Challenge challenge;
        NNT_ASSERT_RESULT_SUCCESS(manager.GetChallenge(&challenge));
        auto keySeed = GetKeySeed(backupStoragePath2);
        auto initialDataMac = GetInitialDataMac(backupStoragePath2);
        auto ksp = GetKeySeedPackageFromServer(challenge, keySeed, initialDataMac);
        NNT_ASSERT_RESULT_SUCCESS(manager.SetKeySeedPackage(ksp));

        // 1 の initialData
        InitialDataForCloudBackUp initialData;
        ReadInitialData(&initialData, backupStoragePath1);

        std::unique_ptr<ISaveDataDivisionImporter> importer;
        std::unique_ptr<ISaveDataDivisionExporter> exporter;

#if 0
        // 新規インポート
        NNT_EXPECT_RESULT_FAILURE(ResultSaveDataTransferInitialDataKeySeedPackageMacVerificationFailed, manager.OpenSaveDataFullImporter(&importer, initialData, TestUserId2, spaceId));

        // 差分インポート
        NNT_EXPECT_RESULT_FAILURE(ResultSaveDataTransferInitialDataKeySeedPackageMacVerificationFailed, manager.OpenSaveDataDiffImporter(&importer, initialData, spaceId, infoSrc2->saveDataId));

        // 複製差分インポート
        NNT_EXPECT_RESULT_SUCCESS(ResultSaveDataTransferInitialDataKeySeedPackageMacVerificationFailed, manager.OpenSaveDataDuplicateDiffImporter(&importer, initialData, spaceId, infoSrc2->saveDataId));

        // TODO: 新規スワップインポート

        // 差分エクスポート
        NNT_EXPECT_RESULT_FAILURE(ResultSaveDataTransferInitialDataKeySeedPackageMacVerificationFailed, manager.OpenSaveDataDiffExporter(&exporter, initialData, spaceId, infoSrc2->saveDataId));
#else
        // 一時的に検証を無効化
        // TODO: テストを有効化
        // keySeed, initialDataMac を FinalizeExport() から引き継いでいなくても操作できること
        {

            // 新規インポート
            NNT_EXPECT_RESULT_SUCCESS(manager.OpenSaveDataFullImporter(&importer, initialData, TestUserId2, spaceId));
            importer.reset();

            // 差分インポート
            NNT_EXPECT_RESULT_SUCCESS(manager.OpenSaveDataDiffImporter(&importer, initialData, spaceId, infoSrc2->saveDataId));
            importer.reset();

            // 差分エクスポート
            NNT_EXPECT_RESULT_SUCCESS(manager.OpenSaveDataDiffExporter(&exporter, initialData, spaceId, infoSrc2->saveDataId));
            exporter.reset();
        }
#endif
    }

    Unmount("cloud1");
    Unmount("cloud2");
}


// TODO:
// KeySeed 違いを検知できること
// データ転送経路の改竄を検知できること
// IV が毎回異なること
// ユーザ ID がつけかわっていること（extradata)
// キャンセルできること、ゴミが残らないこと
// 電断時にゴミが残らないこと
// 差分があるチャンクの処理漏れでエラーが返ること
// Finalize 漏れでエラーが返ること
// 複数回の pull/push
// pull/push のサイズの境界値テスト
// セーブデータサイズ変化・分割数変化時の diff export が full export になること


// 空き領域がゼロフィルされ圧縮でごく小さくなること
// TODO: 念のため DiffExport についてもテスト
TEST(CloudBackup, ExportWithCompression)
{
    DeleteAllTestSaveData();

    int divisionCount = 128;

    const nn::fs::UserId TestUserId = { { 0, 1 } };

    // エクスポート先ストレージ
    nnt::fs::util::TemporaryHostDirectory tempPath;
    tempPath.Create();

    NNT_ASSERT_RESULT_SUCCESS(MountHost("cloud", tempPath.GetPath().c_str()));
    String backupStoragePath("cloud:/");

    // エクスポート元セーブを生成 (src)
    NNT_ASSERT_RESULT_SUCCESS(CreateAndMountSaveData("save", TestUserId));

    // ランダムデータで一度埋めたのち空き領域にする
    {
        const int64_t FileSize = 15 * 1024 * 1024;
        auto buffer = AllocateBuffer(FileSize);
        FillBufferWithRandomValue(buffer.get(), FileSize);

        NNT_ASSERT_RESULT_SUCCESS(CreateAndWriteFileAtOnce("save:/randomData", buffer.get(), FileSize));
        CommitSaveData("save");
        NNT_ASSERT_RESULT_SUCCESS(DeleteFile("save:/randomData"));
        CommitSaveData("save");
        NNT_ASSERT_RESULT_SUCCESS(CreateAndWriteFileAtOnce("save:/randomData", buffer.get(), FileSize));
        CommitSaveData("save");
        NNT_ASSERT_RESULT_SUCCESS(DeleteFile("save:/randomData"));
        CommitSaveData("save");
    }
    Unmount("save");
    SaveDataSpaceId spaceId = SaveDataSpaceId::User;
    nn::util::optional<SaveDataInfo> infoSrc;
    FindSaveData(&infoSrc, spaceId, [&](SaveDataInfo i) {
        return i.saveDataUserId == TestUserId;
    });

    // サムネイルもゼロ埋め
    {
        char thumbnailData[nn::fs::detail::ThumbnailFileSize];
        memset(thumbnailData, 0, sizeof(thumbnailData));
        std::unique_ptr<nn::fs::fsa::IFile> thumbnailFile;
        nn::fs::OpenSaveDataThumbnailFile(&thumbnailFile, infoSrc->applicationId.value, infoSrc->saveDataUserId);
        thumbnailFile->Write(0, thumbnailData, sizeof(thumbnailData), nn::fs::WriteOption());
    }

    // エクスポート
    NN_LOG("Export\n");
    {
        int exportCount;
        int64_t exportSize;
        ExportSaveData(&exportSize, &exportCount, backupStoragePath, *infoSrc, false, divisionCount);
        EXPECT_EQ(divisionCount + 1, exportCount);
        EXPECT_LT(exportSize, 64 * 1024);
    }

    int64_t directorySize;
    NNT_ASSERT_RESULT_SUCCESS(GetDirectorySize(&directorySize, "cloud:/"));
    NN_LOG("Exported total size: %lld\n", directorySize);
    EXPECT_GT(directorySize, 1024);
    EXPECT_LT(directorySize, 64 * 1024); // 15MB -> 50KB 程度になる想定 (@deflate)

    Unmount("cloud");
}


// infoDst に差分インポートして結果の一致確認
void ImportAndVerify(String backupStoragePath, const SaveDataInfo& infoDst, const nnt::fs::util::Hash& hashExpected, bool duplicate)
{
    // dst にインポート
    {
        int importCount;
        int64_t importSize;
        ImportSaveDataDiff(&importSize, &importCount, backupStoragePath, infoDst, duplicate);
    }

    SaveDataInfo newInfoDst;
    NNT_ASSERT_RESULT_SUCCESS(FindSaveDataByApplicationId(&newInfoDst, infoDst.applicationId, infoDst.saveDataUserId));

    // hash 確認
    nnt::fs::util::Hash hashDst;
    CalculateSaveDataHash(&hashDst, newInfoDst);
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(&hashExpected, &hashDst, sizeof(hashDst));
}

// エクスポート元の拡張した経緯の有無で結果が変わらないこと
TEST(CloudBackup, ExporteeExtensionHistory)
{
    DeleteAllTestSaveData();

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

    int divisionCount = 32;

    const size_t WorkBufferSize = 4 * 1024 * 1024;
    auto workBuffer = AllocateBuffer(WorkBufferSize);

    const nn::fs::UserId TestUserId = { { 0, 1 } };
    const nn::fs::UserId TestUserId2 = { { 0, 2 } };

    // エクスポート先ストレージ
    nnt::fs::util::TemporaryHostDirectory tempPath;
    tempPath.Create();

    NNT_ASSERT_RESULT_SUCCESS(MountHost("cloud", tempPath.GetPath().c_str()));
    String backupStoragePath("cloud:/");


    const int64_t TestSaveDataInitialSize = 1 * 1024 * 1024;
    const int64_t TestSaveDataSize = 32 * 1024 * 1024;
    const int64_t TestFileSize = 31 * 1024 * 1024;

    nnt::fs::util::Hash hashSrc1;
    {
        SaveDataInfo infoSrc;

        // 拡張経緯のあるエクスポート元セーブを生成
        NNT_ASSERT_RESULT_SUCCESS(CreateAndExtendSaveData(
            &infoSrc,
            TestApplicationId0,
            TestUserId,
            TestSaveDataInitialSize, TestSaveDataInitialSize,
            TestSaveDataSize, TestSaveDataSize,
            2 // 64
        ));

        // 中身を生成
        NNT_ASSERT_RESULT_SUCCESS(MountSaveData("save", TestApplicationId0, TestUserId));
        NNT_ASSERT_RESULT_SUCCESS(CreateFileWith32BitCount("save:/testFile", TestFileSize, 1));
        CommitSaveData("save");

        NNT_ASSERT_RESULT_SUCCESS(CalculateDirectoryTreeHash(&hashSrc1, "save:/", workBuffer.get(), WorkBufferSize));
        DumpBuffer(&hashSrc1, sizeof(hashSrc1));
        Unmount("save");

        // エクスポート
        {
            int exportCount;
            int64_t exportSize;
            ExportSaveData(&exportSize, &exportCount, backupStoragePath, infoSrc, false, divisionCount);
        }

        // 別のセーブ (dst) にインポート
        {
            int importCount;
            int64_t importSize;
            ImportSaveData(&importSize, &importCount, backupStoragePath, SaveDataSpaceId::User, TestUserId2);
        }

        // dst 確認
        {
            NNT_ASSERT_RESULT_SUCCESS(MountSaveData("save", TestApplicationId0, TestUserId2));
            nnt::fs::util::Hash hashDst1;
            NNT_ASSERT_RESULT_SUCCESS(CalculateDirectoryTreeHash(&hashDst1, "save:/", workBuffer.get(), WorkBufferSize));
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(&hashSrc1, &hashDst1, sizeof(hashDst1));
            Unmount("save");
        }
    }

    DeleteAllTestSaveData();

    {
        SaveDataInfo infoSrc;

        // 拡張経緯の無いエクスポート元セーブを生成
        NNT_ASSERT_RESULT_SUCCESS(CreateAndExtendSaveData(
            &infoSrc,
            TestApplicationId0,
            TestUserId,
            TestSaveDataSize, TestSaveDataSize,
            TestSaveDataSize, TestSaveDataSize,
            0
        ));

        // 同一の中身を生成
        NNT_ASSERT_RESULT_SUCCESS(MountSaveData("save", TestApplicationId0, TestUserId));
        NNT_ASSERT_RESULT_SUCCESS(CreateFileWith32BitCount("save:/testFile", TestFileSize, 1));
        CommitSaveData("save");

        nnt::fs::util::Hash hashSrc2;
        NNT_ASSERT_RESULT_SUCCESS(CalculateDirectoryTreeHash(&hashSrc2, "save:/", workBuffer.get(), WorkBufferSize));
        Unmount("save");

        // エクスポート
        {
            int exportCount;
            int64_t exportSize;
            ExportSaveData(&exportSize, &exportCount, backupStoragePath, infoSrc, false, divisionCount);
        }

        // 別のセーブ (dst) にインポート
        {
            int importCount;
            int64_t importSize;
            ImportSaveData(&importSize, &importCount, backupStoragePath, SaveDataSpaceId::User, TestUserId2);
        }

        // dst 確認
        {
            NNT_ASSERT_RESULT_SUCCESS(MountSaveData("save", TestApplicationId0, TestUserId2));
            nnt::fs::util::Hash hashDst1;
            NNT_ASSERT_RESULT_SUCCESS(CalculateDirectoryTreeHash(&hashDst1, "save:/", workBuffer.get(), WorkBufferSize));
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(&hashSrc1, &hashDst1, sizeof(hashDst1));
            Unmount("save");
        }
    }

    // TODO: 拡張経緯＋DiffExport

    Unmount("cloud");
    DeleteAllTestSaveData();
} // NOLINT(impl/function_size)



// 差分インポート適用先の拡張経緯の有無によらず問題ないこと・結果が同じこと
TEST_P(CloudBackupDiffImport, ImporteeExtensionHistory)
{
    auto param = GetParam();

    DeleteAllTestSaveData();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

    int divisionCount = 32;

    const nn::fs::UserId TestUserId = { { 0, 1 } };
    const nn::fs::UserId TestUserId2 = { { 0, 2 } };

    // エクスポート先ストレージ
    nnt::fs::util::TemporaryHostDirectory tempPath;
    tempPath.Create();

    NNT_ASSERT_RESULT_SUCCESS(MountHost("cloud", tempPath.GetPath().c_str()));
    String backupStoragePath("cloud:/");

    const int64_t TestSaveDataInitialSize = 1 * 1024 * 1024;
    const int64_t TestSaveDataSize = 32 * 1024 * 1024;
    const int64_t TestFileSize = 31 * 1024 * 1024;

    nnt::fs::util::Hash hashSrc1;
    {
        SaveDataInfo infoSrc;

        // エクスポート元セーブを生成
        NNT_ASSERT_RESULT_SUCCESS(CreateAndExtendSaveData(
            &infoSrc,
            TestApplicationId0,
            TestUserId,
            TestSaveDataSize, TestSaveDataSize,
            TestSaveDataSize, TestSaveDataSize,
            0
        ));

        // 中身を生成
        NNT_ASSERT_RESULT_SUCCESS(MountSaveData("save", TestApplicationId0, TestUserId));
        NNT_ASSERT_RESULT_SUCCESS(CreateFileWith32BitCount("save:/testFile", TestFileSize, 1));
        CommitSaveData("save");
        Unmount("save");

        // ハッシュ値計算
        CalculateSaveDataHash(&hashSrc1, infoSrc);

        // エクスポート
        {
            int exportCount;
            int64_t exportSize;
            ExportSaveData(&exportSize, &exportCount, backupStoragePath, infoSrc, false, divisionCount);
        }
    }


    {
        SaveDataInfo infoDst;

        // 拡張経緯のあるインポート先セーブを生成
        NNT_ASSERT_RESULT_SUCCESS(CreateAndExtendSaveData(
            &infoDst,
            TestApplicationId0,
            TestUserId2,
            TestSaveDataInitialSize, TestSaveDataInitialSize,
            TestSaveDataSize, TestSaveDataSize,
            64
        ));

        // インポートしてハッシュ値の一致確認
        ImportAndVerify(backupStoragePath, infoDst, hashSrc1, param.isDuplicate);
    }

    DeleteAllTestSaveData();

    {
        SaveDataInfo infoDst;

        // 拡張経緯のないインポート先セーブを生成
        NNT_ASSERT_RESULT_SUCCESS(CreateAndExtendSaveData(
            &infoDst,
            TestApplicationId0,
            TestUserId2,
            TestSaveDataSize, TestSaveDataSize,
            TestSaveDataSize, TestSaveDataSize,
            0
        ));

        // インポートしてハッシュ値の一致確認
        ImportAndVerify(backupStoragePath, infoDst, hashSrc1, param.isDuplicate);
    }

    Unmount("cloud");
    DeleteAllTestSaveData();
} // NOLINT(impl/function_size)




// 差分インポートで小さくなる・大きくなるケースでも問題ないこと、結果が同じこと
TEST_P(CloudBackupDiffImport, ChangeImporteeSizeByDiffImport)
{
    auto param = GetParam();

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

    int divisionCount = 32;

    const nn::fs::UserId TestUserId = { { 0, 1 } };
    const nn::fs::UserId TestUserId2 = { { 0, 2 } };

    // エクスポート先ストレージ
    nnt::fs::util::TemporaryHostDirectory tempPath;
    tempPath.Create();

    NNT_ASSERT_RESULT_SUCCESS(MountHost("cloud", tempPath.GetPath().c_str()));
    String backupStoragePath("cloud:/");

    DeleteAllTestSaveData();

    const int64_t TestSaveDataInitialSize =  64 * 1024;
    const int64_t TestSaveDataSmallSize   = 128 * 1024;
    const int64_t TestSaveDataFinalSize   = 192 * 1024;
    const int64_t TestSaveDataLargeSize   = 256 * 1024;
    const int64_t TestFileSize            =  96 * 1024;

    nnt::fs::util::Hash hashSrc1;
    SaveDataInfo infoSrc;
    SaveDataExtraData extraDataSrc;
    {
        // エクスポート元セーブを生成
        NNT_ASSERT_RESULT_SUCCESS(CreateAndExtendSaveData(
            &infoSrc,
            TestApplicationId0,
            TestUserId,
            TestSaveDataFinalSize, TestSaveDataFinalSize,
            TestSaveDataFinalSize, TestSaveDataFinalSize,
            0
        ));

        // 中身を生成
        NNT_ASSERT_RESULT_SUCCESS(MountSaveData("save", TestApplicationId0, TestUserId));
        NNT_ASSERT_RESULT_SUCCESS(CreateFileWith32BitCount("save:/testFile", TestFileSize, 1));
        CommitSaveData("save");
        Unmount("save");

        char thumbnailFileHeader[32];
        nnt::fs::util::FillBufferWith8BitCount(thumbnailFileHeader, sizeof(thumbnailFileHeader), 1);
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::WriteSaveDataThumbnailFileHeader(TestApplicationId0.value, TestUserId, thumbnailFileHeader, sizeof(thumbnailFileHeader)));

        // ハッシュ値計算
        CalculateSaveDataHash(&hashSrc1, infoSrc);

        NNT_EXPECT_RESULT_SUCCESS(nn::fs::detail::ReadSaveDataFileSystemExtraData(&extraDataSrc, infoSrc.saveDataSpaceId, infoSrc.saveDataId));

        // エクスポート
        {
            int exportCount;
            int64_t exportSize;
            ExportSaveData(&exportSize, &exportCount, backupStoragePath, infoSrc, false, divisionCount);
        }
    }

    // availableSize, journalSize が 小・一致・大 であるセーブに対してインポート
    struct
    {
        int64_t size;
        int64_t journalSize;
    } testParams[] = {
        { TestSaveDataSmallSize, TestSaveDataSmallSize },
        { TestSaveDataSmallSize, TestSaveDataFinalSize },
        { TestSaveDataSmallSize, TestSaveDataLargeSize },
        { TestSaveDataFinalSize, TestSaveDataSmallSize },
        { TestSaveDataFinalSize, TestSaveDataFinalSize },
        { TestSaveDataFinalSize, TestSaveDataLargeSize },
        { TestSaveDataLargeSize, TestSaveDataSmallSize },
        { TestSaveDataLargeSize, TestSaveDataFinalSize },
        { TestSaveDataLargeSize, TestSaveDataLargeSize },
    };

    for (auto testParam : testParams)
    {
        DeleteAllTestSaveData();

        {
            NN_LOG("import => %lld, %lld\n", testParam.size, testParam.journalSize);

            SaveDataInfo infoDst;

            // インポート先セーブを生成
            NNT_ASSERT_RESULT_SUCCESS(CreateAndExtendSaveData(
                &infoDst,
                TestApplicationId0,
                TestUserId2,
                TestSaveDataInitialSize, TestSaveDataInitialSize,
                testParam.size,          testParam.journalSize,
                4
            ));

            // 適当な中身を生成
            NNT_ASSERT_RESULT_SUCCESS(MountSaveData("save", TestApplicationId0, TestUserId2));
            NNT_ASSERT_RESULT_SUCCESS(CreateFileWith32BitCount("save:/testFile", TestFileSize / 2, 2));
            CommitSaveData("save");
            Unmount("save");

            // インポートしてハッシュ値の一致確認
            ImportAndVerify(backupStoragePath, infoDst, hashSrc1, param.isDuplicate);

            // extra data 確認
            {
                // インポートの結果、再作成が発生した場合は saveDataId が更新される
                // TODO: 引き継ぐように実装する？
                SaveDataInfo newInfoDst;
                NNT_ASSERT_RESULT_SUCCESS(FindSaveDataByApplicationId(&newInfoDst, infoDst.applicationId, infoDst.saveDataUserId));

                SaveDataExtraData extraDataDst;
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::detail::ReadSaveDataFileSystemExtraData(&extraDataDst, newInfoDst.saveDataSpaceId, newInfoDst.saveDataId));
                EXPECT_EQ(newInfoDst.saveDataUserId, extraDataDst.attribute.userId); // user id のみ付け替えられていること
                extraDataDst.attribute.userId = infoSrc.saveDataUserId;
                NNT_FS_UTIL_EXPECT_MEMCMPEQ(&extraDataSrc, &extraDataDst, sizeof(SaveDataExtraData));
            }

            // TODO: メタデータの一致確認
        }
    }

    Unmount("cloud");
    DeleteAllTestSaveData();
} // NOLINT(impl/function_size)


// exporter と 1チャンク目の chunkExporter を開く
void OpenExporter(std::unique_ptr<ISaveDataDivisionExporter>* outExporter, std::unique_ptr<ISaveDataChunkExporter>* outChunkExporter, SaveDataTransferManagerForCloudBackUp& manager, SaveDataInfo& info)
{
    // TODO: 差分エクスポートについてもテスト

    NNT_ASSERT_RESULT_SUCCESS(manager.OpenSaveDataExporter(outExporter, info.saveDataSpaceId, info.saveDataId));
    (*outExporter)->SetDivisionCount(2);

    std::unique_ptr<ISaveDataChunkIterator> iter;
    NNT_ASSERT_RESULT_SUCCESS((*outExporter)->OpenSaveDataDiffChunkIterator(&iter));
    NNT_ASSERT_RESULT_SUCCESS((*outExporter)->OpenSaveDataChunkExporter(outChunkExporter, iter->GetId()));
}

// importer と 1チャンク目の chunkImporter を開く
void OpenImporter(std::unique_ptr<ISaveDataDivisionImporter>* outImporter, std::unique_ptr<ISaveDataChunkImporter>* outChunkImporter, SaveDataChunkId* outChunkId, nn::util::optional<SaveDataInfo>* outInfoWork,
    SaveDataTransferManagerForCloudBackUp& manager, SaveDataInfo& info, nn::fs::UserId uid, String backupStoragePath, bool is_swap=false, bool is_diff=false)
{
    InitialDataForCloudBackUp initialData;
    ReadInitialData(&initialData, backupStoragePath);

    if (is_swap && is_diff)
    {
        NNT_ASSERT_RESULT_SUCCESS(manager.OpenSaveDataDuplicateDiffImporter(outImporter, initialData, info.saveDataSpaceId, info.saveDataId));
    }
    else if(!is_swap && !is_diff)
    {
        NNT_ASSERT_RESULT_SUCCESS(manager.OpenSaveDataFullImporter(outImporter, initialData, nn::account::Uid{ uid._data[0], uid._data[1] }, info.saveDataSpaceId));
    }
    else if(!is_swap && is_diff)
    {
        NNT_ASSERT_RESULT_SUCCESS(manager.OpenSaveDataDiffImporter(outImporter, initialData, info.saveDataSpaceId, info.saveDataId));
    }

    std::unique_ptr<ISaveDataChunkIterator> iter;
    NNT_ASSERT_RESULT_SUCCESS((*outImporter)->OpenSaveDataDiffChunkIterator(&iter));

    int64_t size;
    NNT_ASSERT_RESULT_SUCCESS((*outImporter)->InitializeImport(&size));

    *outChunkId = iter->GetId();
    NNT_ASSERT_RESULT_SUCCESS((*outImporter)->OpenSaveDataChunkImporter(outChunkImporter, *outChunkId));

    // 作業用のセーブデータを取得
    if (is_swap)
    {
        FindSaveData(outInfoWork, info.saveDataSpaceId, [&](SaveDataInfo i) {
            return (i.saveDataUserId == info.saveDataUserId) && (i.applicationId == info.applicationId) && (i.rank == SaveDataRank::Secondary);
        });
    }
    else
    {
        if (is_diff)
        {
            FindSaveData(outInfoWork, info.saveDataSpaceId, [&](SaveDataInfo i) {
                return (i.saveDataUserId == info.saveDataUserId) && (i.applicationId == info.applicationId);
            });
        }
        else
        {
            FindSaveData(outInfoWork, info.saveDataSpaceId, [&](SaveDataInfo i) {
                return (i.saveDataUserId == uid) && (i.applicationId == info.applicationId);
            });
        }
    }
    EXPECT_TRUE(*outInfoWork != util::nullopt);
}

// ストレージから対応するチャンクデータを読み出してインポートする
Result PushChunkData(ISaveDataChunkImporter* chunkImporter, SaveDataChunkId chunkId, String backupStoragePath)
{
    auto name = GenerateStorageFileName(chunkId);

    const size_t BufferSize = 1024 * 1024;
    auto buffer = AllocateBuffer(BufferSize);

    FileHandle handle;
    NNT_EXPECT_RESULT_SUCCESS(OpenFile(&handle, (backupStoragePath + name).c_str(), OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        CloseFile(handle);
    };

    int64_t offset = 0;
    while (true)
    {
        size_t readSize;
        NNT_EXPECT_RESULT_SUCCESS(ReadFile(&readSize, handle, offset, buffer.get(), BufferSize, ReadOption()));

        if (readSize > 0)
        {
            NN_RESULT_DO(chunkImporter->Push(buffer.get(), readSize));
        }
        else
        {
            break;
        }

        offset += readSize;
    }
    NN_RESULT_SUCCESS;
}

// エクスポートの区間禁止
// TODO: DiffExport
TEST(CloudBackup, ProhibitExport)
{
    DeleteAllTestSaveData();

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

    const size_t WorkBufferSize = 4 * 1024 * 1024;
    auto workBuffer = AllocateBuffer(WorkBufferSize);

    const nn::fs::UserId TestUserId = { { 0, 1 } };

    const int64_t TestSaveDataSize = 32 * 1024 * 1024;

    // エクスポート禁止対象のセーブを生成
    SaveDataInfo info;
    NNT_ASSERT_RESULT_SUCCESS(CreateAndExtendSaveData(
        &info,
        TestApplicationId0,
        TestUserId,
        TestSaveDataSize, TestSaveDataSize,
        TestSaveDataSize, TestSaveDataSize,
        0
    ));

    // 無関係のセーブ（別のアプリId）を生成
    SaveDataInfo infoUnrelated;
    NNT_ASSERT_RESULT_SUCCESS(CreateAndExtendSaveData(
        &infoUnrelated,
        TestApplicationId1,
        TestUserId,
        TestSaveDataSize, TestSaveDataSize,
        TestSaveDataSize, TestSaveDataSize,
        0
    ));

    {
        SaveDataTransferManagerForCloudBackUp manager;

        // エクスポータを開く
        std::unique_ptr<ISaveDataDivisionExporter> exporter;
        std::unique_ptr<ISaveDataChunkExporter> chunkExporter;
        OpenExporter(&exporter, &chunkExporter, manager, info);

        // （開いているのでロックされている）
        NNT_EXPECT_RESULT_FAILURE(ResultTargetLocked, DeleteSaveData(SaveDataSpaceId::User, info.saveDataId));

        // 無関係のエクスポータを開く
        std::unique_ptr<ISaveDataDivisionExporter> exporterUnrelated;
        std::unique_ptr<ISaveDataChunkExporter> chunkExporterUnrelated;
        OpenExporter(&exporterUnrelated, &chunkExporterUnrelated, manager, infoUnrelated);

        {
            // TestApplicationId0 についてエクスポートを禁止
            std::unique_ptr<SaveDataTransferProhibiterForCloudBackUp> prohibiter;
            NNT_EXPECT_RESULT_SUCCESS(OpenSaveDataTransferProhibiterForCloudBackUp(&prohibiter, TestApplicationId0));

            // この時点でマウントが通ること（実際は Open 後も通る）
            NNT_EXPECT_RESULT_SUCCESS(MountSaveData("save", TestApplicationId0, TestUserId));
            Unmount("save");

            // （無関係側は引き続きロックされている）
            NNT_EXPECT_RESULT_FAILURE(ResultTargetLocked, DeleteSaveData(SaveDataSpaceId::User, infoUnrelated.saveDataId));

            // エクスポート不可になっていること
            size_t size;
            NNT_EXPECT_RESULT_FAILURE(ResultSaveDataPorterCoreInvalidated, chunkExporter->Pull(&size, workBuffer.get(), WorkBufferSize));

            // 無関係側は可能
            NNT_EXPECT_RESULT_SUCCESS(chunkExporterUnrelated->Pull(&size, workBuffer.get(), WorkBufferSize));

            // エクスポートの開始は禁止
            {
                std::unique_ptr<ISaveDataDivisionExporter> exporterTmp;
                NNT_EXPECT_RESULT_FAILURE(ResultTargetLocked, manager.OpenSaveDataExporter(&exporterTmp, info.saveDataSpaceId, info.saveDataId));
            }

            // 無関係の appId は開始可能
            chunkExporterUnrelated.reset(); // 一旦解放
            exporterUnrelated.reset();
            OpenExporter(&exporterUnrelated, &chunkExporterUnrelated, manager, infoUnrelated);

            // 解禁
            prohibiter.reset();
        }

        // 無効化されたエクスポータは不可のまま
        size_t size;
        NNT_EXPECT_RESULT_FAILURE(ResultSaveDataPorterCoreInvalidated, chunkExporter->Pull(&size, workBuffer.get(), WorkBufferSize));

        // エクスポート開始可
        chunkExporter.reset(); // 一旦解放
        exporter.reset();
        OpenExporter(&exporter, &chunkExporter, manager, info);
        NNT_ASSERT_RESULT_SUCCESS(chunkExporter->Pull(&size, workBuffer.get(), WorkBufferSize));
    }


    // prohibiter によって対象セーブが解放されること
    {
        SaveDataTransferManagerForCloudBackUp manager;
        std::unique_ptr<ISaveDataDivisionExporter> exporter;
        std::unique_ptr<ISaveDataChunkExporter> chunkExporter;
        OpenExporter(&exporter, &chunkExporter, manager, info);

        // ロックされている
        NNT_EXPECT_RESULT_FAILURE(ResultTargetLocked, DeleteSaveData(SaveDataSpaceId::User, info.saveDataId));

        std::unique_ptr<SaveDataExportProhibiter> prohibiter;
        NNT_EXPECT_RESULT_SUCCESS(OpenSaveDataExportProhibiter(&prohibiter, TestApplicationId0));

        // 解放されている
        NNT_EXPECT_RESULT_SUCCESS(DeleteSaveData(SaveDataSpaceId::User, info.saveDataId));
    }

    DeleteAllTestSaveData();
} // NOLINT(impl/function_size)

// 複数の prohibiter によるエクスポートの区間禁止
// TODO: DiffExport
TEST(CloudBackup, ProhibitExportMultiple)
{
    DeleteAllTestSaveData();

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

    const size_t WorkBufferSize = 4 * 1024 * 1024;
    auto workBuffer = AllocateBuffer(WorkBufferSize);

    const nn::fs::UserId TestUserId = { { 0, 1 } };

    const int64_t TestSaveDataSize = 32 * 1024 * 1024;

    // エクスポート禁止対象のセーブを生成
    SaveDataInfo info;
    NNT_ASSERT_RESULT_SUCCESS(CreateAndExtendSaveData(
        &info,
        TestApplicationId0,
        TestUserId,
        TestSaveDataSize, TestSaveDataSize,
        TestSaveDataSize, TestSaveDataSize,
        0
    ));

    {
        SaveDataTransferManagerForCloudBackUp manager;

        {
            std::unique_ptr<SaveDataTransferProhibiterForCloudBackUp> prohibiter[2];
            ncm::ApplicationId appIds[] = { TestApplicationId0, TestApplicationId0 };

            // TestApplicationId0 についてエクスポートを禁止
            // 二重に prohibiter を開けること
            NNT_EXPECT_RESULT_SUCCESS(OpenSaveDataTransferProhibiterForCloudBackUp(prohibiter, appIds, 2));

            // エクスポートの開始は禁止状態
            {
                std::unique_ptr<ISaveDataDivisionExporter> exporterTmp;
                NNT_EXPECT_RESULT_FAILURE(ResultTargetLocked, manager.OpenSaveDataExporter(&exporterTmp, info.saveDataSpaceId, info.saveDataId));
            }

            // 全ての prohibiter を破棄するまで禁止状態
            prohibiter[0].reset();
            {
                std::unique_ptr<ISaveDataDivisionExporter> exporterTmp;
                NNT_EXPECT_RESULT_FAILURE(ResultTargetLocked, manager.OpenSaveDataExporter(&exporterTmp, info.saveDataSpaceId, info.saveDataId));
            }

            // 解禁
            prohibiter[1].reset();
        }

        // エクスポート開始可
        std::unique_ptr<ISaveDataDivisionExporter> exporter;
        std::unique_ptr<ISaveDataChunkExporter> chunkExporter;
        OpenExporter(&exporter, &chunkExporter, manager, info);
        size_t size;
        NNT_ASSERT_RESULT_SUCCESS(chunkExporter->Pull(&size, workBuffer.get(), WorkBufferSize));
    }

    DeleteAllTestSaveData();
} // NOLINT(impl/function_size)


TEST_P(CloudBackupImport, ProhibitImport)
{
    DeleteAllTestSaveData();

    auto param = GetParam();

    bool is_swap = param.isDuplicate;
    bool is_diff = param.isDiff;

    const nn::fs::UserId TestUserId = { { 0, 1 } };
    const nn::fs::UserId TestUserId2 = { { 0, 2 } };

    const int divisionCount = 32;
    const int64_t TestSaveDataSize = 32 * 1024 * 1024;

    // インポート禁止対象のセーブを生成
    SaveDataInfo info;
    NNT_ASSERT_RESULT_SUCCESS(CreateAndExtendSaveData(
        &info,
        TestApplicationId0,
        TestUserId,
        TestSaveDataSize, TestSaveDataSize,
        TestSaveDataSize, TestSaveDataSize,
        0
    ));

    // 無関係のセーブ（別のアプリId）を生成
    SaveDataInfo infoUnrelated;
    NNT_ASSERT_RESULT_SUCCESS(CreateAndExtendSaveData(
        &infoUnrelated,
        TestApplicationId1,
        TestUserId,
        TestSaveDataSize, TestSaveDataSize,
        TestSaveDataSize, TestSaveDataSize,
        0
    ));

    // エクスポート先ストレージ1
    nnt::fs::util::TemporaryHostDirectory tempPath1;
    tempPath1.Create();
    NNT_ASSERT_RESULT_SUCCESS(MountHost("cloud1", tempPath1.GetPath().c_str()));
    String backupStoragePath1("cloud1:/");

    // エクスポート先ストレージ2
    nnt::fs::util::TemporaryHostDirectory tempPath2;
    tempPath2.Create();
    NNT_ASSERT_RESULT_SUCCESS(MountHost("cloud2", tempPath2.GetPath().c_str()));
    String backupStoragePath2("cloud2:/");

    // エクスポート1
    {
        int exportCount;
        int64_t exportSize;
        ExportSaveData(&exportSize, &exportCount, backupStoragePath1, info, false, divisionCount);
    }

    // エクスポート2
    {
        int exportCount;
        int64_t exportSize;
        ExportSaveData(&exportSize, &exportCount, backupStoragePath2, infoUnrelated, false, divisionCount);
    }

    {
        SaveDataTransferManagerForCloudBackUp manager;

        // SetKeySeedPackage
        {
            SaveDataTransferManagerForCloudBackUp::Challenge challenge;
            NNT_ASSERT_RESULT_SUCCESS(manager.GetChallenge(&challenge));
            auto keySeed = GetKeySeed(backupStoragePath1);
            auto initialDataMac = GetInitialDataMac(backupStoragePath1);
            auto ksp = GetKeySeedPackageFromServer(challenge, keySeed, initialDataMac);
            NNT_ASSERT_RESULT_SUCCESS(manager.SetKeySeedPackage(ksp));
        }

        // インポータを開く
        std::unique_ptr<ISaveDataDivisionImporter> importer;
        std::unique_ptr<ISaveDataChunkImporter> chunkImporter;
        SaveDataChunkId chunkId;
        nn::util::optional<SaveDataInfo> infoWork;
        OpenImporter(&importer, &chunkImporter, &chunkId, &infoWork, manager, info, TestUserId2, backupStoragePath1, is_swap, is_diff);
        // テスト用にもう一つ ChunkImporter を開いておく
        std::unique_ptr<ISaveDataChunkImporter> chunkImporter2;
        NNT_ASSERT_RESULT_SUCCESS(importer->OpenSaveDataChunkImporter(&chunkImporter2, chunkId));

        // （開いているのでロックされている）
        {
            NNT_EXPECT_RESULT_FAILURE(ResultTargetLocked, DeleteSaveData(infoWork->saveDataSpaceId, infoWork->saveDataId));
        }

        // 無関係のインポータを開く
        std::unique_ptr<ISaveDataDivisionImporter> importerUnrelated;
        std::unique_ptr<ISaveDataChunkImporter> chunkImporterUnrelated;
        SaveDataChunkId chunkIdUnrelated;
        nn::util::optional<SaveDataInfo> infoWorkUnrelated;
        OpenImporter(&importerUnrelated, &chunkImporterUnrelated, &chunkIdUnrelated, &infoWorkUnrelated, manager, infoUnrelated, TestUserId2, backupStoragePath2, is_swap, is_diff);

        // （開いているのでロックされている）
        {
            NNT_EXPECT_RESULT_FAILURE(ResultTargetLocked, DeleteSaveData(infoWorkUnrelated->saveDataSpaceId, infoWorkUnrelated->saveDataId));
        }

        {
            // TestApplicationId0 についてインポートを禁止
            std::unique_ptr<SaveDataTransferProhibiterForCloudBackUp> prohibiter;
            NNT_EXPECT_RESULT_SUCCESS(OpenSaveDataTransferProhibiterForCloudBackUp(&prohibiter, TestApplicationId0));

            // この時点で解放されていること
            {
                NNT_EXPECT_RESULT_SUCCESS(MountSaveData("save", infoWork->applicationId, infoWork->saveDataUserId));
                Unmount("save");
            }

            // （無関係側は引き続きロックされている）
            {
                NNT_EXPECT_RESULT_FAILURE(ResultTargetLocked, DeleteSaveData(infoWorkUnrelated->saveDataSpaceId, infoWorkUnrelated->saveDataId));
            }

            // インポート不可になっていること
            NNT_EXPECT_RESULT_FAILURE(ResultSaveDataPorterCoreInvalidated, PushChunkData(chunkImporter.get(), chunkId, backupStoragePath1));
            NNT_EXPECT_RESULT_FAILURE(ResultSaveDataTransferDataCorrupted, PushChunkData(chunkImporter.get(), chunkId, backupStoragePath1));

            // 無関係側は可能
            NNT_EXPECT_RESULT_SUCCESS(PushChunkData(chunkImporterUnrelated.get(), chunkIdUnrelated, backupStoragePath2));

            // インポートの開始は禁止
            {
                std::unique_ptr<ISaveDataDivisionImporter> importerTmp;
                InitialDataForCloudBackUp initialDataTmp;
                ReadInitialData(&initialDataTmp, backupStoragePath1);
                nn::account::Uid uid = nn::account::Uid{ TestUserId2._data[0], TestUserId2._data[1] };

                NNT_EXPECT_RESULT_FAILURE(ResultTargetLocked, manager.OpenSaveDataImporter(&importerTmp, initialDataTmp, uid, info.saveDataSpaceId));
            }

            // 無関係の appId は開始可能
            importerUnrelated->CancelImport();
            chunkImporterUnrelated.reset(); // 一旦解放
            importerUnrelated.reset();
            OpenImporter(&importerUnrelated, &chunkImporterUnrelated, &chunkIdUnrelated, &infoWorkUnrelated, manager, infoUnrelated, TestUserId2, backupStoragePath2, is_swap, is_diff);
            NNT_EXPECT_RESULT_SUCCESS(PushChunkData(chunkImporterUnrelated.get(), chunkIdUnrelated, backupStoragePath2));

            // 解禁
            prohibiter.reset();
        }

        // 無効化されたインポータは不可のまま
        NNT_EXPECT_RESULT_FAILURE(ResultSaveDataPorterCoreInvalidated, PushChunkData(chunkImporter2.get(), chunkId, backupStoragePath1));

        // インポート開始可
        importer->CancelImport();
        chunkImporter.reset(); // 一旦解放
        importer.reset();
        OpenImporter(&importer, &chunkImporter, &chunkId, &infoWork, manager, info, TestUserId2, backupStoragePath1, is_swap, is_diff);
        NNT_EXPECT_RESULT_SUCCESS(PushChunkData(chunkImporter.get(), chunkId, backupStoragePath1));
    }

    DeleteAllTestSaveData();

    Unmount("cloud1");
    Unmount("cloud2");
} // NOLINT(impl/function_size)

TEST(CloudBackup, ProhibitImportMultiple)
{
    DeleteAllTestSaveData();

    const nn::fs::UserId TestUserId = { { 0, 1 } };
    const nn::fs::UserId TestUserId2 = { { 0, 2 } };
    const nn::fs::UserId TestUserId3 = { { 0, 3 } };

    const int divisionCount = 32;
    const int64_t TestSaveDataSize = 32 * 1024 * 1024;

    // インポート禁止対象のセーブを生成
    SaveDataInfo info;
    NNT_ASSERT_RESULT_SUCCESS(CreateAndExtendSaveData(
        &info,
        TestApplicationId0,
        TestUserId,
        TestSaveDataSize, TestSaveDataSize,
        TestSaveDataSize, TestSaveDataSize,
        0
    ));

    // エクスポート先ストレージ
    nnt::fs::util::TemporaryHostDirectory tempPath;
    tempPath.Create();
    NNT_ASSERT_RESULT_SUCCESS(MountHost("cloud", tempPath.GetPath().c_str()));
    String backupStoragePath("cloud:/");

    // エクスポート
    {
        int exportCount;
        int64_t exportSize;
        ExportSaveData(&exportSize, &exportCount, backupStoragePath, info, false, divisionCount);
    }

    {
        SaveDataTransferManagerForCloudBackUp manager;

        // SetKeySeedPackage
        {
            SaveDataTransferManagerForCloudBackUp::Challenge challenge;
            NNT_ASSERT_RESULT_SUCCESS(manager.GetChallenge(&challenge));
            auto keySeed = GetKeySeed(backupStoragePath);
            auto initialDataMac = GetInitialDataMac(backupStoragePath);
            auto ksp = GetKeySeedPackageFromServer(challenge, keySeed, initialDataMac);
            NNT_ASSERT_RESULT_SUCCESS(manager.SetKeySeedPackage(ksp));
        }

        {
            // TestApplicationId0 についてエクスポートを禁止
            std::unique_ptr<SaveDataTransferProhibiterForCloudBackUp> prohibiter1;
            NNT_EXPECT_RESULT_SUCCESS(OpenSaveDataTransferProhibiterForCloudBackUp(&prohibiter1, TestApplicationId0));

            // 二重に prohibiter を開けること
            std::unique_ptr<SaveDataTransferProhibiterForCloudBackUp> prohibiter2;
            NNT_EXPECT_RESULT_SUCCESS(OpenSaveDataTransferProhibiterForCloudBackUp(&prohibiter2, TestApplicationId0));

            // インポートの開始は禁止状態
            {
                std::unique_ptr<ISaveDataDivisionImporter> importerTmp;
                InitialDataForCloudBackUp initialDataTmp;
                ReadInitialData(&initialDataTmp, backupStoragePath);
                nn::account::Uid uid = nn::account::Uid{ TestUserId2._data[0], TestUserId2._data[1] };

                NNT_EXPECT_RESULT_FAILURE(ResultTargetLocked, manager.OpenSaveDataImporter(&importerTmp, initialDataTmp, uid, info.saveDataSpaceId));
            }

            // 全ての prohibiter を破棄するまで禁止状態
            prohibiter1.reset();
            {
                std::unique_ptr<ISaveDataDivisionImporter> importerTmp;
                InitialDataForCloudBackUp initialDataTmp;
                ReadInitialData(&initialDataTmp, backupStoragePath);
                nn::account::Uid uid = nn::account::Uid{ TestUserId2._data[0], TestUserId2._data[1] };

                NNT_EXPECT_RESULT_FAILURE(ResultTargetLocked, manager.OpenSaveDataImporter(&importerTmp, initialDataTmp, uid, info.saveDataSpaceId));
            }

            // 解禁
            prohibiter2.reset();
        }

        // インポート開始可
        std::unique_ptr<ISaveDataDivisionImporter> importer;
        std::unique_ptr<ISaveDataChunkImporter> chunkImporter;
        SaveDataChunkId chunkId;
        nn::util::optional<SaveDataInfo> infoWork;
        OpenImporter(&importer, &chunkImporter, &chunkId, &infoWork, manager, info, TestUserId3, backupStoragePath);
        NNT_EXPECT_RESULT_SUCCESS(PushChunkData(chunkImporter.get(), chunkId, backupStoragePath));
    }

    DeleteAllTestSaveData();

    Unmount("cloud");
} // NOLINT(impl/function_size)

// 複製差分インポート時に、完了時まで複製元やセーブデータ列挙に影響がないこと
TEST(CloudBackup, DuplicateDiffImportSource)
{
    DeleteAllTestSaveData();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

    int divisionCount = 32;

    // エクスポート先ストレージ
    nnt::fs::util::TemporaryHostDirectory tempPath;
    tempPath.Create();

    NNT_ASSERT_RESULT_SUCCESS(MountHost("cloud", tempPath.GetPath().c_str()));
    String backupStoragePath("cloud:/");

    const int64_t TestFileSize = 256 * 1024;

    nnt::fs::util::Hash hashSrc0;
    SaveDataInfo infoSrc;
    {

        // エクスポート元セーブを生成
        NNT_ASSERT_RESULT_SUCCESS(CreateAndFindSaveData(&infoSrc));

        // 中身を生成
        NNT_ASSERT_RESULT_SUCCESS(MountSaveData("save", TestApplicationId0, TestUserId0));
        NNT_ASSERT_RESULT_SUCCESS(CreateFileWith32BitCount("save:/testFile", TestFileSize, 1));
        CommitSaveData("save");
        Unmount("save");

        // ハッシュ値計算
        CalculateSaveDataHash(&hashSrc0, infoSrc);

        // エクスポート
        {
            int exportCount;
            int64_t exportSize;
            ExportSaveData(&exportSize, &exportCount, backupStoragePath, infoSrc, false, divisionCount);
        }
    }

    Vector<SaveDataInfo> saveDataInfoArray0;
    FindSaveData(&saveDataInfoArray0, SaveDataSpaceId::User,
        [](SaveDataInfo info) {
        NN_UNUSED(info);
        return true; // 全列挙
    }
    );
    EXPECT_TRUE(saveDataInfoArray0.size() > 0);


    // 更新
    {
        NNT_ASSERT_RESULT_SUCCESS(MountSaveData("save", TestApplicationId0, TestUserId0));
        NNT_ASSERT_RESULT_SUCCESS(CreateFileWith32BitCount("save:/testFile2", TestFileSize, 2));
        CommitSaveData("save");
        Unmount("save");
    }

    // ハッシュ値計算
    nnt::fs::util::Hash hashSrc1;
    CalculateSaveDataHash(&hashSrc1, infoSrc);



    // 複製差分インポート開始
    SaveDataTransferManagerForCloudBackUp manager;
    std::unique_ptr<ISaveDataDivisionImporter> importer;
    {

        SaveDataTransferManagerForCloudBackUp::Challenge challenge;
        NNT_ASSERT_RESULT_SUCCESS(manager.GetChallenge(&challenge));

        auto keySeed = GetKeySeed(backupStoragePath);
        auto initialDataMac = GetInitialDataMac(backupStoragePath);

        auto ksp = GetKeySeedPackageFromServer(challenge, keySeed, initialDataMac);
        NNT_ASSERT_RESULT_SUCCESS(manager.SetKeySeedPackage(ksp));

        InitialDataForCloudBackUp initialData;
        ReadInitialData(&initialData, backupStoragePath);

        NNT_ASSERT_RESULT_SUCCESS(manager.OpenSaveDataDuplicateDiffImporter(&importer, initialData, infoSrc.saveDataSpaceId, infoSrc.saveDataId));
    }

    std::unique_ptr<ISaveDataChunkIterator> iter;
    NNT_ASSERT_RESULT_SUCCESS(importer->OpenSaveDataDiffChunkIterator(&iter));


    // セーブデータ列挙で複製中のセーブデータが列挙されないことをチェック
    auto CheckSaveDataEnumeration = [&saveDataInfoArray0]()
    {
        {
            // 現状(secondary 以外)を全列挙
            Vector<SaveDataInfo> saveDataInfoArray;
            FindSaveData(&saveDataInfoArray, SaveDataSpaceId::User,
                [](SaveDataInfo info) {
                NN_UNUSED(info);
                return info.rank == SaveDataRank::Primary;
            }
            );

            // 一致確認
            EXPECT_TRUE(saveDataInfoArray0.size() == saveDataInfoArray.size());
            if (saveDataInfoArray0.size() == saveDataInfoArray.size())
            {
                NNT_FS_UTIL_EXPECT_MEMCMPEQ(&saveDataInfoArray0[0], &saveDataInfoArray[0], sizeof(SaveDataInfo) * saveDataInfoArray0.size());
            }
        }


        {
            // 作業中の secondary が存在すること
            Vector<SaveDataInfo> saveDataInfoArray;
            FindSaveData(&saveDataInfoArray, SaveDataSpaceId::User,
                [](SaveDataInfo info) {
                NN_UNUSED(info);
                return info.rank == SaveDataRank::Secondary;
            }
            );
            EXPECT_EQ(1, saveDataInfoArray.size());

            // 既存の列挙手段で secondary が列挙されないこと
            std::unique_ptr<nn::fs::SaveDataIterator> iter;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter, SaveDataSpaceId::User));

            int totalCount = 0;
            while (NN_STATIC_CONDITION(true))
            {
                int64_t count;
                nn::fs::SaveDataInfo info;
                NNT_EXPECT_RESULT_SUCCESS(iter->ReadSaveDataInfo(&count, &info, 1));
                if (count == 0)
                {
                    break;
                }

                totalCount++;
                EXPECT_EQ(SaveDataRank::Primary, info.rank);
            }
            EXPECT_EQ(saveDataInfoArray0.size(), totalCount);

        }

    };

    // 対象がロックされていないこと、変化していないことをチェック
    auto CheckSaveDataHash = [](const SaveDataInfo info, const Hash& expected)
    {
        nnt::fs::util::Hash actual;
        CalculateSaveDataHash(&actual, info);
        EXPECT_TRUE(crypto::IsSameBytes(&expected, &actual, sizeof(Hash)));
    };


    // InitializeImport() 中は src をロックする
    {
        const size_t InitializeImportProcessSize = 128 * 1024;

        int64_t restSize;

        // 1 cycle
        NNT_ASSERT_RESULT_SUCCESS(importer->InitializeImport(&restSize, InitializeImportProcessSize));
        CheckSaveDataEnumeration();

        // 完了まで
        while (restSize > 0)
        {
            NNT_ASSERT_RESULT_SUCCESS(importer->InitializeImport(&restSize, InitializeImportProcessSize));
        }
        CheckSaveDataEnumeration();
    }

    // chunkImporter.Push() 中は src は解放されている＆変化無し
    NNT_ASSERT_RESULT_SUCCESS(importer->OpenSaveDataDiffChunkIterator(&iter));
    for (; !iter->IsEnd(); iter->Next())
    {
        std::unique_ptr<ISaveDataChunkImporter> chunkImporter;
        NNT_ASSERT_RESULT_SUCCESS(importer->OpenSaveDataChunkImporter(&chunkImporter, iter->GetId()));

        ChunkAccessor chunk(backupStoragePath, iter->GetId());
        const size_t PushSize = 128 * 1024;

        // 1 push
        {
            auto buffer = chunk.Pull(PushSize);
            NNT_ASSERT_RESULT_SUCCESS(chunkImporter->Push(buffer.buffer.get(), buffer.validDataSize));
            CheckSaveDataEnumeration();
            CheckSaveDataHash(infoSrc, hashSrc1);
        }

        // 完了まで Push
        while (true)
        {
            auto buffer = chunk.Pull(PushSize);
            if (buffer.validDataSize == 0)
            {
                break;
            }

            NNT_ASSERT_RESULT_SUCCESS(chunkImporter->Push(buffer.buffer.get(), buffer.validDataSize));
        }
        CheckSaveDataEnumeration();
        CheckSaveDataHash(infoSrc, hashSrc1);
    }

    // FinalizeImport()
    NNT_ASSERT_RESULT_SUCCESS(importer->FinalizeImport());

    // Swap のゴミ等が無いこと
    {
        SaveDataInfo infoDst;
        NNT_ASSERT_RESULT_SUCCESS(FindSaveDataByApplicationId(&infoDst, TestApplicationId0, TestUserId0)); // saveDataId は更新されている

        {
            {
                // 現状を全列挙
                Vector<SaveDataInfo> saveDataInfoArray;
                FindSaveData(&saveDataInfoArray, SaveDataSpaceId::User,
                    [](SaveDataInfo info) {
                    NN_UNUSED(info);
                    return true;
                }
                );

                // 一致確認、但し対象セーブだけは saveDataId が更新されている
                // (SaveDataInfo の順が乱れていないと仮定したチェック方法)
                EXPECT_TRUE(saveDataInfoArray0.size() == saveDataInfoArray.size());
                if (saveDataInfoArray0.size() == saveDataInfoArray.size())
                {
                    for (int i = 0; i < static_cast<int>(saveDataInfoArray0.size()); i++)
                    {
                        if (saveDataInfoArray[i].saveDataId == infoDst.saveDataId)
                        {
                            saveDataInfoArray[i].saveDataId = infoSrc.saveDataId;
                        }

                        NNT_FS_UTIL_EXPECT_MEMCMPEQ(&saveDataInfoArray0[i], &saveDataInfoArray[i], sizeof(SaveDataInfo));
                    }
                }
            }

            // secondary が存在しないこと
            {
                Vector<SaveDataInfo> saveDataInfoArray;
                FindSaveData(&saveDataInfoArray, SaveDataSpaceId::User,
                    [](SaveDataInfo info) {
                    NN_UNUSED(info);
                    return info.rank == SaveDataRank::Secondary;
                }
                );
                EXPECT_EQ(0, saveDataInfoArray.size());
            }

        }

        CheckSaveDataHash(infoDst, hashSrc0); // インポートの結果 hashSrc0 に戻っている
    }



    Unmount("cloud");
    DeleteAllTestSaveData();
} // NOLINT(impl/function_size)

// export キャンセル
TEST(CloudBackup, CancelExport)
{
    DeleteAllTestSaveData();

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

    const size_t WorkBufferSize = 4 * 1024 * 1024;
    auto workBuffer = AllocateBuffer(WorkBufferSize);

    const nn::fs::UserId TestUserId = { { 0, 1 } };

    const int64_t TestSaveDataSize = 32 * 1024 * 1024;

    // セーブを生成
    SaveDataInfo info;
    NNT_ASSERT_RESULT_SUCCESS(CreateAndExtendSaveData(
        &info,
        TestApplicationId0,
        TestUserId,
        TestSaveDataSize, TestSaveDataSize,
        TestSaveDataSize, TestSaveDataSize,
        0
    ));

    {
        SaveDataTransferManagerForCloudBackUp manager;

        // エクスポータを開く
        std::unique_ptr<ISaveDataDivisionExporter> exporter;
        std::unique_ptr<ISaveDataChunkExporter> chunkExporter;
        OpenExporter(&exporter, &chunkExporter, manager, info);

        // キャンセル
        NNT_EXPECT_RESULT_SUCCESS(exporter->CancelExport());

        // キャンセル後の Pull は ResultSaveDataPorterCoreInvalidated
        {
            size_t size;
            NNT_EXPECT_RESULT_FAILURE(ResultSaveDataPorterCoreInvalidated, chunkExporter->Pull(&size, workBuffer.get(), WorkBufferSize));
        }

        // キャンセル後の ChunkExporter のオープンは事前条件違反
        {
            std::unique_ptr<ISaveDataChunkIterator> iterTmp;
            NNT_EXPECT_RESULT_FAILURE(ResultPreconditionViolation, exporter->OpenSaveDataDiffChunkIterator(&iterTmp));
        }

        // キャンセル後の ChunkExporter のオープンは事前条件違反
        {
            std::unique_ptr<ISaveDataChunkExporter> chunkExporterTmp;
            NNT_EXPECT_RESULT_FAILURE(ResultPreconditionViolation, exporter->OpenSaveDataChunkExporter(&chunkExporterTmp, 0));
        }

        // キャンセル後の Finalize は事前条件違反
        {
            nn::fs::ISaveDataDivisionExporter::KeySeed keySeed;
            nn::fs::ISaveDataDivisionExporter::InitialDataMac mac;
            NNT_EXPECT_RESULT_FAILURE(ResultPreconditionViolation, exporter->FinalizeFullExport(&keySeed, &mac));
        }

        // 新規にエクスポートを開始することは可能
        {
            std::unique_ptr<ISaveDataDivisionExporter> exporter2;
            std::unique_ptr<ISaveDataChunkExporter> chunkExporter2;
            OpenExporter(&exporter2, &chunkExporter2, manager, info);
            size_t size;
            NNT_ASSERT_RESULT_SUCCESS(chunkExporter2->Pull(&size, workBuffer.get(), WorkBufferSize));
        }
    }

    DeleteAllTestSaveData();
} // NOLINT(impl/function_size)

// import キャンセル
TEST_P(CloudBackupImport, CancelImport)
{
    DeleteAllTestSaveData();
    NN_UTIL_SCOPE_EXIT
    {
        DeleteAllTestSaveData();
    };

    auto param = GetParam();

    bool is_swap = param.isDuplicate;
    bool is_diff = param.isDiff;

    const nn::fs::UserId TestUserId = { { 0, 1 } };
    const nn::fs::UserId TestUserId2 = { { 0, 2 } };

    const int divisionCount = 32;
    const int64_t TestSaveDataSize = 32 * 1024 * 1024;
    const int64_t TestFileSize = 256 * 1024;

    // セーブを生成
    SaveDataInfo info;
    NNT_ASSERT_RESULT_SUCCESS(CreateAndExtendSaveData(
        &info,
        TestApplicationId0,
        TestUserId,
        TestSaveDataSize, TestSaveDataSize,
        TestSaveDataSize, TestSaveDataSize,
        0
    ));

    // エクスポート先ストレージ
    nnt::fs::util::TemporaryHostDirectory tempPath;
    tempPath.Create();
    NNT_ASSERT_RESULT_SUCCESS(MountHost("cloud", tempPath.GetPath().c_str()));
    NN_UTIL_SCOPE_EXIT
    {
        Unmount("cloud");
    };
    String backupStoragePath("cloud:/");

    // エクスポート
    {
        int exportCount;
        int64_t exportSize;
        ExportSaveData(&exportSize, &exportCount, backupStoragePath, info, false, divisionCount);
    }

    // 更新
    {
        NNT_ASSERT_RESULT_SUCCESS(MountSaveData("save", TestApplicationId0, TestUserId));
        NNT_ASSERT_RESULT_SUCCESS(CreateFileWith32BitCount("save:/testFile", TestFileSize, 2));
        CommitSaveData("save");
        Unmount("save");
    }

    {
        SaveDataTransferManagerForCloudBackUp manager;

        // SetKeySeedPackage
        {
            SaveDataTransferManagerForCloudBackUp::Challenge challenge;
            NNT_ASSERT_RESULT_SUCCESS(manager.GetChallenge(&challenge));
            auto keySeed = GetKeySeed(backupStoragePath);
            auto initialDataMac = GetInitialDataMac(backupStoragePath);
            auto ksp = GetKeySeedPackageFromServer(challenge, keySeed, initialDataMac);
            NNT_ASSERT_RESULT_SUCCESS(manager.SetKeySeedPackage(ksp));
        }

        // インポータを開く
        std::unique_ptr<ISaveDataDivisionImporter> importer;
        std::unique_ptr<ISaveDataChunkImporter> chunkImporter;
        SaveDataChunkId chunkId;
        nn::util::optional<SaveDataInfo> infoWork;
        OpenImporter(&importer, &chunkImporter, &chunkId, &infoWork, manager, info, TestUserId2, backupStoragePath, is_swap, is_diff);

        // キャンセル
        if (!is_swap && is_diff)
        {
            NNT_EXPECT_RESULT_FAILURE(ResultNotImplemented, importer->CancelImport());
            // キャンセルに失敗しているので以降はチェック不要
            return;
        }
        else
        {
            NNT_EXPECT_RESULT_SUCCESS(importer->CancelImport());
        }

        // キャンセル後の Push は ResultSaveDataPorterCoreInvalidated
        {
            NNT_EXPECT_RESULT_FAILURE(ResultSaveDataPorterCoreInvalidated, PushChunkData(chunkImporter.get(), chunkId, backupStoragePath));
        }

        // キャンセル後の ChunkIterator のオープンは事前条件違反
        {
            std::unique_ptr<ISaveDataChunkIterator> iterTmp;
            NNT_EXPECT_RESULT_FAILURE(ResultPreconditionViolation, importer->OpenSaveDataDiffChunkIterator(&iterTmp));
        }

        // キャンセル後の ChunkImporter のオープンは事前条件違反
        {
            std::unique_ptr<ISaveDataChunkImporter> chunkImporterTmp;
            NNT_EXPECT_RESULT_FAILURE(ResultPreconditionViolation, importer->OpenSaveDataChunkImporter(&chunkImporterTmp, 0));
        }

        // キャンセル後の Finalize は事前条件違反
        {
            NNT_EXPECT_RESULT_FAILURE(ResultPreconditionViolation, importer->FinalizeImport());
        }

        // キャンセル後に作業用セーブデータが存在しないこと
        {
            Vector<SaveDataInfo> saveDataInfoArray;
            FindSaveData(&saveDataInfoArray, SaveDataSpaceId::User,
                [&](SaveDataInfo info) {
                NN_UNUSED(info);
                return (info.rank == infoWork->rank) && (info.saveDataUserId == infoWork->saveDataUserId) && (info.applicationId == infoWork->applicationId);
            }
            );
            EXPECT_EQ(0, saveDataInfoArray.size());
        }

        // 新規にインポートを開始することは可能
        {
            nn::util::optional<SaveDataInfo> infoSrc;
            FindSaveData(&infoSrc, info.saveDataSpaceId, [&](SaveDataInfo i) {
                return i.saveDataUserId == info.saveDataUserId;
            });
            EXPECT_TRUE(infoSrc != util::nullopt);

            std::unique_ptr<ISaveDataDivisionImporter> importer2;
            std::unique_ptr<ISaveDataChunkImporter> chunkImporter2;
            SaveDataChunkId chunkId2;
            nn::util::optional<SaveDataInfo> infoWork2;
            OpenImporter(&importer2, &chunkImporter2, &chunkId2, &infoWork2, manager, *infoSrc, TestUserId2, backupStoragePath, is_swap, is_diff);
            NNT_EXPECT_RESULT_SUCCESS(PushChunkData(chunkImporter2.get(), chunkId2, backupStoragePath));
        }
    }
} // NOLINT(impl/function_size)


INSTANTIATE_TEST_CASE_P(CloudBackupDuplicateDiffImport,
    CloudBackupDiffImport,
    ::testing::ValuesIn(DiffImportParamArray));


INSTANTIATE_TEST_CASE_P(ImportMode,
    CloudBackupImport,
    ::testing::ValuesIn(ImportParamArray));


} // namespace

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();


#if !defined(NN_BUILD_CONFIG_OS_WIN)
    static const size_t BufferPoolSize = 32 * 1024 * 1024;
    static NN_ALIGNAS(4096) char g_BufferPool[BufferPoolSize];
    static const size_t WorkBufferSize = nn::fssystem::BufferPoolWorkSize;
    static NN_ALIGNAS(8) char g_WorkBuffer[WorkBufferSize];
    nn::fssystem::InitializeBufferPool(g_BufferPool, BufferPoolSize, g_WorkBuffer, WorkBufferSize);
#endif

    nn::spl::InitializeForCrypto();

    nn::fs::SetEnabledAutoAbort(false);

    ::testing::InitGoogleTest(&argc, argv);

    nn::fssystem::InitializeAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();

#if defined(NN_BUILD_CONFIG_OS_WIN)
    nn::time::PosixTime posixTime;
    posixTime.value = 1;
    nn::fs::SetCurrentPosixTime(posixTime, 0);
#endif

    auto result = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(result);
}

