﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fs/fs_IStorage.h>
#include <nn/fssystem/save/fs_SaveFileInFile.h>

/**
* @brief 読み込み専用フィルタストレージオブジェクトです。
*/
class ReadOnlyFilterStorage : public nn::fs::IStorage
{
public:
    /**
    * @brief コンストラクタです。
    */
    ReadOnlyFilterStorage() NN_NOEXCEPT
        : m_pStorage(nullptr)
    {
    }

    /**
    * @brief フィルタストレージをマウントします。
    */
    nn::Result Initialize(nn::fs::IStorage* pStorage) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pStorage);
        m_pStorage = pStorage;
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief フィルタストレージをアンマウントします。
    */
    void Finalize() NN_NOEXCEPT
    {
        m_pStorage = nullptr;
    }

    /**
    * @brief 下位ファイルの offset 以降から buffer に size バイト分コピーします。
    *
    * @param[in] offset 読み込み開始位置
    * @param[in,out] buffer 読み込むバッファ
    * @param[in] size 読み込むデータサイズ
    *
    * @return 関数の処理結果を返します。
    */
    virtual nn::Result Read(
                int64_t offset,
                void* buffer,
                size_t size
            ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pStorage);
        return m_pStorage->Read(offset, buffer, size);
    }

    /**
    * @breif ファイルのサイズを取得します。
    *
    * @param[in] size サイズ
    *
    * @return 関数の処理結果を返します。
    */
    virtual nn::Result GetSize(int64_t* outSize) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pStorage);
        return m_pStorage->GetSize(outSize);
    }

    /**
    * @breif ファイルをフラッシュします。
    *
    * @return 関数の処理結果を返します。
    */
    virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pStorage);
        return m_pStorage->Flush();
    }

    /**
    * @brief 下位ファイルの offset 以降に buffer を size バイト分コピーします。
    *
    * @param[in] offset 書き込み開始位置
    * @param[in,out] buffer 書き込むデータ
    * @param[in] size 書き込むデータサイズ
    *
    * @return 関数の処理結果を返します。
    */
    virtual nn::Result Write(
                           int64_t offset,
                           const void* buffer,
                           size_t size
                       ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(offset);
        NN_UNUSED(buffer);
        NN_UNUSED(size);
        return nn::fs::ResultUnsupportedOperation();
    }

    /**
    * @breif ファイルのサイズを変更します。
    *
    * @param[in] size 変更後のサイズ
    *
    * @return 関数の処理結果を返します。
    */
    virtual nn::Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(size);
        return nn::fs::ResultUnsupportedOperation();
    }

private:
    nn::fs::IStorage* m_pStorage;
};

