﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <random>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>
#include "testFs_util_CommonStorageTests.h"
#include "testFs_util_ReadOnlyFilterStorage.h"

TEST(FsReadOnlyFilterStorageTest, Read)
{
    static const size_t FileSize = 1024;

    nnt::fs::util::Vector<char> buf(FileSize);

    // 元となるファイルを初期化します。
    nnt::fs::util::SafeMemoryStorage storageMem;
    storageMem.Initialize(FileSize);
    char* ptr = reinterpret_cast<char*>(storageMem.GetBuffer());

    for( size_t i = 0; i < FileSize; i++ )
    {
        ptr[i] = i & 0xFF;
    }

    // 読み込み専用フィルタファイルを初期化します。
    ReadOnlyFilterStorage storageFilter;
    NNT_ASSERT_RESULT_SUCCESS(storageFilter.Initialize(&storageMem));

    // 書き込みができないことを確認します。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultUnsupportedOperation,
        storageFilter.Write(0, buf.data(), 1)
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultUnsupportedOperation,
        storageFilter.Write(FileSize + 1, buf.data(), 1)
    );

    // サイズが変更できないことを確認します。
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultUnsupportedOperation, storageFilter.SetSize(100));
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultUnsupportedOperation, storageFilter.SetSize(0));

    // サイズの取得はできることを確認します。
    int64_t sizeFile;
    NNT_ASSERT_RESULT_SUCCESS(storageFilter.GetSize(&sizeFile));
    ASSERT_EQ(FileSize, sizeFile);

    // 読み込みはできることを確認します。
    NNT_ASSERT_RESULT_SUCCESS(storageFilter.Read(0, buf.data(), 1));
    NNT_ASSERT_RESULT_SUCCESS(storageFilter.Read(FileSize - 1, buf.data(), 1));
    NNT_ASSERT_RESULT_SUCCESS(storageFilter.Read(0, buf.data(), FileSize - 1));
    NNT_ASSERT_RESULT_SUCCESS(storageFilter.Read(0, buf.data(), FileSize));

    // データ比較
    for( size_t i = 0; i < FileSize; ++i )
    {
        ASSERT_EQ(i & 0xFF, static_cast<uint8_t>(buf[i]));
    }
}

TEST(FsReadOnlyFilterStorageLargeTest, Read)
{
    static const size_t AccessSize = 1024;
    static const int64_t FileSize = static_cast<int64_t>(64) * 1024 * 1024 * 1024 + AccessSize;

    nnt::fs::util::VirtualMemoryStorage baseStorage(FileSize);
    ReadOnlyFilterStorage readOnlyStorage;
    NNT_ASSERT_RESULT_SUCCESS(readOnlyStorage.Initialize(&baseStorage));

    TestLargeOffsetAccess(
        &baseStorage,
        AccessSize,
        [&](int64_t offset, char* readBuffer, const char* writeBuffer, size_t bufferSize) NN_NOEXCEPT
        {
            NNT_ASSERT_RESULT_SUCCESS(readOnlyStorage.Read(offset, readBuffer, bufferSize));
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer, readBuffer, bufferSize);
        });
}

class ReadOnlyFilterStorageTestSetup : public TestStorageSetup
{
public:
    static const int64_t MinFileSize = 1;

public:
    // コンストラクタです。
    ReadOnlyFilterStorageTestSetup() NN_NOEXCEPT
    {
    }

    // ファイルオブジェクトをフォーマットします。
    virtual void Format(
                     int64_t sizeFile,
                     int64_t offset,
                     size_t sizeBlock
                 ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(sizeBlock);
        m_pStorageMem.reset(new nnt::fs::util::SafeMemoryStorage);
        m_pStorageMem->Initialize(static_cast<size_t>(sizeFile + offset * 2));
    }

    // ファイルオブジェクトを初期化します。
    virtual void Initialize(
                     int64_t sizeFile,
                     int64_t offset,
                     size_t sizeBlock
                 ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(sizeFile);
        NN_UNUSED(offset);
        NN_UNUSED(sizeBlock);
        SetStorage(m_pStorageMem.get());
    }

    // ファイルオブジェクトをアンマウントします。
    virtual void Unmount() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    // ファイルオブジェクトを破棄します。
    virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE
    {
        m_pStorageMem.reset(nullptr);
    }

    // 読み込み専用かどうか
    virtual bool IsReadOnly() const NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

    // ファイルへのデータ書き込み時に自動伸長するかどうか
    virtual bool IsAutoExtendSizeOnWrite() const NN_NOEXCEPT NN_OVERRIDE
    {
        return false;
    }

    // ファイルのサイズ変更が可能かどうか
    virtual bool IsEnableResize() const NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

    // サイズ 0 のファイルを作成可能かどうか
    virtual bool IsAcceptsSizeZero() const NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

private:
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> m_pStorageMem;
    std::unique_ptr<ReadOnlyFilterStorage> m_pStorageFilter;
};

// 共通テストを行います。
TEST(FsReadOnlyFilterStorageTest, StorageCommonTest)
{
    StorageTest::RunTest<ReadOnlyFilterStorageTestSetup>();
}

