﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <mutex>
#include <numeric>
#include <map>
#include <sstream>

#include <nn/nn_Common.h>
#include <nn/os/os_Thread.h>
#include <nn/crypto.h>
#include <nn/util/util_IntUtil.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Decompression.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fssystem/save/fs_JournalIntegritySaveDataFileSystem.h>
#include <nn/fssystem/save/fs_JournalIntegritySaveDataFileSystemDriver.h>

// SaveDataFileSystemCore::QueryTotalSize から呼び出される関数のインスタンス化のためにインクルードします。
#include <nn/fssystem/dbm/fs_HierarchicalFileTableTemplate.impl.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Compiler.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>
#include "testFs_util_CommonStorageTests.h"
#include "testFs_util_CommonFileSystemTests.h"

namespace {
    nn::fssystem::IBufferManager* g_pBufferManager = nullptr;

    nn::fssystem::IBufferManager* GetBufferManager() NN_NOEXCEPT
    {
        if( g_pBufferManager == nullptr )
        {
            static const auto MaxCacheCount = 1024;
            static const auto SizeBlock = 16 * 1024;
#if defined(NN_BUILD_CONFIG_OS_WIN)
            static const auto BufferManagerHeapSize = 30 * 1024 * 1024;
#else
            static const auto BufferManagerHeapSize = 32 * 1024 * 1024;
#endif // defined(NN_BUILD_CONFIG_OS_WIN)
            static NN_ALIGNAS(4096) char s_BufferManagerHeap[BufferManagerHeapSize];
            static nn::fssystem::FileSystemBufferManager s_BufferManagerInstance;

            s_BufferManagerInstance.Initialize(
                MaxCacheCount,
                reinterpret_cast<uintptr_t>(s_BufferManagerHeap),
                sizeof(s_BufferManagerHeap),
                SizeBlock);
            g_pBufferManager = &s_BufferManagerInstance;
        }

        return g_pBufferManager;
    }

    void FinalizeBufferManager() NN_NOEXCEPT
    {
        auto pBufferManager(
            reinterpret_cast<nn::fssystem::FileSystemBufferManager*>(g_pBufferManager)
        );
        pBufferManager->Finalize();
        g_pBufferManager = nullptr;
    }
}

class FsJournalIntegritySaveDataFileSystemDriverTest : public ::testing::Test
{
public:
    //! デストラクタ
    virtual ~FsJournalIntegritySaveDataFileSystemDriverTest() NN_NOEXCEPT NN_OVERRIDE {}

#if 0
    // 複数のインスタンスを作成し、同時にリードライトを行っても
    // キャッシュ不足にならないことを検証します。
    static void TestMultiArchiveInternal();

    static void TestRandomErrorDetail(nn::fs::SubStorage storage, uint32_t seed);

    static void TestRandomErrorImpl(uint32_t seed);

    static void TestRandomError2Impl(uint32_t seed);
#endif

#if defined(NN_FS_DESTROY_SIGNATURE)
    // 署名破壊フラグ付き書き込みで検証エラー状態となることを確認します。
    static void TestDestroySignatureInternal(const uint32_t loopCount);

    static void TestReadUnformattedFilesInternal(const uint32_t loopCount);
#endif

protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        m_BufferManagerFreeMemorySize = GetBufferManager()->GetFreeSize();
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        if( !HasFatalFailure() )
        {
            EXPECT_EQ(m_BufferManagerFreeMemorySize, GetBufferManager()->GetFreeSize());
        }
    }

    static size_t SetupBlockSize2ForExtraDataTest(
        size_t sizeBlock,
        int countBlock,
        size_t sizeBlock1
    ) NN_NOEXCEPT
    {
        // ExtraDataTest で使用している sizeBlock1、sizeBlock2 に対して
        // countBlock * sizeBlock の容量が確保できるような sizeBlock2 を求めます
        // (sizeBlock1 / 32 * sizeBlock1 / 32 * sizeBlock2 / 32 * sizeBlock2 >= countBlock * sizeBlock)
        auto valueMin = static_cast<uint32_t>(std::ceil(128.0
            * std::sqrt(2.0 * static_cast<double>(sizeBlock) * static_cast<double>(countBlock))
            / static_cast<double>(sizeBlock1)));
        size_t value = 512;
        if( value < valueMin )
        {
            value = nn::util::ceilp2(valueMin);
        }
        return GenerateRandomBlockSize(value, MaxBlockSize);
    }

private:
    size_t m_BufferManagerFreeMemorySize;
};

static int64_t s_SizeArchiveForCommonTest = 128 * 1024;

class JournalIntegritySaveDataFileSystemDriverTestSetup : public TestFileSystemSetup
{
public:
    // コンストラクタです。
    JournalIntegritySaveDataFileSystemDriverTestSetup() NN_NOEXCEPT
        : m_IsRandomBlockSize(true)
    {
    }

    // ファイルシステムを初期化します。
    virtual void Initialize(int64_t offset, size_t sizeBlock) NN_NOEXCEPT NN_OVERRIDE
    {
        static const int CountExpandMax = 25;

        NN_UNUSED(offset);

        int64_t sizeTotal;
        nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam paramDuplex;
        nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
            paramIntegrity;
        uint32_t countReservedBlock;
        uint32_t countBlock;
        while( NN_STATIC_CONDITION(true) )
        {
            const int64_t sizeFileSystem = s_SizeArchiveForCommonTest * (10 + m_Random.Get(9));
            const int64_t sizeReserved = sizeFileSystem >> (3 + m_Random.Get(2));

            if( m_IsRandomBlockSize )
            {
                sizeBlock = GenerateRandomBlockSize(256, MaxBlockSize);
            }

            // ブロックサイズ
            size_t sizeBlock1 = GenerateRandomBlockSize(512, MaxBlockSize);
            auto minBlockSize = std::max(8 * 1024 * 1024 / sizeBlock1, static_cast<size_t>(512));
            ASSERT_LE(minBlockSize, MaxBlockSize);
            size_t sizeBlock2 = GenerateRandomBlockSize(minBlockSize, MaxBlockSize);
            if( !m_IsRandomBlockSize )
            {
                sizeBlock1 = sizeBlock2 = sizeBlock;
            }
            paramDuplex.sizeBlockLevel[0] = sizeBlock1;
            paramDuplex.sizeBlockLevel[1] = sizeBlock2;
            paramIntegrity.sizeBlockLevel[0] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[1] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[2] = sizeBlock2;
            paramIntegrity.sizeBlockLevel[3] = sizeBlock2;

            countReservedBlock = static_cast<uint32_t>(sizeReserved / sizeBlock);

            countBlock = 0;
            sizeTotal = 0;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryDataBlockCount(
                    &countBlock,
                    sizeFileSystem,
                    sizeReserved,
                    sizeBlock,
                    CountExpandMax,
                    paramDuplex,
                    paramIntegrity
                )
            );
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
                    &sizeTotal,
                    sizeBlock,
                    CountExpandMax,
                    paramDuplex,
                    paramIntegrity,
                    countBlock,
                    countReservedBlock
                )
            );

            uint32_t countBlock2 = 0;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryDataBlockCount(
                    &countBlock2,
                    sizeTotal,
                    sizeReserved,
                    sizeBlock,
                    CountExpandMax,
                    paramDuplex,
                    paramIntegrity
                )
            );

            ASSERT_TRUE(sizeTotal <= sizeFileSystem);
            ASSERT_TRUE(countBlock == countBlock2);

            // サイズが小さすぎないことを大雑把に確認します。
            if( static_cast<int64_t>(sizeTotal + sizeBlock * 150) >= sizeFileSystem )
            {
                break;
            }
        }

        NN_LOG("Test Param:\n");
        NN_LOG("  sizeTotal         : %lld\n", sizeTotal);
        NN_LOG("  sizeBlock1        : %u\n", static_cast<uint32_t>(paramDuplex.sizeBlockLevel[0]));
        NN_LOG("  sizeBlock2        : %u\n", static_cast<uint32_t>(paramDuplex.sizeBlockLevel[1]));
        NN_LOG("  countReservedBlock: %u\n", countReservedBlock);
        NN_LOG("  countBlock        : %u\n", countBlock);

        m_MemStorage.reset(new nnt::fs::util::SafeMemoryStorage());
        m_MemStorage->Initialize(sizeTotal);
        // JournalIntegritySaveDataFileSystemDriver は、必ず 16KB アライメントのアクセスが発生する仕様のため
        // 16 KB を超えるアクセスアライメントのチェックは不要です。
        size_t alignSize = std::min(static_cast<size_t>(16 * 1024), sizeBlock);
        m_AlignCheckStorage.Initialize(
            nn::fs::SubStorage(m_MemStorage.get(), 0, sizeTotal),
            alignSize
        );

        nn::fs::SaveDataHashSalt salt;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::Format(
                nn::fs::SubStorage(&m_AlignCheckStorage, 0, sizeTotal),
                sizeBlock,
                CountExpandMax,
                paramDuplex,
                paramIntegrity,
                countBlock,
                countReservedBlock,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                salt
            )
        );

        m_FileSystem.reset(new nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver());
        NNT_ASSERT_RESULT_SUCCESS(
            m_FileSystem->Initialize(
                nn::fs::SubStorage(&m_AlignCheckStorage, 0, sizeTotal),
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );

        SetFileSystem(m_FileSystem.get());
    } // NOLINT(impl/function_size)

    // ファイルシステムを破棄します。
    virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE
    {
        m_FileSystem->Finalize();
        m_FileSystem.reset();
        m_MemStorage.reset();
        m_MemStorage.reset();
    }

    // ファイルシステムへのアクセスはスレッドセーフかどうか
    virtual bool IsEnableThreadSafeAccess() NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

    // ファイルシステムへの排他制御は有効かどうか
    virtual bool IsEnableExclusiveAccess() NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

    // ファイルのサイズ変更が可能かどうか
    virtual bool IsEnableResizeFile() NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

    //!< ファイルへのデータ書き込み時に自動伸長するかどうか
    virtual bool IsAutoExtendSizeOnWrite() const NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

    // コミット処理を行います。
    nn::Result Commit() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_FileSystem);
        return m_FileSystem->Commit();
    }

    // マウントしなおします。
    void UnmountAndMount() NN_NOEXCEPT
    {
        m_FileSystem->Finalize();
        m_FileSystem.reset(new nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver());
        int64_t sizeTotal;
        NNT_ASSERT_RESULT_SUCCESS(m_AlignCheckStorage.GetSize(&sizeTotal));
        NNT_ASSERT_RESULT_SUCCESS(
            m_FileSystem->Initialize(nn::fs::SubStorage(&m_AlignCheckStorage, 0, sizeTotal),
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );
        SetFileSystem(m_FileSystem.get());
    }

    // ブロックサイズをランダムに決定するかどうかを設定します。
    inline void SetRandomizeBlockSize(bool isRandom) NN_NOEXCEPT
    {
        m_IsRandomBlockSize = isRandom;
    }

private:
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> m_MemStorage;
    std::unique_ptr<nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver> m_FileSystem;
    nnt::fs::util::AlignCheckStorage m_AlignCheckStorage;
    bool m_IsRandomBlockSize;
};

// ファイルシステム共通テストを行います。
TEST_F(FsJournalIntegritySaveDataFileSystemDriverTest, TestArchiveCommonFullHeavy)
{
    s_SizeArchiveForCommonTest = 1024 * 1024;
    FileSystemTest::RunTest<JournalIntegritySaveDataFileSystemDriverTestSetup>();
}

#if !defined(NN_BUILD_CONFIG_ADDRESS_32) || !defined(NN_BUILD_CONFIG_OS_WIN)
// ファイルシステム共通テストを行います。
TEST_F(FsJournalIntegritySaveDataFileSystemDriverTest, TestArchiveCommonFull64Heavy)
{
    s_SizeArchiveForCommonTest = 30 * 1024 * 1024;
    FileSystemTest::RunTest<JournalIntegritySaveDataFileSystemDriverTestSetup>();
}
#endif // !defined(NN_BUILD_CONFIG_ADDRESS_32) || !defined(NN_BUILD_CONFIG_OS_WIN)

// BufferManager が管理するバッファに十分な空き領域がない場合に、
// エラーリザルトが正確に返されない事があるという問題 (SIGLO-34500) が
// 修正されているかを確認するテスト
TEST_F(FsJournalIntegritySaveDataFileSystemDriverTest, TestBufferDeficient)
{
    const auto CountExpandMax = 25;
    const auto CountDataBlock = 3;
    const auto CountReservedBlock = 2;
    const auto SizeBlock = 16 * 1024;
    const auto FileSize =  4096;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    // ファイルシステムの個数
    // 少なすぎると、BufferManager のバッファが埋まる前にテストが終わってしまいます
    const auto FileSystemCount = 230;

    // フォーマット用のデータの準備
    nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam paramDuplex;
    nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
        paramIntegrity;
    paramDuplex.sizeBlockLevel[0] = SizeBlock;
    paramDuplex.sizeBlockLevel[1] = SizeBlock;
    paramIntegrity.sizeBlockLevel[0] = SizeBlock;
    paramIntegrity.sizeBlockLevel[1] = SizeBlock;
    paramIntegrity.sizeBlockLevel[2] = SizeBlock;
    paramIntegrity.sizeBlockLevel[3] = SizeBlock;

    int64_t sizeTotal = 0;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
            &sizeTotal,
            SizeBlock,
            CountExpandMax,
            paramDuplex,
            paramIntegrity,
            CountDataBlock,
            CountReservedBlock
        )
    );

    // バッファマネージャの取得
    nn::fssystem::IBufferManager* pBufferManager = GetBufferManager();
    const size_t sizeFree = pBufferManager->GetFreeSize();

    // バッファマネージャが管理するバッファを消費しながら、
    // 初期化、マウント、ファイル作成、書き込み、コミットした際に、
    // 想定どおりの挙動をするか確かめます
    bool isAnyErrorOcurred = false;
    while( !isAnyErrorOcurred )
    {
        ASSERT_EQ(sizeFree, pBufferManager->GetFreeSize());

        // ファイルシステムとベースストレージを複数用意します
        std::unique_ptr<nnt::fs::util::AccessCountedMemoryStorage[]> storageDataList
            (new nnt::fs::util::AccessCountedMemoryStorage[FileSystemCount]);
        std::unique_ptr<nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver[]> fileSystemsList
            (new nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver[FileSystemCount]);

        for( int i = 0; i < FileSystemCount; ++i )
        {
            // ストレージ作成
            storageDataList[i].Initialize(sizeTotal);

            // ストレージをフォーマット(バッファは消費しない)
            nn::fs::SaveDataHashSalt salt;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::Format(
                    nn::fs::SubStorage(&storageDataList[i], 0, sizeTotal),
                    SizeBlock,
                    CountExpandMax,
                    paramDuplex,
                    paramIntegrity,
                    CountDataBlock,
                    CountReservedBlock,
                    pBufferManager,
                    nnt::fs::util::GetMacGenerator(),
                    salt
                )
            );

            // ストレージをマウント(バッファ消費)
            NNT_ASSERT_RESULT_SUCCESS(
                fileSystemsList[i].Initialize(
                    nn::fs::SubStorage(&storageDataList[i], 0, sizeTotal),
                    GetBufferManager(),
                    nnt::fs::util::GetMacGenerator(),
                    nnt::fs::util::SaveDataMinimumVersion
                )
            );

            // ファイル作成(バッファ消費)
            std::unique_ptr<nn::fssystem::save::Path> pPath(new nn::fssystem::save::Path);
            char fileName[32];
            nn::util::SNPrintf(fileName, sizeof(fileName), "/Save%d.txt", i);
            pPath->Initialize(fileName);

            nn::Result resultCreateFile = fileSystemsList[i].CreateFile(*pPath, FileSize);
            if( resultCreateFile.IsFailure() )
            {
                NNT_ASSERT_RESULT_FAILURE(
                    nn::fs::ResultMappingTableFull,
                    resultCreateFile
                );
                isAnyErrorOcurred = true;
            }

            // ファイル書き込み(バッファ消費)
            // バッファ消費量をテスト実行のたびに変えるために、この処理はランダムで行います
            if( std::uniform_int_distribution<uint32_t>(0, 1)(mt) == 0 )
            {
                // ファイルを開く(バッファは消費しない)
                nn::fssystem::save::IFile* pFile = nullptr;
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystemsList[i].OpenFile(
                        &pFile,
                        *pPath,
                        nn::fs::OpenMode_Write
                    )
                );

                // ファイル書き込み(バッファ消費)
                std::unique_ptr<char[]> buffer(new char[FileSize]);
                nn::Result resultWriteFile = pFile->WriteBytes(0, buffer.get(), FileSize);

                if( resultWriteFile.IsFailure() )
                {
                    NNT_ASSERT_RESULT_FAILURE(
                        nn::fs::ResultMappingTableFull,
                        resultWriteFile
                    );
                    isAnyErrorOcurred = true;
                }

                fileSystemsList[i].CloseFile(pFile);
            }

            // ジャーナルサイズが小さいので、コミットは必ず失敗します
            // この処理はランダムで行います
            if( std::uniform_int_distribution<uint32_t>(0, 1)(mt) == 0 )
            {
                NNT_ASSERT_RESULT_FAILURE(
                    nn::fs::ResultMappingTableFull,
                    fileSystemsList[i].Commit()
                );
            }
            NN_SDK_LOG(".");
        }
        NN_SDK_LOG("\n");
    }
} // NOLINT(impl/function_size)

// 0 バイト書き込みやクローズ時のファイルアクセスなどをテストします。
TEST_F(FsJournalIntegritySaveDataFileSystemDriverTest, Flush)
{
    static const size_t TestLoop = 3;
    static const int CountExpandMax = 25;
    static const auto HashSize = nn::crypto::Sha256Generator::HashSize;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( size_t sizeBlock = 512; sizeBlock <= MaxBlockSize; sizeBlock *= 2 )
    {
        NN_SDK_LOG(".");

        for( int32_t loop = 0; loop < TestLoop; ++loop )
        {
            // ブロックサイズ
            nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam paramDuplex;
            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
                paramIntegrity;
            const size_t sizeBlock1 = GenerateRandomBlockSize(512, MaxBlockSize);
            // sizeBlock1 == sizeBlock2 == 512 だと検証層のサイズが不足することがある
            const size_t minSizeBlock2 = (sizeBlock1 == 512) ? 1024 : 512;
            const size_t sizeBlock2 = GenerateRandomBlockSize(minSizeBlock2, MaxBlockSize);
            paramDuplex.sizeBlockLevel[0] = sizeBlock1 / 4;
            paramDuplex.sizeBlockLevel[1] = sizeBlock2;
            paramIntegrity.sizeBlockLevel[0] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[1] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[2] = sizeBlock2;
            paramIntegrity.sizeBlockLevel[3] = sizeBlock2;

            const auto countMax = std::min<int64_t>(
                                      static_cast<uint64_t>(paramIntegrity.sizeBlockLevel[0])
                                      / HashSize * paramIntegrity.sizeBlockLevel[1]
                                      / HashSize * paramIntegrity.sizeBlockLevel[2]
                                      / HashSize * paramIntegrity.sizeBlockLevel[3]
                                      / sizeBlock,
                                      120
                                  );
            uint32_t countEntryFile =
                std::uniform_int_distribution<uint32_t>(
                    1,
                    static_cast<uint32_t>(countMax * 5) / 6
                )(mt);
            uint32_t countMetaBlock = nn::util::DivideUp(countEntryFile, 4);
            uint32_t countDataBlock;
            if (countMetaBlock > (countMax - 1) / 6)
            {
                countDataBlock = static_cast<uint32_t>(countMax * 5 / 6 + countMetaBlock);
            }
            else
            {
                countDataBlock = static_cast<uint32_t>(
                    std::uniform_int_distribution<int64_t>(countMetaBlock, (countMax - 1) / 6)(mt)
                    + countMax * 5 / 6
                );
            }

            // 予約領域のサイズをデータ領域と等しくすると全領域の書き換えが可能です。
            const auto countReservedBlock = countDataBlock;

            int64_t sizeTotal = 0;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
                    &sizeTotal,
                    sizeBlock,
                    CountExpandMax,
                    paramDuplex,
                    paramIntegrity,
                    countDataBlock,
                    countReservedBlock
                )
            );

            // ファイルシステムを配置する Storage を構築する
            nnt::fs::util::AccessCountedMemoryStorage storageData;
            storageData.Initialize(sizeTotal);

            // JournalIntegritySaveDataFileSystemDriver は、必ず 16KB アライメントのアクセスが発生する仕様のため
            // 16 KB を超えるアクセスアライメントのチェックは不要です。
            size_t sizeAlignment = std::min(sizeBlock, static_cast<size_t>(16 * 1024));
            nnt::fs::util::AlignCheckStorage storageAlignCheck;
            storageAlignCheck.Initialize(
                nn::fs::SubStorage(&storageData, 0, sizeTotal),
                sizeAlignment
            );

            nn::fs::SaveDataHashSalt salt;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::Format(
                    nn::fs::SubStorage(&storageAlignCheck, 0, sizeTotal),
                    sizeBlock,
                    CountExpandMax,
                    paramDuplex,
                    paramIntegrity,
                    countDataBlock,
                    countReservedBlock,
                    GetBufferManager(),
                    nnt::fs::util::GetMacGenerator(),
                    salt
                )
            );

            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver fileSystem;
            NNT_ASSERT_RESULT_SUCCESS(
                fileSystem.Initialize(nn::fs::SubStorage(&storageAlignCheck, 0, sizeTotal),
                    GetBufferManager(),
                    nnt::fs::util::GetMacGenerator(),
                    nnt::fs::util::SaveDataMinimumVersion
                )
            );

            // アクセス回数リセット
            storageData.ResetAccessCounter();

            // バッファ作成
            nnt::fs::util::Vector<char> data(sizeBlock * 2);
            char* ptr = &data[0];
            for( size_t i = 0; i < data.size(); ++i )
            {
                ptr[i] = i & 0xFF;
            }

            nn::fssystem::save::IFileSystem* pFileSystem = &fileSystem;

            // ディレクトリを新規作成します。
            nn::fssystem::save::Path pathDirectory;
            NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize("/directory"));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(pathDirectory));

            // 適当な位置に書き込み
            uint32_t leftBlockCount = countDataBlock - countMetaBlock;
            for( size_t i = 0; i < countEntryFile; ++i )
            {
                char buf[32];
                std::memset(buf, 0, sizeof(buf));
                nn::util::SNPrintf(buf, sizeof(buf), "/directory/file%08u", i);
                nn::fssystem::save::Path filePath;
                NNT_ASSERT_RESULT_SUCCESS(filePath.Initialize(buf));

                nn::fssystem::save::IFile* pFileRW = nullptr;
                const int64_t sizeFile =
                    std::uniform_int_distribution<int64_t>(
                            8,
                            (countEntryFile - i <= leftBlockCount ? sizeBlock : sizeBlock + 7)
                        )(mt);
                leftBlockCount -= static_cast<uint32_t>(nn::util::DivideUp(sizeFile, sizeBlock));
                const int64_t offset =
                    std::uniform_int_distribution<int64_t>(0, sizeFile - 1)(mt);
                size_t size =
                    std::uniform_int_distribution<size_t>(
                        1, static_cast<size_t>(sizeFile - offset))(mt);
                if( offset + static_cast<int64_t>(size) > sizeFile )
                {
                    size = static_cast<size_t>(sizeFile - offset);
                }

                // アクセス回数リセット
                storageData.ResetAccessCounter();

                NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(filePath, sizeFile));
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFileRW, filePath, nn::fs::OpenMode_Write));
                NNT_ASSERT_RESULT_SUCCESS(pFileRW->WriteBytes(offset, ptr, size));

                ASSERT_GE(8, storageData.GetFlushTimes());

                // アクセス回数リセット
                storageData.ResetAccessCounter();

                pFileSystem->CloseFile(pFileRW);

                // クローズでファイルアクセスは発生しない
                ASSERT_EQ(0, storageData.GetReadTimes());
                ASSERT_EQ(0, storageData.GetWriteTimes());
            }

            // アクセス回数リセット
            storageData.ResetAccessCounter();

            // 0 バイト書き込み(flush == true)
            for( size_t i = 0; i < countEntryFile; ++i )
            {
                char buf[32];
                nn::util::SNPrintf(buf, sizeof(buf), "/directory/file%08u", i);
                nn::fssystem::save::Path filePath;
                NNT_ASSERT_RESULT_SUCCESS(filePath.Initialize(buf));

                nn::fssystem::save::IFile* pFile = nullptr;
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, filePath, nn::fs::OpenMode_Write));

                int64_t sizeFile;
                NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&sizeFile));

                const int64_t offset =
                    std::uniform_int_distribution<int64_t>(0, sizeFile - 1)(mt);

                // アクセス回数リセット
                storageData.ResetAccessCounter();

                NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(offset, ptr, 0));

                ASSERT_EQ(0, storageData.GetFlushTimes());

                // アクセス回数リセット
                storageData.ResetAccessCounter();

                NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());

                ASSERT_LE(0, storageData.GetFlushTimes());

                // アクセス回数リセット
                storageData.ResetAccessCounter();

                NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(0, ptr, 0));

                ASSERT_EQ(0, storageData.GetFlushTimes());

                // アクセス回数リセット
                storageData.ResetAccessCounter();

                pFileSystem->CloseFile(pFile);

                // クローズでファイルアクセスは発生しない
                ASSERT_EQ(0, storageData.GetReadTimes());
                ASSERT_EQ(0, storageData.GetWriteTimes());
            }

            // アクセス回数リセット
            storageData.ResetAccessCounter();

            // 範囲外に書き込み
            for( size_t i = 0; i < countEntryFile; ++i )
            {
                char buf[32];
                nn::util::SNPrintf(buf, sizeof(buf), "/directory/file%08u", i);
                nn::fssystem::save::Path filePath;
                NNT_ASSERT_RESULT_SUCCESS(filePath.Initialize(buf));

                nn::fssystem::save::IFile* pFileLoop = nullptr;
                NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(
                    &pFileLoop, filePath, nn::fs::OpenMode_Write));

                int64_t sizeFile;
                NNT_ASSERT_RESULT_SUCCESS(pFileLoop->GetSize(&sizeFile));

                const int64_t offset =
                    std::uniform_int_distribution<int64_t>(sizeFile + 1, sizeFile * 2)(mt);
                const size_t size =
                    std::uniform_int_distribution<size_t>(1, static_cast<size_t>(sizeFile))(mt);

                // アクセス回数リセット
                storageData.ResetAccessCounter();

                NNT_ASSERT_RESULT_FAILURE(
                    nn::fs::ResultInvalidOffset,
                    pFileLoop->WriteBytes(offset, ptr, size)
                );

                // 範囲外エラー発生時に書き込みは行われない
                ASSERT_EQ(0, storageData.GetWriteTimes());
                ASSERT_EQ(0, storageData.GetFlushTimes());

                // アクセス回数リセット
                storageData.ResetAccessCounter();

                pFileSystem->CloseFile(pFileLoop);

                // クローズでファイルアクセスは発生しない
                ASSERT_EQ(0, storageData.GetReadTimes());
                ASSERT_EQ(0, storageData.GetWriteTimes());
            }

            // 明示的なコミットを実行
            NNT_ASSERT_RESULT_SUCCESS(fileSystem.Commit());

            // アクセス回数リセット
            storageData.ResetAccessCounter();

            fileSystem.Finalize();

            // アンマウント処理で書き込みは発生しない
            ASSERT_EQ(0, storageData.GetWriteTimes());

            // クローズでファイルアクセスは発生しない
            ASSERT_EQ(0, storageData.GetReadTimes());
            ASSERT_EQ(0, storageData.GetWriteTimes());
        }
    }

    NN_SDK_LOG("\n");
} // NOLINT(impl/function_size)

// ファイルサイズの上限値をテストします。
TEST_F(FsJournalIntegritySaveDataFileSystemDriverTest, LimitSize)
{
    static const size_t TestLoop = 10;
    static const int CountExpandMax = 25;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( size_t sizeBlock = 512; sizeBlock <= MaxBlockSize; sizeBlock *= 2 )
    {
        NN_SDK_LOG(".");

        for( int32_t loop = 0; loop < TestLoop; ++loop )
        {
            // ブロックサイズ
            nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam paramDuplex;
            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
                paramIntegrity;
            const size_t sizeBlock1 = GenerateRandomBlockSize(512, 4096);
            const size_t sizeBlock2 = GenerateRandomBlockSize(512, 2048);
            paramDuplex.sizeBlockLevel[0] = sizeBlock1 / 4;
            paramDuplex.sizeBlockLevel[1] = sizeBlock2;
            paramIntegrity.sizeBlockLevel[0] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[1] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[2] = sizeBlock2;
            paramIntegrity.sizeBlockLevel[3] =
                (std::uniform_int_distribution<>(0, 1)(mt) != 0) ? sizeBlock1 : sizeBlock2;

            static const auto HashSize = nn::crypto::Sha256Generator::HashSize;
            const auto countMax = std::min<int64_t>(
                                      static_cast<uint64_t>(paramIntegrity.sizeBlockLevel[0])
                                      / HashSize * paramIntegrity.sizeBlockLevel[1]
                                      / HashSize * paramIntegrity.sizeBlockLevel[2]
                                      / HashSize * paramIntegrity.sizeBlockLevel[3]
                                      / sizeBlock,
                                      20
                                  );
            uint32_t countDataBlock =
                static_cast<uint32_t>(
                    std::uniform_int_distribution<int64_t>(0, countMax - 3)(mt) + 3);
            const auto countReservedBlock = countDataBlock;

            int64_t sizeTotal = 0;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
                    &sizeTotal,
                    sizeBlock,
                    CountExpandMax,
                    paramDuplex,
                    paramIntegrity,
                    countDataBlock,
                    countReservedBlock
                )
            );

            nnt::fs::util::AccessCountedMemoryStorage baseStorageData;
            baseStorageData.Initialize(sizeTotal);

            nnt::fs::util::AlignCheckStorage storageData;
            // JournalIntegritySaveDataFileSystemDriver は、必ず 16KB アライメントのアクセスが発生する仕様のため
            // 16 KB を超えるアクセスアライメントのチェックは不要です。
            size_t sizeAlignment = std::min(sizeBlock, static_cast<size_t>(16 * 1024));
            storageData.Initialize(
                nn::fs::SubStorage(&baseStorageData, 0, sizeTotal),
                sizeAlignment
            );

            nn::fs::SaveDataHashSalt salt;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::Format(
                    nn::fs::SubStorage(&storageData, 0, sizeTotal),
                    sizeBlock,
                    CountExpandMax,
                    paramDuplex,
                    paramIntegrity,
                    countDataBlock,
                    countReservedBlock,
                    GetBufferManager(),
                    nnt::fs::util::GetMacGenerator(),
                    salt
                )
            );

            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver fileSystem;
            NNT_ASSERT_RESULT_SUCCESS(
                fileSystem.Initialize(
                    nn::fs::SubStorage(&storageData, 0, sizeTotal),
                    GetBufferManager(),
                    nnt::fs::util::GetMacGenerator(),
                    nnt::fs::util::SaveDataMinimumVersion
                )
            );

            // バッファ作成
            nnt::fs::util::Vector<char> data(sizeBlock * 2);
            char* ptr = &data[0];
            for( size_t i = 0; i < data.size(); ++i )
            {
                ptr[i] = i & 0xFF;
            }

            nn::fssystem::save::IFileSystem* pFileSystem = &fileSystem;

            // エントリ分のブロックを差し引いて上限サイズを計算
            int64_t sizeLimit = sizeBlock * (countDataBlock - 2);
            {
                char buf[32];
                std::memset(buf, 0, sizeof(buf));
                nn::util::SNPrintf(
                    buf, sizeof(buf), "/file%08u", static_cast<uint32_t>(sizeLimit));
                nn::fssystem::save::Path filePath;
                NNT_ASSERT_RESULT_SUCCESS(filePath.Initialize(buf));

                // 上限サイズのファイルを作成します。
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->CreateFile(
                        filePath,
                        sizeLimit
                    )
                );

                // ファイルを削除します。
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->DeleteFile(filePath)
                );

                // 上限 + 1 サイズのファイルを作成します。
                NNT_ASSERT_RESULT_FAILURE(
                    nn::fs::ResultOutOfResource,
                    pFileSystem->CreateFile(
                        filePath,
                        sizeLimit + 1
                    )
                );
            }

            // 明示的なコミットを実行
            NNT_ASSERT_RESULT_SUCCESS(fileSystem.Commit());

            fileSystem.Finalize();
        }
    }

    NN_SDK_LOG("\n");
} // NOLINT(impl/function_size)

//! 不正なサイズを指定すると初期化に失敗することを確認します。
TEST_F(FsJournalIntegritySaveDataFileSystemDriverTest, InvalidSize)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    static const auto LoopCount = 5;
    static const auto CountExpandMax = 25;

    for( size_t sizeBlock = 4096; sizeBlock <= MaxBlockSize; sizeBlock *= 2 )
    {
        NN_SDK_LOG(".");

        for( auto loop = 0; loop < LoopCount; ++loop )
        {
            // ブロックサイズ
            nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam paramDuplex;
            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
                paramIntegrity;
            paramDuplex.sizeBlockLevel[0] = GenerateRandomBlockSize(512, MaxBlockSize) / 4;
            paramDuplex.sizeBlockLevel[1] = GenerateRandomBlockSize(512, MaxBlockSize);
            paramIntegrity.sizeBlockLevel[0] = GenerateRandomBlockSize(512, MaxBlockSize);
            paramIntegrity.sizeBlockLevel[1] = GenerateRandomBlockSize(512, MaxBlockSize);
            paramIntegrity.sizeBlockLevel[2] = GenerateRandomBlockSize(512, MaxBlockSize);
            paramIntegrity.sizeBlockLevel[3] = GenerateRandomBlockSize(512, MaxBlockSize);

            static const auto HashSize = nn::crypto::Sha256Generator::HashSize;
            const int64_t countMax = static_cast<int64_t>(paramIntegrity.sizeBlockLevel[0])
                / HashSize * paramIntegrity.sizeBlockLevel[1]
                / HashSize * paramIntegrity.sizeBlockLevel[2]
                / HashSize * paramIntegrity.sizeBlockLevel[3]
                / sizeBlock;
            int64_t countDataBlock = 0;
            auto countReservedBlock = std::uniform_int_distribution<int64_t>(1, countMax * 2)(mt);

            switch( loop )
            {
            case 0:
                countDataBlock = countMax;
                break;
            case 1:
                countDataBlock = countMax + 1;
                break;
            default:
                countDataBlock = std::uniform_int_distribution<int64_t>(1, countMax * 2)(mt);
                break;
            }

            if( !nn::util::IsIntValueRepresentable<uint32_t>(countDataBlock)
                || !nn::util::IsIntValueRepresentable<uint32_t>(countReservedBlock) )
            {
                continue;
            }

            int64_t sizeTotal = 0;

            for( ; ; )
            {
                const auto result = nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
                                        &sizeTotal,
                                        sizeBlock,
                                        CountExpandMax,
                                        paramDuplex,
                                        paramIntegrity,
                                        static_cast<uint32_t>(countDataBlock),
                                        static_cast<uint32_t>(countReservedBlock)
                                    );
                if( countDataBlock <= countMax )
                {
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    break;
                }
                else
                {
                    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidSize, result);
                    countDataBlock /= 2;
                    countReservedBlock /= 2;
                }
            }

#if 0 // TODO: Format() が高速化されたら有効化する
            {
                nnt::fs::util::VirtualMemoryStorage storageData;
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver fileSystem;

                storageData.Initialize(sizeTotal);

                nn::fs::SubStorage storageBase(&storageData, 0, sizeTotal);

                NNT_ASSERT_RESULT_SUCCESS(
                    nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::Format(
                        nn::fs::SubStorage(&storageData, 0, sizeTotal),
                        sizeBlock,
                        CountExpandMax,
                        paramDuplex,
                        paramIntegrity,
                        static_cast<uint32_t>(countDataBlock),
                        static_cast<uint32_t>(countReservedBlock)
                    )
                );

                NNT_ASSERT_RESULT_SUCCESS(fileSystem.Initialize(storageBase));
                fileSystem.Finalize();
            }
#endif
        }
    }

    NN_SDK_LOG("\n");
} // NOLINT(impl/function_size)


#if 0
struct MThreadTestArg
{
    nn::fssystem::save::IFileSystem* pFileSystem;
    uint32_t nId;
    nn::fssystem::save::Random rnd;
};

// 複数のスレッドから同時にファイルシステムにアクセスします。
static void OnTestMThread(const void *p)
{
    MThreadTestArg* pArg = (MThreadTestArg *)p;
    nn::fssystem::save::IFileSystem* pFileSystem = pArg->pFileSystem;
    uint32_t threadId = pArg->nId;
    nn::fssystem::save::Random *r = &pArg->rnd;
    static const size_t TEST_BUF_SIZE = 128;
    uint8_t tmpbuf1[TEST_BUF_SIZE];
    uint8_t tmpbuf2[TEST_BUF_SIZE];
    char fnbuf[32];
    char fnbuf2[32];
    char fnbuf3[32];
    nn::Result result;

    // テストバッファを乱数で埋めます。
    for( int32_t j = 0; j < TEST_BUF_SIZE; ++j )
    {
        tmpbuf1[j] = static_cast<uint8_t>(r->Get32(256) & 0xFF);
    }
    for( int32_t i = 0; i < 100; ++i )
    {
        if ((threadId == 0) && ((i + 1) % 10 == 0))
        {
            NN_SDK_LOG(".");
        }

        nn::fssystem::save::IFile* pFile;
        nn::fssystem::save::IDirectory* pDir;

        switch (r->Get32(7))
        {
        case 0:
            {
                std::memset(fnbuf, 0, sizeof(fnbuf));
                std::sprintf(fnbuf, "/file%u", threadId);
                nn::fssystem::save::Path pathCurr(static_cast<const char *>(fnbuf));

                result = pFileSystem->CreateFile(pathCurr, 0);
                if (result.IsFailure())
                {
                    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDbmKeyFull, result);
                }
                else
                {
                    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, pathCurr, nn::fs::OpenMode_Write));
                    int64_t size = r->Get32(TEST_BUF_SIZE);
                    NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(size));
                    NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(0, tmpbuf1, static_cast<size_t>(size)));
                    pFileSystem->CloseFile(pFile);

                    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, pathCurr, nn::fs::OpenMode_Read));
                    NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, tmpbuf2, static_cast<size_t>(size)));
                    ASSERT_EQ(0, std::memcmp(tmpbuf1, tmpbuf2, static_cast<size_t>(size)));
                    pFileSystem->CloseFile(pFile);

                    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(pathCurr));
                }
            }
            break;
        case 1:
            {
                std::memset(fnbuf, 0, sizeof(fnbuf));
                std::sprintf(fnbuf, "/dir%u-1", threadId);
                std::memset(fnbuf2, 0, sizeof(fnbuf));
                std::sprintf(fnbuf2, "/dir%u-2", threadId);
                nn::fssystem::save::Path pathCurr1(static_cast<const char *>(fnbuf));
                nn::fssystem::save::Path pathCurr2(static_cast<const char *>(fnbuf2));
                nn::fssystem::save::Path pathRoot("/");

                result = pFileSystem->CreateDirectory(pathCurr1);
                if (result.IsFailure())
                {
                    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDbmKeyFull, result);
                }
                else
                {
                    result = pFileSystem->CreateDirectory(pathCurr2);
                    if (result.IsFailure())
                    {
                        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDbmKeyFull, result);
                        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(pathCurr1));
                    }
                    else
                    {
                        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenDirectory(&pDir, pathRoot));
                        int32_t nDoneCount;
                        nn::fs::DirectoryEntry de[2];
                        NNT_ASSERT_RESULT_SUCCESS(pDir->Read(&nDoneCount, de, 2));
                        ASSERT_EQ(2, nDoneCount);
                        pFileSystem->CloseDirectory(pDir);
                        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(pathCurr1));
                        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(pathCurr2));
                    }
                }
            }
            break;
        case 2:
            {
                std::memset(fnbuf, 0, sizeof(fnbuf));
                std::sprintf(fnbuf, "/dir%u-1", threadId);
                std::memset(fnbuf2, 0, sizeof(fnbuf));
                std::sprintf(fnbuf2, "/dir%u-2", threadId);
                nn::fssystem::save::Path pathCurr1(static_cast<const char *>(fnbuf));
                nn::fssystem::save::Path pathCurr2(static_cast<const char *>(fnbuf2));
                result = pFileSystem->CreateDirectory(pathCurr1);
                if (result.IsFailure())
                {
                    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDbmKeyFull, result);
                }
                else
                {
                    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->RenameDirectory(pathCurr1, pathCurr2));
                    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(pathCurr2));
                }
            }
            break;
        case 3:
            {
                std::memset(fnbuf, 0, sizeof(fnbuf));
                std::sprintf(fnbuf, "/file%u-1", threadId);
                std::memset(fnbuf2, 0, sizeof(fnbuf));
                std::sprintf(fnbuf2, "/file%u-2", threadId);
                nn::fssystem::save::Path pathCurr1(static_cast<const char *>(fnbuf));
                nn::fssystem::save::Path pathCurr2(static_cast<const char *>(fnbuf2));
                result = pFileSystem->CreateFile(pathCurr1, 0);
                if (result.IsFailure())
                {
                    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDbmKeyFull, result);
                }
                else
                {
                    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, pathCurr1, nn::fs::OpenMode_Write));
                    NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(TEST_BUF_SIZE));
                    NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(0, tmpbuf1, TEST_BUF_SIZE));
                    pFileSystem->CloseFile(pFile);

                    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->RenameFile(pathCurr1, pathCurr2));

                    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, pathCurr2, nn::fs::OpenMode_Read));
                    NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, tmpbuf2, TEST_BUF_SIZE));
                    ASSERT_EQ(0, std::memcmp(tmpbuf1,tmpbuf2,TEST_BUF_SIZE));
                    pFileSystem->CloseFile(pFile);

                    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(pathCurr2));
                }
            }
            break;
        case 4:
            {
                std::memset(fnbuf, 0, sizeof(fnbuf));
                std::sprintf(fnbuf, "/dir%u", threadId);
                nn::fssystem::save::Path pathDir(static_cast<const char *>(fnbuf));
                result = pFileSystem->CreateDirectory(pathDir);
                if (result.IsFailure())
                {
                    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDbmKeyFull, result);
                }
                else
                {
                    std::memset(fnbuf2, 0, sizeof(fnbuf2));
                    std::sprintf(fnbuf2, "/dir%u/file%u", threadId, threadId);
                    nn::fssystem::save::Path pathCurr(static_cast<const char *>(fnbuf2));

                    std::memset(fnbuf3, 0, sizeof(fnbuf3));
                    std::sprintf(fnbuf3, "/file%u", threadId);
                    nn::fssystem::save::Path pathRename(static_cast<const char *>(fnbuf3));

                    result = pFileSystem->CreateFile(pathCurr, 0);
                    if (result.IsFailure())
                    {
                        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDbmKeyFull, result);
                        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(pathDir));
                    }
                    else
                    {
                        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, pathCurr, nn::fs::OpenMode_Write));
                        int64_t size = r->Get32(TEST_BUF_SIZE);
                        NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(size));
                        NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(0, tmpbuf1, static_cast<size_t>(size)));
                        pFileSystem->CloseFile(pFile);

                        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, pathCurr, nn::fs::OpenMode_Read));
                        NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, tmpbuf2, static_cast<size_t>(size)));
                        ASSERT_EQ(0, std::memcmp(tmpbuf1, tmpbuf2, static_cast<size_t>(size)));
                        pFileSystem->CloseFile(pFile);

                        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->RenameFile(pathCurr, pathRename));
                        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(pathDir));
                        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(pathRename));
                    }
                }
            }
            break;
        case 5:
            {
                int64_t tmp;
                for (uint32_t n = 0;n < 1000;n++)
                {
                    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->GetFreeBytes(&tmp));
                    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->GetDataAreaBytes(&tmp));
                }
            }
            break;
        case 6:
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
            }
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    delete pArg;
} // NOLINT(impl/function_size)

//! 複数のインスタンスを作成し、同時にリードライトを行っても
//! キャッシュ不足にならないことを検証します。
void FsJournalIntegritySaveDataFileSystemDriverTest::TestMultiArchiveInternal()
{
    static const int32_t INSTANCE_COUNT = 20;
    static const size_t THREAD_STACK_SIZE = 16 * 1024;

    nn::fssystem::save::Random rnd;
    UserSaveDataFileSystemTestSetup setup[INSTANCE_COUNT];
    nn::os::ThreadType threads[INSTANCE_COUNT];

    for( int32_t i = 0; i < sizeof(setup) / sizeof(setup[0]); ++i )
    {
        setup[i].Initialize(0, GenerateRandomBlockSize(256, MaxBlockSize));
    }

    // 複数のスレッドからアクセスします。
    for( uint32_t i = 0; i < sizeof(threads) / sizeof(threads[0]); ++i )
    {
        MThreadTestArg* pArg = new MThreadTestArg ();
        pArg->pFileSystem = setup[i].GetFileSystem();
        pArg->nId = i;
        pArg->rnd.SetSeed(123456789ULL * (i + 1));
        NNT_ASSERT_RESULT_SUCCESS(
            nn::os::CreateThread(
                &threads[i],
                OnTestMThread,
                pArg,
                THREAD_STACK_SIZE,
                NN_OS_DEFAULT_THREAD_PRIORITY - i
            )
        );
        nn::os::StartThread(&threads[i]);
    }
    for (uint32_t i = 0; i < sizeof(threads) / sizeof(threads[0]); i++)
    {
        nn::os::WaitThread(&threads[i]);
        nn::os::DestroyThread(&threads[i]);
    }

    for (int32_t i = 0; i < sizeof(setup) / sizeof(setup[0]); i++)
    {
        setup[i].Finalize();
    }
}

TEST_F(FsJournalIntegritySaveDataFileSystemDriverTest, TestMultiArchive)
{
    TestMultiArchiveInternal();
}

void FsJournalIntegritySaveDataFileSystemDriverTest::TestRandomErrorDetail(
         nn::fs::SubStorage storage,
         uint32_t seed
     )
{
    static const int64_t sizeVerify = 192 * 1024;

    static const nn::Result resultOnErrorWrite = nnt::fs::util::AccessCountedMemoryStorage::GetRandomWriteErrorResult();
    static const nn::Result resultOnErrorRead = nnt::fs::util::AccessCountedMemoryStorage::GetRandomReadErrorResult();

    nn::Result result;
    nnt::fs::util::Random rnd(seed);

    nnt::fs::util::Vector<uint8_t> bufVerify(sizeVerify);
    uint8_t* pBufVerify = &bufVerify[0];
    nnt::fs::util::Vector<uint8_t> bufRead(sizeVerify);
    uint8_t* pBufRead = &bufRead[0];


retry:
    std::unique_ptr<nn::fssystem::save::IFileSystem> pFileSystem;
    pFileSystem.reset(new nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver());
    for( ; ; )
    {
        result = reinterpret_cast<nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver *>(pFileSystem.get())->Initialize(storage);
        if (! result.IsFailure())
        {
            break;
        }
    }

    // ベリファイテストを行なうファイルを作成します。
    char fnbuf[32];
    std::memset(fnbuf, 0, sizeof(fnbuf));
    nn::util::SNPrintf(fnbuf, sizeof(fnbuf), "/file");
    nn::fssystem::save::Path filePath;
    NNT_ASSERT_RESULT_SUCCESS(filePath.Initialize(fnbuf));
    result = pFileSystem->CreateFile(filePath, sizeVerify);
    if (result.IsFailure())
    {
        goto retry;
    }

    // 初期データを書き込みます。
    nn::fssystem::save::IFile* pFileInit = nullptr;
    result = pFileSystem->OpenFile(&pFileInit, filePath, nn::fs::OpenMode_Write | nn::fs::OpenMode_Read);
    if (result.IsFailure())
    {
        goto retry;
    }
    for( size_t i = 0; i < bufVerify.size(); ++i )
    {
        pBufVerify[i] = (i & 255);
    }
    result = pFileInit->WriteBytes(0, pBufVerify, sizeVerify);
    if (result.IsFailure())
    {
        pFileSystem->CloseFile(pFileInit);
        goto retry;
    }
    result = pFileInit->Flush();
    if (result.IsFailure())
    {
        pFileSystem->CloseFile(pFileInit);
        goto retry;
    }

    pFileSystem->CloseFile(pFileInit);

    int64_t bytesFree;
    int64_t bytesDataArea;
    for( size_t i = 0; i < 10; ++i )
    {
        result = pFileSystem->GetDataAreaBytes(&bytesDataArea);
        if (result.IsFailure())
        {
            goto retry;
        }

        result = pFileSystem->GetFreeBytes(&bytesFree);
        if (result.IsFailure())
        {
            goto retry;
        }

        bool bHas;
        nn::fssystem::save::Path pathRoot;
        NNT_ASSERT_RESULT_SUCCESS(pathRoot.Initialize("/"));
        result = pFileSystem->HasDirectory(&bHas, pathRoot);
        if (result.IsFailure())
        {
            goto retry;
        }
        ASSERT_TRUE(bHas);

        nn::fssystem::save::Path pathDir;
        NNT_ASSERT_RESULT_SUCCESS(pathDir.Initialize("/dir"));
        result = pFileSystem->HasDirectory(&bHas, pathDir);
        if (result.IsFailure())
        {
            goto retry;
        }
        ASSERT_TRUE(!bHas);

        nn::fssystem::save::Path pathFile;
        NNT_ASSERT_RESULT_SUCCESS(pathFile.Initialize("/file"));
        result = pFileSystem->HasFile(&bHas, pathFile);
        if (result.IsFailure())
        {
            goto retry;
        }
        ASSERT_TRUE(bHas);

        nn::fssystem::save::Path pathFile2;
        NNT_ASSERT_RESULT_SUCCESS(pathFile2.Initialize("/file2"));
        result = pFileSystem->HasFile(&bHas, pathFile2);
        if (result.IsFailure())
        {
            goto retry;
        }
        ASSERT_TRUE(!bHas);
    }

    // この時点のデータをコミットしておきます。
    result = reinterpret_cast<nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver *>(pFileSystem.get())->Commit();
    if( result.IsFailure() )
    {
        goto retry;
    }

    for( int32_t loop = 0; loop < 10; ++loop )
    {
        //NN_LOG(".");

        nn::fssystem::save::IFile* pFileRW = nullptr;
        result = pFileSystem->OpenFile(&pFileRW, filePath, nn::fs::OpenMode_Write | nn::fs::OpenMode_Read);
        if (result.IsFailure())
        {
            ASSERT_TRUE(
                resultOnErrorRead.CanAccept(result)
             || nn::fs::ResultUnexpected::Includes(result)
            );
            continue;
        }

        // ファイルデータが化けていないかメモリ上のデータとベリファイします。
        result = pFileRW->ReadBytes(0, pBufRead, sizeVerify);
        if (result.IsFailure())
        {
            ASSERT_TRUE(
                resultOnErrorRead.CanAccept(result)
             || nn::fs::ResultUnexpected::Includes(result)
            );
        }
        else
        {
            if (std::memcmp(pBufRead, pBufVerify, sizeVerify) != 0)
            {
                NN_SDK_LOG("Verify error0!!!!!!!!!!!!\n");
                ASSERT_TRUE(false);
            }
        }

        uint8_t bufWrite[1024];
        int64_t sizeRW = rnd.Get32(sizeof(bufWrite) - 1) + 1;
        int64_t offsetRW = rnd.Get32(static_cast<uint32_t>(sizeVerify - sizeRW));

        // 一定回数データを書き込み続けます。
        bool isWriteError = false;
        int32_t countWriteMax = rnd.Get32(20) + 10;
        for (int32_t countWrite = 0; countWrite < countWriteMax; countWrite++)
        {
            std::memset(bufWrite, countWrite & 0xFF, sizeof(bufWrite));
            result = pFileRW->WriteBytes(offsetRW, bufWrite, static_cast<size_t>(sizeRW));
            if (result.IsFailure())
            {
                ASSERT_TRUE(
                    resultOnErrorWrite.CanAccept(result)
                 || resultOnErrorRead.CanAccept(result)
                 || nn::fs::ResultUnexpected::Includes(result)
                );
                isWriteError = true;
                break;
            }
        }
        if( isWriteError )
        {
            pFileSystem->CloseFile(pFileRW);

            // コミットをせずに再度マウントし、データを巻き戻します。
            pFileSystem.reset(new nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver());
            for( ; ; )
            {
                result = reinterpret_cast<nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver *>(pFileSystem.get())->Initialize(storage);
                if (! result.IsFailure())
                {
                    break;
                }
                ASSERT_TRUE(
                    resultOnErrorRead.CanAccept(result)
                 || nn::fs::ResultUnexpected::Includes(result)
                );
            }

            // 巻き戻したファイルデータと保存してあったデータをベリファイを行ないます。
            nn::fssystem::save::IFile* pFileVerify = nullptr;
            result = pFileSystem->OpenFile(&pFileVerify, filePath, nn::fs::OpenMode_Read);
            if (result.IsFailure())
            {
                ASSERT_TRUE(
                    resultOnErrorRead.CanAccept(result)
                 || nn::fs::ResultUnexpected::Includes(result)
                );
            }
            else
            {
                result = pFileVerify->ReadBytes(0, pBufRead, sizeVerify);
                if (result.IsFailure())
                {
                    ASSERT_TRUE(resultOnErrorRead.CanAccept(result));
                }
                else
                {
                    ASSERT_TRUE(std::memcmp(pBufRead, pBufVerify, static_cast<size_t>(sizeVerify)) == 0);
                }
                pFileSystem->CloseFile(pFileVerify);
            }
        }
        else
        {
            bool bCopy = true;

            // 現在のデータをメモリに保存し、コミットを行ないます。
            result = pFileRW->ReadBytes(0, pBufRead, sizeVerify);
            if (result.IsFailure())
            {
                bCopy = false;
            }

            pFileSystem->CloseFile(pFileRW);

            if (bCopy)
            {
                result = reinterpret_cast<nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver *>(pFileSystem.get())->Commit();
                if (result.IsFailure())
                {
                    bCopy = false;
                }
            }

            if (bCopy)
            {
                // Commitに成功したので、正しいデータとして覚えてきます
                std::memcpy(pBufVerify, pBufRead, sizeVerify);
            }
            else
            {
                // コミットをせずに再度マウントし、データを巻き戻します。
                pFileSystem.reset(new nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver());
                for( ; ; )
                {
                    result = reinterpret_cast<nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver *>(pFileSystem.get())->Initialize(storage);
                    if (! result.IsFailure())
                    {
                        break;
                    }
                    ASSERT_TRUE(resultOnErrorRead.CanAccept(result));
                }
            }
        }
    }

    pFileSystem->DeleteFile(filePath);
} // NOLINT(impl/function_size)

void FsJournalIntegritySaveDataFileSystemDriverTest::TestRandomErrorImpl(uint32_t seed)
{
    static size_t COUNT_DATA_BLOCK = 1800;

    nnt::fs::util::Random rnd(seed);

    for( size_t sizeBlock = 512; sizeBlock <= MaxBlockSize; sizeBlock *= 2 )
    {
        for( int loop = 0; loop < 3; ++loop )
        {
            NN_SDK_LOG(".");

            nnt::fs::util::AccessCountedMemoryStorage storageData;

            size_t countDataBlock = COUNT_DATA_BLOCK;

            const auto sizeReserved = (sizeBlock * (countDataBlock + 2));

            // ブロックサイズ
            nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam paramDuplex;
            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam paramIntegrity;
            size_t sizeBlock1 = GenerateRandomBlockSize(512, MaxBlockSize);
            size_t sizeBlock2 = GenerateRandomBlockSize(512, MaxBlockSize);
            paramDuplex.sizeBlockLevel[0] = sizeBlock1 / 4;
            paramDuplex.sizeBlockLevel[1] = sizeBlock2;
            paramIntegrity.sizeBlockLevel[0] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[1] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[2] = sizeBlock2;
            paramIntegrity.sizeBlockLevel[3] = sizeBlock2;

            int64_t sizeTotal = nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
                                    sizeBlock,
                                    sizeReserved,
                                    paramDuplex,
                                    paramIntegrity,
                                    static_cast<uint32_t>(countDataBlock)
                                );

            storageData.Initialize(sizeTotal);
            nn::fs::SubStorage subStorageData(&storageData, 0, sizeTotal);

            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::Format(
                    storageData,
                    sizeBlock,
                    sizeReserved,
                    paramDuplex,
                    paramIntegrity,
                    static_cast<uint32_t>(countDataBlock)
                )
            );

            // ランダムエラーを有効にします。
            storageData.SetRandomError(true);

            TestRandomErrorDetail(subStorageData, rnd.Get32());

            storageData.SetRandomError(false);
        }
    }
}

void FsJournalIntegritySaveDataFileSystemDriverTest::TestRandomError2Impl(uint32_t seed)
{
    static size_t MAX_COUNT_ENTRY_FILE = 1000;
    static size_t MAX_COUNT_ENTRY_DIRECTORY = 1000;
    static size_t COUNT_DATA_BLOCK = 2000;

    static const nn::Result resultOnErrorWrite = AccessCountWrapperFile<nn::fssystem::save::IFile>::GetRandomWriteErrorResult();
    static const nn::Result resultOnErrorRead = AccessCountWrapperFile<nn::fssystem::save::IFile>::GetRandomReadErrorResult();

    nn::fssystem::save::Random rnd(seed);

    for( int64_t sizeBlock = 512; sizeBlock <= MaxBlockSize; sizeBlock *= 2 )
    {
        for( int loop = 0; loop < 300; ++loop )
        {
            NN_SDK_LOG(".");

            TestFileSystemTemplate<AccessCountedMemoryFile> baseFileSystem;
            nn::Result result = nn::ResultSuccess();

            size_t countDataBlock = COUNT_DATA_BLOCK;

            nn::fssystem::save::HierarchicalDuplexFileControlArea::InputParam paramDuplex;
            nn::fssystem::save::HierarchicalIntegrityVerificationFileControlArea::InputParam paramIntegrity;
            size_t sizeBlock1 = GenerateRandomBlockSize(512, MaxBlockSize);
            size_t sizeBlock2 = GenerateRandomBlockSize(512, MaxBlockSize);
            paramDuplex.sizeBlockLevel[0] = sizeBlock1 / 4;
            paramDuplex.sizeBlockLevel[1] = sizeBlock2;
            paramIntegrity.sizeBlockLevel[0] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[1] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[2] = sizeBlock2;
            paramIntegrity.sizeBlockLevel[3] = sizeBlock2;

            int64_t sizeArchive = nn::fssystem::save::SaveDataFileSystem::QueryTotalSize(
                                      static_cast<size_t>(sizeBlock),
                                      paramDuplex,
                                      paramIntegrity,
                                      static_cast<uint32_t>(countDataBlock)
                                  );

            // ランダムエラーの有効化
            AccessCountedMemoryFile* pBaseFile = baseFileSystem.GetFile();
            pBaseFile->SetRandomError(true);

            result = baseFileSystem.Initialize(sizeArchive, sizeBlock);
            if (resultOnErrorRead.CanAccept(result) || resultOnErrorWrite.CanAccept(result))
            {
                continue;
            }
            NNT_ASSERT_RESULT_SUCCESS(result);

            nn::fssystem::save::Path path("/file");

            nn::fssystem::save::IFile* pFile = nullptr;
            result = baseFileSystem.CreateFile(path, 0);
            if (result.IsSuccess())
            {
                result = baseFileSystem.OpenFile(
                            &pFile,
                            path,
                            nn::fs::OpenMode_Read | nn::fs::OpenMode_Write
                         );
            }
            if (!resultOnErrorRead.CanAccept(result) && !resultOnErrorWrite.CanAccept(result))
            {
                baseFileSystem.CloseFile(pFile);
            }
            baseFileSystem.Finalize();
        }
    }
}

TEST_F(FsJournalIntegritySaveDataFileSystemDriverTest, RandomError)
{
    TestRandomErrorImpl(0);
    NN_SDK_LOG("\n");
}

TEST_F(FsJournalIntegritySaveDataFileSystemDriverTest, RandomError2)
{
    TestRandomError2Impl(0);
    NN_SDK_LOG("\n");
}

typedef struct
{
    FsJournalIntegritySaveDataFileSystemDriverTest* m_p;
    uint32_t  seed;
} ThreadArg;

static void OnThreadRandomError(void* p0)
{
    ThreadArg *p = reinterpret_cast<ThreadArg*>(p0);
    p->m_p->TestRandomErrorImpl(p->seed);
}

TEST_F(FsJournalIntegritySaveDataFileSystemDriverTest, ThreadRandomError)
{
    static const uint32_t MAX_THREAD_COUNT = 8;
    static const uint32_t THREAD_STACK_SIZE = 32 * 1024;
    nn::os::Thread threads[MAX_THREAD_COUNT];
    ThreadArg arg[MAX_THREAD_COUNT];

    uint32_t tick = ::GetTickCount();
    NN_LOG("SEED = %d\n", tick);
    for (int32_t i = 0; i < MAX_THREAD_COUNT; i++)
    {
        arg[i].m_p = this;
        arg[i].seed = i + tick;
        NNT_ASSERT_RESULT_SUCCESS(
            threads[i].TryStartUsingAutoStack(
                OnThreadRandomError,
                &arg[i],
                THREAD_STACK_SIZE,
                NN_OS_DEFAULT_THREAD_PRIORITY - 1
            )
        );
        nn::os::StartThread(&threads[i]);
    }

    // スレッドが完了するまで待ちます。
    for (int32_t i = 0; i < MAX_THREAD_COUNT; i++)
    {
        nn::os::WaitThread(&threads[i]);
        nn::os::DestroyThread(&threads[i]);
    }

    nn::dbg::detail::Printf("\n");
}
#endif

#if 0
//! 新規作成したファイルがゼロ埋めされていることを検証します。
TEST_F(FsJournalIntegritySaveDataFileSystemDriverTest, FillZero)
{
    nn::fssystem::save::Random rnd;

    size_t sizeBlock = GenerateRandomBlockSize(256, MaxBlockSize);
    JournalIntegritySaveDataFileSystemDriverTestSetup setup;
    setup.Initialize(0, sizeBlock);

    size_t sizeFillZero = std::min(sizeBlock, static_cast<size_t>(512UL));

    nn::fssystem::save::IFileSystem* pFileSystem = setup.GetFileSystem();

    nn::fssystem::save::IFile* pFile = nullptr;
    uint8_t buf[1024];
    int64_t sizeFile;
    nn::fssystem::save::Path pathFile("/file");

    // 新規作成した sizeFillZero 未満のファイルがゼロ埋めされていることを確認します。
    sizeFile = 10 + rnd.Get32(sizeFillZero - 10UL);
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(pathFile, sizeFile));
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Read));
    NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, buf, sizeFile));
    for (int64_t i = 0; i < sizeFile; i++)
    {
        ASSERT_EQ(0, buf[i]);
    }
    pFileSystem->CloseFile(pFile);

    // 先頭をランダムなデータで埋めます。
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(pathFile));
    sizeFile = sizeFillZero;
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(pathFile, sizeFile));
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Write));
    for (int64_t i = 0; i < sizeFile; i++)
    {
        buf[i] = static_cast<uint8_t>(rnd.Get32() & 0xFF);
    }
    NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(0, buf, sizeFile, nn::fs::WriteOption()));
    pFileSystem->CloseFile(pFile);

    // 先頭 sizeFillZero バイトがゼロ埋めされていることを確認します。
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(pathFile));
    sizeFile = 600;
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(pathFile, sizeFile));
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Read));
    NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, buf, sizeFillZero));
    for (int64_t i = 0; i < sizeFillZero; i++)
    {
        ASSERT_EQ(0, buf[i]);
    }
    pFileSystem->CloseFile(pFile);

    // 再度先頭をランダムなデータで埋めます。
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(pathFile));
    sizeFile = sizeFillZero;
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(pathFile, sizeFile));
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Write));
    for (int64_t i = 0; i < sizeFile; i++)
    {
        buf[i] = static_cast<uint8_t>(rnd.Get32() & 0xFF);
    }
    NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(0, buf, sizeFile, nn::fs::WriteOption()));
    pFileSystem->CloseFile(pFile);

    // 新規作成した sizeFillZero 未満のファイルがゼロ埋めされていることを確認します。
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(pathFile));
    sizeFile = 128 + rnd.Get32(sizeFillZero - 128);
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(pathFile, sizeFile));
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Read));
    NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, buf, sizeFile));
    for (int64_t i = 0; i < sizeFile; i++)
    {
        ASSERT_EQ(0, buf[i]);
    }
    pFileSystem->CloseFile(pFile);

    setup.Finalize();
}
#endif

#if defined(NN_FS_DESTROY_SIGNATURE)
// 署名破壊フラグ付き書き込みで検証エラー状態となることを確認します。
void FsJournalIntegritySaveDataFileSystemDriverTest::TestDestroySignatureInternal(const uint32_t loopCount)
{
    nn::fssystem::save::Random random;

    size_t sizeBlocks[] = { 4096 };

    nn::fssystem::save::IFile* pFile = nullptr;
    nn::fssystem::save::Path pathFile("/file");
    nn::fssystem::save::Path pathFile2("/file2");

    const auto loopBlockMax = loopBlock < sizeof(sizeBlocks) / sizeof(sizeBlocks[0]);
    for( size_t loopBlock = 0; loopBlockMax; loopBlock++ )
    {
        size_t sizeBlock = sizeBlocks[loopBlock];
        //sizeBlock = 256 << m_Random.Get32(8);

        JournalIntegritySaveDataFileSystemDriverTestSetup setup;
        setup.SetRandomizeBlockSize(false);
        setup.Initialize(0, sizeBlock);

        nn::fssystem::save::IFileSystem* pFileSystem = setup.GetFileSystem();

        for( uint32_t loop = 0; loop < loopCount; ++loop )
        {
            NN_LOG(".");

            // ある程度の大きさの適当なサイズ
            int64_t sizeFile = sizeBlock * (8 + random.Get32(8)) + random.Get32(sizeBlock);
            nnt::fs::util::Vector<char> rbuf(sizeFile);
            nnt::fs::util::Vector<char> wbuf(sizeFile);
            nnt::fs::util::Vector<char> wbuf2(sizeFile);

            // ファイルを作成します。
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(pathFile, sizeFile));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(pathFile2, sizeFile));

            // ファイル全体をランダムなデータで埋めます。
            {
                for( int64_t i = 0; i < sizeFile; ++i )
                {
                    wbuf[i] = random.Get32() & 0xFF;
                }

                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Write));
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->WriteBytes(0, &wbuf[0], sizeFile, nn::fs::WriteOption()));
                pFileSystem->CloseFile(pFile);

                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathFile2, nn::fs::OpenMode_Write));
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->WriteBytes(0, &wbuf[0], sizeFile, nn::fs::WriteOption()));
                pFileSystem->CloseFile(pFile);
            }

            // この時点では正常にデータが読み込めることを確認します。
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Read));
                NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, &rbuf[0], sizeFile));
                ASSERT_EQ(0, std::memcmp(&rbuf[0], &wbuf[0], sizeFile));
                pFileSystem->CloseFile(pFile);

                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathFile2, nn::fs::OpenMode_Read));
                NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, &rbuf[0], sizeFile));
                ASSERT_EQ(0, std::memcmp(&rbuf[0], &wbuf[0], sizeFile));
                pFileSystem->CloseFile(pFile);
            }

            // ファイルの一部を破壊オプションありで書き込みます。
            size_t offsetW = random.Get32(sizeFile);
            size_t sizeW = random.Get32(sizeFile - offsetW) + 1;
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Write));
                nn::fs::WriteOption optionDestroy;
                optionDestroy.flush = true;
                optionDestroy.destroySignature = true;
                for( int64_t i = 0; i < sizeW; i++ )
                {
                    wbuf2[i] = random.Get32() & 0xFF;
                }
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->WriteBytes(offsetW, &wbuf2[0], sizeW, optionDestroy));
                pFileSystem->CloseFile(pFile);
            }

            // ファイル2は正常に読み込めることを確認します。
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathFile2, nn::fs::OpenMode_Read));
                NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, &rbuf[0], sizeFile));
                NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, &rbuf[0], sizeFile));
                ASSERT_EQ(0, std::memcmp(&rbuf[0], &wbuf[0], sizeFile));
                pFileSystem->CloseFile(pFile);
            }

            // 破壊した箇所をリード時に検証エラーが発生することを確認します。
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Read));
                NNT_ASSERT_RESULT_FAILURE_RANGE(
                    nn::fs::ResultIntegrityVerificationFailed(),
                    pFile->ReadBytes(offsetW, &rbuf[0], sizeW)
                );
                pFileSystem->CloseFile(pFile);
            }

            // 破壊した箇所を複数回リード時にも検証エラーが発生することを確認します。
            for( int32_t i = 0; i < 10; i++ )
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Read));
                NNT_ASSERT_RESULT_FAILURE_RANGE(
                    nn::fs::ResultIntegrityVerificationFailed(),
                    pFile->ReadBytes(offsetW, &rbuf[0], sizeW)
                );
                pFileSystem->CloseFile(pFile);
            }

            // コミット、アンマウント、マウント後に破壊した箇所を
            // リードしても検証エラーが発生することを確認します。
            {
                NNT_ASSERT_RESULT_SUCCESS(setup.Commit());
                setup.UnmountAndMount();
                pFileSystem = setup.GetFileSystem();
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Read));
                NNT_ASSERT_RESULT_FAILURE_RANGE(
                    nn::fs::ResultIntegrityVerificationFailed(),
                    pFile->ReadBytes(offsetW, &rbuf[0], sizeW)
                );
                pFileSystem->CloseFile(pFile);
            }

            // ファイル2は正常に読み込めることを確認します。
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathFile2, nn::fs::OpenMode_Read));
                NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, &rbuf[0], sizeFile));
                ASSERT_EQ(0, std::memcmp(&rbuf[0], &wbuf[0], sizeFile));
                pFileSystem->CloseFile(pFile);
            }

            // ファイル2の一部を壊して周囲は読み込めることを確認します。
            {
                int32_t sizeW2 = sizeBlock;
                size_t offsetW2 = random.Get32(sizeFile / sizeBlock) * sizeBlock;

                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(
                        &pFile,
                        pathFile2,
                        nn::fs::OpenMode_Read | nn::fs::OpenMode_Write
                    )
                );

                // 破壊オプションなしで書き込み
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->WriteBytes(
                        offsetW2,
                        &wbuf[offsetW2],
                        sizeW2,
                        nn::fs::WriteOption(true)
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(offsetW2, &rbuf[offsetW2], sizeW2));
                ASSERT_EQ(0, std::memcmp(&rbuf[0], &wbuf[0], sizeFile));

                // 破壊オプションありで書き込み
                nn::fs::WriteOption optionDestroy;
                optionDestroy.flush = true;
                optionDestroy.destroySignature = true;
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->WriteBytes(offsetW2, &wbuf[offsetW2], sizeW2, optionDestroy));
                NNT_ASSERT_RESULT_FAILURE_RANGE(
                    nn::fs::ResultIntegrityVerificationFailed(),
                    pFile->ReadBytes(offsetW2, &wbuf2[offsetW2], sizeW2)
                );

                // 破壊していない箇所は読み込めることを確認
                for( int32_t loopCheck = 0; loopCheck < 20; loopCheck++ )
                {
                    size_t offsetW3 = random.Get32(sizeFile / sizeBlock) * sizeBlock;
                    if( offsetW2 != offsetW3 )
                    {
                        NNT_ASSERT_RESULT_SUCCESS(
                            pFile->ReadBytes(offsetW3, &rbuf[offsetW3], sizeW2));
                        ASSERT_EQ(0, std::memcmp(&rbuf[0], &wbuf[0], sizeFile));
                    }
                }

                pFileSystem->CloseFile(pFile);

                // コミットせずにマウントしなおします。
                setup.UnmountAndMount();
                pFileSystem = setup.GetFileSystem();

                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathFile2, nn::fs::OpenMode_Read));

                // 正常にファイルを読み込めることを確認します。
                NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, &rbuf[0], sizeFile));
                ASSERT_EQ(0, std::memcmp(&rbuf[0], &wbuf[0], sizeFile));

                pFileSystem->CloseFile(pFile);
            }

            // テストに使用したファイルを削除します。
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(pathFile));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(pathFile2));
        }

        setup.Finalize();
    }

    NN_LOG("\n");
} // NOLINT(impl/function_size)

TEST_F(FsJournalIntegritySaveDataFileSystemDriverTest, TestDestroySignature)
{
    static const uint32_t LoopCount = 50;

    for( int i = 0; i < 10; ++i )
    {
        TestDestroySignatureInternal(LoopCount);
    }
}
#endif // defined(NN_FS_DESTROY_SIGNATURE)

#if defined(NN_FS_DESTROY_SIGNATURE) && defined(NN_FS_CLEAR_SIGNATURE)
void FsJournalIntegritySaveDataFileSystemDriverTest::TestReadUnformattedFilesInternal(const uint32_t loopCount)
{
    nn::fssystem::save::Random random;

    size_t sizeBlocks[] = { 4096 };

    nn::fssystem::save::IFile* pFile = nullptr;
    nn::fssystem::save::Path pathFile("/file");
    nn::fssystem::save::Path pathFile2("/file2");

    const auto loopBlockMax = loopBlock < sizeof(sizeBlocks) / sizeof(sizeBlocks[0]);
    for( size_t loopBlock = 0; loopBlockMax; loopBlock++ )
    {
        size_t sizeBlock = sizeBlocks[loopBlock];
        //sizeBlock = 256 << m_Random.Get32(8);

        JournalIntegritySaveDataFileSystemDriverTestSetup setup;
        setup.SetRandomizeBlockSize(false);
        setup.Initialize(0, sizeBlock);

        nn::fssystem::save::IFileSystem* pFileSystem = setup.GetFileSystem();

        for( uint32_t loop = 0; loop < loopCount; ++loop )
        {
            NN_LOG(".");

            // ある程度の大きさの適当なサイズ
            int64_t sizeFile = sizeBlock * (8 + random.Get32(8)) + random.Get32(sizeBlock);
            nnt::fs::util::Vector<char> rbuf(sizeFile);
            nnt::fs::util::Vector<char> wbuf(sizeFile);
            nnt::fs::util::Vector<char> wbuf2(sizeFile);

            // ファイルを作成します。
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(pathFile, sizeFile));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(pathFile2, sizeFile));

            // フォーマットせずに全領域を読み込んで 0 が読み込めることを確認します。
            const int PathCount = 2;
            const nn::fssystem::save::Path *pathFiles[PathCount] =
            {
                &pathFile, &pathFile2
            };

            for( int pathIndex = 0; pathIndex < PathCount; ++pathIndex )
            {
                for( int64_t i = 0; i < sizeFile; ++i )
                {
                    rbuf[i] = random.Get32() & 0xFF;
                }

                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, *pathFiles[pathIndex], nn::fs::OpenMode_Read));
                NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, rbuf.data(), sizeFile));
                pFileSystem->CloseFile(pFile);

                for( int64_t i = 0; i < sizeFile; ++i )
                {
                    ASSERT_EQ(0, rbuf[i]);
                }
            }

            // ファイル全体をランダムなデータで埋めます。
            {
                for( int64_t i = 0; i < sizeFile; i++ )
                {
                    wbuf[i] = random.Get32() & 0xFF;
                }

                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Write));
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->WriteBytes(0, &wbuf[0], sizeFile, nn::fs::WriteOption()));
                pFileSystem->CloseFile(pFile);

                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathFile2, nn::fs::OpenMode_Write));
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->WriteBytes(0, &wbuf[0], sizeFile, nn::fs::WriteOption()));
                pFileSystem->CloseFile(pFile);
            }

            // この時点では正常にデータが読み込めることを確認します。
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Read));
                NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, &rbuf[0], sizeFile));
                ASSERT_EQ(0, std::memcmp(&rbuf[0], &wbuf[0], sizeFile));
                pFileSystem->CloseFile(pFile);

                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathFile2, nn::fs::OpenMode_Read));
                NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, &rbuf[0], sizeFile));
                ASSERT_EQ(0, std::memcmp(&rbuf[0], &wbuf[0], sizeFile));
                pFileSystem->CloseFile(pFile);
            }

            // リサイズして拡張した領域を読み込んで 0 が読み込めることを確認します。
            for( int pathIndex = 0; pathIndex < PathCount; ++pathIndex )
            {
                int64_t additionalSize = random.Get32() % (sizeBlock * 4);
                int64_t updatedSize = 0;
                nnt::fs::util::Vector<char> rbuf2(additionalSize);

                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(
                        &pFile,
                        *pathFiles[pathIndex],
                        nn::fs::OpenMode_Read | nn::fs::OpenMode_Write
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(sizeFile + additionalSize));
                NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&updatedSize));
                ASSERT_EQ(sizeFile + additionalSize, updatedSize);
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->ReadBytes(sizeFile, rbuf2.data(), additionalSize));
                pFileSystem->CloseFile(pFile);

                for( int64_t i = 0; i < additionalSize; i++ )
                {
                    ASSERT_EQ(0, rbuf2[i]);
                }
            }

            // ファイルを一度削除して作り直し、フォーマットせずに全領域を読み込んで 0 が読み込めることを確認します。
            for( int pathIndex = 0; pathIndex < PathCount; ++pathIndex )
            {
                NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(*pathFiles[pathIndex]));
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->CreateFile(*pathFiles[pathIndex], sizeFile));
            }

            for( int pathIndex = 0; pathIndex < PathCount; ++pathIndex )
            {
                for( int64_t i = 0; i < sizeFile; i++ )
                {
                    rbuf[i] = random.Get32() & 0xFF;
                }

                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, *pathFiles[pathIndex], nn::fs::OpenMode_Read));
                NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, rbuf.data(), sizeFile));
                pFileSystem->CloseFile(pFile);

                for( int64_t i = 0; i < sizeFile; ++i )
                {
                    ASSERT_EQ(0, rbuf[i]);
                }
            }

            // ファイル全体をランダムなデータで埋め直します。
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Write));
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->WriteBytes(0, &wbuf[0], sizeFile, nn::fs::WriteOption()));
                pFileSystem->CloseFile(pFile);

                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathFile2, nn::fs::OpenMode_Write));
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->WriteBytes(0, &wbuf[0], sizeFile, nn::fs::WriteOption()));
                pFileSystem->CloseFile(pFile);
            }

            // ファイルの一部を破壊オプションありで書き込みます。
            size_t offsetW = random.Get32(sizeFile);
            size_t sizeW = random.Get32(sizeFile - offsetW) + 1;
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Write));
                nn::fs::WriteOption optionDestroy;
                optionDestroy.flush = true;
                optionDestroy.destroySignature = true;
                for( int64_t i = 0; i < sizeW; i++ )
                {
                    wbuf2[i] = random.Get32() & 0xFF;
                }
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->WriteBytes(offsetW, &wbuf2[0], sizeW, optionDestroy));
                pFileSystem->CloseFile(pFile);
            }

            // ファイル2は正常に読み込めることを確認します。
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathFile2, nn::fs::OpenMode_Read));
                NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, &rbuf[0], sizeFile));
                NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, &rbuf[0], sizeFile));
                ASSERT_EQ(0, std::memcmp(&rbuf[0], &wbuf[0], sizeFile));
                pFileSystem->CloseFile(pFile);
            }

            // 破壊した箇所をリード時に検証エラーが発生することを確認します。
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Read));
                NNT_ASSERT_RESULT_FAILURE_RANGE(
                    nn::fs::ResultIntegrityVerificationFailed(),
                    pFile->ReadBytes(offsetW, &rbuf[0], sizeW)
                );
                pFileSystem->CloseFile(pFile);
            }

            // 破壊した箇所を複数回リード時にも検証エラーが発生することを確認します。
            for( int32_t i = 0; i < 10; i++ )
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Read));
                NNT_ASSERT_RESULT_FAILURE_RANGE(
                    nn::fs::ResultIntegrityVerificationFailed(),
                    pFile->ReadBytes(offsetW, &rbuf[0], sizeW)
                );
                pFileSystem->CloseFile(pFile);
            }

            // コミット、アンマウント、マウント後に破壊した箇所を
            // リードしても検証エラーが発生することを確認します。
            {
                NNT_ASSERT_RESULT_SUCCESS(setup.Commit());
                setup.UnmountAndMount();
                pFileSystem = setup.GetFileSystem();
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Read));
                NNT_ASSERT_RESULT_FAILURE_RANGE(
                    nn::fs::ResultIntegrityVerificationFailed(),
                    pFile->ReadBytes(offsetW, &rbuf[0], sizeW)
                );
                pFileSystem->CloseFile(pFile);
            }

            // ファイル2は正常に読み込めることを確認します。
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathFile2, nn::fs::OpenMode_Read));
                NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, &rbuf[0], sizeFile));
                ASSERT_EQ(0, std::memcmp(&rbuf[0], &wbuf[0], sizeFile));
                pFileSystem->CloseFile(pFile);
            }

            // ファイル2の一部を壊して周囲は読み込めることを確認します。
            {
                int32_t sizeW2 = sizeBlock;
                size_t offsetW2 = random.Get32(sizeFile / sizeBlock) * sizeBlock;

                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(
                        &pFile,
                        pathFile2,
                        nn::fs::OpenMode_Read | nn::fs::OpenMode_Write
                    )
                );

                // 破壊オプションなしで書き込み
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->WriteBytes(
                        offsetW2,
                        &wbuf[offsetW2],
                        sizeW2,
                        nn::fs::WriteOption(true)
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(offsetW2, &rbuf[offsetW2], sizeW2));
                ASSERT_EQ(0, std::memcmp(&rbuf[0], &wbuf[0], sizeFile));

                // 破壊オプションありで書き込み
                nn::fs::WriteOption optionDestroy;
                optionDestroy.flush = true;
                optionDestroy.destroySignature = true;
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->WriteBytes(offsetW2, &wbuf[offsetW2], sizeW2, optionDestroy));
                NNT_ASSERT_RESULT_FAILURE_RANGE(
                    nn::fs::ResultIntegrityVerificationFailed(),
                    pFile->ReadBytes(offsetW2, &wbuf2[offsetW2], sizeW2)
                );

                // 破壊していない箇所は読み込めることを確認
                for( int32_t loopCheck = 0; loopCheck < 20; loopCheck++ )
                {
                    size_t offsetW3 = random.Get32(sizeFile / sizeBlock) * sizeBlock;
                    if( offsetW2 != offsetW3 )
                    {
                        NNT_ASSERT_RESULT_SUCCESS(
                            pFile->ReadBytes(offsetW3, &rbuf[offsetW3], sizeW2));
                        ASSERT_EQ(0, std::memcmp(&rbuf[0], &wbuf[0], sizeFile));
                    }
                }

                pFileSystem->CloseFile(pFile);

                // コミットせずにマウントしなおします。
                setup.UnmountAndMount();
                pFileSystem = setup.GetFileSystem();

                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathFile2, nn::fs::OpenMode_Read));

                // 正常にファイルを読み込めることを確認します。
                NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, &rbuf[0], sizeFile));
                ASSERT_EQ(0, std::memcmp(&rbuf[0], &wbuf[0], sizeFile));

                pFileSystem->CloseFile(pFile);
            }

            // テストに使用したファイルを削除します。
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(pathFile));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(pathFile2));
        }

        setup.Finalize();
    }

    NN_LOG("\n");
} // NOLINT(impl/function_size)

TEST_F(FsJournalIntegritySaveDataFileSystemDriverTest, TestReadUnformattedFiles)
{
    static const uint32_t LoopCount = 50;

    for( int i = 0; i < 10; ++i )
    {
        TestReadUnformattedFilesInternal(LoopCount);
    }
}
#endif // defined(NN_FS_DESTROY_SIGNATURE) && defined(NN_FS_CLEAR_SIGNATURE)

//! コミットの挙動をテストします。
TEST_F(FsJournalIntegritySaveDataFileSystemDriverTest, Commit)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    static const auto LoopCount = 3;
    static const auto CountExpandMax = 25;

    for( size_t sizeBlock = 4096; sizeBlock <= MaxBlockSize; sizeBlock *= 2 )
    {
        NN_SDK_LOG(".");

        for( auto loop = 0; loop < LoopCount; ++loop )
        {
            nnt::fs::util::AccessCountedMemoryStorage storageData;
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver fileSystem;

            // ブロックサイズ
            nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam paramDuplex;
            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
                paramIntegrity;
            const size_t sizeBlock1 = GenerateRandomBlockSize(512, MaxBlockSize);
            // sizeBlock1 == sizeBlock2 == 512 だと検証層のサイズが不足することがある
            const size_t minSizeBlock2 = (sizeBlock1 == 512) ? 1024 : 512;
            const size_t sizeBlock2 = GenerateRandomBlockSize(minSizeBlock2, MaxBlockSize);
            paramDuplex.sizeBlockLevel[0] = sizeBlock1 / 4;
            paramDuplex.sizeBlockLevel[1] = sizeBlock2;
            paramIntegrity.sizeBlockLevel[0] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[1] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[2] = sizeBlock2;
            paramIntegrity.sizeBlockLevel[3] = sizeBlock2;

            static const auto HashSize = nn::crypto::Sha256Generator::HashSize;
            const auto countMax = std::min<int64_t>(
                                      static_cast<uint64_t>(paramIntegrity.sizeBlockLevel[0])
                                      / HashSize * paramIntegrity.sizeBlockLevel[1]
                                      / HashSize * paramIntegrity.sizeBlockLevel[2]
                                      / HashSize * paramIntegrity.sizeBlockLevel[3]
                                      / sizeBlock,
                                      120
                                  );
            const auto countDataBlock = std::uniform_int_distribution<int64_t>(
                                            countMax * 5 / 6,
                                            countMax
                                        )(mt);
            const auto countReservedBlock = countDataBlock;

            int64_t sizeTotal = 0;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
                    &sizeTotal,
                    sizeBlock,
                    CountExpandMax,
                    paramDuplex,
                    paramIntegrity,
                    static_cast<uint32_t>(countDataBlock),
                    static_cast<uint32_t>(countReservedBlock)
                )
            );

            // ファイルシステムを配置する Storage を構築する
            storageData.Initialize(sizeTotal);

            nn::fs::SubStorage storageBase(&storageData, 0, sizeTotal);

            nn::fs::SaveDataHashSalt salt;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::Format(
                    nn::fs::SubStorage(&storageData, 0, sizeTotal),
                    sizeBlock,
                    CountExpandMax,
                    paramDuplex,
                    paramIntegrity,
                    static_cast<uint32_t>(countDataBlock),
                    static_cast<uint32_t>(countReservedBlock),
                    GetBufferManager(),
                    nnt::fs::util::GetMacGenerator(),
                    salt
                )
            );

            NNT_ASSERT_RESULT_SUCCESS(
                fileSystem.Initialize(
                    storageBase,
                    GetBufferManager(),
                    nnt::fs::util::GetMacGenerator(),
                    nnt::fs::util::SaveDataMinimumVersion
                )
            );

            // 準備します。
            nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData firstExtraData;
            nnt::fs::util::FillBufferWithRandomValue(&firstExtraData, sizeof(firstExtraData));
            nn::fssystem::save::Path pathDirectory;
            nn::fssystem::save::Path pathFile;
            const auto sizeFile
                = std::uniform_int_distribution<size_t>(sizeBlock / 2, sizeBlock * 2)(mt);
            nnt::fs::util::Vector<char> bufWrite(sizeFile, 0);

            {
                // 拡張データを書き込みます。
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.WriteExtraData(firstExtraData));

                // ファイルを作成し書き込みます。
                nn::fssystem::save::IFile* pFile = nullptr;
                NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize("/directory"));
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory(pathDirectory));
                NNT_ASSERT_RESULT_SUCCESS(pathFile.Initialize("/directory/file.dat"));
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile(pathFile, sizeFile));
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.OpenFile(
                        &pFile,
                        pathFile,
                        nn::fs::OpenMode_Write
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(0, bufWrite.data(), sizeFile));
                fileSystem.CloseFile(pFile);
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.Commit());
                fileSystem.Finalize();
                std::iota(bufWrite.begin(), bufWrite.end(), '\x1');
            }

            // コミットせずに再マウントすると以前のデータが読み込めます。
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.Initialize(
                        storageBase,
                        GetBufferManager(),
                        nnt::fs::util::GetMacGenerator(),
                        nnt::fs::util::SaveDataMinimumVersion
                    )
                );

                // 拡張データを書き込みます。
                nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData notCommitExtraData;
                nnt::fs::util::FillBufferWithRandomValue(
                    &notCommitExtraData, sizeof(notCommitExtraData)
                );
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.WriteExtraData(notCommitExtraData));

                // ファイルを作成し書き込みます。
                nn::fssystem::save::IFile* pFile = nullptr;
                const auto offset = std::uniform_int_distribution<int64_t>(0, sizeFile - 1)(mt);
                const auto sizeMax = static_cast<size_t>(sizeFile - offset);
                const auto size = std::uniform_int_distribution<size_t>(1, sizeMax)(mt);
                nnt::fs::util::Vector<char> bufPrevious(size, 0);
                nnt::fs::util::Vector<char> bufRead(size, 0);

                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.OpenFile(
                        &pFile,
                        pathFile,
                        nn::fs::OpenMode_Read | nn::fs::OpenMode_Write
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(offset, bufPrevious.data(), size));
                NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(offset, bufWrite.data(), size));
                fileSystem.CloseFile(pFile);

                // コミットせずに再マウントします
                fileSystem.Finalize();
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.Initialize(
                        storageBase,
                        GetBufferManager(),
                        nnt::fs::util::GetMacGenerator(),
                        nnt::fs::util::SaveDataMinimumVersion
                    )
                );

                // 以前のデータが読み込めることを確認します
                nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData extraData;
                fileSystem.ReadExtraData(&extraData);
                ASSERT_EQ(0, memcmp(&firstExtraData, &extraData, sizeof(extraData)));

                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.OpenFile(
                        &pFile,
                        pathFile,
                        nn::fs::OpenMode_Read
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(offset, bufRead.data(), size));
                fileSystem.CloseFile(pFile);
                EXPECT_TRUE(std::equal(bufRead.begin(), bufRead.end(), bufPrevious.begin()));
                fileSystem.Finalize();
            }

            // コミットして再マウントすると書き込んだデータが読み込めます。
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.Initialize(
                        storageBase,
                        GetBufferManager(),
                        nnt::fs::util::GetMacGenerator(),
                        nnt::fs::util::SaveDataMinimumVersion
                    )
                );

                // 拡張データを書き込みます。
                nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData commitExtraData;
                nnt::fs::util::FillBufferWithRandomValue(
                    &commitExtraData, sizeof(commitExtraData)
                );
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.WriteExtraData(commitExtraData));

                // ファイルに書き込みます。
                nn::fssystem::save::IFile* pFile = nullptr;
                const auto offset = std::uniform_int_distribution<int64_t>(0, sizeFile - 1)(mt);
                const auto sizeMax = static_cast<size_t>(sizeFile - offset);
                const auto size = std::uniform_int_distribution<size_t>(1, sizeMax)(mt);
                nnt::fs::util::Vector<char> bufRead(size, 0);

                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.OpenFile(
                        &pFile,
                        pathFile,
                        nn::fs::OpenMode_Write
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(offset, bufWrite.data(), size));
                fileSystem.CloseFile(pFile);

                // コミットします
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.Commit());

                // 再マウントします
                fileSystem.Finalize();
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.Initialize(
                        storageBase,
                        GetBufferManager(),
                        nnt::fs::util::GetMacGenerator(),
                        nnt::fs::util::SaveDataMinimumVersion
                    )
                );

                // 書き込んだデータが読み込めることを確認します
                nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData readExtraData;
                fileSystem.ReadExtraData(&readExtraData);
                ASSERT_EQ(0, memcmp(&commitExtraData, &readExtraData, sizeof(readExtraData)));

                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.OpenFile(
                        &pFile,
                        pathFile, nn::fs::OpenMode_Read));
                NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(offset, bufRead.data(), size));
                fileSystem.CloseFile(pFile);
                EXPECT_TRUE(std::equal(bufRead.begin(), bufRead.end(), bufWrite.begin()));
                fileSystem.Finalize();
            }

            // 書き込みモードでのファイルオープン中はコミットできません。
            {
                nn::fssystem::save::IFile* pFile = nullptr;
                nn::fssystem::save::IDirectory* pDirectory = nullptr;
                const auto offset = std::uniform_int_distribution<int64_t>(0, sizeFile - 1)(mt);
                const auto sizeMax = static_cast<size_t>(sizeFile - offset);
                const auto size = std::uniform_int_distribution<size_t>(1, sizeMax)(mt);

                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.Initialize(
                        storageBase,
                        GetBufferManager(),
                        nnt::fs::util::GetMacGenerator(),
                        nnt::fs::util::SaveDataMinimumVersion
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.OpenFile(
                        &pFile,
                        pathFile,
                        nn::fs::OpenMode_Write
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(offset, bufWrite.data(), size));
                NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultUnsupportedOperation, fileSystem.Commit());
                fileSystem.CloseFile(pFile);
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.OpenFile(
                        &pFile,
                        pathFile,
                        nn::fs::OpenMode_Read
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.OpenDirectory(
                        &pDirectory,
                        pathDirectory
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.Commit());
                fileSystem.CloseFile(pFile);
                fileSystem.CloseDirectory(pDirectory);
                fileSystem.Finalize();
            }
        }
    }

    NN_SDK_LOG("\n");
} // NOLINT(impl/function_size)

//! ロールバックの挙動をテストします。
TEST_F(FsJournalIntegritySaveDataFileSystemDriverTest, RollBack)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    static const auto LoopCount = 3;
    static const auto CountExpandMax = 25;

    for( size_t sizeBlock = 4096; sizeBlock <= MaxBlockSize; sizeBlock *= 2 )
    {
        NN_SDK_LOG(".");

        for( auto loop = 0; loop < LoopCount; ++loop )
        {
            nnt::fs::util::AccessCountedMemoryStorage storageData;
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver fileSystem;

            // ブロックサイズ
            nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam paramDuplex;
            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
                paramIntegrity;
            const size_t sizeBlock1 = GenerateRandomBlockSize(512, MaxBlockSize);
            // sizeBlock1 == sizeBlock2 == 512 だと検証層のサイズが不足することがある
            const size_t minSizeBlock2 = (sizeBlock1 == 512) ? 1024 : 512;
            const size_t sizeBlock2 = GenerateRandomBlockSize(minSizeBlock2, MaxBlockSize);
            paramDuplex.sizeBlockLevel[0] = sizeBlock1 / 4;
            paramDuplex.sizeBlockLevel[1] = sizeBlock2;
            paramIntegrity.sizeBlockLevel[0] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[1] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[2] = sizeBlock2;
            paramIntegrity.sizeBlockLevel[3] = sizeBlock2;

            static const auto HashSize = nn::crypto::Sha256Generator::HashSize;
            const auto countMax = std::min<int64_t>(
                                      static_cast<uint64_t>(paramIntegrity.sizeBlockLevel[0])
                                      / HashSize * paramIntegrity.sizeBlockLevel[1]
                                      / HashSize * paramIntegrity.sizeBlockLevel[2]
                                      / HashSize * paramIntegrity.sizeBlockLevel[3]
                                      / sizeBlock,
                                      120
                                  );
            const auto countDataBlock = std::uniform_int_distribution<int64_t>(
                                            countMax * 5 / 6,
                                            countMax
                                        )(mt);
            const auto countReservedBlock = countDataBlock;

            int64_t sizeTotal = 0;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
                    &sizeTotal,
                    sizeBlock,
                    CountExpandMax,
                    paramDuplex,
                    paramIntegrity,
                    static_cast<uint32_t>(countDataBlock),
                    static_cast<uint32_t>(countReservedBlock)
                )
            );

            // ファイルシステムを配置する Storage を構築する
            storageData.Initialize(sizeTotal);

            nn::fs::SubStorage storageBase(&storageData, 0, sizeTotal);

            nn::fs::SaveDataHashSalt salt;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::Format(
                    nn::fs::SubStorage(&storageData, 0, sizeTotal),
                    sizeBlock,
                    CountExpandMax,
                    paramDuplex,
                    paramIntegrity,
                    static_cast<uint32_t>(countDataBlock),
                    static_cast<uint32_t>(countReservedBlock),
                    GetBufferManager(),
                    nnt::fs::util::GetMacGenerator(),
                    salt
                )
            );

            NNT_ASSERT_RESULT_SUCCESS(
                fileSystem.Initialize(
                    storageBase,
                    GetBufferManager(),
                    nnt::fs::util::GetMacGenerator(),
                    nnt::fs::util::SaveDataMinimumVersion
                )
            );

            // 準備します。
            nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData extraData;
            nnt::fs::util::FillBufferWithRandomValue(&extraData, sizeof(extraData));
            nn::fssystem::save::Path pathDirectory;
            nn::fssystem::save::Path pathFile;
            const auto sizeFile
                = std::uniform_int_distribution<size_t>(sizeBlock / 2, sizeBlock * 2)(mt);
            nnt::fs::util::Vector<char> bufWrite(sizeFile, 0);

            {
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.WriteExtraData(extraData));
                nn::fssystem::save::IFile* pFile = nullptr;
                NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize("/directory"));
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory(pathDirectory));
                NNT_ASSERT_RESULT_SUCCESS(pathFile.Initialize("/directory/file.dat"));
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile(pathFile, sizeFile));
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.OpenFile(
                        &pFile,
                        pathFile,
                        nn::fs::OpenMode_Write
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(0, bufWrite.data(), sizeFile));
                fileSystem.CloseFile(pFile);
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.Commit());
                fileSystem.Finalize();
                std::iota(bufWrite.begin(), bufWrite.end(), '\x1');
            }

            // ロールバックすると以前のデータが読み込めます。
            {
                nn::fssystem::save::IFile* pFile = nullptr;
                const auto offset = std::uniform_int_distribution<int64_t>(0, sizeFile - 1)(mt);
                const auto sizeMax = static_cast<size_t>(sizeFile - offset);
                const auto size = std::uniform_int_distribution<size_t>(1, sizeMax)(mt);
                nnt::fs::util::Vector<char> bufPrevious(size, 0);
                nnt::fs::util::Vector<char> bufRead(size, 0);

                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.Initialize(
                        storageBase,
                        GetBufferManager(),
                        nnt::fs::util::GetMacGenerator(),
                        nnt::fs::util::SaveDataMinimumVersion
                    )
                );

                nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData writeExtraData;
                nnt::fs::util::FillBufferWithRandomValue(&writeExtraData, sizeof(writeExtraData));
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.WriteExtraData(writeExtraData));

                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.OpenFile(
                        &pFile,
                        pathFile,
                        nn::fs::OpenMode_Read | nn::fs::OpenMode_Write
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(offset, bufPrevious.data(), size));
                NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(offset, bufWrite.data(), size));
                NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(offset, bufRead.data(), size));
                fileSystem.CloseFile(pFile);
                EXPECT_TRUE(std::equal(bufRead.begin(), bufRead.end(), bufWrite.begin()));

                // ロールバックします
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.Rollback());

                // 拡張データがロールバックしていることを確かめます
                nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData readExtraData;
                NNT_ASSERT_RESULT_SUCCESS(
                    nn::fssystem::save::JournalIntegritySaveDataFileSystem::ReadExtraData(
                        &readExtraData,
                        storageBase,
                        GetBufferManager(),
                        nnt::fs::util::GetMacGenerator(),
                        nnt::fs::util::SaveDataMinimumVersion
                    )
                );
                ASSERT_EQ(0, memcmp(&readExtraData, &extraData, sizeof(readExtraData)));

                // ファイルの中身がロールバックしていることを確かめます
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.OpenFile(
                        &pFile,
                        pathFile,
                        nn::fs::OpenMode_Read
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(offset, bufRead.data(), size));
                fileSystem.CloseFile(pFile);
                EXPECT_TRUE(std::equal(bufRead.begin(), bufRead.end(), bufPrevious.begin()));
                fileSystem.Finalize();
            }

            // エントリオープン中はロールバックできません。
            {
                nn::fssystem::save::IFile* pFile = nullptr;
                nn::fssystem::save::IDirectory* pDirectory = nullptr;
                const auto offset = std::uniform_int_distribution<int64_t>(0, sizeFile - 1)(mt);
                const auto sizeMax = static_cast<size_t>(sizeFile - offset);
                const auto size = std::uniform_int_distribution<size_t>(1, sizeMax)(mt);

                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.Initialize(
                        storageBase,
                        GetBufferManager(),
                        nnt::fs::util::GetMacGenerator(),
                        nnt::fs::util::SaveDataMinimumVersion
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.OpenFile(
                        &pFile,
                        pathFile,
                        nn::fs::OpenMode_Write
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(offset, bufWrite.data(), size));
                NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultUnsupportedOperation, fileSystem.Rollback());
                fileSystem.CloseFile(pFile);
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.OpenFile(
                        &pFile,
                        pathFile,
                        nn::fs::OpenMode_Read
                    )
                );
                NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultUnsupportedOperation, fileSystem.Rollback());
                fileSystem.CloseFile(pFile);
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.OpenDirectory(
                        &pDirectory,
                        pathDirectory
                    )
                );
                NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultUnsupportedOperation, fileSystem.Rollback());
                fileSystem.CloseDirectory(pDirectory);
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.Rollback());
                fileSystem.Finalize();
            }
        }
    }

    NN_SDK_LOG("\n");
} // NOLINT(impl/function_size)

//! 仮コミットの挙動をテストします。
TEST_F(FsJournalIntegritySaveDataFileSystemDriverTest, CommitProvisionally)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    static const auto LoopCount = 5;
    static const auto CountExpandMax = 25;

    for( size_t sizeBlock = 4096; sizeBlock <= MaxBlockSize; sizeBlock *= 2 )
    {
        NN_SDK_LOG(".");

        for( auto loop = 0; loop < LoopCount; ++loop )
        {
            nnt::fs::util::AccessCountedMemoryStorage storageData;
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver fileSystem;

            // ブロックサイズ
            nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam paramDuplex;
            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
                paramIntegrity;
            const size_t sizeBlock1 = GenerateRandomBlockSize(512, MaxBlockSize);
            // sizeBlock1 == sizeBlock2 == 512 だと検証層のサイズが不足することがある
            const size_t minSizeBlock2 = (sizeBlock1 == 512) ? 1024 : 512;
            const size_t sizeBlock2 = GenerateRandomBlockSize(minSizeBlock2, MaxBlockSize);
            paramDuplex.sizeBlockLevel[0] = sizeBlock1 / 4;
            paramDuplex.sizeBlockLevel[1] = sizeBlock2;
            paramIntegrity.sizeBlockLevel[0] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[1] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[2] = sizeBlock2;
            paramIntegrity.sizeBlockLevel[3] = sizeBlock2;

            static const auto HashSize = nn::crypto::Sha256Generator::HashSize;
            const auto countMax = std::min<int64_t>(
                                      static_cast<uint64_t>(paramIntegrity.sizeBlockLevel[0])
                                      / HashSize * paramIntegrity.sizeBlockLevel[1]
                                      / HashSize * paramIntegrity.sizeBlockLevel[2]
                                      / HashSize * paramIntegrity.sizeBlockLevel[3]
                                      / sizeBlock,
                                      120
                                  );
            const auto countDataBlock = std::uniform_int_distribution<int64_t>(
                                            countMax * 5 / 6,
                                            countMax
                                        )(mt);
            const auto countReservedBlock = countDataBlock;

            int64_t sizeTotal = 0;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
                    &sizeTotal,
                    sizeBlock,
                    CountExpandMax,
                    paramDuplex,
                    paramIntegrity,
                    static_cast<uint32_t>(countDataBlock),
                    static_cast<uint32_t>(countReservedBlock)
                )
            );

            // ファイルシステムを配置する Storage を構築する
            storageData.Initialize(sizeTotal);

            nn::fs::SubStorage storageBase(&storageData, 0, sizeTotal);

            nn::fs::SaveDataHashSalt salt;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::Format(
                    nn::fs::SubStorage(&storageData, 0, sizeTotal),
                    sizeBlock,
                    CountExpandMax,
                    paramDuplex,
                    paramIntegrity,
                    static_cast<uint32_t>(countDataBlock),
                    static_cast<uint32_t>(countReservedBlock),
                    GetBufferManager(),
                    nnt::fs::util::GetMacGenerator(),
                    salt
                )
            );

            NNT_ASSERT_RESULT_SUCCESS(
                fileSystem.Initialize(
                    storageBase,
                    GetBufferManager(),
                    nnt::fs::util::GetMacGenerator(),
                    nnt::fs::util::SaveDataMinimumVersion
                )
            );

            // 準備します。
            nn::fssystem::save::Path pathDirectory;
            nn::fssystem::save::Path pathFile;
            const auto sizeFile
                = std::uniform_int_distribution<size_t>(sizeBlock / 2, sizeBlock * 2)(mt);
            nnt::fs::util::Vector<char> bufWrite(sizeFile, 0);

            nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData firstExtraData;
            nnt::fs::util::FillBufferWithRandomValue(&firstExtraData, sizeof(firstExtraData));

            // データを書き込んでおきます。
            {
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.WriteExtraData(firstExtraData));

                nn::fssystem::save::IFile* pFile = nullptr;
                NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize("/directory"));
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory(pathDirectory));
                NNT_ASSERT_RESULT_SUCCESS(pathFile.Initialize("/directory/file.dat"));
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile(pathFile, sizeFile));
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.OpenFile(
                        &pFile,
                        pathFile,
                        nn::fs::OpenMode_Write
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(0, bufWrite.data(), sizeFile));
                fileSystem.CloseFile(pFile);
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.Commit());
                fileSystem.Finalize();
                std::iota(bufWrite.begin(), bufWrite.end(), '\x1');
            }

            nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData secondExtraData;
            nnt::fs::util::FillBufferWithRandomValue(&secondExtraData, sizeof(secondExtraData));
            // 仮コミット後、コミットして再マウントすると書き込んだデータが読み込めます。
            {
                nn::fssystem::save::IFile* pFile = nullptr;
                const auto offset = std::uniform_int_distribution<int64_t>(0, sizeFile - 1)(mt);
                const auto sizeMax = static_cast<size_t>(sizeFile - offset);
                const auto size = std::uniform_int_distribution<size_t>(1, sizeMax)(mt);
                nnt::fs::util::Vector<char> bufRead(size, 0);

                // 仮コミットします。
                {
                    NNT_ASSERT_RESULT_SUCCESS(
                        fileSystem.Initialize(
                            storageBase,
                            GetBufferManager(),
                            nnt::fs::util::GetMacGenerator(),
                            nnt::fs::util::SaveDataMinimumVersion
                        )
                    );

                    // 拡張データを書き込みます
                    NNT_ASSERT_RESULT_SUCCESS(
                        fileSystem.WriteExtraData(
                            secondExtraData
                        )
                    );

                    // ファイルに書き込みます
                    NNT_ASSERT_RESULT_SUCCESS(
                        fileSystem.OpenFile(
                            &pFile,
                            pathFile,
                            nn::fs::OpenMode_Write
                        )
                    );
                    NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(offset, bufWrite.data(), size));
                    fileSystem.CloseFile(pFile);
                    EXPECT_EQ(0, fileSystem.GetCounterForBundledCommit());
                    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CommitProvisionally(1));
                    EXPECT_EQ(1, fileSystem.GetCounterForBundledCommit());
                }

                // コミット前に静的関数で拡張データを読み込むと、最後のコミット時のデータが取得できます。
                {
                    nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData readExtraData;
                    NNT_ASSERT_RESULT_SUCCESS(
                        nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::ReadExtraData(
                            &readExtraData,
                            storageBase,
                            GetBufferManager(),
                            nnt::fs::util::GetMacGenerator(),
                            nnt::fs::util::SaveDataMinimumVersion
                        )
                    );
                    ASSERT_EQ(0, memcmp(&firstExtraData, &readExtraData, sizeof(readExtraData)));
                }

                // コミット前に動的関数で拡張データを読み込むと、仮コミット時のデータが取得できます。
                {
                    nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData readExtraData;
                    fileSystem.ReadExtraData(
                        &readExtraData
                    );
                    ASSERT_EQ(0, memcmp(&secondExtraData, &readExtraData, sizeof(readExtraData)));
                }

                // コミットします。
                {
                    NNT_ASSERT_RESULT_SUCCESS(fileSystem.Commit());
                    EXPECT_EQ(0, fileSystem.GetCounterForBundledCommit());
                    fileSystem.Finalize();
                }

                // 再マウントしても書き込んだデータが読み取れていることを確認します。
                {
                    NNT_ASSERT_RESULT_SUCCESS(
                        fileSystem.Initialize(
                            storageBase,
                            GetBufferManager(),
                            nnt::fs::util::GetMacGenerator(),
                            nnt::fs::util::SaveDataMinimumVersion
                        )
                    );
                    NNT_ASSERT_RESULT_SUCCESS(
                        fileSystem.OpenFile(
                            &pFile,
                            pathFile, nn::fs::OpenMode_Read
                        )
                    );
                    NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(offset, bufRead.data(), size));
                    fileSystem.CloseFile(pFile);
                    EXPECT_TRUE(std::equal(bufRead.begin(), bufRead.end(), bufWrite.begin()));
                    fileSystem.Finalize();

                    nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData readExtraData;
                    NNT_ASSERT_RESULT_SUCCESS(
                        nn::fssystem::save::JournalIntegritySaveDataFileSystem::ReadExtraData(
                            &readExtraData,
                            storageBase,
                            GetBufferManager(),
                            nnt::fs::util::GetMacGenerator(),
                            nnt::fs::util::SaveDataMinimumVersion
                        )
                    );
                    ASSERT_EQ(0, memcmp(&secondExtraData, &readExtraData, sizeof(readExtraData)));
                }
            }

            // 仮コミット後、ロールバックすると以前のデータが読み込めます。
            {
                nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData rollbackExtraData;
                nnt::fs::util::FillBufferWithRandomValue(&rollbackExtraData, sizeof(rollbackExtraData));

                nn::fssystem::save::IFile* pFile = nullptr;
                const auto offset = std::uniform_int_distribution<int64_t>(0, sizeFile - 1)(mt);
                const auto sizeMax = static_cast<size_t>(sizeFile - offset);
                const auto size = std::uniform_int_distribution<size_t>(1, sizeMax)(mt);
                nnt::fs::util::Vector<char> bufPrevious(size, 0);
                nnt::fs::util::Vector<char> bufRead(size, 0);

                // 仮コミットします。
                {
                    NNT_ASSERT_RESULT_SUCCESS(
                        fileSystem.Initialize(
                            storageBase,
                            GetBufferManager(),
                            nnt::fs::util::GetMacGenerator(),
                            nnt::fs::util::SaveDataMinimumVersion
                        )
                    );

                    NNT_ASSERT_RESULT_SUCCESS(
                        fileSystem.WriteExtraData(
                            rollbackExtraData
                        )
                    );

                    NNT_ASSERT_RESULT_SUCCESS(
                        fileSystem.OpenFile(
                            &pFile,
                            pathFile,
                            nn::fs::OpenMode_Read | nn::fs::OpenMode_Write
                        )
                    );
                    NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(offset, bufPrevious.data(), size));
                    NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(offset, bufWrite.data(), size));
                    NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(offset, bufRead.data(), size));
                    fileSystem.CloseFile(pFile);
                    EXPECT_TRUE(std::equal(bufRead.begin(), bufRead.end(), bufWrite.begin()));
                    EXPECT_EQ(0, fileSystem.GetCounterForBundledCommit());
                    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CommitProvisionally(2));
                    EXPECT_EQ(2, fileSystem.GetCounterForBundledCommit());
                }

                // ロールバックします。
                {
                    NNT_ASSERT_RESULT_SUCCESS(fileSystem.Rollback());
                    EXPECT_EQ(0, fileSystem.GetCounterForBundledCommit());
                }

                // 仮コミット前のデータがロールバックされていることを確かめます。
                {
                    NNT_ASSERT_RESULT_SUCCESS(
                        fileSystem.OpenFile(
                            &pFile,
                            pathFile,
                            nn::fs::OpenMode_Read
                        )
                    );
                    NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(offset, bufRead.data(), size));
                    fileSystem.CloseFile(pFile);
                    EXPECT_TRUE(std::equal(bufRead.begin(), bufRead.end(), bufPrevious.begin()));
                    fileSystem.Finalize();

                    nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData readExtraData;
                    NNT_ASSERT_RESULT_SUCCESS(
                        nn::fssystem::save::JournalIntegritySaveDataFileSystem::ReadExtraData(
                            &readExtraData,
                            storageBase,
                            GetBufferManager(),
                            nnt::fs::util::GetMacGenerator(),
                            nnt::fs::util::SaveDataMinimumVersion
                        )
                    );
                    ASSERT_EQ(0, memcmp(&secondExtraData, &readExtraData, sizeof(readExtraData)));
                }
            }

            // ファイル作成、仮コミット、ロールバックします。
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.Initialize(
                        storageBase,
                        GetBufferManager(),
                        nnt::fs::util::GetMacGenerator(),
                        nnt::fs::util::SaveDataMinimumVersion
                    )
                );
                nn::fssystem::save::Path filePath;
                const auto FileSize = 24;
                filePath.Initialize("/file8");
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile(filePath, FileSize));
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.CommitProvisionally(1));
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.Rollback());
                fileSystem.Finalize();

                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.Initialize(
                        storageBase,
                        GetBufferManager(),
                        nnt::fs::util::GetMacGenerator(),
                        nnt::fs::util::SaveDataMinimumVersion
                    )
                );
                nn::fssystem::save::IFile* pFile = nullptr;
                NNT_ASSERT_RESULT_FAILURE(
                    nn::fs::ResultNotFound,
                    fileSystem.OpenFile(&pFile, filePath, nn::fs::OpenMode_Read)
                );
                fileSystem.Finalize();
            }

            // 仮コミットした状態に書き込みはできません。
            // 拡張データのみ書き込みできます。
            {
                nn::fssystem::save::IFile* pFile = nullptr;
                const auto offset = std::uniform_int_distribution<int64_t>(0, sizeFile - 1)(mt);
                const auto sizeMax = static_cast<size_t>(sizeFile - offset);
                const auto size = std::uniform_int_distribution<size_t>(1, sizeMax)(mt);
                nnt::fs::util::Vector<char> bufRead(size, 0);

                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.Initialize(
                        storageBase,
                        GetBufferManager(),
                        nnt::fs::util::GetMacGenerator(),
                        nnt::fs::util::SaveDataMinimumVersion
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.OpenFile(
                        &pFile,
                        pathFile,
                        nn::fs::OpenMode_Write
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(offset, bufWrite.data(), size));
                fileSystem.CloseFile(pFile);
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.CommitProvisionally(3));

                // 仮コミット中でも拡張データは書き込めます。
                nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData writeExtraData = {};
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.WriteExtraData(
                        writeExtraData
                    )
                );

                // 仮コミット中はファイルには書き込めません。
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.OpenFile(
                        &pFile,
                        pathFile,
                        nn::fs::OpenMode_Read | nn::fs::OpenMode_Write
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(offset, bufWrite.data(), size));
                NNT_ASSERT_RESULT_FAILURE(
                    nn::fs::ResultUnexpected, pFile->WriteBytes(offset, bufWrite.data(), size));
                fileSystem.CloseFile(pFile);

                NNT_ASSERT_RESULT_SUCCESS(fileSystem.Commit());

                // コミットを行うと書き込めるようになります
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.OpenFile(
                        &pFile,
                        pathFile,
                        nn::fs::OpenMode_Read | nn::fs::OpenMode_Write
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(offset, bufWrite.data(), size));
                NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(offset, bufWrite.data(), size));
                fileSystem.CloseFile(pFile);
                fileSystem.Finalize();

                nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData readExtraData;
                NNT_ASSERT_RESULT_SUCCESS(
                    nn::fssystem::save::JournalIntegritySaveDataFileSystem::ReadExtraData(
                        &readExtraData,
                        storageBase,
                        GetBufferManager(),
                        nnt::fs::util::GetMacGenerator(),
                        nnt::fs::util::SaveDataMinimumVersion
                    )
                );
                ASSERT_EQ(0, memcmp(&writeExtraData, &readExtraData, sizeof(readExtraData)));
            }
        }
    }

    NN_SDK_LOG("\n");
} // NOLINT(impl/function_size)

//! 拡張の挙動をテストします。
TEST_F(FsJournalIntegritySaveDataFileSystemDriverTest, Expand)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    static const auto LoopCount = 3;
    static const auto CountExpandMax = 25;

    for( size_t sizeBlock = 4096; sizeBlock <= MaxBlockSize; sizeBlock *= 2 )
    {
        NN_SDK_LOG(".");

        for( auto loop = 0; loop < LoopCount; ++loop )
        {
            nnt::fs::util::AccessCountedMemoryStorage baseStorageData;
            nnt::fs::util::AlignCheckStorage storageData;
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver fileSystem;

            // ブロックサイズ
            nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam paramDuplex;
            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
                paramIntegrity;
            const size_t sizeBlock1 = GenerateRandomBlockSize(512, MaxBlockSize);
            // sizeBlock1 + sizeBlock2 <= 1024 + 512 だと検証層のサイズが不足することがある
            const size_t minSizeBlock2 = (sizeBlock1 <= 1024) ? (1024 * 1024 / sizeBlock1) : 512;
            const size_t sizeBlock2 = GenerateRandomBlockSize(minSizeBlock2, MaxBlockSize);
            paramDuplex.sizeBlockLevel[0] = sizeBlock1 / 4;
            paramDuplex.sizeBlockLevel[1] = sizeBlock2;
            paramIntegrity.sizeBlockLevel[0] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[1] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[2] = sizeBlock2;
            paramIntegrity.sizeBlockLevel[3] = sizeBlock2;

            static const auto HashSize = nn::crypto::Sha256Generator::HashSize;
            const auto countMax = std::min<int64_t>(
                                      static_cast<uint64_t>(paramIntegrity.sizeBlockLevel[0])
                                      / HashSize * paramIntegrity.sizeBlockLevel[1]
                                      / HashSize * paramIntegrity.sizeBlockLevel[2]
                                      / HashSize * paramIntegrity.sizeBlockLevel[3]
                                      / sizeBlock,
                                      120
                                  );
            const auto countDataBlock = std::uniform_int_distribution<int64_t>(
                                            countMax * 5 / 6,
                                            countMax
                                        )(mt);
            const auto countDataBlockExpanded = std::uniform_int_distribution<int64_t>(
                                                    countMax,
                                                    countMax * 2
                                                )(mt);
            const auto countReservedBlock = countDataBlock;
            const auto countReservedBlockExpanded = countDataBlockExpanded;

            int64_t sizeTotal = 0;
            int64_t sizeExpanded = 0;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
                    &sizeTotal,
                    sizeBlock,
                    CountExpandMax,
                    paramDuplex,
                    paramIntegrity,
                    static_cast<uint32_t>(countDataBlock),
                    static_cast<uint32_t>(countReservedBlock)
                )
            );
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
                    &sizeExpanded,
                    sizeBlock,
                    CountExpandMax,
                    paramDuplex,
                    paramIntegrity,
                    static_cast<uint32_t>(countDataBlockExpanded) * 2,
                    static_cast<uint32_t>(countReservedBlockExpanded) * 2
                )
            );

            // ファイルシステムを配置する Storage を構築する
            baseStorageData.Initialize(sizeExpanded);
            // JournalIntegritySaveDataFileSystemDriver は、必ず 16KB アライメントのアクセスが発生する仕様のため
            // 16 KB を超えるアクセスアライメントのチェックは不要です。
            size_t sizeAlignment = std::min(sizeBlock, static_cast<size_t>(16 * 1024));
            storageData.Initialize(
                nn::fs::SubStorage(&baseStorageData, 0, sizeExpanded),
                sizeAlignment
            );

            nn::fs::SubStorage storageBase(&storageData, 0, sizeTotal);
            nn::fs::SubStorage storageExpanded(&storageData, 0, sizeExpanded);

            nn::fs::SaveDataHashSalt salt;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::Format(
                    nn::fs::SubStorage(&storageData, 0, sizeTotal),
                    sizeBlock,
                    CountExpandMax,
                    paramDuplex,
                    paramIntegrity,
                    static_cast<uint32_t>(countDataBlock),
                    static_cast<uint32_t>(countReservedBlock),
                    GetBufferManager(),
                    nnt::fs::util::GetMacGenerator(),
                    salt
                )
            );

            NNT_ASSERT_RESULT_SUCCESS(
                fileSystem.Initialize(
                    storageBase,
                    GetBufferManager(),
                    nnt::fs::util::GetMacGenerator(),
                    nnt::fs::util::SaveDataMinimumVersion
                )
            );

            nn::fssystem::save::Path pathDirectory;
            nn::fssystem::save::Path pathFile;

            // ディレクトリとファイルを作成します。
            nn::fssystem::save::IFile* pFile = nullptr;
            NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize("/directory"));
            NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory(pathDirectory));
            NNT_ASSERT_RESULT_SUCCESS(pathFile.Initialize("/directory/file.dat"));
            NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile(pathFile, 0));

            // 指定したサイズを超えるサイズへのファイルサイズ拡張はできません。
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.OpenFile(
                        &pFile,
                        pathFile,
                        nn::fs::OpenMode_Write
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(sizeBlock * countDataBlock / 2));
                NNT_ASSERT_RESULT_FAILURE(
                    nn::fs::ResultAllocationTableFull,
                    pFile->SetSize(sizeBlock * countDataBlock / 2 * 3)
                );
                int64_t sizeFile = 0;
                NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&sizeFile));
                EXPECT_EQ(sizeBlock * countDataBlock / 2, sizeFile);
                fileSystem.CloseFile(pFile);
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.Commit());
            }

            // ファイルシステムを拡張します。
            {
                fileSystem.Finalize();
                const auto logStorageSize
                    = nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryExpandLogSize(
                          sizeBlock,
                          static_cast<uint32_t>(countDataBlockExpanded),
                          static_cast<uint32_t>(countReservedBlockExpanded)
                      );
                nnt::fs::util::SafeMemoryStorage storageLog(logStorageSize);
                NNT_ASSERT_RESULT_SUCCESS(
                    nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::OperateExpand(
                        storageExpanded,
                        nn::fs::SubStorage(&storageLog, 0, logStorageSize),
                        sizeBlock,
                        static_cast<uint32_t>(countDataBlockExpanded),
                        static_cast<uint32_t>(countReservedBlockExpanded),
                        GetBufferManager(),
                        nnt::fs::util::GetMacGenerator(),
                        nnt::fs::util::SaveDataMinimumVersion
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(
                    nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::CommitExpand(
                        storageExpanded,
                        nn::fs::SubStorage(&storageLog, 0, logStorageSize),
                        sizeBlock,
                        GetBufferManager()
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.Initialize(
                        storageExpanded,
                        GetBufferManager(),
                        nnt::fs::util::GetMacGenerator(),
                        nnt::fs::util::SaveDataMinimumVersion
                    )
                );
            }

            // ファイルをより大きいサイズに拡張できるようになります。
            // 指定のブロック数からエントリ用ブロック分を除いたサイズのファイルを作成できます
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.OpenFile(
                        &pFile,
                        pathFile,
                        nn::fs::OpenMode_Write
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->SetSize(sizeBlock * (countDataBlockExpanded - 2)));
                NNT_EXPECT_RESULT_FAILURE(
                    nn::fs::ResultAllocationTableFull,
                    pFile->SetSize(sizeBlock * (countDataBlockExpanded - 2) + 1)
                );
                int64_t sizeFile = 0;
                NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&sizeFile));
                EXPECT_EQ(sizeBlock * (countDataBlockExpanded - 2), sizeFile);
                fileSystem.CloseFile(pFile);
            }

            fileSystem.Finalize();
        }
    }

    NN_SDK_LOG("\n");
} // NOLINT(impl/function_size)

//! 拡張の中断と再開の挙動をテストします。
TEST_F(FsJournalIntegritySaveDataFileSystemDriverTest, InterruptExpand)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    static const auto LoopCount = 5;
    static const auto CountExpandMax = 25;

    for( size_t sizeBlock = 4096; sizeBlock <= MaxBlockSize; sizeBlock *= 2 )
    {
        NN_SDK_LOG(".");

        for( auto loop = 0; loop < LoopCount; ++loop )
        {
            // JournalIntegritySaveDataFileSystemDriver は、必ず 16KB アライメントのアクセスが発生する仕様のため
            // 16 KB を超えるアクセスアライメントのチェックは不要です。
            size_t sizeAlignment = std::min(sizeBlock, static_cast<size_t>(16 * 1024));

            nnt::fs::util::AccessCountedMemoryStorage baseStorageData;
            nnt::fs::util::AlignCheckStorage storageData;
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver fileSystem;

            const auto countDataBlock = std::uniform_int_distribution<>(100, 120)(mt);
            const auto countDataBlockExpanded = std::uniform_int_distribution<>(120, 240)(mt);

            // 最終形態までは予約領域のサイズにはデータ領域以上使用します。
            const auto countReservedBlock = countDataBlock;
            const auto countReservedBlockExpanded = countDataBlockExpanded;

            // ブロックサイズ
            nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam paramDuplex;
            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
                paramIntegrity;
            const size_t sizeBlock1 = GenerateRandomBlockSize(512, MaxBlockSize);
            // sizeBlock1 + sizeBlock2 <= 1024 + 512 だと検証層のサイズが不足することがある
            const size_t minSizeBlock2 = (sizeBlock1 <= 1024) ? (1024 * 1024 / sizeBlock1) : 512;
            const size_t sizeBlock2 = GenerateRandomBlockSize(minSizeBlock2, MaxBlockSize);
            paramDuplex.sizeBlockLevel[0] = sizeBlock1 / 4;
            paramDuplex.sizeBlockLevel[1] = sizeBlock2;
            paramIntegrity.sizeBlockLevel[0] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[1] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[2] = sizeBlock2;
            paramIntegrity.sizeBlockLevel[3] = sizeBlock2;

            int64_t sizeTotal = 0;
            int64_t sizeExpanded = 0;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
                    &sizeTotal,
                    sizeBlock,
                    CountExpandMax,
                    paramDuplex,
                    paramIntegrity,
                    static_cast<uint32_t>(countDataBlock),
                    static_cast<uint32_t>(countReservedBlock)
                )
            );
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
                    &sizeExpanded,
                    sizeBlock,
                    CountExpandMax,
                    paramDuplex,
                    paramIntegrity,
                    static_cast<uint32_t>(countDataBlockExpanded),
                    static_cast<uint32_t>(countReservedBlockExpanded)
                )
            );

            // ファイルシステムを配置する Storage を構築する
            baseStorageData.Initialize(sizeExpanded);
            storageData.Initialize(
                nn::fs::SubStorage(&baseStorageData, 0, sizeExpanded),
                sizeAlignment
            );

            nn::fs::SubStorage storageBase(&storageData, 0, sizeTotal);
            nn::fs::SubStorage storageExpanded(&storageData, 0, sizeExpanded);

            nn::fs::SaveDataHashSalt salt;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::Format(
                    nn::fs::SubStorage(&storageData, 0, sizeTotal),
                    sizeBlock,
                    CountExpandMax,
                    paramDuplex,
                    paramIntegrity,
                    static_cast<uint32_t>(countDataBlock),
                    static_cast<uint32_t>(countReservedBlock),
                    GetBufferManager(),
                    nnt::fs::util::GetMacGenerator(),
                    salt
                )
            );

            NNT_ASSERT_RESULT_SUCCESS(
                fileSystem.Initialize(
                    storageBase,
                    GetBufferManager(),
                    nnt::fs::util::GetMacGenerator(),
                    nnt::fs::util::SaveDataMinimumVersion
                )
            );

            nn::fssystem::save::Path pathDirectory;
            nn::fssystem::save::Path pathFile;

            // ディレクトリとファイルを作成します。
            NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize("/directory"));
            NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory(pathDirectory));
            NNT_ASSERT_RESULT_SUCCESS(pathFile.Initialize("/directory/file.dat"));
            NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile(pathFile, 0));
            fileSystem.Finalize();

            // ファイルシステムを拡張します。
            const auto logStorageSize
                = nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryExpandLogSize(
                      sizeBlock,
                      static_cast<uint32_t>(countDataBlockExpanded),
                      static_cast<uint32_t>(countReservedBlockExpanded)
                  );
            nnt::fs::util::SafeMemoryStorage baseStorageLog(logStorageSize);
            nnt::fs::util::AlignCheckStorage storageLog(
                nn::fs::SubStorage(&baseStorageLog, 0, logStorageSize),
                sizeAlignment
            );
            nnt::fs::util::SafeMemoryStorage baseStoragePreviousData(baseStorageData.GetSize());
            nnt::fs::util::AlignCheckStorage storagePreviousData(
                nn::fs::SubStorage(&baseStoragePreviousData, 0, baseStorageData.GetSize()),
                sizeAlignment
            );

            // リサイズ前のデータを保存しておきます。
            baseStoragePreviousData.Copy(baseStorageData);

            // リサイズを実行します。
            baseStorageLog.FillBuffer(0, static_cast<size_t>(logStorageSize), 0xcc);
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::OperateExpand(
                    storageExpanded,
                    nn::fs::SubStorage(&storageLog, 0, logStorageSize),
                    sizeBlock,
                    static_cast<uint32_t>(countDataBlockExpanded),
                    static_cast<uint32_t>(countReservedBlockExpanded),
                    GetBufferManager(),
                    nnt::fs::util::GetMacGenerator(),
                    nnt::fs::util::SaveDataMinimumVersion
                )
            );

            // リサイズ中に中断しても再度リサイズが実行されます。
            {
                nnt::fs::util::SafeMemoryStorage baseStoragePreviousLog(baseStorageLog.GetSize());
                nnt::fs::util::AlignCheckStorage storagePreviousLog(
                    nn::fs::SubStorage(&baseStoragePreviousLog, 0, baseStorageLog.GetSize()),
                    sizeAlignment
                );
                baseStoragePreviousLog.Copy(baseStorageLog);

                // ログを壊してリサイズ中に中断されたかのような状態を再現します。
                baseStorageLog.FillBuffer(0, static_cast<size_t>(logStorageSize), 0xcc);
                baseStorageData.Copy(baseStoragePreviousData);

                // 再度リサイズが実行されログが作り直されます。
                NNT_ASSERT_RESULT_SUCCESS(
                    nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::OperateExpand(
                        storageExpanded,
                        nn::fs::SubStorage(&storageLog, 0, logStorageSize),
                        sizeBlock,
                        static_cast<uint32_t>(countDataBlockExpanded),
                        static_cast<uint32_t>(countReservedBlockExpanded),
                        GetBufferManager(),
                        nnt::fs::util::GetMacGenerator(),
                        nnt::fs::util::SaveDataMinimumVersion
                    )
                );
                const auto pLogBegin = reinterpret_cast<char*>(baseStorageLog.GetBuffer());
                const auto pLogEnd = pLogBegin + baseStorageLog.GetSize();
                const auto pPreviousBegin =
                    reinterpret_cast<char*>(baseStoragePreviousLog.GetBuffer());
                EXPECT_TRUE(std::equal(pLogBegin, pLogEnd, pPreviousBegin));
            }

            // コミットを実行します
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::CommitExpand(
                    storageExpanded,
                    nn::fs::SubStorage(&storageLog, 0, logStorageSize),
                    sizeBlock,
                    GetBufferManager()
                )
            );

            // コミット中に中断しても再度コミットが実行されます。
            {
                baseStoragePreviousData.Copy(baseStorageData);

                // データを壊してコミット中に中断されたかのような状態を再現します。
                baseStorageData.FillBuffer(0, sizeBlock, 0xcc);

                // 再度コミットが実行されデータが書き直されます。
                NNT_ASSERT_RESULT_SUCCESS(
                    nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::CommitExpand(
                        storageExpanded,
                        nn::fs::SubStorage(&storageLog, 0, logStorageSize),
                        sizeBlock,
                        GetBufferManager()
                    )
                );
                const auto pDataBegin = reinterpret_cast<char*>(baseStorageData.GetBuffer());
                const auto pDataEnd = pDataBegin + baseStorageData.GetSize();
                const auto pPreviousBegin =
                    reinterpret_cast<char*>(baseStoragePreviousData.GetBuffer());
                EXPECT_TRUE(std::equal(pDataBegin, pDataEnd, pPreviousBegin));
            }
        }
    }

    NN_SDK_LOG("\n");
} // NOLINT(impl/function_size)

TEST_F(FsJournalIntegritySaveDataFileSystemDriverTest, MultipleSaveDataFileSystemsHeavy)
{
    static const auto LoopCount = 1000;
    static const auto StackSize = 32 * 1024;
    static const auto CountFileUnit = 6;
    static const auto SaveDataCount = 5 * CountFileUnit;
    static const auto SizeBlock = 16 * 1024;
    static const auto CountExpandMax = 25;

#ifdef NN_BUILD_CONFIG_ADDRESS_64
#ifdef NN_BUILD_CONFIG_OS_WIN
    static const auto CountDataBlockUnit = 100;
    static const auto CountFileBlockUnit = 80;
    static const auto CountBufferBlock = 64;
#else
    static const auto CountDataBlockUnit = 50;
    static const auto CountFileBlockUnit = 40;
    static const auto CountBufferBlock = 32;
#endif
#else
    static const auto CountDataBlockUnit = 20;
    static const auto CountFileBlockUnit = 16;
    static const auto CountBufferBlock = 12;
#endif

    class Statistics
    {
    public:
        Statistics() NN_NOEXCEPT
            : m_Sum(0),
              m_SumSquared(0),
              m_Max(0),
              m_Min(0),
              m_Count(0),
              m_Lock(false)
        {
        }

        void Add(double sample) NN_NOEXCEPT
        {
            m_Lock.lock();
            m_Sum += sample;
            m_SumSquared += sample * sample;
            m_Max = std::max(sample, m_Max);
            m_Min = std::min(sample, m_Min);
            ++m_Count;
            m_Lock.unlock();
        }

        double GetAverage() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_GREATER(m_Count, 0);
            return m_Sum / m_Count;
        }

        double GetVariance() const NN_NOEXCEPT
        {
            const auto average = GetAverage();
            return m_SumSquared / m_Count - average * average;
        }

        double GetStandardDeviation() const NN_NOEXCEPT
        {
            return std::sqrt(GetVariance());
        }

        double GetMax() const NN_NOEXCEPT
        {
            return m_Max;
        }

        double GetMin() const NN_NOEXCEPT
        {
            return m_Min;
        }

    private:
        double m_Sum;
        double m_SumSquared;
        double m_Max;
        double m_Min;
        int m_Count;
        nn::os::Mutex m_Lock;
    };

    static Statistics statisticsWrite;
    static Statistics statisticsRead;

    MeasurableBufferManager<false> bufferManager(GetBufferManager());

    class SaveData
    {
    public:
        nn::Result Initialize(int id, nn::fssystem::IBufferManager* pBufferManager) NN_NOEXCEPT
        {
            m_Id = id;

            m_Mt.seed(nnt::fs::util::GetRandomSeed());

            nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam paramDuplex;
            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
                paramIntegrity;
            paramDuplex.sizeBlockLevel[0] = SizeBlock;
            paramDuplex.sizeBlockLevel[1] = SizeBlock;
            paramIntegrity.sizeBlockLevel[0] = SizeBlock;
            paramIntegrity.sizeBlockLevel[1] = SizeBlock;
            paramIntegrity.sizeBlockLevel[2] = SizeBlock;
            paramIntegrity.sizeBlockLevel[3] = SizeBlock;

            const auto countDataBlock = (id < CountFileUnit ? 100 : (id < 2 * CountFileUnit ? 10 : 1)) * CountDataBlockUnit;
            const auto countReservedBlock = countDataBlock;

            int64_t sizeTotal = 0;
            NN_RESULT_DO(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
                    &sizeTotal,
                    SizeBlock,
                    CountExpandMax,
                    paramDuplex,
                    paramIntegrity,
                    countDataBlock,
                    countReservedBlock
                )
            );

            m_StorageData.Initialize(sizeTotal);

            nn::fs::SubStorage storageBase(&m_StorageData, 0, sizeTotal);

            nn::fs::SaveDataHashSalt salt;
            NN_RESULT_DO(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::Format(
                    storageBase,
                    SizeBlock,
                    CountExpandMax,
                    paramDuplex,
                    paramIntegrity,
                    countDataBlock,
                    countReservedBlock,
                    pBufferManager,
                    nnt::fs::util::GetMacGenerator(),
                    salt
                )
            );

            NN_RESULT_DO(
                m_FileSystem.Initialize(
                    storageBase,
                    pBufferManager,
                    nnt::fs::util::GetMacGenerator(),
                    nnt::fs::util::SaveDataMinimumVersion
                )
            );

            m_SizeFile = (id < CountFileUnit ? 100 : (id < 2 * CountFileUnit ? 10 : 1)) * CountFileBlockUnit * SizeBlock;

            nn::fssystem::save::Path pathDirectory;
            NN_RESULT_DO(pathDirectory.Initialize("/directory"));
            NN_RESULT_DO(m_FileSystem.CreateDirectory(pathDirectory));
            NN_RESULT_DO(m_FilePath.Initialize("/directory/file.dat"));
            NN_RESULT_DO(m_FileSystem.CreateFile(m_FilePath, m_SizeFile));
            NN_RESULT_DO(m_FileSystem.Commit());

            m_Buffer.resize(SizeBlock * CountBufferBlock);
            std::iota(m_Buffer.begin(), m_Buffer.end(), static_cast<char>(id));

            NN_RESULT_SUCCESS;
        }

        nn::Result Access() NN_NOEXCEPT
        {
            const auto close = [this](nn::fssystem::save::IFile* pFile) NN_NOEXCEPT
            {
                m_FileSystem.CloseFile(pFile);
            };
            typedef std::unique_ptr<nn::fssystem::save::IFile, decltype(close)> UniqueFile;

            m_StorageData.ResetAccessCounter();

            for( auto loop = 0; loop < LoopCount; ++loop )
            {
                if( (loop + 1) % 100 == 0 )
                {
                    NN_RESULT_DO(m_FileSystem.Commit());

                    statisticsWrite.Add(m_StorageData.GetWriteTimes());
                    statisticsRead.Add(m_StorageData.GetReadTimes());
                    m_StorageData.ResetAccessCounter();
                }
                else if( loop % 3 != 0 )
                {
                    UniqueFile pUniqueFile(nullptr, close);
                    {
                        nn::fssystem::save::IFile* pFile = nullptr;
                        NN_RESULT_DO(
                            m_FileSystem.OpenFile(
                                &pFile,
                                m_FilePath,
                                nn::fs::OpenMode_Write
                            )
                        );
                        pUniqueFile.reset(pFile);
                    }

                    const auto offsetMax = static_cast<int64_t>(m_SizeFile - m_Buffer.size());
                    const auto offset = std::uniform_int_distribution<int64_t>(0, offsetMax)(m_Mt);
                    NN_RESULT_DO(pUniqueFile->WriteBytes(offset, m_Buffer.data(), m_Buffer.size()));
                }
                else
                {
                    UniqueFile pUniqueFile(nullptr, close);
                    {
                        nn::fssystem::save::IFile* pFile = nullptr;
                        NN_RESULT_DO(
                            m_FileSystem.OpenFile(
                                &pFile,
                                m_FilePath,
                                nn::fs::OpenMode_Read
                            )
                        );
                        pUniqueFile.reset(pFile);
                    }

                    const auto offsetMax = static_cast<int64_t>(m_SizeFile - m_Buffer.size());
                    const auto offset = std::uniform_int_distribution<int64_t>(0, offsetMax)(m_Mt);
                    NN_RESULT_DO(pUniqueFile->ReadBytes(offset, m_Buffer.data(), m_Buffer.size()));
                }

                if( m_Id == 0 && loop % 10 == 0 )
                {
                    NN_SDK_LOG(".");
                }
            }
            NN_RESULT_SUCCESS;
        }

        void Finalize() NN_NOEXCEPT
        {
            m_FileSystem.Finalize();
            m_StorageData.Finalize();
            m_Buffer.clear();
            m_Buffer.shrink_to_fit();
        }

    private:
        int m_Id;
        std::mt19937 m_Mt;
        nnt::fs::util::AccessCountedMemoryStorage m_StorageData;
        nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver m_FileSystem;
        nn::fssystem::save::Path m_FilePath;
        nnt::fs::util::Vector<char> m_Buffer;
        std::size_t m_SizeFile;
    };

    static SaveData s_SaveData[SaveDataCount] = {};
    static nn::os::ThreadType s_Threads[SaveDataCount] = {};
    NN_OS_ALIGNAS_THREAD_STACK static char s_Stack[StackSize * SaveDataCount] = {};

    while( !bufferManager.IsBufferStarved() )
    {
        for( auto& data : s_SaveData )
        {
            NNT_ASSERT_RESULT_SUCCESS(data.Initialize(static_cast<int>(&data - s_SaveData), &bufferManager));
        }

        for( auto& thread : s_Threads )
        {
            const auto index = &thread - s_Threads;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::os::CreateThread(
                    &thread,
                    [](void* pArgument) NN_NOEXCEPT
            {
                NN_SDK_REQUIRES_NOT_NULL(pArgument);
                NNT_EXPECT_RESULT_SUCCESS(reinterpret_cast<SaveData*>(pArgument)->Access());
            },
                    s_SaveData + index,
                s_Stack + StackSize * index,
                StackSize,
                nn::os::DefaultThreadPriority
                )
            );
        }

        for( auto& thread : s_Threads )
        {
            nn::os::StartThread(&thread);
        }

        for( auto& thread : s_Threads )
        {
            nn::os::WaitThread(&thread);
            nn::os::DestroyThread(&thread);
        }

        for( auto& data : s_SaveData )
        {
            data.Finalize();
        }

        EXPECT_LT(statisticsWrite.GetAverage(), 1500);
        EXPECT_LT(statisticsRead.GetAverage(), 4000);
        EXPECT_LT(statisticsWrite.GetMax(), 1500);
        EXPECT_LT(statisticsRead.GetMax(), 8000);
    }

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    // 実行するコア数に依存するため、NX 版でのみチェックする
    EXPECT_LE(static_cast<size_t>(512 * 1024), bufferManager.GetTotalAllocatableSizeMin());
#endif

    NN_SDK_LOG("\n");

    NN_LOG("total allocatable size min: %u bytes\n", static_cast<uint32_t>(bufferManager.GetTotalAllocatableSizeMin()));
} // NOLINT(impl/function_size)

// サイズの大きなストレージを持つファイルシステムのテストします。
TEST_F(FsJournalIntegritySaveDataFileSystemDriverTest, LargeStorageHeavy)
{
    static const int CountExpandMax = 25;
    static const size_t BlockSize = 32 * 1024;
    static const int64_t Giga = 1024 * 1024 * 1024;
    static const int64_t StorageSize = 2 * 1024 * Giga;
    static const uint32_t DataBlockCount = 128;
    static const size_t DataSize = 64 * 1024;
    static const int BlockCountStep = 4;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::Vector<char> readBuf(DataSize);
    nnt::fs::util::Vector<char> baseBuf(DataSize);
    std::iota(baseBuf.begin(), baseBuf.end(), '\x0');

    // ファイルシステムを配置する Storage を構築
    nnt::fs::util::VirtualMemoryStorage storageData;
    storageData.Initialize(StorageSize);

    nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam paramDuplex;
    nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
        paramIntegrity;
    paramDuplex.sizeBlockLevel[0] = MaxBlockSize / 4;
    paramDuplex.sizeBlockLevel[1] = MaxBlockSize;
    paramIntegrity.sizeBlockLevel[0] = MaxBlockSize;
    paramIntegrity.sizeBlockLevel[1] = MaxBlockSize;
    paramIntegrity.sizeBlockLevel[2] = MaxBlockSize;
    paramIntegrity.sizeBlockLevel[3] = MaxBlockSize;

    int64_t sizeTotal = 0;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
            &sizeTotal,
            BlockSize,
            CountExpandMax,
            paramDuplex,
            paramIntegrity,
            DataBlockCount,
            DataBlockCount
        )
    );
    EXPECT_GE(StorageSize, sizeTotal);

    nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver fileSystem;
    nn::fs::SubStorage storageBase(&storageData, 0, sizeTotal);

    nn::fs::SaveDataHashSalt salt;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::Format(
            nn::fs::SubStorage(&storageData, 0, sizeTotal),
            BlockSize,
            CountExpandMax,
            paramDuplex,
            paramIntegrity,
            DataBlockCount,
            DataBlockCount,
            GetBufferManager(),
            nnt::fs::util::GetMacGenerator(),
            salt
        )
    );

    NNT_ASSERT_RESULT_SUCCESS(
        fileSystem.Initialize(
            storageBase,
            GetBufferManager(),
            nnt::fs::util::GetMacGenerator(),
            nnt::fs::util::SaveDataMinimumVersion
        )
    );

    // ファイルを作成
    nn::fssystem::save::Path pathFile;
    NNT_ASSERT_RESULT_SUCCESS(pathFile.Initialize("/file.dat"));
    {
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile(pathFile, 0));

        nn::fssystem::save::IFile* pFile = nullptr;
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.OpenFile(&pFile, pathFile, nn::fs::OpenMode_Write));

        NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(BlockSize * DataBlockCount / 2));
        int64_t sizeFile;
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&sizeFile));
        EXPECT_EQ(BlockSize * DataBlockCount / 2, sizeFile);

        fileSystem.CloseFile(pFile);
    }
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.Commit());

    auto sizeExpanded = sizeTotal;
    auto blockCount = DataBlockCount << BlockCountStep;

    // ある程度の範囲でファイルシステムが拡張できることを確認
    for( ; blockCount < 0x800000; blockCount <<= BlockCountStep )
    {
        int64_t size = 0;
        nn::Result result =
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
                &size,
                BlockSize,
                CountExpandMax,
                paramDuplex,
                paramIntegrity,
                blockCount,
                blockCount
            );

        if( nn::fs::ResultInvalidSize::Includes(result) || StorageSize < size )
        {
            break;
        }
        NNT_ASSERT_RESULT_SUCCESS(result);
        sizeExpanded = size;

        fileSystem.Finalize();

        nn::fs::SubStorage storageExpanded(&storageData, 0, sizeExpanded);

        const auto logStorageSize =
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryExpandLogSize(
                    BlockSize,
                    blockCount,
                    blockCount
                );

        nnt::fs::util::SafeMemoryStorage storageLog(logStorageSize);
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::OperateExpand(
                storageExpanded,
                nn::fs::SubStorage(&storageLog, 0, logStorageSize),
                BlockSize,
                blockCount,
                blockCount,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::CommitExpand(
                storageExpanded,
                nn::fs::SubStorage(&storageLog, 0, logStorageSize),
                BlockSize,
                GetBufferManager()
            )
        );

        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Initialize(
                storageExpanded,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );
    }
    blockCount >>= BlockCountStep;

    // 拡張前のファイルが削除できることを確認
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.DeleteFile(pathFile));

    // 様々なサイズのファイルを作成・削除できることを確認
    for( size_t blockSize = std::uniform_int_distribution<>(128, 512)(mt);
            blockSize < BlockSize;
            blockSize = (blockSize << 1) + std::uniform_int_distribution<>(0, 256)(mt) )
    {
        char fileName[32];
        std::sprintf(fileName, "/%X.dat", static_cast<uint32_t>(blockSize));

        const auto fileSize = static_cast<int64_t>(blockSize) * blockCount;

        nn::fssystem::save::Path path;
        NNT_ASSERT_RESULT_SUCCESS(path.Initialize(fileName));
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile(path, fileSize));
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.DeleteFile(path));
    }

    const auto fileSize = static_cast<int64_t>(BlockSize) * (blockCount - 2);

    // 最大サイズのファイルを作成してデータを書き込む
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile(pathFile, fileSize));
    {
        nn::fssystem::save::IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.OpenFile(&pFile, pathFile, nn::fs::OpenMode_Write));

        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(0, baseBuf.data(), DataSize));
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(fileSize - DataSize, baseBuf.data(), DataSize));

        fileSystem.CloseFile(pFile);
    }
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.Commit());

    fileSystem.Finalize();

    // 再マウント
    NNT_ASSERT_RESULT_SUCCESS(
        fileSystem.Initialize(
            nn::fs::SubStorage(&storageData, 0, sizeExpanded),
            GetBufferManager(),
            nnt::fs::util::GetMacGenerator(),
            nnt::fs::util::SaveDataMinimumVersion
        )
    );

    // マウント前のデータが読み込めるか確認
    {
        nn::fssystem::save::IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.OpenFile(&pFile, pathFile, nn::fs::OpenMode_Read));

        NNT_ASSERT_RESULT_SUCCESS(
            pFile->ReadBytes(0, readBuf.data(), DataSize));
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(baseBuf.data(), readBuf.data(), DataSize);

        NNT_ASSERT_RESULT_SUCCESS(
            pFile->ReadBytes(fileSize - DataSize, readBuf.data(), DataSize));
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(baseBuf.data(), readBuf.data(), DataSize);

        fileSystem.CloseFile(pFile);
    }
} // NOLINT(impl/function_size)

//! 拡張データの読み書きが正常に行えるか確かめます。
TEST_F(FsJournalIntegritySaveDataFileSystemDriverTest, ExtraData)
{
    nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver fileSystem;
    nnt::fs::util::AccessCountedMemoryStorage storageData;
    nn::fs::SubStorage storageBase;
    nn::fs::SubStorage storageExpand;

    // データのサイズ
    const auto countDataBlock = 512;
    const auto countReservedBlock = countDataBlock / 2;

    const auto countDataBlockExpanded = 1024;
    const auto countReservedBlockExpanded = countDataBlockExpanded / 2;

    const size_t sizeBlock = GenerateRandomBlockSize(512, MaxBlockSize);

    // 初期化
    {
        const auto CountExpandMax = 10;

        // ブロックサイズ
        nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam paramDuplex;
        nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
            paramIntegrity;

        const size_t sizeBlock1 = GenerateRandomBlockSize(512, MaxBlockSize);
        const size_t sizeBlock2 = SetupBlockSize2ForExtraDataTest(
            sizeBlock, countDataBlockExpanded, sizeBlock1
        );
        paramDuplex.sizeBlockLevel[0] = sizeBlock1 / 4;
        paramDuplex.sizeBlockLevel[1] = sizeBlock2;
        paramIntegrity.sizeBlockLevel[0] = sizeBlock1;
        paramIntegrity.sizeBlockLevel[1] = sizeBlock1;
        paramIntegrity.sizeBlockLevel[2] = sizeBlock2;
        paramIntegrity.sizeBlockLevel[3] = sizeBlock2;

        int64_t sizeTotal = 0;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
                &sizeTotal,
                sizeBlock,
                CountExpandMax,
                paramDuplex,
                paramIntegrity,
                static_cast<uint32_t>(countDataBlock),
                static_cast<uint32_t>(countReservedBlock)
            )
        );

        // 拡張後のサイズを計算
        int64_t sizeExpanded = 0;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
                &sizeExpanded,
                sizeBlock,
                CountExpandMax,
                paramDuplex,
                paramIntegrity,
                static_cast<uint32_t>(countDataBlockExpanded),
                static_cast<uint32_t>(countReservedBlockExpanded)
            )
        );

        // ファイルシステムを配置する Storage を構築する
        storageData.Initialize(sizeExpanded);
        storageBase = nn::fs::SubStorage(&storageData, 0, sizeTotal);
        storageExpand = nn::fs::SubStorage(&storageData, 0, sizeExpanded);

        nn::fs::SaveDataHashSalt salt;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::Format(
                storageBase,
                sizeBlock,
                CountExpandMax,
                paramDuplex,
                paramIntegrity,
                static_cast<uint32_t>(countDataBlock),
                static_cast<uint32_t>(countReservedBlock),
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                salt
            )
        );

        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Initialize(
                storageBase,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );

        // ファイル作成、書き込み、コミット
        {
            nn::fssystem::save::Path filePath;
            const auto FileSize = 2048;
            filePath.Initialize("/file");
            NNT_ASSERT_RESULT_SUCCESS(
                fileSystem.CreateFile(filePath, FileSize)
            );

            std::unique_ptr<char[]> fileBuffer(new char[FileSize]);
            nnt::fs::util::FillBufferWithRandomValue(fileBuffer.get(), FileSize);

            {
                nn::fssystem::save::IFile* pFile;
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.OpenFile(&pFile, filePath, nn::fs::OpenMode_Write)
                );
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->WriteBytes(0, fileBuffer.get(), FileSize)
                );
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->Flush()
                );
                fileSystem.CloseFile(pFile);
            }
            NNT_ASSERT_RESULT_SUCCESS(
                fileSystem.Commit()
            );
        }
    }

    // 書き込み、読み込み用バッファ作成
    nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData writeData0;
    nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData writeData1;
    nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData readData;

    nnt::fs::util::FillBufferWithRandomValue(&writeData0, 512);
    nnt::fs::util::FillBufferWithRandomValue(&writeData1, 512);

    // フォーマットした状態なら、拡張データの中身は 0 フィルされています
    {
        fileSystem.ReadExtraData(&readData);
        char* buffer = reinterpret_cast<char*>(&readData);
        for( int i=0; i < 512; ++i )
        {
            ASSERT_EQ(0, buffer[i]);
        }
    }

    // 拡張データを書き込み、コミットして静的・動的読み込み
    {
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.WriteExtraData(writeData0)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Commit()
        );

        // 読み込み
        fileSystem.ReadExtraData(&readData);
        ASSERT_EQ(0, memcmp(&readData, &writeData0, 512));

        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::ReadExtraData(
                &readData,
                storageBase,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );
        ASSERT_EQ(0, memcmp(&readData, &writeData0, 512));
    }

    // 拡張データを書き込み、コミットせずに静的・動的読み込み
    {
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.WriteExtraData(writeData1)
        );

        // 動的読み込み
        fileSystem.ReadExtraData(&readData);
        ASSERT_EQ(0, memcmp(&readData, &writeData1, 512));

        // 静的読み込み
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::ReadExtraData(
                &readData,
                storageBase,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );
        ASSERT_EQ(0, memcmp(&readData, &writeData0, 512));
    }

    // アンマウントして静的読み込み
    {
        fileSystem.Finalize();

        // 静的読み込み
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::ReadExtraData(
                &readData,
                storageBase,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );
        ASSERT_EQ(0, memcmp(&readData, &writeData0, 512));
    }

    // マウントして静的・動的読み込み
    {
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Initialize(
                storageBase,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );

        // 動的読み込み
        fileSystem.ReadExtraData(&readData);
        ASSERT_EQ(0, memcmp(&readData, &writeData0, 512));

        // 静的読み込み
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::ReadExtraData(
                &readData,
                storageBase,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );
        ASSERT_EQ(0, memcmp(&readData, &writeData0, 512));
    }

    // 書き込まずにコミットをして静的・動的読み込み
    {
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Commit()
        );

        // 動的読み込み
        fileSystem.ReadExtraData(&readData);
        ASSERT_EQ(0, memcmp(&readData, &writeData0, 512));

        // 静的読み込み
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::ReadExtraData(
                &readData,
                storageBase,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );
        ASSERT_EQ(0, memcmp(&readData, &writeData0, 512));
    }

    // セーブデータを拡張した後にマウントし、静的・動的読み込み
    {
        fileSystem.Finalize();

        // ログストレージ作成
        const auto logStorageSize
            = nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryExpandLogSize(
                sizeBlock,
                static_cast<uint32_t>(countDataBlockExpanded),
                static_cast<uint32_t>(countReservedBlockExpanded)
            );

        nnt::fs::util::SafeMemoryStorage storageLog(logStorageSize);

        // 拡張する
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::OperateExpand(
                storageExpand,
                nn::fs::SubStorage(&storageLog, 0, logStorageSize),
                sizeBlock,
                countDataBlock,
                countDataBlockExpanded,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::CommitExpand(
                storageExpand,
                nn::fs::SubStorage(&storageLog, 0, logStorageSize),
                sizeBlock,
                GetBufferManager()
            )
        );

        //マウント
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Initialize(
                storageExpand,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );

        // 拡張データが書き換えられていないことを確かめる
        // 動的読み込み
        fileSystem.ReadExtraData(&readData);
        ASSERT_EQ(0, memcmp(&readData, &writeData0, 512));

        // 静的読み込み
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::ReadExtraData(
                &readData,
                storageExpand,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );
        ASSERT_EQ(0, memcmp(&readData, &writeData0, 512));
    }
}// NOLINT(impl/function_size)

//! 仮コミットを行い、拡張データが壊れないことを確かめます。
TEST_F(FsJournalIntegritySaveDataFileSystemDriverTest, ExtraDataCommitProvisionally)
{
    nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver fileSystem;
    nnt::fs::util::AccessCountedMemoryStorage storageData;
    nn::fs::SubStorage storageBase;

    // 初期化
    {
        // データのサイズ
        const auto CountDataBlock = 512;
        const auto CountReservedBlock = CountDataBlock / 2;

        // ブロックサイズ
        nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam paramDuplex;
        nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
            paramIntegrity;

        const auto CountExpandMax = 10;

        const size_t sizeBlock = GenerateRandomBlockSize(512, MaxBlockSize);
        const size_t sizeBlock1 = GenerateRandomBlockSize(512, MaxBlockSize);
        const size_t sizeBlock2 = SetupBlockSize2ForExtraDataTest(
            sizeBlock, CountDataBlock, sizeBlock1
        );
        paramDuplex.sizeBlockLevel[0] = sizeBlock1 / 4;
        paramDuplex.sizeBlockLevel[1] = sizeBlock2;
        paramIntegrity.sizeBlockLevel[0] = sizeBlock1;
        paramIntegrity.sizeBlockLevel[1] = sizeBlock1;
        paramIntegrity.sizeBlockLevel[2] = sizeBlock2;
        paramIntegrity.sizeBlockLevel[3] = sizeBlock2;

        int64_t sizeTotal = 0;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
                &sizeTotal,
                sizeBlock,
                CountExpandMax,
                paramDuplex,
                paramIntegrity,
                static_cast<uint32_t>(CountDataBlock),
                static_cast<uint32_t>(CountReservedBlock)
            )
        );

        // ファイルシステムを配置する Storage を構築する
        storageData.Initialize(sizeTotal);

        storageBase = nn::fs::SubStorage(&storageData, 0, sizeTotal);

        nn::fs::SaveDataHashSalt salt;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::Format(
                nn::fs::SubStorage(&storageData, 0, sizeTotal),
                sizeBlock,
                CountExpandMax,
                paramDuplex,
                paramIntegrity,
                static_cast<uint32_t>(CountDataBlock),
                static_cast<uint32_t>(CountReservedBlock),
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                salt
            )
        );

        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Initialize(
                storageBase,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );

        // ファイル作成、書き込み、コミット
        {
            nn::fssystem::save::Path filePath;
            const auto FileSize = 2048;
            filePath.Initialize("/file");
            NNT_ASSERT_RESULT_SUCCESS(
                fileSystem.CreateFile(filePath, FileSize)
            );

            std::unique_ptr<char[]> fileBuffer(new char[FileSize]);
            nnt::fs::util::FillBufferWithRandomValue(fileBuffer.get(), FileSize);

            {
                nn::fssystem::save::IFile* pFile;
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.OpenFile(&pFile, filePath, nn::fs::OpenMode_Write)
                );
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->WriteBytes(0, fileBuffer.get(), FileSize)
                );
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->Flush()
                );
                fileSystem.CloseFile(pFile);
            }

            NNT_ASSERT_RESULT_SUCCESS(
                fileSystem.Commit()
            );
        }
    }

    // 書き込み、読み込み用バッファ作成
    nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData writeData0;
    nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData writeData1;
    nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData writeData2;
    nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData writeData3;
    nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData writeData4;
    nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData readData;

    nnt::fs::util::FillBufferWithRandomValue(&writeData0, 512);
    nnt::fs::util::FillBufferWithRandomValue(&writeData1, 512);
    nnt::fs::util::FillBufferWithRandomValue(&writeData2, 512);
    nnt::fs::util::FillBufferWithRandomValue(&writeData3, 512);
    nnt::fs::util::FillBufferWithRandomValue(&writeData4, 512);

    // 拡張データを書き込みコミットします
    {
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.WriteExtraData(writeData0)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Commit()
        );
    }

    // 拡張データを書き込み、仮コミット
    {
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.WriteExtraData(writeData1)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.CommitProvisionally(1)
        );
    }

    // 仮コミット状態でも書き込可、静的・動的読み込みも可能
    {
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.WriteExtraData(writeData2)
        );

        // 動的読み込み
        fileSystem.ReadExtraData(&readData);
        ASSERT_EQ(0, memcmp(&readData, &writeData2, 512));

        // 静的読み込み
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::ReadExtraData(
                &readData,
                storageBase,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );
        ASSERT_EQ(0, memcmp(&readData, &writeData0, 512));
    }

    // アンマウントして静的読み込み
    {
        fileSystem.Finalize();

        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::ReadExtraData(
                &readData,
                storageBase,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );
        ASSERT_EQ(0, memcmp(&readData, &writeData0, 512));
    }

    // マウントし、再マウント前と同様の状態であることを確認
    {
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Initialize(
                storageBase,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );

        // 書き込みは可能
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.WriteExtraData(writeData3)
        );

        // 動的読み込み
        fileSystem.ReadExtraData(&readData);
        ASSERT_EQ(0, memcmp(&readData, &writeData3, 512));

        // 静的読み込み
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::ReadExtraData(
                &readData,
                storageBase,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );
        ASSERT_EQ(0, memcmp(&readData, &writeData0, 512));
    }

    // コミットし、読み込み
    {
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Commit()
        );

        // 動的読み込み
        fileSystem.ReadExtraData(&readData);
        ASSERT_EQ(0, memcmp(&readData, &writeData3, 512));

        // 静的読み込み
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::ReadExtraData(
                &readData,
                storageBase,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );
        ASSERT_EQ(0, memcmp(&readData, &writeData3, 512));
    }

    // 書き込み、再マウントして静的・動的読み込みし、
    // 仮コミット状態が継続していないことを確認
    {
        // 書き込みは可能
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.WriteExtraData(writeData4)
        );

        fileSystem.Finalize();
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Initialize(
                storageBase,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );

        // 動的読み込み
        fileSystem.ReadExtraData(&readData);
        ASSERT_EQ(0, memcmp(&readData, &writeData3, 512));

        // 静的読み込み
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::ReadExtraData(
                &readData,
                storageBase,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );
        ASSERT_EQ(0, memcmp(&readData, &writeData3, 512));
    }
}// NOLINT(impl/function_size)

//! ロールバックを行い、拡張データが壊れないことを確かめます
TEST_F(FsJournalIntegritySaveDataFileSystemDriverTest, ExtraDataRollBack)
{
    nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver fileSystem;
    nnt::fs::util::AccessCountedMemoryStorage storageData;
    nn::fs::SubStorage storageBase;

    // 初期化
    {
        // データのサイズ
        const auto countDataBlock = 512;
        const auto countReservedBlock = countDataBlock / 2;

        // ブロックサイズ
        nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam paramDuplex;
        nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
            paramIntegrity;

        const auto CountExpandMax = 10;

        const size_t sizeBlock = GenerateRandomBlockSize(512, MaxBlockSize);
        const size_t sizeBlock1 = GenerateRandomBlockSize(512, MaxBlockSize);
        const size_t sizeBlock2 = SetupBlockSize2ForExtraDataTest(
            sizeBlock, countDataBlock, sizeBlock1
        );
        paramDuplex.sizeBlockLevel[0] = sizeBlock1 / 4;
        paramDuplex.sizeBlockLevel[1] = sizeBlock2;
        paramIntegrity.sizeBlockLevel[0] = sizeBlock1;
        paramIntegrity.sizeBlockLevel[1] = sizeBlock1;
        paramIntegrity.sizeBlockLevel[2] = sizeBlock2;
        paramIntegrity.sizeBlockLevel[3] = sizeBlock2;

        int64_t sizeTotal = 0;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
                &sizeTotal,
                sizeBlock,
                CountExpandMax,
                paramDuplex,
                paramIntegrity,
                static_cast<uint32_t>(countDataBlock),
                static_cast<uint32_t>(countReservedBlock)
            )
        );

        // ファイルシステムを配置する Storage を構築する
        storageData.Initialize(sizeTotal);

        storageBase = nn::fs::SubStorage(&storageData, 0, sizeTotal);

        nn::fs::SaveDataHashSalt salt;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::Format(
                nn::fs::SubStorage(&storageData, 0, sizeTotal),
                sizeBlock,
                CountExpandMax,
                paramDuplex,
                paramIntegrity,
                static_cast<uint32_t>(countDataBlock),
                static_cast<uint32_t>(countReservedBlock),
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                salt
            )
        );

        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Initialize(
                storageBase,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );

        // ファイル作成、書き込み、コミット
        {
            nn::fssystem::save::Path filePath;
            const auto FileSize = 2048;
            filePath.Initialize("/file");
            NNT_ASSERT_RESULT_SUCCESS(
                fileSystem.CreateFile(filePath, FileSize)
            );

            std::unique_ptr<char[]> fileBuffer(new char[FileSize]);
            nnt::fs::util::FillBufferWithRandomValue(fileBuffer.get(), FileSize);

            {
                nn::fssystem::save::IFile* pFile;
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.OpenFile(&pFile, filePath, nn::fs::OpenMode_Write)
                );
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->WriteBytes(0, fileBuffer.get(), FileSize)
                );
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->Flush()
                );
                fileSystem.CloseFile(pFile);
            }

            NNT_ASSERT_RESULT_SUCCESS(
                fileSystem.Commit()
            );
        }
    }

    // 書き込み、読み込み用バッファ作成
    nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData writeData0;
    nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData writeData1;
    nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData writeData2;
    nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData readData;

    nnt::fs::util::FillBufferWithRandomValue(&writeData0, 512);
    nnt::fs::util::FillBufferWithRandomValue(&writeData1, 512);
    nnt::fs::util::FillBufferWithRandomValue(&writeData2, 512);

    // 拡張データを書き込みコミットします
    {
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.WriteExtraData(writeData0)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Commit()
        );
    }

    // 拡張データを書き込み、仮コミット、コミット
    {
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.WriteExtraData(writeData0)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.CommitProvisionally(1)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Commit()
        );
    }

    // 拡張データを書き込み、ロールバック
    {
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.WriteExtraData(writeData0)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Rollback()
        );
    }

    // 仮コミット、ロールバック
    {
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.CommitProvisionally(1)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Rollback()
        );
    }

    // 拡張データを書き込み、仮コミット、ロールバック
    {
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.WriteExtraData(writeData0)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.CommitProvisionally(1)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Rollback()
        );
    }

    // 仮コミット状態が解除されているので書き込み可能
    {
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.WriteExtraData(writeData1)
        );
    }

    // アンマウントして静的読み込み
    {
        fileSystem.Finalize();

        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::ReadExtraData(
                &readData,
                storageBase,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );
        ASSERT_EQ(0, memcmp(&readData, &writeData0, 512));
    }

    // マウントして静的・動的読み込み
    {
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Initialize(
                storageBase,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );

        // 動的読み込み
        fileSystem.ReadExtraData(&readData);
        ASSERT_EQ(0, memcmp(&readData, &writeData0, 512));

        // 静的読み込み
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::ReadExtraData(
                &readData,
                storageBase,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );
        ASSERT_EQ(0, memcmp(&readData, &writeData0, 512));
    }

    // 書き込み、仮コミットせずにロールバック
    {
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.WriteExtraData(writeData2)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Rollback()
        );
    }

    // 静的・動的読み込み
    {
        // 動的読み込み
        fileSystem.ReadExtraData(&readData);
        ASSERT_EQ(0, memcmp(&readData, &writeData0, 512));

        // 静的読み込み
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::ReadExtraData(
                &readData,
                storageBase,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );
        ASSERT_EQ(0, memcmp(&readData, &writeData0, 512));
    }
}// NOLINT(impl/function_size)

TEST_F(FsJournalIntegritySaveDataFileSystemDriverTest, RandomAllocationFailureHeavy)
{
    typedef nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver SaveDataFileSystemDriver;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    MeasurableBufferManager<false> measurableBufferManager(GetBufferManager());
    nnt::fs::util::RandomFailureBufferManager randomFailureBufferManager(&measurableBufferManager, &mt);

    static const auto LoopCount = 5;
    static const auto InnerLoopCount = 50;
    static const auto CountExpandMax = 25;

    // Allocation Count の実測値の * 4 / 3 ぐらいにする
    // サイズ変動系は小さめの値にして問題ない (リトライ時に * 2 される)
    static const auto RandomDividerFormat = 1000;
    static const auto RandomDividerOperateExpand = 1400;
    static const auto RandomDividerCommitExpand = 25;
    static const auto RandomDividerMount = 25;
    static const auto RandomDividerUnmount = 350;
    static const auto RandomDividerCommit = 25;
    static const auto RandomDividerRollback = 15;
    static const auto RandomDividerFileSystem = 50;
    static const auto RandomDividerFile = 5;

    auto doTest = [&](const char* name, int divider, std::function<void ()> action) NN_NOEXCEPT
    {
        randomFailureBufferManager.SetDivider(divider);
        NNT_FS_ASSERT_NO_FATAL_FAILURE(action());
        ASSERT_EQ(nullptr, nn::fssystem::buffers::GetBufferManagerContext());
        NN_LOG("Allocation Count %s %d\n", name, randomFailureBufferManager.GetAllocationCount());
    };

    const size_t sizeFreeSizeStart = GetBufferManager()->GetFreeSize();
    for( size_t sizeBlock = 4096; sizeBlock <= MaxBlockSize; sizeBlock *= 2 )
    {
        NN_LOG("block size: %d/%d\n", static_cast<int>(sizeBlock), static_cast<int>(MaxBlockSize));

        for( auto loop = 0; loop < LoopCount; ++loop )
        {
            NN_LOG("loop count: %d/%d\n", loop + 1, LoopCount);

            nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam paramDuplex;
            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
                paramIntegrity;
            const size_t sizeBlock1 = GenerateRandomBlockSize(512, MaxBlockSize);
            const size_t sizeBlock2
                = GenerateRandomBlockSize(512 < sizeBlock1 ? 512 : 1024, MaxBlockSize);
            paramDuplex.sizeBlockLevel[0] = sizeBlock1 / 4;
            paramDuplex.sizeBlockLevel[1] = sizeBlock2;
            paramIntegrity.sizeBlockLevel[0] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[1] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[2] = sizeBlock2;
            paramIntegrity.sizeBlockLevel[3] = sizeBlock2;

            const auto countDataBlock = std::uniform_int_distribution<>(100, 120)(mt);
            const auto countReservedBlock = countDataBlock;
            const auto sizeFile = countDataBlock * sizeBlock * 4 / 5;
            const auto sizeBuffer = sizeFile * 3 / 4;

            int64_t sizeTotal = 0;
            NNT_ASSERT_RESULT_SUCCESS(
                SaveDataFileSystemDriver::QueryTotalSize(
                    &sizeTotal,
                    sizeBlock,
                    CountExpandMax,
                    paramDuplex,
                    paramIntegrity,
                    countDataBlock,
                    countReservedBlock
                )
            );
            ASSERT_EQ(sizeFreeSizeStart, GetBufferManager()->GetFreeSize());

            nnt::fs::util::SafeMemoryStorage storageData(sizeTotal);
            nn::fs::SubStorage storageBase(&storageData, 0, sizeTotal);

            NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                "Format", RandomDividerFormat,
                [&]() NN_NOEXCEPT
                {
                    nn::fs::SaveDataHashSalt salt;
                    NNT_ASSERT_RESULT_SUCCESS(
                        SaveDataFileSystemDriver::Format(
                            storageBase,
                            sizeBlock,
                            CountExpandMax,
                            paramDuplex,
                            paramIntegrity,
                            countDataBlock / 2,
                            countReservedBlock / 2,
                            &randomFailureBufferManager,
                            nnt::fs::util::GetMacGenerator(),
                            salt
                        )
                    );
                }
            ));
            ASSERT_EQ(sizeFreeSizeStart, GetBufferManager()->GetFreeSize());

            NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                "ReadExtraData", RandomDividerFormat,
                [&]() NN_NOEXCEPT
                {
                    nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData extraData;
                    NNT_ASSERT_RESULT_SUCCESS(
                        SaveDataFileSystemDriver::ReadExtraData(
                            &extraData,
                            storageBase,
                            &randomFailureBufferManager,
                            nnt::fs::util::GetMacGenerator(),
                            nnt::fs::util::SaveDataMinimumVersion
                        )
                    );
                }
            ));
            ASSERT_EQ(sizeFreeSizeStart, GetBufferManager()->GetFreeSize());

            const auto sizeLogStorage = SaveDataFileSystemDriver::QueryExpandLogSize(
                sizeBlock,
                countDataBlock,
                countReservedBlock
            );
            nnt::fs::util::SafeMemoryStorage storageLog(sizeLogStorage);
            ASSERT_EQ(sizeFreeSizeStart, GetBufferManager()->GetFreeSize());

            NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                "OperateExpand", RandomDividerOperateExpand,
                [&]() NN_NOEXCEPT
                {
                    NNT_ASSERT_RESULT_SUCCESS(
                        SaveDataFileSystemDriver::OperateExpand(
                            storageBase,
                            nn::fs::SubStorage(&storageLog, 0, sizeLogStorage),
                            sizeBlock,
                            countDataBlock,
                            countReservedBlock,
                            &randomFailureBufferManager,
                            nnt::fs::util::GetMacGenerator(),
                            nnt::fs::util::SaveDataMinimumVersion
                        )
                    );
                }
            ));
            ASSERT_EQ(sizeFreeSizeStart, GetBufferManager()->GetFreeSize());

            NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                "CommitExpand", RandomDividerCommitExpand,
                [&]() NN_NOEXCEPT
                {
                    NNT_ASSERT_RESULT_SUCCESS(
                        SaveDataFileSystemDriver::CommitExpand(
                            storageBase,
                            nn::fs::SubStorage(&storageLog, 0, sizeLogStorage),
                            sizeBlock,
                            &randomFailureBufferManager
                        )
                    );
                }
            ));
            ASSERT_EQ(sizeFreeSizeStart, GetBufferManager()->GetFreeSize());

            // ファイルを読み書きして Rollback する
            {
                SaveDataFileSystemDriver fileSystem;

                randomFailureBufferManager.SetDivider(RandomDividerMount);
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.Initialize(
                        storageBase,
                        &randomFailureBufferManager,
                        nnt::fs::util::GetMacGenerator(),
                        nnt::fs::util::SaveDataMinimumVersion
                    )
                );
                ASSERT_EQ(nullptr, nn::fssystem::buffers::GetBufferManagerContext());
                NN_LOG("Allocate Count Mount %d\n", randomFailureBufferManager.GetAllocationCount());

                nn::fssystem::save::Path pathFile;
                NNT_ASSERT_RESULT_SUCCESS(pathFile.Initialize("/directory/file.dat"));

                nn::fssystem::save::Path pathDirectory;
                NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize("/directory"));

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "CreateDirectory", RandomDividerFileSystem,
                    [&]() NN_NOEXCEPT
                    {
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory(pathDirectory));
                    }
                ));

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "CreateFile", RandomDividerFileSystem,
                    [&]() NN_NOEXCEPT
                    {
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile(pathFile, 1));
                    }
                ));

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "Commit", RandomDividerCommit,
                    [&]() NN_NOEXCEPT
                    {
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.Commit());
                    }
                ));

                // ファイルアクセス
                {
                    nn::fssystem::save::IFile* pFile = nullptr;

                    NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                        "OpenFile", RandomDividerFile,
                        [&]() NN_NOEXCEPT
                        {
                            const auto openMode = nn::fs::OpenMode_Read | nn::fs::OpenMode_Write;
                            NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&pFile, pathFile, openMode));
                        }
                    ));

                    NN_UTIL_SCOPE_EXIT
                    {
                        NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                            "CloseFile", 0,
                            [&]() NN_NOEXCEPT
                            {
                                fileSystem.CloseFile(pFile);
                            }
                        ));
                    };

                    // サイズの取得と設定
                    {
                        int64_t sizeCurrent;
                        NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                            "File::GetSize", RandomDividerFile,
                            [&]() NN_NOEXCEPT
                            {
                                NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&sizeCurrent));
                                ASSERT_EQ(1, sizeCurrent);
                            }
                        ));

                        NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                            "File::SetSize", RandomDividerFile,
                            [&]() NN_NOEXCEPT
                            {
                                NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(sizeFile));
                                ASSERT_EQ(1, sizeCurrent);
                            }
                        ));
                    }

                    NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                        "File::OperateRange", 2,
                        [&]() NN_NOEXCEPT
                        {
                            char inBuffer[32];
                            char outBuffer[32];

                            NNT_ASSERT_RESULT_FAILURE(
                                nn::fs::ResultUnsupportedOperation,
                                pFile->OperateRange(
                                    outBuffer, sizeof(outBuffer),
                                    nn::fs::OperationId::Invalidate,
                                    0, sizeof(inBuffer),
                                    inBuffer, sizeof(inBuffer)
                                )
                            );
                        }
                    ));

                    nnt::fs::util::Vector<char> readBuffer(sizeBuffer, 0);
                    nnt::fs::util::Vector<char> writeBuffer(sizeBuffer);
                    std::iota(writeBuffer.begin(), writeBuffer.end(), '\x0');

                    for( auto innerLoop = 0; innerLoop < InnerLoopCount; ++innerLoop )
                    {
                        const auto offsetMax = static_cast<int64_t>(sizeFile - sizeBuffer);
                        const auto offset = std::uniform_int_distribution<int64_t>(0, offsetMax)(mt);

                        NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                            "File::WriteBytes", RandomDividerFile,
                            [&]() NN_NOEXCEPT
                            {
                                NNT_ASSERT_RESULT_SUCCESS(
                                    pFile->WriteBytes(
                                        offset,
                                        writeBuffer.data(),
                                        writeBuffer.size()
                                    )
                                );
                            }
                        ));

                        NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                            "File::ReadBytes", RandomDividerFile,
                            [&]() NN_NOEXCEPT
                            {
                                NNT_ASSERT_RESULT_SUCCESS(
                                    pFile->ReadBytes(
                                        offset,
                                        readBuffer.data(),
                                        readBuffer.size()
                                    )
                                );
                            }
                        ));

                        NNT_FS_UTIL_EXPECT_MEMCMPEQ(
                            readBuffer.data(), writeBuffer.data(), readBuffer.size()
                        );

                        NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                            "File::Flush", RandomDividerFile,
                            [&]() NN_NOEXCEPT
                            {
                                NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());
                            }
                        ));
                    }
                }

                // ディレクトリアクセス
                {
                    nn::fssystem::save::IDirectory* pDirectory = nullptr;

                    NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                        "OpenDirectory", RandomDividerFile,
                        [&]() NN_NOEXCEPT
                        {
                            NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenDirectory(&pDirectory, pathDirectory));
                        }
                    ));

                    NN_UTIL_SCOPE_EXIT
                    {
                        NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                            "CloseDirectory", 0,
                            [&]() NN_NOEXCEPT
                            {
                                fileSystem.CloseDirectory(pDirectory);
                            }
                        ));
                    };

                    NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                        "Directory::Read", RandomDividerFile,
                        [&]() NN_NOEXCEPT
                        {
                            nn::fs::DirectoryEntry entry;
                            int currentCount;
                            NNT_ASSERT_RESULT_SUCCESS(pDirectory->Read(&currentCount, &entry, 1));
                        }
                    ));
                }

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "Rollback", RandomDividerRollback,
                    [&]() NN_NOEXCEPT
                    {
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.Rollback());
                    }
                ));

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "Unmount without Buffer Allocation", RandomDividerUnmount,
                    [&]() NN_NOEXCEPT
                    {
                        fileSystem.Finalize();
                    }
                ));
            }

            ASSERT_EQ(sizeFreeSizeStart, GetBufferManager()->GetFreeSize());

            // アンマウントでバッファ確保するケース
            {
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver fileSystem;

                randomFailureBufferManager.SetDivider(RandomDividerMount);
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.Initialize(
                        storageBase,
                        &randomFailureBufferManager,
                        nnt::fs::util::GetMacGenerator(),
                        nnt::fs::util::SaveDataMinimumVersion
                    )
                );

                nn::fssystem::save::Path pathFile;
                NNT_ASSERT_RESULT_SUCCESS(pathFile.Initialize("/directory/file2.dat"));

                randomFailureBufferManager.SetDivider(RandomDividerFileSystem);
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile(pathFile, 0));

                randomFailureBufferManager.SetDivider(RandomDividerRollback);
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.Rollback());

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "Unmount with Buffer Allocation", RandomDividerUnmount,
                    [&]() NN_NOEXCEPT
                    {
                        fileSystem.Finalize();
                    }
                ));
            }

            ASSERT_EQ(sizeFreeSizeStart, GetBufferManager()->GetFreeSize());

            // ファイルの削除やリネーム
            {
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver fileSystem;

                randomFailureBufferManager.SetDivider(RandomDividerMount);
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.Initialize(
                        storageBase,
                        &randomFailureBufferManager,
                        nnt::fs::util::GetMacGenerator(),
                        nnt::fs::util::SaveDataMinimumVersion
                    )
                );

                nn::fssystem::save::Path pathFile;
                NNT_ASSERT_RESULT_SUCCESS(pathFile.Initialize("/directory/file.dat"));

                nn::fssystem::save::Path pathFile2;
                NNT_ASSERT_RESULT_SUCCESS(pathFile2.Initialize("/directory/file3.dat"));

                nn::fssystem::save::Path pathDirectory;
                NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize("/directory/directory"));

                nn::fssystem::save::Path pathDirectory2;
                NNT_ASSERT_RESULT_SUCCESS(pathDirectory2.Initialize("/directory/directory2"));

                randomFailureBufferManager.SetDivider(RandomDividerFileSystem);
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory(pathDirectory));

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "GetFreeBytes", RandomDividerFileSystem,
                    [&]() NN_NOEXCEPT
                    {
                        int64_t size;
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.GetFreeBytes(&size));
                    }
                ));

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "GetDataAreaBytes", RandomDividerFileSystem,
                    [&]() NN_NOEXCEPT
                    {
                        int64_t size;
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.GetDataAreaBytes(&size));
                    }
                ));

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "HasFile", RandomDividerFileSystem,
                    [&]() NN_NOEXCEPT
                    {
                        bool isExists;
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.HasFile(&isExists, pathFile));
                        ASSERT_TRUE(isExists);
                    }
                ));

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "HasDirectory", RandomDividerFileSystem,
                    [&]() NN_NOEXCEPT
                    {
                        bool isExists;
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.HasDirectory(&isExists, pathDirectory));
                        ASSERT_TRUE(isExists);
                    }
                ));

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "RenameFile", RandomDividerFileSystem,
                    [&]() NN_NOEXCEPT
                    {
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.RenameFile(pathFile, pathFile2));
                    }
                ));

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "RenameDirectory", RandomDividerFileSystem,
                    [&]() NN_NOEXCEPT
                    {
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.RenameDirectory(pathDirectory, pathDirectory2));
                    }
                ));

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "DeleteFile", RandomDividerFileSystem,
                    [&]() NN_NOEXCEPT
                    {
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.DeleteFile(pathFile2));
                    }
                ));

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "DeleteDirectory", RandomDividerFileSystem,
                    [&]() NN_NOEXCEPT
                    {
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.DeleteDirectory(pathDirectory2));
                    }
                ));

                NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize("/directory"));
                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "DeleteDirectoryRecursively", RandomDividerFileSystem,
                    [&]() NN_NOEXCEPT
                    {
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.DeleteDirectoryRecursively(pathDirectory));
                    }
                ));

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "CommitProvisionally", RandomDividerCommit,
                    [&]() NN_NOEXCEPT
                    {
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.CommitProvisionally(1));
                    }
                ));

                randomFailureBufferManager.SetDivider(RandomDividerMount);
                fileSystem.Finalize();
            }

            ASSERT_EQ(sizeFreeSizeStart, GetBufferManager()->GetFreeSize());
        }
    }

    ASSERT_EQ(sizeFreeSizeStart, GetBufferManager()->GetFreeSize());
} // NOLINT(impl/function_size)

/**
* JournalIntegritySaveDataFileSystemDriver の電源断テストを行います。
* 特定回数の書き込みを行うと更新できなくなるストレージを使うことで、
* 電源断からの復旧及び復旧後の挙動を再現させます。
* 電源断発生処理は基本的にマルチスレッドで行います。
* 復旧後のバリデーションはシングルスレッドで行います。
*/
typedef void(*ThreadFunction)(void*);
namespace
{
    static const auto MaxThreadCount = 8;
    static const auto StackSize = 32 * 1024;
    NN_OS_ALIGNAS_THREAD_STACK static char g_ThreadStack[StackSize * MaxThreadCount];
}

struct PowerInterruptionTestParam
{
    size_t blockSize;
    uint32_t dataBlockCount;
    int maxExpandCount;
    size_t duplexBlockLevel1;
    size_t duplexBlockLevel2;
    size_t integrityBlockLevel1;
    size_t integrityBlockLevel2;
    size_t integrityBlockLevel3;
    size_t integrityBlockLevel4;
    int threadCount;
};

// テスト用のパラメータを設定
nnt::fs::util::Vector<PowerInterruptionTestParam> MakePowerInterruptionTestParam()
{
    nnt::fs::util::Vector<PowerInterruptionTestParam> params;
    {
        PowerInterruptionTestParam param;
        param.blockSize = 1024;
        param.dataBlockCount = 512;
        param.maxExpandCount = 25;
        param.duplexBlockLevel1 = 1024;
        param.duplexBlockLevel2 = 1024;
        param.integrityBlockLevel1 = 1024;
        param.integrityBlockLevel2 = 1024;
        param.integrityBlockLevel3 = 1024;
        param.integrityBlockLevel4 = 1024;
        param.threadCount = 8;
        params.push_back(param);
    }
    {
        PowerInterruptionTestParam param;
        param.blockSize = 4096;
        param.dataBlockCount = 512;
        param.maxExpandCount = 25;
        param.duplexBlockLevel1 = 4 * 1024;
        param.duplexBlockLevel2 = 4 * 1024;
        param.integrityBlockLevel1 = 1024;
        param.integrityBlockLevel2 = 1024;
        param.integrityBlockLevel3 = 4 * 1024;
        param.integrityBlockLevel4 = 16 * 1024;
        param.threadCount = 8;
        params.push_back(param);
    }
    {
        PowerInterruptionTestParam param;
        param.blockSize = 1024;
        param.dataBlockCount = 2048;
        param.maxExpandCount = 25;
        param.duplexBlockLevel1 = 512;
        param.duplexBlockLevel2 = 512;
        param.integrityBlockLevel1 = 1024;
        param.integrityBlockLevel2 = 8 * 1024;
        param.integrityBlockLevel3 = 4 * 1024;
        param.integrityBlockLevel4 = 2 * 1024;
        param.threadCount = 8;
        params.push_back(param);
    }

    return params;
}

/**
* @brief    電源断テストクラス
*
* @details
*   電源断後の復旧時の挙動をテストします。
*   ディレクトリやファイルを操作するたびにディレクトリ構成等を記録し、
*   電源断後のバリデートで記録と相違ないかチェックします。
*/
class FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTest
    : public ::testing::TestWithParam<PowerInterruptionTestParam>
{
public:
    //! ディレクトリやファイルの状態を表す列挙型
    enum class DataState
    {
        Creating,   //!< Create を呼び出す前
        Created,    //!< Create が Success、Commit を呼び出す前
        Exist,      //!< Commit が Success
        Deleting,   //!< Delete を呼び出す前
        Deleted,    //!< Delete が Success、Commit を呼び出す前
        Empty,      //!< Commit が Success
        ReWriting,  //!< Write を呼び出す前
        ReWrited    //!< Write が Success、Commit を呼び出す前
    };

    /**
    * @brief    マルチスレッド関数用引数となる構造体
    *
    * @details
    *   マルチスレッドで実行する関数にこのクラスのインスタンスを渡すための引数となる構造体です
    */
    struct ThreadArgument
    {
        int threadId;
        int randomSeed;
        FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTest* pFileSystemTest;
        bool CheckInterruption() const NN_NOEXCEPT
        {
            return pFileSystemTest->CheckInterruption();
        }
    };

    /**
    * @brief    パスデータ
    *
    * @details  バリデーション用の記録を参照する際のキーとなります。
    *           isDirectory=true, directoryId={2,3} の場合、/dir2/dir3 というディレクトリを表します
    *           depth <= 2 を前提に実装しています。
    */
    struct PathData
    {
        bool isDirectory;           //!< ディレクトリかファイルか
        int depth;                  //!< directoryId の長さ
        int directoryId[2];         //!< パスを求めるための値

        //! パスを求めます
        void GetPath(nn::fssystem::save::Path* outPath) const NN_NOEXCEPT
        {
            std::ostringstream stream;
            for( int i = 0; i < this->depth - 1; ++i )
            {
                stream << "/dir" << this->directoryId[i];
            }
            stream << (this->isDirectory ? "/dir" : "/file") << this->directoryId[this->depth - 1];
            outPath->Initialize(stream.str().c_str());
        }

        //! 記録からデータを参照するために必要です
        bool operator < (const PathData& rhs) const NN_NOEXCEPT
        {
            if( this->isDirectory != rhs.isDirectory )
            {
                return this->isDirectory;
            }
            if( this->depth != rhs.depth )
            {
                return this->depth < rhs.depth;
            }
            for( int i = 0; i < this->depth; ++i )
            {
                if( this->directoryId[i] != rhs.directoryId[i] )
                {
                    return this->directoryId[i] < rhs.directoryId[i];
                }
            }
            return false;
        }
    };

    /**
    * @brief    ファイルデータ
    *
    * @details  バリデーションのためのファイルのデータです
    */
    struct FileData
    {
        size_t size;        //!< ファイルサイズ
        int dataIndex;      //!< データ書き込みの際の std::iota の引数
        int oldDataIndex;   //!< データ書き変えをする場合の書き換え前の dataIndex
        int threadId;       //!< 最後にこのファイルを操作したスレッド
        DataState state;    //!< ファイルの状態

        /**
        * @brief    ファイルが持っているべきデータを入れたバッファを返します。
        *
        * @details
        *       dataIndex == 8 とすると、"89ABCD..."という値が入ったバッファを返します
        *       dataIndex == -1 の場合は 0 フィルされたバッファを返します
        */
        std::unique_ptr<char[]> GetExpectedFileData() const NN_NOEXCEPT
        {
            std::unique_ptr<char[]> buffer(new char[size]);
            if( dataIndex != -1 )
            {
                std::iota(buffer.get(), buffer.get() + size, static_cast<char>(dataIndex));
            }
            else
            {
                memset(buffer.get(), 0, size);
            }
            return buffer;
        }
    };

    /**
    * @brief    ディレクトリデータ
    *
    * @details  バリデーションのためのディレクトリのデータです
    */
    struct DirectoryData
    {
        int threadId;       // 最後にこのディレクトリを操作したスレッド
        DataState state;    // ファイルの状態
    };

public:
    static const int MaxWritableCount = INT_MAX;

public:
    //! コンストラクタ
    FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTest() NN_NOEXCEPT
      : m_Locker(true)
    {
    }

    //! マルチスレッド用のスレッド準備
    virtual void SetUp() NN_OVERRIDE
    {
        memset(g_ThreadStack, 0, sizeof(g_ThreadStack));
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        // このテストはテスト中に確保したバッファーの解放が保証されない
        FinalizeBufferManager();
    }

    /**
    * @brief        シングルスレッドでフォーマットします。
    *
    * @param[in]    param   テストパラメータ
    *
    * @details      シングルスレッドでフォーマットします。
    */
    void Format(const PowerInterruptionTestParam& param) NN_NOEXCEPT
    {
        NNT_ASSERT_RESULT_SUCCESS(
            Format(param, MaxWritableCount)
        );
    }

    /**
    * @brief        シングルスレッドでフォーマットします。
    *
    * @param[in]    param           テストパラメータ
    * @param[in]    writableCount   書き込み可能回数
    *
    * @return       初期化に失敗(初期化中に電源断)すると false が返ります。
    *
    * @details      シングルスレッドで呼び出すことを想定しています。
    *               writableCount を減らせば失敗します
    */
    nn::Result Format(const PowerInterruptionTestParam& param, int writableCount) NN_NOEXCEPT
    {
        m_IsMounted = false;

        // 内部変数を初期化
        m_FileMap.clear();
        m_DirectoryMap.clear();
        for( int threadId = 0; threadId < param.threadCount; ++threadId )
        {
            m_Argument[threadId].threadId = threadId;
            m_Argument[threadId].pFileSystemTest = this;
        }

        // SaveDataFileSystem 用にストレージをフォーマット
        nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam paramDuplex;
        nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam paramIntegrity;
        paramDuplex.sizeBlockLevel[0] = param.duplexBlockLevel1;
        paramDuplex.sizeBlockLevel[1] = param.duplexBlockLevel2;
        paramIntegrity.sizeBlockLevel[0] = param.integrityBlockLevel1;
        paramIntegrity.sizeBlockLevel[1] = param.integrityBlockLevel2;
        paramIntegrity.sizeBlockLevel[2] = param.integrityBlockLevel3;
        paramIntegrity.sizeBlockLevel[3] = param.integrityBlockLevel4;
        int64_t sizeTotal = 0;
        NN_RESULT_DO(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
                &sizeTotal,
                param.blockSize,
                param.maxExpandCount,
                paramDuplex,
                paramIntegrity,
                param.dataBlockCount,
                param.dataBlockCount
            )
        );
        m_BaseStorage.Initialize(static_cast<size_t>(sizeTotal));
        m_Storage.Initialize(
            nn::fs::SubStorage(&m_BaseStorage, 0, sizeTotal)
        );
        m_Storage.StartCounting(writableCount);
        nn::fs::SaveDataHashSalt salt;
        NN_RESULT_DO(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::Format(
                nn::fs::SubStorage(&m_Storage, 0, sizeTotal),
                param.blockSize,
                param.maxExpandCount,
                paramDuplex,
                paramIntegrity,
                param.dataBlockCount,
                param.dataBlockCount,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                salt
            )
        );
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief    マウントします。
    *
    * @details  シングルスレッドで呼び出すことを想定しています。
    *           マウントに失敗すると false が返ります。
    */
    bool Mount() NN_NOEXCEPT
    {
        int64_t storageSize = 0;
        m_Storage.GetSize(&storageSize);
        nn::Result result = m_SaveDataFileSystem.Initialize(
                                nn::fs::SubStorage(&m_Storage, 0, storageSize),
                                GetBufferManager(),
                                nnt::fs::util::GetMacGenerator(),
                                nnt::fs::util::SaveDataMinimumVersion
                            );
        m_IsMounted = result.IsSuccess();
        return m_IsMounted;
    }

    /**
    * @brief    アンマウントします。
    *
    * @details  シングルスレッドで呼び出すことを想定しています。
    *           複数回マウントする必要がある場合は、必ずアンマウントしてください。
    */
    void Unmount() NN_NOEXCEPT
    {
        m_SaveDataFileSystem.Finalize();
        m_IsMounted = false;
    }

    /**
    * @brief        マルチスレッドで関数を実行します。
    *
    * @param[in]    function    マルチスレッドで実行する関数
    *
    * @details      引数に渡された関数をマルチスレッドで実行します。
    */
    void ExecuteMultiThreadFunction(ThreadFunction function) NN_NOEXCEPT
    {
        const PowerInterruptionTestParam& param = GetParam();

        // param.threadCount != MaxThreadCount の可能性があるので、範囲 for 文は不可
        for( int threadId = 0; threadId < param.threadCount; ++threadId )
        {
            NNT_ASSERT_RESULT_SUCCESS(
                nn::os::CreateThread(
                    &m_Threads[threadId],
                    function,
                    m_Argument + threadId,
                    g_ThreadStack + StackSize * threadId,
                    StackSize,
                    nn::os::DefaultThreadPriority
                )
            );
            nn::os::StartThread(&m_Threads[threadId]);
        }
        for( int threadId = 0; threadId < param.threadCount; ++threadId )
        {
            nn::os::WaitThread(&m_Threads[threadId]);
            nn::os::DestroyThread(&m_Threads[threadId]);
        }
    }

    /**
    * @brief        ディレクトリ作成
    *
    * @param[in]    depth           ディレクトリの深さ(directoryId の個数)
    * @param[in]    pDirectoryId    パスを求めるための ID
    * @param[in]    threadId        スレッドID、記録のために必要
    *
    * @return       関数の実行結果を返します
    *
    * @pre          1 <= depth <= 2
    *
    * @details
    *   depth=2、directoryId={1,2} のとき、/dir1/dir2 というディレクトリを作成します。
    */
    nn::Result CreateDirectory(int depth, const int* pDirectoryId, int threadId) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsMounted);
        NN_SDK_REQUIRES_LESS_EQUAL(1, depth);
        NN_SDK_REQUIRES_LESS_EQUAL(depth, 2);

        std::lock_guard<nn::os::Mutex> locker(m_Locker);
        if( CheckInterruption() )
        {
            return nnt::fs::util::PowerInterruptionStorage::ResultPowerInterruptionOccurred();
        }
        RefreshMapData(threadId);

        // ディレクトリ作成中である旨を記録
        PathData pathData;
        {
            pathData.isDirectory = true;
            pathData.depth = depth;
            for( int i = 0; i < depth; ++i )
            {
                pathData.directoryId[i] = pDirectoryId[i];
            }
        }

        // ディレクトリ作成
        StartCreateDirectory(pathData, threadId);
        {
            nn::fssystem::save::Path path;
            pathData.GetPath(&path);
            NN_RESULT_DO(m_SaveDataFileSystem.CreateDirectory(path));
        }
        EndCreateDirectory(pathData);
        NN_RESULT_DO(m_SaveDataFileSystem.Commit());
        RefreshMapData(threadId);
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        ディレクトリ削除
    *
    * @param[in]    depth           ディレクトリの深さ(directoryId の個数)
    * @param[in]    pDirectoryId    パスを求めるための ID
    * @param[in]    threadId        スレッドID、記録のために必要
    *
    * @return       関数の実行結果を返します
    *
    * @pre          1 <= depah <= 2
    *
    * @details
    *   depth=2、directoryId={1,2} のとき、/dir1/dir2 というディレクトリを削除します。
    */
    nn::Result DeleteDirectory(
            int depth,
            const int* pDirectoryId,
            int threadId
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsMounted);
        NN_SDK_REQUIRES_LESS_EQUAL(1, depth);
        NN_SDK_REQUIRES_LESS_EQUAL(depth, 2);

        std::lock_guard<nn::os::Mutex> locker(m_Locker);
        if( CheckInterruption() )
        {
            return nnt::fs::util::PowerInterruptionStorage::ResultPowerInterruptionOccurred();
        }
        RefreshMapData(threadId);

        // 削除するディレクトリのデータを取得
        PathData pathData;
        {
            pathData.isDirectory = true;
            pathData.depth = depth;
            for( int i = 0; i < depth; ++i )
            {
                pathData.directoryId[i] = pDirectoryId[i];
            }
        }

        // ディレクトリ削除
        StartDeleteDirectory(pathData, threadId);
        {
            nn::fssystem::save::Path path;
            pathData.GetPath(&path);
            NN_RESULT_DO(m_SaveDataFileSystem.DeleteDirectory(path));
        }
        EndDeleteDirectory(pathData);
        NN_RESULT_DO(m_SaveDataFileSystem.Commit());
        RefreshMapData(threadId);
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        ディレクトリリネーム
    *
    * @param[in]    srcDepth            ディレクトリの深さ(directoryId の個数)
    * @param[in]    pSrcDirectoryId     パスを求めるための ID
    * @param[in]    dstDepth            ディレクトリの深さ(directoryId の個数)
    * @param[in]    pDstDirectoryId     パスを求めるための ID
    * @param[in]    threadId            スレッドID、記録のために必要
    *
    * @return       関数の実行結果を返します
    *
    * @pre          1 <= srcDepth <= 2
    * @pre          1 <= dstDepth <= 2
    *
    * @details
    *   srcDepth=2、pSrcDirectoryId={1,2}、dstDepth=2、pDstDirectoryId={1,4} のとき、
    *   /dir1/dir2 というディレクトリを /dir1/dir4 にリネームします。
    */
    nn::Result RenameDirectory(
            int srcDepth,
            const int* pSrcDirectoryId,
            int dstDepth,
            const int* pDstDirectoryId,
            int threadId
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsMounted);
        NN_SDK_REQUIRES_LESS_EQUAL(1, srcDepth);
        NN_SDK_REQUIRES_LESS_EQUAL(srcDepth, 2);
        NN_SDK_REQUIRES_LESS_EQUAL(1, dstDepth);
        NN_SDK_REQUIRES_LESS_EQUAL(dstDepth, 2);

        std::lock_guard<nn::os::Mutex> locker(m_Locker);
        if( CheckInterruption() )
        {
            return nnt::fs::util::PowerInterruptionStorage::ResultPowerInterruptionOccurred();
        }
        RefreshMapData(threadId);

        // データ作成
        PathData srcPathData;
        PathData dstPathData;
        {
            srcPathData.isDirectory = true;
            dstPathData.isDirectory = true;
            srcPathData.depth = srcDepth;
            dstPathData.depth = dstDepth;
            for( int i = 0; i < srcDepth; ++i )
            {
                srcPathData.directoryId[i] = pSrcDirectoryId[i];
            }
            for( int i = 0; i < dstDepth; ++i )
            {
                dstPathData.directoryId[i] = pDstDirectoryId[i];
            }
        }

        // リネームする
        StartDeleteDirectory(srcPathData, threadId);
        StartCreateDirectory(dstPathData, threadId);
        {
            nn::fssystem::save::Path srcPath;
            nn::fssystem::save::Path dstPath;
            srcPathData.GetPath(&srcPath);
            dstPathData.GetPath(&dstPath);
            NN_RESULT_DO(m_SaveDataFileSystem.RenameDirectory(srcPath, dstPath));
        }
        EndDeleteDirectory(srcPathData);
        EndCreateDirectory(dstPathData);
        NN_RESULT_DO(m_SaveDataFileSystem.Commit());
        RefreshMapData(threadId);
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        ファイル作成
    *
    * @param[in]    depth           ディレクトリの深さ(directoryId の個数)
    * @param[in]    pDirectoryId    パスを求めるための ID
    * @param[in]    threadId        スレッドID、記録のために必要
    * @param[in]    size            ファイルサイズ
    *
    * @return       関数の実行結果を返します
    *
    * @pre          1 <= depah <= 2
    *
    * @details
    *   depth=2、directoryId={1,2} のとき、/dir1/file2 というファイルを作成します。
    *   ファイルの中身はデフォルト (0 フィル) です
    */
    nn::Result CreateFile(int depth, const int* pDirectoryId, int threadId, int64_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsMounted);
        NN_SDK_REQUIRES_LESS_EQUAL(1, depth);
        NN_SDK_REQUIRES_LESS_EQUAL(depth, 2);

        std::lock_guard<nn::os::Mutex> locker(m_Locker);
        if( CheckInterruption() )
        {
            return nnt::fs::util::PowerInterruptionStorage::ResultPowerInterruptionOccurred();
        }
        RefreshMapData(threadId);
        PathData pathData;
        {
            pathData.isDirectory = false;
            pathData.depth = depth;
            for( int i = 0; i < depth; ++i )
            {
                pathData.directoryId[i] = pDirectoryId[i];
            }
        }

        // ファイル作成
        StartCreateFile(pathData, threadId, static_cast<int>(size));
        {
            nn::fssystem::save::Path path;
            pathData.GetPath(&path);
            NN_RESULT_DO(m_SaveDataFileSystem.CreateFile(path, size));
        }
        EndCreateFile(pathData);
        NN_RESULT_DO(m_SaveDataFileSystem.Commit());
        RefreshMapData(threadId);
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        ファイル書き込み
    *
    * @param[in]    depth           ディレクトリの深さ(directoryId の個数)
    * @param[in]    pDirectoryId    パスを求めるための ID
    * @param[in]    data            ファイルデータの先頭の値
    *
    * @return       関数の実行結果を返します
    *
    * @pre          1 <= depah <= 2
    *
    * @details
    *   depth=2、directoryId={1,2} のとき、/dir1/file2 というファイルに書き込みます。
    *   data = 8 のとき、"89ABCD..." という値をファイルに書き込みます。
    */
    nn::Result WriteFile(int depth, const int* pDirectoryId, int threadId, int data) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsMounted);
        NN_SDK_REQUIRES_LESS_EQUAL(1, depth);
        NN_SDK_REQUIRES_LESS_EQUAL(depth, 2);

         std::lock_guard<nn::os::Mutex> locker(m_Locker);
        if( !m_IsMounted )
        {
            return nn::fs::ResultNotInitialized();
        }
        if( CheckInterruption() )
        {
            return nnt::fs::util::PowerInterruptionStorage::ResultPowerInterruptionOccurred();
        }
        RefreshMapData(threadId);
        PathData pathData;
        {
            pathData.isDirectory = false;
            pathData.depth = depth;
            for( int i = 0; i < depth; ++i )
            {
                pathData.directoryId[i] = pDirectoryId[i];
            }
        }

        // ファイル書き込み
        StartWriteFile(pathData, threadId, data);
        {
            nn::fssystem::save::IFile* pFile = nullptr;
            nn::fssystem::save::Path path;

            // ファイルがあるかどうかは、StartWriteFile でチェックしているのでここではチェックしない
            FileData fileData = m_FileMap[pathData];
            pathData.GetPath(&path);

            NN_RESULT_DO(m_SaveDataFileSystem.OpenFile(&pFile, path, nn::fs::OpenMode_Write));
            const auto closer = [this](nn::fssystem::save::IFile* ptr) NN_NOEXCEPT
            {
                m_SaveDataFileSystem.CloseFile(ptr);
            };
            std::unique_ptr<nn::fssystem::save::IFile, decltype(closer)> autoClose(pFile, closer);
            int64_t fileSize = 0;
            NN_RESULT_DO(pFile->GetSize(&fileSize));
            NN_RESULT_DO(
                pFile->WriteBytes(0, fileData.GetExpectedFileData().get(), static_cast<size_t>(fileSize))
            );
            NN_RESULT_DO(pFile->Flush());
        }
        EndWriteFile(pathData);
        NN_RESULT_DO(m_SaveDataFileSystem.Commit());
        RefreshMapData(threadId);
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        ファイル削除
    *
    * @param[in]    depth           ディレクトリの深さ(directoryId の個数)
    * @param[in]    pDirectoryId    パスを求めるための ID
    * @param[in]    threadId        スレッドID、記録のために必要
    *
    * @return       関数の実行結果を返します
    *
    * @pre          1 <= depah <= 2
    *
    * @details
    *   depth=2、directoryId={1,2} のとき、/dir1/file2 というファイルを削除します。
    */
    nn::Result DeleteFile(int depth, const int* pDirectoryId, int threadId) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsMounted);
        NN_SDK_REQUIRES_LESS_EQUAL(1, depth);
        NN_SDK_REQUIRES_LESS_EQUAL(depth, 2);

        std::lock_guard<nn::os::Mutex> locker(m_Locker);
        if( CheckInterruption() )
        {
            return nnt::fs::util::PowerInterruptionStorage::ResultPowerInterruptionOccurred();
        }
        RefreshMapData(threadId);
        PathData pathData;
        {
            pathData.isDirectory = false;
            pathData.depth = depth;
            for( int i = 0; i < depth; ++i )
            {
                pathData.directoryId[i] = pDirectoryId[i];
            }
        }

        // 削除します
        StartDeleteFile(pathData, threadId);
        {
            nn::fssystem::save::Path path;
            pathData.GetPath(&path);
            NN_RESULT_DO(m_SaveDataFileSystem.DeleteFile(path));
        }
        EndDeleteFile(pathData);
        NN_RESULT_DO(m_SaveDataFileSystem.Commit());
        RefreshMapData(threadId);
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        ファイルリネーム
    *
    * @param[in]    srcDepth            ディレクトリの深さ(srcDirectoryId の個数)
    * @param[in]    pSrcDirectoryId     パスを求めるための ID
    * @param[in]    dstDepth            ディレクトリの深さ(dstDirectoryId の個数)
    * @param[in]    pDstDirectoryId     パスを求めるための ID
    * @param[in]    threadId            スレッドID、記録のために必要
    *
    * @return       関数の実行結果を返します
    *
    * @pre          1 <= srcDepth <= 2
    * @pre          1 <= dstDepth <= 2
    *
    * @details
    *   depth=2、pSrcDirectoryId={1,2} のとき、/dir1/file2 というファイルをリネームします。
    */
    nn::Result RenameFile(
        int srcDepth,
        const int* pSrcDirectoryId,
        int dstDepth,
        const int* pDstDirectoryId,
        int threadId
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsMounted);
        NN_SDK_REQUIRES_LESS_EQUAL(1, srcDepth);
        NN_SDK_REQUIRES_LESS_EQUAL(srcDepth, 2);
        NN_SDK_REQUIRES_LESS_EQUAL(1, dstDepth);
        NN_SDK_REQUIRES_LESS_EQUAL(dstDepth, 2);

        std::lock_guard<nn::os::Mutex> locker(m_Locker);
        if( CheckInterruption() )
        {
            return nnt::fs::util::PowerInterruptionStorage::ResultPowerInterruptionOccurred();
        }
        RefreshMapData(threadId);
        PathData srcPathData;
        PathData dstPathData;
        {
            srcPathData.isDirectory = false;
            dstPathData.isDirectory = false;
            srcPathData.depth = srcDepth;
            dstPathData.depth = dstDepth;
            for( int i = 0; i < srcDepth; ++i )
            {
                srcPathData.directoryId[i] = pSrcDirectoryId[i];
            }
            for( int i = 0; i < dstDepth; ++i )
            {
                dstPathData.directoryId[i] = pDstDirectoryId[i];
            }
        }

        // リネームする
        StartDeleteFile(srcPathData, threadId);
        StartCreateFile(
            dstPathData,
            threadId,
            static_cast<int>(m_FileMap[srcPathData].size),
            static_cast<int>(m_FileMap[srcPathData].dataIndex)
        );
        {
            nn::fssystem::save::Path srcPath;
            nn::fssystem::save::Path dstPath;
            srcPathData.GetPath(&srcPath);
            dstPathData.GetPath(&dstPath);
            m_FileMap[srcPathData].oldDataIndex = m_FileMap[srcPathData].dataIndex;
            NN_RESULT_DO(m_SaveDataFileSystem.RenameFile(srcPath, dstPath));
        }
        EndCreateFile(dstPathData);
        EndDeleteFile(srcPathData);
        NN_RESULT_DO(m_SaveDataFileSystem.Commit());
        RefreshMapData(threadId);
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        バリデーション
    *
    * @details
    *   記録とファイルシステムを比較し、誤りが無いかバリデートします。
    *   誤りが検出された場合は assertion failure となります。
    *   マルチスレッドには対応していません。
    */
    void CheckValid() NN_NOEXCEPT
    {
        ASSERT_TRUE(m_IsMounted);

        nn::fssystem::save::Path path;
        StopCounting();

        // ディレクトリ
        for( auto iter = m_DirectoryMap.begin(); iter != m_DirectoryMap.end(); ++iter )
        {
            iter->first.GetPath(&path);
            switch( iter->second.state )
            {
            case DataState::Deleting:
            case DataState::Exist:
                {
                    // 存在していることを確認
                    bool hasDirectory = false;
                    NNT_ASSERT_RESULT_SUCCESS(
                        m_SaveDataFileSystem.HasDirectory(&hasDirectory, path)
                    );
                    ASSERT_TRUE(hasDirectory);
                }
                break;
            case DataState::Creating:
            case DataState::Empty:
                {
                    // 存在しないことを確認
                    bool hasDirectory = true;
                    NNT_ASSERT_RESULT_SUCCESS(
                        m_SaveDataFileSystem.HasDirectory(&hasDirectory, path)
                    );
                    ASSERT_FALSE(hasDirectory);
                }
                break;
            case DataState::Created:
            case DataState::Deleted:
                // 状態が不明なのでチェックしません
                // IndicateFileState を呼び出すことで状態を確定させます
                break;
            default: NN_UNEXPECTED_DEFAULT;
            }
        }
        // ファイル
        for( auto iter = m_FileMap.begin(); iter != m_FileMap.end(); ++iter )
        {
            iter->first.GetPath(&path);
            switch( iter->second.state )
            {
            case DataState::Deleting:
            case DataState::Exist:
                {
                    // 存在していることを確認
                    bool hasFile = false;
                    NNT_ASSERT_RESULT_SUCCESS(
                        m_SaveDataFileSystem.HasFile(&hasFile, path)
                    );
                    ASSERT_TRUE(hasFile);

                    // 中身が正しいことを確認
                    nn::fssystem::save::IFile* pFile;
                    NNT_ASSERT_RESULT_SUCCESS(
                        m_SaveDataFileSystem.OpenFile(&pFile, path, nn::fs::OpenMode_Read)
                    );
                    const auto closer = [this](nn::fssystem::save::IFile* ptr) NN_NOEXCEPT
                    {
                        m_SaveDataFileSystem.CloseFile(ptr);
                    };
                    std::unique_ptr<nn::fssystem::save::IFile, decltype(closer)> autoClose(pFile, closer);

                    int64_t fileSize = 0;
                    {
                        // ファイルサイズ
                        NNT_ASSERT_RESULT_SUCCESS(
                            pFile->GetSize(&fileSize)
                        );
                        ASSERT_EQ(fileSize, iter->second.size);
                    }
                    {
                        // データ
                        std::unique_ptr<char[]> buffer(new char[static_cast<size_t>(fileSize)]);
                        NNT_ASSERT_RESULT_SUCCESS(
                            pFile->ReadBytes(0, buffer.get(), static_cast<size_t>(fileSize))
                        );
                        ASSERT_EQ(
                            0,
                            memcmp(
                                buffer.get(),
                                iter->second.GetExpectedFileData().get(),
                                static_cast<size_t>(fileSize)
                            )
                        );
                    }
                }
                break;
            case DataState::Creating:
            case DataState::Empty:
                {
                    // 存在しないことを確認
                    bool hasDirectory = true;
                    NNT_ASSERT_RESULT_SUCCESS(
                        m_SaveDataFileSystem.HasDirectory(&hasDirectory, path)
                    );
                    ASSERT_FALSE(hasDirectory);
                }
                break;
            case DataState::ReWriting:
            case DataState::ReWrited:
            case DataState::Created:
            case DataState::Deleted:
                // 状態が不明なのでチェックしません
                // IndicateFileState を呼び出すことで状態を確定させます
                break;
            default: NN_UNEXPECTED_DEFAULT;
            }
        }
    }

    // 状態が不明なデータを調査し、
    // 中途半端な状態ではないことを確かめるとともに、
    // 不安定ではない状態にする
    void IndicateDirectoryState()
    {
        if( !m_IsMounted )
        {
            ASSERT_FALSE(true);
        }
        nn::fssystem::save::Path path;

        StopCounting();
        for( auto iter = m_DirectoryMap.begin(); iter != m_DirectoryMap.end(); ++iter )
        {
            switch( iter->second.state )
            {
            case DataState::Deleting:
            case DataState::Exist:
            {
                iter->second.state = DataState::Exist;
            }
            break;
            case DataState::Creating:
            case DataState::Empty:
            {
                iter->second.state = DataState::Empty;
            }
            break;
            case DataState::Created:
            case DataState::Deleted:
            {
                iter->first.GetPath(&path);

                // ディレクトリが存在する or 存在しない
                bool hasDirectory = false;
                NNT_ASSERT_RESULT_SUCCESS(
                    m_SaveDataFileSystem.HasDirectory(&hasDirectory, path)
                );
                iter->second.state = hasDirectory ? DataState::Exist : DataState::Empty;
            }
            break;
            default: NN_UNEXPECTED_DEFAULT;
            }
        }
    }

    // 状態が不明なデータを調査し、
    // 中途半端な状態ではないことを確かめるとともに、
    // 不安定ではない状態にする
    void IndicateFileState()
    {
        if( !m_IsMounted )
        {
            ASSERT_FALSE(true);
        }
        nn::fssystem::save::Path path;
        StopCounting();
        for( auto iter = m_FileMap.begin(); iter != m_FileMap.end(); ++iter )
        {
            switch( iter->second.state )
            {
            case DataState::Deleting:
            case DataState::Exist:
                {
                    iter->second.state = DataState::Exist;
                }
                break;
            case DataState::Creating:
            case DataState::Empty:
                {
                    iter->second.state = DataState::Empty;
                }
                break;
            case DataState::Created:
            case DataState::Deleted:
                {
                    iter->first.GetPath(&path);

                    // ファイルが存在する or 存在しない
                    bool hasFile = false;
                    NNT_ASSERT_RESULT_SUCCESS(
                        m_SaveDataFileSystem.HasFile(&hasFile, path)
                    );
                    iter->second.state = hasFile ? DataState::Exist : DataState::Empty;
                }
                break;
            case DataState::ReWriting:
                {
                    iter->second.state = DataState::Exist;
                    iter->second.dataIndex = iter->second.oldDataIndex;
                }
                break;
            case DataState::ReWrited:
                {
                    iter->second.state = DataState::Exist;
                    std::unique_ptr<char[]> buffer(new char[iter->second.size]);
                    int memcmpResult;

                    // 2 回処理を行う
                    // 1 回目は書き換え後のデータ、2 回めは書き換え前のデータで調査
                    for( int loop = 0; loop < 2; ++loop)
                    {
                        nn::fssystem::save::IFile* pFile = nullptr;
                        iter->first.GetPath(&path);
                        NNT_ASSERT_RESULT_SUCCESS(
                            m_SaveDataFileSystem.OpenFile(&pFile, path, nn::fs::OpenMode_Read)
                        );
                        const auto closer = [this](nn::fssystem::save::IFile* ptr) NN_NOEXCEPT
                        {
                            m_SaveDataFileSystem.CloseFile(ptr);
                        };
                        std::unique_ptr<nn::fssystem::save::IFile, decltype(closer)> autoClose(pFile, closer);

                        NNT_ASSERT_RESULT_SUCCESS(
                            pFile->ReadBytes(0, buffer.get(), iter->second.size)
                        );
                        memcmpResult = memcmp(
                            buffer.get(),
                            iter->second.GetExpectedFileData().get(),
                            iter->second.size
                        );
                        if( memcmpResult == 0 )
                        {
                            break;
                        }
                        iter->second.dataIndex = iter->second.oldDataIndex;
                    }
                    ASSERT_EQ(0, memcmpResult);
                }
                break;
            default: NN_UNEXPECTED_DEFAULT;
            }
        }
    }

public:
    /**
    * @briaf        ディレクトリの数を求めます
    *
    * @param[in]    threadId    スレッド ID
    *
    * @details      ID が threadId であるスレッドによって作成された、
    *               存在するディレクトリの個数を求めます。
    *               削除されたディレクトリや作成中のディレクトリはカウントしません。
    */
    int GetDirectoryCount(int threadId) const NN_NOEXCEPT
    {
        int count = 0;
        for( auto iter = m_DirectoryMap.begin(); iter != m_DirectoryMap.end(); ++iter )
        {
            if( iter->second.state == DataState::Exist
                && iter->second.threadId == threadId )
            {
                ++count;
            }
        }
        return count;
    }

    /**
    * @briaf        ファイルの数を求めます
    *
    * @param[in]    threadId    スレッド ID
    *
    * @details      ID が threadId であるスレッドによって作成された、
    *               存在するファイルの個数を求めます。
    *               削除されたファイルや作成中のファイルはカウントしません。
    */
    int GetFileCount(int threadId) const NN_NOEXCEPT
    {
        int count = 0;
        for( auto iter = m_FileMap.begin(); iter != m_FileMap.end(); ++iter )
        {
            if( iter->second.state == DataState::Exist
                && iter->second.threadId == threadId )
            {
                ++count;
            }
        }
        return count;
    }

    /**
    * @brief        ディレクトリのパスデータ取得
    *
    * @param[out]   outPathData     パスデータ
    * @param[in]    directoryNumber ディレクトリの番号(記録順、0 はじまり)
    * @param[in]    threadId        スレッドID
    *
    * @return       関数の実行結果を返します
    *
    * @details      ディレクトリを番号で指定してそのパスデータを取得します
    */
    void GetDirectoryPathByNumber(
            PathData* outPathData,
            int directoryNumber,
            int threadId
        ) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> locker(m_Locker);
        NN_ASSERT_NOT_NULL(outPathData);

        if( directoryNumber >= GetDirectoryCount(threadId) )
        {
            outPathData->depth = 0;
            return;
        }

        int directoryId = -1;
        for( auto iter = m_DirectoryMap.begin(); iter != m_DirectoryMap.end(); ++iter )
        {
            if( iter->second.threadId != threadId )
            {
                continue;
            }
            if( iter->second.state == DataState::Exist )
            {
                ++directoryId;
            }
            if( directoryId == directoryNumber )
            {
                *outPathData = iter->first;
                return;
            }
        }
        NN_ASSERT(false);
    }

    /**
    * @brief        ファイルのパスデータ取得
    *
    * @param[out]   outPathData     パスデータ
    * @param[in]    fileNumber      ファイルの番号(記録順、0 はじまり)
    * @param[in]    threadId        スレッドID
    *
    * @return       関数の実行結果を返します
    *
    * @details      ファイルを番号で指定してそのパスデータを取得します
    */
    void GetFilePathByNumber(
            PathData* outPathData,
            int fileNumber,
            int threadId
        ) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> locker(m_Locker);
        NN_ASSERT_NOT_NULL(outPathData);
        if( fileNumber >= GetFileCount(threadId) )
        {
            outPathData->depth = 0;
            return;
        }

        int directoryId = -1;
        for( auto iter = m_FileMap.begin(); iter != m_FileMap.end(); ++iter )
        {
            if( iter->second.threadId != threadId )
            {
                continue;
            }
            if( iter->second.state == DataState::Exist )
            {
                ++directoryId;
            }
            if( directoryId == fileNumber )
            {
                *outPathData = iter->first;
                return;
            }
        }
        NN_ASSERT(false);
    }

    // 電源断が発生したかどうかチェックする
    bool CheckInterruption() const NN_NOEXCEPT
    {
        return m_Storage.IsPowerInterruptionOccurred();
    }

    // writableCount 回以上の書き込みが行われたら電源断が発生する
    void StartCounting(int writableCount) NN_NOEXCEPT
    {
        m_Storage.StartCounting(writableCount);
    }

    // 電源断発生状態を解除する
    void StopCounting() NN_NOEXCEPT
    {
        m_Storage.StopCounting();
    }

    // 引数に乱数のシード値をセットする
    void SetRandomSeed() NN_NOEXCEPT
    {
        const PowerInterruptionTestParam& param = GetParam();
        for( int threadId = 0; threadId < param.threadCount; ++threadId )
        {
            m_Argument[threadId].randomSeed = nnt::fs::util::GetRandomSeed();
        }
    }

private:
    /**
    * 記録をリフレッシュする
    * コミット時に電源断が発生したため状態が不明なディレクトリやファイルを、
    * コミット前後どちらの状態になったかを特定します。
    * バリデーションの前に行います。
    */
    void RefreshMapData(int threadId) NN_NOEXCEPT
    {
        // 電源断が発生していないことを確認
        if( m_Storage.IsPowerInterruptionOccurred() )
        {
            return;
        }
        for( auto iter = m_DirectoryMap.begin(); iter != m_DirectoryMap.end(); ++iter )
        {
            if( iter->second.state == DataState::Created && iter->second.threadId == threadId )
            {
                iter->second.state = DataState::Exist;
            }
            if( iter->second.state == DataState::Deleted && iter->second.threadId == threadId )
            {
                iter->second.state = DataState::Empty;
            }
        }
        for( auto iter = m_FileMap.begin(); iter != m_FileMap.end(); ++iter )
        {
            if( iter->second.state == DataState::Created && iter->second.threadId == threadId )
            {
                iter->second.state = DataState::Exist;
            }
            if( iter->second.state == DataState::Deleted && iter->second.threadId == threadId )
            {
                iter->second.state = DataState::Empty;
            }
            if( iter->second.state == DataState::ReWrited && iter->second.threadId == threadId )
            {
                iter->second.state = DataState::Exist;
            }
        }
    }

    void StartCreateDirectory(const PathData& pathData, int threadId) NN_NOEXCEPT
    {
        ASSERT_TRUE(
            m_DirectoryMap.find(pathData) == m_DirectoryMap.end()
            || m_DirectoryMap[pathData].state == DataState::Empty
        );
        DirectoryData directory;
        directory.state = DataState::Creating;
        directory.threadId = threadId;
        m_DirectoryMap[pathData] = directory;
    }

    void EndCreateDirectory(const PathData& pathData) NN_NOEXCEPT
    {
        ASSERT_EQ(
            m_DirectoryMap[pathData].state,
            DataState::Creating
        );
        m_DirectoryMap[pathData].state = DataState::Created;
    }

    void StartDeleteDirectory(const PathData& pathData, int threadId) NN_NOEXCEPT
    {
        ASSERT_EQ(
            m_DirectoryMap[pathData].state,
            DataState::Exist
        );
        m_DirectoryMap[pathData].state = DataState::Deleting;
        m_DirectoryMap[pathData].threadId = threadId;
    }

    void EndDeleteDirectory(const PathData& pathData) NN_NOEXCEPT
    {
        ASSERT_EQ(
            m_DirectoryMap[pathData].state,
            DataState::Deleting
        );
        m_DirectoryMap[pathData].state = DataState::Deleted;
    }

    void StartCreateFile(const PathData& pathData, int threadId, int size) NN_NOEXCEPT
    {
        StartCreateFile(pathData, threadId, size , -1);
    }
    void StartCreateFile(const PathData& pathData, int threadId, int size, int data) NN_NOEXCEPT
    {
        ASSERT_TRUE(
            m_FileMap.find(pathData) == m_FileMap.end()
            || m_FileMap[pathData].state == DataState::Empty
        );
        FileData file;
        file.state = DataState::Creating;
        file.threadId = threadId;
        file.dataIndex = data;
        file.size = size;
        m_FileMap[pathData] = file;
    }

    void EndCreateFile(const PathData& pathData) NN_NOEXCEPT
    {
        ASSERT_EQ(
            m_FileMap[pathData].state,
            DataState::Creating
        );
        m_FileMap[pathData].state = DataState::Created;
    }

    void StartDeleteFile(const PathData& pathData, int threadId) NN_NOEXCEPT
    {
        ASSERT_EQ(
            m_FileMap[pathData].state,
            DataState::Exist
        );
        m_FileMap[pathData].state = DataState::Deleting;
        m_FileMap[pathData].threadId = threadId;
    }

    void EndDeleteFile(const PathData& pathData) NN_NOEXCEPT
    {
        ASSERT_EQ(
            m_FileMap[pathData].state,
            DataState::Deleting
        );
        m_FileMap[pathData].state = DataState::Deleted;
    }

    void StartWriteFile(const PathData& pathData, int threadId, int dataIndex) NN_NOEXCEPT
    {
        ASSERT_EQ(
            m_FileMap[pathData].state,
            DataState::Exist
        );
        m_FileMap[pathData].state = DataState::ReWriting;
        m_FileMap[pathData].threadId = threadId;
        m_FileMap[pathData].oldDataIndex = m_FileMap[pathData].dataIndex;
        m_FileMap[pathData].dataIndex = dataIndex;
    }

    void EndWriteFile(const PathData& pathData) NN_NOEXCEPT
    {
        ASSERT_EQ(
            m_FileMap[pathData].state,
            DataState::ReWriting
        );
        m_FileMap[pathData].state = DataState::ReWrited;
    }

private:
    bool m_IsMounted;
    nn::os::ThreadType m_Threads[MaxThreadCount];
    ThreadArgument m_Argument[MaxThreadCount];
    nnt::fs::util::SafeMemoryStorage m_BaseStorage;
    nnt::fs::util::PowerInterruptionStorage m_Storage;
    nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver m_SaveDataFileSystem;
    std::map<PathData, DirectoryData> m_DirectoryMap;
    std::map<PathData, FileData> m_FileMap;
    nn::os::Mutex m_Locker;
};

/**
* マウント中に電源断を行うテスト。
* シングルスレッドで実行します。
* 電源断までの書き込み可能回数を徐々に減らし、一定数以下でエラーが発生することを確認します。
* フォーマットに失敗したストレージはマウントできないことを確認します。
*/
TEST_P(FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTest, MountTest)
{
    const PowerInterruptionTestParam& param = GetParam();
    bool formatErrorOccurred = false;

    // writableCount の初期値は決め打ち
    for( int writableCount = 60; writableCount > 0; --writableCount )
    {
        nn::Result result = Format(param, writableCount);
        if( result.IsSuccess() )
        {
            // 過去に失敗し今回成功したならエラー
            ASSERT_FALSE(formatErrorOccurred);

            // 電源断が発生したか確認
            if( CheckInterruption() )
            {
                StopCounting();
                bool isMountSuccess = Mount();
                if( isMountSuccess )
                {
                    // 電源断発生 & フォーマット成功 & マウント成功
                    NN_SDK_LOG("Power Interrupted, But Format Success and Initialize Success\n");
                }
            }
        }
        else
        {
            formatErrorOccurred = true;

            // 電源断が発生したか確認
            ASSERT_TRUE(CheckInterruption());
        }
        Unmount();
    }
}

/**
* ディレクトリ作成中 or 作成後のコミット中に電源断
* ディレクトリもファイルも無い状態から開始
*/
TEST_P(FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTest, CreateDirectory1)
{
    const PowerInterruptionTestParam& param = GetParam();

    // バッファが剥がせず無限ループになることがあるので注意
    for( int writableCount = 10; writableCount < 50; ++writableCount )
    {
        // フォーマット、マウントする(前回のループの内容はリセットされる)
        Format(param);
        ASSERT_TRUE(Mount());

        // 書き込み可能回数を設定
        StartCounting(writableCount);

        // マルチスレッドでディレクトリ作成
        ExecuteMultiThreadFunction(
            [](void* pArg) NN_NOEXCEPT
            {
                nn::Result result;
                const auto pArgument = reinterpret_cast<ThreadArgument*>(pArg);
                int directoryIdList[1];
                int directoryId = 0;
                do
                {
                    // /dir<threadId * 50 + directoryId> というディレクトリを作り続ける
                    // 電源断されたら終了
                    directoryIdList[0] = pArgument->threadId * 50 + directoryId;
                    result = pArgument->pFileSystemTest->CreateDirectory(
                        1,
                        directoryIdList,
                        pArgument->threadId
                    );
                    ++directoryId;
                } while( result.IsSuccess() );

                // 本当に電源断されたのかチェック
                ASSERT_TRUE(
                    pArgument->CheckInterruption()
                );
            }
        );

        // 電源断が発生したのか再度確認
        ASSERT_TRUE(CheckInterruption());

        // 初期化できるか確認
        StopCounting();
        Unmount();
        bool initializeSuccess = Mount();
        ASSERT_TRUE(initializeSuccess);

        // バリデーション
        IndicateDirectoryState();
        IndicateFileState();
        CheckValid();

        Unmount();
    }
}

/**
* ディレクトリ作成中 or 作成後のコミット中に電源断
* 複数ディレクトリ、複数ファイルが有る状態から開始
*/
TEST_P(FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTest, CreateDirectory2)
{
    const PowerInterruptionTestParam& param = GetParam();

    // バッファが剥がせず無限ループになることがあるので注意
    for( int writableCount = 10; writableCount < 30; ++writableCount )
    {
        // フォーマット、マウントする(前回のループの内容はリセットされる)
        Format(param);
        ASSERT_TRUE(Mount());
        // 念のため
        StopCounting();

        // ファイルを 12 個、ディレクトリを10個作っておく
        {
            int directoryIdList[2];
            directoryIdList[0] = 9999;
            CreateDirectory(1, directoryIdList, 0);
            for( int i = 0; i < 10; ++i )
            {
                directoryIdList[1] = i;
                CreateDirectory(2, directoryIdList, 0);
                CreateFile(2, directoryIdList, 0, 100);
            }
            directoryIdList[0] = 0;
            CreateDirectory(1, directoryIdList, 0);
        }

        // 書き込み可能回数を設定
        StartCounting(writableCount);

        // マルチスレッドでディレクトリ作成
        ExecuteMultiThreadFunction(
            [](void* pArg) NN_NOEXCEPT
            {
                nn::Result result;
                const auto pArgument = reinterpret_cast<ThreadArgument*>(pArg);
                int directoryIdList[2];
                directoryIdList[0] = 0;
                int directoryId = 0;
                do
                {
                    // /dir0/dir<threadId * 50 + directoryId> というディレクトリを作り続ける
                    // 電源断されたら終了
                    directoryIdList[1] = pArgument->threadId * 50 + directoryId;
                    result = pArgument->pFileSystemTest->CreateDirectory(
                        2,
                        directoryIdList,
                        pArgument->threadId
                    );
                    ++directoryId;
                } while( result.IsSuccess() );

                // 本当に電源断されたのかチェック
                ASSERT_TRUE(pArgument->CheckInterruption());
            }
        );

        // 電源断が発生したのか再度確認
        ASSERT_TRUE(CheckInterruption());

        // 初期化できるか確認
        StopCounting();
        Unmount();
        bool initializeSuccess = Mount();
        ASSERT_TRUE(initializeSuccess);

        // バリデーション
        IndicateDirectoryState();
        IndicateFileState();
        CheckValid();

        Unmount();
    }
}

/**
* ディレクトリリネーム中 or リネーム後のコミット中に電源断
*/
TEST_P(FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTest, RenameDirectoryHeavy)
{
    const PowerInterruptionTestParam& param = GetParam();
    for( int writableCount = 10; writableCount < 50; ++writableCount )
    {
        // フォーマット、マウントする(前回のループの内容はリセットされる)
        Format(param);
        ASSERT_TRUE(Mount());

        // シングルスレッドでサイズ1024のファイルを作成しておく
        int directoryId[1];
        for( int threadId = 0; threadId < param.threadCount; ++threadId )
        {
            for( int i = 0; i < 10; ++i )
            {
                directoryId[0] = i + 100 * threadId;
                NNT_ASSERT_RESULT_SUCCESS(
                    CreateDirectory(
                        1,
                        directoryId,
                        threadId
                    )
                );
            }
        }
        // 書き込み可能回数を設定
        StartCounting(writableCount);
        ExecuteMultiThreadFunction(
            [](void* pArg) NN_NOEXCEPT
            {
                nn::Result result;
                const auto pArgument = reinterpret_cast<ThreadArgument*>(pArg);
                int srcDirectoryIdList[1];
                int dstDirectoryIdList[1];
                int directoryId = 0;
                do
                {
                    // /dir<threadId * 100 + directoryId> というディレクトリを
                    // /dir<threadId * 100 + directoryId + 50> にリネームし続ける
                    // 電源断されたら終了
                    srcDirectoryIdList[0] = pArgument->threadId * 100 + directoryId;
                    dstDirectoryIdList[0] = pArgument->threadId * 100 + directoryId + 50;
                    result = pArgument->pFileSystemTest->RenameDirectory(
                        1,
                        srcDirectoryIdList,
                        1,
                        dstDirectoryIdList,
                        pArgument->threadId
                    );
                    ++directoryId;
                } while( result.IsSuccess() );

                // 本当に電源断されたのかチェック
                ASSERT_TRUE(pArgument->CheckInterruption());
            }
        );
        Unmount();

        // 電源断が発生したのか再度確認
        ASSERT_TRUE(CheckInterruption());

        // 初期化できるか確認
        StopCounting();
        bool initializeSuccess = Mount();
        ASSERT_TRUE(initializeSuccess);

        // バリデーション
        IndicateDirectoryState();
        IndicateFileState();
        CheckValid();

        Unmount();
    }
}

/**
* ディレクトリ削除中 or 削除後のコミット中に電源断
*/
TEST_P(FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTest, DeleteDirectoryHeavy)
{
    const PowerInterruptionTestParam& param = GetParam();
    for( int writableCount = 10; writableCount < 50; ++writableCount )
    {
        // フォーマット、マウントする(前回のループの内容はリセットされる)
        Format(param);
        ASSERT_TRUE(Mount());

        // シングルスレッドでディレクトリを作成しておく
        StopCounting();
        int directoryId[1];
        for( int threadId = 0; threadId < param.threadCount; ++threadId )
        {
            for( int i = 0; i < 10; ++i )
            {
                directoryId[0] = i + 100 * threadId;
                NNT_ASSERT_RESULT_SUCCESS(
                    CreateDirectory(
                        1,
                        directoryId,
                        threadId
                    )
                );
            }
        }
        // 書き込み可能回数を設定
        StartCounting(writableCount);
        ExecuteMultiThreadFunction(
            [](void* pArg) NN_NOEXCEPT
            {
                nn::Result result;
                const auto pArgument = reinterpret_cast<ThreadArgument*>(pArg);
                int directoryIdList[1];
                int directoryId = 0;
                do
                {
                    // /file<threadId * 100 + directoryId> というディレクトリを削除し続ける
                    // 電源断されたら終了
                    directoryIdList[0] = pArgument->threadId * 100 + directoryId;
                    result = pArgument->pFileSystemTest->DeleteDirectory(
                        1,
                        directoryIdList,
                        pArgument->threadId
                    );
                    ++directoryId;
                } while( result.IsSuccess() );

                // 本当に電源断されたのかチェック
                ASSERT_TRUE(pArgument->CheckInterruption());
            }
        );
        Unmount();

        // 電源断が発生したのか再度確認
        ASSERT_TRUE(CheckInterruption());

        // 初期化できるか確認
        StopCounting();
        bool initializeSuccess = Mount();
        ASSERT_TRUE(initializeSuccess);

        // バリデーション
        IndicateDirectoryState();
        IndicateFileState();
        CheckValid();

        Unmount();
    }
}

/**
* ファイル作成中 or 作成後のコミット中に電源断
* ディレクトリもファイルも無い状態から開始
*/
TEST_P(FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTest, CreateFile1)
{
    const PowerInterruptionTestParam& param = GetParam();

    // バッファが剥がせず無限ループになることがあるので注意
    for( int writableCount = 10; writableCount < 50; ++writableCount )
    {
        // フォーマット、マウントする(前回のループの内容はリセットされる)
        Format(param);
        ASSERT_TRUE(Mount());

        // 書き込み可能回数を設定
        StartCounting(writableCount);

        // マルチスレッドでディレクトリ作成
        ExecuteMultiThreadFunction(
            [](void* pArg) NN_NOEXCEPT
            {
                nn::Result result;
                const auto pArgument = reinterpret_cast<ThreadArgument*>(pArg);
                nn::fssystem::save::Path path;
                int directoryIdList[1];
                int directoryId = 0;
                do
                {
                    // /file<threadId * 100 + directoryId> というファイルを作り続ける
                    // 電源断されたら終了
                    directoryIdList[0] = pArgument->threadId * 100 + directoryId;
                    int64_t fileSize = pArgument->threadId * directoryId * 10 + 800;
                    result = pArgument->pFileSystemTest->CreateFile(
                        1,
                        directoryIdList,
                        pArgument->threadId,
                        fileSize
                    );
                    if( result.IsSuccess() )
                    {
                        result = pArgument->pFileSystemTest->WriteFile(
                            1,
                            directoryIdList,
                            pArgument->threadId,
                            directoryId
                        );
                    }
                    ++directoryId;
                } while( result.IsSuccess() );

                // 本当に電源断されたのかチェック
                ASSERT_TRUE(pArgument->CheckInterruption());
            }
        );
        Unmount();

        // 電源断が発生したのか再度確認
        ASSERT_TRUE(CheckInterruption());
        StopCounting();

        // 初期化できるか確認
        bool initializeSuccess = Mount();
        ASSERT_TRUE(initializeSuccess);

        // バリデーション
        IndicateDirectoryState();
        IndicateFileState();
        CheckValid();

        Unmount();
    }
}

/**
* ファイル作成中 or 作成後のコミット中に電源断
* 複数ディレクトリ、複数ファイルが有る状態から開始
*/
TEST_P(FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTest, CreateFile2)
{
    const PowerInterruptionTestParam& param = GetParam();

    // バッファが剥がせず無限ループになることがあるので注意
    for( int writableCount = 10; writableCount < 50; ++writableCount )
    {
        // フォーマット、マウントする(前回のループの内容はリセットされる)
        Format(param);
        ASSERT_TRUE(Mount());
        // 念のため
        StopCounting();

        // ファイルを 12 個、ディレクトリを10個作っておく
        {
            int directoryIdList[2];
            directoryIdList[0] = 9999;
            CreateDirectory(1, directoryIdList, 0);
            for( int i = 0; i < 10; ++i )
            {
                directoryIdList[1] = i;
                CreateDirectory(2, directoryIdList, 0);
                CreateFile(2, directoryIdList, 0, 100);
            }
            directoryIdList[0] = 0;
            CreateDirectory(1, directoryIdList, 0);
        }

        // 書き込み可能回数を設定
        StartCounting(writableCount);

        // マルチスレッドでディレクトリ作成
        ExecuteMultiThreadFunction(
            [](void* pArg) NN_NOEXCEPT
            {
                nn::Result result;
                const auto pArgument = reinterpret_cast<ThreadArgument*>(pArg);
                nn::fssystem::save::Path path;
                int directoryIdList[2];
                directoryIdList[0] = 0;
                int directoryId = 0;
                do
                {
                    // /dir0/file<threadId * 100 * directoryId> というファイルを作り続ける
                    // 電源断されたら終了
                    directoryIdList[1] = pArgument->threadId * 100 + directoryId;
                    int64_t fileSize = pArgument->threadId * directoryId * 10 + 800;
                    result = pArgument->pFileSystemTest->CreateFile(
                        2,
                        directoryIdList,
                        pArgument->threadId,
                        fileSize
                    );
                    if( result.IsSuccess() )
                    {
                        result = pArgument->pFileSystemTest->WriteFile(
                            2,
                            directoryIdList,
                            pArgument->threadId,
                            directoryId
                        );
                    }
                    ++directoryId;
                } while( result.IsSuccess() );

                // 本当に電源断されたのかチェック
                ASSERT_TRUE(pArgument->CheckInterruption());
            }
        );
        Unmount();

        // 電源断が発生したのか再度確認
        ASSERT_TRUE(CheckInterruption());

        // 初期化できるか確認
        StopCounting();
        bool initializeSuccess = Mount();
        ASSERT_TRUE(initializeSuccess);

        // バリデーション
        IndicateDirectoryState();
        IndicateFileState();
        CheckValid();

        Unmount();
    }
}

/**
* ファイル書き込み中 or 書き込み後のコミット中に電源断
*/
TEST_P(FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTest, WriteFile)
{
    const PowerInterruptionTestParam& param = GetParam();

    // バッファが剥がせず無限ループになることがあるので注意
    for( int writableCount = 20; writableCount < 50; ++writableCount )
    {
        // フォーマット、マウントする(前回のループの内容はリセットされる)
        Format(param);
        ASSERT_TRUE(Mount());

        // 先にファイルを作成しておく
        for( int threadId = 0; threadId < param.threadCount; ++threadId )
        {
            StopCounting();
            int directoryIdList[1];
            directoryIdList[0] = threadId;
            NNT_ASSERT_RESULT_SUCCESS(
                CreateFile(
                    1,
                    directoryIdList,
                    threadId,
                    100
                )
            );
        }

        // 書き込み可能回数を設定
        StartCounting(writableCount);
        ExecuteMultiThreadFunction(
            [](void* pArg) NN_NOEXCEPT
            {
                nn::Result result;
                const auto pArgument = reinterpret_cast<ThreadArgument*>(pArg);
                int directoryIdList[1];
                directoryIdList[0] = pArgument->threadId;
                int directoryId = 0;
                do
                {
                    // /file<threadId> というファイルを何度も書き換える
                    int fileDataIndex = pArgument->threadId * directoryId;
                    result = pArgument->pFileSystemTest->WriteFile(
                        1,
                        directoryIdList,
                        pArgument->threadId,
                        fileDataIndex
                    );
                    ++directoryId;
                } while( result.IsSuccess() );

                // 電源断されたのかチェック
                ASSERT_TRUE(pArgument->CheckInterruption());
            }
        );
        Unmount();

        // 電源断が発生したのか再度確認
        ASSERT_TRUE(CheckInterruption());

        // 初期化できるか確認
        StopCounting();
        bool initializeSuccess = Mount();
        ASSERT_TRUE(initializeSuccess);

        // バリデーション
        IndicateDirectoryState();
        IndicateFileState();
        CheckValid();

        Unmount();
    }
}

/**
* ファイル削除中 or 削除後のコミット中に電源断
*/
TEST_P(FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTest, DeleteFileHeavy)
{
    const PowerInterruptionTestParam& param = GetParam();
    for( int writableCount = 10; writableCount < 50; ++writableCount )
    {
        // フォーマット、マウントする(前回のループの内容はリセットされる)
        Format(param);
        ASSERT_TRUE(Mount());

        // シングルスレッドでサイズ1024のファイルを作成しておく
        int directoryId[1];
        for( int threadId = 0; threadId < param.threadCount; ++threadId )
        {
            for( int i = 0; i < 20; ++i )
            {
                directoryId[0] = i + 100 * threadId;
                NNT_ASSERT_RESULT_SUCCESS(
                    CreateFile(
                        1,
                        directoryId,
                        threadId,
                        1024
                    )
                );
            }
        }
        // 書き込み可能回数を設定
        StartCounting(writableCount);
        ExecuteMultiThreadFunction(
            [](void* pArg) NN_NOEXCEPT
            {
                nn::Result result;
                const auto pArgument = reinterpret_cast<ThreadArgument*>(pArg);
                int directoryIdList[1];
                int directoryId = 0;
                do
                {
                    // /file<threadId * 100 + directoryId> というファイルを削除し続ける
                    // 電源断されたら終了
                    directoryIdList[0] = pArgument->threadId * 100 + directoryId;
                    result = pArgument->pFileSystemTest->DeleteFile(
                        1,
                        directoryIdList,
                        pArgument->threadId
                    );
                    ++directoryId;
                } while( result.IsSuccess() );

                // 本当に電源断されたのかチェック
                ASSERT_TRUE(pArgument->CheckInterruption());
            }
        );
        // 電源断が発生したのか再度確認
        ASSERT_TRUE(CheckInterruption());

        // 初期化できるか確認
        StopCounting();
        Unmount();
        bool initializeSuccess = Mount();
        ASSERT_TRUE(initializeSuccess);

        // バリデーション
        IndicateDirectoryState();
        IndicateFileState();
        CheckValid();

        Unmount();
    }
}

/**
* ファイルリネーム中 or リネーム後のコミット中に電源断
*/
TEST_P(FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTest, RenameFileHeavy)
{
    const PowerInterruptionTestParam& param = GetParam();
    for( int writableCount = 10; writableCount < 50; ++writableCount )
    {
        // フォーマット、マウントする(前回のループの内容はリセットされる)
        Format(param);
        ASSERT_TRUE(Mount());

        // シングルスレッドでサイズ1024のファイルを作成しておく
        int directoryId[1];
        for( int threadId = 0; threadId < param.threadCount; ++threadId )
        {
            for( int i = 0; i < 10; ++i )
            {
                directoryId[0] = i + 100 * threadId;
                NNT_ASSERT_RESULT_SUCCESS(
                    CreateFile(
                        1,
                        directoryId,
                        threadId,
                        1024
                    )
                );
            }
        }

        // 書き込み可能回数を設定
        StartCounting(writableCount);
        ExecuteMultiThreadFunction(
            [](void* pArg) NN_NOEXCEPT
            {
                nn::Result result;
                const auto pArgument = reinterpret_cast<ThreadArgument*>(pArg);
                int srcDirectoryIdList[1];
                int dstDirectoryIdList[1];
                int directoryId = 0;
                do
                {
                    // /file<threadId * 100 + directoryId> というファイルを
                    // /file<threadId * 100 + directoryId + 50> にリネームし続ける
                    // 電源断されたら終了
                    srcDirectoryIdList[0] = pArgument->threadId * 100 + directoryId;
                    dstDirectoryIdList[0] = pArgument->threadId * 100 + directoryId + 50;
                    result = pArgument->pFileSystemTest->RenameFile(
                        1,
                        srcDirectoryIdList,
                        1,
                        dstDirectoryIdList,
                        pArgument->threadId
                    );
                    ++directoryId;
                } while( result.IsSuccess() );

                // 本当に電源断されたのかチェック
                ASSERT_TRUE(pArgument->CheckInterruption());
            }
        );
        Unmount();

        // 電源断が発生したのか再度確認
        ASSERT_TRUE(CheckInterruption());

        // 初期化できるか確認
        StopCounting();
        bool initializeSuccess = Mount();
        ASSERT_TRUE(initializeSuccess);

        // バリデーション
        IndicateDirectoryState();
        IndicateFileState();
        CheckValid();

        Unmount();
    }
}

//! ランダムにディレクトリを選択し削除する
nn::Result RundomTestDeleteDirectory(
        FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTest* pFileSystemDriverTest,
        int threadId,
        std::mt19937* random
    )
{
    int directoryCount = pFileSystemDriverTest->GetDirectoryCount(threadId);
    if( directoryCount > 0 )
    {
        int directoryNumber
            = std::uniform_int_distribution<>(0, directoryCount - 1)(*random);
        FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTest::PathData path;
        pFileSystemDriverTest->GetDirectoryPathByNumber(
            &path,
            directoryNumber,
            threadId
        );
        if( path.depth > 1 )
        {
            NN_RESULT_DO(
                pFileSystemDriverTest->DeleteDirectory(
                    path.depth,
                    path.directoryId,
                    threadId
                )
            );
        }
    }
    NN_RESULT_SUCCESS;
}

//! ランダムにファイルを選択し削除する
nn::Result RundomTestDeleteFile(
        FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTest* pFileSystemDriverTest,
        int threadId,
        std::mt19937* random
    )
{
    int fileCount = pFileSystemDriverTest->GetFileCount(threadId);
    if( fileCount > 0 )
    {
        int fileNumber = std::uniform_int_distribution<>(0, fileCount - 1)(*random);
        FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTest::PathData path;
        pFileSystemDriverTest->GetFilePathByNumber(
            &path,
            fileNumber,
            threadId
        );
        if( path.depth > 1 )
        {
            NN_RESULT_DO(
                pFileSystemDriverTest->DeleteFile(
                    path.depth,
                    path.directoryId,
                    threadId
                )
            );
        }
    }
    NN_RESULT_SUCCESS;
}

//! ランダムにディレクトリを取得し、renameDirectoryId へリネームする
nn::Result RundomTestRenameDirectory(
        FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTest* pFileSystemDriverTest,
        int threadId,
        int renameDirectoryId,
        std::mt19937* random
    )
{
    int directoryCount = pFileSystemDriverTest->GetDirectoryCount(threadId);
    if( directoryCount > 0 )
    {
        int directoryNumber = std::uniform_int_distribution<>(0, directoryCount - 1)(*random);
        FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTest::PathData path;
        pFileSystemDriverTest->GetDirectoryPathByNumber(
            &path,
            directoryNumber,
            threadId
        );
        if( path.depth > 1 )
        {
            int directoryIdList[2] = {threadId, renameDirectoryId};
            NN_RESULT_DO(
                pFileSystemDriverTest->RenameDirectory(
                    path.depth,
                    path.directoryId,
                    2,
                    directoryIdList,
                    threadId
                )
            );
        }
    }
    NN_RESULT_SUCCESS;
}

//! ランダムにファイルを取得し、renameFileId へリネームする
nn::Result RundomTestRenameFile(
        FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTest* pFileSystemDriverTest,
        int threadId,
        int renameFileId,
        std::mt19937* random
    )
{
    int fileCount = pFileSystemDriverTest->GetFileCount(threadId);
    if( fileCount > 0 )
    {
        int fileNumber = std::uniform_int_distribution<>(0, fileCount - 1)(*random);
        FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTest::PathData path;
        pFileSystemDriverTest->GetFilePathByNumber(&path, fileNumber, threadId);
        if( path.depth > 1 )
        {
            int directoryIdList[2] = {threadId, renameFileId};
            NN_RESULT_DO(
                pFileSystemDriverTest->RenameFile(
                    path.depth,
                    path.directoryId,
                    2,
                    directoryIdList,
                    threadId
                )
            );
        }
    }
    NN_RESULT_SUCCESS;
}

NNT_DISABLE_OPTIMIZATION
/**
* @brief    ランダムテスト
*
* @details
*   FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTest.RandamTest の中身です
*   乱数を用いランダムに以下の処理のいずれかをランダムに行います
*       ディレクトリ追加
*       ディレクトリリネーム
*       ディレクトリ削除
*       ファイル追加
*       ファイル書き込み
*       ファイルリネーム
*       ファイル削除
*   電源断の発生までの書き込み回数はランダムです。
*   電源断が確認されたら即座にオブジェクトを開放し、ストレージマウントからやり直します。
*   やり直した際にデータに異常が無いか確認します。
*   マルチスレッドで実行することを想定しています。
*/
void RundomTest(
        FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTest* pFileSystemDriverTest,
        int threadId,
        int seed
    ) NN_NOEXCEPT
{
    nn::Result result = nn::ResultSuccess();
    int directoryIdList[2];
    directoryIdList[0] = threadId;
    static int directoryId[MaxThreadCount] = {0};

    std::mt19937 random(seed);

    while( result.IsSuccess() )
    {
        directoryIdList[1] = directoryId[threadId];
        ++directoryId[threadId];
        // 乱数で処理を変える
        switch( std::uniform_int_distribution<>(0, 6)(random) )
        {
        case 0:
            // ディレクトリを作成
            {
                result = pFileSystemDriverTest->CreateDirectory(2, directoryIdList, threadId);
            }
            break;
        case 1:
            // ファイルを作成
            {
                int64_t fileSize = std::uniform_int_distribution<>(16, 4096)(random);
                result = pFileSystemDriverTest->CreateFile(2, directoryIdList, threadId, fileSize);
            }
            break;
        case 2:
            // ファイルに書き込み
            {
                // ランダムにファイル取得
                int fileCount = pFileSystemDriverTest->GetFileCount(threadId);
                if( fileCount > 0 )
                {
                    int fileNumber = std::uniform_int_distribution<>(0, fileCount - 1)(random);
                    FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTest::PathData pathData;
                    pFileSystemDriverTest->GetFilePathByNumber(&pathData, fileNumber, threadId);
                    if( pathData.depth > 1 )
                    {
                        int writeData = std::uniform_int_distribution<>(0, 100)(random);
                        result = pFileSystemDriverTest->WriteFile(
                            pathData.depth,
                            pathData.directoryId,
                            threadId,
                            writeData
                        );
                    }
                }
            }
            break;
        case 3:
            // ディレクトリを削除
            {
                result = RundomTestDeleteDirectory(pFileSystemDriverTest, threadId, &random);
            }
            break;
        case 4:
            // ファイルを削除
            {
                result = RundomTestDeleteFile(pFileSystemDriverTest, threadId, &random);
        }
            break;
        case 5:
            // ディレクトリリネーム
            {
                result = RundomTestRenameDirectory(
                    pFileSystemDriverTest,
                    threadId,
                    directoryIdList[1],
                    &random
                );
            }
            break;
        case 6:
            // ファイルをリネーム
            {
                result = RundomTestRenameFile(
                    pFileSystemDriverTest,
                    threadId,
                    directoryIdList[1],
                    &random
                );
            }
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }
    ASSERT_TRUE(pFileSystemDriverTest->CheckInterruption());
}
NNT_RESTORE_OPTIMIZATION

TEST_P(FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTest, RandomTest)
{
    const PowerInterruptionTestParam& param = GetParam();

    // フォーマット、マウントする(このテストではフォーマットしなおさない)
    Format(param);
    ASSERT_TRUE(Mount());

    // /dir<threadId> というディレクトリを作成しておく
    int directoryIdList[1];
    for( int threadId = 0; threadId < param.threadCount; ++threadId )
    {
        directoryIdList[0] = threadId;
        NNT_ASSERT_RESULT_SUCCESS(
            CreateDirectory(
                1,
                directoryIdList,
                threadId
            )
        );
    }
    Unmount();
    std::mt19937 random(nnt::fs::util::GetRandomSeed());
    static const int LoopCount = 50;

    // バッファが剥がせず無限ループになることがあるので対策が必要
    // とりあえず 50 周します
    for( int loop = 0; loop < LoopCount; ++loop )
    {
        ASSERT_TRUE(Mount());

        // 各スレッドの Seed 値をセット
        SetRandomSeed();

        // 書き込み可能回数をセット
        StartCounting(std::uniform_int_distribution<>(20, 200)(random));

        ExecuteMultiThreadFunction(
            [](void* pArg) NN_NOEXCEPT
            {
                const auto pArgument = reinterpret_cast<ThreadArgument*>(pArg);
                RundomTest(
                    pArgument->pFileSystemTest,
                    pArgument->threadId,
                    pArgument->randomSeed
                );

                // 終了の原因が電源断かチェック
                ASSERT_TRUE(pArgument->CheckInterruption());
            }
        );
        Unmount();

        // 電源断が発生したのか再度確認
        ASSERT_TRUE(CheckInterruption());
        StopCounting();

        // 初期化できるか確認
        {
            bool initializeSuccess = Mount();
            ASSERT_TRUE(initializeSuccess);

            // バリデーション
            IndicateDirectoryState();
            IndicateFileState();
            CheckValid();
        }
        Unmount();
    }
}

INSTANTIATE_TEST_CASE_P(
    FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTestSet,
    FsJournalIntegritySaveDataFileSystemDriverPowerInterruptionTest,
    ::testing::ValuesIn(MakePowerInterruptionTestParam())
);

#include "savedata1_v4.h"
#include "savedata2_v4.h"
#include "savedata3_v4.h"
#include "savedata4_v5.h"

TEST(FsJournalIntegritySaveDataFileSystemDriver, CompatibleDataV4)
{
    struct CompatibleDataTable
    {
        const uint8_t* pData;
        size_t size;
        char extraData;
    };
    const CompatibleDataTable CompatibleDataTableV4[] =
    {
        { SaveData1_v4, sizeof(SaveData1_v4), 0x11 },
        { SaveData2_v4, sizeof(SaveData2_v4), 0x22 },
        { SaveData3_v4, sizeof(SaveData3_v4), 0x33 },
    };

    for( size_t indexTable = 0; indexTable < sizeof(CompatibleDataTableV4) / sizeof(CompatibleDataTable); ++indexTable)
    {
        size_t sizeData = nn::util::GetGzipDecompressedSize(
                              CompatibleDataTableV4[indexTable].pData,
                              CompatibleDataTableV4[indexTable].size
                          );
        nnt::fs::util::Vector<char> buffer(sizeData);
        nnt::fs::util::Vector<char> work(64 * 1024);
        bool isDecompressSuccess = nn::util::DecompressGzip(
                                       &buffer[0],
                                       buffer.size(),
                                       CompatibleDataTableV4[indexTable].pData,
                                       CompatibleDataTableV4[indexTable].size,
                                       &work[0],
                                       work.size()
                                    );
        ASSERT_TRUE(isDecompressSuccess);

        nnt::fs::util::SafeMemoryStorage storageData(buffer.size());
        std::memcpy(storageData.GetBuffer(), &buffer[0], buffer.size());
        nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver fileSystem;
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Initialize(
                nn::fs::SubStorage(&storageData, 0, buffer.size()),
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );

        // 拡張データが読み込めることを確認
        {
            nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData extraData;
            fileSystem.ReadExtraData(&extraData);
            const char* p = reinterpret_cast<const char *>(&extraData);
            for( size_t i = 0; i < sizeof(extraData); ++i )
            {
                ASSERT_EQ(CompatibleDataTableV4[indexTable].extraData, p[i]);
            }
        }

        // ルートディレクトリ直下のファイル読み込み
        {
            nn::fssystem::save::IDirectory* pDirectory = nullptr;
            nn::fssystem::save::Path pathDirectory;
            NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize("/"));
            NNT_ASSERT_RESULT_SUCCESS(
                fileSystem.OpenDirectory(
                    &pDirectory,
                    pathDirectory
                )
            );
            while( NN_STATIC_CONDITION(true) )
            {
                int32_t count;
                nn::fs::DirectoryEntry de;
                NNT_ASSERT_RESULT_SUCCESS(pDirectory->Read(&count, &de, 1));
                if( count == 0 )
                {
                    break;
                }
            }
            fileSystem.CloseDirectory(pDirectory);
        }
    }
}

TEST(FsJournalIntegritySaveDataFileSystemDriver, CompatibleDataV4AndExpand)
{
    struct CompatibleDataTable
    {
        const uint8_t* pData;
        size_t size;
    };
    const CompatibleDataTable CompatibleDataTableV4[] =
    {
        { SaveData1_v4, sizeof(SaveData1_v4) },
        { SaveData2_v4, sizeof(SaveData2_v4) },
        { SaveData3_v4, sizeof(SaveData3_v4) },
    };

    const int MinimumVersion = 4;
    for( size_t indexTable = 0; indexTable < sizeof(CompatibleDataTableV4) / sizeof(CompatibleDataTable); ++indexTable)
    {
        const auto sizeBlock = 4 * 1024;
        const auto countDataBlockExpanded = 1024;
        const auto countReservedBlockExpanded = countDataBlockExpanded;

        size_t sizeData = nn::util::GetGzipDecompressedSize(
                              CompatibleDataTableV4[indexTable].pData,
                              CompatibleDataTableV4[indexTable].size
                          );

        int64_t sizeExpanded = 0;
        {
            auto CountExpandMax = 25;
            nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam paramDuplex;
            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam paramIntegrity;
            paramDuplex.sizeBlockLevel[0] = sizeBlock;
            paramDuplex.sizeBlockLevel[1] = sizeBlock;
            paramIntegrity.sizeBlockLevel[0] = sizeBlock;
            paramIntegrity.sizeBlockLevel[1] = sizeBlock;
            paramIntegrity.sizeBlockLevel[2] = sizeBlock;
            paramIntegrity.sizeBlockLevel[3] = sizeBlock;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
                    &sizeExpanded,
                    sizeBlock,
                    CountExpandMax,
                    paramDuplex,
                    paramIntegrity,
                    static_cast<uint32_t>(countDataBlockExpanded) * 2,
                    static_cast<uint32_t>(countReservedBlockExpanded) * 2
                )
            );
        }

        nnt::fs::util::Vector<char> buffer(sizeData);
        nnt::fs::util::Vector<char> work(64 * 1024);
        bool isDecompressSuccess = nn::util::DecompressGzip(
                                       &buffer[0],
                                       buffer.size(),
                                       CompatibleDataTableV4[indexTable].pData,
                                       CompatibleDataTableV4[indexTable].size,
                                       &work[0],
                                       work.size()
                                   );
        ASSERT_TRUE(isDecompressSuccess);

        nnt::fs::util::SafeMemoryStorage storageData(sizeExpanded);
        std::memcpy(storageData.GetBuffer(), &buffer[0], sizeData);
        nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver fileSystem;
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Initialize(
                nn::fs::SubStorage(&storageData, 0, sizeData),
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                MinimumVersion
            )
        );

        // 拡張データが読み込めることを確認
        {
            nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData extraData;
            fileSystem.ReadExtraData(&extraData);
        }

        // ルートディレクトリ直下のファイル読み込み
        {
            nn::fssystem::save::IDirectory* pDirectory = nullptr;
            nn::fssystem::save::Path pathDirectory;
            NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize("/"));
            NNT_ASSERT_RESULT_SUCCESS(
                fileSystem.OpenDirectory(
                    &pDirectory,
                    pathDirectory
                )
            );
            while( NN_STATIC_CONDITION(true) )
            {
                int32_t count;
                nn::fs::DirectoryEntry de;
                NNT_ASSERT_RESULT_SUCCESS(pDirectory->Read(&count, &de, 1));
                if( count == 0 )
                {
                    break;
                }
            }
            fileSystem.CloseDirectory(pDirectory);
        }

        // ファイルシステムを拡張します。
        {
            fileSystem.Finalize();

            nn::fs::SubStorage storageExpanded(&storageData, 0, sizeExpanded);

            const auto logStorageSize
                = nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryExpandLogSize(
                      sizeBlock,
                      static_cast<uint32_t>(countDataBlockExpanded),
                      static_cast<uint32_t>(countReservedBlockExpanded)
                  );
            nnt::fs::util::SafeMemoryStorage storageLog(logStorageSize);
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::OperateExpand(
                    storageExpanded,
                    nn::fs::SubStorage(&storageLog, 0, logStorageSize),
                    sizeBlock,
                    static_cast<uint32_t>(countDataBlockExpanded),
                    static_cast<uint32_t>(countReservedBlockExpanded),
                    GetBufferManager(),
                    nnt::fs::util::GetMacGenerator(),
                    nnt::fs::util::SaveDataMinimumVersion
                )
            );
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::CommitExpand(
                    storageExpanded,
                    nn::fs::SubStorage(&storageLog, 0, logStorageSize),
                    sizeBlock,
                    GetBufferManager()
                )
            );

            // マウントしてアクセスできることを確認します。
            NNT_ASSERT_RESULT_SUCCESS(
                fileSystem.Initialize(
                    storageExpanded,
                    GetBufferManager(),
                    nnt::fs::util::GetMacGenerator(),
                    MinimumVersion
                )
            );
            nn::fssystem::save::Path pathDirectory;
            NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize("/NewDirectory"));
            NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory(pathDirectory));

            // 変更内容をコミットして、再度マウント、アクセスできることを確認します。
            NNT_ASSERT_RESULT_SUCCESS(fileSystem.Commit());
            fileSystem.Finalize();

            NNT_ASSERT_RESULT_SUCCESS(
                fileSystem.Initialize(
                    storageExpanded,
                    GetBufferManager(),
                    nnt::fs::util::GetMacGenerator(),
                    MinimumVersion
                )
            );

            nn::fssystem::save::IDirectory* pDirectory = nullptr;
            NNT_ASSERT_RESULT_SUCCESS(
                fileSystem.OpenDirectory(
                    &pDirectory,
                    pathDirectory
                )
            );
            fileSystem.CloseDirectory(pDirectory);
        }
    }
} // NOLINT(impl/function_size)

TEST(FsJournalIntegritySaveDataFileSystemDriver, MinimumVersion)
{
    struct CompatibleDataTable
    {
        int version;
        const uint8_t* pData;
        size_t size;
    };
    const CompatibleDataTable CompatibleDataTableMix[] =
    {
        { 4, SaveData1_v4, sizeof(SaveData1_v4) },
        { 4, SaveData2_v4, sizeof(SaveData2_v4) },
        { 4, SaveData3_v4, sizeof(SaveData3_v4) },
        { 5, SaveData4_v5, sizeof(SaveData4_v5) },
    };

    // 初期状態では v4、v5 どちらのセーブデータも読み込める
    for( size_t indexTable = 0; indexTable < sizeof(CompatibleDataTableMix) / sizeof(CompatibleDataTable); ++indexTable)
    {
        const auto sizeBlock = 4 * 1024;
        const auto countDataBlockExpanded = 1024;
        const auto countReservedBlockExpanded = countDataBlockExpanded;

        size_t sizeData = nn::util::GetGzipDecompressedSize(
                              CompatibleDataTableMix[indexTable].pData,
                              CompatibleDataTableMix[indexTable].size
                          );

        int64_t sizeExpanded = 0;
        {
            auto CountExpandMax = 25;
            nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam paramDuplex;
            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam paramIntegrity;
            paramDuplex.sizeBlockLevel[0] = sizeBlock;
            paramDuplex.sizeBlockLevel[1] = sizeBlock;
            paramIntegrity.sizeBlockLevel[0] = sizeBlock;
            paramIntegrity.sizeBlockLevel[1] = sizeBlock;
            paramIntegrity.sizeBlockLevel[2] = sizeBlock;
            paramIntegrity.sizeBlockLevel[3] = sizeBlock;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
                    &sizeExpanded,
                    sizeBlock,
                    CountExpandMax,
                    paramDuplex,
                    paramIntegrity,
                    static_cast<uint32_t>(countDataBlockExpanded) * 2,
                    static_cast<uint32_t>(countReservedBlockExpanded) * 2
                )
            );
        }

        nnt::fs::util::Vector<char> buffer(sizeData);
        nnt::fs::util::Vector<char> work(64 * 1024);
        bool isDecompressSuccess = nn::util::DecompressGzip(
                                       &buffer[0],
                                       buffer.size(),
                                       CompatibleDataTableMix[indexTable].pData,
                                       CompatibleDataTableMix[indexTable].size,
                                       &work[0],
                                       work.size()
                                   );
        ASSERT_TRUE(isDecompressSuccess);

        nnt::fs::util::SafeMemoryStorage storageData(sizeExpanded);
        std::memcpy(storageData.GetBuffer(), &buffer[0], sizeData);
        nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver fileSystem;
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Initialize(
                nn::fs::SubStorage(&storageData, 0, sizeData),
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator(),
                4
            )
        );

        // 拡張データが読み込めることを確認
        {
            nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData extraData;
            fileSystem.ReadExtraData(&extraData);
        }

        // ルートディレクトリ直下のファイル読み込み
        {
            nn::fssystem::save::IDirectory* pDirectory = nullptr;
            nn::fssystem::save::Path pathDirectory;
            NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize("/"));
            NNT_ASSERT_RESULT_SUCCESS(
                fileSystem.OpenDirectory(
                    &pDirectory,
                    pathDirectory
                )
            );
            while( NN_STATIC_CONDITION(true) )
            {
                int32_t count;
                nn::fs::DirectoryEntry de;
                NNT_ASSERT_RESULT_SUCCESS(pDirectory->Read(&count, &de, 1));
                if( count == 0 )
                {
                    break;
                }
            }
            fileSystem.CloseDirectory(pDirectory);
        }
    }

    // 最低バージョンを 5 とすると v4 セーブデータは読み込めなくなる
    for( size_t indexTable = 0; indexTable < sizeof(CompatibleDataTableMix) / sizeof(CompatibleDataTable); ++indexTable)
    {
        const auto sizeBlock = 4 * 1024;
        const auto countDataBlockExpanded = 1024;
        const auto countReservedBlockExpanded = countDataBlockExpanded;

        size_t sizeData = nn::util::GetGzipDecompressedSize(
                              CompatibleDataTableMix[indexTable].pData,
                              CompatibleDataTableMix[indexTable].size
                          );

        int64_t sizeExpanded = 0;
        {
            auto CountExpandMax = 25;
            nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam paramDuplex;
            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam paramIntegrity;
            paramDuplex.sizeBlockLevel[0] = sizeBlock;
            paramDuplex.sizeBlockLevel[1] = sizeBlock;
            paramIntegrity.sizeBlockLevel[0] = sizeBlock;
            paramIntegrity.sizeBlockLevel[1] = sizeBlock;
            paramIntegrity.sizeBlockLevel[2] = sizeBlock;
            paramIntegrity.sizeBlockLevel[3] = sizeBlock;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
                    &sizeExpanded,
                    sizeBlock,
                    CountExpandMax,
                    paramDuplex,
                    paramIntegrity,
                    static_cast<uint32_t>(countDataBlockExpanded) * 2,
                    static_cast<uint32_t>(countReservedBlockExpanded) * 2
                )
            );
        }

        nnt::fs::util::Vector<char> buffer(sizeData);
        nnt::fs::util::Vector<char> work(64 * 1024);
        bool isDecompressSuccess = nn::util::DecompressGzip(
                                       &buffer[0],
                                       buffer.size(),
                                       CompatibleDataTableMix[indexTable].pData,
                                       CompatibleDataTableMix[indexTable].size,
                                       &work[0],
                                       work.size()
                                   );
        ASSERT_TRUE(isDecompressSuccess);

        nnt::fs::util::SafeMemoryStorage storageData(sizeExpanded);
        std::memcpy(storageData.GetBuffer(), &buffer[0], sizeData);
        nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver fileSystem;

        // 最低バージョン未満のセーブデータはマウントできません。
        if( CompatibleDataTableMix[indexTable].version == 4 )
        {
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultUnsupportedVersion,
                fileSystem.Initialize(
                    nn::fs::SubStorage(&storageData, 0, sizeData),
                    GetBufferManager(),
                    nnt::fs::util::GetMacGenerator(),
                    5
                )
            );
            fileSystem.Finalize();
        }
        else
        {
            NNT_ASSERT_RESULT_SUCCESS(
                fileSystem.Initialize(
                    nn::fs::SubStorage(&storageData, 0, sizeData),
                    GetBufferManager(),
                    nnt::fs::util::GetMacGenerator(),
                    5
                )
            );
        }

        // 最低バージョン未満のセーブデータは拡張きません。
        {
            nn::fs::SubStorage storageExpanded(&storageData, 0, sizeExpanded);

            const auto logStorageSize
                = nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryExpandLogSize(
                      sizeBlock,
                      static_cast<uint32_t>(countDataBlockExpanded),
                      static_cast<uint32_t>(countReservedBlockExpanded)
                  );
            nnt::fs::util::SafeMemoryStorage storageLog(logStorageSize);
            if( CompatibleDataTableMix[indexTable].version == 4 )
            {
                NNT_ASSERT_RESULT_FAILURE(
                    nn::fs::ResultUnsupportedVersion,
                    nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::OperateExpand(
                        storageExpanded,
                        nn::fs::SubStorage(&storageLog, 0, logStorageSize),
                        sizeBlock,
                        static_cast<uint32_t>(countDataBlockExpanded),
                        static_cast<uint32_t>(countReservedBlockExpanded),
                        GetBufferManager(),
                        nnt::fs::util::GetMacGenerator(),
                        5
                    )
                );
            }
            else
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::OperateExpand(
                        storageExpanded,
                        nn::fs::SubStorage(&storageLog, 0, logStorageSize),
                        sizeBlock,
                        static_cast<uint32_t>(countDataBlockExpanded),
                        static_cast<uint32_t>(countReservedBlockExpanded),
                        GetBufferManager(),
                        nnt::fs::util::GetMacGenerator(),
                        5
                    )
                );
            }
        }

        // 最低バージョン未満のセーブデータはマスターヘッダの修復もできません。
        {
            int64_t sizeNewData;
            NNT_ASSERT_RESULT_SUCCESS(storageData.GetSize(&sizeNewData));
            nn::fs::SubStorage storageNewData(&storageData, 0, sizeNewData);
            if( CompatibleDataTableMix[indexTable].version == 4 )
            {
                NNT_ASSERT_RESULT_FAILURE(
                    nn::fs::ResultUnsupportedVersion,
                    nn::fssystem::save::JournalIntegritySaveDataFileSystem::RecoverMasterHeader(
                        storageNewData,
                        GetBufferManager(),
                        nnt::fs::util::GetMacGenerator(),
                        5
                    )
                );
            }
            else
            {
                NNT_EXPECT_RESULT_SUCCESS(
                    nn::fssystem::save::JournalIntegritySaveDataFileSystem::RecoverMasterHeader(
                        storageNewData,
                        GetBufferManager(),
                        nnt::fs::util::GetMacGenerator(),
                        5
                    )
                );
            }
        }
    }
} // NOLINT(impl/function_size)
