﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <numeric>
#include <memory>

#include <nn/nn_Common.h>
#include <nn/fssystem/save/fs_IntegritySaveDataStorage.h>

#include <nnt/nntest.h>
#include <nnt/fsUtil/testFs_util.h>

#include "testFs_util_CommonFileSystemTests.h"

namespace nn { namespace fssystem { namespace save {

namespace
{
struct TestParam
{
    uint32_t countBlock;            // ブロック数
    int64_t sizeBlock;              // ブロックサイズ
    size_t verificationInputParam0; // 完全性検証のハッシュストレージのサイズレベル0
    size_t verificationInputParam1; // 完全性検証のハッシュストレージのサイズレベル1
    size_t verificationInputParam2; // 完全性検証のハッシュストレージのサイズレベル2
    size_t verificationInputParam3; // 完全性検証のハッシュストレージのサイズレベル3
    int maxCacheCount;              // 完全性検証用キャッシュバッファの最大キャッシュ数
    size_t cacheBufferPageSize;     // 完全性検証用キャッシュバッファのページサイズ
    size_t cacheBufferMaxOrder;     // 完全性検証用キャッシュバッファの最大キャッシュサイズのオーダー

    int64_t GetEntrySize() const NN_NOEXCEPT
    {
        return countBlock * sizeBlock;
    }

    int64_t GetStorageSize() const NN_NOEXCEPT
    {
        return GetEntrySize();
    }

    size_t GetCacheSize() const NN_NOEXCEPT
    {
        return cacheBufferPageSize * maxCacheCount;
    }
};

nnt::fs::util::Vector<TestParam> MakeTestParam() NN_NOEXCEPT
{
    nnt::fs::util::Vector<TestParam> testParams;

    {
        TestParam data;
        data.countBlock = 128;
        data.sizeBlock = 128;
        data.verificationInputParam0 = 1 << 10;
        data.verificationInputParam1 = 1 << 10;
        data.verificationInputParam2 = 1 << 10;
        data.verificationInputParam3 = 1 << 11;
        data.maxCacheCount = 1024;
        data.cacheBufferPageSize = 4096;
        data.cacheBufferMaxOrder = 4;
        testParams.push_back(data);
    }

    {
        TestParam data;
        data.countBlock = 512;
        data.sizeBlock = 512;
        data.verificationInputParam0 = 1 << 12;
        data.verificationInputParam1 = 1 << 12;
        data.verificationInputParam2 = 1 << 12;
        data.verificationInputParam3 = 1 << 12;
        data.maxCacheCount = 1024;
        data.cacheBufferPageSize = 4096;
        data.cacheBufferMaxOrder = 4;
        testParams.push_back(data);
    }

    {
        TestParam data;
        data.countBlock = 256;
        data.sizeBlock = 1024;
        data.verificationInputParam0 = 1 << 10;
        data.verificationInputParam1 = 1 << 11;
        data.verificationInputParam2 = 1 << 12;
        data.verificationInputParam3 = 1 << 13;
        data.maxCacheCount = 1024;
        data.cacheBufferPageSize = 4096;
        data.cacheBufferMaxOrder = 8;
        testParams.push_back(data);
    }
#ifdef NN_BUILD_CONFIG_OS_WIN
    {
        TestParam data;
        data.countBlock = 1024;
        data.sizeBlock = 1024;
        data.verificationInputParam0 = 1 << 10;
        data.verificationInputParam1 = 1 << 10;
        data.verificationInputParam2 = 1 << 10;
        data.verificationInputParam3 = 1 << 10;
        data.maxCacheCount = 2048;
        data.cacheBufferPageSize = 4096;
        data.cacheBufferMaxOrder = 12;
        testParams.push_back(data);
    }
#endif
    return testParams;
}

nnt::fs::util::Vector<TestParam> MakeLargeTestParam() NN_NOEXCEPT
{
    nnt::fs::util::Vector<TestParam> testParams;

    {
        TestParam data;
        data.countBlock = 8 * 1024 * 1024;
        data.sizeBlock = 16 * 1024;
        data.verificationInputParam0 = 16 * 1024;
        data.verificationInputParam1 = 16 * 1024;
        data.verificationInputParam2 = 16 * 1024;
        data.verificationInputParam3 = 16 * 1024;
        data.maxCacheCount = 4 * 1024;
        data.cacheBufferPageSize = 16 * 1024;
        data.cacheBufferMaxOrder = 8;
        testParams.push_back(data);
    }

    return testParams;
}

}

template<typename TBaseStorage, typename THashStorage>
class FsIntegritySaveDataStorageTestTemplate : public ::testing::TestWithParam<TestParam>
{
public:
    FsIntegritySaveDataStorageTestTemplate() NN_NOEXCEPT
        : m_BufferManagerSet(),
        m_VerificationInfo(),
        m_Locker(true)
    {
    }

    // テストのセットアップ
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        const TestParam& param = GetParam();
        NNT_ASSERT_RESULT_SUCCESS(FormatStorage(param));
        NNT_ASSERT_RESULT_SUCCESS(MountStorage());
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        m_SaveDataStorage.Finalize();
    }

    Result FormatStorage(const TestParam& param) NN_NOEXCEPT
    {
        nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam inputParamIntegrity;
        inputParamIntegrity.sizeBlockLevel[0] = param.verificationInputParam0;
        inputParamIntegrity.sizeBlockLevel[1] = param.verificationInputParam1;
        inputParamIntegrity.sizeBlockLevel[2] = param.verificationInputParam2;
        inputParamIntegrity.sizeBlockLevel[3] = param.verificationInputParam3;

        const int64_t entrySize = param.GetEntrySize();

        // 初期化に必要な値を QuerySize で取得します
        nn::fssystem::save::HierarchicalIntegrityVerificationSizeSet sizeSet = { 0 };
        NN_RESULT_DO(IntegritySaveDataStorage::QuerySize(
            &sizeSet,
            inputParamIntegrity,
            entrySize
        ));

        // ハッシュ検証のためのメタデータを設定します
        nn::fssystem::save::HierarchicalIntegrityVerificationMetaInformation metaInfo = { 0 };
        nn::fs::SubStorage verificationControlAreaStorage;
        {
            const auto bodySize = param.GetStorageSize() + sizeSet.layeredHashSizes[0] + sizeSet.layeredHashSizes[1] + sizeSet.layeredHashSizes[2];
            m_StorageVerificationControlArea.Initialize(static_cast<size_t>(sizeSet.controlSize));
            {
                int64_t offset = 0;
                metaInfo.Format();
                metaInfo.sizeMasterHash = static_cast<uint32_t>(sizeSet.masterHashSize);
                metaInfo.infoLevelHash.maxLayers = 5;
                metaInfo.infoLevelHash.info[3].size = bodySize;
                for( auto level = 0; level < 3; ++level )
                {
                    const auto size = sizeSet.layeredHashSizes[level];
                    metaInfo.infoLevelHash.info[level].offset = offset;
                    metaInfo.infoLevelHash.info[level].size = size;
                    metaInfo.infoLevelHash.info[level].orderBlock
                        = nn::fssystem::save::ILog2(
                            static_cast<uint32_t>(inputParamIntegrity.sizeBlockLevel[level]));
                    metaInfo.infoLevelHash.info[3].size
                        = metaInfo.infoLevelHash.info[3].size - size;
                    offset += size;
                }
                metaInfo.infoLevelHash.info[3].offset = offset;
                metaInfo.infoLevelHash.info[3].orderBlock
                    = nn::fssystem::save::ILog2(
                        static_cast<uint32_t>(inputParamIntegrity.sizeBlockLevel[3]));
            }
            verificationControlAreaStorage = fs::SubStorage(&m_StorageVerificationControlArea, 0, sizeSet.controlSize);
            NN_RESULT_DO(
                nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::Format(
                    verificationControlAreaStorage,
                    metaInfo
                )
            );
            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea controlArea;
            NN_RESULT_DO(controlArea.Initialize(verificationControlAreaStorage));
            controlArea.GetLevelHashInfo(&m_VerificationInfo);
        }

        // 各ストレージの初期化します
        m_StorageBase.Initialize(entrySize);
        m_StorageMasterHash.Initialize(sizeSet.masterHashSize);
        m_StorageHash.Initialize(m_VerificationInfo.info[0].size + m_VerificationInfo.info[1].size + m_VerificationInfo.info[2].size + m_VerificationInfo.info[3].size);

        // 各ストレージをサブストレージ化します
        nn::fs::SubStorage storageMasterHash(&m_StorageMasterHash, 0, sizeSet.masterHashSize);
        nn::fs::SubStorage subStorageData(&m_StorageBase, 0, entrySize);
        nn::fs::SubStorage storageLayeredHashL1(&m_StorageHash, m_VerificationInfo.info[0].offset, m_VerificationInfo.info[0].size);
        nn::fs::SubStorage storageLayeredHashL2(&m_StorageHash, m_VerificationInfo.info[1].offset, m_VerificationInfo.info[1].size);
        nn::fs::SubStorage storageLayeredHashL3(&m_StorageHash, m_VerificationInfo.info[2].offset, m_VerificationInfo.info[2].size);

        // 初期化用にサブストレージをコピーします
        m_SubStorageVerificationMasterHash = storageMasterHash;
        m_SubStorageVerificationLayeredHashL1 = storageLayeredHashL1;
        m_SubStorageVerificationLayeredHashL2 = storageLayeredHashL2;
        m_SubStorageVerificationLayeredHashL3 = storageLayeredHashL3;
        m_SubStorageData = subStorageData;

        InitializeBuffer(param);

        // フォーマット
        NN_RESULT_DO(m_SaveDataStorage.Format(
            verificationControlAreaStorage,
            storageMasterHash,
            metaInfo
        ));

        NN_RESULT_SUCCESS;
    } // NOLINT(impl/function_size)

    // 完全性検証ハッシュに割り当てるキャッシュバッファを設定
    void InitializeBuffer(const TestParam& param) NN_NOEXCEPT
    {
        const size_t cacheBufferMax = static_cast<size_t>(1) << (param.cacheBufferMaxOrder - 1);
        const size_t sizeUnitBlock = cacheBufferMax * param.cacheBufferPageSize;
        const size_t sizeCacheBuffer = sizeUnitBlock * nn::fssystem::save::IntegrityLayerCountSave;
        m_CacheBuffer.reset(new char[sizeCacheBuffer]);
        {
            NNT_ASSERT_RESULT_SUCCESS(m_BufferManager.Initialize(
                param.maxCacheCount,
                reinterpret_cast<uintptr_t>(m_CacheBuffer.get()),
                sizeCacheBuffer,
                param.cacheBufferPageSize
            ));
        }
        for( size_t i = 0; i < IntegrityLayerCountSave; ++i )
        {
            m_BufferManagerSet.pBuffer[i] = &m_BufferManager;
        }
    }

    // 初期化
    Result MountStorage() NN_NOEXCEPT
    {
        NN_RESULT_DO(m_SaveDataStorage.Initialize(
            m_SubStorageVerificationMasterHash,
            m_SubStorageVerificationLayeredHashL1,
            m_SubStorageVerificationLayeredHashL2,
            m_SubStorageVerificationLayeredHashL3,
            m_SubStorageData,
            m_VerificationInfo,
            &m_BufferManagerSet,
            &m_Locker
        ));
        NN_RESULT_SUCCESS;
    }

    void UnmountStorage() NN_NOEXCEPT
    {
        m_SaveDataStorage.Finalize();
    }

    Result ReadStorage(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_SaveDataStorage.Read(offset, buffer, size));
        NN_RESULT_SUCCESS;
    }
    Result WriteStorage(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_SaveDataStorage.Write(offset, buffer, size));
        NN_RESULT_SUCCESS;
    }
    Result OperateStorage(
        nn::fs::OperationId operationId,
        int64_t offset,
        int64_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_SaveDataStorage.OperateRange(operationId, offset, size));
        NN_RESULT_SUCCESS;
    }
    Result CommitStorage() NN_NOEXCEPT
    {
        NN_RESULT_DO(m_SaveDataStorage.Commit());
        NN_RESULT_SUCCESS;
    }
    void ResetBuffer() const NN_NOEXCEPT
    {
        nn::fssystem::FileSystemBufferManager* bufferManager
            = reinterpret_cast<nn::fssystem::FileSystemBufferManager*>(m_BufferManagerSet.pBuffer[0]);
        bufferManager->Finalize();
    }
    fs::SubStorage* GetSubStorageMasterHash() NN_NOEXCEPT
    {
        return &m_SubStorageVerificationMasterHash;
    }
    void ResetAccessCounter() NN_NOEXCEPT
    {
        m_StorageMasterHash.ResetAccessCounter();
        m_StorageHash.ResetAccessCounter();
    }
    bool IsWriteCounterZero() const NN_NOEXCEPT
    {
        int writeCount = m_StorageMasterHash.GetWriteTimes() + m_StorageHash.GetWriteTimes();
        return writeCount == 0;
    }
    static bool IsWriteHashExpected(size_t sizeWrite) NN_NOEXCEPT
    {
        return sizeWrite >= 512 * 1024;
    }

    nn::fs::IStorage* GetStorage() NN_NOEXCEPT
    {
        return &m_SaveDataStorage;
    }

private:
    // ストレージ本体
    IntegritySaveDataStorage m_SaveDataStorage;

    // 下に敷くストレージ
    // データストレージ
    TBaseStorage m_StorageBase;
    // 署名検証のマスターハッシュ
    nnt::fs::util::AccessCountedMemoryStorage m_StorageMasterHash;
    // 署名検証のハッシュレイヤ x 3
    THashStorage m_StorageHash;
    // 署名検証のコントロール情報
    nnt::fs::util::SafeMemoryStorage m_StorageVerificationControlArea;

    // バッファマネージャ
    FilesystemBufferManagerSet m_BufferManagerSet;
    nn::fssystem::FileSystemBufferManager m_BufferManager;
    std::unique_ptr<char[]> m_CacheBuffer;

    // マウント用の値
    fs::SubStorage m_SubStorageVerificationMasterHash;
    fs::SubStorage m_SubStorageVerificationLayeredHashL1;
    fs::SubStorage m_SubStorageVerificationLayeredHashL2;
    fs::SubStorage m_SubStorageVerificationLayeredHashL3;
    fs::SubStorage m_SubStorageData;
    HierarchicalIntegrityVerificationInformation m_VerificationInfo;
    nn::os::Mutex m_Locker;
};

typedef FsIntegritySaveDataStorageTestTemplate<nnt::fs::util::SafeMemoryStorage, nnt::fs::util::AccessCountedMemoryStorage> FsIntegritySaveDataStorageTest;
typedef FsIntegritySaveDataStorageTestTemplate<nnt::fs::util::VirtualMemoryStorage, nnt::fs::util::VirtualMemoryStorage> FsIntegritySaveDataStorageLargeTest;

/**
* 成功パターンテスト
*/
TEST_P(FsIntegritySaveDataStorageTest, TestSimple)
{
    int offset = 20;
    int size = 40;

    std::unique_ptr<char[]> writeBuffer(new char[size]);
    std::unique_ptr<char[]> readBuffer(new char[size]);
    char* pWriteBuffer = writeBuffer.get();
    char* pReadBuffer = readBuffer.get();
    std::memset(pWriteBuffer, 0xFF, size);

    // 書き込んでコミット
    NNT_ASSERT_RESULT_SUCCESS(WriteStorage(offset + 10, pWriteBuffer, size - 10));
    NNT_ASSERT_RESULT_SUCCESS(WriteStorage(offset + 5, pWriteBuffer, size - 5));
    NNT_ASSERT_RESULT_SUCCESS(WriteStorage(offset, pWriteBuffer, size));
    NNT_ASSERT_RESULT_SUCCESS(WriteStorage(offset, pWriteBuffer, size));
    NNT_ASSERT_RESULT_SUCCESS(CommitStorage());

    // 書き込んだものが読み込めるかテスト
    NNT_ASSERT_RESULT_SUCCESS(ReadStorage(offset + 8, pReadBuffer, size - 8));
    NNT_ASSERT_RESULT_SUCCESS(ReadStorage(offset, pReadBuffer, size));
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(pReadBuffer, pWriteBuffer, size);

    // 書き込んだ後、コミットしません
    NNT_ASSERT_RESULT_SUCCESS(WriteStorage(offset + 1, pWriteBuffer, size - 1));
    NNT_ASSERT_RESULT_SUCCESS(WriteStorage(offset + 5, pWriteBuffer, size - 5));
    NNT_ASSERT_RESULT_SUCCESS(ReadStorage(offset, pReadBuffer, size));

    // 最初に書き込んだものが読めるかテスト
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(pReadBuffer, pWriteBuffer, size);
}

/**
* コミットしなくても下位ストレージにアクセスすることを確かめます
*/
TEST_P(FsIntegritySaveDataStorageTest, TestAccessCountSingleWrite)
{
    const TestParam& param = GetParam();
    int64_t entrySize = param.GetEntrySize();
    size_t bufferSize = static_cast<size_t>(entrySize / 2);
    std::unique_ptr<char[]> buffer(new char[bufferSize]);
    std::iota(buffer.get(), buffer.get() + bufferSize, '\x1');
    int64_t offsetMax = entrySize - bufferSize;
    for( int i = 0; i < 3; ++i )
    {
        int64_t offset = offsetMax * i / 2;

        // Write を行ったときのアクセスの有無をチェック
        {
            ResetAccessCounter();
            NNT_ASSERT_RESULT_SUCCESS(
                WriteStorage(
                    offset,
                    buffer.get(),
                    bufferSize
                )
            );

            // 条件によってはハッシュが更新される
            if( IsWriteHashExpected(bufferSize) )
            {
                ASSERT_FALSE(IsWriteCounterZero());
            }
            else
            {
                ASSERT_TRUE(IsWriteCounterZero());
            }
        }

        // コミットすることで下位ストレージを書き換えます
        {
            ResetAccessCounter();
            NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
            ASSERT_FALSE(IsWriteCounterZero());
        }

        // 初期化し直します
        UnmountStorage();
        NNT_ASSERT_RESULT_SUCCESS(MountStorage());
    }
}

/**
* コミットするまで下位ストレージにアクセスしないことを確かめます
*/
TEST_P(FsIntegritySaveDataStorageTest, TestAccessCountMultipleWrite)
{
    const TestParam& param = GetParam();
    int64_t entrySize = param.GetEntrySize();
    size_t bufferSize = static_cast<size_t>(entrySize / 2);
    std::unique_ptr<char[]> buffer(new char[bufferSize]);
    std::iota(buffer.get(), buffer.get() + bufferSize, '\x1');
    int64_t offsetMax = entrySize - bufferSize;
    for( int i = 0; i < 3; ++i )
    {
        int64_t offset = offsetMax * i / 2;

        // Write を行ったときのアクセスの有無をチェック
        {
            ResetAccessCounter();
            NNT_ASSERT_RESULT_SUCCESS(
                WriteStorage(offset, buffer.get(), bufferSize / 4)
            );
            offset += bufferSize / 4;
            NNT_ASSERT_RESULT_SUCCESS(
                WriteStorage(offset, buffer.get(), bufferSize / 4)
            );
            offset += bufferSize / 4;
            NNT_ASSERT_RESULT_SUCCESS(
                WriteStorage(offset, buffer.get(), bufferSize / 4)
            );
            offset += bufferSize / 4;
            NNT_ASSERT_RESULT_SUCCESS(
                WriteStorage(offset, buffer.get(), bufferSize / 4)
            );
            if( IsWriteHashExpected(bufferSize) )
            {
                ASSERT_FALSE(IsWriteCounterZero());
            }
            else
            {
                ASSERT_TRUE(IsWriteCounterZero());
            }
        }

        // コミットすることで下位ストレージを書き換えます
        {
            ResetAccessCounter();
            NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
            ASSERT_FALSE(IsWriteCounterZero());
        }

        // 初期化し直す
        UnmountStorage();
        NNT_ASSERT_RESULT_SUCCESS(MountStorage());
    }
}

/**
* 0 バイト書き込みでは、コミットしても下位ストレージにアクセスしないことを確かめます
*/
TEST_P(FsIntegritySaveDataStorageTest, TestAccessCountZeroWrite)
{
    const TestParam& param = GetParam();
    int64_t storageSize = param.GetStorageSize();
    std::unique_ptr<char[]> buffer(new char[1]);
    int64_t offsetMax = storageSize;
    for( int i = 0; i < 3; ++i )
    {
        int64_t offset = offsetMax * i / 2;
        ResetAccessCounter();
        NNT_ASSERT_RESULT_SUCCESS(
            WriteStorage(
                offset,
                buffer.get(),
                0
            )
        );
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        ASSERT_TRUE(IsWriteCounterZero());
    }
}

/**
* マウント直後のコミット等で、下位ストレージにアクセスしないことを確かめます
*/
TEST_P(FsIntegritySaveDataStorageTest, TestUnmount)
{
    const TestParam& param = GetParam();
    int64_t entrySize = param.GetEntrySize();
    size_t bufferSize = static_cast<size_t>(entrySize / 2);
    std::unique_ptr<char[]> buffer(new char[bufferSize]);
    std::iota(buffer.get(), buffer.get() + bufferSize, '\x1');

    // Write を行うだけでは、下位ストレージにアクセスしません
    int64_t offsetMax = entrySize - bufferSize;
    for( int i = 0; i < 3; ++i )
    {
        int64_t offset = offsetMax * i / 2;

        NNT_ASSERT_RESULT_SUCCESS(
            WriteStorage(offset, buffer.get(), bufferSize / 4)
        );
        offset += bufferSize / 4;
        NNT_ASSERT_RESULT_SUCCESS(
            WriteStorage(offset, buffer.get(), bufferSize / 4)
        );
        offset += bufferSize / 4;
        NNT_ASSERT_RESULT_SUCCESS(
            WriteStorage(offset, buffer.get(), bufferSize / 4)
        );
        offset += bufferSize / 4;
        NNT_ASSERT_RESULT_SUCCESS(
            WriteStorage(offset, buffer.get(), bufferSize / 4)
        );

        // 書き込みを行ってからアンマウントすると、
        // コミットが行われるためアクセスが発生します
        ResetAccessCounter();
        UnmountStorage();
        ASSERT_FALSE(IsWriteCounterZero());

        // マウント時に下位ストレージにアクセスが発生しません
        ResetAccessCounter();
        NNT_ASSERT_RESULT_SUCCESS(MountStorage());
        ASSERT_TRUE(IsWriteCounterZero());

        // マウント直後にコミットしても下位ストレージにアクセスが発生しません
        ResetAccessCounter();
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        ASSERT_TRUE(IsWriteCounterZero());

        // コミット直後のアンマウント時はアクセスが発生しません
        ResetAccessCounter();
        UnmountStorage();
        ASSERT_TRUE(IsWriteCounterZero());

        NNT_ASSERT_RESULT_SUCCESS(MountStorage());
    }
}

/**
* Write した値を Commit した後 Read で読み取ります
*/
TEST_P(FsIntegritySaveDataStorageTest, TestReadWrite)
{
    const TestParam& param = GetParam();

    int64_t entrySize = param.GetEntrySize();
    size_t bufferSize = static_cast<size_t>(entrySize / 2);
    int64_t offsetMax = entrySize - bufferSize;

    for( int i = 0; i < 3; ++i )
    {
        int64_t offset = offsetMax * i / 2;

        // 書き込み
        std::unique_ptr<char[]> writeBuffer(new char[bufferSize]);
        std::iota(writeBuffer.get(), writeBuffer.get() + bufferSize, static_cast<char>(i));
        NNT_ASSERT_RESULT_SUCCESS(WriteStorage(offset, writeBuffer.get(), bufferSize));

        // コミット
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());

        // 読み込み
        std::unique_ptr<char[]> readBuffer(new char[bufferSize]);
        NNT_ASSERT_RESULT_SUCCESS(ReadStorage(offset, readBuffer.get(), bufferSize));
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(writeBuffer.get(), readBuffer.get(), bufferSize);
    }
}

/**
* ストレージ全体に書き込みを行い、コミットし、それが Read できるか確かめます
*/
TEST_P(FsIntegritySaveDataStorageTest, TestReadWriteEntire)
{
    const TestParam& param = GetParam();

    size_t entrySize = static_cast<size_t>(param.GetEntrySize());
    size_t bufferSize = entrySize / 2;

    std::unique_ptr<char[]> writeBuffer(new char[entrySize]);
    std::iota(writeBuffer.get(), writeBuffer.get() + entrySize, '\x1');

    // ストレージ全体に書き込みを行います
    {
        size_t writtenSize = 0;
        for( ; writtenSize < entrySize - bufferSize; writtenSize += bufferSize )
        {
            NNT_ASSERT_RESULT_SUCCESS(
                WriteStorage(
                    writtenSize,
                    writeBuffer.get() + writtenSize,
                    static_cast<size_t>(bufferSize)
                )
            );
            NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        }
        size_t remainSize = entrySize - writtenSize;
        NNT_ASSERT_RESULT_SUCCESS(
            WriteStorage(writtenSize, writeBuffer.get() + writtenSize, remainSize)
        );
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
    }

    // 読み込み
    std::unique_ptr<char[]> readBuffer(new char[entrySize]);
    NNT_ASSERT_RESULT_SUCCESS(ReadStorage(0, readBuffer.get(), entrySize));
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(writeBuffer.get(), readBuffer.get(), entrySize);
}

/**
* ストレージ全体に書き込みを行い、コミットし、再マウントしてから Read できるか確かめます
*/
TEST_P(FsIntegritySaveDataStorageTest, TestReadWriteEntireRemount)
{
    const TestParam& param = GetParam();

    size_t entrySize = static_cast<size_t>(param.GetEntrySize());
    size_t bufferSize = entrySize / 2;

    std::unique_ptr<char[]> writeBuffer(new char[entrySize]);
    std::iota(writeBuffer.get(), writeBuffer.get() + entrySize, '\x1');

    // ストレージ全体に書き込みを行います
    {
        size_t writtenSize = 0;
        for( ; writtenSize < entrySize - bufferSize; writtenSize += bufferSize )
        {
            NNT_ASSERT_RESULT_SUCCESS(
                WriteStorage(writtenSize, writeBuffer.get() + writtenSize, bufferSize)
            );
            NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        }
        size_t remainSize = entrySize - writtenSize;
        NNT_ASSERT_RESULT_SUCCESS(
            WriteStorage(writtenSize, writeBuffer.get() + writtenSize, remainSize)
        );
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
    }

    // 再マウント
    UnmountStorage();
    NNT_ASSERT_RESULT_SUCCESS(MountStorage());

    // 読み込み
    std::unique_ptr<char[]> readBuffer(new char[entrySize]);
    NNT_ASSERT_RESULT_SUCCESS(ReadStorage(0, readBuffer.get(), entrySize));
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(writeBuffer.get(), readBuffer.get(), entrySize);
}

/**
* 書き込み直後に読み込みを行ったときの値を確かめます
*/
TEST_P(FsIntegritySaveDataStorageTest, TestReadWriteWithoutCommit)
{
    const TestParam& param = GetParam();

    size_t entrySize = static_cast<size_t>(param.GetEntrySize());
    size_t bufferSize = entrySize / 2;

    // ストレージ全体に事前に書き込みを行います
    {
        std::unique_ptr<char[]> writeBuffer(new char[entrySize]);
        std::iota(writeBuffer.get(), writeBuffer.get() + entrySize, '\x1');
        size_t writtenSize = 0;
        for( ; writtenSize < entrySize - bufferSize; writtenSize += bufferSize )
        {
            NNT_ASSERT_RESULT_SUCCESS(
                WriteStorage(writtenSize, writeBuffer.get() + writtenSize, bufferSize)
            );
            NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        }
        size_t remainSize = entrySize - writtenSize;
        NNT_ASSERT_RESULT_SUCCESS(
            WriteStorage(writtenSize, writeBuffer.get() + writtenSize, remainSize)
        );
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
    }

    int64_t offsetMax = entrySize - bufferSize;
    for( int i = 0; i < 3; ++i )
    {
        // 書き込み
        int64_t offset = offsetMax * i / 2;
        std::unique_ptr<char[]> writeBuffer(new char[bufferSize]);
        std::iota(writeBuffer.get(), writeBuffer.get() + bufferSize, '\xA');
        NNT_ASSERT_RESULT_SUCCESS(WriteStorage(offset, writeBuffer.get(), bufferSize));

        // 読み込み
        std::unique_ptr<char[]> readBuffer(new char[entrySize]);
        NNT_ASSERT_RESULT_SUCCESS(ReadStorage(0, readBuffer.get(), entrySize));
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(writeBuffer.get(), readBuffer.get() + offset, bufferSize);

        // コミットして再マウント
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        UnmountStorage();
        NNT_ASSERT_RESULT_SUCCESS(MountStorage());
    }
}

/**
* とびとびで Write した後 Commit してから Read を行います
*/
TEST_P(FsIntegritySaveDataStorageTest, TestReadWriteSkipping)
{
    const TestParam& param = GetParam();

    size_t entrySize = static_cast<size_t>(param.GetEntrySize());
    size_t bufferSize = entrySize / 2;

    // いったん全体に書き込んでおきます
    {
        std::unique_ptr<char[]> writeBuffer(new char[entrySize]);
        std::iota(writeBuffer.get(), writeBuffer.get() + entrySize, '\x1');
        size_t writtenSize = 0;
        for( ; writtenSize < entrySize - bufferSize; writtenSize += bufferSize )
        {
            NNT_ASSERT_RESULT_SUCCESS(
                WriteStorage(writtenSize, writeBuffer.get() + writtenSize, bufferSize)
            );
            NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        }
        size_t remainSize = entrySize - writtenSize;
        NNT_ASSERT_RESULT_SUCCESS(
            WriteStorage(writtenSize, writeBuffer.get() + writtenSize, remainSize)
        );
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
    }

    // skipSize ずつとばしながら writeSize ずつ書き込みます
    const int64_t skipSize = param.verificationInputParam3;
    const size_t writeSize = param.verificationInputParam3;

    // 比較用のデータも作成
    std::unique_ptr<char[]> compareBuffer(new char[entrySize]);
    std::iota(compareBuffer.get(), compareBuffer.get() + entrySize, '\x1');

    int64_t offset = skipSize;
    int64_t writtenSize = 0;
    std::unique_ptr<char[]> writeBuffer(new char[writeSize]);
    std::iota(writeBuffer.get(), writeBuffer.get() + writeSize, '\xA');
    for( ;; )
    {
        NNT_ASSERT_RESULT_SUCCESS(WriteStorage(offset, writeBuffer.get(), writeSize));
        std::memcpy(compareBuffer.get() + offset, writeBuffer.get(), writeSize);
        offset += writeSize + skipSize;
        writtenSize += writeSize;
        if( writtenSize + writeSize > bufferSize
            || offset + writeSize > entrySize )
        {
            break;
        }
    }
    NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
    UnmountStorage();
    NNT_ASSERT_RESULT_SUCCESS(MountStorage());

    // 読み込まれる値は書き込んだ値と等しくなります
    {
        std::unique_ptr<char[]> readBuffer(new char[entrySize]);
        NNT_ASSERT_RESULT_SUCCESS(ReadStorage(0, readBuffer.get(), entrySize));
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(compareBuffer.get(), readBuffer.get(), entrySize);
    }
}

/**
* 重複させながら Write した後 commit してから Read を行います
*/
TEST_P(FsIntegritySaveDataStorageTest, TestReadWriteOverlap)
{
    const TestParam& param = GetParam();

    size_t entrySize = static_cast<size_t>(param.GetEntrySize());
    size_t bufferSize = entrySize / 2;

    // いったん全体に書き込んでおきます
    {
        std::unique_ptr<char[]> writeBuffer(new char[entrySize]);
        std::iota(writeBuffer.get(), writeBuffer.get() + entrySize, '\x1');
        size_t writtenSize = 0;
        for( ; writtenSize < entrySize - bufferSize; writtenSize += bufferSize )
        {
            NNT_ASSERT_RESULT_SUCCESS(
                WriteStorage(writtenSize, writeBuffer.get() + writtenSize, bufferSize)
            );
            NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        }
        size_t remainSize = entrySize - writtenSize;
        NNT_ASSERT_RESULT_SUCCESS(
            WriteStorage(writtenSize, writeBuffer.get() + writtenSize, remainSize)
        );
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
    }

    const size_t overlapSize = bufferSize / 8;
    const size_t writeSize = bufferSize / 4 + overlapSize;

    // 比較用のデータも作成
    std::unique_ptr<char[]> compareBuffer(new char[entrySize]);
    std::iota(compareBuffer.get(), compareBuffer.get() + entrySize, '\x1');

    // 重複させながら Write します
    {
        std::unique_ptr<char[]> writeBuffer(new char[writeSize]);
        std::iota(writeBuffer.get(), writeBuffer.get() + writeSize, '\xA');
        int64_t offset = 0;
        NNT_ASSERT_RESULT_SUCCESS(WriteStorage(offset, writeBuffer.get(), writeSize));
        std::memcpy(compareBuffer.get() + offset, writeBuffer.get(), writeSize);
        offset += writeSize - overlapSize;
        NNT_ASSERT_RESULT_SUCCESS(WriteStorage(offset, writeBuffer.get(), writeSize));
        std::memcpy(compareBuffer.get() + offset, writeBuffer.get(), writeSize);
        offset += writeSize - overlapSize;
        NNT_ASSERT_RESULT_SUCCESS(WriteStorage(offset, writeBuffer.get(), writeSize));
        std::memcpy(compareBuffer.get() + offset, writeBuffer.get(), writeSize);
    }
    NNT_ASSERT_RESULT_SUCCESS(CommitStorage());

    // 読み込まれる値は書き込んだ値と等しくなります
    {
        std::unique_ptr<char[]> readBuffer(new char[entrySize]);
        NNT_ASSERT_RESULT_SUCCESS(ReadStorage(0, readBuffer.get(), entrySize));
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(compareBuffer.get(), readBuffer.get(), entrySize);
    }
}

/**
* 読み込み時と書き込み時のオフセットが違っていても正しく読み込めるか確かめます
*/
TEST_P(FsIntegritySaveDataStorageTest, TestReadWriteOffset)
{
    const TestParam& param = GetParam();

    size_t entrySize = static_cast<size_t>(param.GetEntrySize());
    size_t bufferSize = entrySize / 2;

    // 全体に書き込み
    {
        std::unique_ptr<char[]> writeBuffer(new char[entrySize]);
        std::iota(writeBuffer.get(), writeBuffer.get() + entrySize, '\x1');
        size_t writtenSize = 0;
        for( ; writtenSize < entrySize - bufferSize; writtenSize += bufferSize )
        {
            NNT_ASSERT_RESULT_SUCCESS(
                WriteStorage(writtenSize, writeBuffer.get() + writtenSize, bufferSize)
            );
            NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        }
        size_t remainSize = entrySize - writtenSize;
        NNT_ASSERT_RESULT_SUCCESS(
            WriteStorage(writtenSize, writeBuffer.get() + writtenSize, remainSize)
        );
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
    }

    // offset を 0～ブロックサイズ でずらしながら読み込みます
    std::unique_ptr<char[]> compareBuffer(new char[bufferSize]);
    std::unique_ptr<char[]> readBuffer(new char[bufferSize]);
    for( int i = 0; i <= param.sizeBlock; ++i )
    {
        std::iota(compareBuffer.get(), compareBuffer.get() + bufferSize, static_cast<char>(i + 1));
        NNT_ASSERT_RESULT_SUCCESS(ReadStorage(i, readBuffer.get(), bufferSize));
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(compareBuffer.get(), readBuffer.get(), bufferSize);
    }
}

/**
* 境界周辺に書き込み、正しく書き込めているか確認します
*/
TEST_P(FsIntegritySaveDataStorageTest, TestBoundaryValue)
{
    const TestParam& param = GetParam();

    const int checkSizeMax = 10;
    int64_t entrySize = param.GetEntrySize();

    std::unique_ptr<char[]> writeBuffer(new char[checkSizeMax]);
    std::unique_ptr<char[]> readBuffer(new char[checkSizeMax]);

    for( int size = 10; size > 0; --size )
    {
        std::iota(writeBuffer.get(), writeBuffer.get() + checkSizeMax, static_cast<char>(size * 2));
        NNT_ASSERT_RESULT_SUCCESS(WriteStorage(entrySize - size, writeBuffer.get(), size));
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        NNT_ASSERT_RESULT_SUCCESS(ReadStorage(entrySize - size, readBuffer.get(), size));
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(writeBuffer.get(), readBuffer.get(), size);
    }
}

/**
* 同じ箇所に何度も書き込むテスト
* 100 回書き込みと読み込みを行います
* 10 回に 1 回、コミットします
* 10 回に 1 回、再マウントします
*/
TEST_P(FsIntegritySaveDataStorageTest, TestWriteOverlap)
{
    const TestParam& param = GetParam();
    size_t entrySize = static_cast<size_t>(param.GetEntrySize());
    size_t bufferSize = entrySize / 2;

    std::unique_ptr<char[]> writeBuffer(new char[entrySize]);
    std::unique_ptr<char[]> compareBuffer(new char[entrySize]);
    std::unique_ptr<char[]> readBuffer(new char[entrySize]);

    // いったん全体に書き込んでおきます
    {
        std::iota(writeBuffer.get(), writeBuffer.get() + entrySize, '\x1');
        std::iota(compareBuffer.get(), compareBuffer.get() + entrySize, '\x1');
        size_t writtenSize = 0;
        for( ; writtenSize < entrySize - bufferSize; writtenSize += bufferSize )
        {
            NNT_ASSERT_RESULT_SUCCESS(
                WriteStorage(writtenSize, writeBuffer.get() + writtenSize, bufferSize)
            );
            NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        }
        size_t remainSize = entrySize - writtenSize;
        NNT_ASSERT_RESULT_SUCCESS(WriteStorage(writtenSize, writeBuffer.get() + writtenSize, remainSize));
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
    }

    for( int i = 0; i < 100; ++i )
    {
        int64_t offset = i / 10;

        // 書き込む値は毎回変えます
        std::iota(writeBuffer.get(), writeBuffer.get() + bufferSize, static_cast<char>(i));
        ResetAccessCounter();
        NNT_ASSERT_RESULT_SUCCESS(
            WriteStorage(
                offset,
                writeBuffer.get(),
                bufferSize - 10
            )
        );
        if( IsWriteHashExpected(bufferSize) )
        {
            ASSERT_FALSE(IsWriteCounterZero());
        }
        else
        {
            ASSERT_TRUE(IsWriteCounterZero());
        }
        std::memcpy(compareBuffer.get() + offset, writeBuffer.get(), bufferSize - 10);

        // 10 回に 1 回コミットします
        if( i % 10 == 9 )
        {
            NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        }

        // 10 回に 1 回再マウントします
        if( i % 10 == 4 )
        {
            UnmountStorage();
            NNT_ASSERT_RESULT_SUCCESS(MountStorage());
        }

        // 読み込み
        NNT_ASSERT_RESULT_SUCCESS(ReadStorage(0, readBuffer.get(), entrySize));
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(compareBuffer.get(), readBuffer.get(), entrySize);
    }
}

/**
* 下位ストレージを書き換えた場合にハッシュ検証エラーが発生するか確かめます
*/
TEST_P(FsIntegritySaveDataStorageTest, TestSignatureVerificationFailed)
{
    const TestParam& param = GetParam();
    size_t entrySize = static_cast<size_t>(param.GetEntrySize());
    size_t stepSize = entrySize / 2;

    std::unique_ptr<char[]> fillBuffer(new char[entrySize]);

    // いったん全体に書き込んでおきます
    {
        std::iota(fillBuffer.get(), fillBuffer.get() + entrySize, '\x1');
        size_t writtenSize = 0;
        for( ; writtenSize < entrySize - stepSize; writtenSize += stepSize )
        {
            NNT_ASSERT_RESULT_SUCCESS(
                WriteStorage(writtenSize, fillBuffer.get() + writtenSize, stepSize)
            );
            NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
        }
        size_t remainSize = entrySize - writtenSize;
        NNT_ASSERT_RESULT_SUCCESS(WriteStorage(writtenSize, fillBuffer.get() + writtenSize, remainSize));
        NNT_ASSERT_RESULT_SUCCESS(CommitStorage());
    }

    int64_t bufferSize = 512;
    std::unique_ptr<char[]> buffer(new char[static_cast<size_t>(bufferSize)]);
    std::iota(buffer.get(), buffer.get() + bufferSize, '\x5');

    // ハッシュデータがキャッシュされているので、
    // キャッシュをリセットしておきます
    {
        UnmountStorage();
        ResetBuffer();
        InitializeBuffer(param);
        NNT_ASSERT_RESULT_SUCCESS(MountStorage());
    }

    // マスターハッシュの値を書き換えます
    NNT_ASSERT_RESULT_SUCCESS(GetSubStorageMasterHash()->Write(0, buffer.get(), 32));

    // 検証に失敗します
    int64_t readOffset = 256;
    size_t readSize = 256;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultIntegrityVerificationStorageCorrupted,
        ReadStorage(readOffset, buffer.get(), readSize)
    );
}

/**
* 範囲操作をテストします
*/
TEST_P(FsIntegritySaveDataStorageTest, TestOperateRange)
{
    // 書き込み可能ストレージではキャッシュ無効化は非対応
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultUnsupportedOperation,
        OperateStorage(nn::fs::OperationId::Invalidate, 0, 1));
}

TEST_P(FsIntegritySaveDataStorageLargeTest, TestLarge)
{
    TestLargeOffsetAccess(GetStorage(), 1024);
}

INSTANTIATE_TEST_CASE_P(
    TestIntegritySaveDataStorage,
    FsIntegritySaveDataStorageTest,
    ::testing::ValuesIn(MakeTestParam())
);

INSTANTIATE_TEST_CASE_P(
    TestIntegritySaveDataStorage,
    FsIntegritySaveDataStorageLargeTest,
    ::testing::ValuesIn(MakeLargeTestParam())
);

}}}
