﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <numeric>
#include <map>
#include <sstream>

#include <nn/nn_Common.h>
#include <nn/os/os_Thread.h>
#include <nn/crypto.h>
#include <nn/util/util_IntUtil.h>
#include <nn/util/util_FormatString.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fssystem/save/fs_IntegritySaveDataFileSystemDriver.h>

// SaveDataFileSystemCore::QueryTotalSize から呼び出される関数の
// インスタンス化のためにインクルードします。
#include <nn/fssystem/dbm/fs_HierarchicalFileTableTemplate.impl.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>
#include "testFs_util_CommonStorageTests.h"
#include "testFs_util_CommonFileSystemTests.h"

namespace {
    typedef nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam IntegrityInputParam;
    typedef nn::fssystem::save::IntegritySaveDataFileSystemDriver FileSystemDriver;

    nn::fssystem::IBufferManager* GetBufferManager() NN_NOEXCEPT
    {
        static nn::fssystem::IBufferManager* s_pBufferManager = nullptr;

        if( s_pBufferManager == nullptr )
        {
            static const auto MaxCacheCount = 1024;
            static const auto SizeBlock = 16 * 1024;
#if defined(NN_BUILD_CONFIG_OS_WIN)
            static const auto BufferManagerHeapSize = 30 * 1024 * 1024;
#else
            static const auto BufferManagerHeapSize = 31 * 1024 * 1024;
#endif // defined(NN_BUILD_CONFIG_OS_WIN)
            static NN_ALIGNAS(4096) char s_BufferManagerHeap[BufferManagerHeapSize];
            static nn::fssystem::FileSystemBufferManager s_BufferManagerInstance;

            s_BufferManagerInstance.Initialize(
                MaxCacheCount,
                reinterpret_cast<uintptr_t>(s_BufferManagerHeap),
                sizeof(s_BufferManagerHeap),
                SizeBlock);
            s_pBufferManager = &s_BufferManagerInstance;
        }

        return s_pBufferManager;
    }

    uint32_t g_CountBlockForTestArchiveCommonFullHeavyTestMin = 0;
    uint32_t g_CountBlockForTestArchiveCommonFullHeavyTestMax = 0;
}

class FsIntegritySaveDataFileSystemDriverTest : public ::testing::Test
{
public:
    virtual ~FsIntegritySaveDataFileSystemDriverTest() NN_NOEXCEPT NN_OVERRIDE
    {
    }

protected:
    void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        m_BufferManagerFreeMemorySize = GetBufferManager()->GetFreeSize();
    }

    void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        if( !HasFatalFailure() )
        {
            EXPECT_EQ(m_BufferManagerFreeMemorySize, GetBufferManager()->GetFreeSize());
        }
    }

    static size_t SetupBlockSize2ForExtraDataTest(
        size_t sizeBlock,
        int countBlock,
        size_t sizeBlock1
    ) NN_NOEXCEPT
    {
        // ExtraDataTest で使用している sizeBlock1、sizeBlock2 に対して
        // countBlock * sizeBlock の容量が確保できるような sizeBlock2 を求めます
        // (sizeBlock1 / 32 * sizeBlock1 / 32 * sizeBlock2 / 32 * sizeBlock2 >= countBlock * sizeBlock)
        auto valueMin = static_cast<uint32_t>(std::ceil(128.0
            * std::sqrt(2.0 * static_cast<double>(sizeBlock) * static_cast<double>(countBlock))
            / static_cast<double>(sizeBlock1)));
        size_t value = 512;
        if( value < valueMin )
        {
            value = nn::util::ceilp2(valueMin);
        }
        return GenerateRandomBlockSize(value, MaxBlockSize);
    }

private:
    size_t m_BufferManagerFreeMemorySize;

};

class IntegritySaveDataFileSystemDriverTestSetup : public TestFileSystemSetup
{
public:
    // コンストラクタです。
    IntegritySaveDataFileSystemDriverTestSetup() NN_NOEXCEPT
        : m_IsRandomBlockSize(true)
    {
    }

    virtual ~IntegritySaveDataFileSystemDriverTestSetup() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    // ファイルシステムを初期化します。
    virtual void Initialize(int64_t offset, size_t sizeBlockPreferred) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(offset);

        size_t sizeBlockActual = sizeBlockPreferred;

        ASSERT_NE(0u, g_CountBlockForTestArchiveCommonFullHeavyTestMin);
        ASSERT_NE(0u, g_CountBlockForTestArchiveCommonFullHeavyTestMax);

        std::mt19937 mt(nnt::fs::util::GetRandomSeed());

        int64_t sizeTotal;
        nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
            paramIntegrity;
        uint32_t countBlock;
        {
            // ブロックサイズ
            if (m_IsRandomBlockSize)
            {
                sizeBlockActual = GenerateRandomBlockSize(256, MaxBlockSize);

                size_t sizeVerificationBlock1 = GenerateRandomBlockSize(512, MaxBlockSize);
                auto minBlockSize = std::max(
                    8 * 1024 * 1024 / sizeVerificationBlock1,
                    static_cast<size_t>(512)
                );
                ASSERT_LE(minBlockSize, MaxBlockSize);
                size_t sizeVerificationBlock2
                    = GenerateRandomBlockSize(minBlockSize, MaxBlockSize);

                paramIntegrity.sizeBlockLevel[0] = sizeVerificationBlock1;
                paramIntegrity.sizeBlockLevel[1] = sizeVerificationBlock1;
                paramIntegrity.sizeBlockLevel[2] = sizeVerificationBlock2;
                paramIntegrity.sizeBlockLevel[3] = sizeVerificationBlock2;
            }
            else
            {
                paramIntegrity.sizeBlockLevel[0] = sizeBlockActual;
                paramIntegrity.sizeBlockLevel[1] = sizeBlockActual;
                paramIntegrity.sizeBlockLevel[2] = sizeBlockActual;
                paramIntegrity.sizeBlockLevel[3] = sizeBlockActual;
            }

            countBlock = std::uniform_int_distribution<>(
                g_CountBlockForTestArchiveCommonFullHeavyTestMin,
                g_CountBlockForTestArchiveCommonFullHeavyTestMax)(mt);

            sizeTotal = 0;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::IntegritySaveDataFileSystemDriver::QueryTotalSize(
                    &sizeTotal,
                    sizeBlockActual,
                    paramIntegrity,
                    countBlock
                )
            );
        }

        NN_LOG("Test Param:\n");
        NN_LOG("  sizeTotal         : %lld\n", sizeTotal);
        NN_LOG("  sizeBlock1        : %u\n",
            static_cast<uint32_t>(paramIntegrity.sizeBlockLevel[0])
        );
        NN_LOG("  sizeBlock2        : %u\n",
            static_cast<uint32_t>(paramIntegrity.sizeBlockLevel[2])
        );
        NN_LOG("  countBlock        : %u\n", countBlock);

        m_MemStorage.reset(new nnt::fs::util::SafeMemoryStorage());
        ASSERT_NE(nullptr, m_MemStorage.get());
        m_MemStorage->Initialize(sizeTotal);
        // IntegritySaveDataFileSystemDriver は必ず 16KB アライメントのアクセスが発生する
        // 仕様のため 16 KB を超えるアクセスアライメントのチェックは不要です。
        size_t alignSize = std::min(static_cast<size_t>(16 * 1024), sizeBlockActual);
        m_AlignmentCheckStorage.Initialize(
            nn::fs::SubStorage(m_MemStorage.get(), 0, sizeTotal),
            alignSize
        );

        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::IntegritySaveDataFileSystemDriver::Format(
                nn::fs::SubStorage(&m_AlignmentCheckStorage, 0, sizeTotal),
                sizeBlockActual,
                paramIntegrity,
                countBlock,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator()
            )
        );

        m_FileSystem.reset(new nn::fssystem::save::IntegritySaveDataFileSystemDriver());
        ASSERT_NE(nullptr, m_FileSystem.get());
        NNT_ASSERT_RESULT_SUCCESS(
            m_FileSystem->Initialize(
                nn::fs::SubStorage(&m_AlignmentCheckStorage, 0, sizeTotal),
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator()
            )
        );

        SetFileSystem(m_FileSystem.get());
    }

    // ファイルシステムを破棄します。
    virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE
    {
        m_FileSystem->Finalize();
        m_FileSystem.reset();
        m_MemStorage.reset();
        m_MemStorage.reset();
    }

    // ファイルシステムへのアクセスはスレッドセーフかどうか
    virtual bool IsEnableThreadSafeAccess() NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

    // ファイルシステムへの排他制御は有効かどうか
    virtual bool IsEnableExclusiveAccess() NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

    // ファイルのサイズ変更が可能かどうか
    virtual bool IsEnableResizeFile() NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

    //!< ファイルへのデータ書き込み時に自動伸長するかどうか
    virtual bool IsAutoExtendSizeOnWrite() const NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

    // コミット処理を行います。
    nn::Result Commit() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_FileSystem);
        NN_RESULT_DO(m_FileSystem->Commit());
        NN_RESULT_SUCCESS;
    }

    // マウントしなおします。
    void UnmountAndMount() NN_NOEXCEPT
    {
        m_FileSystem->Finalize();
        m_FileSystem.reset(new nn::fssystem::save::IntegritySaveDataFileSystemDriver());
        ASSERT_NE(nullptr, m_FileSystem.get());

        int64_t sizeTotal;
        NNT_ASSERT_RESULT_SUCCESS(m_AlignmentCheckStorage.GetSize(&sizeTotal));
        NNT_ASSERT_RESULT_SUCCESS(
            m_FileSystem->Initialize(nn::fs::SubStorage(&m_AlignmentCheckStorage, 0, sizeTotal),
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator()));
        SetFileSystem(m_FileSystem.get());
    }

    // ブロックサイズをランダムに決定するかどうかを設定します。
    inline void SetRandomizeBlockSize(bool isRandom) NN_NOEXCEPT
    {
        m_IsRandomBlockSize = isRandom;
    }

private:
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> m_MemStorage;
    std::unique_ptr<nn::fssystem::save::IntegritySaveDataFileSystemDriver> m_FileSystem;
    nnt::fs::util::AlignCheckStorage m_AlignmentCheckStorage;
    bool m_IsRandomBlockSize;
};

// ファイルシステム共通テストを行います。
TEST_F(FsIntegritySaveDataFileSystemDriverTest, TestArchiveCommonFullHeavy)
{
    g_CountBlockForTestArchiveCommonFullHeavyTestMin = 500;
    g_CountBlockForTestArchiveCommonFullHeavyTestMax = 1000;
    FileSystemTest::RunTest<IntegritySaveDataFileSystemDriverTestSetup>();
}

#if !defined(NN_BUILD_CONFIG_ADDRESS_32) || !defined(NN_BUILD_CONFIG_OS_WIN)
// ファイルシステム共通テストを行います。
TEST_F(FsIntegritySaveDataFileSystemDriverTest, TestArchiveCommonFull64Heavy)
{
    g_CountBlockForTestArchiveCommonFullHeavyTestMin = 15000;
    g_CountBlockForTestArchiveCommonFullHeavyTestMax = 30000;
    FileSystemTest::RunTest<IntegritySaveDataFileSystemDriverTestSetup>();
}
#endif // !defined(NN_BUILD_CONFIG_ADDRESS_32) || !defined(NN_BUILD_CONFIG_OS_WIN)

// 0 バイト書き込みやクローズ時のファイルアクセスなどをテストします。
TEST_F(FsIntegritySaveDataFileSystemDriverTest, Flush)
{
    static const size_t TestLoop = 3;
    static const auto HashSize = nn::crypto::Sha256Generator::HashSize;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( size_t sizeBlock = 512; sizeBlock <= MaxBlockSize; sizeBlock *= 2 )
    {
        NN_SDK_LOG(".");

        for( int32_t loop = 0; loop < TestLoop; ++loop )
        {
            // ブロックサイズ
            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
                paramIntegrity;
            const size_t sizeBlock1 = GenerateRandomBlockSize(512, MaxBlockSize);
            // sizeBlock1 == sizeBlock2 == 512 だと検証層のサイズが不足することがある
            const size_t minSizeBlock2 = (sizeBlock1 == 512) ? 1024 : 512;
            const size_t sizeBlock2 = GenerateRandomBlockSize(minSizeBlock2, MaxBlockSize);
            paramIntegrity.sizeBlockLevel[0] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[1] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[2] = sizeBlock2;
            paramIntegrity.sizeBlockLevel[3] = sizeBlock2;

            const auto countMax = std::min<int64_t>(
                                      static_cast<uint64_t>(paramIntegrity.sizeBlockLevel[0])
                                      / HashSize * paramIntegrity.sizeBlockLevel[1]
                                      / HashSize * paramIntegrity.sizeBlockLevel[2]
                                      / HashSize * paramIntegrity.sizeBlockLevel[3]
                                      / sizeBlock,
                                      120
                                  );
            uint32_t countEntryFile =
                std::uniform_int_distribution<uint32_t>(
                    1,
                    static_cast<uint32_t>(countMax * 5) / 6
                )(mt);
            uint32_t countMetaBlock = nn::util::DivideUp(countEntryFile, 4);
            uint32_t countDataBlock;
            if( countMetaBlock > (countMax - 1) / 6)
            {
                countDataBlock = static_cast<uint32_t>(countMax * 5 / 6 + countMetaBlock);
            }
            else
            {
                countDataBlock = static_cast<uint32_t>(
                    std::uniform_int_distribution<int64_t>(countMetaBlock, (countMax - 1) / 6)(mt)
                    + countMax * 5 / 6
                );
            }

            int64_t sizeTotal = 0;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::IntegritySaveDataFileSystemDriver::QueryTotalSize(
                    &sizeTotal,
                    sizeBlock,
                    paramIntegrity,
                    countDataBlock
                )
            );

            // ファイルシステムを配置する Storage を構築する
            nnt::fs::util::AccessCountedMemoryStorage storageData;
            storageData.Initialize(sizeTotal);

            // IntegritySaveDataFileSystemDriver は必ず 16KB アライメントのアクセスが発生する
            // 仕様のため 16 KB を超えるアクセスアライメントのチェックは不要です。
            size_t sizeAlignment = std::min(sizeBlock, static_cast<size_t>(16 * 1024));
            nnt::fs::util::AlignCheckStorage storageAlignCheck;
            storageAlignCheck.Initialize(
                nn::fs::SubStorage(&storageData, 0, sizeTotal),
                sizeAlignment
            );

            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::IntegritySaveDataFileSystemDriver::Format(
                    nn::fs::SubStorage(&storageAlignCheck, 0, sizeTotal),
                    sizeBlock,
                    paramIntegrity,
                    countDataBlock,
                    GetBufferManager(),
                    nnt::fs::util::GetMacGenerator()
                )
            );

            nn::fssystem::save::IntegritySaveDataFileSystemDriver fileSystem;
            NNT_ASSERT_RESULT_SUCCESS(
                fileSystem.Initialize(nn::fs::SubStorage(&storageAlignCheck, 0, sizeTotal),
                    GetBufferManager(),
                    nnt::fs::util::GetMacGenerator()));

            // アクセス回数リセット
            storageData.ResetAccessCounter();

            // バッファ作成
            nnt::fs::util::Vector<char> data(sizeBlock * 2);
            char* ptr = &data[0];
            for( size_t i = 0; i < data.size(); ++i )
            {
                ptr[i] = i & 0xFF;
            }

            nn::fssystem::save::IFileSystem* pFileSystem = &fileSystem;

            // ディレクトリを新規作成します。
            nn::fssystem::save::Path pathDirectory;
            NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize("/directory"));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(pathDirectory));

            // 適当な位置に書き込み
            uint32_t leftBlockCount = countDataBlock - countMetaBlock;
            for( size_t i = 0; i < countEntryFile; ++i )
            {
                {
                    char buf[32];
                    std::memset(buf, 0, sizeof(buf));
                    nn::util::SNPrintf(buf, sizeof(buf), "/directory/file%08u", i);
                    nn::fssystem::save::Path filePath;
                    NNT_ASSERT_RESULT_SUCCESS(filePath.Initialize(buf));

                    nn::fssystem::save::IFile* pFileRW = nullptr;
                    const int64_t sizeFile =
                        std::uniform_int_distribution<int64_t>(
                                8,
                                (countEntryFile - i <= leftBlockCount ? sizeBlock : sizeBlock + 7)
                            )(mt);
                    leftBlockCount -=
                        static_cast<uint32_t>(nn::util::DivideUp(sizeFile, sizeBlock));
                    const int64_t offset =
                        std::uniform_int_distribution<int64_t>(0, sizeFile - 1)(mt);
                    size_t size =
                        std::uniform_int_distribution<size_t>(
                            1, static_cast<size_t>(sizeFile - offset))(mt);
                    if( offset + static_cast<int64_t>(size) > sizeFile )
                    {
                        size = static_cast<size_t>(sizeFile - offset);
                    }

                    // アクセス回数リセット
                    storageData.ResetAccessCounter();

                    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(filePath, sizeFile));
                    NNT_ASSERT_RESULT_SUCCESS(
                        pFileSystem->OpenFile(&pFileRW, filePath, nn::fs::OpenMode_Write));
                    NN_UTIL_SCOPE_EXIT
                    {
                        pFileSystem->CloseFile(pFileRW);
                    };
                    NNT_ASSERT_RESULT_SUCCESS(pFileRW->WriteBytes(offset, ptr, size));

                    ASSERT_GE(8, storageData.GetFlushTimes());

                    // アクセス回数リセット
                    storageData.ResetAccessCounter();
                }

                // クローズでファイルアクセスは発生しない
                ASSERT_EQ(0, storageData.GetReadTimes());
                ASSERT_EQ(0, storageData.GetWriteTimes());
            }

            // アクセス回数リセット
            storageData.ResetAccessCounter();

            // 0 バイト書き込み(flush == true)
            for( size_t i = 0; i < countEntryFile; ++i )
            {
                {
                    char buf[32];
                    nn::util::SNPrintf(buf, sizeof(buf), "/directory/file%08u", i);
                    nn::fssystem::save::Path filePath;
                    NNT_ASSERT_RESULT_SUCCESS(filePath.Initialize(buf));

                    nn::fssystem::save::IFile* pFile = nullptr;
                    NNT_ASSERT_RESULT_SUCCESS(
                        pFileSystem->OpenFile(&pFile, filePath, nn::fs::OpenMode_Write));
                    NN_UTIL_SCOPE_EXIT
                    {
                        pFileSystem->CloseFile(pFile);
                    };

                    int64_t sizeFile;
                    NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&sizeFile));

                    const int64_t offset =
                        std::uniform_int_distribution<int64_t>(0, sizeFile - 1)(mt);

                    // アクセス回数リセット
                    storageData.ResetAccessCounter();

                    NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(offset, ptr, 0));

                    ASSERT_EQ(0, storageData.GetFlushTimes());

                    // アクセス回数リセット
                    storageData.ResetAccessCounter();

                    NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());

                    ASSERT_LE(0, storageData.GetFlushTimes());

                    // アクセス回数リセット
                    storageData.ResetAccessCounter();

                    NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(0, ptr, 0));

                    ASSERT_EQ(0, storageData.GetFlushTimes());

                    // アクセス回数リセット
                    storageData.ResetAccessCounter();
                }

                // クローズでファイルアクセスは発生しない
                ASSERT_EQ(0, storageData.GetReadTimes());
                ASSERT_EQ(0, storageData.GetWriteTimes());
            }

            // アクセス回数リセット
            storageData.ResetAccessCounter();

            // 範囲外に書き込み
            for( size_t i = 0; i < countEntryFile; ++i )
            {
                {
                    char buf[32];
                    nn::util::SNPrintf(buf, sizeof(buf), "/directory/file%08u", i);
                    nn::fssystem::save::Path filePath;
                    NNT_ASSERT_RESULT_SUCCESS(filePath.Initialize(buf));

                    nn::fssystem::save::IFile* pFileLoop = nullptr;
                    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(
                        &pFileLoop, filePath, nn::fs::OpenMode_Write));
                    NN_UTIL_SCOPE_EXIT
                    {
                        pFileSystem->CloseFile(pFileLoop);
                    };

                    int64_t sizeFile;
                    NNT_ASSERT_RESULT_SUCCESS(pFileLoop->GetSize(&sizeFile));

                    const int64_t offset =
                        std::uniform_int_distribution<int64_t>(sizeFile + 1, sizeFile * 2)(mt);
                    const size_t size =
                        std::uniform_int_distribution<size_t>(1, static_cast<size_t>(sizeFile))(mt);

                    // アクセス回数リセット
                    storageData.ResetAccessCounter();

                    NNT_ASSERT_RESULT_FAILURE(
                        nn::fs::ResultInvalidOffset,
                        pFileLoop->WriteBytes(offset, ptr, size)
                    );

                    // 範囲外エラー発生時に書き込みは行われない
                    ASSERT_EQ(0, storageData.GetWriteTimes());
                    ASSERT_EQ(0, storageData.GetFlushTimes());

                    // アクセス回数リセット
                    storageData.ResetAccessCounter();
                }

                // クローズでファイルアクセスは発生しない
                ASSERT_EQ(0, storageData.GetReadTimes());
                ASSERT_EQ(0, storageData.GetWriteTimes());
            }

            // 明示的なコミットを実行
            NNT_ASSERT_RESULT_SUCCESS(fileSystem.Commit());

            // アクセス回数リセット
            storageData.ResetAccessCounter();

            fileSystem.Finalize();

            // アンマウント処理で書き込みは発生しない
            ASSERT_EQ(0, storageData.GetWriteTimes());

            // クローズでファイルアクセスは発生しない
            ASSERT_EQ(0, storageData.GetReadTimes());
            ASSERT_EQ(0, storageData.GetWriteTimes());
        }
    }

    NN_SDK_LOG("\n");
} // NOLINT(impl/function_size)

// ファイルサイズの上限値をテストします。
TEST_F(FsIntegritySaveDataFileSystemDriverTest, LimitSize)
{
    static const size_t TestLoop = 10;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( size_t sizeBlock = 512; sizeBlock <= MaxBlockSize; sizeBlock *= 2 )
    {
        NN_SDK_LOG(".");

        for( int32_t loop = 0; loop < TestLoop; ++loop )
        {
            // ブロックサイズ
            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
                paramIntegrity;
            const size_t sizeBlock1 = GenerateRandomBlockSize(512, 4096);
            const size_t sizeBlock2 = GenerateRandomBlockSize(512, 2048);
            paramIntegrity.sizeBlockLevel[0] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[1] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[2] = sizeBlock2;
            paramIntegrity.sizeBlockLevel[3] =
                (std::uniform_int_distribution<>(0, 1)(mt) != 0) ? sizeBlock1 : sizeBlock2;

            static const auto HashSize = nn::crypto::Sha256Generator::HashSize;
            const auto countMax = std::min<int64_t>(
                                      static_cast<uint64_t>(paramIntegrity.sizeBlockLevel[0])
                                      / HashSize * paramIntegrity.sizeBlockLevel[1]
                                      / HashSize * paramIntegrity.sizeBlockLevel[2]
                                      / HashSize * paramIntegrity.sizeBlockLevel[3]
                                      / sizeBlock,
                                      20
                                  );
            uint32_t countDataBlock =
                static_cast<uint32_t>(
                    std::uniform_int_distribution<int64_t>(0, countMax - 3)(mt) + 3);

            int64_t sizeTotal = 0;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::IntegritySaveDataFileSystemDriver::QueryTotalSize(
                    &sizeTotal,
                    sizeBlock,
                    paramIntegrity,
                    countDataBlock
                )
            );

            nnt::fs::util::AccessCountedMemoryStorage baseStorageData;
            baseStorageData.Initialize(sizeTotal);

            nnt::fs::util::AlignCheckStorage storageData;
            // IntegritySaveDataFileSystemDriver は必ず 16KB アライメントのアクセスが発生する
            // 仕様のため 16 KB を超えるアクセスアライメントのチェックは不要です。
            size_t sizeAlignment = std::min(sizeBlock, static_cast<size_t>(16 * 1024));
            storageData.Initialize(
                nn::fs::SubStorage(&baseStorageData, 0, sizeTotal),
                sizeAlignment
            );

            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::IntegritySaveDataFileSystemDriver::Format(
                    nn::fs::SubStorage(&storageData, 0, sizeTotal),
                    sizeBlock,
                    paramIntegrity,
                    countDataBlock,
                    GetBufferManager(),
                    nnt::fs::util::GetMacGenerator()
                )
            );

            nn::fssystem::save::IntegritySaveDataFileSystemDriver fileSystem;
            NNT_ASSERT_RESULT_SUCCESS(
                fileSystem.Initialize(
                    nn::fs::SubStorage(&storageData, 0, sizeTotal),
                    GetBufferManager(),
                    nnt::fs::util::GetMacGenerator()
                )
            );

            // バッファ作成
            nnt::fs::util::Vector<char> data(sizeBlock * 2);
            char* ptr = &data[0];
            for( size_t i = 0; i < data.size(); ++i )
            {
                ptr[i] = i & 0xFF;
            }

            nn::fssystem::save::IFileSystem* pFileSystem = &fileSystem;

            // エントリ分のブロックを差し引いて上限サイズを計算
            int64_t sizeLimit = sizeBlock * (countDataBlock - 2);
            {
                char buf[32];
                std::memset(buf, 0, sizeof(buf));
                nn::util::SNPrintf(
                    buf, sizeof(buf), "/file%08u", static_cast<uint32_t>(sizeLimit));
                nn::fssystem::save::Path filePath;
                NNT_ASSERT_RESULT_SUCCESS(filePath.Initialize(buf));

                // 上限サイズのファイルを作成します。
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->CreateFile(
                        filePath,
                        sizeLimit
                    )
                );

                // ファイルを削除します。
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->DeleteFile(filePath)
                );

                // 上限 + 1 サイズのファイルを作成します。
                NNT_ASSERT_RESULT_FAILURE(
                    nn::fs::ResultOutOfResource,
                    pFileSystem->CreateFile(
                        filePath,
                        sizeLimit + 1
                    )
                );
            }

            // 明示的なコミットを実行
            NNT_ASSERT_RESULT_SUCCESS(fileSystem.Commit());

            fileSystem.Finalize();
        }
    }

    NN_SDK_LOG("\n");
} // NOLINT(impl/function_size)

//! 不正なサイズを指定すると初期化に失敗することを確認します。
TEST_F(FsIntegritySaveDataFileSystemDriverTest, InvalidSize)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    static const auto LoopCount = 5;

    for( size_t sizeBlock = 4096; sizeBlock <= MaxBlockSize; sizeBlock *= 2 )
    {
        NN_SDK_LOG(".");

        for( auto loop = 0; loop < LoopCount; ++loop )
        {
            // ブロックサイズ
            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
                paramIntegrity;
            paramIntegrity.sizeBlockLevel[0] = GenerateRandomBlockSize(512, MaxBlockSize);
            paramIntegrity.sizeBlockLevel[1] = GenerateRandomBlockSize(512, MaxBlockSize);
            paramIntegrity.sizeBlockLevel[2] = GenerateRandomBlockSize(512, MaxBlockSize);
            paramIntegrity.sizeBlockLevel[3] = GenerateRandomBlockSize(512, MaxBlockSize);

            static const auto HashSize = nn::crypto::Sha256Generator::HashSize;
            const int64_t countMax = static_cast<int64_t>(paramIntegrity.sizeBlockLevel[0])
                / HashSize * paramIntegrity.sizeBlockLevel[1]
                / HashSize * paramIntegrity.sizeBlockLevel[2]
                / HashSize * paramIntegrity.sizeBlockLevel[3]
                / sizeBlock;
            int64_t countDataBlock = 0;
            auto countReservedBlock = std::uniform_int_distribution<int64_t>(1, countMax * 2)(mt);

            switch( loop )
            {
            case 0:
                countDataBlock = countMax;
                break;
            case 1:
                countDataBlock = countMax + 1;
                break;
            default:
                countDataBlock = std::uniform_int_distribution<int64_t>(1, countMax * 2)(mt);
                break;
            }

            if( !nn::util::IsIntValueRepresentable<uint32_t>(countDataBlock)
                || !nn::util::IsIntValueRepresentable<uint32_t>(countReservedBlock) )
            {
                continue;
            }

            int64_t sizeTotal = 0;

            for( ; ; )
            {
                const auto result = FileSystemDriver::QueryTotalSize(
                                        &sizeTotal,
                                        sizeBlock,
                                        paramIntegrity,
                                        static_cast<uint32_t>(countDataBlock)
                                    );
                if( countDataBlock <= countMax )
                {
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    break;
                }
                else
                {
                    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidSize, result);
                    countDataBlock /= 2;
                    countReservedBlock /= 2;
                }
            }

#if 0 // TODO: Format() が高速化されたら有効化する
            {
                nnt::fs::util::VirtualMemoryStorage storageData;
                nn::fssystem::save::IntegritySaveDataFileSystemDriver fileSystem;

                storageData.Initialize(sizeTotal);

                nn::fs::SubStorage storageBase(&storageData, 0, sizeTotal);

                NNT_ASSERT_RESULT_SUCCESS(
                    nn::fssystem::save::IntegritySaveDataFileSystemDriver::Format(
                        nn::fs::SubStorage(&storageData, 0, sizeTotal),
                        sizeBlock,
                        paramIntegrity,
                        static_cast<uint32_t>(countDataBlock),
                        GetBufferManager(),
                        nnt::fs::util::GetMacGenerator())
                );

                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.Initialize(storageBase, GetBufferManager(), nnt::fs::util::GetMacGenerator())
                );
                fileSystem.Finalize();
            }
#endif
        }
    }

    NN_SDK_LOG("\n");
} // NOLINT(impl/function_size)

//! コミットの挙動をテストします。
TEST_F(FsIntegritySaveDataFileSystemDriverTest, Commit)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    static const auto LoopCount = 3;

    bool isAnyErrorOccurred = false;
    while( !isAnyErrorOccurred )
    {
        for( size_t sizeBlock = 4096; sizeBlock <= MaxBlockSize; sizeBlock *= 2 )
        {
            NN_SDK_LOG(".");

            for( auto loop = 0; loop < LoopCount; ++loop )
            {
                nnt::fs::util::AccessCountedMemoryStorage storageData;
                nn::fssystem::save::IntegritySaveDataFileSystemDriver fileSystem;

                // ブロックサイズ
                nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
                    paramIntegrity;
                const size_t sizeBlock1 = GenerateRandomBlockSize(512, MaxBlockSize);
                // sizeBlock1 == sizeBlock2 == 512 だと検証層のサイズが不足することがある
                const size_t minSizeBlock2 = (sizeBlock1 == 512) ? 1024 : 512;
                const size_t sizeBlock2 = GenerateRandomBlockSize(minSizeBlock2, MaxBlockSize);
                paramIntegrity.sizeBlockLevel[0] = sizeBlock1;
                paramIntegrity.sizeBlockLevel[1] = sizeBlock1;
                paramIntegrity.sizeBlockLevel[2] = sizeBlock2;
                paramIntegrity.sizeBlockLevel[3] = sizeBlock2;

                static const auto HashSize = nn::crypto::Sha256Generator::HashSize;
                const auto countMax = std::min<int64_t>(
                    static_cast<uint64_t>(paramIntegrity.sizeBlockLevel[0])
                    / HashSize * paramIntegrity.sizeBlockLevel[1]
                    / HashSize * paramIntegrity.sizeBlockLevel[2]
                    / HashSize * paramIntegrity.sizeBlockLevel[3]
                    / sizeBlock,
                    120
                    );
                const auto countDataBlock = std::uniform_int_distribution<int64_t>(
                    countMax * 5 / 6,
                    countMax
                    )(mt);

                int64_t sizeTotal = 0;
                NNT_ASSERT_RESULT_SUCCESS(
                    nn::fssystem::save::IntegritySaveDataFileSystemDriver::QueryTotalSize(
                        &sizeTotal,
                        sizeBlock,
                        paramIntegrity,
                        static_cast<uint32_t>(countDataBlock)
                    )
                );

                // ファイルシステムを配置する Storage を構築する
                storageData.Initialize(sizeTotal);

                nn::fs::SubStorage storageBase(&storageData, 0, sizeTotal);

                NNT_ASSERT_RESULT_SUCCESS(
                    nn::fssystem::save::IntegritySaveDataFileSystemDriver::Format(
                        nn::fs::SubStorage(&storageData, 0, sizeTotal),
                        sizeBlock,
                        paramIntegrity,
                        static_cast<uint32_t>(countDataBlock),
                        GetBufferManager(),
                        nnt::fs::util::GetMacGenerator()
                    )
                );

                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.Initialize(storageBase, GetBufferManager(), nnt::fs::util::GetMacGenerator())
                );

                // 準備します。
                nn::fssystem::save::IntegritySaveDataFileSystem::ExtraData firstExtraData;
                nnt::fs::util::FillBufferWithRandomValue(&firstExtraData, sizeof(firstExtraData));
                nn::fssystem::save::Path pathDirectory;
                nn::fssystem::save::Path pathFile;
                const auto sizeFile
                    = std::uniform_int_distribution<size_t>(sizeBlock / 2, sizeBlock * 2)(mt);
                nnt::fs::util::Vector<char> bufWrite(sizeFile, 0);

                {
                    // 拡張データを書き込みます。
                    NNT_ASSERT_RESULT_SUCCESS(fileSystem.WriteExtraData(firstExtraData));

                    {
                        // ファイルを作成し書き込みます。
                        nn::fssystem::save::IFile* pFile = nullptr;
                        NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize("/directory"));
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory(pathDirectory));
                        NNT_ASSERT_RESULT_SUCCESS(pathFile.Initialize("/directory/file.dat"));
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile(pathFile, sizeFile));
                        NNT_ASSERT_RESULT_SUCCESS(
                            fileSystem.OpenFile(
                                &pFile,
                                pathFile,
                                nn::fs::OpenMode_Write
                            )
                        );
                        NN_UTIL_SCOPE_EXIT
                        {
                            fileSystem.CloseFile(pFile);
                        };
                        NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(0, bufWrite.data(), sizeFile));
                    }

                    NNT_ASSERT_RESULT_SUCCESS(fileSystem.Commit());
                    fileSystem.Finalize();
                    std::iota(bufWrite.begin(), bufWrite.end(), '\x1');
                }

                // コミットせずに再マウントするとデータが読めたり読めなかったりします
                bool isSuccessToOpenFile = false;
                {
                    NNT_ASSERT_RESULT_SUCCESS(
                        fileSystem.Initialize(storageBase, GetBufferManager(), nnt::fs::util::GetMacGenerator())
                    );

                    // 拡張データを書き込みます。
                    nn::fssystem::save::IntegritySaveDataFileSystem::ExtraData notCommitExtraData;
                    nnt::fs::util::FillBufferWithRandomValue(
                        &notCommitExtraData, sizeof(notCommitExtraData)
                    );
                    NNT_ASSERT_RESULT_SUCCESS(fileSystem.WriteExtraData(notCommitExtraData));

                    // ファイルを作成し書き込みます。
                    const auto offset = std::uniform_int_distribution<int64_t>(0, sizeFile - 1)(mt);
                    const auto sizeMax = static_cast<size_t>(sizeFile - offset);
                    const auto size = std::uniform_int_distribution<size_t>(1, sizeMax)(mt);
                    nnt::fs::util::Vector<char> bufPrevious(size, 0);
                    nnt::fs::util::Vector<char> bufRead(size, 0);

                    {
                        nn::fssystem::save::IFile* pFile = nullptr;
                        NNT_ASSERT_RESULT_SUCCESS(
                            fileSystem.OpenFile(
                                &pFile,
                                pathFile,
                                nn::fs::OpenMode_Read | nn::fs::OpenMode_Write
                            )
                        );
                        NN_UTIL_SCOPE_EXIT
                        {
                            fileSystem.CloseFile(pFile);
                        };
                        NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(offset, bufPrevious.data(), size));
                        NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(offset, bufWrite.data(), size));
                    }

                    // コミットせずに再マウントします
                    fileSystem.Finalize();
                    NNT_ASSERT_RESULT_SUCCESS(
                        fileSystem.Initialize(storageBase, GetBufferManager(), nnt::fs::util::GetMacGenerator())
                    );

                    // 以前のデータが読み込めることを確認します
                    nn::fssystem::save::IntegritySaveDataFileSystem::ExtraData extraData;
                    fileSystem.ReadExtraData(&extraData);
                    NNT_FS_UTIL_ASSERT_MEMCMPEQ(&firstExtraData, &extraData, sizeof(extraData));

                    // ファイルの Open はハッシュ検証エラーになることがあります
                    {
                        nn::fssystem::save::IFile* pFile = nullptr;
                        auto result = fileSystem.OpenFile(
                            &pFile,
                            pathFile,
                            nn::fs::OpenMode_Read
                        );
                        if( nn::fs::ResultNonRealDataVerificationFailed::Includes(result)
                            || nn::fs::ResultUnclearedRealDataVerificationFailed::Includes(result) )
                        {
                            isAnyErrorOccurred = true;
                        }
                        else
                        {
                            NNT_ASSERT_RESULT_SUCCESS(result);

                            NN_UTIL_SCOPE_EXIT
                            {
                                fileSystem.CloseFile(pFile);
                            };

                            NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(offset, bufRead.data(), size));
                            NNT_FS_UTIL_ASSERT_MEMCMPEQ(bufRead.data(), bufPrevious.data(), size);

                            isSuccessToOpenFile = true;
                        }
                    }

                    fileSystem.Finalize();
                }

                if( isSuccessToOpenFile )
                {
                    // コミットして再マウントすると書き込んだデータが読み込めます。
                    {
                        NNT_ASSERT_RESULT_SUCCESS(
                            fileSystem.Initialize(storageBase, GetBufferManager(), nnt::fs::util::GetMacGenerator())
                        );

                        // 拡張データを書き込みます。
                        nn::fssystem::save::IntegritySaveDataFileSystem::ExtraData commitExtraData;
                        nnt::fs::util::FillBufferWithRandomValue(
                            &commitExtraData, sizeof(commitExtraData)
                        );
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.WriteExtraData(commitExtraData));

                        // ファイルに書き込みます。
                        const auto offset = std::uniform_int_distribution<int64_t>(0, sizeFile - 1)(mt);
                        const auto sizeMax = static_cast<size_t>(sizeFile - offset);
                        const auto size = std::uniform_int_distribution<size_t>(1, sizeMax)(mt);
                        {
                            nn::fssystem::save::IFile* pFile = nullptr;

                            NNT_ASSERT_RESULT_SUCCESS(
                                fileSystem.OpenFile(
                                    &pFile,
                                    pathFile,
                                    nn::fs::OpenMode_Write
                                )
                            );
                            NN_UTIL_SCOPE_EXIT
                            {
                                fileSystem.CloseFile(pFile);
                            };

                            NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(offset, bufWrite.data(), size));
                        }

                        // コミットします
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.Commit());
                        // 再マウントします
                        fileSystem.Finalize();
                        NNT_ASSERT_RESULT_SUCCESS(
                            fileSystem.Initialize(storageBase, GetBufferManager(), nnt::fs::util::GetMacGenerator())
                        );

                        // 書き込んだデータが読み込めることを確認します
                        nn::fssystem::save::IntegritySaveDataFileSystem::ExtraData readExtraData;
                        fileSystem.ReadExtraData(&readExtraData);
                        NNT_FS_UTIL_ASSERT_MEMCMPEQ(
                            &commitExtraData, &readExtraData, sizeof(readExtraData)
                        );

                        {
                            nnt::fs::util::Vector<char> bufRead(size, 0);
                            nn::fssystem::save::IFile* pFile = nullptr;

                            NNT_ASSERT_RESULT_SUCCESS(
                                fileSystem.OpenFile(
                                    &pFile,
                                    pathFile, nn::fs::OpenMode_Read));
                            NN_UTIL_SCOPE_EXIT
                            {
                                fileSystem.CloseFile(pFile);
                            };

                            NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(offset, bufRead.data(), size));
                            NNT_FS_UTIL_ASSERT_MEMCMPEQ(
                                bufRead.data(), bufWrite.data(), bufRead.size()
                            );
                        }

                        fileSystem.Finalize();
                    }

                    {
                        const auto offset = std::uniform_int_distribution<int64_t>(0, sizeFile - 1)(mt);
                        const auto sizeMax = static_cast<size_t>(sizeFile - offset);
                        const auto size = std::uniform_int_distribution<size_t>(1, sizeMax)(mt);

                        NNT_ASSERT_RESULT_SUCCESS(
                            fileSystem.Initialize(storageBase, GetBufferManager(), nnt::fs::util::GetMacGenerator())
                        );

                        // 書き込みモードでのファイルオープン中はコミットできません。
                        {
                            nn::fssystem::save::IFile* pFile = nullptr;
                            NNT_ASSERT_RESULT_SUCCESS(
                                fileSystem.OpenFile(
                                    &pFile,
                                    pathFile,
                                    nn::fs::OpenMode_Write
                                )
                            );
                            NN_UTIL_SCOPE_EXIT
                            {
                                fileSystem.CloseFile(pFile);
                            };

                            NNT_ASSERT_RESULT_SUCCESS(
                                pFile->WriteBytes(offset, bufWrite.data(), size)
                            );
                            NNT_ASSERT_RESULT_FAILURE(
                                nn::fs::ResultUnsupportedOperation, fileSystem.Commit()
                            );
                        }

                        // 読み込みモードでのファイルオープン中はコミットできます。
                        {
                            nn::fssystem::save::IFile* pFile = nullptr;
                            NNT_ASSERT_RESULT_SUCCESS(
                                fileSystem.OpenFile(
                                    &pFile,
                                    pathFile,
                                    nn::fs::OpenMode_Read
                                )
                            );
                            NN_UTIL_SCOPE_EXIT
                            {
                                fileSystem.CloseFile(pFile);
                            };

                            nn::fssystem::save::IDirectory* pDirectory = nullptr;
                            NNT_ASSERT_RESULT_SUCCESS(
                                fileSystem.OpenDirectory(
                                    &pDirectory,
                                    pathDirectory
                                )
                            );
                            NN_UTIL_SCOPE_EXIT
                            {
                                fileSystem.CloseDirectory(pDirectory);
                            };

                            NNT_ASSERT_RESULT_SUCCESS(fileSystem.Commit());
                        }

                        fileSystem.Finalize();
                    }
                }
            }
        }

        NN_SDK_LOG("\n");
    }
} // NOLINT(impl/function_size)

TEST_F(FsIntegritySaveDataFileSystemDriverTest, MultipleSaveDataFileSystemsHeavy)
{
    static const auto LoopCount = 1000;
    static const auto StackSize = 32 * 1024;
    static const auto CountFileUnit = 6;
    static const auto SaveDataCount = 5 * CountFileUnit;
    static const auto SizeBlock = 16 * 1024;

#ifdef NN_BUILD_CONFIG_ADDRESS_64
#ifdef NN_BUILD_CONFIG_OS_WIN
    static const auto CountDataBlockUnit = 100;
    static const auto CountFileBlockUnit = 80;
    static const auto CountBufferBlock = 64;
#else
    static const auto CountDataBlockUnit = 50;
    static const auto CountFileBlockUnit = 40;
    static const auto CountBufferBlock = 32;
#endif
#else
    static const auto CountDataBlockUnit = 20;
    static const auto CountFileBlockUnit = 16;
    static const auto CountBufferBlock = 12;
#endif

    class Statistics
    {
    public:
        Statistics() NN_NOEXCEPT
            : m_Sum(0),
              m_SumSquared(0),
              m_Max(0),
              m_Min(0),
              m_Count(0),
              m_Lock(false)
        {
        }

        void Add(double sample) NN_NOEXCEPT
        {
            m_Lock.lock();
            m_Sum += sample;
            m_SumSquared += sample * sample;
            m_Max = std::max(sample, m_Max);
            m_Min = std::min(sample, m_Min);
            ++m_Count;
            m_Lock.unlock();
        }

        double GetAverage() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_GREATER(m_Count, 0);
            return m_Sum / m_Count;
        }

        double GetVariance() const NN_NOEXCEPT
        {
            const auto average = GetAverage();
            return m_SumSquared / m_Count - average * average;
        }

        double GetStandardDeviation() const NN_NOEXCEPT
        {
            return std::sqrt(GetVariance());
        }

        double GetMax() const NN_NOEXCEPT
        {
            return m_Max;
        }

        double GetMin() const NN_NOEXCEPT
        {
            return m_Min;
        }

    private:
        double m_Sum;
        double m_SumSquared;
        double m_Max;
        double m_Min;
        int m_Count;
        nn::os::Mutex m_Lock;
    };

    static Statistics statisticsWrite;
    static Statistics statisticsRead;

    MeasurableBufferManager<false> bufferManager(GetBufferManager());

    class SaveData
    {
    public:
        nn::Result Initialize(int id, nn::fssystem::IBufferManager* pBufferManager) NN_NOEXCEPT
        {
            m_Id = id;

            m_Mt.seed(id + 1);

            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
                paramIntegrity;
            paramIntegrity.sizeBlockLevel[0] = SizeBlock;
            paramIntegrity.sizeBlockLevel[1] = SizeBlock;
            paramIntegrity.sizeBlockLevel[2] = SizeBlock;
            paramIntegrity.sizeBlockLevel[3] = SizeBlock;

            int countDataBlock;
            if (id < CountFileUnit)
            {
                countDataBlock = 100 * CountDataBlockUnit;
            }
            else if( id < 2 * CountFileUnit )
            {
                countDataBlock = 10 * CountDataBlockUnit;
            }
            else
            {
                countDataBlock = 1 * CountDataBlockUnit;
            }

            int64_t sizeTotal = 0;
            NN_RESULT_DO(
                nn::fssystem::save::IntegritySaveDataFileSystemDriver::QueryTotalSize(
                    &sizeTotal,
                    SizeBlock,
                    paramIntegrity,
                    countDataBlock
                )
            );

            m_StorageData.Initialize(sizeTotal);

            nn::fs::SubStorage storageBase(&m_StorageData, 0, sizeTotal);

            NN_RESULT_DO(
                nn::fssystem::save::IntegritySaveDataFileSystemDriver::Format(
                    storageBase,
                    SizeBlock,
                    paramIntegrity,
                    countDataBlock,
                    pBufferManager,
                    nnt::fs::util::GetMacGenerator()
                )
            );

            NN_RESULT_DO(m_FileSystem.Initialize(storageBase, pBufferManager, nnt::fs::util::GetMacGenerator()));

            if( id < CountFileUnit )
            {
                m_SizeFile = 100 * CountFileBlockUnit * SizeBlock;
            }
            else if( id < 2 * CountFileUnit )
            {
                m_SizeFile = 10 * CountFileBlockUnit * SizeBlock;
            }
            else
            {
                m_SizeFile = 1 * CountFileBlockUnit * SizeBlock;
            }

            nn::fssystem::save::Path pathDirectory;
            NN_RESULT_DO(pathDirectory.Initialize("/directory"));
            NN_RESULT_DO(m_FileSystem.CreateDirectory(pathDirectory));
            NN_RESULT_DO(m_FilePath.Initialize("/directory/file.dat"));
            NN_RESULT_DO(m_FileSystem.CreateFile(m_FilePath, m_SizeFile));
            NN_RESULT_DO(m_FileSystem.Commit());

            m_Buffer.resize(SizeBlock * CountBufferBlock);
            std::iota(m_Buffer.begin(), m_Buffer.end(), static_cast<char>(id));

            NN_RESULT_SUCCESS;
        }

        nn::Result Access() NN_NOEXCEPT
        {
            const auto close = [this](nn::fssystem::save::IFile* pFile) NN_NOEXCEPT
            {
                m_FileSystem.CloseFile(pFile);
            };
            typedef std::unique_ptr<nn::fssystem::save::IFile, decltype(close)> UniqueFile;

            m_StorageData.ResetAccessCounter();

            for( auto loop = 0; loop < LoopCount; ++loop )
            {
                if( (loop + 1) % 100 == 0 )
                {
                    NN_RESULT_DO(m_FileSystem.Commit());

                    statisticsWrite.Add(m_StorageData.GetWriteTimes());
                    statisticsRead.Add(m_StorageData.GetReadTimes());
                    m_StorageData.ResetAccessCounter();
                }
                else if( loop % 3 != 0 )
                {
                    UniqueFile pUniqueFile(nullptr, close);
                    {
                        nn::fssystem::save::IFile* pFile = nullptr;
                        NN_RESULT_DO(
                            m_FileSystem.OpenFile(
                                &pFile,
                                m_FilePath,
                                nn::fs::OpenMode_Write
                            )
                        );
                        pUniqueFile.reset(pFile);
                    }

                    const auto offsetMax = static_cast<int64_t>(m_SizeFile - m_Buffer.size());
                    const auto offset = std::uniform_int_distribution<int64_t>(0, offsetMax)(m_Mt);
                    NN_RESULT_DO(pUniqueFile->WriteBytes(offset, m_Buffer.data(), m_Buffer.size()));
                }
                else
                {
                    UniqueFile pUniqueFile(nullptr, close);
                    {
                        nn::fssystem::save::IFile* pFile = nullptr;
                        NN_RESULT_DO(
                            m_FileSystem.OpenFile(
                                &pFile,
                                m_FilePath,
                                nn::fs::OpenMode_Read
                            )
                        );
                        pUniqueFile.reset(pFile);
                    }

                    const auto offsetMax = static_cast<int64_t>(m_SizeFile - m_Buffer.size());
                    const auto offset = std::uniform_int_distribution<int64_t>(0, offsetMax)(m_Mt);
                    NN_RESULT_DO(pUniqueFile->ReadBytes(offset, m_Buffer.data(), m_Buffer.size()));
                }

                if( m_Id == 0 && loop % 10 == 0 )
                {
                    NN_SDK_LOG(".");
                }
            }
            NN_RESULT_SUCCESS;
        }

        void Finalize() NN_NOEXCEPT
        {
            m_FileSystem.Finalize();
            m_StorageData.Finalize();
            m_Buffer.clear();
            m_Buffer.shrink_to_fit();
        }

    private:
        int m_Id;
        std::mt19937 m_Mt;
        nnt::fs::util::AccessCountedMemoryStorage m_StorageData;
        nn::fssystem::save::IntegritySaveDataFileSystemDriver m_FileSystem;
        nn::fssystem::save::Path m_FilePath;
        nnt::fs::util::Vector<char> m_Buffer;
        std::size_t m_SizeFile;
    };

    static SaveData s_SaveData[SaveDataCount] = {};
    static nn::os::ThreadType s_Threads[SaveDataCount] = {};
    NN_OS_ALIGNAS_THREAD_STACK static char s_Stack[StackSize * SaveDataCount] = {};

    while( !bufferManager.IsBufferStarved() )
    {
        for( auto& data : s_SaveData )
        {
            NNT_ASSERT_RESULT_SUCCESS(
                data.Initialize(static_cast<int>(&data - s_SaveData), &bufferManager)
            );
        }

        for( auto& thread : s_Threads )
        {
            const auto index = &thread - s_Threads;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::os::CreateThread(
                    &thread,
                    [](void* pArgument) NN_NOEXCEPT
            {
                NN_SDK_REQUIRES_NOT_NULL(pArgument);
                NNT_EXPECT_RESULT_SUCCESS(reinterpret_cast<SaveData*>(pArgument)->Access());
            },
                    s_SaveData + index,
                s_Stack + StackSize * index,
                StackSize,
                nn::os::DefaultThreadPriority
                )
            );
        }

        for( auto& thread : s_Threads )
        {
            nn::os::StartThread(&thread);
        }

        for( auto& thread : s_Threads )
        {
            nn::os::WaitThread(&thread);
            nn::os::DestroyThread(&thread);
        }

        for( auto& data : s_SaveData )
        {
            data.Finalize();
        }

        EXPECT_LT(statisticsWrite.GetAverage(), 1500);
        EXPECT_LT(statisticsRead.GetAverage(), 2500);
        EXPECT_LT(statisticsWrite.GetMax(), 1500);
        EXPECT_LT(statisticsRead.GetMax(), 5000);

        EXPECT_LE(static_cast<size_t>(256 * 1024), bufferManager.GetTotalAllocatableSizeMin());
    }

    NN_SDK_LOG("\n");

    NN_LOG(
        "total allocatable size min: %u bytes\n",
        static_cast<uint32_t>(bufferManager.GetTotalAllocatableSizeMin())
    );
} // NOLINT(impl/function_size)

// サイズの大きなストレージを持つファイルシステムのテストします。
TEST_F(FsIntegritySaveDataFileSystemDriverTest, LargeStorageHeavy)
{
    static const size_t BlockSize = 32 * 1024;
    static const int64_t Giga = 1024 * 1024 * 1024;
    static const int64_t StorageSizeMin = 2 * 1024 * Giga;
    static const uint32_t DataBlockCount = 64 * 1024;
    static const size_t DataSize = 64 * 1024;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::Vector<char> readBuf(DataSize);
    nnt::fs::util::Vector<char> baseBuf(DataSize);
    std::iota(baseBuf.begin(), baseBuf.end(), '\x0');

    // ファイルシステムを配置する Storage を構築
    nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
        paramIntegrity;
    paramIntegrity.sizeBlockLevel[0] = MaxBlockSize;
    paramIntegrity.sizeBlockLevel[1] = MaxBlockSize;
    paramIntegrity.sizeBlockLevel[2] = MaxBlockSize;
    paramIntegrity.sizeBlockLevel[3] = MaxBlockSize;

    int64_t sizeTotal = 0;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::fssystem::save::IntegritySaveDataFileSystemDriver::QueryTotalSize(
            &sizeTotal,
            BlockSize,
            paramIntegrity,
            DataBlockCount
        )
    );
    EXPECT_GE(StorageSizeMin, sizeTotal);

    nnt::fs::util::VirtualMemoryStorage storageData;
    storageData.Initialize(sizeTotal);

    nn::fssystem::save::IntegritySaveDataFileSystemDriver fileSystem;
    nn::fs::SubStorage storageBase(&storageData, 0, sizeTotal);

    NNT_ASSERT_RESULT_SUCCESS(
        nn::fssystem::save::IntegritySaveDataFileSystemDriver::Format(
            nn::fs::SubStorage(&storageData, 0, sizeTotal),
            BlockSize,
            paramIntegrity,
            DataBlockCount,
            GetBufferManager(),
            nnt::fs::util::GetMacGenerator()
        )
    );

    NNT_ASSERT_RESULT_SUCCESS(
        fileSystem.Initialize(storageBase, GetBufferManager(), nnt::fs::util::GetMacGenerator())
    );

    // ファイルを作成
    nn::fssystem::save::Path pathFile;
    NNT_ASSERT_RESULT_SUCCESS(pathFile.Initialize("/file.dat"));
    {
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile(pathFile, 0));

        nn::fssystem::save::IFile* pFile = nullptr;
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.OpenFile(&pFile, pathFile, nn::fs::OpenMode_Write));
        NN_UTIL_SCOPE_EXIT
        {
            fileSystem.CloseFile(pFile);
        };

        NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(BlockSize * DataBlockCount / 2));
        int64_t sizeFile;
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&sizeFile));
        EXPECT_EQ(BlockSize * DataBlockCount / 2, sizeFile);
    }
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.Commit());

    // ファイルが削除できることを確認
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.DeleteFile(pathFile));

    // 様々なサイズのファイルを作成・削除できることを確認
    for( size_t blockSize = std::uniform_int_distribution<>(128, 512)(mt);
            blockSize < BlockSize;
            blockSize = (blockSize << 1) + std::uniform_int_distribution<>(0, 256)(mt) )
    {
        char fileName[32];
        std::sprintf(fileName, "/%X.dat", static_cast<uint32_t>(blockSize));

        const auto fileSize = static_cast<int64_t>(blockSize) * DataBlockCount;

        nn::fssystem::save::Path path;
        NNT_ASSERT_RESULT_SUCCESS(path.Initialize(fileName));
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile(path, fileSize));
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.DeleteFile(path));
    }

    const auto fileSize = static_cast<int64_t>(BlockSize) * (DataBlockCount - 2);

    // 最大サイズのファイルを作成してデータを書き込む
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile(pathFile, fileSize));
    {
        nn::fssystem::save::IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.OpenFile(&pFile, pathFile, nn::fs::OpenMode_Write));
        NN_UTIL_SCOPE_EXIT
        {
            fileSystem.CloseFile(pFile);
        };

        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(0, baseBuf.data(), DataSize));
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(fileSize - DataSize, baseBuf.data(), DataSize));
    }
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.Commit());

    fileSystem.Finalize();

    // 再マウント
    NNT_ASSERT_RESULT_SUCCESS(
        fileSystem.Initialize(
            nn::fs::SubStorage(&storageData, 0, sizeTotal),
            GetBufferManager(),
            nnt::fs::util::GetMacGenerator()
        )
    );

    // マウント前のデータが読み込めるか確認
    {
        nn::fssystem::save::IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.OpenFile(&pFile, pathFile, nn::fs::OpenMode_Read)
        );
        NN_UTIL_SCOPE_EXIT
        {
            fileSystem.CloseFile(pFile);
        };

        NNT_ASSERT_RESULT_SUCCESS(
            pFile->ReadBytes(0, readBuf.data(), DataSize)
        );
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(baseBuf.data(), readBuf.data(), DataSize);

        NNT_ASSERT_RESULT_SUCCESS(
            pFile->ReadBytes(fileSize - DataSize, readBuf.data(), DataSize)
        );
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(baseBuf.data(), readBuf.data(), DataSize);
    }
} // NOLINT(impl/function_size)

//! 拡張データの読み書きが正常に行えるか確かめます。
TEST_F(FsIntegritySaveDataFileSystemDriverTest, ExtraData)
{
    nn::fssystem::save::IntegritySaveDataFileSystemDriver fileSystem;
    nnt::fs::util::AccessCountedMemoryStorage storageData;
    nn::fs::SubStorage storageBase;
    nn::fs::SubStorage storageExpand;

    // データのサイズ
    const auto countDataBlock = 512;
    const size_t sizeBlock = GenerateRandomBlockSize(512, MaxBlockSize);

    // 初期化
    {
        // ブロックサイズ
        nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
            paramIntegrity;

        const size_t sizeBlock1 = GenerateRandomBlockSize(512, MaxBlockSize);
        const size_t sizeBlock2 = SetupBlockSize2ForExtraDataTest(
            sizeBlock, countDataBlock, sizeBlock1
        );
        paramIntegrity.sizeBlockLevel[0] = sizeBlock1;
        paramIntegrity.sizeBlockLevel[1] = sizeBlock1;
        paramIntegrity.sizeBlockLevel[2] = sizeBlock2;
        paramIntegrity.sizeBlockLevel[3] = sizeBlock2;

        int64_t sizeTotal = 0;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::IntegritySaveDataFileSystemDriver::QueryTotalSize(
                &sizeTotal,
                sizeBlock,
                paramIntegrity,
                static_cast<uint32_t>(countDataBlock)
            )
        );

        // ファイルシステムを配置する Storage を構築する
        storageData.Initialize(sizeTotal);
        storageBase = nn::fs::SubStorage(&storageData, 0, sizeTotal);

        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::IntegritySaveDataFileSystemDriver::Format(
                storageBase,
                sizeBlock,
                paramIntegrity,
                static_cast<uint32_t>(countDataBlock),
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator()
            )
        );

        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Initialize(storageBase, GetBufferManager(), nnt::fs::util::GetMacGenerator())
        );

        // ファイル作成、書き込み、コミット
        {
            nn::fssystem::save::Path filePath;
            const auto FileSize = 2048;
            NNT_ASSERT_RESULT_SUCCESS(filePath.Initialize("/file"));
            NNT_ASSERT_RESULT_SUCCESS(
                fileSystem.CreateFile(filePath, FileSize)
            );

            std::unique_ptr<char[]> fileBuffer(new char[FileSize]);
            nnt::fs::util::FillBufferWithRandomValue(fileBuffer.get(), FileSize);

            {
                nn::fssystem::save::IFile* pFile;
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.OpenFile(&pFile, filePath, nn::fs::OpenMode_Write)
                );
                NN_UTIL_SCOPE_EXIT
                {
                    fileSystem.CloseFile(pFile);
                };

                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->WriteBytes(0, fileBuffer.get(), FileSize)
                );
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->Flush()
                );
            }
            NNT_ASSERT_RESULT_SUCCESS(
                fileSystem.Commit()
            );
        }
    }

    // 書き込み、読み込み用バッファ作成
    nn::fssystem::save::IntegritySaveDataFileSystem::ExtraData writeData0;
    nn::fssystem::save::IntegritySaveDataFileSystem::ExtraData writeData1;
    nn::fssystem::save::IntegritySaveDataFileSystem::ExtraData readData;

    nnt::fs::util::FillBufferWithRandomValue(&writeData0, 512);
    nnt::fs::util::FillBufferWithRandomValue(&writeData1, 512);

    // フォーマットした状態なら、拡張データの中身は 0 フィルされています
    {
        fileSystem.ReadExtraData(&readData);
        char* buffer = reinterpret_cast<char*>(&readData);
        for( int i=0; i < 512; ++i )
        {
            ASSERT_EQ(0, buffer[i]);
        }
    }

    // 拡張データを書き込み、コミットして静的・動的読み込み
    {
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.WriteExtraData(writeData0)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Commit()
        );

        // 読み込み
        fileSystem.ReadExtraData(&readData);
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(&readData, &writeData0, 512);

        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::IntegritySaveDataFileSystemDriver::ReadExtraData(
                &readData,
                storageBase,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator()
            )
        );
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(&readData, &writeData0, 512);
    }

    // 拡張データを書き込み、コミットせずに静的・動的読み込み
    {
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.WriteExtraData(writeData1)
        );

        // 動的読み込み
        fileSystem.ReadExtraData(&readData);
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(&readData, &writeData1, 512);

        // 静的読み込み
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::IntegritySaveDataFileSystemDriver::ReadExtraData(
                &readData,
                storageBase,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator()
            )
        );
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(&readData, &writeData0, 512);
    }

    // アンマウントして静的読み込み
    {
        fileSystem.Finalize();

        // 静的読み込み
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::IntegritySaveDataFileSystemDriver::ReadExtraData(
                &readData,
                storageBase,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator()
            )
        );
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(&readData, &writeData0, 512);
    }

    // マウントして静的・動的読み込み
    {
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Initialize(storageBase, GetBufferManager(), nnt::fs::util::GetMacGenerator())
        );

        // 動的読み込み
        fileSystem.ReadExtraData(&readData);
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(&readData, &writeData0, 512);

        // 静的読み込み
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::IntegritySaveDataFileSystemDriver::ReadExtraData(
                &readData,
                storageBase,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator()
            )
        );
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(&readData, &writeData0, 512);
    }

    // 書き込まずにコミットをして静的・動的読み込み
    {
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Commit()
        );

        // 動的読み込み
        fileSystem.ReadExtraData(&readData);
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(&readData, &writeData0, 512);

        // 静的読み込み
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::IntegritySaveDataFileSystemDriver::ReadExtraData(
                &readData,
                storageBase,
                GetBufferManager(),
                nnt::fs::util::GetMacGenerator()
            )
        );
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(&readData, &writeData0, 512);
    }
}// NOLINT(impl/function_size)

TEST_F(FsIntegritySaveDataFileSystemDriverTest, RandomAllocationFailureHeavy)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    MeasurableBufferManager<false> measurableBufferManager(GetBufferManager());
    nnt::fs::util::RandomFailureBufferManager randomFailureBufferManager(&measurableBufferManager, &mt);

    static const auto LoopCount = 5;
    static const auto InnerLoopCount = 50;

    // Allocation Count の実測値の * 4 / 3 ぐらいにする
    // サイズ変動系は小さめの値にして問題ない (リトライ時に * 2 される)
    static const auto RandomDividerFormat = 1000;
    static const auto RandomDividerMount = 25;
    static const auto RandomDividerUnmount = 350;
    static const auto RandomDividerCommit = 25;
    static const auto RandomDividerFileSystem = 50;
    static const auto RandomDividerFile = 5;

    auto doTest = [&](const char* name, int divider, std::function<void ()> action) NN_NOEXCEPT
    {
        randomFailureBufferManager.SetDivider(divider);
        NNT_FS_ASSERT_NO_FATAL_FAILURE(action());
        ASSERT_EQ(nullptr, nn::fssystem::buffers::GetBufferManagerContext());
        NN_LOG("Allocation Count %s %d\n", name, randomFailureBufferManager.GetAllocationCount());
    };

    const size_t sizeFreeSizeStart = GetBufferManager()->GetFreeSize();
    for( size_t sizeBlock = 4096; sizeBlock <= MaxBlockSize; sizeBlock *= 2 )
    {
        NN_LOG("block size: %d/%d\n", static_cast<int>(sizeBlock), static_cast<int>(MaxBlockSize));

        for( auto loop = 0; loop < LoopCount; ++loop )
        {
            NN_LOG("loop count: %d/%d\n", loop + 1, LoopCount);

            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam paramIntegrity;
            const size_t sizeBlock1 = GenerateRandomBlockSize(512, MaxBlockSize);
            const size_t sizeBlock2
                = GenerateRandomBlockSize(512 < sizeBlock1 ? 512 : 1024, MaxBlockSize);
            paramIntegrity.sizeBlockLevel[0] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[1] = sizeBlock1;
            paramIntegrity.sizeBlockLevel[2] = sizeBlock2;
            paramIntegrity.sizeBlockLevel[3] = sizeBlock2;

            const auto countDataBlock = std::uniform_int_distribution<>(100, 120)(mt);
            const auto sizeFile = countDataBlock * sizeBlock * 4 / 5;
            const auto sizeBuffer = sizeFile * 3 / 4;

            int64_t sizeTotal = 0;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::IntegritySaveDataFileSystemDriver::QueryTotalSize(
                    &sizeTotal,
                    sizeBlock,
                    paramIntegrity,
                    countDataBlock
                )
            );
            ASSERT_EQ(sizeFreeSizeStart, GetBufferManager()->GetFreeSize());

            nnt::fs::util::SafeMemoryStorage storageData(sizeTotal);
            nn::fs::SubStorage storageBase(&storageData, 0, sizeTotal);

            NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                "Format", RandomDividerFormat,
                [&]() NN_NOEXCEPT
                {
                    NNT_ASSERT_RESULT_SUCCESS(
                        nn::fssystem::save::IntegritySaveDataFileSystemDriver::Format(
                            storageBase,
                            sizeBlock,
                            paramIntegrity,
                            countDataBlock,
                            &randomFailureBufferManager,
                            nnt::fs::util::GetMacGenerator())
                    );
                }
            ));
            ASSERT_EQ(sizeFreeSizeStart, GetBufferManager()->GetFreeSize());

            NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                "ReadExtraData", RandomDividerFormat,
                [&]() NN_NOEXCEPT
                {
                    nn::fssystem::save::IntegritySaveDataFileSystem::ExtraData extraData;
                    NNT_ASSERT_RESULT_SUCCESS(
                        nn::fssystem::save::IntegritySaveDataFileSystemDriver::ReadExtraData(
                            &extraData,
                            storageBase,
                            &randomFailureBufferManager,
                            nnt::fs::util::GetMacGenerator()
                        )
                    );
                }
            ));
            ASSERT_EQ(sizeFreeSizeStart, GetBufferManager()->GetFreeSize());

            // Mount してファイル読み書きする
            {
                nn::fssystem::save::IntegritySaveDataFileSystemDriver fileSystem;

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "Mount", RandomDividerMount,
                    [&]() NN_NOEXCEPT
                    {
                        NNT_ASSERT_RESULT_SUCCESS(
                            fileSystem.Initialize(storageBase, &randomFailureBufferManager, nnt::fs::util::GetMacGenerator())
                        );
                    }
                ));

                nn::fssystem::save::Path pathFile;
                NNT_ASSERT_RESULT_SUCCESS(pathFile.Initialize("/directory/file.dat"));

                nn::fssystem::save::Path pathDirectory;
                NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize("/directory"));

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "CreateDirectory", RandomDividerFileSystem,
                    [&]() NN_NOEXCEPT
                    {
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory(pathDirectory));
                    }
                ));

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "CreateFile", RandomDividerFileSystem,
                    [&]() NN_NOEXCEPT
                    {
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile(pathFile, 1));
                    }
                ));

                // ファイルアクセス
                {
                    nn::fssystem::save::IFile* pFile = nullptr;
                    const auto openMode = nn::fs::OpenMode_Read | nn::fs::OpenMode_Write;

                    NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                        "OpenFile", RandomDividerFile,
                        [&]() NN_NOEXCEPT
                        {
                            NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&pFile, pathFile, openMode));
                        }
                    ));

                    NN_UTIL_SCOPE_EXIT
                    {
                        NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                            "CloseFile", 0,
                            [&]() NN_NOEXCEPT
                            {
                                fileSystem.CloseFile(pFile);
                            }
                        ));
                    };

                    // サイズの取得と設定
                    {
                        int64_t sizeCurrent;
                        NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                            "File::GetSize", RandomDividerFile,
                            [&]() NN_NOEXCEPT
                            {
                                NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&sizeCurrent));
                                ASSERT_EQ(1, sizeCurrent);
                            }
                        ));

                        NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                            "File::SetSize", RandomDividerFile,
                            [&]() NN_NOEXCEPT
                            {
                                NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(sizeFile));
                                ASSERT_EQ(1, sizeCurrent);
                            }
                        ));
                    }

                    NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                        "File::OperateRange", 2,
                        [&]() NN_NOEXCEPT
                        {
                            char inBuffer[32];
                            char outBuffer[32];

                            NNT_ASSERT_RESULT_FAILURE(
                                nn::fs::ResultUnsupportedOperation,
                                pFile->OperateRange(
                                    outBuffer, sizeof(outBuffer),
                                    nn::fs::OperationId::Invalidate,
                                    0, sizeof(inBuffer),
                                    inBuffer, sizeof(inBuffer)
                                )
                            );
                        }
                    ));

                    nnt::fs::util::Vector<char> readBuffer(sizeBuffer, 0);
                    nnt::fs::util::Vector<char> writeBuffer(sizeBuffer);
                    std::iota(writeBuffer.begin(), writeBuffer.end(), '\x0');

                    for( auto innerLoop = 0; innerLoop < InnerLoopCount; ++innerLoop )
                    {
                        const auto offsetMax = static_cast<int64_t>(sizeFile - sizeBuffer);
                        const auto offset = std::uniform_int_distribution<int64_t>(0, offsetMax)(mt);

                        NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                            "File::WriteBytes", RandomDividerFile,
                            [&]() NN_NOEXCEPT
                            {
                                NNT_ASSERT_RESULT_SUCCESS(
                                    pFile->WriteBytes(
                                        offset,
                                        writeBuffer.data(),
                                        writeBuffer.size()
                                    )
                                );
                            }
                        ));

                        NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                            "File::ReadBytes", RandomDividerFile,
                            [&]() NN_NOEXCEPT
                            {
                                NNT_ASSERT_RESULT_SUCCESS(
                                    pFile->ReadBytes(
                                        offset,
                                        readBuffer.data(),
                                        readBuffer.size()
                                    )
                                );
                            }
                        ));

                        NNT_FS_UTIL_ASSERT_MEMCMPEQ(
                            readBuffer.data(), writeBuffer.data(), readBuffer.size()
                        );

                        NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                            "File::Flush", RandomDividerFile,
                            [&]() NN_NOEXCEPT
                            {
                                NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());
                            }
                        ));
                    }
                }

                // ディレクトリアクセス
                {
                    nn::fssystem::save::IDirectory* pDirectory = nullptr;

                    NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                        "OpenDirectory", RandomDividerFile,
                        [&]() NN_NOEXCEPT
                        {
                            NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenDirectory(&pDirectory, pathDirectory));
                        }
                    ));

                    NN_UTIL_SCOPE_EXIT
                    {
                        NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                            "CloseDirectory", 0,
                            [&]() NN_NOEXCEPT
                            {
                                fileSystem.CloseDirectory(pDirectory);
                            }
                        ));
                    };

                    NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                        "Directory::Read", RandomDividerFile,
                        [&]() NN_NOEXCEPT
                        {
                            nn::fs::DirectoryEntry entry;
                            int currentCount;
                            NNT_ASSERT_RESULT_SUCCESS(pDirectory->Read(&currentCount, &entry, 1));
                        }
                    ));
                }

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "Commit", RandomDividerCommit,
                    [&]() NN_NOEXCEPT
                    {
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.Commit());
                    }
                ));

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "Unmount", RandomDividerUnmount,
                    [&]() NN_NOEXCEPT
                    {
                        fileSystem.Finalize();
                    }
                ));
            }

            ASSERT_EQ(sizeFreeSizeStart, GetBufferManager()->GetFreeSize());

            // 再度マウントしなおして書き込みできることを確認
            {
                nn::fssystem::save::IntegritySaveDataFileSystemDriver fileSystem;

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "Re-Mount", RandomDividerMount,
                    [&]() NN_NOEXCEPT
                    {
                        NNT_ASSERT_RESULT_SUCCESS(
                            fileSystem.Initialize(storageBase, &randomFailureBufferManager, nnt::fs::util::GetMacGenerator())
                        );
                    }
                ));

                nn::fssystem::save::Path pathFile;
                NNT_ASSERT_RESULT_SUCCESS(pathFile.Initialize("/directory/file2.dat"));

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "Re-Mount and CreateFile", RandomDividerFileSystem,
                    [&]() NN_NOEXCEPT
                    {
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile(pathFile, 0));
                    }
                ));

                randomFailureBufferManager.SetDivider(RandomDividerCommit);
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.Commit());

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "Re-Mount and Unmount", RandomDividerUnmount,
                    [&]() NN_NOEXCEPT
                    {
                        fileSystem.Finalize();
                    }
                ));
            }

            ASSERT_EQ(sizeFreeSizeStart, GetBufferManager()->GetFreeSize());

            // ファイルの削除やリネーム
            {
                nn::fssystem::save::IntegritySaveDataFileSystemDriver fileSystem;

                randomFailureBufferManager.SetDivider(RandomDividerMount);
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.Initialize(storageBase, &randomFailureBufferManager, nnt::fs::util::GetMacGenerator())
                );

                nn::fssystem::save::Path pathFile;
                NNT_ASSERT_RESULT_SUCCESS(pathFile.Initialize("/directory/file.dat"));

                nn::fssystem::save::Path pathFile2;
                NNT_ASSERT_RESULT_SUCCESS(pathFile2.Initialize("/directory/file3.dat"));

                nn::fssystem::save::Path pathDirectory;
                NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize("/directory/directory"));

                nn::fssystem::save::Path pathDirectory2;
                NNT_ASSERT_RESULT_SUCCESS(pathDirectory2.Initialize("/directory/directory2"));

                randomFailureBufferManager.SetDivider(RandomDividerFileSystem);
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory(pathDirectory));

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "GetFreeBytes", RandomDividerFileSystem,
                    [&]() NN_NOEXCEPT
                    {
                        int64_t size;
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.GetFreeBytes(&size));
                    }
                ));

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "GetDataAreaBytes", RandomDividerFileSystem,
                    [&]() NN_NOEXCEPT
                    {
                        int64_t size;
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.GetDataAreaBytes(&size));
                    }
                ));

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "HasFile", RandomDividerFileSystem,
                    [&]() NN_NOEXCEPT
                    {
                        bool isExists;
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.HasFile(&isExists, pathFile));
                        ASSERT_TRUE(isExists);
                    }
                ));

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "HasDirectory", RandomDividerFileSystem,
                    [&]() NN_NOEXCEPT
                    {
                        bool isExists;
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.HasDirectory(&isExists, pathDirectory));
                        ASSERT_TRUE(isExists);
                    }
                ));

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "RenameFile", RandomDividerFileSystem,
                    [&]() NN_NOEXCEPT
                    {
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.RenameFile(pathFile, pathFile2));
                    }
                ));

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "RenameDirectory", RandomDividerFileSystem,
                    [&]() NN_NOEXCEPT
                    {
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.RenameDirectory(pathDirectory, pathDirectory2));
                    }
                ));

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "DeleteFile", RandomDividerFileSystem,
                    [&]() NN_NOEXCEPT
                    {
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.DeleteFile(pathFile2));
                    }
                ));

                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "DeleteDirectory", RandomDividerFileSystem,
                    [&]() NN_NOEXCEPT
                    {
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.DeleteDirectory(pathDirectory2));
                    }
                ));

                NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize("/directory"));
                NNT_FS_ASSERT_NO_FATAL_FAILURE(doTest(
                    "DeleteDirectoryRecursively", RandomDividerFileSystem,
                    [&]() NN_NOEXCEPT
                    {
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.DeleteDirectoryRecursively(pathDirectory));
                    }
                ));

                randomFailureBufferManager.SetDivider(RandomDividerCommit);
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.Commit());

                randomFailureBufferManager.SetDivider(RandomDividerMount);
                fileSystem.Finalize();
            }

            ASSERT_EQ(sizeFreeSizeStart, GetBufferManager()->GetFreeSize());
        }
    }

    ASSERT_EQ(sizeFreeSizeStart, GetBufferManager()->GetFreeSize());
} // NOLINT(impl/function_size)
