﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <numeric>
#include <memory>

#include <nn/nn_Common.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os/os_Thread.h>
#include <nn/fssystem/save/fs_IntegritySaveDataFileSystem.h>

#include <nnt/nntest.h>
#include <nnt/fsUtil/testFs_util.h>

#include "testFs_util_CommonFileSystemTests.h"

namespace nn { namespace fssystem { namespace save {

namespace
{
struct TestParam
{
    uint32_t countDataBlock;        // データ本体のブロック数
    uint32_t sizeBlock;             // ブロックサイズ
    size_t sizeBlockLevel0;         // 完全性検証のハッシュストレージのサイズレベル0
    size_t sizeBlockLevel1;         // 完全性検証のハッシュストレージのサイズレベル1
    size_t verificationInputParam2; // 完全性検証のハッシュストレージのサイズレベル2
    size_t verificationInputParam3; // 完全性検証のハッシュストレージのサイズレベル3
    int maxCacheCount;              // 完全性検証用キャッシュバッファの最大キャッシュ数
    size_t cacheBufferPageSize;     // 完全性検証用キャッシュバッファのページサイズ
    size_t cacheBufferMaxOrder;     // 完全性検証用キャッシュバッファの最大キャッシュサイズのオーダー

    int64_t GetEntrySize() const NN_NOEXCEPT
    {
        return static_cast<int64_t>(countDataBlock * sizeBlock);
    }
    size_t GetCacheSize() const NN_NOEXCEPT
    {
        return cacheBufferPageSize * maxCacheCount;
    }
};

nnt::fs::util::Vector<TestParam> MakeTestParam() NN_NOEXCEPT
{
    nnt::fs::util::Vector<TestParam> testParams;

    // 小さめの値
    {
        TestParam data;
        data.countDataBlock = 512;
        data.sizeBlock = 1024;
        data.sizeBlockLevel0 = 4 * 1024;
        data.sizeBlockLevel1 = 256;
        data.verificationInputParam2 = 256;
        data.verificationInputParam3 = 256;
        data.maxCacheCount = 1024;
        data.cacheBufferPageSize = 4 * 1024;
        data.cacheBufferMaxOrder = 6;
        testParams.push_back(data);
    }

    // 大きめの値
    {
        TestParam data;
        data.countDataBlock = 4096;
        data.sizeBlock = 8 * 1024;
        data.sizeBlockLevel0 = 4 * 1024;
        data.sizeBlockLevel1 = 4 * 1024;
        data.verificationInputParam2 = 4 * 1024;
        data.verificationInputParam3 = 4 * 1024;
        data.maxCacheCount = 4 * 1024;
        data.cacheBufferPageSize = 16 * 1024;
        data.cacheBufferMaxOrder = 8;
        testParams.push_back(data);
    }

    // ブロック数少、ブロックサイズ大
    {
        TestParam data;
        data.countDataBlock = 256;
        data.sizeBlock = 32 * 1024;
        data.sizeBlockLevel0 = 16 * 1024;
        data.sizeBlockLevel1 = 4 * 1024;
        data.verificationInputParam2 = 256;
        data.verificationInputParam3 = 64;
        data.maxCacheCount = 1024;
        data.cacheBufferPageSize = 4 * 1024;
        data.cacheBufferMaxOrder = 8;
        testParams.push_back(data);
    }

// PC 上でのみ実行(cacheBufferMaxOrder 大きめ)
#ifdef NN_BUILD_CONFIG_OS_WIN
    {
        TestParam data;
        data.countDataBlock = 1024;
        data.sizeBlock = 256;
        data.sizeBlockLevel0 = 4 * 1024;
        data.sizeBlockLevel1 = 2 * 1024;
        data.verificationInputParam2 = 1024;
        data.verificationInputParam3 = 512;
        data.maxCacheCount = 2048;
        data.cacheBufferPageSize = 4096;
        data.cacheBufferMaxOrder = 12;
        testParams.push_back(data);
    }
#endif
    return testParams;
}
}

nnt::fs::util::Vector<TestParam> MakeLargeTestParam() NN_NOEXCEPT
{
    nnt::fs::util::Vector<TestParam> testParams;

    {
        TestParam data;
        data.countDataBlock = 8 * 1024 * 1024;
        data.sizeBlock = 16 * 1024;
        data.sizeBlockLevel0 = 32 * 1024;
        data.sizeBlockLevel1 = 64 * 1024;
        data.verificationInputParam2 = 4 * 1024;
        data.verificationInputParam3 = 4 * 1024;
        data.maxCacheCount = 4 * 1024;
        data.cacheBufferPageSize = 16 * 1024;
        data.cacheBufferMaxOrder = 8;
        testParams.push_back(data);
    }

    return testParams;
}

class FsIntegritySaveDataFileSystemTestBase : public ::testing::TestWithParam<TestParam>
{
public:

    FsIntegritySaveDataFileSystemTestBase() NN_NOEXCEPT
        : m_BufferManagerSet(),
        m_Locker(true)
    {
    }

    // テストのセットアップ
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        const TestParam& param = GetParam();
        NNT_ASSERT_RESULT_SUCCESS(FormatFileSystem(param));
        NNT_ASSERT_RESULT_SUCCESS(MountFileSystem());
    }

    // テスト終了処理
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        m_SaveDataFileSystem.Finalize();
    }

    Result FormatFileSystem(const TestParam& param) NN_NOEXCEPT
    {
        nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam inputParamIntegrity;
        inputParamIntegrity.sizeBlockLevel[0] = param.sizeBlockLevel0;
        inputParamIntegrity.sizeBlockLevel[1] = param.sizeBlockLevel1;
        inputParamIntegrity.sizeBlockLevel[2] = param.verificationInputParam2;
        inputParamIntegrity.sizeBlockLevel[3] = param.verificationInputParam3;
        InitializeBuffer(param);

        uint32_t sizeBlock = param.sizeBlock;

        // ストレージのサイズを取得し、ストレージ確保します
        int64_t mainStorageSize;
        NN_RESULT_DO(
            m_SaveDataFileSystem.QuerySize(
                &mainStorageSize,
                inputParamIntegrity,
                sizeBlock,
                param.countDataBlock
            )
        );
        InitializeMainStorage(mainStorageSize);
        m_MainSubStorage = fs::SubStorage(GetMainStorage(), 0, mainStorageSize);

        // フォーマット
        NN_RESULT_DO(m_SaveDataFileSystem.Format(
            m_MainSubStorage,
            inputParamIntegrity,
            sizeBlock,
            param.countDataBlock,
            &m_BufferManagerSet,
            &m_BufferManager,
            &m_Locker,
            nnt::fs::util::GetMacGenerator()
        ));

        NN_RESULT_SUCCESS;
    }

    Result MountFileSystem() NN_NOEXCEPT
    {
        // 初期化
        NN_RESULT_DO(m_SaveDataFileSystem.Initialize(
            m_MainSubStorage,
            &m_BufferManagerSet,
            &m_BufferManager,
            &m_Locker,
            nnt::fs::util::GetMacGenerator()
        ));
        NN_RESULT_SUCCESS;
    }

    void UnmountFileSystem() NN_NOEXCEPT
    {
        m_SaveDataFileSystem.Finalize();
    }

protected:
    virtual void InitializeMainStorage(int64_t size) NN_NOEXCEPT = 0;
    virtual nn::fs::IStorage* GetMainStorage() NN_NOEXCEPT = 0;

    IntegritySaveDataFileSystem* GetSaveDataFileSystem() NN_NOEXCEPT
    {
        return &m_SaveDataFileSystem;
    }

    // 完全性検証ハッシュに割り当てるキャッシュバッファを設定します
    void InitializeBuffer(const TestParam& param) NN_NOEXCEPT
    {
        const size_t cacheBufferMax = static_cast<size_t>(1) << (param.cacheBufferMaxOrder - 1);
        const size_t sizeUnitBlock = cacheBufferMax * param.cacheBufferPageSize;
        const size_t sizeCacheBuffer = sizeUnitBlock * nn::fssystem::save::IntegrityLayerCountSave;
        m_CacheBuffer.reset(new char[sizeCacheBuffer]);

        // バッファマネージャ初期化
        NNT_ASSERT_RESULT_SUCCESS(
            m_BufferManager.Initialize(
                param.maxCacheCount,
                reinterpret_cast<uintptr_t>(m_CacheBuffer.get()),
                sizeCacheBuffer,
                param.cacheBufferPageSize
            )
        );

        // バッファマネージャセット作成
        for( size_t i = 0; i < IntegrityLayerCountSave; ++i )
        {
            m_BufferManagerSet.pBuffer[i] = &m_BufferManager;
        }
    }

    // /dir<directoryId> というパスが取得できます
    void GetDirectoryPath(Path* outPath, int directoryId) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(directoryId < 1000);
        char direcrtoryPath[8];
        size_t bufferSize = 8;
        nn::util::SNPrintf(direcrtoryPath, bufferSize, "/dir%d/", directoryId);
        NNT_ASSERT_RESULT_SUCCESS(outPath->Initialize(direcrtoryPath));
    };

    // /dir<directoryId1>/dir<directoryId2> というパスが取得できます
    void GetDirectoryPath(Path* outPath, int directoryId1, int directoryId2) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(directoryId1 < 1000);
        NN_SDK_REQUIRES(directoryId2 < 1000);
        char direcrtoryPath[15];
        size_t bufferSize = 15;
        nn::util::SNPrintf(direcrtoryPath, bufferSize, "/dir%d/dir%d/", directoryId1, directoryId2);
        NNT_ASSERT_RESULT_SUCCESS(outPath->Initialize(direcrtoryPath));
    };

    // /file<fileId> というパスが取得できます
    void GetFilePath(Path* outPath, int fileId) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(fileId < 1000);
        char filePath[8];
        size_t bufferSize = 8;
        nn::util::SNPrintf(filePath, bufferSize, "/file%d", fileId);
        NNT_ASSERT_RESULT_SUCCESS(outPath->Initialize(filePath));
    };

    // /dir<directoryId>/file<fileId> というパスが取得できます
    void GetFilePath(Path* outPath, int directoryId, int fileId) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(directoryId < 1000);
        NN_SDK_REQUIRES(fileId < 1000);
        char filePath[15];
        size_t bufferSize = 15;
        nn::util::SNPrintf(filePath, bufferSize, "/dir%d/file%d", directoryId, fileId);
        NNT_ASSERT_RESULT_SUCCESS(outPath->Initialize(filePath));
    };

private:
    // ストレージ本体
    IntegritySaveDataFileSystem m_SaveDataFileSystem;

    // 下に敷くストレージなど
    nnt::fs::util::AccessCountedMemoryStorage m_MainStorage;
    nn::fssystem::save::FilesystemBufferManagerSet m_BufferManagerSet;
    nn::fssystem::FileSystemBufferManager m_BufferManager;
    std::unique_ptr<char[]> m_CacheBuffer;

    // マウント用の値
    fs::SubStorage m_MainSubStorage;
    nn::os::Mutex m_Locker;
};

class FsIntegritySaveDataFileSystemTest : public FsIntegritySaveDataFileSystemTestBase
{
protected:
    virtual void InitializeMainStorage(int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        m_MainStorage.Initialize(size);
    }

    virtual nn::fs::IStorage* GetMainStorage() NN_NOEXCEPT NN_OVERRIDE
    {
        return &m_MainStorage;
    }

    nnt::fs::util::AccessCountedMemoryStorage* GetAccessCountStorage() NN_NOEXCEPT
    {
        return &m_MainStorage;
    }

private:
    nnt::fs::util::AccessCountedMemoryStorage m_MainStorage;
};

class FsIntegritySaveDataFileSystemLargeTest : public FsIntegritySaveDataFileSystemTestBase
{
protected:
    virtual void InitializeMainStorage(int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        m_MainStorage.Initialize(size);
    }

    virtual nn::fs::IStorage* GetMainStorage() NN_NOEXCEPT NN_OVERRIDE
    {
        return &m_MainStorage;
    }

private:
    nnt::fs::util::VirtualMemoryStorage m_MainStorage;
};

/**
* 成功パターンテスト
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestSimple)
{
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    const int fileSize = 1024;
    std::unique_ptr<char[]> writeBuffer(new char[static_cast<size_t>(fileSize)]);
    std::iota(writeBuffer.get(), writeBuffer.get() + fileSize, '\x1');
    Path path;

    // ファイルを作成しコミット
    {
        GetDirectoryPath(&path, 1);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));

        GetFilePath(&path, 1, 1);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateFile(path, fileSize));

        {
            IFile* pFile;
            NNT_ASSERT_RESULT_SUCCESS(
                pSaveDataFileSystem->OpenFile(
                    &pFile,
                    path,
                    nn::fs::OpenMode_Write
                )
            );
            NN_UTIL_SCOPE_EXIT
            {
                pSaveDataFileSystem->CloseFile(pFile);
            };

            NNT_ASSERT_RESULT_SUCCESS(
                pFile->WriteBytes(0, writeBuffer.get(), static_cast<size_t>(fileSize))
            );
            NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());
        }

        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());
    }

    // 作成したファイルを読み込み、書き込んだバッファと比較します
    {
        GetFilePath(&path, 1, 1);

        IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(
                &pFile,
                path,
                nn::fs::OpenMode_Read
            )
        );
        NN_UTIL_SCOPE_EXIT
        {
            pSaveDataFileSystem->CloseFile(pFile);
        };

        std::unique_ptr<char[]> readBuffer(new char[static_cast<size_t>(fileSize)]);
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->ReadBytes(0, readBuffer.get(), static_cast<size_t>(fileSize))
        );
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(writeBuffer.get(), readBuffer.get(), fileSize);
    }
}

/**
* 4 GB を超えるオフセットのテスト
*/
TEST_P(FsIntegritySaveDataFileSystemLargeTest, TestSimple)
{
    TestLargeOffsetAccess(GetSaveDataFileSystem(), "/file");
}

/**
* サイズが 0バイトのファイルと 0 バイトでないファイルをいくつか作成し、0バイト書き込みを行います
* 0 バイト書き込みでは、コミットしても下位ストレージにアクセスしないことを確かめます
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestCheckAccsessCountZero)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    const TestParam& param = GetParam();

    int64_t bufferSize = param.GetEntrySize();
    std::unique_ptr<char[]> writeBuffer(new char[static_cast<size_t>(bufferSize)]);
    std::iota(writeBuffer.get(), writeBuffer.get() + bufferSize, '\x1');

    nn::fssystem::save::Path path;
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();

    auto pAccessCountStorage = GetAccessCountStorage();

    // サイズ 0 のファイルをいくつか作成しコミット
    int countSize0 = std::uniform_int_distribution<>(1, 5)(mt);
    GetDirectoryPath(&path, 0);
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));
    for( int i = 0; i < countSize0; ++i )
    {
        GetFilePath(&path, 0, i);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateFile(path, 0));
    }
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());

    // ランダムサイズのファイルをいくつか作成しコミット
    int countSizeRandom = std::uniform_int_distribution<>(1, 5)(mt);
    GetDirectoryPath(&path, 1);
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));
    for( int i = 0; i < countSizeRandom; ++i )
    {
        GetFilePath(&path, 1, i);
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->CreateFile(
                path,
                std::uniform_int_distribution<>(1, 1024)(mt)
            )
        );
    }
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());

    pAccessCountStorage->ResetAccessCounter();

    // 各ファイルを開き、0 バイト書き込み
    {
        for( int i = 0; i < countSize0; ++i )
        {
            GetFilePath(&path, 0, i);

            IFile* pFile;
            NNT_ASSERT_RESULT_SUCCESS(
                pSaveDataFileSystem->OpenFile(
                    &pFile,
                    path,
                    nn::fs::OpenMode_Write
                )
            );
            NN_UTIL_SCOPE_EXIT
            {
                pSaveDataFileSystem->CloseFile(pFile);
            };
            NNT_ASSERT_RESULT_SUCCESS(
                pFile->WriteBytes(0, writeBuffer.get(), 0)
            );
        }
        for( int i = 0; i < countSizeRandom; ++i )
        {
            GetFilePath(&path, 1, i);

            IFile* pFile;
            NNT_ASSERT_RESULT_SUCCESS(
                pSaveDataFileSystem->OpenFile(
                    &pFile,
                    path,
                    nn::fs::OpenMode_Write
                )
            );
            NN_UTIL_SCOPE_EXIT
            {
                pSaveDataFileSystem->CloseFile(pFile);
            };

            int64_t fileSize = 0;
            NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&fileSize));
            int64_t offset
                = std::uniform_int_distribution<int64_t>(1, fileSize)(mt);
            NNT_ASSERT_RESULT_SUCCESS(
                pFile->WriteBytes(offset, writeBuffer.get(), 0)
            );
        }
    }

    // アクセス回数をチェック
    ASSERT_EQ(0, pAccessCountStorage->GetWriteTimes());
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());
    ASSERT_EQ(0, pAccessCountStorage->GetWriteTimes());
}

/**
* コミットしない場合、アンマウント時に下位ストレージにアクセスしないことを確かめます
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestCheckAccsessCountUnmount)
{
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    const TestParam& param = GetParam();

    int64_t bufferSize = param.GetEntrySize() - param.sizeBlock * 8;
    std::unique_ptr<char[]> writeBuffer(new char[static_cast<size_t>(bufferSize)]);
    std::iota(writeBuffer.get(), writeBuffer.get() + bufferSize, '\x1');

    nn::fssystem::save::Path path;
    {
        int64_t fileSize = bufferSize * 1 / 8;

        // ファイルを 8 つ作成
        GetDirectoryPath(&path, 0);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));
        for( int i = 0; i < 8; ++i )
        {
            GetFilePath(&path, 0, i);
            NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateFile(path, fileSize));
        }
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());

        // ファイル書き込み
        {
            for( int i = 0; i < 8; ++i )
            {
                GetFilePath(&path, 0, i);
                IFile* pFile;
                NNT_ASSERT_RESULT_SUCCESS(
                    pSaveDataFileSystem->OpenFile(
                        &pFile,
                        path,
                        nn::fs::OpenMode_Write
                    )
                );
                NN_UTIL_SCOPE_EXIT
                {
                    pSaveDataFileSystem->CloseFile(pFile);
                };
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->WriteBytes(0, writeBuffer.get(), static_cast<size_t>(fileSize))
                );
                NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());
            }
        }

        auto pAccessCountStorage = GetAccessCountStorage();

        // 書き込みを行ってからアンマウントすると、キャッシュ解放のためにアクセスが発生します
        UnmountFileSystem();
        ASSERT_LT(0, pAccessCountStorage->GetWriteTimes());

        // アクセスチェックをしながら再マウント
        pAccessCountStorage->ResetAccessCounter();
        NNT_ASSERT_RESULT_SUCCESS(MountFileSystem());
        UnmountFileSystem();
        ASSERT_EQ(0, pAccessCountStorage->GetWriteTimes());
    }
}

/**
* ディレクトリを 100 個作成するテスト
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestCreateDirectory)
{
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    nn::fssystem::save::Path path;

    for( int i = 0; i < 100; ++i )
    {
        GetDirectoryPath(&path, i);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());
    }
}

/**
* これ以上ファイルが作成できなくなるまでファイルを作成します
* 1 ブロックに作成できる数のファイルエントリを作成し、残りのブロックをファイルデータで埋めます
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestCreateFileHeavy)
{
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    nn::fssystem::save::Path path;

    const TestParam& param = GetParam();

    // 1 ブロック内に作成できるファイルエントリ数
    const int fileEntryCount = param.sizeBlock / 96 - 2;

    // ファイルの中身が書きこめるブロック数
    // (ファイルエントリ用とディレクトリエントリ用に 2 ブロック必要)
    const int fileBlockCount = param.countDataBlock - 2;

    if( fileBlockCount <= 1 || fileEntryCount <= 1)
    {
        return;
    }

    // サイズが fileSize のファイルを (fileEntryCount - 1) 個作成することで
    // (fileBlockCount - 1) 個以下のブロックを埋めます
    const int64_t fileSize = param.sizeBlock * ((fileBlockCount - 1) / (fileEntryCount - 1));
    if( fileSize == 0 )
    {
        // ブロック数が多くエントリ数が少ない場合は、このテストをスキップします
        return;
    }

    GetDirectoryPath(&path, 0);
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));
    for( int i = 0; i < fileEntryCount - 1; ++i )
    {
        GetFilePath(&path, 0, i);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateFile(path, fileSize));
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());

        // 再マウントでキャッシュをリセット
        UnmountFileSystem();
        NNT_ASSERT_RESULT_SUCCESS(MountFileSystem());
    }

    // この時点で、usedBlockByFileData 個のブロックが使われています
    const int64_t usedBlockByFileData = (fileSize / param.sizeBlock) * (fileEntryCount - 1) + 2;

    // なので使われていないブロックは leftBlockCount 個
    const int64_t leftBlockCount = param.countDataBlock - usedBlockByFileData;

    // 1 個以上は残っています
    ASSERT_LT(0, leftBlockCount);

    // 残りのブロックを 1 つだけ残して、1つのファイルで埋めます
    const int64_t paddingFileSize = leftBlockCount * param.sizeBlock;

    // freeByte を取得すると、残りサイズと同じになります
    int64_t freeByte = 0;
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->GetFreeBytes(&freeByte));
    ASSERT_EQ(freeByte, leftBlockCount * param.sizeBlock);

    GetFilePath(&path, 0, fileEntryCount);
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateFile(path, paddingFileSize));

    // これ以上はサイズ 1 のファイルすら作成できません
    GetFilePath(&path, 0, fileEntryCount + 1);
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAllocationTableFull,
        pSaveDataFileSystem->CreateFile(path, 1)
    );
}

/**
* 空き容量の計算が正しいか確認します
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestCalculateFreeBytesHeavy)
{
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    nn::fssystem::save::Path path;
    const TestParam& param = GetParam();

    const auto loopCount = 3;

    const auto directoryEntrySize = 96;
    const auto fileEntrySize = 96;

    // 作成したエントリの数を記憶しておきます
    int countDirectoryEntry = 3;    // ディレクトリエントリリストは予約ブロックが 3 個あります
    int countFileEntry = 2;         // ファイルエントリリストはヘッダブロックが 2 個あります

    // すでに確保したブロックと作成したエントリの数を記憶しておきます
    int countDirectoryEntryBlock = nn::util::DivideUp(
        countDirectoryEntry * directoryEntrySize, param.sizeBlock);
    int countFileEntryBlock = nn::util::DivideUp(countFileEntry * fileEntrySize, param.sizeBlock);

    // フリーブロック数が引数と同じかチェックするラムダ式
    auto freeBytesCheck = [&](int64_t expectedFreeBlockCount) NN_NOEXCEPT
    {
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());
        int64_t actualFreeBytes = 0;
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->GetFreeBytes(&actualFreeBytes));
        const int64_t expectedSize = expectedFreeBlockCount * param.sizeBlock;
        ASSERT_EQ(expectedSize, actualFreeBytes);
    };
    int lastFreeBlockCount = param.countDataBlock
        - (countDirectoryEntryBlock + countFileEntryBlock);

    // 初期状態の空き容量を計算します
    {
        freeBytesCheck(lastFreeBlockCount);
    }

    // ディレクトリとファイルを作成しながら残り空き容量を計算する処理を 3 回行います
    for( int loop = 0; loop < loopCount; ++loop )
    {
        // ディレクトリ作成後の空き容量計算
        {
            // ルート直下に 1 個作成します
            GetDirectoryPath(&path, loop);
            NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));
            NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());

            // エントリ確保
            ++countDirectoryEntry;

            const int countCreate = countDirectoryEntryBlock * param.sizeBlock / directoryEntrySize
                - countDirectoryEntry;

            // 作成したディレクトリの中に countCreate 個作成します
            for( int i = 0; i < countCreate; ++i )
            {
                GetDirectoryPath(&path, loop, i);
                NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));
                NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());

                ++countDirectoryEntry;

                // この時点ではまだブロックが消費されていない
                freeBytesCheck(lastFreeBlockCount);
            }

            // さらに 1 個作成すると、ブロックが消費されます
            GetDirectoryPath(&path, loop, countCreate);
            NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));
            NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());

            --lastFreeBlockCount;
            ++countDirectoryEntry;
            ++countDirectoryEntryBlock;

            // 減っているか確認します。
            freeBytesCheck(lastFreeBlockCount);
        }

        // ファイル作成後の空き容量計算
        {
            const int countCreate = countFileEntryBlock * param.sizeBlock / fileEntrySize
                - countFileEntry;

            // これ以上作成できないなら終了します
            if( lastFreeBlockCount < countCreate + 1 )
            {
                return;
            }

            // 作成したディレクトリの中に countCreate 個作成します
            for( int i = 0; i < countCreate; ++i )
            {
                GetFilePath(&path, loop, i);
                NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateFile(path, 1));
                NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());
                ++countFileEntry;

                // ファイル 1 つにつきブロック 1 つを消費します
                --lastFreeBlockCount;
                freeBytesCheck(lastFreeBlockCount);
            }

            // さらに 1 個作成すると、ブロックを消費します
            GetFilePath(&path, loop, countCreate);
            NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateFile(path, 1));

            --lastFreeBlockCount;
            --lastFreeBlockCount;
            ++countFileEntry;
            ++countFileEntryBlock;

            // 減っているか確認します。
            freeBytesCheck(lastFreeBlockCount);
        }

        // これ以上作成できないなら終了します
        if( lastFreeBlockCount < 1 )
        {
            return;
        }
    }
}

/**
* ファイルに書き込みコミットした後に、書いた内容を読み取ります
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestReadWriteSimple)
{
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    const TestParam& param = GetParam();

    int64_t bufferSize = param.GetEntrySize();
    std::unique_ptr<char[]> writeBuffer(new char[static_cast<size_t>(bufferSize)]);
    std::unique_ptr<char[]> readBuffer(new char[static_cast<size_t>(bufferSize)]);

    std::iota(writeBuffer.get(), writeBuffer.get() + bufferSize, '\x1');

    nn::fssystem::save::Path path;
    int64_t fileSize = bufferSize * 1 / 2;
    // ディレクトリとファイルを作成します
    {
        GetDirectoryPath(&path, 0);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));
        GetFilePath(&path, 0, 1);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateFile(path, fileSize));
    }

    // ファイルに書き込んでコミットします
    IFile* pFile;
    {
        GetFilePath(&path, 0, 1);
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(
                &pFile,
                path,
                nn::fs::OpenMode_Write
            )
        );
        NN_UTIL_SCOPE_EXIT
        {
            pSaveDataFileSystem->CloseFile(pFile);
        };

        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(0, writeBuffer.get(), static_cast<size_t>(fileSize))
        );
        NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());
    }
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());

    // ファイルに書き込んだデータが読み込めるか確認します
    {
        GetFilePath(&path, 0, 1);
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(
                &pFile,
                path,
                nn::fs::OpenMode_Read
            )
        );
        NN_UTIL_SCOPE_EXIT
        {
            pSaveDataFileSystem->CloseFile(pFile);
        };
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->ReadBytes(0, readBuffer.get(), static_cast<size_t>(fileSize))
        );
    }
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(readBuffer.get(), writeBuffer.get(), static_cast<size_t>(fileSize));
}

/**
* ランダムなオフセットで書き込みを行い、書いた内容をランダムなオフセットで読み取ります
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestReadWriteFractions)
{
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    const TestParam& param = GetParam();
    int64_t bufferSize = (param.GetEntrySize() - 4 * param.countDataBlock) / 10;

    std::unique_ptr<char[]> writeBuffer(new char[static_cast<size_t>(bufferSize)]);
    std::unique_ptr<char[]> readBuffer(new char[static_cast<size_t>(bufferSize)]);
    std::iota(writeBuffer.get(), writeBuffer.get() + bufferSize, '\x1');

    nn::fssystem::save::Path path;
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    // 10 回行います
    for( int i = 0; i < 10; ++i )
    {
        int64_t fileSize = std::uniform_int_distribution<int64_t>(1, bufferSize)(mt);
        // ファイルを作成
        {
            GetDirectoryPath(&path, 0);
            NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));
            GetFilePath(&path, 0, i);
            NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateFile(path, fileSize));
            NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());
        }

        // 書き込んでコミット
        int64_t writeOffset
            = std::uniform_int_distribution<int64_t>(1, fileSize)(mt);
        {
            GetFilePath(&path, 0, i);
            IFile* pFile;
            NNT_ASSERT_RESULT_SUCCESS(
                pSaveDataFileSystem->OpenFile(
                    &pFile,
                    path,
                    nn::fs::OpenMode_Write
                )
            );
            NN_UTIL_SCOPE_EXIT
            {
                pSaveDataFileSystem->CloseFile(pFile);
            };
            NNT_ASSERT_RESULT_SUCCESS(
                pFile->WriteBytes(
                    writeOffset,
                    writeBuffer.get(),
                    static_cast<size_t>(fileSize - writeOffset)
                )
            );
            NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());
        }
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());

        // 書き込んだデータを読み込みます
        int64_t readOffset
            = std::uniform_int_distribution<int64_t>(
                  writeOffset,
                  fileSize
              )(mt);
        {
            GetFilePath(&path, 0, i);
            IFile* pFile;
            NNT_ASSERT_RESULT_SUCCESS(
                pSaveDataFileSystem->OpenFile(
                    &pFile,
                    path,
                    nn::fs::OpenMode_Read
                )
            );
            NN_UTIL_SCOPE_EXIT
            {
                pSaveDataFileSystem->CloseFile(pFile);
            };
            NNT_ASSERT_RESULT_SUCCESS(
                pFile->ReadBytes(
                    readOffset,
                    readBuffer.get(),
                    static_cast<size_t>(fileSize - readOffset)
                )
            );
        }
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(
            readBuffer.get(),
            writeBuffer.get() + (readOffset - writeOffset),
            static_cast<size_t>(fileSize - readOffset)
        );

        // ファイルを削除
        {
            GetFilePath(&path, 0, i);
            NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->DeleteFile(path));
            NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());
            GetDirectoryPath(&path, 0);
            NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->DeleteDirectory(path));
        }
    }
}

/**
* 書き込み、コミットした後に再マウントしてから Read できるか確かめます
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestReadWriteRemount)
{
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    const TestParam& param = GetParam();
    int64_t bufferSize = param.GetEntrySize();

    std::unique_ptr<char[]> writeBuffer(new char[static_cast<size_t>(bufferSize)]);
    std::unique_ptr<char[]> readBuffer(new char[static_cast<size_t>(bufferSize)]);
    std::iota(writeBuffer.get(), writeBuffer.get() + bufferSize, '\x1');

    nn::fssystem::save::Path path;
    int64_t fileSize = bufferSize * 1 / 2;
    // ファイルを作成
    {
        GetDirectoryPath(&path, 0);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));
        GetFilePath(&path, 0, 1);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateFile(path, fileSize));
    }

    // 書き込みます
    IFile* pFile;
    {
        GetFilePath(&path, 0, 1);
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(
                &pFile,
                path,
                nn::fs::OpenMode_Write
            )
        );
        NN_UTIL_SCOPE_EXIT
        {
            pSaveDataFileSystem->CloseFile(pFile);
        };
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(0, writeBuffer.get(), static_cast<size_t>(fileSize))
        );
        NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());
    }

    // コミットしてアンマウント
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());
    UnmountFileSystem();
    NNT_ASSERT_RESULT_SUCCESS(MountFileSystem());

    // 書き込んだデータを読み込みます
    {
        GetFilePath(&path, 0, 1);
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(
                &pFile,
                path,
                nn::fs::OpenMode_Read
            )
        );
        NN_UTIL_SCOPE_EXIT
        {
            pSaveDataFileSystem->CloseFile(pFile);
        };
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->ReadBytes(0, readBuffer.get(), static_cast<size_t>(fileSize))
        );
    }
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(readBuffer.get(), writeBuffer.get(), static_cast<size_t>(fileSize));
}

/**
* 書き込んだ後に読み込みを行うと、書き込んだ値が読み込めることを確かめます
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestReadWritePreCommit)
{
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    const TestParam& param = GetParam();

    int64_t bufferSize = param.GetEntrySize() - param.sizeBlock * 4;

    std::unique_ptr<char[]> writeBuffer(new char[static_cast<size_t>(bufferSize)]);
    std::unique_ptr<char[]> readBuffer(new char[static_cast<size_t>(bufferSize)]);
    std::iota(writeBuffer.get(), writeBuffer.get() + bufferSize, '\x1');

    nn::fssystem::save::Path path;

    int64_t fileSize = bufferSize;
    // ファイルを作成
    {
        GetDirectoryPath(&path, 0);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));
        GetFilePath(&path, 0, 1);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateFile(path, fileSize));
    }
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());

    // 書き込みます
    {
        GetFilePath(&path, 0, 1);
        IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(
                &pFile,
                path,
                nn::fs::OpenMode_Write
            )
        );
        NN_UTIL_SCOPE_EXIT
        {
            pSaveDataFileSystem->CloseFile(pFile);
        };
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(0, writeBuffer.get(), static_cast<size_t>(fileSize))
        );
        NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());
    }

    // コミットせずに読み込みます
    {
        GetFilePath(&path, 0, 1);
        IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(
                &pFile,
                path,
                nn::fs::OpenMode_Read
            )
        );
        NN_UTIL_SCOPE_EXIT
        {
            pSaveDataFileSystem->CloseFile(pFile);
        };
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->ReadBytes(0, readBuffer.get(), static_cast<size_t>(fileSize))
        );
    }
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(readBuffer.get(), writeBuffer.get(), static_cast<size_t>(fileSize));
}

/**
* 書き込み後にコミットせずアンマウントすると、
* ハッシュ検証エラーが発生することを確かめます
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestReadWriteNoCommit)
{
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    const TestParam& param = GetParam();
    int64_t bufferSize = param.GetEntrySize() - param.sizeBlock * 4;

    std::unique_ptr<char[]> writeBuffer(new char[static_cast<size_t>(bufferSize)]);
    std::iota(writeBuffer.get(), writeBuffer.get() + bufferSize, '\x1');
    std::unique_ptr<char[]> readBuffer(new char[static_cast<size_t>(bufferSize)]);

    nn::fssystem::save::Path path;

    // ファイルを作成
    int64_t fileSize = bufferSize * 1 / 4;
    {
        GetDirectoryPath(&path, 0);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));
        GetFilePath(&path, 0, 0);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateFile(path, fileSize));
        GetFilePath(&path, 0, 1);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateFile(path, fileSize));
        GetFilePath(&path, 0, 2);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateFile(path, fileSize));
    }
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());

    // 書き込んでコミットします
    {
        GetFilePath(&path, 0, 1);
        IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(
                &pFile,
                path,
                nn::fs::OpenMode_Write
            )
        );
        NN_UTIL_SCOPE_EXIT
        {
            pSaveDataFileSystem->CloseFile(pFile);
        };
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(0, writeBuffer.get(), static_cast<size_t>(fileSize))
        );
        NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());
    }
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());

    // 書き込んでコミットしません
    {
        std::unique_ptr<char[]> overwriteBuffer(new char[static_cast<size_t>(bufferSize)]);
        std::iota(overwriteBuffer.get(), overwriteBuffer.get() + bufferSize, '\x64');
        GetFilePath(&path, 0, 1);
        IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(
                &pFile,
                path,
                nn::fs::OpenMode_Write
            )
        );
        NN_UTIL_SCOPE_EXIT
        {
            pSaveDataFileSystem->CloseFile(pFile);
        };
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(0, overwriteBuffer.get(), static_cast<size_t>(fileSize))
        );
        NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());
    }

    // 再マウント
    UnmountFileSystem();
    NNT_ASSERT_RESULT_SUCCESS(MountFileSystem());

    // 書いた内容を読み取ります
    {
        GetFilePath(&path, 0, 1);

        IFile* pFile;
        auto result = pSaveDataFileSystem->OpenFile(
            &pFile,
            path,
            nn::fs::OpenMode_Read
        );

        if( result.IsFailure() )
        {
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultNonRealDataVerificationFailed,
                result
            );
        }
        else
        {
            NN_UTIL_SCOPE_EXIT
            {
                pSaveDataFileSystem->CloseFile(pFile);
            };

            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultUnclearedRealDataVerificationFailed,
                pFile->ReadBytes(0, readBuffer.get(), static_cast<size_t>(fileSize))
            );
        }
    }
}

/**
* とびとびで Write した後 Commit してから Read を行います
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestReadWriteSplit)
{
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    const TestParam& param = GetParam();
    int64_t bufferSize = param.GetEntrySize();
    int64_t fileSize = bufferSize - param.sizeBlock * 4;

    std::unique_ptr<char[]> writeBuffer(new char[static_cast<size_t>(bufferSize)]);
    std::unique_ptr<char[]> readBuffer(new char[static_cast<size_t>(bufferSize)]);
    std::unique_ptr<char[]> overwriteBuffer(new char[static_cast<size_t>(bufferSize)]);

    std::iota(writeBuffer.get(), writeBuffer.get() + bufferSize, '\x1');
    std::iota(overwriteBuffer.get(), overwriteBuffer.get() + bufferSize, '\x64');

    nn::fssystem::save::Path path;

    // ファイルを作成
    {
        GetDirectoryPath(&path, 0);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));
        GetFilePath(&path, 0, 1);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateFile(path, fileSize));
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());
    }

    // いったん全体に書き込んでコミット
    {
        GetFilePath(&path, 0, 1);
        IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(
                &pFile,
                path,
                nn::fs::OpenMode_Write
            )
        );
        NN_UTIL_SCOPE_EXIT
        {
            pSaveDataFileSystem->CloseFile(pFile);
        };
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(0, writeBuffer.get(), static_cast<size_t>(fileSize))
        );
        NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());
    }
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());

    // とびとびで書き込んでコミットします
    {
        GetFilePath(&path, 0, 1);
        IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(
                &pFile,
                path,
                nn::fs::OpenMode_Write
            )
        );
        NN_UTIL_SCOPE_EXIT
        {
            pSaveDataFileSystem->CloseFile(pFile);
        };
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(
                fileSize / 8,
                overwriteBuffer.get() + fileSize / 8,
                static_cast<size_t>(fileSize / 8)
            )
        );
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(
                fileSize * 3 / 8,
                overwriteBuffer.get() + fileSize * 3 / 8,
                static_cast<size_t>(fileSize / 8)
            )
        );
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(
                fileSize * 6 / 8,
                overwriteBuffer.get() + fileSize * 6 / 8,
                static_cast<size_t>(fileSize / 8)
            )
        );
        NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());
    }
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());

    // 読み込んだデータがとびとびで書き込んだものであることを確かめます
    {
        GetFilePath(&path, 0, 1);
        IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(
                &pFile,
                path,
                nn::fs::OpenMode_Read
            )
        );
        NN_UTIL_SCOPE_EXIT
        {
            pSaveDataFileSystem->CloseFile(pFile);
        };
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->ReadBytes(0, readBuffer.get(), static_cast<size_t>(fileSize))
        );
    }
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(
        readBuffer.get() + fileSize * 0 / 8,
        writeBuffer.get() + fileSize * 0 / 8,
        static_cast<size_t>(fileSize / 8)
    );
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(
        readBuffer.get() + fileSize * 1 / 8,
        overwriteBuffer.get() + fileSize * 1 / 8,
        static_cast<size_t>(fileSize / 8)
    );
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(
        readBuffer.get() + fileSize * 2 / 8,
        writeBuffer.get() + fileSize * 2 / 8,
        static_cast<size_t>(fileSize / 8)
    );
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(
        readBuffer.get() + fileSize * 3 / 8,
        overwriteBuffer.get() + fileSize * 3 / 8,
        static_cast<size_t>(fileSize / 8)
    );
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(
        readBuffer.get() + fileSize * 4 / 8,
        writeBuffer.get() + fileSize * 4 / 8,
        static_cast<size_t>(fileSize / 8)
    );
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(
        readBuffer.get() + fileSize * 5 / 8,
        writeBuffer.get() + fileSize * 5 / 8,
        static_cast<size_t>(fileSize / 8)
    );
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(
        readBuffer.get() + fileSize * 6 / 8,
        overwriteBuffer.get() + fileSize * 6 / 8,
        static_cast<size_t>(fileSize / 8)
    );
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(
        readBuffer.get() + fileSize * 7 / 8,
        writeBuffer.get() + fileSize * 7 / 8,
        static_cast<size_t>(fileSize / 8)
    );
} // NOLINT(impl/function_size)

/**
* とびとびで Write した後 commit せずに再マウントします
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestReadWriteSplitNoCommit)
{
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    const TestParam& param = GetParam();
    int64_t bufferSize = param.GetEntrySize() - param.sizeBlock * 4;
    std::unique_ptr<char[]> writeBuffer(new char[static_cast<size_t>(bufferSize)]);
    std::unique_ptr<char[]> readBuffer(new char[static_cast<size_t>(bufferSize)]);

    std::iota(writeBuffer.get(), writeBuffer.get() + bufferSize, '\x1');
    nn::fssystem::save::Path path;

    int64_t fileSize = bufferSize;

    // ファイルを作成
    {
        GetDirectoryPath(&path, 0);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));
        GetFilePath(&path, 0, 1);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateFile(path, fileSize));
    }
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());

    // 書き込んでコミットします
    {
        GetFilePath(&path, 0, 1);
        IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(
                &pFile,
                path,
                nn::fs::OpenMode_Write
            )
        );
        NN_UTIL_SCOPE_EXIT
        {
            pSaveDataFileSystem->CloseFile(pFile);
        };
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(0, writeBuffer.get(), static_cast<size_t>(fileSize))
        );
        NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());
    }
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());

    // 違う値を書き込んでコミットしません
    {
        std::unique_ptr<char[]> overwriteBuffer(new char[static_cast<size_t>(bufferSize)]);
        std::iota(overwriteBuffer.get(), overwriteBuffer.get() + bufferSize, '\x64');
        GetFilePath(&path, 0, 1);
        IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(
                &pFile,
                path,
                nn::fs::OpenMode_Write
            )
        );
        NN_UTIL_SCOPE_EXIT
        {
            pSaveDataFileSystem->CloseFile(pFile);
        };
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(
                fileSize / 8,
                overwriteBuffer.get(),
                static_cast<size_t>(fileSize / 8)
            )
        );
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(
                fileSize * 3 / 8,
                overwriteBuffer.get(),
                static_cast<size_t>(fileSize / 8)
            )
        );
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(
                fileSize * 6 / 8,
                overwriteBuffer.get(),
                static_cast<size_t>(fileSize / 8)
            )
        );
        NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());
    }

    // 再マウントします
    UnmountFileSystem();
    NNT_ASSERT_RESULT_SUCCESS(MountFileSystem());

    // ハッシュ検証エラーが発生することを確認します
    {
        GetFilePath(&path, 0, 1);
        IFile* pFile;
        auto result = pSaveDataFileSystem->OpenFile(
            &pFile,
            path,
            nn::fs::OpenMode_Read
        );
        if( result.IsFailure() )
        {
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultNonRealDataVerificationFailed,
                result
            );
        }
        else
        {
            NN_UTIL_SCOPE_EXIT
            {
                pSaveDataFileSystem->CloseFile(pFile);
            };

            result = pFile->ReadBytes(0, readBuffer.get(), static_cast<size_t>(fileSize));
            if( !nn::fs::ResultUnclearedRealDataVerificationFailed::Includes(result) )
            {
                NNT_ASSERT_RESULT_FAILURE(
                    nn::fs::ResultNonRealDataVerificationFailed,
                    result
                );
            }
        }
    }
} // NOLINT(impl/function_size)

/**
* 重複させながら書き込み、コミットし、読み込みます
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestReadWriteOverlap)
{
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    const TestParam& param = GetParam();
    int64_t bufferSize = param.GetEntrySize();
    int64_t fileSize = bufferSize - param.sizeBlock * 4;
    std::unique_ptr<char[]> writeBuffer(new char[static_cast<size_t>(bufferSize)]);
    std::iota(writeBuffer.get(), writeBuffer.get() + bufferSize, '\x1');
    std::unique_ptr<char[]> readBuffer(new char[static_cast<size_t>(bufferSize)]);
    nn::fssystem::save::Path path;

    // ファイルを作成
    {
        GetDirectoryPath(&path, 0);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));
        GetFilePath(&path, 0, 1);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateFile(path, fileSize));
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());
    }

    // 重複させながらデータを書き込みます
    {
        GetFilePath(&path, 0, 1);
        IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(
                &pFile,
                path,
                nn::fs::OpenMode_Write
            )
        );
        NN_UTIL_SCOPE_EXIT
        {
            pSaveDataFileSystem->CloseFile(pFile);
        };
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(0, writeBuffer.get(), static_cast<size_t>(fileSize / 2))
        );
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(fileSize / 4, writeBuffer.get(), static_cast<size_t>(fileSize / 2))
        );
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(fileSize / 2, writeBuffer.get(), static_cast<size_t>(fileSize / 2))
        );
        NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());
    }

    // コミット
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());

    //再マウント
    UnmountFileSystem();
    NNT_ASSERT_RESULT_SUCCESS(MountFileSystem());

    // 書き込んだデータが読み込めることを確かめます
    {
        GetFilePath(&path, 0, 1);
        IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(
                &pFile,
                path,
                nn::fs::OpenMode_Read
            )
        );
        NN_UTIL_SCOPE_EXIT
        {
            pSaveDataFileSystem->CloseFile(pFile);
        };
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->ReadBytes(0, readBuffer.get(), static_cast<size_t>(fileSize))
        );
    }
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(
        readBuffer.get(), writeBuffer.get(), static_cast<size_t>(fileSize / 4)
    );
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(
        readBuffer.get() + fileSize / 4, writeBuffer.get(), static_cast<size_t>(fileSize / 4)
    );
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(
        readBuffer.get() + fileSize / 2, writeBuffer.get(), static_cast<size_t>(fileSize / 2)
    );
}

/**
* ディレクトリの追加/削除を何度も行います
* ファイルもディレクトリも無い状態で行います
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestCreateDeleteDirectory)
{
    nn::fssystem::save::Path path;
    GetDirectoryPath(&path, 1);

    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    for( int i = 0; i < 100; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->DeleteDirectory(path));
    }
}

/**
* ファイルの追加/削除を何度も行います
* ファイルもディレクトリも無い状態で行います
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestCreateDeleteFile)
{
    nn::fssystem::save::Path path;
    GetFilePath(&path, 1);

    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    for( int i = 0; i < 100; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateFile(path, 100));
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->DeleteFile(path));
    }
}

/**
* ディレクトリの追加/削除を何度も行います
* いくつかファイルとディレクトリがある状態で行います
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestCreateDeleteDirectoryWithOthers)
{
    const TestParam& param = GetParam();
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    nn::fssystem::save::Path path;

    // ディレクトリとファイルをランダムな個数作成
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    auto directoryCount = std::uniform_int_distribution<>(1, 20)(mt);
    for( int i = 0; i < directoryCount; ++i )
    {
        GetDirectoryPath(&path, i);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));
    }
    auto fileCount = std::uniform_int_distribution<>(1, 20)(mt);
    for( int i = 0; i < fileCount; ++i )
    {
        GetFilePath(&path, 0, i);
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->CreateFile(
                path,
                param.sizeBlock
            )
        );
    }

    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());

    // dir0/dir0 の作成/削除を 100 回
    GetDirectoryPath(&path, 0, 0);
    for( int i = 0; i < 100; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->DeleteDirectory(path));
    }
}

/**
* ファイルの追加/削除を何度も行います
* いくつかファイルとディレクトリがある状態で行います
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestCreateDeleteFileWithOthers)
{
    const TestParam& param = GetParam();
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    nn::fssystem::save::Path path;

    // ディレクトリとファイルをランダムな個数作成
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    auto directoryCount = std::uniform_int_distribution<>(1, 20)(mt);
    for( int i = 0; i < directoryCount; ++i )
    {
        GetDirectoryPath(&path, i);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));
    }
    auto fileCount = std::uniform_int_distribution<>(1, 20)(mt);
    for( int i = 1; i <= fileCount; ++i )
    {
        GetFilePath(&path, 0, i);
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->CreateFile(
                path,
                param.sizeBlock
            )
        );
    }

    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());

    // dir0/file0 の作成/削除を 100 回
    // サイズはランダムです
    GetFilePath(&path, 0, 0);
    for( int i = 0; i < 100; ++i )
    {
        auto fileSize = std::uniform_int_distribution<>(1, param.sizeBlock * 2)(mt);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateFile(path, fileSize));
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->DeleteFile(path));
    }
}

/**
* ディレクトリを 20 個追加し、20 個削除する処理を 30 回繰り返します
* ディレクトリ名はランダムです
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestCreateDeleteDirectoryRandom)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nn::fssystem::save::Path path;
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();

    const int FileCount = 20;

    // ファイル名のリスト
    std::unique_ptr<nnt::fs::util::String[]> pDirectoryNameList(
        new nnt::fs::util::String[FileCount]
    );

    for( int i = 0; i < 30; ++i )
    {
        // ファイルを 20 個作成します
        for( int j = 0; j < FileCount; ++j )
        {
            pDirectoryNameList[j] = nnt::fs::util::String("/");
            pDirectoryNameList[j] += nnt::fs::util::GenerateRandomEntryName(32);
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize(pDirectoryNameList[j].c_str()));
            NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));
        }
        // 作成したファイル 20 個を削除します
        for( int j = 0; j < FileCount; ++j )
        {
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize(pDirectoryNameList[j].c_str()));
            NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->DeleteDirectory(path));
        }
    }
}

/**
* ファイルをいくつか追加し、全て削除する処理を 30 回繰り返します
* ファイル名とファイルサイズはランダムです
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestCreateDeleteFileRandom)
{
    static const size_t FileEntrySize = 96;
    static const uint32_t MaxFileCount = 32;

    // ファイル名のリスト
    static nnt::fs::util::String pFileNameList[MaxFileCount];
    static nn::fssystem::save::Path path;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    TestParam param = GetParam();

    // 作成可能なファイル数を計算
    auto creatableFileCount = MaxFileCount;
    for( ;; )
    {
        // creatableFileCount 個ファイルを作成する場合、
        // 作成するファイル数 * エントリサイズ / ブロックサイズ
        // (切り上げ) 個のブロックをファイルエントリに消費します
        // 親ディレクトリを更新することになるので、ファイル作成に必要なブロック数に 1 足します
        auto requiredBlockCount = nn::util::DivideUp(
            (creatableFileCount) * FileEntrySize, param.sizeBlock) + 1;
        if( requiredBlockCount + creatableFileCount > param.countDataBlock )
        {
            creatableFileCount--;
        }
        else
        {
            break;
        }
    }

    for( int i = 0; i < 30; ++i )
    {
        auto fileCount = std::uniform_int_distribution<>(1, creatableFileCount)(mt);

        // ファイルを作成
        for( auto j = 0; j < fileCount; ++j )
        {
            pFileNameList[j] = nnt::fs::util::String("/");
            pFileNameList[j] += nnt::fs::util::GenerateRandomEntryName(
                nn::fssystem::dbm::MaxFileNameLength);
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize(pFileNameList[j].c_str()));

            NNT_ASSERT_RESULT_SUCCESS(
                pSaveDataFileSystem->CreateFile(
                    path,
                    std::uniform_int_distribution<>(1, param.sizeBlock)(mt)
                )
            );
        }

        // 作成したファイルを全て削除
        for( auto j = 0; j < fileCount; ++j )
        {
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize(pFileNameList[j].c_str()));
            NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->DeleteFile(path));
        }
    }
}

/**
* OpenMode_Write で開いて読み込みを行い、エラーが発生することを確かめます
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestInvalidModeRead)
{
    const TestParam& param = GetParam();
    int64_t fileSize = param.GetEntrySize() - param.sizeBlock * 4;
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    nn::fssystem::save::Path path;

    // ディレクトリとファイル作成
    GetDirectoryPath(&path, 0);
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));
    GetFilePath(&path, 0, 0);
    NNT_ASSERT_RESULT_SUCCESS(
        pSaveDataFileSystem->CreateFile(
            path,
            fileSize
        )
    );
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());
    std::unique_ptr<char[]> buffer(new char[static_cast<size_t>(fileSize)]);
    std::iota(buffer.get(), buffer.get() + fileSize, '\x0');

    // OpenMode_Write で開いて読み込みます
    {
        auto pAccessCountStorage = GetAccessCountStorage();
        pAccessCountStorage->ResetAccessCounter();

        IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(&pFile, path, nn::fs::OpenMode_Write)
        );
        NN_UTIL_SCOPE_EXIT
        {
            pSaveDataFileSystem->CloseFile(pFile);
        };

        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultReadUnpermitted,
            pFile->ReadBytes(0, buffer.get(), static_cast<size_t>(fileSize))
        );
        ASSERT_EQ(0, pAccessCountStorage->GetWriteTimes());
    }
}

/**
* OpenMode_Read で開いて書き込みを行いエラーが発生することを確かめます
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestInvalidModeWrite)
{
    const TestParam& param = GetParam();
    int64_t fileSize = param.GetEntrySize() - param.sizeBlock * 4;
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    nn::fssystem::save::Path path;

    // ディレクトリとファイル作成
    GetDirectoryPath(&path, 0);
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));
    GetFilePath(&path, 0, 0);
    NNT_ASSERT_RESULT_SUCCESS(
        pSaveDataFileSystem->CreateFile(
            path,
            fileSize
        )
    );
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());
    std::unique_ptr<char[]> buffer(new char[static_cast<size_t>(fileSize)]);
    std::iota(buffer.get(), buffer.get() + fileSize, '\x0');

    // OpenMode_Read で開いて書き込みます
    {
        auto pAccessCountStorage = GetAccessCountStorage();
        pAccessCountStorage->ResetAccessCounter();

        IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(&pFile, path, nn::fs::OpenMode_Read)
        );
        NN_UTIL_SCOPE_EXIT
        {
            pSaveDataFileSystem->CloseFile(pFile);
        };

        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultWriteUnpermitted,
            pFile->WriteBytes(0, buffer.get(), static_cast<size_t>(fileSize))
        );
        ASSERT_EQ(0, pAccessCountStorage->GetWriteTimes());
    }
}

/**
* 2 つのファイルを同時に開き、交互に読み書きを行うことができることを確かめます
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestParallelWrite)
{
    const TestParam& param = GetParam();
    int64_t fileSize = (param.GetEntrySize() - param.sizeBlock * 4) / 2;
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    nn::fssystem::save::Path path1;
    nn::fssystem::save::Path path2;

    // ディレクトリとファイル作成
    GetDirectoryPath(&path1, 0);
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path1));
    GetFilePath(&path1, 0, 0);
    NNT_ASSERT_RESULT_SUCCESS(
        pSaveDataFileSystem->CreateFile(
            path1,
            fileSize
        )
    );
    GetFilePath(&path2, 0, 1);
    NNT_ASSERT_RESULT_SUCCESS(
        pSaveDataFileSystem->CreateFile(
            path2,
            fileSize
        )
    );
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());
    int64_t bufferSize = param.GetEntrySize() / 6;
    std::unique_ptr<char[]> writeBuffer(new char[static_cast<size_t>(bufferSize)]);
    std::iota(writeBuffer.get(), writeBuffer.get() + bufferSize, '\x0');

    // 2 つのファイルを同時に開いて交互に書き込みます
    {
        {
            IFile* pFile1;
            NNT_ASSERT_RESULT_SUCCESS(
                pSaveDataFileSystem->OpenFile(&pFile1, path1, nn::fs::OpenMode_Write)
            );
            NN_UTIL_SCOPE_EXIT
            {
                pSaveDataFileSystem->CloseFile(pFile1);
            };

            IFile* pFile2;
            NNT_ASSERT_RESULT_SUCCESS(
                pSaveDataFileSystem->OpenFile(&pFile2, path2, nn::fs::OpenMode_Write)
            );
            NN_UTIL_SCOPE_EXIT
            {
                pSaveDataFileSystem->CloseFile(pFile2);
            };

            NNT_ASSERT_RESULT_SUCCESS(
                pFile1->WriteBytes(0, writeBuffer.get(), static_cast<size_t>(bufferSize))
            );
            NNT_ASSERT_RESULT_SUCCESS(
                pFile2->WriteBytes(0, writeBuffer.get(), static_cast<size_t>(bufferSize))
            );
            NNT_ASSERT_RESULT_SUCCESS(
                pFile1->WriteBytes(bufferSize, writeBuffer.get(), static_cast<size_t>(bufferSize))
            );
            NNT_ASSERT_RESULT_SUCCESS(
                pFile2->WriteBytes(bufferSize, writeBuffer.get(), static_cast<size_t>(bufferSize))
            );
            NNT_ASSERT_RESULT_SUCCESS(pFile1->Flush());
            NNT_ASSERT_RESULT_SUCCESS(pFile2->Flush());
        }

        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());
    }

    // 再マウントします
    UnmountFileSystem();
    NNT_ASSERT_RESULT_SUCCESS(MountFileSystem());

    // 正しく書き込めていることを確かめます
    for( int i = 0; i < 2; ++i )
    {
        std::unique_ptr<char[]> readBuffer(new char[static_cast<size_t>(bufferSize * 2)]);
        IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(&pFile, path1, nn::fs::OpenMode_Read)
        );
        NN_UTIL_SCOPE_EXIT
        {
            pSaveDataFileSystem->CloseFile(pFile);
        };
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->ReadBytes(0, readBuffer.get(), static_cast<size_t>(bufferSize) * 2)
        );
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(
            writeBuffer.get(), readBuffer.get(), static_cast<size_t>(bufferSize)
        );
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(
            writeBuffer.get(), readBuffer.get() + bufferSize, static_cast<size_t>(bufferSize)
        );
    }
}

/**
* マルチスレッドで 2 つのファイルを開き、同時に読み書きを行います
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestMultiWrite)
{
    static const auto StackSize = 64 * 1024;
    static const auto ThreadCount = 2;
    NN_OS_ALIGNAS_THREAD_STACK static char ThreadStack[StackSize * ThreadCount] = {0};
    static nn::os::ThreadType threads[ThreadCount];
    static nn::fssystem::save::Path directoryPath;
    static nn::fssystem::save::Path pathList[ThreadCount];

    TestParam param(GetParam());
    const int64_t fileSize = param.GetEntrySize() / 16;
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();

    // マルチスレッド用の引数
    struct Argument
    {
        nn::fssystem::save::IntegritySaveDataFileSystem* pFileSystem;
        nn::fssystem::save::Path* pPath;
        int index;
    };
    Argument args[ThreadCount] = {};

    // ディレクトリを作成しておきます
    {
        GetDirectoryPath(&directoryPath, 0);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(directoryPath));

        for( int i = 0; i < ThreadCount; ++i )
        {
            GetFilePath(&(pathList[i]), 0, i);
            NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateFile(pathList[i], fileSize));
            args[i].pFileSystem = pSaveDataFileSystem;
            args[i].pPath = &pathList[i];
            args[i].index = i * 100;
        }
    }

    // 複数スレッドから書き込みを行います
    for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
    {
        const auto ThreadFunc = [](void* pArg) NN_NOEXCEPT
        {
            Argument* pArgument = reinterpret_cast<Argument*>(pArg);

            nn::fssystem::save::IFile* pFile;
            NNT_ASSERT_RESULT_SUCCESS(
                pArgument->pFileSystem->OpenFile(&pFile, *pArgument->pPath, nn::fs::OpenMode_Write)
            );
            NN_UTIL_SCOPE_EXIT
            {
                pArgument->pFileSystem->CloseFile(pFile);
            };

            int64_t writeSize = 0;
            NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&writeSize));

            // 1 バイトずつ書き込みます
            for( int i = 0; i < writeSize; ++i )
            {
                char data = static_cast<char>(i + pArgument->index);
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->WriteBytes(i, &data, 1)
                );
                NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());
            }
        };

        NNT_ASSERT_RESULT_SUCCESS(
            nn::os::CreateThread(
                &threads[threadIndex],
                ThreadFunc,
                &args[threadIndex],
                &ThreadStack[StackSize * threadIndex],
                StackSize,
                nn::os::DefaultThreadPriority
            )
        );
    }
    for( auto& thread : threads )
    {
        nn::os::StartThread(&thread);
    }
    for( auto& thread : threads )
    {
        nn::os::WaitThread(&thread);
        nn::os::DestroyThread(&thread);
    }

    NNT_ASSERT_RESULT_SUCCESS(
        pSaveDataFileSystem->CommitFileSystem()
    );

    // 再マウントします
    UnmountFileSystem();
    NNT_ASSERT_RESULT_SUCCESS(MountFileSystem());

    // 正しく書き込めているか確かめます
    {
        std::unique_ptr<char[]> bufferExpect(new char[static_cast<size_t>(fileSize)]);
        std::unique_ptr<char[]> bufferActual(new char[static_cast<size_t>(fileSize)]);
        nn::fssystem::save::IFile* pFile;
        for( int i = 0; i < ThreadCount; ++i )
        {
            std::iota(
                bufferExpect.get(),
                bufferExpect.get() + fileSize,
                static_cast<char>(args[i].index)
            );
            NNT_ASSERT_RESULT_SUCCESS(
                pSaveDataFileSystem->OpenFile(&pFile, *args[i].pPath, nn::fs::OpenMode_Read)
            );
            NN_UTIL_SCOPE_EXIT
            {
                pSaveDataFileSystem->CloseFile(pFile);
            };

            NNT_ASSERT_RESULT_SUCCESS(
                pFile->ReadBytes(0, bufferActual.get(), static_cast<size_t>(fileSize))
            );
            NNT_FS_UTIL_ASSERT_MEMCMPEQ(
                bufferExpect.get(), bufferActual.get(), static_cast<size_t>(fileSize)
            );
        }
    }
} // NOLINT(impl/function_size)

/**
* 1 つのファイルを複数のインスタンスから開きます
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestDuplicateOpen)
{
    const TestParam& param = GetParam();
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    nn::fssystem::save::Path path;

    // ディレクトリとファイル作成
    int64_t fileSize = param.GetEntrySize() / 2;

    GetDirectoryPath(&path, 0);
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));

    GetFilePath(&path, 0, 0);
    NNT_ASSERT_RESULT_SUCCESS(
        pSaveDataFileSystem->CreateFile(
            path,
            fileSize
        )
    );

    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());

    // OpenMode_Write で開いているとき、他インスタンスで同じファイルは開けません
    {
        {
            std::unique_ptr<char[]> writeBuffer(new char[static_cast<size_t>(fileSize)]);
            std::iota(writeBuffer.get(), writeBuffer.get() + fileSize, '\x0');

            GetFilePath(&path, 0, 0);

            IFile* pFile1 = nullptr;
            NNT_ASSERT_RESULT_SUCCESS(
                pSaveDataFileSystem->OpenFile(&pFile1, path, nn::fs::OpenMode_Write)
            );
            NN_UTIL_SCOPE_EXIT
            {
                pSaveDataFileSystem->CloseFile(pFile1);
            };

            NNT_ASSERT_RESULT_SUCCESS(
                pFile1->WriteBytes(0, writeBuffer.get(), static_cast<size_t>(fileSize))
            );
            NNT_ASSERT_RESULT_SUCCESS(pFile1->Flush());

            IFile* pFile2 = nullptr;

            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultPermissionDenied,
                pSaveDataFileSystem->OpenFile(&pFile2, path, nn::fs::OpenMode_Write)
            );
            ASSERT_EQ(nullptr, pFile2);

            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultPermissionDenied,
                pSaveDataFileSystem->OpenFile(&pFile2, path, nn::fs::OpenMode_Read)
            );
            ASSERT_EQ(nullptr, pFile2);
        }

        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());
    }

    // OpenMode_Read で開いているときは、別のインスタンスで OpenMode_Read で開くことは可能です
    {
        GetFilePath(&path, 0, 0);

        IFile* pFile1 = nullptr;
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(&pFile1, path, nn::fs::OpenMode_Read)
        );
        NN_UTIL_SCOPE_EXIT
        {
            pSaveDataFileSystem->CloseFile(pFile1);
        };

        {
            IFile* pFile2 = nullptr;
            NNT_ASSERT_RESULT_SUCCESS(
                pSaveDataFileSystem->OpenFile(&pFile2, path, nn::fs::OpenMode_Read)
            );
            NN_UTIL_SCOPE_EXIT
            {
                pSaveDataFileSystem->CloseFile(pFile2);
            };

            std::unique_ptr<char[]> readBuffer1(new char[static_cast<size_t>(fileSize)]);
            NNT_ASSERT_RESULT_SUCCESS(
                pFile1->ReadBytes(0, readBuffer1.get(), static_cast<size_t>(fileSize))
            );
            std::unique_ptr<char[]> readBuffer2(new char[static_cast<size_t>(fileSize)]);
            NNT_ASSERT_RESULT_SUCCESS(
                pFile2->ReadBytes(0, readBuffer2.get(), static_cast<size_t>(fileSize))
            );
            NNT_FS_UTIL_ASSERT_MEMCMPEQ(
                readBuffer1.get(), readBuffer2.get(), static_cast<size_t>(fileSize)
            );
        }

        {
            // OpenMode_Write で開くことはできない
            IFile* pFile2 = nullptr;
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultPermissionDenied,
                pSaveDataFileSystem->OpenFile(&pFile2, path, nn::fs::OpenMode_Write)
            );
        }
    }

    UnmountFileSystem();
}

/**
* ファイルを開いたままアンマウントするとエラーが発生することを確かめます
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestFileUnmountWithOpen)
{
    const TestParam& param = GetParam();
    int64_t fileSize = param.GetEntrySize() / 2;
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    nn::fssystem::save::Path path;

    // ディレクトリとファイル作成しコミット
    GetDirectoryPath(&path, 0);
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));

    GetFilePath(&path, 0, 0);
    NNT_ASSERT_RESULT_SUCCESS(
        pSaveDataFileSystem->CreateFile(
            path,
            fileSize
        )
    );
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());

    std::unique_ptr<char[]> writeBuffer(new char[static_cast<size_t>(fileSize)]);
    IFile* pFile1;
    bool isFile1Closed = false;

    // データを書き込みます
    {
        {
            std::iota(writeBuffer.get(), writeBuffer.get() + fileSize, '\x0');
            GetFilePath(&path, 0, 0);
            NNT_ASSERT_RESULT_SUCCESS(
                pSaveDataFileSystem->OpenFile(&pFile1, path, nn::fs::OpenMode_Write)
            );
            NN_UTIL_SCOPE_EXIT
            {
                pSaveDataFileSystem->CloseFile(pFile1);
            };

            NNT_ASSERT_RESULT_SUCCESS(
                pFile1->WriteBytes(0, writeBuffer.get(), static_cast<size_t>(fileSize))
            );
            NNT_ASSERT_RESULT_SUCCESS(pFile1->Flush());
        }

        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());
    }

    // データを書き込み、ファイルを開いたままにします
    {
        std::iota(writeBuffer.get(), writeBuffer.get() + fileSize, '\x1');
        GetFilePath(&path, 0, 0);
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(&pFile1, path, nn::fs::OpenMode_Write)
        );
    }

    NN_UTIL_SCOPE_EXIT
    {
        if( !isFile1Closed )
        {
            isFile1Closed = true;
            pSaveDataFileSystem->CloseFile(pFile1);
        }
    };

    {
        NNT_ASSERT_RESULT_SUCCESS(
            pFile1->WriteBytes(0, writeBuffer.get(), static_cast<size_t>(fileSize))
        );
        NNT_ASSERT_RESULT_SUCCESS(pFile1->Flush());
    }

    // 再マウント
    UnmountFileSystem();
    NNT_ASSERT_RESULT_SUCCESS(MountFileSystem());

    // ハッシュ検証エラーかパーミッションエラーが発生することを確認します
    {
        std::unique_ptr<char[]> readBuffer(new char[static_cast<size_t>(fileSize)]);

        GetFilePath(&path, 0, 0);
        IFile* pFile2;
        auto result = pSaveDataFileSystem->OpenFile(&pFile2, path, nn::fs::OpenMode_Read);
        if( !nn::fs::ResultNonRealDataVerificationFailed::Includes(result) )
        {
            NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPermissionDenied, result);

            isFile1Closed = true;
            pSaveDataFileSystem->CloseFile(pFile1);

            NNT_ASSERT_RESULT_SUCCESS(
                pSaveDataFileSystem->OpenFile(&pFile2, path, nn::fs::OpenMode_Read)
            );
            NN_UTIL_SCOPE_EXIT
            {
                pSaveDataFileSystem->CloseFile(pFile2);
            };

            result = pFile2->ReadBytes(0, readBuffer.get(), static_cast<size_t>(fileSize));
            if( !nn::fs::ResultUnclearedRealDataVerificationFailed::Includes(result) )
            {
                NNT_ASSERT_RESULT_FAILURE(
                    nn::fs::ResultNonRealDataVerificationFailed,
                    result
                );
            }
        }
    }
}

/**
* ファイルを開いたままコミットしてアンマウントします
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestMissFileCommitWithOpen)
{
    const TestParam& param = GetParam();
    int64_t fileSize = param.GetEntrySize() / 2;
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    nn::fssystem::save::Path path;

    // ディレクトリとファイル作成
    GetDirectoryPath(&path, 0);
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));

    GetFilePath(&path, 0, 0);
    NNT_ASSERT_RESULT_SUCCESS(
        pSaveDataFileSystem->CreateFile(
            path,
            fileSize
        )
    );

    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());
    std::unique_ptr<char[]> writeBuffer(new char[static_cast<size_t>(fileSize)]);
    IFile* pFile1;
    bool isFile1Closed = false;

    // データを書き込みます
    {
        {
            std::iota(writeBuffer.get(), writeBuffer.get() + fileSize, '\x0');
            GetFilePath(&path, 0, 0);
            NNT_ASSERT_RESULT_SUCCESS(
                pSaveDataFileSystem->OpenFile(&pFile1, path, nn::fs::OpenMode_Write)
            );
            NN_UTIL_SCOPE_EXIT
            {
                pSaveDataFileSystem->CloseFile(pFile1);
            };

            NNT_ASSERT_RESULT_SUCCESS(
                pFile1->WriteBytes(0, writeBuffer.get(), static_cast<size_t>(fileSize))
            );
            NNT_ASSERT_RESULT_SUCCESS(pFile1->Flush());
        }

        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());
    }

    // データを書き込み、クローズする前にコミットします
    {
        std::iota(writeBuffer.get(), writeBuffer.get() + fileSize, '\x1');
        GetFilePath(&path, 0, 0);
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(&pFile1, path, nn::fs::OpenMode_Write)
        );
    }

    NN_UTIL_SCOPE_EXIT
    {
        if( !isFile1Closed )
        {
            isFile1Closed = true;
            pSaveDataFileSystem->CloseFile(pFile1);
        }
    };

    {
        NNT_ASSERT_RESULT_SUCCESS(
            pFile1->WriteBytes(0, writeBuffer.get(), static_cast<size_t>(fileSize))
        );
        NNT_ASSERT_RESULT_SUCCESS(pFile1->Flush());
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultUnsupportedOperation,
            pSaveDataFileSystem->CommitFileSystem()
        );
    }

    // 再マウントします
    UnmountFileSystem();
    NNT_ASSERT_RESULT_SUCCESS(MountFileSystem());

    // ファイルが書き込めているか書き込めていないかどちらかであることを確かめます
    {
        std::unique_ptr<char[]> readBuffer(new char[static_cast<size_t>(fileSize)]);
        std::iota(writeBuffer.get(), writeBuffer.get() + fileSize, '\x0');

        // この時点ではエラー
        GetFilePath(&path, 0, 0);
        IFile* pFile2;
        auto result = pSaveDataFileSystem->OpenFile(&pFile2, path, nn::fs::OpenMode_Read);
        if( !nn::fs::ResultNonRealDataVerificationFailed::Includes(result) )
        {
            NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPermissionDenied, result);

            isFile1Closed = true;
            pSaveDataFileSystem->CloseFile(pFile1);

            NNT_ASSERT_RESULT_SUCCESS(
                pSaveDataFileSystem->OpenFile(&pFile2, path, nn::fs::OpenMode_Read)
            );
            NN_UTIL_SCOPE_EXIT
            {
                pSaveDataFileSystem->CloseFile(pFile2);
            };

            result = pFile2->ReadBytes(0, readBuffer.get(), static_cast<size_t>(fileSize));
            if( !nn::fs::ResultUnclearedRealDataVerificationFailed::Includes(result) )
            {
                NNT_ASSERT_RESULT_FAILURE(
                    nn::fs::ResultNonRealDataVerificationFailed,
                    result
                );
            }
        }
    }
}

/**
* OpenMode_AllowAppend を指定したときに、
* 書き込み時のオフセットがファイルサイズ以上の場合
* エラーが発生することを確かめます
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestWriteOffsetOutOfRange)
{
    const TestParam& param = GetParam();
    int64_t fileSize = param.GetEntrySize() / 2;
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    nn::fssystem::save::Path path;

    // ディレクトリとファイル作成
    GetDirectoryPath(&path, 0);
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));

    GetFilePath(&path, 0, 0);
    NNT_ASSERT_RESULT_SUCCESS(
        pSaveDataFileSystem->CreateFile(
            path,
            fileSize
        )
    );

    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());
    std::unique_ptr<char[]> writeBuffer(new char[static_cast<size_t>(fileSize)]);
    std::iota(writeBuffer.get(), writeBuffer.get() + fileSize, '\x0');

    // データを書き込みます
    {
        {
            GetFilePath(&path, 0, 0);

            IFile* pFile;
            NNT_ASSERT_RESULT_SUCCESS(
                pSaveDataFileSystem->OpenFile(&pFile, path, nn::fs::OpenMode_Write)
            );
            NN_UTIL_SCOPE_EXIT
            {
                pSaveDataFileSystem->CloseFile(pFile);
            };

            NNT_ASSERT_RESULT_SUCCESS(
                pFile->WriteBytes(0, writeBuffer.get(), static_cast<size_t>(fileSize))
            );
            NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());
        }

        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());
    }

    // 範囲外オフセットで書き込みます
    {
        GetFilePath(&path, 0, 0);

        IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(&pFile, path, nn::fs::OpenMode_Write)
        );
        NN_UTIL_SCOPE_EXIT
        {
            pSaveDataFileSystem->CloseFile(pFile);
        };

        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultFileExtensionWithoutOpenModeAllowAppend,
            pFile->WriteBytes(fileSize, writeBuffer.get(), static_cast<size_t>(fileSize))
        );
    }
}

/**
* ファイルサイズからはみ出して読み込む処理であっても、
* 範囲内の部分は読み込めていることを確かめます
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestReadSizeOutOfRange)
{
    const TestParam& param = GetParam();
    int64_t fileSize = param.GetEntrySize() / 2;
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    nn::fssystem::save::Path path;

    // ディレクトリとファイル作成
    GetDirectoryPath(&path, 0);
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));

    GetFilePath(&path, 0, 0);
    NNT_ASSERT_RESULT_SUCCESS(
        pSaveDataFileSystem->CreateFile(
            path,
            fileSize
        )
    );
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());

    // データを書き込みクローズ、コミットする
    {
        std::unique_ptr<char[]> writeBuffer(new char[static_cast<size_t>(fileSize)]);
        std::iota(writeBuffer.get(), writeBuffer.get() + fileSize, '\x0');
        GetFilePath(&path, 0, 0);

        {
            IFile* pFile;
            NNT_ASSERT_RESULT_SUCCESS(
                pSaveDataFileSystem->OpenFile(&pFile, path, nn::fs::OpenMode_Write)
            );
            NN_UTIL_SCOPE_EXIT
            {
                pSaveDataFileSystem->CloseFile(pFile);
            };

            NNT_ASSERT_RESULT_SUCCESS(
                pFile->WriteBytes(0, writeBuffer.get(), static_cast<size_t>(fileSize))
            );
            NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());
        }

        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());
    }

    // ファイルからはみ出して読み込みます
    {
        GetFilePath(&path, 0, 0);
        IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(&pFile, path, nn::fs::OpenMode_Read)
        );
        NN_UTIL_SCOPE_EXIT
        {
            pSaveDataFileSystem->CloseFile(pFile);
        };

        std::unique_ptr<char[]> readBuffer(new char[static_cast<size_t>(fileSize)]);
        std::unique_ptr<char[]> expectBuffer(new char[static_cast<size_t>(fileSize)]);

        {
            // [1, fileSize + 1) を読み込み
            NNT_ASSERT_RESULT_SUCCESS(
                pFile->ReadBytes(1, readBuffer.get(), static_cast<size_t>(fileSize))
            );

            // [1, fileSize) の部分は読み込み成功していることを確かめます
            std::iota(expectBuffer.get(), expectBuffer.get() + fileSize, '\x1');
            NNT_FS_UTIL_ASSERT_MEMCMPEQ(
                readBuffer.get(), expectBuffer.get(), static_cast<size_t>(fileSize) - 1
            );
        }
        {
            // [fileSize - 3, fileSize + 3) を読み込み
            NNT_ASSERT_RESULT_SUCCESS(
                pFile->ReadBytes(fileSize - 3, readBuffer.get(), 6)
            );

            // [fileSize - 3, fileSize) の部分は読み込み成功していることを確かめます
            std::iota(expectBuffer.get(), expectBuffer.get() + 3, static_cast<char>(fileSize - 3));
            NNT_FS_UTIL_ASSERT_MEMCMPEQ(readBuffer.get(), expectBuffer.get(), 3);
        }
    }
}

/**
* OpenMode_AllowAppend を指定せずに境界外まで書き込むとエラーが発生することを確かめます
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestWriteSizeOutOfRangeNotAllowAppend)
{
    const TestParam& param = GetParam();
    int64_t fileSize = param.GetEntrySize() / 2;
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    nn::fssystem::save::Path path;

    // ディレクトリとファイル作成
    GetDirectoryPath(&path, 0);
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));

    GetFilePath(&path, 0, 0);
    NNT_ASSERT_RESULT_SUCCESS(
        pSaveDataFileSystem->CreateFile(
            path,
            fileSize
        )
    );
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());

    // データを書き込みます
    {
        {
            std::unique_ptr<char[]> writeBuffer(new char[static_cast<size_t>(fileSize)]);
            std::iota(writeBuffer.get(), writeBuffer.get() + fileSize, '\x0');

            GetFilePath(&path, 0, 0);
            IFile* pFile;
            NNT_ASSERT_RESULT_SUCCESS(
                pSaveDataFileSystem->OpenFile(&pFile, path, nn::fs::OpenMode_Write)
            );
            NN_UTIL_SCOPE_EXIT
            {
                pSaveDataFileSystem->CloseFile(pFile);
            };
            NNT_ASSERT_RESULT_SUCCESS(
                pFile->WriteBytes(0, writeBuffer.get(), static_cast<size_t>(fileSize))
            );
            NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());
        }

        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());
    }

    // OpenMode_AllowAppend を指定せずにファイルからはみ出す書き込みをします
    {
        std::unique_ptr<char[]> writeBuffer(new char[static_cast<size_t>(fileSize)]);
        std::iota(writeBuffer.get(), writeBuffer.get() + fileSize, '\x5');
        std::unique_ptr<char[]> readBuffer(new char[static_cast<size_t>(fileSize)]);
        std::unique_ptr<char[]> expectBuffer(new char[static_cast<size_t>(fileSize)]);
        std::iota(expectBuffer.get(), expectBuffer.get() + fileSize, '\x0');

        GetFilePath(&path, 0, 0);

        IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(
                &pFile,
                path,
                nn::fs::OpenMode_Read | nn::fs::OpenMode_Write
            )
        );
        NN_UTIL_SCOPE_EXIT
        {
            pSaveDataFileSystem->CloseFile(pFile);
        };

        // [1, fileSize + 1) を書き込み
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultFileExtensionWithoutOpenModeAllowAppend,
            pFile->WriteBytes(1, writeBuffer.get(), static_cast<size_t>(fileSize))
        );
        NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());
        // ファイルサイズが拡張していないことを確認します
        int64_t currentFileSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->GetSize(&currentFileSize)
        );
        ASSERT_EQ(fileSize, currentFileSize);

        // 書き込み前と内容が同じことを確認します
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->ReadBytes(0, readBuffer.get(), static_cast<size_t>(fileSize) + 1)
        );
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(
            readBuffer.get(), expectBuffer.get(), static_cast<size_t>(fileSize)
        );

        // [fileSize + 3, fileSize + 10) を書き込みます
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset,
            pFile->WriteBytes(fileSize + 3, writeBuffer.get(), 7)
        );
        NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());

        // ファイルサイズが拡張していないことを確認します
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&currentFileSize));
        ASSERT_EQ(fileSize, currentFileSize);

        // 書き込み前と内容が同じことを確認します
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->ReadBytes(0, readBuffer.get(), static_cast<size_t>(fileSize) + 10)
        );
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(
            readBuffer.get(), expectBuffer.get(), static_cast<size_t>(fileSize)
        );
    }
}

/**
* OpenMode_AllowAppend を指定してファイルオープンした場合、
* ファイルからはみ出して書き込む処理を行うと、
* ファイルサイズを拡張させ書き込まれることを確かめます
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestWriteSizeOutOfRangeAllowAppend)
{
    const TestParam& param = GetParam();
    int64_t fileSize = param.GetEntrySize() / 2;
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    nn::fssystem::save::Path path;

    // ディレクトリとファイル作成
    GetDirectoryPath(&path, 0);
    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));

    GetFilePath(&path, 0, 0);
    NNT_ASSERT_RESULT_SUCCESS(
        pSaveDataFileSystem->CreateFile(
            path,
            fileSize
        )
    );

    NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());

    // データを書き込みます
    {
        {
            std::unique_ptr<char[]> writeBuffer(new char[static_cast<size_t>(fileSize)]);
            std::iota(writeBuffer.get(), writeBuffer.get() + fileSize, '\x0');
            GetFilePath(&path, 0, 0);

            IFile* pFile;
            NNT_ASSERT_RESULT_SUCCESS(
                pSaveDataFileSystem->OpenFile(&pFile, path, nn::fs::OpenMode_Write)
            );
            NN_UTIL_SCOPE_EXIT
            {
                pSaveDataFileSystem->CloseFile(pFile);
            };

            NNT_ASSERT_RESULT_SUCCESS(
                pFile->WriteBytes(0, writeBuffer.get(), static_cast<size_t>(fileSize))
            );
            NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());
        }

        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());
    }

    // OpenMode_AllowAppend を指定してファイルからはみ出して書き込みます
    {
        std::unique_ptr<char[]> writeBuffer(new char[static_cast<size_t>(fileSize)]);
        std::iota(writeBuffer.get(), writeBuffer.get() + fileSize, '\x5');
        std::unique_ptr<char[]> readBuffer(new char[static_cast<size_t>(fileSize + 10)]);
        std::unique_ptr<char[]> expectBuffer(new char[static_cast<size_t>(fileSize + 10)]);
        std::memset(expectBuffer.get(), 0, static_cast<size_t>(fileSize) + 10);
        std::iota(expectBuffer.get(), expectBuffer.get() + fileSize, '\x0');

        GetFilePath(&path, 0, 0);
        IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(
                &pFile,
                path,
                nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend
            )
        );
        NN_UTIL_SCOPE_EXIT
        {
            pSaveDataFileSystem->CloseFile(pFile);
        };

        // [1, fileSize + 1) を書き込みます
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(1, writeBuffer.get(), static_cast<size_t>(fileSize))
        );
        NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());
        std::iota(expectBuffer.get() + 1, expectBuffer.get() + fileSize + 1, '\x5');

        // ファイルサイズが拡張していることを確認します
        int64_t currentFileSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&currentFileSize));
        ASSERT_EQ(fileSize + 1, currentFileSize);

        // [1, fileSize + 1) の部分に書き込み成功していることを確認します
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->ReadBytes(0, readBuffer.get(), static_cast<size_t>(fileSize) + 1)
        );
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(
            readBuffer.get(), expectBuffer.get(), static_cast<size_t>(fileSize) + 1
        );

        // [fileSize + 3, fileSize + 10) を書き込みます
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(fileSize + 3, writeBuffer.get(), 7)
        );
        NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());
        std::iota(expectBuffer.get() + fileSize + 3, expectBuffer.get() + fileSize + 10, '\x5');

        // ファイルサイズが拡張していることを確認します
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&currentFileSize));
        ASSERT_EQ(fileSize + 10, currentFileSize);

        // [fileSize + 3, fileSize + 10) の部分に書き込み成功していることを確認します
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->ReadBytes(0, readBuffer.get(), static_cast<size_t>(fileSize) + 10)
        );
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(
            readBuffer.get(), expectBuffer.get(), static_cast<size_t>(fileSize) + 10
        );
    }
}

/**
* ファイルを開いたり閉じたりしながら何度も書き込むテスト
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestOverlapWrite)
{
    int directoryCount = 5;
    int fileCount = 5;
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    nn::fssystem::save::Path path;

    auto GetFileSize = [](int directoryId, int fileId) NN_NOEXCEPT
    {
        return static_cast<int64_t>((directoryId + 1) * (fileId + 2) * 16);
    };

    // ディレクトリとファイル作成
    for( int directoryId = 0; directoryId  < directoryCount; ++directoryId )
    {
        GetDirectoryPath(&path, directoryId);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));

        for( int fileId = 0; fileId < fileCount; ++fileId )
        {
            int64_t freeBytes = 0;
            NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->GetFreeBytes(&freeBytes));

            GetFilePath(&path, directoryId, fileId);
            NNT_ASSERT_RESULT_SUCCESS(
                pSaveDataFileSystem->CreateFile(
                    path,
                    GetFileSize(directoryId, fileId)
                )
            );
            NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());
        }
    }
    for( int loop = 0; loop < 1024; ++loop )
    {
        // 適当に一つのファイルを選びます
        int directoryId = (fileCount * 3 / 2) % directoryCount;
        int fileId = (directoryCount >> 2 ^ directoryCount << 2) % fileCount;

        {
            IFile* pFile;
            bool hasFile = false;
            GetFilePath(&path, directoryId, fileId);
            NNT_ASSERT_RESULT_SUCCESS(
                pSaveDataFileSystem->HasFile(
                    &hasFile,
                    path
                )
            );
            ASSERT_TRUE(hasFile);
            NNT_ASSERT_RESULT_SUCCESS(
                pSaveDataFileSystem->OpenFile(
                    &pFile,
                    path,
                    nn::fs::OpenMode_Write
                )
            );
            NN_UTIL_SCOPE_EXIT
            {
                pSaveDataFileSystem->CloseFile(pFile);
            };

            int64_t fileSize = GetFileSize(directoryId, fileId);
            std::unique_ptr<char[]> buffer(new char[static_cast<size_t>(fileSize)]);
            std::memset(buffer.get(), directoryId * fileId, static_cast<size_t>(fileSize));
            int64_t fileSizeActual = 0;
            NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&fileSizeActual));
            ASSERT_EQ(fileSize, fileSizeActual);
            NNT_ASSERT_RESULT_SUCCESS(
                pFile->WriteBytes(0, buffer.get(), static_cast<size_t>(fileSizeActual))
            );
        }

        // 10 回に 1 回、コミットします
        if( loop % 10 == 3 )
        {
            NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CommitFileSystem());
        }

        // 250 回に 1 回、ディレクトリ数を増やします
        if( loop % 250 == 95 )
        {
            ++directoryCount;
            GetDirectoryPath(&path, directoryCount - 1);
            NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));
            for( int id = 0; id < fileCount; ++id )
            {
                GetFilePath(&path, directoryCount - 1, id);
                NNT_ASSERT_RESULT_SUCCESS(
                    pSaveDataFileSystem->CreateFile(path, GetFileSize(directoryCount - 1, id))
                );
            }
        }

        // 160 回に 1 回、ファイル数を増やします
        if( loop % 160 == 110 )
        {
            ++fileCount;
            for( int id = 0; id < directoryCount; ++id )
            {
                GetFilePath(&path, id, fileCount - 1);
                NNT_ASSERT_RESULT_SUCCESS(
                    pSaveDataFileSystem->CreateFile(path, GetFileSize(id, fileCount - 1))
                );
            }
        }
    }
}

/**
* 作成したファイルがゼロフィルされていることを確認するテスト
*/
TEST_P(FsIntegritySaveDataFileSystemTest, TestFileIsFilledWithZero)
{
    IntegritySaveDataFileSystem* pSaveDataFileSystem = GetSaveDataFileSystem();
    const int fileSize = 1024;
    std::unique_ptr<char[]> readBuffer(new char[static_cast<size_t>(fileSize)]);
    Path path;

    // ファイルを作成しコミット
    {
        GetDirectoryPath(&path, 1);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateDirectory(path));

        GetFilePath(&path, 1, 1);
        NNT_ASSERT_RESULT_SUCCESS(pSaveDataFileSystem->CreateFile(path, fileSize));
        IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(
            pSaveDataFileSystem->OpenFile(
                &pFile,
                path,
                nn::fs::OpenMode_Read
            )
        );
        NN_UTIL_SCOPE_EXIT
        {
            pSaveDataFileSystem->CloseFile(pFile);
        };

        NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, readBuffer.get(), fileSize));
        for( int i = 0; i < fileSize; ++i )
        {
            EXPECT_EQ(0, readBuffer.get()[i]);
        }
    }
}

INSTANTIATE_TEST_CASE_P(
    TestIntegritySaveDataFileSystem,
    FsIntegritySaveDataFileSystemTest,
    ::testing::ValuesIn(MakeTestParam())
);

INSTANTIATE_TEST_CASE_P(
    TestIntegritySaveDataFileSystem,
    FsIntegritySaveDataFileSystemLargeTest,
    ::testing::ValuesIn(MakeLargeTestParam())
);

}}}
