﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <nn/fssystem/save/fs_HierarchicalIntegrityVerificationStorage.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>
#include "testFs_util_CommonStorageTests.h"
#include "testFs_util_CommonFileSystemTests.h"

// メタデータのテスト
// (こちらは実際にアプリケーションに組み込む場合の参考にしてください)
TEST(FsHierarchicalIntegrityVerificationStorageTest, MetaDataHeavy)
{
    static const size_t TestSize = 20 * 1024 * 1024;
    static const size_t TestLoop = 10;
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 4 * 1024;
    static const size_t CacheBufferMaxOrder = 12;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::Vector<char> readBuf(TestSize);
    nnt::fs::util::Vector<char> writeBuf(TestSize);
    nnt::fs::util::Vector<char> writeBufMaster(TestSize);
    char* pReadBuf = &readBuf[0];
    char* pWriteBuf = &writeBuf[0];
    char* pWriteBufMaster = &writeBufMaster[0];

    for( int64_t sizeRealData = 0; sizeRealData <= TestSize; )
    {
        NN_SDK_LOG(".");

        // 各階層のブロックサイズは可変です(256 ～ MaxBlockSize)
        // ただし、以下のパラメーターを考慮する必要が有ります。
        //
        //  ・キャッシュを有効に使うためには、1ブロックの大きさをキャッシュのページサイズと同等か、
        //    それより大きなサイズを設定すべきです。
        //    キャッシュのページサイズは "BlockCacheBufferManager" テンプレートの
        //    コンパイル時に決定します。

        const size_t sizeBlockLevel1 = GenerateRandomBlockSize(256, MaxBlockSize);
        const size_t sizeBlockLevel2 = GenerateRandomBlockSize(256, MaxBlockSize);
        const size_t sizeBlockLevel3 = GenerateRandomBlockSize(256, MaxBlockSize);
        const size_t sizeBlockLevel4 = GenerateRandomBlockSize(256, MaxBlockSize);
        nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
            inputParamIntegrity;
        inputParamIntegrity.sizeBlockLevel[0] = sizeBlockLevel1;
        inputParamIntegrity.sizeBlockLevel[1] = sizeBlockLevel2;
        inputParamIntegrity.sizeBlockLevel[2] = sizeBlockLevel3;
        inputParamIntegrity.sizeBlockLevel[3] = sizeBlockLevel4;

        // キャッシュを初期化します。
        //
        // キャッシュマネージャーはBuddyHeapを用いており、
        // 特定のバイト数 (2^x) がメモリ管理単位になっています。
        // 1ページ4096バイトの場合は、4096*2^(5-1) = 64KB を"一単位"とし、その倍数を与えます。
        // "一単位"は、BlockCacheBufferManager::QueryInitializationBlockSize() より求められます。
        // 但し、上記は推奨であり無駄な領域が多少発生する可能性はありますが任意のサイズを渡すことができます。
        // この場合でもページサイズの倍数であれば適切に分割されますので無駄な領域は発生しません。

        nn::os::Mutex locker(true);
        nn::fssystem::save::FilesystemBufferManagerSet cacheBufferSet;

        const size_t sizeUnitBlock =
            QueryUnitBlockSize(CacheBufferMaxOrder, CacheBlockSize);
        const size_t sizeCacheBuffer =
            sizeUnitBlock * nn::fssystem::save::IntegrityLayerCountSave;

        nnt::fs::util::Vector<char> bufCache(sizeCacheBuffer);
        nn::fssystem::FileSystemBufferManager cacheBuffer;
        // キャッシュバッファを確保、初期化
        NNT_ASSERT_RESULT_SUCCESS(
            cacheBuffer.Initialize(
                CacheCountMax,
                reinterpret_cast<uintptr_t>(&bufCache[0]),
                bufCache.size(),
                CacheBlockSize
            )
        );

        // 完全性検証ハッシュに割り当てるキャッシュバッファ
        for( size_t i = 0; i < nn::fssystem::save::IntegrityLayerCountSave; ++i )
        {
            cacheBufferSet.pBuffer[i] = &cacheBuffer;
        }

        // メタデータに必要なサイズを取得します。
        // メタデータは各階層のパラメーター、オプション情報などを格納する領域です。
        nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea controlArea;
        nn::fssystem::save::HierarchicalIntegrityVerificationSizeSet sizes = {};

        // それぞれのサイズを求めます
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::QuerySize(
                &sizes,
                inputParamIntegrity,
                nn::fssystem::save::IntegrityLayerCountSave,
                sizeRealData
            )
        );

        const int64_t bodySize = sizeRealData +
                                 sizes.layeredHashSizes[0] +
                                 sizes.layeredHashSizes[1] +
                                 sizes.layeredHashSizes[2];

        // 管理領域 + オプション領域の格納先ファイルを用意します
        nnt::fs::util::SafeMemoryStorage baseStorageCtrlAndOptionInfo;
        baseStorageCtrlAndOptionInfo.Initialize(sizes.controlSize);

        // ルートハッシュを格納するメモリファイルを用意します
        nnt::fs::util::SafeMemoryStorage baseStorageMasterHash;
        baseStorageMasterHash.Initialize(sizes.masterHashSize);

        // 実データ+L1,L2,L3ハッシュを格納するメモリファイルを用意します
        nnt::fs::util::SafeMemoryStorage baseStorageDataBody;
        baseStorageDataBody.Initialize(bodySize);

        // メタデータ領域をフォーマットします。
        // こちらはあくまでメタデータ領域のフォーマットであり
        // 実データ領域をフォーマットするわけでは有りません。
        nn::fs::SubStorage
            storageControlArea(&baseStorageCtrlAndOptionInfo, 0, sizes.controlSize);
        nn::fssystem::save::HierarchicalIntegrityVerificationMetaInformation meta = {};
        {
            int64_t offset = 0;
            meta.Format();
            meta.sizeMasterHash = static_cast<uint32_t>(sizes.masterHashSize);
            meta.infoLevelHash.info[3].size = bodySize;
            meta.infoLevelHash.maxLayers = nn::fssystem::save::IntegrityLayerCountSave;
            for( auto level = 0; level < 3; ++level )
            {
                const int64_t size = sizes.layeredHashSizes[level];
                meta.infoLevelHash.info[level].offset = offset;
                meta.infoLevelHash.info[level].size = size;
                meta.infoLevelHash.info[level].orderBlock =
                    nn::fssystem::save::ILog2(
                        static_cast<uint32_t>(inputParamIntegrity.sizeBlockLevel[level]));
                meta.infoLevelHash.info[3].size
                    = meta.infoLevelHash.info[3].size - size;
                offset += size;
            }
            meta.infoLevelHash.info[3].offset = offset;
            meta.infoLevelHash.info[3].orderBlock =
                nn::fssystem::save::ILog2(
                    static_cast<uint32_t>(inputParamIntegrity.sizeBlockLevel[3]));
        }

        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::Format(
                storageControlArea,
                meta
            )
        );

        // メタデータをマウントします。
        NNT_ASSERT_RESULT_SUCCESS(controlArea.Initialize(storageControlArea));

        // 管理領域からそれぞれのレベルのオフセット情報を取得します
        nn::fssystem::save::HierarchicalIntegrityVerificationInformation infoLevelHash;
        controlArea.GetLevelHashInfo(&infoLevelHash);

        // フォーマットで指定した値のとおりになっているか確認します
        ASSERT_EQ(sizes.masterHashSize, controlArea.GetMasterHashSize());
        ASSERT_EQ(
            nn::fssystem::save::ILog2(static_cast<uint32_t>(sizeBlockLevel1)),
            static_cast<uint32_t>(infoLevelHash.info[0].orderBlock)
        );
        ASSERT_EQ(
            nn::fssystem::save::ILog2(static_cast<uint32_t>(sizeBlockLevel2)),
            static_cast<uint32_t>(infoLevelHash.info[1].orderBlock)
        );
        ASSERT_EQ(
            nn::fssystem::save::ILog2(static_cast<uint32_t>(sizeBlockLevel3)),
            static_cast<uint32_t>(infoLevelHash.info[2].orderBlock)
        );
        ASSERT_EQ(
            nn::fssystem::save::ILog2(static_cast<uint32_t>(sizeBlockLevel4)),
            static_cast<uint32_t>(infoLevelHash.info[3].orderBlock)
        );

        // バッファを0クリアします
        std::memset(pWriteBufMaster, 0, TestSize);

        // マスターハッシュをつぶします
        NNT_ASSERT_RESULT_SUCCESS(
            baseStorageMasterHash.Write(
                0,
                pWriteBufMaster,
                static_cast<size_t>(sizes.masterHashSize)
            )
        );

        for( int32_t loop1 = 0; loop1 < TestLoop; ++loop1 )
        {
            // メタデータを元に完全性検証を初期化します。
            nn::fssystem::save::HierarchicalIntegrityVerificationStorage storage;
            nn::fs::SubStorage storageMasterHash(
                &baseStorageMasterHash, 0, controlArea.GetMasterHashSize());
            nn::fs::SubStorage storageLayeredHash(
                &baseStorageDataBody, 0, infoLevelHash.GetLayeredHashSize());
            nn::fs::SubStorage storageLayeredHashL1(
                &storageLayeredHash,
                infoLevelHash.info[0].offset,
                infoLevelHash.info[0].size
            );
            nn::fs::SubStorage storageLayeredHashL2(
                &storageLayeredHash,
                infoLevelHash.info[1].offset,
                infoLevelHash.info[1].size
            );
            nn::fs::SubStorage storageLayeredHashL3(
                &storageLayeredHash,
                infoLevelHash.info[2].offset,
                infoLevelHash.info[2].size
            );
            nn::fs::SubStorage storageData(
                &baseStorageDataBody,
                infoLevelHash.GetDataOffset(),
                infoLevelHash.GetDataSize()
            );
            nn::fssystem::save::HierarchicalIntegrityVerificationStorage::HierarchicalStorageInformation storageInfo;
            storageInfo.SetMasterHashStorage(storageMasterHash);
            storageInfo.SetLayer1HashStorage(storageLayeredHashL1);
            storageInfo.SetLayer2HashStorage(storageLayeredHashL2);
            storageInfo.SetLayer3HashStorage(storageLayeredHashL3);
            storageInfo.SetDataStorage(storageData);
            NNT_ASSERT_RESULT_SUCCESS(
                storage.Initialize(
                    infoLevelHash,
                    storageInfo,
                    &cacheBufferSet,
                    &locker,
                    nn::fs::StorageType::StorageType_SaveData
                )
            );

            // ランダムなオフセット、サイズで読み書きを行います。
            for( int32_t loop2 = 0; loop2 < TestLoop; ++loop2 )
            {
                int64_t offset =
                    std::uniform_int_distribution<int64_t>(0, sizeRealData)(mt);
                size_t size =
                    std::uniform_int_distribution<size_t>(
                        0, static_cast<size_t>(sizeRealData - offset) / 16)(mt);
                if( sizeRealData == 0 )
                {
                    offset = 0;
                    size = 0;
                }
                if( sizeRealData < offset + static_cast<int64_t>(size) )
                {
                    size = static_cast<size_t>(sizeRealData - offset);
                }

                std::memset(pWriteBuf, loop2 & 0xFF, size);

                // データを書き込みます。
                NNT_ASSERT_RESULT_SUCCESS(storage.Write(offset, pWriteBuf, size));
                std::memcpy(pWriteBufMaster + offset, pWriteBuf, size);

                if( std::uniform_int_distribution<>(0, 7)(mt) == 0 || loop2 == TestLoop - 2 )
                {
                    // コミット処理のテスト
                    NNT_ASSERT_RESULT_SUCCESS(storage.Commit());

                    // コミット後、同じ空間を別のStorageでマウントし
                    // 同一の内容が読み込みできるか確認します
                    nn::fssystem::save::FilesystemBufferManagerSet cacheBufferSet2;
                    nn::os::Mutex locker2(true);

                    // キャッシュバッファを確保、初期化
                    nnt::fs::util::Vector<char> bufCache2(sizeCacheBuffer);
                    nn::fssystem::FileSystemBufferManager cacheBuffer2;
                    NNT_ASSERT_RESULT_SUCCESS(
                        cacheBuffer2.Initialize(
                            CacheCountMax,
                            reinterpret_cast<uintptr_t>(&bufCache2[0]),
                            bufCache2.size(),
                            CacheBlockSize
                        )
                    );

                    // 完全性検証ハッシュに割り当てるキャッシュバッファ
                    for( size_t i = 0; i < nn::fssystem::save::IntegrityLayerCountSave; i++ )
                    {
                        cacheBufferSet2.pBuffer[i] = &cacheBuffer2;
                    }

                    // マウントします
                    {
                        nn::fssystem::save::HierarchicalIntegrityVerificationStorage storage2;
                        NNT_ASSERT_RESULT_SUCCESS(
                            storage2.Initialize(
                                infoLevelHash,
                                storageInfo,
                                &cacheBufferSet2,
                                &locker2,
                                nn::fs::StorageType::StorageType_SaveData
                            )
                        );

                        NNT_ASSERT_RESULT_SUCCESS(storage2.Read(offset, pReadBuf, size));

                        // 内容を比較します
                        ASSERT_EQ(0, std::memcmp(pReadBuf, pWriteBuf, size));
                    }
                }
                else if( std::uniform_int_distribution<>(0, 7)(mt) || (loop2 == TestLoop - 1) )
                {
                    // データを読み込んでベリファイします。
                    NNT_ASSERT_RESULT_SUCCESS(storage.Read(offset, pReadBuf, size));

                    ASSERT_EQ(0, std::memcmp(pReadBuf, pWriteBuf, size));
                }
            }
        }
        controlArea.Finalize();

        sizeRealData = 1 +
                       sizeRealData * 2 +
                       std::uniform_int_distribution<int64_t>(0, sizeRealData)(mt);
    }
    NN_SDK_LOG("\n");
} // NOLINT(impl/function_size)

// フラッシュのテスト
void FlushTest(nn::fs::StorageType storageType)
{
    static const size_t TestSize = 1024 * 1024;
    static const int TestLoop = (storageType == nn::fs::StorageType::StorageType_SaveData) ? 10 : 1;
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 4096;
    static const size_t CacheBufferMaxOrder = 12;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::Vector<char> buf(TestSize);
    char* ptr = &buf[0];
    nn::fssystem::save::HierarchicalIntegrityVerificationSizeSet sizes = {};

    for( int loop = 0; loop < TestLoop; ++loop )
    {
        int64_t sizeStorage =
            (loop == 0) ? 0 : std::uniform_int_distribution<int64_t>(0, TestSize - 1)(mt);
        int64_t offsetBase = std::uniform_int_distribution<>(0, 127)(mt);

        // 各階層のブロックサイズは可変です(256 ～ MaxBlockSize)
        // ただし、以下のパラメーターを考慮する必要が有ります。
        //
        //  ・キャッシュを有効に使うためには、1ブロックの大きさをキャッシュのページサイズと同等か、
        //    それより大きなサイズを設定すべきです。
        //    キャッシュのページサイズは "BlockCacheBufferManager" テンプレートの
        //    コンパイル時に決定します。

        size_t sizeBlockLevel1 = GenerateRandomBlockSize(256, MaxBlockSize);
        size_t sizeBlockLevel2 = GenerateRandomBlockSize(256, MaxBlockSize);
        size_t sizeBlockLevel3 = GenerateRandomBlockSize(256, MaxBlockSize);
        size_t sizeBlockLevel4 = GenerateRandomBlockSize(256, MaxBlockSize);
        nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
            inputParamIntegrity;
        inputParamIntegrity.sizeBlockLevel[0] = sizeBlockLevel1;
        inputParamIntegrity.sizeBlockLevel[1] = sizeBlockLevel2;
        inputParamIntegrity.sizeBlockLevel[2] = sizeBlockLevel3;
        inputParamIntegrity.sizeBlockLevel[3] = sizeBlockLevel4;

        // 検証ブロックサイズの最大値を求める
        const size_t sizeBlockLevelMax =
            std::max(inputParamIntegrity.sizeBlockLevel[0],
                std::max(inputParamIntegrity.sizeBlockLevel[1],
                    inputParamIntegrity.sizeBlockLevel[2]));

        // ファイルサイズを最大検証ブロック単位に合わせる
        sizeStorage = (sizeStorage + (sizeBlockLevelMax - 1)) & ~(sizeBlockLevelMax - 1);

        // メタデータに必要なサイズを取得します。
        // メタデータは各階層のパラメーター、オプション情報などを格納する領域です。
        nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea controlArea;

        // キャッシュを初期化します。
        //
        // キャッシュマネージャーはBuddyHeapを用いており、
        // 特定のバイト数 (2^x) がメモリ管理単位になっています。
        // 1ページ4096バイトの場合は、4096*2^(5-1) = 64KB を"一単位"とし、その倍数を与えます。
        // "一単位"は、BlockCacheBufferManager::QueryInitializationBlockSize() より求められます。
        // 但し、上記は推奨であり無駄な領域が多少発生する可能性はありますが任意のサイズを渡すことができます。
        // この場合でもページサイズの倍数であれば適切に分割されますので無駄な領域は発生しません。

        nn::os::Mutex locker(true);
        nn::fssystem::save::FilesystemBufferManagerSet cacheBufferSet;

        const size_t sizeUnitBlock =
            QueryUnitBlockSize(CacheBufferMaxOrder, CacheBlockSize);
        const size_t sizeCacheBuffer =
            sizeUnitBlock * nn::fssystem::save::IntegrityLayerCountSave;

        // キャッシュバッファを確保、初期化
        nnt::fs::util::Vector<char> bufCache(sizeCacheBuffer);
        nn::fssystem::FileSystemBufferManager cacheBuffer;
        NNT_ASSERT_RESULT_SUCCESS(
            cacheBuffer.Initialize(
                CacheCountMax,
                reinterpret_cast<uintptr_t>(&bufCache[0]),
                bufCache.size(),
                CacheBlockSize
            )
        );

        // 完全性検証ハッシュに割り当てるキャッシュバッファ
        for( size_t i = 0; i < nn::fssystem::save::IntegrityLayerCountSave; ++i )
        {
            cacheBufferSet.pBuffer[i] = &cacheBuffer;
        }

        // それぞれのサイズを求めます
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::QuerySize(
                &sizes,
                inputParamIntegrity,
                nn::fssystem::save::IntegrityLayerCountSave,
                sizeStorage
            )
        );

        const int64_t bodySize = sizeStorage +
                                 sizes.layeredHashSizes[0] +
                                 sizes.layeredHashSizes[1] +
                                 sizes.layeredHashSizes[2];

        {
            nnt::fs::util::AccessCountedMemoryStorage baseStorageCtrlAndOptionInfo;
            nnt::fs::util::AccessCountedMemoryStorage baseStorageMasterHash;
            nnt::fs::util::AccessCountedMemoryStorage baseStorageDataBody;
            nn::fssystem::save::HierarchicalIntegrityVerificationStorage storageIntegrity;

            // 管理領域 + オプション領域の格納先ファイルを用意します
            baseStorageCtrlAndOptionInfo.Initialize(sizes.controlSize + offsetBase * 2);

            // ルートハッシュを格納するメモリファイルを用意します
            baseStorageMasterHash.Initialize(sizes.masterHashSize + offsetBase * 2);

            // 実データ+L1,L2,L3ハッシュを格納するメモリファイルを用意します
            baseStorageDataBody.Initialize(bodySize + offsetBase * 2);

            // メタデータ領域をフォーマットします。
            // こちらはあくまでメタデータ領域のフォーマットであり
            // 実データ領域をフォーマットするわけでは有りません。
            nn::fs::SubStorage storageControlArea(
                &baseStorageCtrlAndOptionInfo,
                offsetBase,
                sizes.controlSize
            );
            nn::fssystem::save::HierarchicalIntegrityVerificationMetaInformation meta = {};
            {
                int64_t offsetHashInfo = 0;
                meta.Format();
                meta.sizeMasterHash = static_cast<uint32_t>(sizes.masterHashSize);
                meta.infoLevelHash.info[3].size = bodySize;
                meta.infoLevelHash.maxLayers = nn::fssystem::save::IntegrityLayerCountSave;
                for( auto level = 0; level < 3; ++level )
                {
                    const int64_t size = sizes.layeredHashSizes[level];
                    meta.infoLevelHash.info[level].offset = offsetHashInfo;
                    meta.infoLevelHash.info[level].size = size;
                    meta.infoLevelHash.info[level].orderBlock =
                        nn::fssystem::save::ILog2(
                            static_cast<uint32_t>(inputParamIntegrity.sizeBlockLevel[level]));
                    meta.infoLevelHash.info[3].size
                        = meta.infoLevelHash.info[3].size - size;
                    offsetHashInfo += size;
                }
                meta.infoLevelHash.info[3].offset = offsetHashInfo;
                meta.infoLevelHash.info[3].orderBlock =
                    nn::fssystem::save::ILog2(
                        static_cast<uint32_t>(inputParamIntegrity.sizeBlockLevel[3]));
            }
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::Format(
                    storageControlArea,
                    meta
                )
            );

            // ルートハッシュを0で埋めます
            baseStorageMasterHash.OperateRange(nn::fs::OperationId::FillZero, offsetBase, sizes.masterHashSize);

            // メタデータをマウントします。
            NNT_ASSERT_RESULT_SUCCESS(controlArea.Initialize(storageControlArea));

            // 管理領域からそれぞれのレベルのオフセット情報を取得します
            nn::fssystem::save::HierarchicalIntegrityVerificationInformation infoLevelHash;
            controlArea.GetLevelHashInfo(&infoLevelHash);

            // フォーマットで指定した値のとおりになっているか確認します
            ASSERT_EQ(sizes.masterHashSize, controlArea.GetMasterHashSize());
            ASSERT_EQ(
                nn::fssystem::save::ILog2(static_cast<uint32_t>(sizeBlockLevel1)),
                static_cast<uint32_t>(infoLevelHash.info[0].orderBlock)
            );
            ASSERT_EQ(
                nn::fssystem::save::ILog2(static_cast<uint32_t>(sizeBlockLevel2)),
                static_cast<uint32_t>(infoLevelHash.info[1].orderBlock)
            );
            ASSERT_EQ(
                nn::fssystem::save::ILog2(static_cast<uint32_t>(sizeBlockLevel3)),
                static_cast<uint32_t>(infoLevelHash.info[2].orderBlock)
            );

            // メタデータを元に完全性検証を初期化します。
            nn::fs::SubStorage storageMasterHash(
                &baseStorageMasterHash, offsetBase, controlArea.GetMasterHashSize());
            nn::fs::SubStorage storageLayeredHash(
                &baseStorageDataBody, offsetBase, infoLevelHash.GetLayeredHashSize());
            nn::fs::SubStorage storageLayeredHashL1(
                &storageLayeredHash,
                infoLevelHash.info[0].offset,
                infoLevelHash.info[0].size
            );
            nn::fs::SubStorage storageLayeredHashL2(
                &storageLayeredHash,
                infoLevelHash.info[1].offset,
                infoLevelHash.info[1].size
            );
            nn::fs::SubStorage storageLayeredHashL3(
                &storageLayeredHash,
                infoLevelHash.info[2].offset,
                infoLevelHash.info[2].size
            );
            nn::fs::SubStorage storageData(
                &baseStorageDataBody,
                offsetBase + infoLevelHash.GetDataOffset(),
                infoLevelHash.GetDataSize()
            );
            nn::fssystem::save::HierarchicalIntegrityVerificationStorage::HierarchicalStorageInformation storageInfo;
            storageInfo.SetMasterHashStorage(storageMasterHash);
            storageInfo.SetLayer1HashStorage(storageLayeredHashL1);
            storageInfo.SetLayer2HashStorage(storageLayeredHashL2);
            storageInfo.SetLayer3HashStorage(storageLayeredHashL3);
            storageInfo.SetDataStorage(storageData);
            NNT_ASSERT_RESULT_SUCCESS(
                storageIntegrity.Initialize(
                    infoLevelHash,
                    storageInfo,
                    &cacheBufferSet,
                    &locker,
                    storageType
                )
            );

            controlArea.Finalize();

            baseStorageCtrlAndOptionInfo.ResetAccessCounter();
            baseStorageMasterHash.ResetAccessCounter();
            baseStorageDataBody.ResetAccessCounter();

            // 適当な位置に書き込み
            if( 0 < sizeStorage )
            {
                for( int loop3 = 0; loop3 <= TestLoop; ++loop3 )
                {
                    const int64_t offset = nn::util::align_down(
                        std::uniform_int_distribution<int64_t>(0, sizeStorage)(mt),
                        sizeBlockLevelMax
                    );
                    const size_t size = nn::util::align_up(
                        std::uniform_int_distribution<size_t>(
                            0, static_cast<size_t>(sizeStorage - offset))(mt),
                        sizeBlockLevelMax
                    );
                    std::memset(ptr, (loop3 + 1) & 0xFF, size);

                    NNT_ASSERT_RESULT_SUCCESS(storageIntegrity.Write(offset, ptr, size));
                }

                // CtrlAndOption は更新されない
                ASSERT_EQ(0, baseStorageCtrlAndOptionInfo.GetWriteTimes());
                ASSERT_EQ(0, baseStorageMasterHash.GetFlushTimes());
                ASSERT_EQ(0, baseStorageDataBody.GetFlushTimes());
            }

            // アクセス回数リセット
            baseStorageCtrlAndOptionInfo.ResetAccessCounter();
            baseStorageMasterHash.ResetAccessCounter();
            baseStorageDataBody.ResetAccessCounter();

            // 0 バイト書き込み
            for( int loop3 = 0; loop3 <= TestLoop; ++loop3 )
            {
                size_t offset = nn::util::align_down(
                    std::uniform_int_distribution<size_t>(
                        0, static_cast<size_t>(sizeStorage))(mt),
                    sizeBlockLevelMax
                );
                if( sizeStorage == 0 )
                {
                    offset = 0;
                }

                baseStorageCtrlAndOptionInfo.ResetAccessCounter();
                baseStorageMasterHash.ResetAccessCounter();
                baseStorageDataBody.ResetAccessCounter();

                NNT_ASSERT_RESULT_SUCCESS(storageIntegrity.Write(offset, ptr, 0));

                // 0 バイト書き込みではファイルアクセスは発生しません。
                ASSERT_EQ(0, baseStorageCtrlAndOptionInfo.GetWriteTimes());
                ASSERT_EQ(0, baseStorageMasterHash.GetWriteTimes());
                ASSERT_EQ(0, baseStorageDataBody.GetFlushTimes());

                baseStorageCtrlAndOptionInfo.ResetAccessCounter();
                baseStorageMasterHash.ResetAccessCounter();
                baseStorageDataBody.ResetAccessCounter();
            }

            // 範囲外に書き込み
            for( int loop3 = 0; loop3 <= TestLoop; ++loop3 )
            {
                int64_t offset;
                size_t size;
                if( sizeStorage == 0 )
                {
                    offset = std::uniform_int_distribution<>(1, 10)(mt);
                    size = std::uniform_int_distribution<>(1, 10)(mt);
                }
                else
                {
                    offset = std::uniform_int_distribution<int64_t>(
                                sizeStorage + 1, sizeStorage * 2)(mt);
                    size = std::uniform_int_distribution<size_t>(
                                1, static_cast<size_t>(sizeStorage))(mt);
                }

                NNT_ASSERT_RESULT_FAILURE(
                    nn::fs::ResultInvalidOffset,
                    storageIntegrity.Write(offset, ptr, size)
                );
            }

            // 範囲外エラー発生時に書き込みは行われない
            ASSERT_EQ(0, baseStorageCtrlAndOptionInfo.GetWriteTimes());
            ASSERT_EQ(0, baseStorageMasterHash.GetWriteTimes());
            ASSERT_EQ(0, baseStorageDataBody.GetWriteTimes());

            // キャッシュ無効化のテストです
            for( auto count = 0; count < 100; ++count )
            {
                if( sizeStorage <= 0 )
                {
                    continue;
                }

                const auto offset = std::uniform_int_distribution<int64_t>(0, sizeStorage - 1)(mt);
                char buffer[1] = {};

                // 下位ストレージからデータを読み込む
                NNT_EXPECT_RESULT_SUCCESS(storageIntegrity.Read(offset, buffer, 1));

                // 同じ個所を読み込んでもキャッシュされているので下位ストレージへのアクセスは発生しないはず
                baseStorageDataBody.ResetAccessCounter();
                NNT_EXPECT_RESULT_SUCCESS(storageIntegrity.Read(offset, buffer, 1));
                EXPECT_EQ(0, baseStorageDataBody.GetReadTimes());

                if( storageType == nn::fs::StorageType::StorageType_SaveData )
                {
                    // StorageType_SaveDataではInvalidateしてはいけない
                    NNT_ASSERT_RESULT_FAILURE(
                        nn::fs::ResultUnsupportedOperation,
                        storageIntegrity.OperateRange(
                        nn::fs::OperationId::Invalidate,
                        0,
                        std::numeric_limits<int64_t>::max()));
                }
                else
                {
                    // キャッシュを破棄するとデータが壊れて読めなくなる
                    baseStorageDataBody.ResetAccessCounter();
                    NNT_EXPECT_RESULT_SUCCESS(storageIntegrity.OperateRange(
                        nn::fs::OperationId::Invalidate,
                        0,
                        std::numeric_limits<int64_t>::max()));
                    NNT_ASSERT_RESULT_FAILURE(
                        nn::fs::ResultNonRealDataVerificationFailed,
                        storageIntegrity.Read(offset, buffer, 1));
                }
            }
        }
    }
} // NOLINT(impl/function_size)

TEST(FsHierarchicalIntegrityVerificationStorageTest, Flush)
{
    FlushTest(nn::fs::StorageType::StorageType_RomFs);
    FlushTest(nn::fs::StorageType::StorageType_SaveData);
}

#if 0
TEST(FsHierarchicalIntegrityVerificationStorageTest, UnknownOperation)
{
    static const size_t TestSize = 1024 * 1024;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    int64_t controlSize;
    int64_t masterHashSize;

    const int64_t offset = std::uniform_int_distribution<int64_t>(0, 127)(mt);
    const size_t sizeBlock = static_cast<size_t>(1) << std::uniform_int_distribution<>(3, 10)(mt);

    NN_UNUSED(sizeBlock);

    // 各階層のブロックサイズは可変です(256-8192)
    // ただし、以下のパラメーターを考慮する必要が有ります。
    //
    //  ・キャッシュを有効に使うためには、1ブロックの大きさをキャッシュのページサイズと同等か、
    //　　それより大きなサイズを設定すべきです。
    //　　キャッシュのページサイズは "BlockCacheBufferManager" テンプレートの
    //　　コンパイル時に決定します。

    const size_t sizeBlockLevel1 = static_cast<size_t>(1) << std::uniform_int_distribution<>(8, 12)(mt);
    const size_t sizeBlockLevel2 = static_cast<size_t>(1) << std::uniform_int_distribution<>(8, 12)(mt);
    const size_t sizeBlockLevel3 = static_cast<size_t>(1) << std::uniform_int_distribution<>(8, 12)(mt);
    const size_t sizeBlockLevel4 = static_cast<size_t>(1) << std::uniform_int_distribution<>(8, 12)(mt);
    nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam inputParamIntegrity;
    inputParamIntegrity.sizeBlockLevel[0] = sizeBlockLevel1;
    inputParamIntegrity.sizeBlockLevel[1] = sizeBlockLevel2;
    inputParamIntegrity.sizeBlockLevel[2] = sizeBlockLevel3;
    inputParamIntegrity.sizeBlockLevel[3] = sizeBlockLevel4;

    // 検証ブロックサイズの最大値を求める
    size_t sizeBlockLevelMax = std::max(
                                   inputParamIntegrity.sizeBlockLevel[0],
                                   std::max(
                                       inputParamIntegrity.sizeBlockLevel[1],
                                       inputParamIntegrity.sizeBlockLevel[2]
                                   )
                               );

    // ファイルサイズを最大検証ブロック単位に合わせる
    const auto sizeStorage = nn::util::align_up(
                                 std::uniform_int_distribution<int64_t>(0, TestSize)(mt),
                                 sizeBlockLevelMax
                             );

    // メタデータに必要なサイズを取得します。
    // メタデータは各階層のパラメーター、オプション情報などを格納する領域です。
    nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea controlArea;

    // キャッシュを初期化します。
    //
    // キャッシュマネージャーはBuddyHeapを用いており、
    // 特定のバイト数 (2^x) がメモリ管理単位になっています。
    // 1ページ4096バイトの場合は、4096*2^(5-1) = 64KB を"一単位"とし、その倍数を与えます。
    // "一単位"は、BlockCacheBufferManager::QueryInitializationBlockSize() より求められます。
    // 但し、上記は推奨であり無駄な領域が多少発生する可能性はありますが任意のサイズを渡すことができます。
    // この場合でもページサイズの倍数であれば適切に分割されますので無駄な領域は発生しません。

    nn::os::Mutex locker(true);
    CacheBufferParam bufferParam;
    nn::fssystem::save::FilesystemBufferManagerSet cacheBufferSet;

    static const int MaxCacheCount = 1024;
    static const size_t CacheBufferPageSize = 4096;
    static const size_t CacheBufferMaxOrder = 12;

    const size_t sizeUnitBlock = QueryUnitBlockSize(CacheBufferMaxOrder, CacheBufferPageSize);

    size_t sizeCacheBuffer = sizeUnitBlock * nn::fssystem::save::IntegrityLayerCountSave;

    // キャッシュバッファを確保、初期化
    {
        auto& bufCache = bufferParam.bufCache;
        bufCache.resize(sizeCacheBuffer);
        bufferParam.memoryRange.first = reinterpret_cast<uintptr_t>(bufCache.data());
        bufferParam.memoryRange.second = bufCache.size();
        bufferParam.cacheBuffer.Initialize(
            MaxCacheCount,
            bufferParam.memoryRange.first,
            bufferParam.memoryRange.second,
            CacheBufferPageSize
        );
    }

    // 完全性検証ハッシュに割り当てるキャッシュバッファ
    for( auto& pBuffer : cacheBufferSet.pBuffer )
    {
        pBuffer = &bufferParam.cacheBuffer;
    }

    int64_t layerdHashSize = 0;
    int64_t bodySize = 0;

    // それぞれのサイズを求めます
    NNT_ASSERT_RESULT_SUCCESS(
        nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::QuerySize(
            &controlSize,
            &masterHashSize,
            &layerdHashSize,
            &bodySize,
            inputParamIntegrity,
            sizeStorage
        )
    );

    {
        nnt::fs::util::AccessCountedMemoryStorage baseStorageCtrlAndOptionInfo;
        nnt::fs::util::AccessCountedMemoryStorage baseStorageMasterHash;
        nnt::fs::util::AccessCountedMemoryStorage baseStorageDataBody;
        nn::fssystem::save::HierarchicalIntegrityVerificationStorage storageIntegrity;

        // 管理領域 + オプション領域の格納先ファイルを用意します
        baseStorageCtrlAndOptionInfo.Initialize(static_cast<size_t>(controlSize + offset * 2));

        // ルートハッシュを格納するメモリファイルを用意します
        baseStorageMasterHash.Initialize(static_cast<size_t>(masterHashSize + offset * 2));

        // 実データ+L1,L2,L3ハッシュを格納するメモリファイルを用意します
        baseStorageDataBody.Initialize(static_cast<size_t>(bodySize + offset * 2));

        // メタデータ領域をフォーマットします。
        // こちらはあくまでメタデータ領域のフォーマットであり
        // 実データ領域をフォーマットするわけでは有りません。
        nn::fs::SubStorage storageControlArea(&baseStorageCtrlAndOptionInfo, offset, controlSize);
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::Format(
                storageControlArea,
                inputParamIntegrity,
                sizeStorage
            )
        );

        // メタデータをマウントします。
        NNT_ASSERT_RESULT_SUCCESS(controlArea.Initialize(storageControlArea));

        // 管理領域からそれぞれのレベルのオフセット情報を取得します
        nn::fssystem::save::HierarchicalIntegrityVerificationInformation infoLevelHash;
        controlArea.GetLevelHashInfo(&infoLevelHash);

        // フォーマットで指定した値のとおりになっているか確認します
        ASSERT_EQ(masterHashSize, controlArea.GetMasterHashSize());
        ASSERT_EQ(
            nn::fssystem::save::ILog2(static_cast<uint32_t>(sizeBlockLevel1)),
            static_cast<size_t>(infoLevelHash.info[0].orderBlock)
        );
        ASSERT_EQ(
            nn::fssystem::save::ILog2(static_cast<uint32_t>(sizeBlockLevel2)),
            static_cast<size_t>(infoLevelHash.info[1].orderBlock)
        );
        ASSERT_EQ(
            nn::fssystem::save::ILog2(static_cast<uint32_t>(sizeBlockLevel3)),
            static_cast<size_t>(infoLevelHash.info[2].orderBlock)
        );

        // メタデータを元に完全性検証を初期化します。
        nn::fs::SubStorage storageMasterHash(&baseStorageMasterHash, offset, controlArea.GetMasterHashSize());
        nn::fs::SubStorage storageLayeredHash(&baseStorageDataBody, offset, infoLevelHash.GetLayeredHashSize());
        nn::fs::SubStorage storageData(&baseStorageDataBody, offset + infoLevelHash.GetDataOffset(), infoLevelHash.GetDataSize());
        NNT_ASSERT_RESULT_SUCCESS(
            storageIntegrity.Initialize(
                infoLevelHash,
                storageMasterHash,
                storageLayeredHash,
                storageData,
                &cacheBufferSet,
                &locker
            )
        );

        controlArea.Finalize();

        // 不明な操作は失敗するはず
        const auto offsetOperate
            = std::uniform_int_distribution<int64_t>(0, infoLevelHash.GetDataSize() - 1)(mt);
        const auto sizeOperate
            = std::uniform_int_distribution<int64_t>(1, infoLevelHash.GetDataSize() - offsetOperate)(mt);
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultUnsupportedOperation,
            storageIntegrity.OperateRange(
                static_cast<nn::fs::OperationId>(-1),
                offsetOperate,
                sizeOperate
            )
        );
    }
} // NOLINT(impl/function_size)
#endif

class HierarchicalIntegrityVerificationStorageTestSetup : public TestStorageSetup
{
public:
    static const int64_t MinFileSize = 2;

public:
    // コンストラクタです。
    HierarchicalIntegrityVerificationStorageTestSetup() NN_NOEXCEPT
        : m_Lock(true)
    {
    }

    // ファイルオブジェクトをフォーマットします。
    virtual void Format(
                     int64_t sizeStorage,
                     int64_t offset,
                     size_t sizeBlock
                 ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(sizeBlock);

        static const int CacheCountMax = 1024;
        static const size_t CacheBlockSize = 4 * 1024;
        static const size_t CacheBufferMaxOrder = 12;

        //
        // 各階層のブロックサイズは可変です(256 ～ MaxBlockSize)
        // ただし、以下のパラメーターを考慮する必要が有ります。
        //
        //  ・キャッシュを有効に使うためには、1ブロックの大きさをキャッシュのページサイズと同等か、
        //    それより大きなサイズを設定すべきです。
        //    キャッシュのページサイズは "BlockCacheBufferManager" テンプレートの
        //    コンパイル時に決定します。
        //

        m_SizeBlockLevel1 = GenerateRandomBlockSize(256, MaxBlockSize);
        m_SizeBlockLevel2 = GenerateRandomBlockSize(256, MaxBlockSize);
        m_SizeBlockLevel3 = GenerateRandomBlockSize(256, MaxBlockSize);
        m_SizeBlockLevel4 = GenerateRandomBlockSize(256, MaxBlockSize);
        nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam
            inputParamIntegrity;
        inputParamIntegrity.sizeBlockLevel[0] = m_SizeBlockLevel1;
        inputParamIntegrity.sizeBlockLevel[1] = m_SizeBlockLevel2;
        inputParamIntegrity.sizeBlockLevel[2] = m_SizeBlockLevel3;
        inputParamIntegrity.sizeBlockLevel[3] = m_SizeBlockLevel4;

        // メタデータに必要なサイズを取得します。
        // メタデータは各階層のパラメーター、オプション情報などを格納する領域です。
        nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea controlArea;

        // キャッシュを初期化します。
        //
        // キャッシュマネージャーはBuddyHeapを用いており、
        // 特定のバイト数 (2^x) がメモリ管理単位になっています。
        // 1ページ4096バイトの場合は、4096*2^(5-1) = 64KB を"一単位"とし、その倍数を与えます。
        // "一単位"は、BlockCacheBufferManager::QueryInitializationBlockSize() より求められます。
        // 但し、上記は推奨であり無駄な領域が多少発生する可能性はありますが任意のサイズを渡すことができます。
        // この場合でもページサイズの倍数であれば適切に分割されますので無駄な領域は発生しません。

        const size_t sizeUnitBlock =
            QueryUnitBlockSize(CacheBufferMaxOrder, CacheBlockSize);
        const size_t sizeCacheBuffer =
            sizeUnitBlock * nn::fssystem::save::IntegrityLayerCountSave;

        // キャッシュバッファを確保、初期化
        m_BufCache.resize(sizeCacheBuffer);
        NNT_ASSERT_RESULT_SUCCESS(
            m_CacheBuffer.Initialize(
                CacheCountMax,
                reinterpret_cast<uintptr_t>(&m_BufCache[0]),
                m_BufCache.size(),
                CacheBlockSize
            )
        );

        // 完全性検証ハッシュに割り当てるキャッシュバッファ
        for( size_t i = 0; i < nn::fssystem::save::IntegrityLayerCountSave; i++ )
        {
            m_CacheBufferSet.pBuffer[i] = &m_CacheBuffer;
        }

        // それぞれのサイズを求めます
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::QuerySize(
                &m_Sizes,
                inputParamIntegrity,
                nn::fssystem::save::IntegrityLayerCountSave,
                sizeStorage
            )
        );
        const int64_t bodySize = sizeStorage +
                                 m_Sizes.layeredHashSizes[0] +
                                 m_Sizes.layeredHashSizes[1] +
                                 m_Sizes.layeredHashSizes[2];

        // 管理領域 + オプション領域の格納先ファイルを用意します
        m_BaseStorageCtrlAndOptionInfo.reset(new nnt::fs::util::SafeMemoryStorage);
        m_BaseStorageCtrlAndOptionInfo->Initialize(m_Sizes.controlSize + offset * 2);

        // ルートハッシュを格納するメモリファイルを用意します
        m_BaseStorageMasterHash.reset(new nnt::fs::util::SafeMemoryStorage);
        m_BaseStorageMasterHash->Initialize(m_Sizes.masterHashSize + offset * 2);

        // 実データ+L1,L2,L3ハッシュを格納するメモリファイルを用意します
        m_BaseStorageDataBody.reset(new nnt::fs::util::SafeMemoryStorage);
        m_BaseStorageDataBody->Initialize(bodySize + offset * 2);

        // メタデータ領域をフォーマットします。
        // こちらはあくまでメタデータ領域のフォーマットであり
        // 実データ領域をフォーマットするわけでは有りません。
        nn::fs::SubStorage storageControlArea(
            m_BaseStorageCtrlAndOptionInfo.get(), offset, m_Sizes.controlSize);
        nn::fssystem::save::HierarchicalIntegrityVerificationMetaInformation meta = {};
        {
            int64_t offsetHashInfo = 0;
            meta.Format();
            meta.sizeMasterHash = static_cast<uint32_t>(m_Sizes.masterHashSize);
            meta.infoLevelHash.info[3].size = bodySize;
            meta.infoLevelHash.maxLayers = nn::fssystem::save::IntegrityLayerCountSave;
            for( auto level = 0; level < 3; ++level )
            {
                const int64_t size = m_Sizes.layeredHashSizes[level];
                meta.infoLevelHash.info[level].offset = offsetHashInfo;
                meta.infoLevelHash.info[level].size = size;
                meta.infoLevelHash.info[level].orderBlock =
                    nn::fssystem::save::ILog2(
                        static_cast<uint32_t>(inputParamIntegrity.sizeBlockLevel[level]));
                meta.infoLevelHash.info[3].size
                    = meta.infoLevelHash.info[3].size - size;
                offsetHashInfo += size;
            }
            meta.infoLevelHash.info[3].offset = offsetHashInfo;
            meta.infoLevelHash.info[3].orderBlock =
                nn::fssystem::save::ILog2(
                    static_cast<uint32_t>(inputParamIntegrity.sizeBlockLevel[3]));
        }
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::Format(
                storageControlArea,
                meta
            )
        );

        // ルートハッシュを0で埋めます
        m_BaseStorageMasterHash->OperateRange(nn::fs::OperationId::FillZero, offset, m_Sizes.masterHashSize);
    }

    // ファイルオブジェクトを初期化します。
    virtual void Initialize(
                     int64_t sizeStorage,
                     int64_t offset,
                     size_t sizeBlock
                 ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(sizeBlock);
        NN_UNUSED(sizeStorage);

        // メタデータをマウントします。
        nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea controlArea;
        nn::fs::SubStorage
            storageControlArea(m_BaseStorageCtrlAndOptionInfo.get(), offset, m_Sizes.controlSize);
        NNT_ASSERT_RESULT_SUCCESS(controlArea.Initialize(storageControlArea));

        // 管理領域からそれぞれのレベルのオフセット情報を取得します
        nn::fssystem::save::HierarchicalIntegrityVerificationInformation infoLevelHash;
        controlArea.GetLevelHashInfo(&infoLevelHash);

        // フォーマットで指定した値のとおりになっているか確認します
        ASSERT_EQ(m_Sizes.masterHashSize, controlArea.GetMasterHashSize());
        ASSERT_EQ(
            nn::fssystem::save::ILog2(static_cast<uint32_t>(m_SizeBlockLevel1)),
            static_cast<uint32_t>(infoLevelHash.info[0].orderBlock)
        );
        ASSERT_EQ(
            nn::fssystem::save::ILog2(static_cast<uint32_t>(m_SizeBlockLevel2)),
            static_cast<uint32_t>(infoLevelHash.info[1].orderBlock)
        );
        ASSERT_EQ(
            nn::fssystem::save::ILog2(static_cast<uint32_t>(m_SizeBlockLevel3)),
            static_cast<uint32_t>(infoLevelHash.info[2].orderBlock)
        );

        // メタデータを元に完全性検証を初期化します。
        m_StorageIntegrity.reset(
            new nn::fssystem::save::HierarchicalIntegrityVerificationStorage());

        nn::fs::SubStorage storageMasterHash(
            m_BaseStorageMasterHash.get(), offset, controlArea.GetMasterHashSize());
        nn::fs::SubStorage storageLayeredHash(
            m_BaseStorageDataBody.get(), offset, infoLevelHash.GetLayeredHashSize());
        nn::fs::SubStorage storageLayeredHashL1(
            &storageLayeredHash,
            infoLevelHash.info[0].offset,
            infoLevelHash.info[0].size
        );
        nn::fs::SubStorage storageLayeredHashL2(
            &storageLayeredHash,
            infoLevelHash.info[1].offset,
            infoLevelHash.info[1].size
        );
        nn::fs::SubStorage storageLayeredHashL3(
            &storageLayeredHash,
            infoLevelHash.info[2].offset,
            infoLevelHash.info[2].size
        );
        nn::fs::SubStorage storageData(
            m_BaseStorageDataBody.get(),
            offset + infoLevelHash.GetDataOffset(),
            infoLevelHash.GetDataSize()
        );
        nn::fssystem::save::HierarchicalIntegrityVerificationStorage::HierarchicalStorageInformation storageInfo;
        storageInfo.SetMasterHashStorage(storageMasterHash);
        storageInfo.SetLayer1HashStorage(storageLayeredHashL1);
        storageInfo.SetLayer2HashStorage(storageLayeredHashL2);
        storageInfo.SetLayer3HashStorage(storageLayeredHashL3);
        storageInfo.SetDataStorage(storageData);
        NNT_ASSERT_RESULT_SUCCESS(
            m_StorageIntegrity->Initialize(
                infoLevelHash,
                storageInfo,
                &m_CacheBufferSet,
                &m_Lock,
                nn::fs::StorageType::StorageType_SaveData
            )
        );

        SetStorage(m_StorageIntegrity.get());

        controlArea.Finalize();
    }

    // ファイルオブジェクトをアンマウントします。
    virtual void Unmount() NN_NOEXCEPT NN_OVERRIDE
    {
        m_StorageIntegrity.reset(nullptr);
    }

    // ファイルオブジェクトを破棄します。
    virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE
    {
        m_StorageIntegrity.reset(nullptr);
        m_BaseStorageCtrlAndOptionInfo.reset(nullptr);
        m_BaseStorageMasterHash.reset(nullptr);
        m_BaseStorageDataBody.reset(nullptr);
        m_CacheBuffer.Finalize();
    }

    // 読み込み専用かどうか
    virtual bool IsReadOnly() const NN_NOEXCEPT NN_OVERRIDE
    {
        return false;
    }

    // ファイルへのデータ書き込み時に自動伸長するかどうか
    virtual bool IsAutoExtendSizeOnWrite() const NN_NOEXCEPT NN_OVERRIDE
    {
        return false;
    }

    // ファイルのサイズ変更が可能かどうか
    virtual bool IsEnableResize() const NN_NOEXCEPT NN_OVERRIDE
    {
        return false;
    }

    // サイズ 0 のファイルを作成可能かどうか
    virtual bool IsAcceptsSizeZero() const NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

private:
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> m_BaseStorageCtrlAndOptionInfo;
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> m_BaseStorageMasterHash;
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> m_BaseStorageDataBody;
    std::unique_ptr<nn::fssystem::save::HierarchicalIntegrityVerificationStorage>
        m_StorageIntegrity;

    nn::fssystem::save::FilesystemBufferManagerSet m_CacheBufferSet;
    nnt::fs::util::Vector<char> m_BufCache;
    nn::fssystem::FileSystemBufferManager m_CacheBuffer;
    nn::os::Mutex m_Lock;

    size_t m_SizeBlockLevel1;
    size_t m_SizeBlockLevel2;
    size_t m_SizeBlockLevel3;
    size_t m_SizeBlockLevel4;
    nn::fssystem::save::HierarchicalIntegrityVerificationSizeSet m_Sizes;
};

// 共通テストを行います。
TEST(FsHierarchicalIntegrityVerificationStorageTest, StorageCommonTest)
{
    StorageTest::RunTest<HierarchicalIntegrityVerificationStorageTestSetup>();
}
