﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <nn/fssystem/save/fs_DuplexBitmapHolder.h>
#include <nn/fssystem/save/fs_HierarchicalDuplexStorage.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>
#include "testFs_util_ReadOnlyFilterStorage.h"

// 二重化ビットマップファイルのテスト
// 1 バイトよりも上 + 中途半端なサイズの二重化ビットマップのテストを行います。
TEST(FsDuplexBitmapHolderTest, Simple)
{
    static const uint32_t BlockCount = 19;

    const int64_t sizeStorage =
        nn::fssystem::save::DuplexBitmapHolder::QuerySize(BlockCount) + 16;

    nnt::fs::util::SafeMemoryStorage storageOriginal;
    nnt::fs::util::SafeMemoryStorage storageModified;
    storageOriginal.Initialize(sizeStorage);
    storageModified.Initialize(sizeStorage);

    NNT_ASSERT_RESULT_SUCCESS(
        nn::fssystem::save::DuplexBitmapHolder::Format(
            BlockCount,
            nn::fs::SubStorage(&storageModified, 8, sizeStorage),
            nn::fs::SubStorage(&storageOriginal, 8, sizeStorage)
        )
    );

    nn::fssystem::save::DuplexBitmapHolder bitmap;
    bitmap.Initialize(
        BlockCount,
        nn::fs::SubStorage(&storageModified, 8, sizeStorage),
        nn::fs::SubStorage(&storageOriginal, 8, sizeStorage)
    );

    size_t zeroCount = 0;
    size_t oneCount = 0;

    // オリジナルメタデータに対するイテレーションをテスト
    {
        // イテレータを初期化
        nn::fssystem::save::DuplexBitmapHolder::Iterator iter;
        bitmap.IterateBegin(&iter, 0, 19);

        // 0 が 19 ビット分連続していることをチェック
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateOriginalNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(zeroCount, 19);
        ASSERT_EQ(oneCount, 0);

        // イテレーションが完了していることをチェック
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateOriginalNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(zeroCount, 0);
        ASSERT_EQ(oneCount, 0);
    }

    // メタデータに対するイテレーションをテスト
    {
        // イテレータを初期化
        nn::fssystem::save::DuplexBitmapHolder::Iterator iter;
        bitmap.IterateBegin(&iter, 0, 19);

        // 0 が 19 ビット分連続していることをチェック
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(zeroCount, 19);
        ASSERT_EQ(oneCount, 0);

        // イテレーションが完了していることをチェック
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateOriginalNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(zeroCount, 0);
        ASSERT_EQ(oneCount, 0);
    }

    // ビットを書き換え
    {
        // ビットを書き換え
        //
        // 書き換え後のビット
        // org: 0000000000000000000
        // cur: 0000011111111100000
        NNT_ASSERT_RESULT_SUCCESS(bitmap.MarkModified(5, 9));

        {
            // イテレータを初期化
            nn::fssystem::save::DuplexBitmapHolder::Iterator iter;
            bitmap.IterateBegin(&iter, 0, 19);

            // 先頭に 0 が 5 個、1 が 9 個連続していることをチェック
            NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
            ASSERT_EQ(zeroCount, 5);
            ASSERT_EQ(oneCount, 9);

            // 先頭に 0 が 5 個連続していることをチェック
            NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
            ASSERT_EQ(zeroCount, 5);
            ASSERT_EQ(oneCount, 0);

            // イテレーションが完了していることをチェック
            NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
            ASSERT_EQ(zeroCount, 0);
            ASSERT_EQ(oneCount, 0);
        }

        // 途中からイテレーションを行う
        {
            // イテレータを初期化
            nn::fssystem::save::DuplexBitmapHolder::Iterator iter;
            bitmap.IterateBegin(&iter, 6, 10);

            // 先頭に 1 が 8 個連続していることをチェック
            NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
            ASSERT_EQ(zeroCount, 0);
            ASSERT_EQ(oneCount, 8);

            // 先頭に 0 が 2 個連続していることをチェック
            NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
            ASSERT_EQ(zeroCount, 2);
            ASSERT_EQ(oneCount, 0);

            // イテレーションが完了していることをチェック
            NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
            ASSERT_EQ(zeroCount, 0);
            ASSERT_EQ(oneCount, 0);
        }
    }

    bitmap.Finalize();
}

// マウントだけでは書き込まれないことをテスト
TEST(FsDuplexBitmapHolderTest, ReadOnly)
{
    static const uint32_t BlockCount = 128;

    const int64_t sizeStorage =
        nn::fssystem::save::DuplexBitmapHolder::QuerySize(BlockCount) + 16;

    nnt::fs::util::SafeMemoryStorage storageOriginal;
    nnt::fs::util::SafeMemoryStorage storageModified;
    storageOriginal.Initialize(sizeStorage);
    storageModified.Initialize(sizeStorage);

    NNT_ASSERT_RESULT_SUCCESS(
        nn::fssystem::save::DuplexBitmapHolder::Format(
            BlockCount,
            nn::fs::SubStorage(&storageModified, 8, sizeStorage),
            nn::fs::SubStorage(&storageOriginal, 8, sizeStorage)
        )
    );

    ReadOnlyFilterStorage storageReadOnlyOriginal;
    ReadOnlyFilterStorage storageReadOnlyModified;
    storageReadOnlyOriginal.Initialize(&storageOriginal);
    storageReadOnlyModified.Initialize(&storageModified);

    nn::fssystem::save::DuplexBitmapHolder bitmap;
    bitmap.Initialize(
        BlockCount,
        nn::fs::SubStorage(&storageReadOnlyModified, 8, sizeStorage),
        nn::fs::SubStorage(&storageReadOnlyOriginal, 8, sizeStorage)
    );

    size_t zeroCount = 0;
    size_t oneCount = 0;

    // 読み込みは失敗しない
    {
        // イテレータを初期化
        nn::fssystem::save::DuplexBitmapHolder::Iterator iter;
        bitmap.IterateBegin(&iter, 0, BlockCount);

        // 0が連続していることをチェック
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(zeroCount, BlockCount);
        ASSERT_EQ(oneCount, 0);

        // イテレーションが完了していることをチェック
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateOriginalNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(zeroCount, 0);
        ASSERT_EQ(oneCount, 0);
    }

    // 変更は失敗する
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultUnsupportedOperation, bitmap.MarkModified(0, 1));

    bitmap.Finalize();
}

// 二重化ビットマップファイルの拡張のテストを行います。
TEST(FsDuplexBitmapHolderTest, Expand)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( uint32_t blockCountOld = 1; blockCountOld <= 1024; blockCountOld <<= 2 )
    {
        for( uint32_t blockCountNew = blockCountOld + 1;
             blockCountNew <= 2048;
             blockCountNew <<= 1 )
        {
            // 拡張前のストレージを確保します。
            const int64_t sizeOld =
                nn::fssystem::save::DuplexBitmapHolder::QuerySize(blockCountOld);
            nnt::fs::util::SafeMemoryStorage storageOriginalOld(sizeOld);
            nnt::fs::util::SafeMemoryStorage storageModifiedOld(sizeOld);
            nn::fs::SubStorage subStorageOriginalOld(&storageOriginalOld, 0, sizeOld);
            nn::fs::SubStorage subStorageModifiedOld(&storageModifiedOld, 0, sizeOld);

            // フォーマットして初期化します。
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::DuplexBitmapHolder::Format(
                    blockCountOld, subStorageModifiedOld, subStorageOriginalOld
                )
            );

            nn::fssystem::save::DuplexBitmapHolder bitmap;
            bitmap.Initialize(blockCountOld, subStorageModifiedOld, subStorageOriginalOld);

            // ランダムにビットを反転させます。
            nnt::fs::util::Vector<bool> modifiedBits(blockCountNew, false);
            for( uint32_t index = 0; index < blockCountOld; ++index )
            {
                if( std::uniform_int_distribution<>(0, 1)(mt) == 0 )
                {
                    // ビットを反転させます。
                    NNT_ASSERT_RESULT_SUCCESS(bitmap.MarkModified(index, 1));

                    // ビットを反転させたことを記録します。
                    modifiedBits[index] = true;
                }
            }

            // ビットが反転したことを確認します。
            {
                nn::fssystem::dbm::DuplexBitmap::Iterator iter;
                bitmap.IterateBegin(&iter, 0, blockCountOld);

                size_t oneCount = 0;
                size_t zeroCount = 0;

                for( size_t index = 0; index < blockCountOld; )
                {
                    NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
                    ASSERT_LE(index + zeroCount + oneCount, blockCountOld);

                    // 0 の続く範囲内にビットを反転させた記録が無いことを確認します。
                    for( size_t zeroIndex = index; zeroIndex < index + zeroCount; ++zeroIndex )
                    {
                        ASSERT_FALSE(modifiedBits[zeroIndex]);
                    }

                    index += zeroCount;

                    // 1 の続く範囲内にビットを反転させた記録があることを確認します。
                    for( size_t oneIndex = index; oneIndex < index + oneCount; ++oneIndex )
                    {
                        ASSERT_TRUE(modifiedBits[oneIndex]);
                    }

                    index += oneCount;
                }

                // イテレーションが終了します。
                NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
                ASSERT_EQ(zeroCount, 0);
                ASSERT_EQ(oneCount, 0);
            }

            // 拡張のために一旦アンマウントします。
            bitmap.Finalize();

            // 拡張後のストレージを確保します。
            const int64_t sizeNew =
                nn::fssystem::save::DuplexBitmapHolder::QuerySize(blockCountNew);
            nnt::fs::util::SafeMemoryStorage storageOriginalNew(sizeNew);
            nnt::fs::util::SafeMemoryStorage storageModifiedNew(sizeNew);
            nn::fs::SubStorage subStorageOriginalNew(&storageOriginalNew, 0, sizeNew);
            nn::fs::SubStorage subStorageModifiedNew(&storageModifiedNew, 0, sizeNew);

            // 拡張前のストレージの内容を書き込みます。
            NNT_ASSERT_RESULT_SUCCESS(
                storageOriginalNew.Write(
                    0,
                    storageOriginalOld.GetBuffer(),
                    static_cast<size_t>(storageOriginalOld.GetSize())
                )
            );
            NNT_ASSERT_RESULT_SUCCESS(
                storageModifiedNew.Write(
                    0,
                    storageModifiedOld.GetBuffer(),
                    static_cast<size_t>(storageModifiedOld.GetSize())
                )
            );
            NNT_ASSERT_RESULT_SUCCESS(storageOriginalNew.Flush());
            NNT_ASSERT_RESULT_SUCCESS(storageModifiedNew.Flush());

            // ビットマップを拡張します。
            NNT_ASSERT_RESULT_SUCCESS(
                bitmap.Expand(
                    blockCountOld,
                    blockCountNew,
                    subStorageModifiedNew,
                    subStorageOriginalNew
                )
            );

            // 再度マウントします。
            bitmap.Initialize(blockCountNew, subStorageModifiedNew, subStorageOriginalNew);

            // ビットの状態が変化していないことを確認します。
            {
                nn::fssystem::dbm::DuplexBitmap::Iterator iter;
                bitmap.IterateBegin(&iter, 0, blockCountNew);

                size_t oneCount = 0;
                size_t zeroCount = 0;

                for( size_t index = 0; index < blockCountNew; )
                {
                    NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
                    ASSERT_LE(index + zeroCount + oneCount, blockCountNew);

                    // 0 の続く範囲内にビットを反転させた記録が無いことを確認します。
                    for( size_t zeroIndex = index; zeroIndex < index + zeroCount; ++zeroIndex )
                    {
                        if( zeroIndex < blockCountOld )
                        {
                            ASSERT_FALSE(modifiedBits[zeroIndex]);
                        }
                    }

                    index += zeroCount;

                    if( oneCount > 0 )
                    {
                        // 1 の続く範囲内に拡張された領域はありません。
                        ASSERT_LE(index + oneCount, blockCountOld);
                    }

                    // 1 の続く範囲内にビットを反転させた記録があることを確認します。
                    for( size_t oneIndex = index; oneIndex < index + oneCount; ++oneIndex )
                    {
                        ASSERT_TRUE(modifiedBits[oneIndex]);
                    }

                    index += oneCount;
                }

                // イテレーションが終了します。
                NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
                ASSERT_EQ(zeroCount, 0);
                ASSERT_EQ(oneCount, 0);
            }

            // ランダムにビットを反転させます。
            for( uint32_t index = 0; index < blockCountNew; ++index )
            {
                if( std::uniform_int_distribution<>(0, 1)(mt) == 0 )
                {
                    // ビットを反転させます。
                    NNT_ASSERT_RESULT_SUCCESS(bitmap.MarkModified(index, 1));

                    // ビットを反転させたことを記録します。
                    modifiedBits[index] = true;
                }
            }

            // ビットが反転したことを確認します。
            {
                nn::fssystem::dbm::DuplexBitmap::Iterator iter;
                bitmap.IterateBegin(&iter, 0, blockCountNew);

                size_t oneCount = 0;
                size_t zeroCount = 0;

                for( size_t index = 0; index < blockCountNew; )
                {
                    NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
                    ASSERT_LE(index + zeroCount + oneCount, blockCountNew);

                    // 0 の続く範囲内にビットを反転させた記録が無いことを確認します。
                    for( size_t zeroIndex = index; zeroIndex < index + zeroCount; ++zeroIndex )
                    {
                        ASSERT_FALSE(modifiedBits[zeroIndex]);
                    }

                    index += zeroCount;

                    // 1 の続く範囲内にビットを反転させた記録があることを確認します。
                    for( size_t oneIndex = index; oneIndex < index + oneCount; ++oneIndex )
                    {
                        ASSERT_TRUE(modifiedBits[oneIndex]);
                    }

                    index += oneCount;
                }

                // イテレーションが終了します。
                NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
                ASSERT_EQ(zeroCount, 0);
                ASSERT_EQ(oneCount, 0);
            }

            bitmap.Finalize();
        }
    }
} // NOLINT(impl/function_size)
