﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <numeric>
#include <deque>
#include <random>
#include <nn/os/os_Thread.h>
#include <nn/fssystem/save/fs_BufferedStorage.h>
#include <nn/fssystem/buffers/fs_FileSystemBufferManager.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>
#include "testFs_util_CommonFileSystemTests.h"

class FsBufferedStorageTest : public ::testing::TestWithParam<bool>
{
protected:
    bool IsBulkReadEnabled() const NN_NOEXCEPT
    {
        return GetParam();
    }
};

typedef FsBufferedStorageTest FsBufferedStorageDeathTest;

// ストレージの初期化・書き込みを行う簡単なテスト
TEST_P(FsBufferedStorageTest, Simple)
{
    static const size_t TestSize = 1024 * 1024;
    static const size_t BufferCacheSize = 2 * 1024;
    static const int BufferCacheCount = 4;

    nnt::fs::util::Vector<char> buf1(TestSize);
    nnt::fs::util::Vector<char> buf2(TestSize);
    nnt::fs::util::Vector<char> bufCache(BufferCacheSize * BufferCacheCount);

    nn::fssystem::FileSystemBufferManager bufManager;
    NNT_ASSERT_RESULT_SUCCESS(
        bufManager.Initialize(
            BufferCacheCount,
            reinterpret_cast<uintptr_t>(bufCache.data()),
            bufCache.size(),
            BufferCacheSize
        )
    );

    // テストデータ設定
    for( size_t i = 0; i < TestSize; ++i )
    {
        buf1[i] = i & 0xFF;
    }

    for( size_t i = 0; i < TestSize; )
    {
        nnt::fs::util::SafeMemoryStorage storageMem(i);

        std::memset(&buf2[0], 0x55, TestSize);
        NNT_ASSERT_RESULT_SUCCESS(storageMem.Write(0, &buf2[0], i));

        nn::fssystem::save::BufferedStorage storage;
        NNT_ASSERT_RESULT_SUCCESS(
            storage.Initialize(
                nn::fs::SubStorage(&storageMem, 0, i),
                &bufManager,
                BufferCacheSize,
                BufferCacheCount
            )
        );
        if( IsBulkReadEnabled() )
        {
            storage.EnableBulkRead();
        }

        // バイト単位での書き込み戻り値確認
        for( size_t j = 0; j <= i; ++j )
        {
            NNT_ASSERT_RESULT_SUCCESS(storage.Write(j, &buf1[j], 1));
        }

        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset,
            storage.Write(i + 1, &buf1[0], 1)
        );

        // データ比較
        NNT_ASSERT_RESULT_SUCCESS(storage.Read(0, &buf2[0], i));

        ASSERT_EQ(0, std::memcmp(&buf1[0], &buf2[0], i));
        ASSERT_EQ(0x55, buf2[i]);

        storage.Flush();

        if( i == 0 )
        {
            i = 1;
        }
        else
        {
            i *= 2;
        }
    }
}

// 4 GB を超えるオフセットの読み書きテスト
TEST_P(FsBufferedStorageTest, LargeOffset)
{
    static const size_t AccessSize = 1024;
    static const int64_t StorageSize = static_cast<int64_t>(64) * 1024 * 1024 * 1024 + AccessSize;
    static const size_t BufferCacheSize = 2 * 1024;
    static const int BufferCacheCount = 4;

    nnt::fs::util::Vector<char> bufCache(BufferCacheSize * BufferCacheCount);
    nn::fssystem::FileSystemBufferManager bufManager;
    NNT_ASSERT_RESULT_SUCCESS(
        bufManager.Initialize(
            BufferCacheCount,
            reinterpret_cast<uintptr_t>(bufCache.data()),
            bufCache.size(),
            BufferCacheSize
        )
    );

    nnt::fs::util::VirtualMemoryStorage storageMem(StorageSize);

    // ストレージの初期化
    nn::fssystem::save::BufferedStorage storage;
    NNT_ASSERT_RESULT_SUCCESS(
        storage.Initialize(
            nn::fs::SubStorage(&storageMem, 0, StorageSize),
            &bufManager,
            BufferCacheSize,
            BufferCacheCount
        )
    );
    if( IsBulkReadEnabled() )
    {
        storage.EnableBulkRead();
    }

    TestLargeOffsetAccess(
        &storage,
        AccessSize,
        [&](int64_t offset, char* readBuffer, const char* writeBuffer, size_t bufferSize) NN_NOEXCEPT
        {
            // BufferedStorage から読み込み
            NNT_ASSERT_RESULT_SUCCESS(storage.Read(offset, readBuffer, bufferSize));
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer, readBuffer, bufferSize);

            // 元ストレージから読み込み
            NNT_ASSERT_RESULT_SUCCESS(storageMem.Read(offset, readBuffer, bufferSize));
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer, readBuffer, bufferSize);

        });
}

// ストレージの読み書きのテスト
TEST_P(FsBufferedStorageTest, ReadWrite)
{
    static const size_t TestSize = 1024 * 1024;
    static const int BufferCacheCount = 8;
    static const size_t BufferCacheSizeMin = 1024;
    static const size_t BufferCacheSizeMax = 8 * 1024;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::Vector<char> readBuffer(TestSize);
    nnt::fs::util::Vector<char> writeBuffer(TestSize);
    nnt::fs::util::Vector<char> writeBuffer2(TestSize, 0); // BufferedStorage と同じ内容を書き込むバッファ
    nnt::fs::util::Vector<char> bufCache(BufferCacheSizeMax * BufferCacheCount);

    nn::fssystem::FileSystemBufferManager bufManager;
    NNT_ASSERT_RESULT_SUCCESS(
        bufManager.Initialize(
            BufferCacheCount,
            reinterpret_cast<uintptr_t>(bufCache.data()),
            bufCache.size(),
            BufferCacheSizeMin
        )
    );

    for( size_t sizeTest = 1; sizeTest <= TestSize; sizeTest *= 2 )
    {
        // テストバッファに初期値を書き込み
        std::memset(writeBuffer2.data(), 0, TestSize);

        // バッファードファイルをセットアップ
        size_t bufferCacheSize = GenerateRandomBlockSize(512, 4096);

        nnt::fs::util::SafeMemoryStorage storageMem(sizeTest);
        NNT_ASSERT_RESULT_SUCCESS(storageMem.Write(0, writeBuffer2.data(), sizeTest));

        nn::fssystem::save::BufferedStorage storage;
        NNT_ASSERT_RESULT_SUCCESS(
            storage.Initialize(
                nn::fs::SubStorage(&storageMem, 0, sizeTest),
                &bufManager,
                bufferCacheSize,
                BufferCacheCount
            )
        );
        if( IsBulkReadEnabled() )
        {
            storage.EnableBulkRead();
        }

        // ランダムな位置にデータを書き、内容を比較
        for( int loop = 1000; 0 < loop; --loop )
        {
            // テストデータを書き込みます。
            size_t offset = std::uniform_int_distribution<size_t>(0, sizeTest - 1)(mt);
            size_t size = std::uniform_int_distribution<size_t>(1, sizeTest - offset)(mt);

            // 二つのバッファにそれぞれ書き込み
            std::memset(writeBuffer.data(), loop & 0xFF, size);
            NNT_ASSERT_RESULT_SUCCESS(storage.Write(offset, writeBuffer.data(), size));
            std::memcpy(writeBuffer2.data() + offset, writeBuffer.data(), size);

            // 書き込んだ位置を読み込んでデータ比較
            NNT_ASSERT_RESULT_SUCCESS(storage.Read(offset, readBuffer.data(), size));
            NNT_FS_UTIL_ASSERT_MEMCMPEQ(writeBuffer.data(), readBuffer.data(), size);

            if( (loop == 1) || ((loop % 100) == 0) )
            {
                // ストレージ全域を読み込んでデータ比較
                NNT_ASSERT_RESULT_SUCCESS(storage.Read(0, readBuffer.data(), sizeTest));
                NNT_FS_UTIL_ASSERT_MEMCMPEQ(writeBuffer2.data(), readBuffer.data(), sizeTest);
            }
        }

        // 元ストレージに書き込まれているか比較
        NNT_ASSERT_RESULT_SUCCESS(storage.Flush());
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(writeBuffer2.data(), storageMem.GetBuffer(), sizeTest);
    }
}

// フラッシュのテスト
TEST_P(FsBufferedStorageTest, Flush)
{
    static const size_t SizeTest = 8 * 1024;
    static const int BufferCacheCount = 4;
    static const size_t BufferCacheSize = 1 * 1024;

    nnt::fs::util::Vector<char> buf1(SizeTest);
    char* ptr1 = &buf1[0];

    nnt::fs::util::Vector<char> bufCache(BufferCacheSize * BufferCacheCount);
    nn::fssystem::FileSystemBufferManager bufManager;
    NNT_ASSERT_RESULT_SUCCESS(
        bufManager.Initialize(
            BufferCacheCount,
            reinterpret_cast<uintptr_t>(bufCache.data()),
            bufCache.size(),
            BufferCacheSize
        )
    );

    // テストデータ設定
    for( size_t i = 0; i < SizeTest; ++i )
    {
        buf1[i] = i & 0xFF;
    }

    nnt::fs::util::AccessCountedMemoryStorage memStorage;
    memStorage.Initialize(SizeTest);
    NNT_ASSERT_RESULT_SUCCESS(memStorage.Write(0, ptr1, SizeTest));

    nn::fssystem::save::BufferedStorage storage;
    NNT_ASSERT_RESULT_SUCCESS(
        storage.Initialize(
            nn::fs::SubStorage(&memStorage, 0, SizeTest),
            &bufManager,
            BufferCacheSize,
            BufferCacheCount
        )
    );
    if( IsBulkReadEnabled() )
    {
        storage.EnableBulkRead();
    }

    // アクセス回数リセット
    memStorage.ResetAccessCounter();

    // 範囲内へのデータ書き込み
    NNT_ASSERT_RESULT_SUCCESS(storage.Write(0, ptr1, SizeTest / 2));
    NNT_ASSERT_RESULT_SUCCESS(storage.Write(SizeTest / 2, ptr1, SizeTest / 2));
    NNT_ASSERT_RESULT_SUCCESS(storage.Write(0, ptr1, SizeTest));

    ASSERT_EQ(0, memStorage.GetFlushTimes());

    // アクセス回数リセット
    memStorage.ResetAccessCounter();

    // Flush を呼ぶと下位の Flush も呼ばれる
    NNT_ASSERT_RESULT_SUCCESS(storage.Flush());
    ASSERT_EQ(1, memStorage.GetFlushTimes());

    // アクセス回数リセット
    memStorage.ResetAccessCounter();

    // 範囲外へのアクセス
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidOffset,
        storage.Write(24 * 1024 + 1, ptr1, 8 * 1024)
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidOffset,
        storage.Write(32 * 1024, ptr1, 8 * 1024)
    );

    ASSERT_EQ(0, memStorage.GetFlushTimes());
}

// フラッシュとベリファイのテスト
TEST_P(FsBufferedStorageTest, FlushVerify)
{
    static const int BufCount = 2;
    static const size_t BlockSizeMin = 1024 / 2;

    nnt::fs::util::Vector<char> bufCache(256 * 1024);
    nn::fssystem::FileSystemBufferManager bufManager;
    NNT_ASSERT_RESULT_SUCCESS(
        bufManager.Initialize(
            BufCount,
            reinterpret_cast<uintptr_t>(bufCache.data()),
            bufCache.size(),
            BlockSizeMin
        )
    );

    for( size_t memSize = 1024; memSize <= 64 * 1024; memSize *= 4 )
    {
        nnt::fs::util::AccessCountedMemoryStorage memStorage;
        memStorage.Initialize(memSize);

        size_t blockSize = memSize / BufCount;

        nn::fssystem::save::BufferedStorage bufStorage;
        bufStorage.Initialize(
            nn::fs::SubStorage(&memStorage, 0, memSize),
            &bufManager,
            blockSize,
            BufCount
        );
        if( IsBulkReadEnabled() )
        {
            bufStorage.EnableBulkRead();
        }

        // バッファとファイルの初期化
        nnt::fs::util::Vector<char> writeBuffer(memSize - 1);
        nnt::fs::util::Vector<char> readBuffer(memSize - 1, 0);
        std::iota(writeBuffer.begin(), writeBuffer.end(), '\x1');

        nn::Result result = memStorage.Write(0, readBuffer.data(), readBuffer.size());
        NNT_ASSERT_RESULT_SUCCESS(result);

        // 書き込みを行っても、フラッシュするまではキャッシュされているデータは元ファイルに反映されない
        for( int count = 0; count < 3; ++count )
        {
            // 書き込む
            result = bufStorage.Write(0, writeBuffer.data(), writeBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);

            // キャッシュされているので読み込むことはできる
            memStorage.ResetAccessCounter();
            result = bufStorage.Read(0, readBuffer.data(), readBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(std::equal(writeBuffer.begin(), writeBuffer.end(), readBuffer.begin()));
            EXPECT_EQ(0, memStorage.GetReadTimes());

            // しかし元ファイルには反映されていない
            result = memStorage.Read(0, readBuffer.data(), readBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(
                std::find_if(
                    readBuffer.begin(),
                    readBuffer.end(),
                    [](const char& value) NN_NOEXCEPT
                    {
                        return value != 0;
                    }
                ) == readBuffer.end()
            );
        }

        // フラッシュする
        result = bufStorage.Flush();
        NNT_ASSERT_RESULT_SUCCESS(result);

        // フラッシュすると反映される
        {
            // キャッシュから読み込む
            memStorage.ResetAccessCounter();
            result = bufStorage.Read(0, readBuffer.data(), readBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(std::equal(writeBuffer.begin(), writeBuffer.end(), readBuffer.begin()));
            EXPECT_EQ(0, memStorage.GetReadTimes());

            // 元ファイルからも同じデータが読み込める
            result = memStorage.Read(0, readBuffer.data(), readBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(std::equal(writeBuffer.begin(), writeBuffer.end(), readBuffer.begin()));
        }
    }
}

// キャッシュ無効化のテスト
TEST_P(FsBufferedStorageTest, Invalidate)
{
    static const auto BufferCacheCount = 4;
    static const auto BufferCacheSize = 256;
    static const auto StorageSize = 16 * 1024;

    // ストレージを用意する
    nnt::fs::util::Vector<char> bufCache(BufferCacheSize * BufferCacheCount);
    nn::fssystem::FileSystemBufferManager bufManager;
    NNT_ASSERT_RESULT_SUCCESS(
        bufManager.Initialize(
            BufferCacheCount,
            reinterpret_cast<uintptr_t>(bufCache.data()),
            bufCache.size(),
            BufferCacheSize
        )
    );

    nnt::fs::util::AccessCountedMemoryStorage memStorage;
    memStorage.Initialize(StorageSize);

    nn::fssystem::save::BufferedStorage storage;
    NNT_ASSERT_RESULT_SUCCESS(
        storage.Initialize(
            nn::fs::SubStorage(&memStorage, 0, StorageSize),
            &bufManager,
            BufferCacheSize,
            BufferCacheCount
        )
    );
    if( IsBulkReadEnabled() )
    {
        storage.EnableBulkRead();
    }

    std::mt19937 rng(nnt::fs::util::GetRandomSeed());

    for( auto count = 0; count < 100; ++count )
    {
        const auto offset = std::uniform_int_distribution<int64_t>(0, StorageSize - 1)(rng);
        char buffer[1] = {};

        // 下位ストレージからデータを読み込む
        NNT_EXPECT_RESULT_SUCCESS(storage.Read(offset, buffer, 1));

        // 同じ個所を読み込んでもキャッシュされているので下位ストレージへのアクセスは発生しないはず
        memStorage.ResetAccessCounter();
        NNT_EXPECT_RESULT_SUCCESS(storage.Read(offset, buffer, 1));
        EXPECT_EQ(0, memStorage.GetReadTimes());

        // キャッシュを破棄してから読むと再び下位ストレージへのアクセスが発生するはず
        memStorage.ResetAccessCounter();
        NNT_EXPECT_RESULT_SUCCESS(storage.OperateRange(
            nn::fs::OperationId::Invalidate,
            0,
            std::numeric_limits<int64_t>::max()));
        NNT_EXPECT_RESULT_SUCCESS(storage.Read(offset, buffer, 1));
        EXPECT_GT(memStorage.GetInvalidateTimes(), 0);
        EXPECT_GT(memStorage.GetReadTimes(), 0);
    }
}

// 境界値分析
TEST_P(FsBufferedStorageTest, Boundary)
{
    static const int BufCount = 4;
    static const size_t BlockSize = 1024;
    static const size_t MemSize = 16 * 1024;

    nnt::fs::util::Vector<char> bufCache(BlockSize * BufCount);
    nn::fssystem::FileSystemBufferManager bufManager;
    NNT_ASSERT_RESULT_SUCCESS(
        bufManager.Initialize(
            BufCount,
            reinterpret_cast<uintptr_t>(bufCache.data()),
            bufCache.size(),
            BlockSize
        )
    );

    nnt::fs::util::SafeMemoryStorage memStorage(MemSize);
    nn::fssystem::save::BufferedStorage bufStorage;
    bufStorage.Initialize(
        nn::fs::SubStorage(&memStorage, 0, MemSize),
        &bufManager,
        BlockSize,
        BufCount
    );
    if( IsBulkReadEnabled() )
    {
        bufStorage.EnableBulkRead();
    }

    nnt::fs::util::Vector<char> buffer(MemSize);
    nn::Result result;

    // 読み込み
    result = bufStorage.Read(-1, buffer.data(), 1);
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidOffset, result);
    result = bufStorage.Read(MemSize, buffer.data(), 1);
    NNT_ASSERT_RESULT_SUCCESS(result);
    result = bufStorage.Read(MemSize + 1, buffer.data(), 1);
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidOffset, result);
    result = bufStorage.Read(0, buffer.data(), MemSize + 1);
    NNT_ASSERT_RESULT_SUCCESS(result);

    // 書き込み
    result = bufStorage.Write(-1, buffer.data(), 1);
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidOffset, result);
    result = bufStorage.Write(MemSize, buffer.data(), 1);
    NNT_ASSERT_RESULT_SUCCESS(result);
    result = bufStorage.Write(MemSize + 1, buffer.data(), 1);
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidOffset, result);
    result = bufStorage.Write(0, buffer.data(), MemSize + 1);
    NNT_ASSERT_RESULT_SUCCESS(result);
}

// ランダムなオフセット・サイズで読み書きするテスト
TEST_P(FsBufferedStorageTest, RandomAccess)
{
    static const int BufCount = 4;
    static const size_t BlockSize = 1024;
    static const size_t MemSize = 16 * 1024;

    nnt::fs::util::Vector<char> bufCache(BlockSize * BufCount);
    nn::fssystem::FileSystemBufferManager bufManager;
    NNT_ASSERT_RESULT_SUCCESS(
        bufManager.Initialize(
            BufCount,
            reinterpret_cast<uintptr_t>(bufCache.data()),
            bufCache.size(),
            BlockSize
        )
    );

    nnt::fs::util::SafeMemoryStorage memStorage(MemSize);
    nn::fssystem::save::BufferedStorage bufStorage;
    bufStorage.Initialize(
        nn::fs::SubStorage(&memStorage, 0, MemSize),
        &bufManager,
        BlockSize,
        BufCount
    );
    if( IsBulkReadEnabled() )
    {
        bufStorage.EnableBulkRead();
    }

    // バッファとファイルの初期化
    nnt::fs::util::Vector<char> writeBuffer(MemSize);
    nnt::fs::util::Vector<char> writeBuffer2(MemSize, 0); // BufferedStorage と同じ内容を書き込むバッファ
    nnt::fs::util::Vector<char> readBuffer(MemSize, 0);

    NNT_ASSERT_RESULT_SUCCESS(bufStorage.Write(0, writeBuffer2.data(), MemSize));

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    // 飛ばし飛ばしアクセス
    for( size_t unitSize = 16; unitSize < BlockSize; unitSize *= 4 )
    {
        for( int64_t unitIndex = 0; unitIndex < BlockSize; unitIndex += unitSize )
        {
            for( int64_t blockIndex = 0; blockIndex < MemSize / BlockSize; ++blockIndex )
            {
                for( size_t i = 0; i < unitSize; ++i )
                {
                    writeBuffer[i] = static_cast<char>(std::uniform_int_distribution<>(0, 255)(mt));
                }

                int64_t offset = BlockSize * blockIndex + unitIndex;
                NNT_ASSERT_RESULT_SUCCESS(bufStorage.Write(offset, writeBuffer.data(), unitSize));
                std::memcpy(writeBuffer2.data() + offset, writeBuffer.data(), unitSize);

                // 書き込んだ領域を正しく読み込めるかテスト
                NNT_ASSERT_RESULT_SUCCESS(bufStorage.Read(offset, readBuffer.data(), unitSize));
                NNT_FS_UTIL_ASSERT_MEMCMPEQ(writeBuffer.data(), readBuffer.data(), unitSize);
            }
        }

        // ストレージ全域を正しく読み込めるかテスト
        NNT_ASSERT_RESULT_SUCCESS(bufStorage.Read(0, readBuffer.data(), readBuffer.size()));
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(writeBuffer2.data(), readBuffer.data(), readBuffer.size());
    }

    // 元ストレージに書き込まれているかテスト
    NNT_ASSERT_RESULT_SUCCESS(bufStorage.Flush());
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(writeBuffer2.data(), memStorage.GetBuffer(), MemSize);
}

// 読み書きで正常にキャッシュが動作しているかテスト
TEST_P(FsBufferedStorageTest, Cache)
{
    static const int BufCountMax = 16;
    static const size_t BlockSizeMin = 8 * 1024 / 16;
    static const size_t BlockSizeMax = 128 * 1024 / 4;

    nnt::fs::util::Vector<char> bufCache(BlockSizeMax * BufCountMax);
    nn::fssystem::FileSystemBufferManager bufManager;
    NNT_ASSERT_RESULT_SUCCESS(
        bufManager.Initialize(
            BufCountMax,
            reinterpret_cast<uintptr_t>(bufCache.data()),
            bufCache.size(),
            BlockSizeMin
        )
    );

    for( size_t memSize = 8 * 1024; memSize <= 128 * 1024; memSize *= 4 )
    {
        for( size_t blockSize = memSize / 16; blockSize <= memSize / 4; blockSize *= 2 )
        {
            for( int bufCount = 4; bufCount <= 16; bufCount *= 2 )
            {
                nnt::fs::util::AccessCountedMemoryStorage memStorage;
                memStorage.Initialize(memSize);

                nn::fssystem::save::BufferedStorage bufStorage;
                bufStorage.Initialize(
                    nn::fs::SubStorage(&memStorage, 0, memSize),
                    &bufManager,
                    blockSize,
                    bufCount
                );
                if( IsBulkReadEnabled() )
                {
                    bufStorage.EnableBulkRead();
                }

                nnt::fs::util::Vector<char> buffer(memSize);

                // ブロックサイズでアラインされた広範囲のアクセスは一括で実行されキャッシュされない
                {
                    memStorage.ResetAccessCounter();

                    // ちょうど 1 回書き込みが行われる
                    nn::Result result = bufStorage.Write(0, buffer.data(), memSize);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(0, memStorage.GetReadTimes());
                    EXPECT_EQ(1, memStorage.GetWriteTimes());

                    // ちょうど 1 回読み込みが行われる
                    result = bufStorage.Read(0, buffer.data(), memSize);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(1, memStorage.GetReadTimes());
                    EXPECT_EQ(1, memStorage.GetWriteTimes());
                }

                // 半端な位置からアクセスすると先頭がキャッシュされる
                {
                    memStorage.ResetAccessCounter();

                    // ちょうど 1 回ずつ読み書きが発生する
                    nn::Result result = bufStorage.Write(1, buffer.data(), blockSize * 3 - 1);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(1, memStorage.GetReadTimes());
                    EXPECT_EQ(1, memStorage.GetWriteTimes());

                    // キャッシュ範囲のみのアクセスでは読み書きは行われない
                    result = bufStorage.Read(1, buffer.data(), blockSize - 1);
                    result = bufStorage.Write(1, buffer.data(), blockSize - 1);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(1, memStorage.GetReadTimes());
                    EXPECT_EQ(1, memStorage.GetWriteTimes());
                }

                // 半端な位置でアクセスが終わると末尾がキャッシュされる
                {
                    memStorage.ResetAccessCounter();

                    // ちょうど 1 回ずつ読み書きが発生する
                    nn::Result result = bufStorage.Write(
                        memSize - blockSize * 3, buffer.data(), blockSize * 3 - 1);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(1, memStorage.GetReadTimes());
                    EXPECT_EQ(1, memStorage.GetWriteTimes());

                    // キャッシュ範囲のみのアクセスでは読み書きは行われない
                    result = bufStorage.Read(memSize - blockSize, buffer.data(), blockSize - 1);
                    result = bufStorage.Write(memSize - blockSize, buffer.data(), blockSize - 1);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(1, memStorage.GetReadTimes());
                    EXPECT_EQ(1, memStorage.GetWriteTimes());
                }

                // ブロックサイズ以下のアクセスはキャッシュされる
                {
                    memStorage.ResetAccessCounter();

                    // ちょうど 1 回だけ読み込みが発生する
                    nn::Result result = bufStorage.Write(blockSize, buffer.data(), blockSize);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(1, memStorage.GetReadTimes());
                    EXPECT_EQ(0, memStorage.GetWriteTimes());

                    // キャッシュ範囲のみのアクセスでは読み書きは行われない
                    result = bufStorage.Read(blockSize, buffer.data(), blockSize);
                    result = bufStorage.Write(blockSize, buffer.data(), blockSize);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(1, memStorage.GetReadTimes());
                    EXPECT_EQ(0, memStorage.GetWriteTimes());
                }

                // ブロックサイズでアラインされた広範囲のアクセスはキャッシュをフラッシュする
                {
                    memStorage.ResetAccessCounter();

                    // 既にキャッシュされた領域のフラッシュと全体の読み込みが行われる
                    nn::Result result = bufStorage.Read(0, buffer.data(), memSize);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(1, memStorage.GetReadTimes());
                    if( IsBulkReadEnabled() )
                    {
                        // ダーティキャッシュのある先頭2ブロックと末尾1ブロックの領域はキャッシュから読み込まれ、フラッシュされない
                        EXPECT_EQ(0, memStorage.GetWriteTimes());
                    }
                    else
                    {
                        EXPECT_EQ(3, memStorage.GetWriteTimes());
                    }

                    // ちょうど 1 回書き込みが行われる
                    result = bufStorage.Write(0, buffer.data(), memSize);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(1, memStorage.GetReadTimes());
                    EXPECT_EQ(4, memStorage.GetWriteTimes());
                }
            }
        }
    }
} // NOLINT(impl/function_size)

// キャッシュの優先順位テスト
TEST_P(FsBufferedStorageTest, CacheOrder)
{
    static const int BlockCount = 64;
    static const size_t BlockSize = 256;
    static const size_t MemSize = BlockSize * BlockCount;

    nnt::fs::util::Vector<char> bufCache(256 * 1024);
    nn::fssystem::FileSystemBufferManager bufManager;
    NNT_ASSERT_RESULT_SUCCESS(
        bufManager.Initialize(
            BlockCount,
            reinterpret_cast<uintptr_t>(bufCache.data()),
            bufCache.size(),
            BlockSize
        )
    );

    nnt::fs::util::AccessCountedMemoryStorage memStorage;
    memStorage.Initialize(MemSize);

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int loopCount = 0; loopCount < 20; ++loopCount )
    {
        int bufCount = std::uniform_int_distribution<>(1, BlockCount - 1)(mt);
        nn::fssystem::save::BufferedStorage bufStorage;
        bufStorage.Initialize(
            nn::fs::SubStorage(&memStorage, 0, MemSize),
            &bufManager,
            BlockSize,
            bufCount
        );
        if( IsBulkReadEnabled() )
        {
            bufStorage.EnableBulkRead();
        }

        std::deque<int> caches;

        for( int innerLoopCount = 0; innerLoopCount < 500; ++innerLoopCount )
        {
            int cacheIndex = std::uniform_int_distribution<>(0, bufCount - 1)(mt);
            int oldIndex = -1;
            bool isAlreadyCached = true;

            if( std::remove(caches.begin(), caches.end(), cacheIndex) == caches.end() )
            {
                // 新しくキャッシュされる
                caches.push_back(cacheIndex);
                isAlreadyCached = false;

                if( bufCount < static_cast<int>(caches.size()) )
                {
                    // 古いキャッシュは破棄される
                    oldIndex = caches.front();
                    caches.pop_front();

                    // 確認のために破棄された領域は再びキャッシュされる
                    caches.pop_front();
                    caches.push_back(oldIndex);
                }
            }
            else
            {
                // 既にあるならリストの最後尾に移動させる
                caches.back() = cacheIndex;
            }

            memStorage.ResetAccessCounter();

            int data;
            nn::Result result;

            // 古いキャッシュもこの段階ではまだ有効
            // そのためアクセスしても元ファイルへの読み書きは発生しない
            if( 0 <= oldIndex )
            {
                result = bufStorage.Read(BlockSize * oldIndex, &data, sizeof(int));
                NNT_ASSERT_RESULT_SUCCESS(result);
                result = bufStorage.Write(BlockSize * oldIndex, &data, sizeof(int));
                NNT_ASSERT_RESULT_SUCCESS(result);
                EXPECT_EQ(0, memStorage.GetReadTimes());
                EXPECT_EQ(0, memStorage.GetWriteTimes());
            }

            // 新しくキャッシュされる
            // 新しいキャッシュの読み込みが発生するかもしれない
            // 古いキャッシュのフラッシュの書き込みが発生するかもしれない
            result = bufStorage.Read(BlockSize * cacheIndex, &data, sizeof(int));
            NNT_ASSERT_RESULT_SUCCESS(result);
            result = bufStorage.Write(BlockSize * cacheIndex, &data, sizeof(int));
            NNT_ASSERT_RESULT_SUCCESS(result);
            EXPECT_EQ(isAlreadyCached ? 0 : 1, memStorage.GetReadTimes());
            EXPECT_EQ(oldIndex < 0 ? 0 : 1, memStorage.GetWriteTimes());

            // 古いキャッシュはこの段階では破棄されている
            // このことを確認するために破棄された領域は再びキャッシュされる
            if( 0 <= oldIndex )
            {
                result = bufStorage.Read(BlockSize * oldIndex, &data, sizeof(int));
                NNT_ASSERT_RESULT_SUCCESS(result);
                result = bufStorage.Write(BlockSize * oldIndex, &data, sizeof(int));
                NNT_ASSERT_RESULT_SUCCESS(result);
                EXPECT_EQ(2, memStorage.GetReadTimes());
                EXPECT_EQ(2, memStorage.GetWriteTimes());
            }
        }
    }
}

// マルチスレッドでの読み込み
TEST_P(FsBufferedStorageTest, ReadFromMultiThread)
{
    static const size_t StackSize = 32 * 1024;
    static const int ThreadCount = 8;
    static const int LoopCount = 500;
    static const size_t BlockSize = 512;
    static const int BlockCount = 32;
    static const size_t MemSize = 16 * BlockSize * BlockCount;
    static const int BufCount = ThreadCount;

    NN_OS_ALIGNAS_THREAD_STACK static char s_ThreadStack[StackSize * ThreadCount] = {};

    // ストレージを用意する
    nnt::fs::util::Vector<char> bufCache(BlockCount * BlockSize);
    nn::fssystem::FileSystemBufferManager bufManager;
    NNT_ASSERT_RESULT_SUCCESS(
        bufManager.Initialize(
            BlockCount,
            reinterpret_cast<uintptr_t>(bufCache.data()),
            bufCache.size(),
            BlockSize
        )
    );

    nnt::fs::util::AccessCountedMemoryStorage memStorage;
    memStorage.Initialize(MemSize);

    nn::fssystem::save::BufferedStorage bufStorage;
    bufStorage.Initialize(
        nn::fs::SubStorage(&memStorage, 0, MemSize),
        &bufManager,
        BlockSize,
        BufCount
    );
    if( IsBulkReadEnabled() )
    {
        bufStorage.EnableBulkRead();
    }

    nnt::fs::util::Vector<char> writeBuffer(MemSize);
    std::iota(writeBuffer.begin(), writeBuffer.end(), '\x0');
    NNT_ASSERT_RESULT_SUCCESS(bufStorage.Write(0, writeBuffer.data(), writeBuffer.size()));

    struct Argument
    {
        int seed;
        nn::fssystem::save::BufferedStorage* pBufStorage;
        nnt::fs::util::Vector<char>* pBufWrite;
    };

    Argument args[ThreadCount] = {};
    nn::os::ThreadType threads[ThreadCount];

    // 複数スレッドから読み込みを行う
    for( int threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
    {
        const auto ThreadFunc = [](void* pArg) NN_NOEXCEPT
        {
            const auto seed = reinterpret_cast<Argument*>(pArg)->seed;
            const auto pBufStorage = reinterpret_cast<Argument*>(pArg)->pBufStorage;
            auto& writeBuffer = *reinterpret_cast<Argument*>(pArg)->pBufWrite;

            nnt::fs::util::Vector<char> readBuffer(MemSize);
            std::mt19937 mt(seed);

            for( int count = 0; count < LoopCount; ++count )
            {
                size_t offset = std::uniform_int_distribution<size_t>(0, MemSize - 1)(mt);
                size_t size = std::uniform_int_distribution<size_t>(1, MemSize - offset)(mt);

                nn::Result result = pBufStorage->Read(offset, readBuffer.data(), size);
                NNT_ASSERT_RESULT_SUCCESS(result);

                EXPECT_TRUE(
                    std::equal(
                        readBuffer.begin(),
                        std::next(readBuffer.begin(), size),
                        std::next(writeBuffer.begin(), offset)
                    )
                );
            }
        };

        args[threadIndex].seed = nnt::fs::util::GetRandomSeed();
        args[threadIndex].pBufStorage = &bufStorage;
        args[threadIndex].pBufWrite = &writeBuffer;

        nn::Result result = nn::os::CreateThread(
            &threads[threadIndex],
            ThreadFunc,
            &args[threadIndex],
            s_ThreadStack + StackSize * threadIndex,
            StackSize,
            nn::os::DefaultThreadPriority
        );
        NNT_ASSERT_RESULT_SUCCESS(result);
        nn::os::StartThread(&threads[threadIndex]);
    }

    // スレッドの後始末
    for( auto& thread : threads )
    {
        nn::os::WaitThread(&thread);
        nn::os::DestroyThread(&thread);
    }
}

#if !defined(NN_SDK_BUILD_RELEASE)
// 初期化前のテスト
TEST_P(FsBufferedStorageDeathTest, Precondition)
{
    static const size_t MemSize = 16 * 1024;
    static const size_t BlockSize = 1024;
    static const int BufCount = 4;

    nnt::fs::util::Vector<char> bufCache(BufCount * BlockSize);
    nn::fssystem::FileSystemBufferManager bufManager;
    NNT_ASSERT_RESULT_SUCCESS(
        bufManager.Initialize(
            BufCount,
            reinterpret_cast<uintptr_t>(bufCache.data()),
            bufCache.size(),
            BlockSize
        )
    );

    nnt::fs::util::Vector<char> buffer(1024);
    int64_t size = 0;

    nn::fssystem::save::BufferedStorage bufStorage;
    if( IsBulkReadEnabled() )
    {
        bufStorage.EnableBulkRead();
    }

    // 未初期化
    EXPECT_DEATH_IF_SUPPORTED(bufStorage.Read(0, buffer.data(), buffer.size()), "");
    EXPECT_DEATH_IF_SUPPORTED(bufStorage.Write(0, buffer.data(), buffer.size()), "");
    EXPECT_DEATH_IF_SUPPORTED(bufStorage.GetSize(&size), "");
    EXPECT_DEATH_IF_SUPPORTED(bufStorage.SetSize(1024), "");
    EXPECT_DEATH_IF_SUPPORTED(bufStorage.Flush(), "");

    // 初期化
    nnt::fs::util::SafeMemoryStorage memStorage(MemSize);
    nn::fs::SubStorage storage(&memStorage, 0, MemSize);
    EXPECT_DEATH_IF_SUPPORTED(bufStorage.Initialize(storage, nullptr, 0, BufCount), "");
    EXPECT_DEATH_IF_SUPPORTED(bufStorage.Initialize(storage, &bufManager, 0, BufCount), "");
    EXPECT_DEATH_IF_SUPPORTED(bufStorage.Initialize(storage, &bufManager, 1023, BufCount), "");
    EXPECT_DEATH_IF_SUPPORTED(bufStorage.Initialize(storage, &bufManager, 1025, BufCount), "");
    EXPECT_DEATH_IF_SUPPORTED(bufStorage.Initialize(storage, &bufManager, BlockSize, 0), "");
    nn::Result result = bufStorage.Initialize(storage, &bufManager, BlockSize, BufCount);
    NNT_ASSERT_RESULT_SUCCESS(result);

    // 初期化済み
    EXPECT_DEATH_IF_SUPPORTED(bufStorage.GetSize(nullptr), "");
    result = bufStorage.GetSize(&size);
    NNT_ASSERT_RESULT_SUCCESS(result);

    // UnsupportedOperation
    //result = bufStorage.SetSize(1024);
    //NNT_ASSERT_RESULT_SUCCESS(result);

    result = bufStorage.Flush();
    NNT_ASSERT_RESULT_SUCCESS(result);
}
#endif

INSTANTIATE_TEST_CASE_P(FsBufferedStorageTest,
                        FsBufferedStorageTest,
                        ::testing::Values(true, false));
