﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include "testFs_util_CommonStorageTests.h"
#include "testFs_util_AccessCountWrapperStorage.h"

// アクセスカウント付きメモリストレージのテスト
TEST(FsAccessCountWrapperStorageTest, Simple)
{
    static const int LoopCount = 512;
    static const size_t StorageSize = 1024;

    nnt::fs::util::AccessCountedMemoryStorage storageMem;
    nnt::fs::util::Vector<char> buf1(StorageSize);
    nnt::fs::util::Vector<char> buf2(StorageSize);

    // データ初期化
    for( size_t i = 0; i < StorageSize; i++ )
    {
        buf1[i] = i & 0xFF;
    }

    // 元となるファイルを初期化します。
    storageMem.Initialize(StorageSize);

    for( size_t transferSize = 1; transferSize < StorageSize; transferSize <<= 1 )
    {
        const int verifyReadTimes =
            static_cast<int>(LoopCount * ((StorageSize + (transferSize - 1)) / transferSize));
        const int verifyWriteTimes = verifyReadTimes;
        const int verifyWriteWithFlushTimes = verifyWriteTimes;

        storageMem.ResetAccessCounter();

        // sequential access
        for( int i = 0; i < LoopCount; ++i )
        {
            if( (0 < i) && (0 == (i % (LoopCount >> 4))) )
            {
                NN_SDK_LOG(".");
            }

            // write
            for( size_t offset = 0; offset < StorageSize; offset += transferSize )
            {
                size_t length =
                    (offset + transferSize <= StorageSize) ? transferSize : StorageSize - offset;
                NNT_ASSERT_RESULT_SUCCESS(
                    storageMem.Write(offset, &buf1[offset], length));
                NNT_ASSERT_RESULT_SUCCESS(storageMem.Flush());
            }

            // read
            for( size_t offset = 0; offset < StorageSize; offset += transferSize )
            {
                size_t length =
                    (offset + transferSize <= StorageSize) ? transferSize : StorageSize - offset;
                NNT_ASSERT_RESULT_SUCCESS(
                    storageMem.Read(offset, &buf2[offset], length));
            }

            // 読み込みデータ検証
            // 毎ループ行うべきだが遅すぎるので最後に１回だけ行う
            for( size_t j = 0; j < StorageSize; ++j )
            {
                ASSERT_EQ(j & 0xFF, static_cast<uint8_t>(buf2[j]));
            }

            // 読み込みバッファクリア
            std::fill(buf2.begin(), buf2.end(), '\x0');
        }

        // アクセス回数の確認
        ASSERT_EQ(verifyReadTimes, storageMem.GetReadTimes());
        ASSERT_EQ(verifyWriteTimes, storageMem.GetWriteTimes());
        ASSERT_EQ(verifyWriteWithFlushTimes, storageMem.GetFlushTimes());

        NN_SDK_LOG("\n");
    }

    // サイズの取得ができることを確認します。
    int64_t sizeStorage;
    NNT_ASSERT_RESULT_SUCCESS(storageMem.GetSize(&sizeStorage));
    ASSERT_EQ(StorageSize, sizeStorage);
}

class AccessCountWrapperStorageTestSetup : public TestStorageSetup
{
public:
    static const int64_t MinFileSize = 1;

public:
    // コンストラクタです。
    AccessCountWrapperStorageTestSetup() NN_NOEXCEPT
    {
    }

    // ファイルオブジェクトをフォーマットします。
    virtual void Format(
                     int64_t sizeStorage,
                     int64_t offset,
                     size_t sizeBlock
                 ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(sizeBlock);
        m_pStorageMem.reset(new AccessCountedMemoryStorage);
        m_pStorageMem->Initialize(static_cast<size_t>(sizeStorage + offset * 2));
    }

    // ファイルオブジェクトを初期化します。
    virtual void Initialize(
                     int64_t sizeStorage,
                     int64_t offset,
                     size_t sizeBlock
                 ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(sizeStorage);
        NN_UNUSED(offset);
        NN_UNUSED(sizeBlock);
        SetStorage(m_pStorageMem.get());
    }

    // ファイルオブジェクトをアンマウントします。
    virtual void Unmount() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    // ファイルオブジェクトを破棄します。
    virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE
    {
        m_pStorageMem.reset();
    }

    // 読み込み専用かどうか
    virtual bool IsReadOnly() const NN_NOEXCEPT NN_OVERRIDE
    {
        return false;
    }

    // ファイルへのデータ書き込み時に自動伸長するかどうか
    virtual bool IsAutoExtendSizeOnWrite() const NN_NOEXCEPT NN_OVERRIDE
    {
        return false;
    }

    // ファイルのサイズ変更が可能かどうか
    virtual bool IsEnableResize() const NN_NOEXCEPT NN_OVERRIDE
    {
        return false;
    }

    // サイズ 0 のファイルを作成可能かどうか
    virtual bool IsAcceptsSizeZero() const NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

private:
    std::unique_ptr<AccessCountedMemoryStorage> m_pStorageMem;
};

// 共通テストを行います。
TEST(FsAccessCountWrapperStorageTest, StorageCommonTest)
{
    StorageTest::RunTest<AccessCountWrapperStorageTestSetup>();
}

