﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/crypto.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_SaveDataFileSystemCacheManager.h>
#include <nn/fssystem/fs_SaveDataFileSystemCacheRegister.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h.>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>

namespace
{
    class TestSaveDataFileSystem : public nn::fssystem::SaveDataFileSystem
    {
    public:
        static int GetInstanceCount() NN_NOEXCEPT
        {
            return g_InstanceCount;
        }

        TestSaveDataFileSystem(
                nn::fs::SaveDataSpaceId spaceId,
                nn::fs::SaveDataId saveDataId
            ) NN_NOEXCEPT
        {
            ++g_InstanceCount;
            RegisterCacheObserver(nullptr, spaceId, saveDataId);
        }

        virtual ~TestSaveDataFileSystem() NN_NOEXCEPT NN_OVERRIDE
        {
            --g_InstanceCount;
        }

    private:
        static int g_InstanceCount;
    };
    int TestSaveDataFileSystem::g_InstanceCount = 0;

    // SaveData Parameters
    const int64_t AvailableSize = 32 * 1024 * 1024;
    const int64_t ReservedSize = 32 * 1024 * 1024;
    const size_t BlockSize = 16 * 1024;
    const uint32_t CountBlock = AvailableSize / BlockSize;
    const uint32_t CountReservedBlock = ReservedSize / BlockSize;
    const int CountExpandMax = 1;
    const auto MaxBufferCacheCount = 1024;
    const auto BufferManagerHeapSize = 8 * 1024 * 1024;
    const size_t BufferManagerWorkBufferSize = 128 * 1024;

    const int HeapSize = 10 * 1024 * 1024;
    char g_HeapStack[HeapSize];

    NN_ALIGNAS(4096) char g_BufferManagerHeap[BufferManagerHeapSize];
    char g_BufferManagerWorkBuffer[BufferManagerWorkBufferSize];
    nn::fssystem::FileSystemBufferManager g_BufferManager;

    const size_t BufferPoolSize = 1024 * 1024;
    NN_ALIGNAS(4096) char g_BufferPool[BufferPoolSize];

    const size_t WorkBufferSize = nn::fssystem::BufferPoolWorkSize;
    NN_ALIGNAS(8) char g_WorkBuffer[WorkBufferSize];

    // Memory Storage version
    nnt::fs::util::SafeMemoryStorage g_BaseStorage;

    void InitializeTestState() NN_NOEXCEPT
    {
        nnt::fs::util::InitializeSaveDataTestHelper();
        nnt::fs::util::InitializeTestLibraryHeap(g_HeapStack, HeapSize);

        nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
        nn::fssystem::InitializeAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
        nn::fssystem::save::HierarchicalIntegrityVerificationStorage::SetGenerateRandomFunction(nnt::fs::util::GenerateRandomForSaveDataTest);
        nn::fssystem::InitializeBufferPool(g_BufferPool, BufferPoolSize, g_WorkBuffer, WorkBufferSize);

        g_BufferManager.Initialize(
            MaxBufferCacheCount,
            reinterpret_cast<uintptr_t>(g_BufferManagerHeap),
            sizeof(g_BufferManagerHeap),
            BlockSize,
            g_BufferManagerWorkBuffer,
            BufferManagerWorkBufferSize
        );
    }
    void FinalizeTestState() NN_NOEXCEPT
    {
        g_BufferManager.Finalize();
        nnt::fs::util::FinalizeSaveDataTestHelper();
    }

    void SetUpSaveDataFs(
             std::shared_ptr<nn::fssystem::SaveDataFileSystem>* pOutValue,
             nn::fs::SaveDataSpaceId spaceId,
             nn::fs::SaveDataId saveDataId
         ) NN_NOEXCEPT
    {
        nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam paramDuplex
            = {{BlockSize, BlockSize}};
        nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam paramIntegrity
            = {{BlockSize, BlockSize, BlockSize, BlockSize}};

        int64_t totalSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
                &totalSize,
                BlockSize,
                CountExpandMax,
                paramDuplex,
                paramIntegrity,
                CountBlock,
                CountReservedBlock
            )
        );

        g_BaseStorage.Initialize(totalSize);

        nn::fs::SubStorage storage(&g_BaseStorage, 0, totalSize);
        nn::fs::SaveDataHashSalt salt;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::Format(
                storage,
                BlockSize,
                CountExpandMax,
                paramDuplex,
                paramIntegrity,
                CountBlock,
                CountReservedBlock,
                &g_BufferManager,
                nnt::fs::util::GetMacGenerator(),
                salt
            )
        );

        // テスト用の SaveDataFileSystem を生成
        std::shared_ptr<nn::fssystem::SaveDataFileSystem> saveDataFs(
            new TestSaveDataFileSystem(
                spaceId,
                saveDataId
            )
        );
        ASSERT_NE(saveDataFs, nullptr);

        // 初期化・マウント
        NNT_ASSERT_RESULT_SUCCESS(
            saveDataFs->Initialize(
                &g_BaseStorage,
                &g_BufferManager,
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );

        *pOutValue = std::move(saveDataFs);
    }
}

// キャッシュを登録し取得するテスト
TEST(TestSaveDataFileSystemCacheManager, GetCache)
{
    const int MaxCacheCount = 4;
    const int SaveDataCount = 5;

    nn::fssystem::SaveDataFileSystemCacheManager manager;
    manager.Initialize(MaxCacheCount);

    for( int i = 0; i < SaveDataCount; ++i )
    {
        nn::fs::SaveDataId saveDataId = static_cast<nn::fs::SaveDataId>(i);
        std::shared_ptr<nn::fssystem::SaveDataFileSystem> pFileSystem;
        SetUpSaveDataFs(
            &pFileSystem,
            nn::fs::SaveDataSpaceId::User,
            saveDataId
        );

        std::shared_ptr<nn::fs::fsa::IFileSystem> saveDataFs
            = nn::fssystem::AllocateShared<nn::fssystem::SaveDataFileSystemCacheRegister>(
                  std::move(pFileSystem), &manager
              );
        NN_UNUSED(saveDataFs);
    }

    // SaveDataFileSystem のインスタンスが 4 つ有ることを確認
    ASSERT_EQ(MaxCacheCount, TestSaveDataFileSystem::GetInstanceCount());

    // キャッシュしていない ID は見つからない
    {
        nn::fs::SaveDataId saveDataId = static_cast<nn::fs::SaveDataId>(0);
        std::shared_ptr<nn::fssystem::SaveDataFileSystem> pFileSystem;
        ASSERT_FALSE(
            manager.GetCache(
                &pFileSystem,
                nn::fs::SaveDataSpaceId::User,
                saveDataId
            )
        );
    }
    {
        nn::fs::SaveDataId saveDataId = static_cast<nn::fs::SaveDataId>(1);
        std::shared_ptr<nn::fssystem::SaveDataFileSystem> pFileSystem;
        ASSERT_FALSE(
            manager.GetCache(
                &pFileSystem,
                nn::fs::SaveDataSpaceId::System,
                saveDataId
            )
        );
    }
    ASSERT_EQ(MaxCacheCount, TestSaveDataFileSystem::GetInstanceCount());

    // キャッシュされていることの確認
    // 取得されるとインスタンスが解放される
    for( int i = SaveDataCount - MaxCacheCount; i < SaveDataCount; ++i )
    {
        {
            nn::fs::SaveDataId saveDataId = static_cast<nn::fs::SaveDataId>(i);
            std::shared_ptr<nn::fssystem::SaveDataFileSystem> pFileSystem;
            ASSERT_TRUE(
                manager.GetCache(
                    &pFileSystem,
                    nn::fs::SaveDataSpaceId::User,
                    saveDataId
                )
            );
            ASSERT_EQ(saveDataId, pFileSystem->GetSaveDataId());
        }
        ASSERT_EQ(MaxCacheCount - i, TestSaveDataFileSystem::GetInstanceCount());
    }
}

// キャッシュを解放するテスト
TEST(TestSaveDataFileSystemCacheManager, Unregister)
{
    const int MaxCacheCount = 8;
    const int SaveDataCount = 12;

    nn::fssystem::SaveDataFileSystemCacheManager manager;
    manager.Initialize(MaxCacheCount);

    for( int i = 0; i < SaveDataCount; ++i )
    {
        nn::fs::SaveDataId saveDataId = static_cast<nn::fs::SaveDataId>(i);
        std::shared_ptr<nn::fssystem::SaveDataFileSystem> pFileSystem;
        SetUpSaveDataFs(
            &pFileSystem,
            nn::fs::SaveDataSpaceId::User,
            saveDataId
        );

        std::shared_ptr<nn::fs::fsa::IFileSystem> saveDataFs
            = nn::fssystem::AllocateShared<nn::fssystem::SaveDataFileSystemCacheRegister>(
                  std::move(pFileSystem), &manager
              );
        NN_UNUSED(saveDataFs);
    }

    // SaveDataFileSystem のインスタンスが 8 つ有ることを確認
    ASSERT_EQ(MaxCacheCount, TestSaveDataFileSystem::GetInstanceCount());

    // 解放するとインスタンスの数が減る
    int minCachedSaveDataId = SaveDataCount - MaxCacheCount;
    for( int i = SaveDataCount - 1; i >= minCachedSaveDataId; --i )
    {
        nn::fs::SaveDataId saveDataId = static_cast<nn::fs::SaveDataId>(i);
        manager.Unregister(nn::fs::SaveDataSpaceId::User, saveDataId);
        ASSERT_EQ(i - minCachedSaveDataId, TestSaveDataFileSystem::GetInstanceCount());
    }
}

// 同じ ID でも重複して登録されることを確認するテスト
TEST(TestSaveDataFileSystemCacheManager, SameId)
{
    const int MaxCacheCount = 6;
    const int SaveDataIdCount = 4;
    const int SaveDataCount = SaveDataIdCount * 2;

    nn::fssystem::SaveDataFileSystemCacheManager manager;
    manager.Initialize(MaxCacheCount);

    for( int i = 0; i < SaveDataCount; ++i )
    {
        nn::fs::SaveDataId saveDataId = static_cast<nn::fs::SaveDataId>(i % SaveDataIdCount);
        std::shared_ptr<nn::fssystem::SaveDataFileSystem> pFileSystem;
        SetUpSaveDataFs(
            &pFileSystem,
            nn::fs::SaveDataSpaceId::User,
            saveDataId
        );

        std::shared_ptr<nn::fs::fsa::IFileSystem> saveDataFs
            = nn::fssystem::AllocateShared<nn::fssystem::SaveDataFileSystemCacheRegister>(
                  std::move(pFileSystem), &manager
              );
        NN_UNUSED(saveDataFs);
    }

    // 同じ ID でも重複して登録可能
    ASSERT_EQ(MaxCacheCount, TestSaveDataFileSystem::GetInstanceCount());
}

// SD 上のセーブデータはキャッシュしない事を確認するテスト
TEST(TestSaveDataFileSystemCacheManager, SdSystem)
{
    const int MaxCacheCount = 1;
    const int SaveDataCount = 4;

    nn::fssystem::SaveDataFileSystemCacheManager manager;
    manager.Initialize(MaxCacheCount);

    for( int i = 0; i < SaveDataCount; ++i )
    {
        nn::fs::SaveDataId saveDataId = static_cast<nn::fs::SaveDataId>(i);
        std::shared_ptr<nn::fssystem::SaveDataFileSystem> pFileSystem;
        SetUpSaveDataFs(
            &pFileSystem,
            nn::fs::SaveDataSpaceId::SdSystem,
            saveDataId
        );

        std::shared_ptr<nn::fs::fsa::IFileSystem> saveDataFs
            = nn::fssystem::AllocateShared<nn::fssystem::SaveDataFileSystemCacheRegister>(
                  std::move(pFileSystem), &manager
              );
        NN_UNUSED(saveDataFs);
    }

    // 登録されていないことを確認
    ASSERT_EQ(0, TestSaveDataFileSystem::GetInstanceCount());
}

// キャッシュ数が 0 でも動作することを確認するテスト
TEST(TestSaveDataFileSystemCacheManager, NoCache)
{
    const int MaxCacheCount = 0;

    nn::fssystem::SaveDataFileSystemCacheManager manager;
    manager.Initialize(MaxCacheCount);
    {
        nn::fs::SaveDataId saveDataId = static_cast<nn::fs::SaveDataId>(1);
        std::shared_ptr<nn::fssystem::SaveDataFileSystem> pFileSystem;
        SetUpSaveDataFs(
            &pFileSystem,
            nn::fs::SaveDataSpaceId::User,
            saveDataId
        );

        std::shared_ptr<nn::fs::fsa::IFileSystem> saveDataFs
            = nn::fssystem::AllocateShared<nn::fssystem::SaveDataFileSystemCacheRegister>(
                  std::move(pFileSystem), &manager
              );
        NN_UNUSED(saveDataFs);
    }

    // 登録されていないことを確認
    ASSERT_EQ(0, TestSaveDataFileSystem::GetInstanceCount());
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nnt::fs::util::ResetAllocateCount();

    InitializeTestState();

    auto result = RUN_ALL_TESTS();

    FinalizeTestState();

    if( nnt::fs::util::CheckMemoryLeak() )
    {
        nnt::Exit(1);
    }
    nnt::Exit(result);
}
